<?php

use Illuminate\Support\Facades\Password;
use Livewire\Attributes\Layout;
use Livewire\Volt\Component;

new #[Layout('layouts.guest')] class extends Component {
    public string $email = '';

    public function sendPasswordResetLink(): void
    {
        $this->validate([
            'email' => ['required', 'string', 'email'],
        ]);

        // We will send the password reset link to this user. Once we have attempted
        // to send the link, we will examine the response then see the message we
        // need to show to the user. Finally, we'll send out a proper response.
        $status = Password::sendResetLink($this->only('email'));

        if ($status != Password::RESET_LINK_SENT) {
            $this->addError('email', __($status));

            return;
        }

        $this->reset('email');

        session()->flash('status', __($status));
    }
}; ?>

<div>
    <x-slot name="title">
        {{ __('Forgot your password?') }}
    </x-slot>
    <x-slot name="description">
        {{ __('Let us know your email address and we will help get you back in!') }}
    </x-slot>
    <!-- Session Status -->
    <x-auth-session-status class="mb-4" :status="session('status')" />
    <x-auth-session-error class="mb-4" :loginError="session('loginError')" />

    <form wire:submit="sendPasswordResetLink" class="mt-8 space-y-6">
        <!-- Email Address -->
        <div>
            <x-input-label
                for="email"
                :value="__('Email Address')"
                class="block text-sm font-medium text-gray-700 dark:text-gray-300"
            />
            <x-text-input
                wire:model="email"
                id="email"
                class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-primary-500 focus:ring-primary-500 sm:text-sm dark:border-gray-700 dark:bg-gray-800 dark:text-white"
                type="email"
                name="email"
                required
                autofocus
            />
            <x-input-error :messages="$errors->get('email')" class="mt-2" />
        </div>

        <x-primary-button
            class="w-full justify-center rounded-md border border-transparent bg-primary-600 px-4 py-2 text-sm font-medium text-white shadow-sm hover:bg-primary-700 focus:outline-none focus:ring-2 focus:ring-primary-500 focus:ring-offset-2 dark:bg-primary-500 dark:hover:bg-primary-600"
        >
            {{ __('Request Password Reset Email') }}
            @svg('hugeicons-arrow-right-02', 'ms-2 inline h-5 w-5')
        </x-primary-button>

        <div class="mt-6 text-center">
            <a
                class="text-sm font-medium text-primary-600 hover:text-primary-500 dark:text-primary-400 dark:hover:text-primary-300"
                href="{{ route('login') }}"
                wire:navigate
            >
                {{ __('Back to login') }}
            </a>
        </div>
    </form>
</div>
