/*
 * Copyright (c) 2024 See AUTHORS file.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.tommyettinger.fontwriter;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.files.FileHandle;
import com.badlogic.gdx.graphics.Texture;
import com.badlogic.gdx.graphics.g2d.BitmapFont;
import com.badlogic.gdx.graphics.g2d.TextureAtlas;
import com.badlogic.gdx.graphics.g2d.TextureRegion;
import com.badlogic.gdx.scenes.scene2d.ui.Skin;
import com.badlogic.gdx.utils.Array;
import com.badlogic.gdx.utils.Json;
import com.badlogic.gdx.utils.JsonValue;
import com.badlogic.gdx.utils.SerializationException;

/**
 * A subclass of {@link Skin} that includes a serializer for Structured JSON Fonts, which are typically generated by
 * <a href="https://github.com/tommyettinger/fontwriter">fontwriter</a>. This can load
 * FontWriter's JSON format into either {@link BitmapFont} objects. It can also
 * load {@link BitmapFont}s from AngelCode BMFont files (with a ".fnt" extension).
 * <br>
 * If you are using {@link com.badlogic.gdx.assets.AssetManager}, use {@link JsonSkinLoader}.
 */

public class JsonSkin extends Skin {
    /** Creates an empty skin. */
    public JsonSkin() {
    }

    /** Creates a skin containing the resources in the specified skin JSON file. If a file in the same directory with a ".atlas"
     * extension exists, it is loaded as a {@link TextureAtlas} and the texture regions added to the skin. The atlas is
     * automatically disposed when the skin is disposed.
     * @param  skinFile The JSON file to be read.
     */
    public JsonSkin(FileHandle skinFile) {
        super(skinFile);

    }

    /** Creates a skin containing the resources in the specified skin JSON file and the texture regions from the specified atlas.
     * The atlas is automatically disposed when the skin is disposed.
     * @param skinFile The JSON file to be read.
     * @param atlas The texture atlas to be associated with the {@link Skin}.
     */
    public JsonSkin(FileHandle skinFile, TextureAtlas atlas) {
        super(skinFile, atlas);
    }

    /** Creates a skin containing the texture regions from the specified atlas. The atlas is automatically disposed when the skin
     * is disposed.
     * @param atlas The texture atlas to be associated with the {@link Skin}.
     */
    public JsonSkin(TextureAtlas atlas) {
        super(atlas);
    }

    /**
     * Overrides the default JSON loader to process Structured JSON Fonts from a Skin JSON.
     * This allows BitmapFont items to be loaded from either .fnt or .json files.
     * @param skinFile The JSON file to be processed.
     * @return The {@link Json} used to read the file.
     */
    @Override
    protected Json getJsonLoader(final FileHandle skinFile) {
        Json json = super.getJsonLoader(skinFile);
        final Skin skin = this;

        json.setSerializer(BitmapFont.class, new Json.ReadOnlySerializer<BitmapFont>() {
            public BitmapFont read (Json json, JsonValue jsonData, Class type) {
                String path = json.readValue("file", String.class, jsonData);

                FileHandle fontFile = skinFile.sibling(path);
                if (!fontFile.exists()) fontFile = Gdx.files.internal(path);
                if (!fontFile.exists()) throw new SerializationException("Font file not found: " + fontFile);

                boolean lzb = "dat".equalsIgnoreCase(fontFile.extension());
                boolean fw = "json".equalsIgnoreCase(fontFile.extension());

                float scaledSize = json.readValue("scaledSize", float.class, -1f, jsonData);
                Boolean flip = json.readValue("flip", Boolean.class, false, jsonData);
                Boolean markupEnabled = json.readValue("markupEnabled", Boolean.class, false, jsonData);
                // This defaults to true if loading from .fnt, or false if loading from .json :
                Boolean useIntegerPositions = json.readValue("useIntegerPositions", Boolean.class, !(fw || lzb), jsonData);

                // Use a region with the same name as the font, else use a PNG file in the same directory as the FNT file.
                String regionName = fontFile.nameWithoutExtension();
                try {
                    BitmapFont bitmapFont;
                    Array<TextureRegion> regions = skin.getRegions(regionName);
                    if (regions != null && regions.notEmpty()) {
                        if(fw || lzb) {
                            bitmapFont = BitmapFontSupport.loadStructuredJson(fontFile, regions.first(), flip);
                        }
                        else {
                            bitmapFont = new BitmapFont(new BitmapFont.BitmapFontData(fontFile, flip), regions, true);
                        }
                    } else {
                        TextureRegion region = skin.optional(regionName, TextureRegion.class);
                        if (region != null)
                        {
                            if(fw || lzb) {
                                bitmapFont = BitmapFontSupport.loadStructuredJson(fontFile, region, flip);
                            }
                            else {
                                bitmapFont = new BitmapFont(fontFile, region, flip);
                            }
                        }
                        else {
                            FileHandle imageFile = fontFile.sibling(regionName + ".png");
                            if (imageFile.exists()) {
                                region = new TextureRegion(new Texture(imageFile));
                                if(fw || lzb) {
                                    bitmapFont = BitmapFontSupport.loadStructuredJson(fontFile, region, flip);
                                } else {
                                    bitmapFont = new BitmapFont(fontFile, region, flip);
                                }
                            } else {
                                if(fw || lzb)
                                    throw new RuntimeException("Missing image file or TextureRegion.");
                                else {
                                    bitmapFont = new BitmapFont(fontFile, flip);
                                }
                            }
                        }
                    }
                    bitmapFont.getData().markupEnabled = markupEnabled;
                    bitmapFont.setUseIntegerPositions(useIntegerPositions);
                    // For BitmapFont, scaled size is the desired cap height to scale the font to.
                    // But we scale it based on line height here because that's what Font does.
                    if (scaledSize != -1) {
                        bitmapFont.getData().setScale(scaledSize / bitmapFont.getLineHeight());
                    }

                    return bitmapFont;
                } catch (RuntimeException ex) {
                    throw new SerializationException("Error loading BitmapFont: " + fontFile, ex);
                }
            }
        });
        return json;
    }
}