# NAME

CGI::Emulate::PSGI - PSGI adapter for CGI

# SYNOPSIS

    my $app = CGI::Emulate::PSGI->handler(sub {
        # Existing CGI code
    });

# DESCRIPTION

This module allows an application designed for the CGI environment to
run in a PSGI environment, and thus on any of the backends that PSGI
supports.

It works by translating the environment provided by the PSGI
specification to one expected by the CGI specification. Likewise, it
captures output as it would be prepared for the CGI standard, and
translates it to the format expected for the PSGI standard using
[CGI::Parse::PSGI](https://metacpan.org/pod/CGI::Parse::PSGI) module.

# CGI.pm

If your application uses [CGI](https://metacpan.org/pod/CGI), be sure to cleanup the global
variables in the handler loop yourself, so:

    my $app = CGI::Emulate::PSGI->handler(sub {
        use CGI;
        CGI::initialize_globals();
        my $q = CGI->new;
        # ...
    });

Otherwise previous request variables will be reused in the new
requests.

Alternatively, you can install and use [CGI::Compile](https://metacpan.org/pod/CGI::Compile) from CPAN and
compiles your existing CGI scripts into a sub that is perfectly ready
to be converted to PSGI application using this module.

    my $sub = CGI::Compile->compile("/path/to/script.cgi");
    my $app = CGI::Emulate::PSGI->handler($sub);

This will take care of assigning a unique namespace for each script
etc. See [CGI::Compile](https://metacpan.org/pod/CGI::Compile) for details.

You can also consider using [CGI::PSGI](https://metacpan.org/pod/CGI::PSGI) but that would require you to
slightly change your code from:

    my $q = CGI->new;
    # ...
    print $q->header, $output;

into:

    use CGI::PSGI;

    my $app = sub {
        my $env = shift;
        my $q = CGI::PSGI->new($env);
        # ...
        return [ $q->psgi_header, [ $output ] ];
    };

See [CGI::PSGI](https://metacpan.org/pod/CGI::PSGI) for details.

# METHODS

- handler

        my $app = CGI::Emulate::PSGI->handler($code);

    Creates a PSGI application code reference out of CGI code reference.

- emulate\_environment

        my %env = CGI::Emulate::PSGI->emulate_environment($env);

    Creates an environment hash out of PSGI environment hash. If your code
    or framework just needs an environment variable emulation, use this
    method like:

        local %ENV = (%ENV, CGI::Emulate::PSGI->emulate_environment($env));
        # run your application

    If you use `handler` method to create a PSGI environment hash, this
    is automatically called in the created application.

# AUTHOR

Tokuhiro Matsuno <tokuhirom@cpan.org>

Tatsuhiko Miyagawa

# COPYRIGHT AND LICENSE

Copyright (c) 2009-2010 by tokuhirom.

This program is free software; you can redistribute
it and/or modify it under the same terms as Perl itself.

The full text of the license can be found in the
LICENSE file included with this module.

# SEE ALSO

[PSGI](https://metacpan.org/pod/PSGI) [CGI::Compile](https://metacpan.org/pod/CGI::Compile) [CGI::PSGI](https://metacpan.org/pod/CGI::PSGI) [Plack](https://metacpan.org/pod/Plack) [CGI::Parse::PSGI](https://metacpan.org/pod/CGI::Parse::PSGI)
