import Color from 'color';
import { source } from 'common-tags';
import { AnnotatedVariant, colorSetToVariants } from '../color-set/index.js';
import { dirname, join, Template, version } from './index.js';

function packageName(variant: AnnotatedVariant): string {
  return `hyper-${variant.title.kebab}`;
}

const template: Template = {
  name: 'Hyper',
  render: async function* (colorSet) {
    const variants = colorSetToVariants(colorSet);
    for (const variant of variants) {
      const indexFileName = 'index.js';
      yield {
        path: join(packageName(variant), 'package.json'),
        content: JSON.stringify(
          {
            name: packageName(variant),
            version,
            description: `${variant.title.human} theme for Hyper.app, generated by themer`,
            keywords: ['themer', colorSet.name, variant.name, 'hyper'],
            main: indexFileName,
          },
          null,
          2,
        ),
      };
      yield {
        path: join(packageName(variant), indexFileName),
        content: source`
          module.exports.decorateConfig = config => {
            return Object.assign({}, config, {
              cursorColor: '${Color(variant.colors.accent6)
                .fade(0.5)
                .rgb()
                .string()}',
              cursorAccentColor: '${variant.colors.shade0}',
              foregroundColor: '${variant.colors.shade6}',
              backgroundColor: '${variant.colors.shade0}',
              selectionColor: '${Color(variant.colors.accent5)
                .fade(0.9)
                .rgb()
                .string()}',
              borderColor: '${variant.colors.accent4}',
              colors: {
                black: '${variant.colors.shade0}',
                red: '${variant.colors.accent0}',
                green: '${variant.colors.accent3}',
                yellow: '${variant.colors.accent2}',
                blue: '${variant.colors.accent5}',
                magenta: '${variant.colors.accent7}',
                cyan: '${variant.colors.accent4}',
                white: '${variant.colors.shade6}',
                lightBlack: '${variant.colors.shade1}',
                lightRed: '${variant.colors.accent1}',
                lightGreen: '${variant.colors.accent3}',
                lightYellow: '${variant.colors.accent2}',
                lightBlue: '${variant.colors.accent5}',
                lightMagenta: '${variant.colors.accent7}',
                lightCyan: '${variant.colors.accent4}',
                lightWhite: '${variant.colors.shade7}',
              },
            });
          };
        `,
      };
    }
  },
  renderInstructions: (paths, colorSet) => {
    const directories = [...new Set(paths.map(dirname))];
    const packageNames = colorSetToVariants(colorSet).map(packageName);
    return source`
      First, copy (or symlink) the outputted package ${
        directories.length > 1 ? 'directories' : 'directory'
      } to the Hyper local plugins directory:

      \`\`\`
      ${directories.map((dir) => `cp -R '${dir}' ~/.hyper_plugins/local/`)}
      \`\`\`

      Then edit \`~/.hyper.js\` and ad the package to the \`localPlugins\` array:

      \`\`\`
      ...
      localPlugins: [
        ${packageNames.map((name) => `'${name}'`).join(' // or ')}
      ],
      ...
      \`\`\`
    `;
  },
};

export default template;
