<?php

/*
 * This file is part of the Thelia package.
 * http://www.thelia.net
 *
 * (c) OpenStudio <info@thelia.net>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Thelia\Condition;

use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Thelia\Condition\Implementation\MatchForTotalAmount;
use Thelia\Model\CurrencyQuery;

/**
 * Unit Test ConditionCollection Class
 * Generated by PHPUnit_SkeletonGenerator 1.2.1 on 2013-11-17 at 18:59:24.
 *
 * @author  Guillaume MOREL <gmorel@openstudio.fr>
 */
class ConditionCollectionTest extends TestCase
{
    /**
     * @var ConditionCollection
     */
    protected $object;

    /**
     * Generate adapter stub.
     *
     * @param int    $cartTotalPrice   Cart total price
     * @param string $checkoutCurrency Checkout currency
     * @param string $i18nOutput       Output from each translation
     *
     * @return MockObject
     */
    public function generateFacadeStub($cartTotalPrice = 400, $checkoutCurrency = 'EUR', $i18nOutput = '')
    {
        $stubFacade = $this->getMockBuilder('\Thelia\Coupon\BaseFacade')
            ->disableOriginalConstructor()
            ->getMock();

        $currencies = CurrencyQuery::create();
        $currencies = $currencies->find();
        $stubFacade->expects($this->any())
            ->method('getAvailableCurrencies')
            ->willReturn($currencies);

        $stubFacade->expects($this->any())
            ->method('getCartTotalPrice')
            ->willReturn($cartTotalPrice);

        $stubFacade->expects($this->any())
            ->method('getCheckoutCurrency')
            ->willReturn($checkoutCurrency);

        $stubFacade->expects($this->any())
            ->method('getConditionEvaluator')
            ->willReturn(new ConditionEvaluator());

        $stubTranslator = $this->getMockBuilder('\Thelia\Core\Translation\Translator')
            ->disableOriginalConstructor()
            ->getMock();
        $stubTranslator->expects($this->any())
            ->method('trans')
            ->willReturn($i18nOutput);

        $stubFacade->expects($this->any())
            ->method('getTranslator')
            ->willReturn($stubTranslator);

        return $stubFacade;
    }

    public function testGetConditions(): void
    {
        $stubFacade = $this->generateFacadeStub();

        $condition1 = new MatchForTotalAmount($stubFacade);
        $operators = [
            MatchForTotalAmount::CART_TOTAL => Operators::EQUAL,
            MatchForTotalAmount::CART_CURRENCY => Operators::EQUAL,
        ];
        $values = [
            MatchForTotalAmount::CART_TOTAL => 400,
            MatchForTotalAmount::CART_CURRENCY => 'EUR', ];
        $condition1->setValidatorsFromForm($operators, $values);

        $collection = new ConditionCollection();
        $collection[] = $condition1;

        $expected = $condition1;
        $actual = $collection[0];

        $this->assertEquals($expected, $actual);

        $this->assertFalse($collection->count() == 0);
    }

    /**
     * @covers \Thelia\Condition\ConditionCollection::count
     */
    public function testIsEmpty(): void
    {
        $collection = new ConditionCollection();
        $this->assertTrue($collection->count() == 0);
    }

    /**
     * @covers \Thelia\Condition\ConditionCollection::__toString
     */
    public function testToString(): void
    {
        $stubFacade = $this->generateFacadeStub();

        $condition1 = new MatchForTotalAmount($stubFacade);
        $operators1 = [
            MatchForTotalAmount::CART_TOTAL => Operators::SUPERIOR,
            MatchForTotalAmount::CART_CURRENCY => Operators::EQUAL,
        ];
        $values1 = [
            MatchForTotalAmount::CART_TOTAL => 400,
            MatchForTotalAmount::CART_CURRENCY => 'EUR', ];
        $condition1->setValidatorsFromForm($operators1, $values1);

        $condition2 = new MatchForTotalAmount($stubFacade);
        $operators2 = [
            MatchForTotalAmount::CART_TOTAL => Operators::INFERIOR,
            MatchForTotalAmount::CART_CURRENCY => Operators::EQUAL,
        ];
        $values2 = [
            MatchForTotalAmount::CART_TOTAL => 600,
            MatchForTotalAmount::CART_CURRENCY => 'EUR', ];
        $condition2->setValidatorsFromForm($operators2, $values2);

        $collection = new ConditionCollection();
        $collection[] = $condition1;
        $collection[] = $condition2;

        $expected = '[{"conditionServiceId":"thelia.condition.match_for_total_amount","operators":{"price":">","currency":"=="},"values":{"price":400,"currency":"EUR"}},{"conditionServiceId":"thelia.condition.match_for_total_amount","operators":{"price":"<","currency":"=="},"values":{"price":600,"currency":"EUR"}}]';
        $actual = $collection->__toString();

        $this->assertEquals($expected, $actual);
    }
}
