<?php

/*
 * This file is part of the Thelia package.
 * http://www.thelia.net
 *
 * (c) OpenStudio <info@thelia.net>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Thelia\Core\Template;

use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Thelia\Core\Event\Cache\CacheEvent;
use Thelia\Core\Event\Config\ConfigUpdateEvent;
use Thelia\Core\Event\TheliaEvents;
use Thelia\Model\ConfigQuery;

class TheliaTemplateHelper implements TemplateHelperInterface, EventSubscriberInterface
{
    protected $kernelCacheDir;

    public function __construct($kernelCacheDir)
    {
        $this->kernelCacheDir = $kernelCacheDir;
    }

    /**
     * @return TemplateDefinition
     */
    public function getActiveMailTemplate()
    {
        return new TemplateDefinition(
            ConfigQuery::read('active-mail-template', 'default'),
            TemplateDefinition::EMAIL
        );
    }

    /**
     * Check if a template definition is the current active template.
     *
     * @return bool true is the given template is the active template
     */
    public function isActive(TemplateDefinition $tplDefinition)
    {
        $tplVar = '';

        switch ($tplDefinition->getType()) {
            case TemplateDefinition::FRONT_OFFICE:
                $tplVar = 'active-front-template';
                break;
            case TemplateDefinition::BACK_OFFICE:
                $tplVar = 'active-admin-template';
                break;
            case TemplateDefinition::PDF:
                $tplVar = 'active-pdf-template';
                break;
            case TemplateDefinition::EMAIL:
                $tplVar = 'active-mail-template';
                break;
        }

        return $tplDefinition->getName() == ConfigQuery::read($tplVar, 'default');
    }

    /**
     * @return TemplateDefinition
     */
    public function getActivePdfTemplate()
    {
        return new TemplateDefinition(
            ConfigQuery::read('active-pdf-template', 'default'),
            TemplateDefinition::PDF
        );
    }

    /**
     * @return TemplateDefinition
     */
    public function getActiveAdminTemplate()
    {
        return new TemplateDefinition(
            ConfigQuery::read('active-admin-template', 'default'),
            TemplateDefinition::BACK_OFFICE
        );
    }

    /**
     * @return TemplateDefinition
     */
    public function getActiveFrontTemplate()
    {
        return new TemplateDefinition(
            ConfigQuery::read('active-front-template', 'default'),
            TemplateDefinition::FRONT_OFFICE
        );
    }

    /**
     * Returns an array which contains all standard template definitions.
     */
    public function getStandardTemplateDefinitions()
    {
        return [
            $this->getActiveFrontTemplate(),
            $this->getActiveAdminTemplate(),
            $this->getActivePdfTemplate(),
            $this->getActiveMailTemplate(),
        ];
    }

    /**
     * Return a list of existing templates for a given template type.
     *
     * @param int    $templateType the template type
     * @param string $base         the template base (module or core, default to core)
     *
     * @return TemplateDefinition[] of \Thelia\Core\Template\TemplateDefinition
     */
    public function getList($templateType, $base = THELIA_TEMPLATE_DIR)
    {
        $list = $exclude = [];

        $tplIterator = TemplateDefinition::getStandardTemplatesSubdirsIterator();

        foreach ($tplIterator as $type => $subdir) {
            if ($templateType == $type) {
                $baseDir = rtrim($base, DS).DS.$subdir;

                try {
                    // Every subdir of the basedir is supposed to be a template.
                    $di = new \DirectoryIterator($baseDir);

                    /** @var \DirectoryIterator $file */
                    foreach ($di as $file) {
                        // Ignore 'dot' elements
                        if ($file->isDot() || !$file->isDir()) {
                            continue;
                        }

                        // Ignore reserved directory names
                        if (\in_array($file->getFilename(), $exclude)) {
                            continue;
                        }

                        $list[] = new TemplateDefinition($file->getFilename(), $templateType);
                    }
                } catch (\UnexpectedValueException $ex) {
                    // Ignore the exception and continue
                }
            }
        }

        return $list;
    }

    /**
     * Clear the cache if the front or admin template is changed in the back-office.
     */
    public function clearCache(ConfigUpdateEvent $event, string $eventName, EventDispatcherInterface $dispatcher): void
    {
        if (
            (null === $config = ConfigQuery::create()->findPk($event->getConfigId()))
            || ($config->getValue() === $event->getValue())
            || ($config->getName() !== TemplateDefinition::BACK_OFFICE_CONFIG_NAME
             && $config->getName() !== TemplateDefinition::FRONT_OFFICE_CONFIG_NAME)
        ) {
            return;
        }

        $cacheEvent = new CacheEvent($this->kernelCacheDir);
        $dispatcher->dispatch($cacheEvent, TheliaEvents::CACHE_CLEAR);
    }

    public static function getSubscribedEvents()
    {
        return [
            TheliaEvents::CONFIG_SETVALUE => ['clearCache', 130],
        ];
    }
}
