---
id: rule-preset
title: Creating Preset
---

"preset" is a collection of rules and rulesConfig.

The basic source code format for a preset is:

```js
module.exports = {
    rules: {
        "no-todo": require("textlint-rule-no-todo").default
    },
    rulesConfig: {
        "no-todo": true
    }
};
```

- `"rules"` is rule creator object.
- `"rulesConfig"` is rule config object for `"rules"`.

## Example
 
e.g.) "textlint-rule-preset-gizmo"

textlint-rule-preset-gizmo includes the following rules:

- ruleA
- ruleB

`textlint-rule-preset-gizmo.js`: 

```js
module.exports = {
    rules: {
        ruleA: require("textlint-rule-A"),
        ruleB: require("textlint-rule-B")
    },
    rulesConfig: {
        ruleA: true,
        ruleB: true
    }
};
```

Usage of "textlint-rule-preset-gizmo":

`.textlintrc.json`

```json
{
    "rules": {
        "preset-gizmo": {
            "ruleA": false
            /* ruleB's options is defined by preset-gizmo */
        }
    }
}
```

**Recommended:** use [@textlint/module-interop](https://www.npmjs.com/package/@textlint/module-interop) for loading external package.

When require external textlint rule package, you should use [@textlint/module-interop](https://www.npmjs.com/package/@textlint/module-interop) instead of `require` function.
Because, some rule package expose `exports.default` instead of `exports`.
That `.default` property is generated by ES2015+ transpiler like Babel, TypeScript.

[@textlint/module-interop](https://www.npmjs.com/package/@textlint/module-interop) help to load both of `exports.default` and `exports`.

```js
const { moduleInterop } = require("@textlint/module-interop");
module.exports = {
    rules: {
        ruleA: moduleInterop(require("textlint-rule-external-A")),
        ruleB: moduleInterop(require("@scope/textlint-rule-external-B"))
    },
    rulesConfig: {
        ruleA: true,
        ruleB: true
    }
};
```
 
Of course, If you use [Babel](https://babeljs.io/) or [TypeScript](https://www.typescriptlang.org/)(with [`esModuleInterop` option](https://www.typescriptlang.org/docs/handbook/compiler-options.html)), you can just use `import` statement.
These transpiler wrap `require` function with interop helper function automatically.

```js
import RuleA from "textlint-rule-external-A";
// It will be transpiled to `const RuleA = __importDefault(require("textlint-rule-external-A"))`
import RuleB from "@scope/textlint-rule-external-B";
// It will be transpiled to `const RuleB = __importDefault(require("textlint-rule-external-B"))`
module.exports = {
    rules: {
        ruleA: RuleA,
        ruleB: RuleB
    },
    rulesConfig: {
        ruleA: true,
        ruleB: true
    }
};
```

## Publishing

If you want to publish your textlint rule preset, see following documents.


### Package Naming Conventions

textlint rule package naming should have `textlint-rule-preset-` prefix.
 
- `textlint-rule-preset-<name>`
- `@scope/textlint-rule-preset-<name>`
    - textlint supports [Scoped packages](https://docs.npmjs.com/misc/scope "Scoped packages")

Example: `textlint-rule-preset-example`

textlint user use it following:

```json
{
    "rules": {
        "preset-example": true
    }
}
```

Example: `@textlint-rule/textlint-rule-preset-google-developer`

textlint user use it following:

```json
{
    "rules": {
        "@textlint-rule/preset-google-developer": true
    }
}
```

### Keywords

You should add `textlintrule` to npm's `keywords`

```json
{
  "name": "textlint-rule-preset-foo-bar",
  "description": "Your preset of rules description",
  "version": "1.0.1",
  "keywords": [
    "textlintrule"
  ]
}
```
