import { dayjs } from '@tachybase/utils';

import { lowerCase } from 'lodash';

/**
 * 为json对象中collections节点的name、title属性添加后缀
 * 为json对象中fields节点的collectionName、through、target性添加后缀
 * @param jsonObj json对象
 * @param appendText 后缀文本
 */
export function appendJsonCollectionName(jsonObj: any, appendText: string) {
  appendCollectionsNode(jsonObj, appendText);
  for (const item of jsonObj['collections'][0]['fields']) {
    appendFieldsNode(item, appendText);
  }
  return jsonObj;
}

function appendCollectionsNode(jsonObj: any, appendText: string) {
  jsonObj['collections'][0]['name'] += appendText;
  jsonObj['collections'][0]['title'] += appendText;
}

function appendFieldsNode(jsonObj: any, appendText: string) {
  for (const key in jsonObj) {
    if (key === 'collectionName' || key === 'through' || key === 'target') {
      jsonObj[key] += appendText;
    }
  }
}

/**
 * builtin_ 为TachyBase内置模板，同TachyBase在页面创建效果一致主要用法3种：
 *  1是测试用例仅需一个无业务字段的表即可验证，
 *  2是作为多对多的中间表，
 *  3是存在先表后关系字段的情况，使用create内置模板表再update表的方式更新数据表来添加全部字段
 * e2e_ 按照不同场景的数据表，可以多个测试用例共用一个或一套数据表
 */
export const builtinGeneral = {
  collections: [
    {
      key: 'hqh4qy21m46',
      name: 'tt_amt_org',
      title: '自动>组织[普通表]',
      inherit: false,
      hidden: false,
      description: null,
      fields: [
        {
          key: 'z3fvals50xp',
          name: 'id',
          type: 'bigInt',
          interface: 'id',
          description: null,
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          autoIncrement: true,
          primaryKey: true,
          allowNull: false,
          uiSchema: {
            type: 'number',
            title: '{{t("ID")}}',
            'x-component': 'InputNumber',
            'x-read-pretty': true,
          },
        },
        {
          key: '3cp1zy4ogle',
          name: 'createdAt',
          type: 'date',
          interface: 'createdAt',
          description: null,
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          field: 'createdAt',
          uiSchema: {
            type: 'datetime',
            title: '{{t("Created at")}}',
            'x-component': 'DatePicker',
            'x-component-props': {},
            'x-read-pretty': true,
          },
        },
        {
          key: '6l38n4nt7nb',
          name: 'createdBy',
          type: 'belongsTo',
          interface: 'createdBy',
          description: null,
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          target: 'users',
          foreignKey: 'createdById',
          uiSchema: {
            type: 'object',
            title: '{{t("Created by")}}',
            'x-component': 'AssociationField',
            'x-component-props': {
              fieldNames: {
                value: 'id',
                label: 'nickname',
              },
            },
            'x-read-pretty': true,
          },
          targetKey: 'id',
        },
        {
          key: '3xds0p7ersr',
          name: 'updatedAt',
          type: 'date',
          interface: 'updatedAt',
          description: null,
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          field: 'updatedAt',
          uiSchema: {
            type: 'string',
            title: '{{t("Last updated at")}}',
            'x-component': 'DatePicker',
            'x-component-props': {},
            'x-read-pretty': true,
          },
        },
        {
          key: 'slqyx974mzc',
          name: 'updatedBy',
          type: 'belongsTo',
          interface: 'updatedBy',
          description: null,
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          target: 'users',
          foreignKey: 'updatedById',
          uiSchema: {
            type: 'object',
            title: '{{t("Last updated by")}}',
            'x-component': 'AssociationField',
            'x-component-props': {
              fieldNames: {
                value: 'id',
                label: 'nickname',
              },
            },
            'x-read-pretty': true,
          },
          targetKey: 'id',
        },
      ],
      category: [],
      logging: true,
      autoGenId: true,
      createdBy: true,
      updatedBy: true,
      createdAt: true,
      updatedAt: true,
      sortable: true,
      template: 'general',
      view: false,
      schema: 'public',
    },
  ],
};

export const builtinCalendar = {
  collections: [
    {
      key: '7lfgblnyr0w',
      name: 'tt_amt_org_cld',
      title: '自动>组织[日历表]',
      inherit: false,
      hidden: false,
      description: null,
      fields: [
        {
          key: 'nq6a40fz0nd',
          name: 'cron',
          type: 'string',
          interface: 'select',
          description: null,
          collectionName: 'tt_amt_org_cld',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'string',
            title: '{{t("Repeats")}}',
            'x-component': 'CronSet',
            'x-component-props': 'allowClear',
            enum: [
              {
                label: '{{t("Daily")}}',
                value: '0 0 0 * * ?',
              },
              {
                label: '{{t("Weekly")}}',
                value: 'every_week',
              },
              {
                label: '{{t("Monthly")}}',
                value: 'every_month',
              },
              {
                label: '{{t("Yearly")}}',
                value: 'every_year',
              },
            ],
          },
        },
        {
          key: 'qtxt638o4nq',
          name: 'exclude',
          type: 'json',
          interface: null,
          description: null,
          collectionName: 'tt_amt_org_cld',
          parentKey: null,
          reverseKey: null,
        },
        {
          key: 'ly88an4fg8f',
          name: 'id',
          type: 'bigInt',
          interface: 'id',
          description: null,
          collectionName: 'tt_amt_org_cld',
          parentKey: null,
          reverseKey: null,
          autoIncrement: true,
          primaryKey: true,
          allowNull: false,
          uiSchema: {
            type: 'number',
            title: '{{t("ID")}}',
            'x-component': 'InputNumber',
            'x-read-pretty': true,
          },
        },
        {
          key: '8fiqtpovg79',
          name: 'createdAt',
          type: 'date',
          interface: 'createdAt',
          description: null,
          collectionName: 'tt_amt_org_cld',
          parentKey: null,
          reverseKey: null,
          field: 'createdAt',
          uiSchema: {
            type: 'datetime',
            title: '{{t("Created at")}}',
            'x-component': 'DatePicker',
            'x-component-props': {},
            'x-read-pretty': true,
          },
        },
        {
          key: 'hup33jdnp9j',
          name: 'createdBy',
          type: 'belongsTo',
          interface: 'createdBy',
          description: null,
          collectionName: 'tt_amt_org_cld',
          parentKey: null,
          reverseKey: null,
          target: 'users',
          foreignKey: 'createdById',
          uiSchema: {
            type: 'object',
            title: '{{t("Created by")}}',
            'x-component': 'AssociationField',
            'x-component-props': {
              fieldNames: {
                value: 'id',
                label: 'nickname',
              },
            },
            'x-read-pretty': true,
          },
          targetKey: 'id',
        },
        {
          key: '4ky6wvfhduq',
          name: 'updatedAt',
          type: 'date',
          interface: 'updatedAt',
          description: null,
          collectionName: 'tt_amt_org_cld',
          parentKey: null,
          reverseKey: null,
          field: 'updatedAt',
          uiSchema: {
            type: 'string',
            title: '{{t("Last updated at")}}',
            'x-component': 'DatePicker',
            'x-component-props': {},
            'x-read-pretty': true,
          },
        },
        {
          key: '1xht59blk26',
          name: 'updatedBy',
          type: 'belongsTo',
          interface: 'updatedBy',
          description: null,
          collectionName: 'tt_amt_org_cld',
          parentKey: null,
          reverseKey: null,
          target: 'users',
          foreignKey: 'updatedById',
          uiSchema: {
            type: 'object',
            title: '{{t("Last updated by")}}',
            'x-component': 'AssociationField',
            'x-component-props': {
              fieldNames: {
                value: 'id',
                label: 'nickname',
              },
            },
            'x-read-pretty': true,
          },
          targetKey: 'id',
        },
      ],
      category: [],
      logging: true,
      template: 'calendar',
      view: false,
      createdBy: true,
      updatedBy: true,
      createdAt: true,
      updatedAt: true,
      sortable: true,
      schema: 'public',
    },
  ],
};

export const builtinTree = {
  collections: [
    {
      key: 'wvgsufr1uxv',
      name: 'tt_amt_org_tree',
      title: '自动>组织[树表]',
      inherit: false,
      hidden: false,
      description: null,
      fields: [
        {
          key: '4ihc87rzn00',
          name: 'parentId',
          type: 'bigInt',
          interface: 'integer',
          description: null,
          collectionName: 'tt_amt_org_tree',
          parentKey: null,
          reverseKey: null,
          isForeignKey: true,
          uiSchema: {
            type: 'number',
            title: '{{t("Parent ID")}}',
            'x-component': 'InputNumber',
            'x-read-pretty': true,
          },
          target: 'tt_amt_org_tree',
        },
        {
          key: 'j58bfihyjdy',
          name: 'parent',
          type: 'belongsTo',
          interface: 'm2o',
          description: null,
          collectionName: 'tt_amt_org_tree',
          parentKey: null,
          reverseKey: null,
          foreignKey: 'parentId',
          treeParent: true,
          onDelete: 'CASCADE',
          uiSchema: {
            title: '{{t("Parent")}}',
            'x-component': 'AssociationField',
            'x-component-props': {
              multiple: false,
              fieldNames: {
                label: 'id',
                value: 'id',
              },
            },
          },
          target: 'tt_amt_org_tree',
          targetKey: 'id',
        },
        {
          key: '6r6b98g0hcz',
          name: 'children',
          type: 'hasMany',
          interface: 'o2m',
          description: null,
          collectionName: 'tt_amt_org_tree',
          parentKey: null,
          reverseKey: null,
          foreignKey: 'parentId',
          treeChildren: true,
          onDelete: 'CASCADE',
          uiSchema: {
            title: '{{t("Children")}}',
            'x-component': 'AssociationField',
            'x-component-props': {
              multiple: true,
              fieldNames: {
                label: 'id',
                value: 'id',
              },
            },
          },
          target: 'tt_amt_org_tree',
          targetKey: 'id',
          sourceKey: 'id',
        },
        {
          key: 'hju1ggqps4n',
          name: 'id',
          type: 'bigInt',
          interface: 'id',
          description: null,
          collectionName: 'tt_amt_org_tree',
          parentKey: null,
          reverseKey: null,
          autoIncrement: true,
          primaryKey: true,
          allowNull: false,
          uiSchema: {
            type: 'number',
            title: '{{t("ID")}}',
            'x-component': 'InputNumber',
            'x-read-pretty': true,
          },
        },
        {
          key: '1y7rpmh8muj',
          name: 'createdAt',
          type: 'date',
          interface: 'createdAt',
          description: null,
          collectionName: 'tt_amt_org_tree',
          parentKey: null,
          reverseKey: null,
          field: 'createdAt',
          uiSchema: {
            type: 'datetime',
            title: '{{t("Created at")}}',
            'x-component': 'DatePicker',
            'x-component-props': {},
            'x-read-pretty': true,
          },
        },
        {
          key: 'ieknujjdkv9',
          name: 'createdBy',
          type: 'belongsTo',
          interface: 'createdBy',
          description: null,
          collectionName: 'tt_amt_org_tree',
          parentKey: null,
          reverseKey: null,
          target: 'users',
          foreignKey: 'createdById',
          uiSchema: {
            type: 'object',
            title: '{{t("Created by")}}',
            'x-component': 'AssociationField',
            'x-component-props': {
              fieldNames: {
                value: 'id',
                label: 'nickname',
              },
            },
            'x-read-pretty': true,
          },
          targetKey: 'id',
        },
        {
          key: 'cv35rzoqna6',
          name: 'updatedAt',
          type: 'date',
          interface: 'updatedAt',
          description: null,
          collectionName: 'tt_amt_org_tree',
          parentKey: null,
          reverseKey: null,
          field: 'updatedAt',
          uiSchema: {
            type: 'string',
            title: '{{t("Last updated at")}}',
            'x-component': 'DatePicker',
            'x-component-props': {},
            'x-read-pretty': true,
          },
        },
        {
          key: 'z21wfpry7uh',
          name: 'updatedBy',
          type: 'belongsTo',
          interface: 'updatedBy',
          description: null,
          collectionName: 'tt_amt_org_tree',
          parentKey: null,
          reverseKey: null,
          target: 'users',
          foreignKey: 'updatedById',
          uiSchema: {
            type: 'object',
            title: '{{t("Last updated by")}}',
            'x-component': 'AssociationField',
            'x-component-props': {
              fieldNames: {
                value: 'id',
                label: 'nickname',
              },
            },
            'x-read-pretty': true,
          },
          targetKey: 'id',
        },
      ],
      category: [],
      logging: true,
      autoGenId: true,
      createdBy: true,
      updatedBy: true,
      createdAt: true,
      updatedAt: true,
      sortable: true,
      template: 'tree',
      view: false,
      tree: 'adjacencyList',
      schema: 'public',
    },
  ],
};

export const builtinFile = {
  collections: [
    {
      key: '6hzed4cvpxo',
      name: 'tt_amt_file',
      title: '自动>档案[文件表]',
      inherit: false,
      hidden: false,
      description: null,
      fields: [
        {
          key: 'qo0w8dcl45u',
          name: 'title',
          type: 'string',
          interface: 'input',
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          deletable: false,
          uiSchema: {
            type: 'string',
            title: '{{t("Title")}}',
            'x-component': 'Input',
          },
        },
        {
          key: 'ayzc9b3aee5',
          name: 'filename',
          type: 'string',
          interface: 'input',
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          deletable: false,
          uiSchema: {
            type: 'string',
            title: '{{t("File name", { ns: "file-manager" })}}',
            'x-component': 'Input',
            'x-read-pretty': true,
          },
        },
        {
          key: 'ftn60zv1rfr',
          name: 'extname',
          type: 'string',
          interface: 'input',
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          deletable: false,
          uiSchema: {
            type: 'string',
            title: '{{t("Extension name", { ns: "file-manager" })}}',
            'x-component': 'Input',
            'x-read-pretty': true,
          },
        },
        {
          key: '59oj6m0r8be',
          name: 'size',
          type: 'integer',
          interface: 'integer',
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          deletable: false,
          uiSchema: {
            type: 'number',
            title: '{{t("Size", { ns: "file-manager" })}}',
            'x-component': 'InputNumber',
            'x-read-pretty': true,
            'x-component-props': {
              stringMode: true,
              step: '0',
            },
          },
        },
        {
          key: 'burt1tfbcjx',
          name: 'mimetype',
          type: 'string',
          interface: 'input',
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          deletable: false,
          uiSchema: {
            type: 'string',
            title: '{{t("Mime type", { ns: "file-manager" })}}',
            'x-component': 'Input',
            'x-read-pretty': true,
          },
        },
        {
          key: '1zfhbqod0i1',
          name: 'path',
          type: 'string',
          interface: 'input',
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          deletable: false,
          uiSchema: {
            type: 'string',
            title: '{{t("Path")}}',
            'x-component': 'Input',
            'x-read-pretty': true,
          },
        },
        {
          key: 'b0bmsmrfqqw',
          name: 'url',
          type: 'string',
          interface: 'input',
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          deletable: false,
          uiSchema: {
            type: 'string',
            title: '{{t("URL")}}',
            'x-component': 'Input.URL',
            'x-read-pretty': true,
          },
        },
        {
          key: 'indgsp9sqqq',
          name: 'preview',
          type: 'string',
          interface: 'url',
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          field: 'url',
          deletable: false,
          uiSchema: {
            type: 'string',
            title: '{{t("Preview")}}',
            'x-component': 'Preview',
            'x-read-pretty': true,
          },
        },
        {
          key: '2p8d1h46x9l',
          name: 'storage',
          type: 'belongsTo',
          interface: null,
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          comment: '存储引擎',
          target: 'storages',
          foreignKey: 'storageId',
          deletable: false,
          targetKey: 'id',
        },
        {
          key: 'im0yegvb5jt',
          name: 'meta',
          type: 'jsonb',
          interface: null,
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          deletable: false,
          defaultValue: {},
        },
        {
          key: 'tjr9nr5thoh',
          name: 'id',
          type: 'bigInt',
          interface: 'id',
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          autoIncrement: true,
          primaryKey: true,
          allowNull: false,
          uiSchema: {
            type: 'number',
            title: '{{t("ID")}}',
            'x-component': 'InputNumber',
            'x-read-pretty': true,
          },
        },
        {
          key: 's48vjiqbjcq',
          name: 'createdAt',
          type: 'date',
          interface: 'createdAt',
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          field: 'createdAt',
          uiSchema: {
            type: 'datetime',
            title: '{{t("Created at")}}',
            'x-component': 'DatePicker',
            'x-component-props': {},
            'x-read-pretty': true,
          },
        },
        {
          key: 'y3t7jii4m8w',
          name: 'createdBy',
          type: 'belongsTo',
          interface: 'createdBy',
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          target: 'users',
          foreignKey: 'createdById',
          uiSchema: {
            type: 'object',
            title: '{{t("Created by")}}',
            'x-component': 'RecordPicker',
            'x-component-props': {
              fieldNames: {
                value: 'id',
                label: 'nickname',
              },
            },
            'x-read-pretty': true,
          },
          targetKey: 'id',
        },
        {
          key: 'z5gvejdbfyb',
          name: 'updatedAt',
          type: 'date',
          interface: 'updatedAt',
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          field: 'updatedAt',
          uiSchema: {
            type: 'string',
            title: '{{t("Last updated at")}}',
            'x-component': 'DatePicker',
            'x-component-props': {},
            'x-read-pretty': true,
          },
        },
        {
          key: 'qgd8zko7b0z',
          name: 'updatedBy',
          type: 'belongsTo',
          interface: 'updatedBy',
          description: null,
          collectionName: 'tt_amt_file',
          parentKey: null,
          reverseKey: null,
          target: 'users',
          foreignKey: 'updatedById',
          uiSchema: {
            type: 'object',
            title: '{{t("Last updated by")}}',
            'x-component': 'RecordPicker',
            'x-component-props': {
              fieldNames: {
                value: 'id',
                label: 'nickname',
              },
            },
            'x-read-pretty': true,
          },
          targetKey: 'id',
        },
      ],
      category: [],
      logging: true,
      template: 'file',
      view: false,
      createdBy: true,
      updatedBy: true,
      storage: 'local',
      schema: 'public',
    },
  ],
};

export const builtinExpression = {
  collections: [
    {
      key: 'w77iro3197g',
      name: 'tt_amt_exp',
      title: '自动>表达式[表达式表]',
      inherit: false,
      hidden: false,
      description: null,
      fields: [
        {
          key: 'slc2i7yz2ck',
          name: 'engine',
          type: 'string',
          interface: 'radioGroup',
          description: null,
          collectionName: 'tt_amt_exp',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'string',
            title: '{{t("Calculation engine")}}',
            'x-component': 'Radio.Group',
            enum: [
              {
                value: 'math.js',
                label: 'Math.js',
                tooltip:
                  "{{t('Math.js comes with a large set of built-in functions and constants, and offers an integrated solution to work with different data types')}}",
                link: 'https://mathjs.org/',
              },
              {
                value: 'formula.js',
                label: 'Formula.js',
                tooltip: '{{t("Formula.js supports most Microsoft Excel formula functions.")}}',
                link: 'https://formulajs.info/functions/',
              },
            ],
            default: 'formula.js',
          },
        },
        {
          key: 'mlvkdcg2yvq',
          name: 'sourceCollection',
          type: 'string',
          interface: 'select',
          description: null,
          collectionName: 'tt_amt_exp',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'string',
            title: '{{t("Collection")}}',
            'x-component': 'CollectionSelect',
            'x-component-props': {},
          },
        },
        {
          key: 'vdzigakt9ne',
          name: 'expression',
          type: 'text',
          interface: 'expression',
          description: null,
          collectionName: 'tt_amt_exp',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'string',
            title: '{{t("Expression")}}',
            'x-component': 'DynamicExpression',
          },
        },
        {
          key: 'fx72pqt0fjv',
          name: 'id',
          type: 'bigInt',
          interface: 'id',
          description: null,
          collectionName: 'tt_amt_exp',
          parentKey: null,
          reverseKey: null,
          autoIncrement: true,
          primaryKey: true,
          allowNull: false,
          uiSchema: {
            type: 'number',
            title: '{{t("ID")}}',
            'x-component': 'InputNumber',
            'x-read-pretty': true,
          },
        },
        {
          key: 'owiio66cw1u',
          name: 'createdAt',
          type: 'date',
          interface: 'createdAt',
          description: null,
          collectionName: 'tt_amt_exp',
          parentKey: null,
          reverseKey: null,
          field: 'createdAt',
          uiSchema: {
            type: 'datetime',
            title: '{{t("Created at")}}',
            'x-component': 'DatePicker',
            'x-component-props': {},
            'x-read-pretty': true,
          },
        },
        {
          key: '9fu8kmc69dj',
          name: 'createdBy',
          type: 'belongsTo',
          interface: 'createdBy',
          description: null,
          collectionName: 'tt_amt_exp',
          parentKey: null,
          reverseKey: null,
          target: 'users',
          foreignKey: 'createdById',
          uiSchema: {
            type: 'object',
            title: '{{t("Created by")}}',
            'x-component': 'RecordPicker',
            'x-component-props': {
              fieldNames: {
                value: 'id',
                label: 'nickname',
              },
            },
            'x-read-pretty': true,
          },
          targetKey: 'id',
        },
        {
          key: 'uie8q16z2jj',
          name: 'updatedAt',
          type: 'date',
          interface: 'updatedAt',
          description: null,
          collectionName: 'tt_amt_exp',
          parentKey: null,
          reverseKey: null,
          field: 'updatedAt',
          uiSchema: {
            type: 'string',
            title: '{{t("Last updated at")}}',
            'x-component': 'DatePicker',
            'x-component-props': {},
            'x-read-pretty': true,
          },
        },
        {
          key: '6jzyuc1jzro',
          name: 'updatedBy',
          type: 'belongsTo',
          interface: 'updatedBy',
          description: null,
          collectionName: 'tt_amt_exp',
          parentKey: null,
          reverseKey: null,
          target: 'users',
          foreignKey: 'updatedById',
          uiSchema: {
            type: 'object',
            title: '{{t("Last updated by")}}',
            'x-component': 'RecordPicker',
            'x-component-props': {
              fieldNames: {
                value: 'id',
                label: 'nickname',
              },
            },
            'x-read-pretty': true,
          },
          targetKey: 'id',
        },
      ],
      category: [],
      logging: true,
      template: 'expression',
      view: false,
      createdBy: true,
      updatedBy: true,
      createdAt: true,
      updatedAt: true,
      sortable: true,
      schema: 'public',
    },
  ],
};

//普通表单表场景，无关系字段情况
export const generalWithNoRelationalFields = {
  collections: [
    {
      key: '86lpobjxamt',
      name: 'tt_amt_org',
      title: '自动>组织[普通表]',
      inherit: false,
      hidden: false,
      fields: [
        {
          key: 'ohxa8xp9amt',
          name: 'id',
          type: 'bigInt',
          interface: 'id',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          autoIncrement: true,
          primaryKey: true,
          allowNull: false,
          uiSchema: {
            type: 'number',
            title: '{{t("ID")}}',
            'x-component': 'InputNumber',
            'x-read-pretty': true,
          },
        },
        {
          key: 'qjit9pcvamt',
          name: 'createdAt',
          type: 'date',
          interface: 'createdAt',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          field: 'createdAt',
          uiSchema: {
            type: 'datetime',
            title: '{{t("Created at")}}',
            'x-component': 'DatePicker',
            'x-component-props': {},
            'x-read-pretty': true,
          },
        },
        {
          key: 'vijs3wcnamt',
          name: 'createdBy',
          type: 'belongsTo',
          interface: 'createdBy',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          target: 'users',
          foreignKey: 'createdById',
          uiSchema: {
            type: 'object',
            title: '{{t("Created by")}}',
            'x-component': 'AssociationField',
            'x-component-props': {
              fieldNames: {
                value: 'id',
                label: 'nickname',
              },
            },
            'x-read-pretty': true,
          },
          targetKey: 'id',
        },
        {
          key: 'zsebjtwiamt',
          name: 'updatedAt',
          type: 'date',
          interface: 'updatedAt',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          field: 'updatedAt',
          uiSchema: {
            type: 'string',
            title: '{{t("Last updated at")}}',
            'x-component': 'DatePicker',
            'x-component-props': {},
            'x-read-pretty': true,
          },
        },
        {
          key: 'fa4c74phamt',
          name: 'updatedBy',
          type: 'belongsTo',
          interface: 'updatedBy',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          target: 'users',
          foreignKey: 'updatedById',
          uiSchema: {
            type: 'object',
            title: '{{t("Last updated by")}}',
            'x-component': 'AssociationField',
            'x-component-props': {
              fieldNames: {
                value: 'id',
                label: 'nickname',
              },
            },
            'x-read-pretty': true,
          },
          targetKey: 'id',
        },
        {
          key: 'mv6h7f8lamt',
          name: 'orgcode',
          type: 'sequence',
          interface: 'sequence',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          patterns: [
            {
              type: 'string',
              options: {
                value: '区域编码',
              },
            },
            {
              type: 'integer',
              options: {
                digits: 9,
                start: 1,
                key: 57450,
              },
            },
          ],
          uiSchema: {
            type: 'string',
            'x-component': 'Input',
            'x-component-props': {},
            title: '公司编号(自动编码)',
          },
          inputable: false,
        },
        {
          key: '44u5zoyyamt',
          name: 'orgname',
          type: 'string',
          interface: 'input',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'string',
            'x-component': 'Input',
            title: '公司名称(单行文本)',
          },
        },
        {
          key: 'tdq82bb0amt',
          name: 'address',
          type: 'text',
          interface: 'textarea',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'string',
            'x-component': 'Input.TextArea',
            title: '公司地址(多行文本)',
          },
        },
        {
          key: '10xi6i24amt',
          name: 'phone',
          type: 'string',
          interface: 'phone',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'string',
            'x-component': 'Input',
            'x-component-props': {
              type: 'tel',
            },
            title: '负责人电话(手机号码)',
          },
        },
        {
          key: 'n3ykqhe4amt',
          name: 'email',
          type: 'string',
          interface: 'email',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'string',
            'x-component': 'Input',
            'x-validator': 'email',
            title: '电子邮箱(电子邮箱)',
          },
        },
        {
          key: '5ddcgbpqamt',
          name: 'staffnum',
          type: 'bigInt',
          interface: 'integer',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'number',
            'x-component': 'InputNumber',
            'x-component-props': {
              stringMode: true,
              step: '1',
            },
            'x-validator': 'integer',
            title: '员工人数(整数)',
          },
        },
        {
          key: 'z59sf4ilamt',
          name: 'insurednum',
          type: 'bigInt',
          interface: 'integer',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'number',
            'x-component': 'InputNumber',
            'x-component-props': {
              stringMode: true,
              step: '1',
            },
            'x-validator': 'integer',
            title: '参保人数(整数)',
          },
        },
        {
          key: 'ecvb3a8xamt',
          name: 'regcapital',
          type: 'double',
          interface: 'number',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            'x-component-props': {
              step: '0.0001',
              stringMode: true,
            },
            type: 'number',
            'x-component': 'InputNumber',
            title: '注册资本(数字)',
          },
        },
        {
          key: 'ierg1fbvamt',
          name: 'paidcapital',
          type: 'double',
          interface: 'number',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            'x-component-props': {
              step: '0.0001',
              stringMode: true,
            },
            type: 'number',
            'x-component': 'InputNumber',
            title: '实缴资本(数字)',
          },
        },
        {
          key: '46kjnywaamt',
          name: 'insuranceratio',
          type: 'float',
          interface: 'percent',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            'x-component-props': {
              step: '0.01',
              stringMode: true,
              addonAfter: '%',
            },
            type: 'string',
            'x-component': 'Percent',
            title: '参保占比(百分比)',
          },
        },
        {
          key: '8j2vkys9amt',
          name: 'isenable',
          type: 'boolean',
          interface: 'checkbox',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'boolean',
            'x-component': 'Checkbox',
            title: '是否启用(勾选)',
          },
        },
        {
          key: '6f480de5amt',
          name: 'status_singleselect',
          type: 'string',
          interface: 'select',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            enum: [
              {
                value: '1',
                label: '存续',
              },
              {
                value: '2',
                label: '在业',
              },
              {
                value: '3',
                label: '吊销',
              },
              {
                value: '4',
                label: '注销',
              },
              {
                value: '5',
                label: '迁入',
              },
              {
                value: '6',
                label: '迁出',
              },
              {
                value: '7',
                label: '停业',
              },
              {
                value: '8',
                label: '清算',
              },
            ],
            type: 'string',
            'x-component': 'Select',
            title: '公司状态(下拉单选)',
          },
        },
        {
          key: 'fcx8cw1ramt',
          name: 'range_multipleselect',
          type: 'array',
          interface: 'multipleSelect',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            enum: [
              {
                value: 'F3134',
                label: '软件销售',
              },
              {
                value: 'I3006',
                label: '软件开发',
              },
              {
                value: 'I3007',
                label: '人工智能基础软件开发',
              },
              {
                value: 'I3008',
                label: '人工智能应用软件开发',
              },
              {
                value: 'I3010',
                label: '软件外包服务',
              },
              {
                value: 'I3011',
                label: '网络与信息安全软件开发',
              },
              {
                value: 'I3012',
                label: '人工智能理论与算法软件开发',
              },
              {
                value: 'I3014',
                label: '数字文化创意软件开发',
              },
              {
                value: 'I3027',
                label: '信息技术咨询服务',
              },
              {
                value: 'I3032',
                label: '数据处理服务',
              },
              {
                value: 'I3034',
                label: '计算机系统服务',
              },
              {
                value: 'L2032',
                label: '信息咨询服务（不含许可类信息咨询服务）',
              },
              {
                value: 'L2095',
                label: '企业管理咨询',
              },
              {
                value: 'M2070',
                label: '技术服务、技术开发、技术咨询、技术交流、技术转让、技术推广',
              },
              {
                value: 'O3010',
                label: '咨询策划服务',
              },
              {
                value: 'P1029',
                label: '业务培训（不含教育培训、职业技能培训等需取得许可的培训）',
              },
            ],
            type: 'array',
            'x-component': 'Select',
            'x-component-props': {
              mode: 'multiple',
            },
            title: '经营范围(下拉多选)',
          },
          defaultValue: [],
        },
        {
          key: 'z71al7aaamt',
          name: 'status_radio',
          type: 'string',
          interface: 'radioGroup',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            enum: [
              {
                value: '1',
                label: '存续',
              },
              {
                value: '2',
                label: '在业',
              },
              {
                value: '3',
                label: '吊销',
              },
              {
                value: '4',
                label: '注销',
              },
              {
                value: '5',
                label: '迁入',
              },
              {
                value: '6',
                label: '迁出',
              },
              {
                value: '7',
                label: '停业',
              },
              {
                value: '8',
                label: '清算',
              },
            ],
            type: 'string',
            'x-component': 'Radio.Group',
            title: '公司状态(单选)',
          },
        },
        {
          key: 'x8v9qf95amt',
          name: 'range_check',
          type: 'array',
          interface: 'checkboxGroup',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            enum: [
              {
                value: 'F3134',
                label: '软件销售',
              },
              {
                value: 'I3006',
                label: '软件开发',
              },
              {
                value: 'I3007',
                label: '人工智能基础软件开发',
              },
              {
                value: 'I3008',
                label: '人工智能应用软件开发',
              },
              {
                value: 'I3010',
                label: '软件外包服务',
              },
              {
                value: 'I3011',
                label: '网络与信息安全软件开发',
              },
              {
                value: 'I3012',
                label: '人工智能理论与算法软件开发',
              },
              {
                value: 'I3014',
                label: '数字文化创意软件开发',
              },
              {
                value: 'I3027',
                label: '信息技术咨询服务',
              },
              {
                value: 'I3032',
                label: '数据处理服务',
              },
              {
                value: 'I3034',
                label: '计算机系统服务',
              },
              {
                value: 'L2032',
                label: '信息咨询服务（不含许可类信息咨询服务）',
              },
              {
                value: 'L2095',
                label: '企业管理咨询',
              },
              {
                value: 'M2070',
                label: '技术服务、技术开发、技术咨询、技术交流、技术转让、技术推广',
              },
              {
                value: 'O3010',
                label: '咨询策划服务',
              },
              {
                value: 'P1029',
                label: '业务培训（不含教育培训、职业技能培训等需取得许可的培训）',
              },
            ],
            type: 'string',
            'x-component': 'Checkbox.Group',
            title: '经营范围(复选)',
          },
          defaultValue: [],
        },
        {
          key: 'c50l1zguamt',
          name: 'range_markdown',
          type: 'text',
          interface: 'markdown',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'string',
            'x-component': 'Markdown',
            title: '经营范围(Markdown)',
          },
        },
        {
          key: 'msj1qa4damt',
          name: 'range_richtext',
          type: 'text',
          interface: 'richText',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'string',
            'x-component': 'RichText',
            title: '经营范围(富文本)',
          },
        },
        {
          key: 'q755hkxmamt',
          name: 'establishdate',
          type: 'date',
          interface: 'datetime',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            'x-component-props': {
              dateFormat: 'YYYY-MM-DD',
              gmt: false,
              showTime: false,
            },
            type: 'string',
            'x-component': 'DatePicker',
            title: '成立日期(日期)',
          },
        },
        {
          key: '9kn4ujeramt',
          name: 'range_json',
          type: 'json',
          interface: 'json',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          defaultValue: null,
          uiSchema: {
            type: 'object',
            'x-component': 'Input.JSON',
            'x-component-props': {
              autoSize: {
                minRows: 5,
              },
            },
            default: null,
            title: '经营范围(JSON)',
          },
        },
        {
          key: 'vqcsj7htamt',
          name: 'url',
          type: 'string',
          interface: 'url',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'string',
            title: '官网地址(URL)',
            'x-component': 'Input.URL',
          },
        },
        {
          key: '0naiw2nq4j0',
          name: 'insuranceratio_formula',
          type: 'formula',
          interface: 'formula',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          dataType: 'double',
          uiSchema: {
            'x-component-props': {
              step: '0.01',
              stringMode: true,
            },
            type: 'number',
            'x-component': 'Formula.Result',
            'x-read-pretty': true,
            title: '参保占比(公式)',
          },
          engine: 'math.js',
          expression: '100*{{insurednum}}/{{staffnum}}',
        },
        {
          key: 'fd9sqp8usoj',
          name: 'sharesnum',
          type: 'bigInt',
          interface: 'integer',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'number',
            'x-component': 'InputNumber',
            'x-component-props': {
              stringMode: true,
              step: '1',
            },
            'x-validator': 'integer',
            title: '股票数量(整数)',
          },
        },
        {
          key: 'fd9sqp8uamt',
          name: 'testdataid',
          type: 'bigInt',
          interface: 'integer',
          collectionName: 'tt_amt_org',
          parentKey: null,
          reverseKey: null,
          uiSchema: {
            type: 'number',
            'x-component': 'InputNumber',
            'x-component-props': {
              stringMode: true,
              step: '1',
            },
            'x-validator': 'integer',
            title: '测试数据id',
          },
        },
      ],
      category: [],
      logging: true,
      autoGenId: true,
      createdBy: true,
      updatedBy: true,
      createdAt: true,
      updatedAt: true,
      sortable: true,
      template: 'general',
    },
  ],
};

export const generalWithNoRelationalFieldsData = [
  {
    createdAt: '2023-07-16T21:30:58.000Z',
    updatedAt: '2023-06-20T03:32:01.000Z',
    createdById: 0,
    updatedById: 3,
    testdataid: 86,
    orgcode: '350086',
    orgname: 'Black Logistic LLC',
    address: '闵行区宾川路314号',
    phone: '21-040-5877',
    email: 'cwlam@mail.com',
    url: 'http://www.osara.co.jp/Food',
    staffnum: 25,
    regcapital: 85.8799,
    paidcapital: 297.1762,
    insuranceratio: 29.93,
    isenable: true,
    status_singleselect: '1',
    range_multipleselect: ['F3134', 'I3007', 'I3011', 'I3012', 'I3014', 'I3027'],
    status_radio: '4',
    range_check: ['I3010', 'I3011', 'I3014', 'I3027', 'L2095', 'M2070', 'O3010'],
    establishdate: '2022-03-17T19:44:50.000Z',
  },
  {
    createdAt: '2023-07-13T13:20:10.000Z',
    updatedAt: '2023-07-10T02:25:11.737Z',
    createdById: 4,
    updatedById: null,
    testdataid: 41,
    orgcode: '230041',
    orgname: '孙記有限责任公司',
    address: '770 Shennan E Rd, Cai Wu Wei, Luohu District',
    phone: '755-4939-4147',
    email: 'kars16@icloud.com',
    url: 'http://www.lamhy.org/HouseholdKitchenAppliances',
    staffnum: 67,
    regcapital: 14.9121,
    paidcapital: 215.5018,
    insuranceratio: 42.26,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: [
      'F3134',
      'I3006',
      'I3007',
      'I3008',
      'I3012',
      'I3014',
      'I3027',
      'I3034',
      'L2095',
      'M2070',
      'P1029',
    ],
    status_radio: '1',
    range_check: ['F3134', 'I3010', 'I3032', 'I3034', 'O3010', 'P1029'],
    establishdate: '2022-08-01T01:51:06.000Z',
  },
  {
    createdAt: '2023-03-14T23:05:00.000Z',
    updatedAt: '2023-07-10T02:25:11.729Z',
    createdById: null,
    updatedById: 4,
    testdataid: 27,
    orgcode: '340027',
    orgname: 'Lisa Telecommunication Inc.',
    address: '43 Zhongshan 5th Rd, Zimaling Shangquan',
    phone: '760-7344-2889',
    email: 'onnaku@gmail.com',
    url: 'https://video.ziyihe404.net/CenturionGardenOutdoor',
    staffnum: 104,
    regcapital: 237.9354,
    paidcapital: 131.6008,
    insuranceratio: 4.22,
    isenable: true,
    status_singleselect: '2',
    range_multipleselect: ['F3134', 'I3032', 'I3034', 'L2032', 'M2070', 'O3010'],
    status_radio: '6',
    range_check: ['F3134', 'I3006', 'I3008', 'I3012', 'M2070', 'O3010'],
    establishdate: '2021-02-19T00:02:48.000Z',
  },
  {
    createdAt: '2023-08-20T18:07:24.000Z',
    updatedAt: '2023-07-10T02:25:11.741Z',
    createdById: 5,
    updatedById: null,
    testdataid: 77,
    orgcode: '450077',
    orgname: '子异有限责任公司',
    address: '619 4th Section  Renmin South Road, Jinjiang District',
    phone: '174-6088-1418',
    email: 'taoshihan1@hotmail.com',
    url: 'https://www.litc908.jp/ArtsHandicraftsSewing',
    staffnum: 37,
    regcapital: 237.5694,
    paidcapital: 94.707,
    insuranceratio: null,
    isenable: true,
    status_singleselect: '2',
    range_multipleselect: ['F3134', 'I3008', 'I3011', 'I3014'],
    status_radio: '5',
    range_check: ['I3006', 'I3007', 'I3008', 'I3010', 'I3034', 'L2032', 'L2095', 'M2070', 'O3010'],
    establishdate: '2023-10-07T23:30:43.000Z',
  },
  {
    createdAt: '2023-07-11T07:34:49.000Z',
    updatedAt: '2023-07-10T02:25:11.736Z',
    createdById: 5,
    updatedById: null,
    testdataid: 25,
    orgcode: '550025',
    orgname: '赵記发展贸易有限责任公司',
    address: '607 Sanlitun Road, Chaoyang District',
    phone: '144-4751-3753',
    email: null,
    url: 'https://video.kakeungw.net/VideoGames',
    staffnum: 64,
    regcapital: 78.8595,
    paidcapital: 71.565,
    insuranceratio: 58.15,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: ['F3134', 'I3006', 'I3011', 'I3027', 'I3032', 'L2032', 'L2095'],
    status_radio: '6',
    range_check: ['I3008', 'I3027', 'I3034', 'L2032', 'O3010', 'P1029'],
    establishdate: null,
  },
  {
    createdAt: '2023-04-28T09:13:50.000Z',
    updatedAt: '2023-05-29T08:08:24.000Z',
    createdById: 6,
    updatedById: 5,
    testdataid: 66,
    orgcode: '440066',
    orgname: '吴技术有限责任公司',
    address: '793 Hongqiao Rd., Xu Hui District',
    phone: '21-0094-0906',
    email: 'yyut@icloud.com',
    url: 'https://video.yeowtszhin.us/Appliances',
    staffnum: 45,
    regcapital: null,
    paidcapital: 76.1281,
    insuranceratio: 76.17,
    isenable: true,
    status_singleselect: '2',
    range_multipleselect: ['F3134', 'I3008', 'I3011', 'I3014', 'I3027', 'I3034', 'L2032', 'O3010'],
    status_radio: '4',
    range_check: ['F3134', 'I3011', 'I3014', 'I3027', 'I3032', 'I3034', 'L2095', 'O3010', 'P1029'],
    establishdate: '2023-09-08T01:51:09.000Z',
  },
  {
    createdAt: '2023-03-20T17:13:24.000Z',
    updatedAt: '2023-05-16T04:16:34.000Z',
    createdById: 1,
    updatedById: 6,
    testdataid: 62,
    orgcode: '510062',
    orgname: 'Salazar Food LLC',
    address: '716 Huaxia St, Jinghua Shangquan',
    phone: '198-4847-7122',
    email: 'auwingfat1219@gmail.com',
    url: 'https://www.xielu.us/CellPhonesAccessories',
    staffnum: 131,
    regcapital: 273.1562,
    paidcapital: 287.3437,
    insuranceratio: 76.86,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: [
      'I3006',
      'I3007',
      'I3008',
      'I3010',
      'I3011',
      'I3012',
      'I3032',
      'L2032',
      'L2095',
      'O3010',
      'P1029',
    ],
    status_radio: '5',
    range_check: [
      'F3134',
      'I3006',
      'I3007',
      'I3011',
      'I3014',
      'I3027',
      'I3034',
      'L2032',
      'L2095',
      'M2070',
      'O3010',
      'P1029',
    ],
    establishdate: '2021-12-14T18:38:12.000Z',
  },
  {
    createdAt: '2023-05-10T18:09:11.000Z',
    updatedAt: '2023-08-13T01:37:53.000Z',
    createdById: 9,
    updatedById: 1,
    testdataid: 20,
    orgcode: '620020',
    orgname: '岚有限责任公司',
    address: '8 Tremont Road',
    phone: '614-635-9277',
    email: 'salazar1985@gmail.com',
    url: 'https://www.gordemily.com/FilmSupplies',
    staffnum: null,
    regcapital: 219.7301,
    paidcapital: 63.6256,
    insuranceratio: 45.43,
    isenable: null,
    status_singleselect: '8',
    range_multipleselect: ['I3006', 'I3007', 'I3008', 'I3012', 'I3014', 'I3027', 'I3034', 'L2032', 'P1029'],
    status_radio: '8',
    range_check: ['F3134', 'I3006', 'I3011', 'I3014', 'I3027', 'I3032', 'L2095', 'M2070'],
    establishdate: '2022-07-05T10:21:51.000Z',
  },
  {
    createdAt: '2023-05-05T06:12:02.000Z',
    updatedAt: '2023-05-04T19:41:58.000Z',
    createdById: 3,
    updatedById: 6,
    testdataid: 31,
    orgcode: '430031',
    orgname: 'Hughes LLC',
    address: '980 4th Section  Renmin South Road, Jinjiang District',
    phone: '147-7853-0142',
    email: 'scao@mail.com',
    url: 'http://image.ikedayu1224.com/Baby',
    staffnum: 135,
    regcapital: 139.2834,
    paidcapital: 173.014,
    insuranceratio: 62.83,
    isenable: true,
    status_singleselect: '2',
    range_multipleselect: ['F3134', 'I3006', 'I3014', 'I3032', 'I3034', 'P1029'],
    status_radio: '2',
    range_check: ['F3134', 'I3006', 'I3008', 'I3010', 'I3034', 'M2070', 'O3010', 'P1029'],
    establishdate: '2023-11-10T02:30:06.000Z',
  },
  {
    createdAt: '2023-05-03T05:56:11.000Z',
    updatedAt: '2023-07-29T02:33:20.000Z',
    createdById: 8,
    updatedById: 4,
    testdataid: 55,
    orgcode: '310055',
    orgname: '睿有限责任公司',
    address: '75 Huaxia St, Jinghua Shangquan',
    phone: '760-789-1994',
    email: 'skudo73@outlook.com',
    url: 'http://image.kenta1.com/ToolsHomeDecoration',
    staffnum: 101,
    regcapital: 168.9348,
    paidcapital: 242.9576,
    insuranceratio: 28.11,
    isenable: true,
    status_singleselect: '8',
    range_multipleselect: ['I3006', 'I3008', 'I3014', 'L2095', 'M2070'],
    status_radio: '8',
    range_check: ['F3134', 'I3006', 'I3007', 'I3008', 'I3012', 'I3014', 'I3034', 'L2095', 'M2070', 'O3010'],
    establishdate: '2021-12-29T21:52:28.000Z',
  },
  {
    createdAt: '2023-08-14T14:48:46.000Z',
    updatedAt: '2023-08-08T22:23:16.000Z',
    createdById: 5,
    updatedById: 9,
    testdataid: 84,
    orgcode: '530084',
    orgname: '璐有限责任公司',
    address: '32 Jiangnan West Road, Haizhu District',
    phone: '20-611-5899',
    email: 'kffan@gmail.com',
    url: 'http://auth.nakayama40.jp/Others',
    staffnum: 27,
    regcapital: 211.2634,
    paidcapital: 251.2448,
    insuranceratio: 8.95,
    isenable: true,
    status_singleselect: '3',
    range_multipleselect: ['F3134', 'I3006', 'I3008', 'I3027', 'I3032', 'P1029'],
    status_radio: '5',
    range_check: ['I3006', 'I3008', 'I3012', 'L2095', 'M2070'],
    establishdate: '2022-12-20T04:42:54.000Z',
  },
  {
    createdAt: '2023-06-30T09:53:02.000Z',
    updatedAt: '2023-07-18T22:24:59.000Z',
    createdById: 3,
    updatedById: 4,
    testdataid: 56,
    orgcode: '210056',
    orgname: 'Lopez Brothers Technology LLC',
    address: '延庆区028县道953号',
    phone: '10-641-1187',
    email: 'naks@hotmail.com',
    url: 'https://image.guzh.info/CenturionGardenOutdoor',
    staffnum: 185,
    regcapital: 63.5178,
    paidcapital: 153.6774,
    insuranceratio: 91.75,
    isenable: true,
    status_singleselect: '3',
    range_multipleselect: ['I3008', 'I3010', 'I3012', 'I3014', 'I3034', 'M2070', 'P1029'],
    status_radio: '4',
    range_check: ['I3008', 'I3011', 'L2032', 'M2070', 'P1029'],
    establishdate: '2021-05-21T11:02:59.000Z',
  },
  {
    createdAt: '2023-06-24T08:12:11.000Z',
    updatedAt: '2023-07-10T02:25:11.743Z',
    createdById: 8,
    updatedById: null,
    testdataid: 89,
    orgcode: '640089',
    orgname: 'Curtis Pharmaceutical Inc.',
    address: null,
    phone: '179-9231-4455',
    email: 'aoi110@icloud.com',
    url: 'https://www.yuziyi.co.jp/ToolsHomeDecoration',
    staffnum: 119,
    regcapital: 196.5298,
    paidcapital: 195.5359,
    insuranceratio: null,
    isenable: null,
    status_singleselect: '4',
    range_multipleselect: ['F3134', 'I3012', 'I3014', 'I3027', 'L2095', 'O3010', 'P1029'],
    status_radio: '4',
    range_check: ['I3006', 'I3008', 'I3010', 'I3011', 'I3014', 'I3027', 'L2032', 'M2070'],
    establishdate: '2023-10-08T21:08:37.000Z',
  },
  {
    createdAt: '2023-08-20T12:48:08.000Z',
    updatedAt: '2023-06-13T11:27:58.000Z',
    createdById: 7,
    updatedById: 5,
    testdataid: 44,
    orgcode: '410044',
    orgname: 'Lee Brothers Inc.',
    address: '成华区双庆路300号',
    phone: null,
    email: 'tajiehong602@outlook.com',
    url: 'http://image.tonkay9.info/Baby',
    staffnum: 176,
    regcapital: 297.447,
    paidcapital: 16.0008,
    insuranceratio: 69.5,
    isenable: true,
    status_singleselect: '4',
    range_multipleselect: ['F3134', 'I3007', 'I3011', 'I3014', 'I3034', 'M2070'],
    status_radio: '3',
    range_check: ['F3134', 'I3006', 'I3010', 'I3027', 'I3034', 'L2032', 'O3010'],
    establishdate: '2023-01-04T12:48:43.000Z',
  },
  {
    createdAt: '2023-03-10T11:17:24.000Z',
    updatedAt: '2023-07-25T07:52:00.000Z',
    createdById: 3,
    updatedById: 4,
    testdataid: 35,
    orgcode: '150035',
    orgname: '邵記系统有限责任公司',
    address: '209 1st Ave',
    phone: '718-246-6197',
    email: 'akat@hotmail.com',
    url: 'http://video.fuchungyin1009.co.jp/ArtsHandicraftsSewing',
    staffnum: 185,
    regcapital: 262.7247,
    paidcapital: 299.6833,
    insuranceratio: 45.48,
    isenable: null,
    status_singleselect: '1',
    range_multipleselect: ['F3134', 'I3007', 'I3014', 'I3027', 'I3032', 'I3034', 'L2032', 'M2070', 'O3010'],
    status_radio: '2',
    range_check: ['I3006', 'I3007', 'I3010', 'I3032', 'I3034', 'M2070', 'P1029'],
    establishdate: '2022-06-04T09:44:46.000Z',
  },
  {
    createdAt: '2023-05-22T02:10:55.000Z',
    updatedAt: '2023-07-19T22:44:41.000Z',
    createdById: 9,
    updatedById: 3,
    testdataid: 4,
    orgcode: '430004',
    orgname: '宇宁制药有限责任公司',
    address: '328 Jingtian East 1st St, Futian District',
    phone: '755-3659-2323',
    email: 'cmsi92@icloud.com',
    url: 'http://www.czhennan4.biz/ToolsHomeDecoration',
    staffnum: 132,
    regcapital: 175.7828,
    paidcapital: 290.7203,
    insuranceratio: 45.74,
    isenable: true,
    status_singleselect: '8',
    range_multipleselect: ['I3006', 'I3007', 'I3008', 'I3010', 'I3011', 'I3012', 'I3027', 'M2070'],
    status_radio: '7',
    range_check: ['F3134', 'I3006', 'I3007', 'I3012', 'I3014', 'I3032', 'L2095', 'P1029'],
    establishdate: '2021-01-08T16:46:38.000Z',
  },
  {
    createdAt: '2023-03-08T16:04:01.000Z',
    updatedAt: '2023-05-05T22:57:27.000Z',
    createdById: 4,
    updatedById: 3,
    testdataid: 81,
    orgcode: '110081',
    orgname: '子异有限责任公司',
    address: '229 Rush Street',
    phone: '312-600-3390',
    email: 'lokwy@icloud.com',
    url: 'https://image.anakagawa.us/MusicalInstrument',
    staffnum: 26,
    regcapital: null,
    paidcapital: 186.0802,
    insuranceratio: 18.24,
    isenable: true,
    status_singleselect: '2',
    range_multipleselect: ['I3006', 'I3007', 'I3008', 'I3012', 'I3014', 'I3032', 'L2032', 'L2095', 'M2070', 'O3010'],
    status_radio: '5',
    range_check: ['F3134', 'I3006', 'I3007', 'I3008', 'I3012', 'I3027', 'L2032', 'O3010'],
    establishdate: '2021-04-08T05:24:21.000Z',
  },
  {
    createdAt: '2023-06-15T13:47:48.000Z',
    updatedAt: '2023-05-20T12:37:39.000Z',
    createdById: 0,
    updatedById: 4,
    testdataid: 34,
    orgcode: '210034',
    orgname: '董記电子有限责任公司',
    address: '265 Huaxia St, Jinghua Shangquan',
    phone: null,
    email: 'rivashle1114@gmail.com',
    url: 'http://auth.takeda1.cn/Handcrafts',
    staffnum: 176,
    regcapital: 281.1833,
    paidcapital: 67.1949,
    insuranceratio: 29.93,
    isenable: true,
    status_singleselect: '4',
    range_multipleselect: ['F3134', 'I3007', 'I3010', 'I3012', 'I3027', 'I3034', 'L2032', 'M2070', 'O3010', 'P1029'],
    status_radio: '2',
    range_check: ['F3134', 'I3006', 'I3007', 'I3011', 'I3012', 'I3014', 'L2095', 'O3010'],
    establishdate: '2023-02-08T16:44:29.000Z',
  },
  {
    createdAt: '2023-08-04T08:58:50.000Z',
    updatedAt: '2023-08-04T04:34:11.000Z',
    createdById: 9,
    updatedById: 8,
    testdataid: 87,
    orgcode: '420087',
    orgname: '史記物业代理有限责任公司',
    address: '549 Ridgewood Road',
    phone: '330-982-5220',
    email: 'maruyamashi@hotmail.com',
    url: 'https://image.kondohik1.co.jp/ToysGames',
    staffnum: 126,
    regcapital: 227.7387,
    paidcapital: 139.9423,
    insuranceratio: 80.19,
    isenable: true,
    status_singleselect: '5',
    range_multipleselect: ['I3008', 'I3010', 'I3011', 'I3027', 'I3034', 'P1029'],
    status_radio: '3',
    range_check: ['I3006', 'I3010', 'I3014', 'I3032', 'I3034', 'M2070', 'O3010'],
    establishdate: '2023-04-08T00:21:42.000Z',
  },
  {
    createdAt: '2023-06-17T22:39:01.000Z',
    updatedAt: '2023-05-12T07:34:03.000Z',
    createdById: 0,
    updatedById: 7,
    testdataid: 65,
    orgcode: '250065',
    orgname: '林有限责任公司',
    address: '595 3rd Section Hongxing Road, Jinjiang District',
    phone: '147-2716-7825',
    email: 'kathy09@gmail.com',
    url: 'http://auth.shenj.net/AppsGames',
    staffnum: 146,
    regcapital: 63.0539,
    paidcapital: 18.8649,
    insuranceratio: 0.22,
    isenable: true,
    status_singleselect: '2',
    range_multipleselect: ['I3008', 'I3011', 'I3012', 'I3027', 'L2032', 'L2095', 'M2070', 'P1029'],
    status_radio: '1',
    range_check: ['F3134', 'I3006', 'I3007', 'I3008', 'I3011', 'I3027', 'I3032', 'L2032', 'L2095'],
    establishdate: '2023-08-06T08:03:27.000Z',
  },
  {
    createdAt: '2023-04-26T07:55:22.000Z',
    updatedAt: '2023-06-15T15:44:16.000Z',
    createdById: 8,
    updatedById: 4,
    testdataid: 74,
    orgcode: '240074',
    orgname: "Schmidt's Food Inc.",
    address: '罗湖区蔡屋围深南东路630号',
    phone: '755-938-8862',
    email: 'jialun9@outlook.com',
    url: 'https://www.kwokwing614.org/ToysGames',
    staffnum: 34,
    regcapital: 291.2407,
    paidcapital: 235.8852,
    insuranceratio: 5.3,
    isenable: true,
    status_singleselect: '5',
    range_multipleselect: ['F3134', 'I3007', 'I3010', 'I3012', 'L2032', 'L2095', 'M2070'],
    status_radio: '5',
    range_check: ['F3134', 'I3006', 'I3010', 'I3011', 'I3014', 'I3032', 'M2070', 'O3010', 'P1029'],
    establishdate: '2021-06-15T18:47:52.000Z',
  },
  {
    createdAt: '2023-06-13T04:08:45.000Z',
    updatedAt: '2023-05-19T22:14:25.000Z',
    createdById: 9,
    updatedById: 4,
    testdataid: 19,
    orgcode: '650019',
    orgname: 'Myers Brothers Telecommunication Inc.',
    address: '294 Dong Zhi Men, Dongcheng District',
    phone: '155-7317-7341',
    email: 'daisukeotsu@mail.com',
    url: 'https://drive.floresla8.org/BeautyPersonalCare',
    staffnum: 128,
    regcapital: 197.9632,
    paidcapital: 90.5853,
    insuranceratio: 64.9,
    isenable: true,
    status_singleselect: '5',
    range_multipleselect: ['I3007', 'I3008', 'I3014', 'L2095', 'P1029'],
    status_radio: '2',
    range_check: ['I3006', 'I3008', 'I3010', 'I3032', 'L2095', 'M2070'],
    establishdate: '2023-04-30T14:56:20.000Z',
  },
  {
    createdAt: '2023-03-22T19:09:18.000Z',
    updatedAt: '2023-08-25T18:37:39.000Z',
    createdById: 7,
    updatedById: 5,
    testdataid: 21,
    orgcode: '620021',
    orgname: 'Williams Consultants Inc.',
    address: '786 Tremont Road',
    phone: '614-644-3993',
    email: 'aota96@yahoo.com',
    url: 'https://image.hilldo.biz/BeautyPersonalCare',
    staffnum: 195,
    regcapital: 94.6208,
    paidcapital: 113.2067,
    insuranceratio: 0.42,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: ['F3134', 'I3006', 'I3010', 'I3011', 'I3027', 'I3032', 'I3034'],
    status_radio: '6',
    range_check: ['F3134', 'I3007', 'I3008', 'I3012', 'I3014', 'I3027', 'I3034', 'L2032', 'M2070'],
    establishdate: '2021-06-14T09:07:34.000Z',
  },
  {
    createdAt: '2023-08-28T21:13:03.000Z',
    updatedAt: '2023-07-17T22:47:34.000Z',
    createdById: 8,
    updatedById: 6,
    testdataid: 43,
    orgcode: '220043',
    orgname: 'Mendez Logistic LLC',
    address: '浦东新区健祥路790号',
    phone: '172-1090-3775',
    email: 'suzukidaisuke@mail.com',
    url: 'https://www.zhengr.info/Books',
    staffnum: 173,
    regcapital: 173.8671,
    paidcapital: null,
    insuranceratio: 72.41,
    isenable: true,
    status_singleselect: '3',
    range_multipleselect: ['I3011', 'I3032', 'I3034', 'L2032', 'L2095', 'P1029'],
    status_radio: '8',
    range_check: [
      'F3134',
      'I3006',
      'I3007',
      'I3008',
      'I3010',
      'I3011',
      'I3012',
      'I3014',
      'I3032',
      'L2095',
      'M2070',
      'P1029',
    ],
    establishdate: '2021-12-27T04:23:26.000Z',
  },
  {
    createdAt: '2023-08-25T08:01:03.000Z',
    updatedAt: '2023-07-10T02:25:11.739Z',
    createdById: 4,
    updatedById: null,
    testdataid: 33,
    orgcode: '230033',
    orgname: '詩涵电脑有限责任公司',
    address: '825 2nd Zhongshan Road, Yuexiu District',
    phone: '166-7145-2872',
    email: null,
    url: 'http://video.scy.info/BeautyPersonalCare',
    staffnum: 184,
    regcapital: 237.116,
    paidcapital: 7.2748,
    insuranceratio: 66.74,
    isenable: null,
    status_singleselect: '1',
    range_multipleselect: ['F3134', 'I3006', 'I3007', 'I3011', 'I3012', 'I3027', 'I3032', 'L2032', 'O3010'],
    status_radio: '5',
    range_check: null,
    establishdate: '2021-10-10T20:57:37.000Z',
  },
  {
    createdAt: '2023-07-09T05:39:52.000Z',
    updatedAt: '2023-06-26T04:30:41.000Z',
    createdById: 6,
    updatedById: 3,
    testdataid: 10,
    orgcode: '230010',
    orgname: 'Medina Brothers Software LLC',
    address: '472 West Market Street',
    phone: '330-026-0212',
    email: 'murphygrace@yahoo.com',
    url: 'https://www.ngws.com/Books',
    staffnum: 17,
    regcapital: 74.6449,
    paidcapital: 281.953,
    insuranceratio: 26.67,
    isenable: true,
    status_singleselect: '2',
    range_multipleselect: ['F3134', 'I3006', 'I3007', 'I3008', 'I3011', 'I3014', 'I3034', 'L2032', 'M2070', 'P1029'],
    status_radio: '8',
    range_check: ['F3134', 'I3007', 'I3008', 'I3011', 'I3012', 'I3014', 'I3027', 'L2095', 'P1029'],
    establishdate: '2023-01-25T11:54:37.000Z',
  },
  {
    createdAt: '2023-08-15T23:23:20.000Z',
    updatedAt: '2023-05-09T00:00:10.000Z',
    createdById: 1,
    updatedById: 3,
    testdataid: 36,
    orgcode: '440036',
    orgname: '李記电子有限责任公司',
    address: '750 Lark Street',
    phone: '838-441-5793',
    email: 'ryonishimura530@gmail.com',
    url: 'https://image.cwk.biz/Baby',
    staffnum: 88,
    regcapital: 292.6699,
    paidcapital: 223.948,
    insuranceratio: 36.35,
    isenable: true,
    status_singleselect: '5',
    range_multipleselect: ['I3006', 'I3007', 'I3010', 'I3011', 'I3012', 'I3032', 'I3034', 'L2032', 'L2095', 'P1029'],
    status_radio: '4',
    range_check: ['I3007', 'I3008', 'I3012', 'I3014', 'L2032', 'O3010'],
    establishdate: null,
  },
  {
    createdAt: '2023-03-24T11:13:11.000Z',
    updatedAt: '2023-06-02T14:17:58.000Z',
    createdById: 10,
    updatedById: 4,
    testdataid: 51,
    orgcode: '550051',
    orgname: '胡有限责任公司',
    address: '西城区复兴门内大街438号',
    phone: '137-8041-7460',
    email: 'saumankwo@mail.com',
    url: 'https://auth.csoto.net/Others',
    staffnum: null,
    regcapital: 51.757,
    paidcapital: 67.2645,
    insuranceratio: 29.38,
    isenable: true,
    status_singleselect: '4',
    range_multipleselect: ['I3006', 'I3007', 'I3008', 'I3011', 'I3012', 'I3014', 'I3027', 'I3032', 'I3034'],
    status_radio: '7',
    range_check: ['F3134', 'I3006', 'I3008', 'I3010', 'I3011', 'I3014', 'I3032', 'L2032', 'O3010', 'P1029'],
    establishdate: '2021-01-02T11:44:41.000Z',
  },
  {
    createdAt: '2023-04-02T04:03:53.000Z',
    updatedAt: '2023-06-14T12:56:19.000Z',
    createdById: 6,
    updatedById: 1,
    testdataid: 59,
    orgcode: '210059',
    orgname: '陶記制药有限责任公司',
    address: null,
    phone: '179-0776-0684',
    email: 'yuto426@gmail.com',
    url: 'http://drive.pati.co.jp/ArtsHandicraftsSewing',
    staffnum: 154,
    regcapital: null,
    paidcapital: 78.4742,
    insuranceratio: 51.93,
    isenable: true,
    status_singleselect: '1',
    range_multipleselect: ['F3134', 'I3010', 'I3011', 'I3012', 'I3014', 'M2070', 'O3010', 'P1029'],
    status_radio: '7',
    range_check: ['F3134', 'I3006', 'I3007', 'I3010', 'I3027', 'I3032', 'I3034', 'L2095', 'M2070'],
    establishdate: '2023-01-31T01:55:31.000Z',
  },
  {
    createdAt: '2023-03-02T08:06:09.000Z',
    updatedAt: '2023-07-10T02:25:11.741Z',
    createdById: 7,
    updatedById: null,
    testdataid: 73,
    orgcode: '410073',
    orgname: '晓明有限责任公司',
    address: '696 North Michigan Ave',
    phone: '312-545-7935',
    email: 'jocox1229@outlook.com',
    url: null,
    staffnum: 181,
    regcapital: 208.5642,
    paidcapital: 124.9526,
    insuranceratio: 88.64,
    isenable: true,
    status_singleselect: '6',
    range_multipleselect: ['F3134', 'I3006', 'I3008', 'I3010', 'I3014', 'I3034', 'L2095', 'P1029'],
    status_radio: '2',
    range_check: ['F3134', 'I3006', 'I3008', 'I3010', 'I3011', 'I3014', 'I3027', 'I3032', 'L2095', 'M2070', 'O3010'],
    establishdate: null,
  },
  {
    createdAt: '2023-03-25T22:18:26.000Z',
    updatedAt: '2023-06-27T06:57:51.000Z',
    createdById: 5,
    updatedById: 3,
    testdataid: 5,
    orgcode: '630005',
    orgname: "Grant's Network Systems LLC",
    address: '成华区双庆路397号',
    phone: '149-8059-2712',
    email: 'turneal@gmail.com',
    url: 'http://auth.fujr.jp/Food',
    staffnum: 193,
    regcapital: 116.4198,
    paidcapital: 267.387,
    insuranceratio: 62.02,
    isenable: true,
    status_singleselect: '3',
    range_multipleselect: ['F3134', 'I3006', 'I3007', 'I3008', 'I3010', 'I3012', 'I3014', 'I3032', 'I3034', 'L2032'],
    status_radio: '6',
    range_check: ['I3006', 'I3027', 'L2095', 'P1029'],
    establishdate: '2023-09-09T15:50:04.000Z',
  },
  {
    createdAt: '2023-07-13T05:22:20.000Z',
    updatedAt: '2023-08-11T12:32:14.000Z',
    createdById: 3,
    updatedById: 6,
    testdataid: 29,
    orgcode: '440029',
    orgname: 'Thomas Brothers Pharmaceutical Inc.',
    address: '472 West Market Street',
    phone: '330-960-6930',
    email: 'shirleypay@yahoo.com',
    url: 'http://auth.jiang1103.jp/CollectiblesFineArt',
    staffnum: 68,
    regcapital: 4.9611,
    paidcapital: 142.3588,
    insuranceratio: 40.34,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: ['F3134', 'I3008', 'I3010', 'I3034', 'L2032', 'M2070', 'O3010'],
    status_radio: '3',
    range_check: ['F3134', 'I3010', 'I3011', 'I3012', 'I3014', 'I3027', 'O3010', 'P1029'],
    establishdate: '2021-09-05T07:53:50.000Z',
  },
  {
    createdAt: '2023-08-04T12:20:48.000Z',
    updatedAt: '2023-05-02T21:09:34.000Z',
    createdById: 3,
    updatedById: 4,
    testdataid: 95,
    orgcode: '350095',
    orgname: '陈記有限责任公司',
    address: '268 Bergen St',
    phone: '718-862-1720',
    email: 'yara1953@gmail.com',
    url: 'https://auth.riverae.jp/PetSupplies',
    staffnum: 66,
    regcapital: 282.8759,
    paidcapital: 67.7269,
    insuranceratio: 46.59,
    isenable: null,
    status_singleselect: '6',
    range_multipleselect: ['I3006', 'I3007', 'I3010', 'I3011', 'I3012', 'I3014', 'I3027', 'L2032', 'M2070'],
    status_radio: '3',
    range_check: ['I3006', 'I3010', 'I3011', 'I3014', 'I3027', 'I3034', 'P1029'],
    establishdate: '2021-11-25T14:56:54.000Z',
  },
  {
    createdAt: '2023-05-01T17:43:16.000Z',
    updatedAt: '2023-08-20T08:30:36.000Z',
    createdById: 1,
    updatedById: 4,
    testdataid: 8,
    orgcode: '540008',
    orgname: 'Christine Communications Inc.',
    address: '630 Canal Street',
    phone: '212-162-6691',
    email: 'hazelcraw@yahoo.com',
    url: 'http://auth.jialunhu.xyz/Others',
    staffnum: 47,
    regcapital: 140.3844,
    paidcapital: 236.435,
    insuranceratio: 51.21,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: ['I3006', 'I3007', 'I3008', 'I3014', 'I3032', 'L2095', 'O3010'],
    status_radio: '7',
    range_check: ['I3008', 'I3010', 'I3032', 'L2032', 'M2070', 'O3010'],
    establishdate: '2021-11-09T15:13:33.000Z',
  },
  {
    createdAt: '2023-06-20T03:38:25.000Z',
    updatedAt: '2023-06-28T02:29:27.000Z',
    createdById: 10,
    updatedById: 5,
    testdataid: 57,
    orgcode: '130057',
    orgname: 'Bobby Inc.',
    address: '631 Huaxia St, Jinghua Shangquan',
    phone: '167-3235-8699',
    email: 'zhend@gmail.com',
    url: 'http://drive.yunxi2015.biz/ClothingShoesandJewelry',
    staffnum: 161,
    regcapital: 20.9988,
    paidcapital: 188.2148,
    insuranceratio: 74.21,
    isenable: true,
    status_singleselect: '8',
    range_multipleselect: ['F3134', 'I3006', 'I3010', 'I3011', 'I3012', 'I3014', 'I3034', 'L2032', 'O3010', 'P1029'],
    status_radio: '3',
    range_check: null,
    establishdate: '2023-04-27T11:30:53.000Z',
  },
  {
    createdAt: '2023-04-21T22:18:05.000Z',
    updatedAt: '2023-05-02T02:52:18.000Z',
    createdById: 4,
    updatedById: 8,
    testdataid: 60,
    orgcode: '440060',
    orgname: 'Wilson Engineering LLC',
    address: '894 Jianxiang Rd, Pudong',
    phone: '147-8272-9832',
    email: 'hmwong@icloud.com',
    url: 'https://drive.ayano303.jp/Beauty',
    staffnum: null,
    regcapital: 129.6063,
    paidcapital: 211.5024,
    insuranceratio: 50.02,
    isenable: true,
    status_singleselect: '4',
    range_multipleselect: ['I3010', 'I3012', 'I3027', 'I3034', 'L2032', 'M2070'],
    status_radio: '2',
    range_check: ['I3006', 'I3007', 'I3010', 'I3011', 'I3012', 'I3034', 'L2032', 'L2095', 'M2070', 'P1029'],
    establishdate: '2023-02-02T04:12:05.000Z',
  },
  {
    createdAt: '2023-03-07T17:22:42.000Z',
    updatedAt: '2023-05-17T00:30:59.000Z',
    createdById: 9,
    updatedById: 7,
    testdataid: 28,
    orgcode: '140028',
    orgname: '嘉伦有限责任公司',
    address: '189 West Market Street',
    phone: '330-159-6349',
    email: 'hotheod@gmail.com',
    url: 'https://drive.warren98.biz/CellPhonesAccessories',
    staffnum: 95,
    regcapital: 35.1485,
    paidcapital: 60.3831,
    insuranceratio: 21.62,
    isenable: null,
    status_singleselect: '6',
    range_multipleselect: [
      'F3134',
      'I3006',
      'I3008',
      'I3010',
      'I3011',
      'I3012',
      'I3032',
      'L2032',
      'L2095',
      'M2070',
      'O3010',
      'P1029',
    ],
    status_radio: '5',
    range_check: ['F3134', 'I3008', 'I3010', 'I3011', 'I3034', 'P1029'],
    establishdate: '2022-01-13T23:52:05.000Z',
  },
  {
    createdAt: '2023-04-26T13:15:06.000Z',
    updatedAt: '2023-07-13T07:01:14.000Z',
    createdById: 1,
    updatedById: 1,
    testdataid: 49,
    orgcode: '150049',
    orgname: 'Owens Brothers Software LLC',
    address: '422 W Ring Rd, Buji Town, Longgang',
    phone: '755-057-2623',
    email: 'zhanglan@gmail.com',
    url: 'http://www.sotogregory502.org/ArtsHandicraftsSewing',
    staffnum: 130,
    regcapital: 46.1374,
    paidcapital: 3.4208,
    insuranceratio: 80.43,
    isenable: null,
    status_singleselect: '7',
    range_multipleselect: ['I3007', 'I3012', 'I3027', 'I3032', 'L2095', 'O3010', 'P1029'],
    status_radio: '4',
    range_check: ['F3134', 'I3007', 'I3010', 'I3011', 'I3027', 'I3032', 'I3034', 'L2032'],
    establishdate: '2022-11-07T06:43:42.000Z',
  },
  {
    createdAt: '2023-08-25T07:59:57.000Z',
    updatedAt: '2023-08-30T00:27:44.000Z',
    createdById: 0,
    updatedById: 8,
    testdataid: 12,
    orgcode: '150012',
    orgname: '张記有限责任公司',
    address: '531 4th Section  Renmin South Road, Jinjiang District',
    phone: '28-987-2142',
    email: 'tsubasayamaz5@outlook.com',
    url: null,
    staffnum: 188,
    regcapital: 282.0915,
    paidcapital: 230.6067,
    insuranceratio: 24.06,
    isenable: true,
    status_singleselect: '1',
    range_multipleselect: ['I3011', 'I3012', 'L2032', 'L2095', 'M2070', 'O3010', 'P1029'],
    status_radio: '7',
    range_check: ['F3134', 'I3007', 'I3008', 'I3010', 'I3011', 'I3012', 'I3014', 'I3027', 'I3032', 'I3034', 'P1029'],
    establishdate: '2021-01-18T20:54:21.000Z',
  },
  {
    createdAt: '2023-08-06T11:00:11.000Z',
    updatedAt: '2023-07-10T02:25:11.737Z',
    createdById: null,
    updatedById: null,
    testdataid: 47,
    orgcode: '620047',
    orgname: 'Williams Brothers LLC',
    address: '东泰五街621号',
    phone: null,
    email: 'martin2002@icloud.com',
    url: 'https://auth.kwokwaiman.us/HouseholdKitchenAppliances',
    staffnum: 78,
    regcapital: 279.4844,
    paidcapital: 266.3442,
    insuranceratio: 32.12,
    isenable: true,
    status_singleselect: '1',
    range_multipleselect: ['I3006', 'I3007', 'I3008', 'I3010', 'I3011', 'I3027', 'L2032'],
    status_radio: '8',
    range_check: ['F3134', 'I3006', 'I3007', 'I3010', 'I3011', 'I3012', 'I3027', 'I3034', 'L2032', 'L2095', 'O3010'],
    establishdate: '2022-08-14T17:55:17.000Z',
  },
  {
    createdAt: '2023-08-03T18:16:07.000Z',
    updatedAt: '2023-05-11T02:43:43.000Z',
    createdById: 0,
    updatedById: 9,
    testdataid: 69,
    orgcode: '610069',
    orgname: '致远有限责任公司',
    address: '龙岗区学园一巷968号',
    phone: '755-7571-7551',
    email: 'ynakagawa925@icloud.com',
    url: 'http://video.tzhiyuan.net/SportsOutdoor',
    staffnum: 90,
    regcapital: 2.7332,
    paidcapital: 108.5354,
    insuranceratio: 64.76,
    isenable: null,
    status_singleselect: '2',
    range_multipleselect: ['F3134', 'I3007', 'I3010', 'I3011', 'I3012', 'I3034', 'O3010'],
    status_radio: '2',
    range_check: ['I3007', 'I3008', 'I3010', 'I3012', 'I3014', 'I3032'],
    establishdate: '2021-06-18T12:50:21.000Z',
  },
  {
    createdAt: '2023-05-16T16:27:22.000Z',
    updatedAt: '2023-05-13T11:17:34.000Z',
    createdById: 5,
    updatedById: 6,
    testdataid: 80,
    orgcode: '140080',
    orgname: '睿贸易有限责任公司',
    address: '白云区小坪东路90号',
    phone: '197-2431-7197',
    email: 'wuwm@icloud.com',
    url: 'http://drive.hz4.com/HealthBabyCare',
    staffnum: 36,
    regcapital: 132.0943,
    paidcapital: 278.8981,
    insuranceratio: 40.03,
    isenable: true,
    status_singleselect: '8',
    range_multipleselect: ['F3134', 'I3007', 'I3010', 'I3011', 'I3032', 'L2032', 'L2095', 'M2070'],
    status_radio: '8',
    range_check: ['I3006', 'I3007', 'I3011', 'I3014', 'I3034', 'L2095'],
    establishdate: '2021-07-12T22:04:31.000Z',
  },
  {
    createdAt: '2023-03-16T02:59:56.000Z',
    updatedAt: '2023-07-03T00:12:24.000Z',
    createdById: 4,
    updatedById: 6,
    testdataid: 45,
    orgcode: '120045',
    orgname: 'Adams Brothers Inc.',
    address: '847 West Houston Street',
    phone: '212-675-9018',
    email: null,
    url: 'https://image.ks4.info/Handcrafts',
    staffnum: null,
    regcapital: 179.2233,
    paidcapital: 203.447,
    insuranceratio: 70.8,
    isenable: true,
    status_singleselect: '8',
    range_multipleselect: ['I3008', 'I3010', 'I3012', 'I3014', 'I3027', 'L2095', 'M2070', 'O3010'],
    status_radio: '8',
    range_check: ['F3134', 'I3008', 'I3012', 'I3014', 'I3027', 'I3032', 'L2095', 'M2070', 'P1029'],
    establishdate: '2021-03-27T22:05:05.000Z',
  },
  {
    createdAt: '2023-07-17T15:00:25.000Z',
    updatedAt: '2023-05-22T19:30:52.000Z',
    createdById: 8,
    updatedById: 9,
    testdataid: 85,
    orgcode: '310085',
    orgname: '秀英系统有限责任公司',
    address: '492 Pedway',
    phone: null,
    email: 'huyu1@outlook.com',
    url: 'http://image.shilan.net/ArtsHandicraftsSewing',
    staffnum: null,
    regcapital: 224.7069,
    paidcapital: null,
    insuranceratio: 27.21,
    isenable: true,
    status_singleselect: '8',
    range_multipleselect: ['I3008', 'I3012', 'I3014', 'I3034', 'L2095', 'M2070', 'P1029'],
    status_radio: '1',
    range_check: [
      'F3134',
      'I3006',
      'I3007',
      'I3008',
      'I3010',
      'I3011',
      'I3032',
      'I3034',
      'L2032',
      'L2095',
      'M2070',
      'O3010',
    ],
    establishdate: '2021-03-30T21:18:22.000Z',
  },
  {
    createdAt: '2023-04-02T16:10:12.000Z',
    updatedAt: '2023-07-10T02:25:11.740Z',
    createdById: 7,
    updatedById: null,
    testdataid: 72,
    orgcode: '350072',
    orgname: '邱記电讯有限责任公司',
    address: '白云区机场路棠苑街五巷951号',
    phone: '175-9040-0888',
    email: 'chiuwaiwong@gmail.com',
    url: null,
    staffnum: 162,
    regcapital: 59.8279,
    paidcapital: 141.9588,
    insuranceratio: 86.98,
    isenable: true,
    status_singleselect: '5',
    range_multipleselect: ['F3134', 'I3006', 'I3007', 'I3010', 'I3011', 'I3027', 'I3034', 'O3010'],
    status_radio: '6',
    range_check: null,
    establishdate: '2022-01-31T21:23:17.000Z',
  },
  {
    createdAt: '2023-05-07T01:59:43.000Z',
    updatedAt: '2023-08-13T07:41:11.000Z',
    createdById: 4,
    updatedById: 10,
    testdataid: 100,
    orgcode: '310100',
    orgname: '陈記发展贸易有限责任公司',
    address: '环区南街二巷453号',
    phone: '769-5777-9743',
    email: 'siu817@icloud.com',
    url: 'https://auth.donfreem506.org/SportsOutdoor',
    staffnum: 193,
    regcapital: 264.7365,
    paidcapital: 32.0125,
    insuranceratio: 46.66,
    isenable: null,
    status_singleselect: '7',
    range_multipleselect: null,
    status_radio: '7',
    range_check: ['F3134', 'I3006', 'I3011', 'I3027', 'I3034', 'O3010', 'P1029'],
    establishdate: '2023-05-23T04:27:32.000Z',
  },
  {
    createdAt: '2023-08-01T08:19:39.000Z',
    updatedAt: '2023-07-17T02:48:36.000Z',
    createdById: 3,
    updatedById: 5,
    testdataid: 30,
    orgcode: '550030',
    orgname: '子异技术有限责任公司',
    address: '成华区二仙桥东三路698号',
    phone: null,
    email: null,
    url: 'https://drive.patterson44.net/ArtsHandicraftsSewing',
    staffnum: null,
    regcapital: 88.4317,
    paidcapital: 232.3953,
    insuranceratio: 88.86,
    isenable: true,
    status_singleselect: '6',
    range_multipleselect: ['I3007', 'L2032', 'L2095', 'M2070', 'P1029'],
    status_radio: '7',
    range_check: ['F3134', 'I3011', 'I3032'],
    establishdate: '2022-01-26T08:14:41.000Z',
  },
  {
    createdAt: '2023-07-20T08:54:02.000Z',
    updatedAt: '2023-06-15T20:12:54.000Z',
    createdById: 6,
    updatedById: 8,
    testdataid: 75,
    orgcode: '310075',
    orgname: '武記有限责任公司',
    address: '545 Rush Street',
    phone: '312-022-1358',
    email: 'xiuying1125@hotmail.com',
    url: 'https://video.zhezi.jp/Books',
    staffnum: 197,
    regcapital: 136.3425,
    paidcapital: 96.297,
    insuranceratio: 28.79,
    isenable: true,
    status_singleselect: '6',
    range_multipleselect: ['I3008', 'I3012', 'I3034', 'O3010', 'P1029'],
    status_radio: '7',
    range_check: ['F3134', 'I3006', 'I3007', 'I3014', 'I3027', 'I3032', 'L2095', 'M2070'],
    establishdate: '2021-01-26T06:55:36.000Z',
  },
  {
    createdAt: '2023-07-12T18:10:51.000Z',
    updatedAt: '2023-08-19T13:15:44.000Z',
    createdById: 3,
    updatedById: 1,
    testdataid: 23,
    orgcode: '420023',
    orgname: '秀英有限责任公司',
    address: null,
    phone: '330-890-8118',
    email: 'grahadiana1127@icloud.com',
    url: 'https://auth.kkin.org/CellPhonesAccessories',
    staffnum: 152,
    regcapital: 190.0139,
    paidcapital: 274.6092,
    insuranceratio: 6.39,
    isenable: true,
    status_singleselect: '5',
    range_multipleselect: ['I3008', 'I3010', 'I3011', 'I3012', 'I3027', 'L2032', 'O3010', 'P1029'],
    status_radio: '2',
    range_check: ['I3006', 'I3007', 'I3010', 'I3012', 'I3014', 'I3032', 'I3034', 'L2095', 'M2070'],
    establishdate: '2023-06-21T21:47:05.000Z',
  },
  {
    createdAt: '2023-06-30T09:34:24.000Z',
    updatedAt: '2023-07-10T02:25:11.728Z',
    createdById: null,
    updatedById: 3,
    testdataid: 26,
    orgcode: '520026',
    orgname: '钱記有限责任公司',
    address: '106 S Broadway',
    phone: null,
    email: 'hla53@gmail.com',
    url: 'http://drive.songzhiyuan.jp/SportsOutdoor',
    staffnum: 128,
    regcapital: 45.8576,
    paidcapital: 49.0852,
    insuranceratio: 69.17,
    isenable: true,
    status_singleselect: '5',
    range_multipleselect: ['F3134', 'I3010', 'L2095', 'P1029'],
    status_radio: '5',
    range_check: ['F3134', 'I3006', 'I3008', 'I3010', 'I3027', 'I3032', 'L2095', 'O3010'],
    establishdate: '2022-03-22T20:41:25.000Z',
  },
  {
    createdAt: '2023-05-24T19:59:54.000Z',
    updatedAt: '2023-08-20T10:07:16.000Z',
    createdById: 8,
    updatedById: 3,
    testdataid: 37,
    orgcode: '620037',
    orgname: 'Cindy Logistic Inc.',
    address: '791 Columbia St',
    phone: '718-173-2935',
    email: 'satoyuna2@yahoo.com',
    url: 'https://www.yunxi8.cn/MusicalInstrument',
    staffnum: 6,
    regcapital: 39.6587,
    paidcapital: 118.6882,
    insuranceratio: 25.94,
    isenable: true,
    status_singleselect: '2',
    range_multipleselect: ['F3134', 'I3007', 'I3008', 'I3012', 'L2032', 'L2095', 'O3010'],
    status_radio: '5',
    range_check: null,
    establishdate: null,
  },
  {
    createdAt: '2023-07-10T12:04:12.000Z',
    updatedAt: '2023-07-10T02:25:11.742Z',
    createdById: 6,
    updatedById: null,
    testdataid: 78,
    orgcode: '140078',
    orgname: '陆記有限责任公司',
    address: '578 Xiaoping E Rd, Baiyun ',
    phone: '155-0484-9694',
    email: 'fukudahikari@gmail.com',
    url: 'https://auth.gibsondon.us/CollectiblesFineArt',
    staffnum: 79,
    regcapital: 21.868,
    paidcapital: 252.1746,
    insuranceratio: 58.72,
    isenable: true,
    status_singleselect: '8',
    range_multipleselect: ['I3010', 'I3032', 'L2032', 'L2095', 'O3010', 'P1029'],
    status_radio: '5',
    range_check: ['F3134', 'I3008', 'I3010', 'I3012', 'I3027', 'I3034', 'L2032', 'M2070', 'P1029'],
    establishdate: '2021-04-16T16:14:38.000Z',
  },
  {
    createdAt: '2023-07-05T20:02:07.000Z',
    updatedAt: '2023-05-27T21:16:46.000Z',
    createdById: 10,
    updatedById: 5,
    testdataid: 71,
    orgcode: '630071',
    orgname: '晓明有限责任公司',
    address: "19 West Chang'an Avenue, Xicheng District",
    phone: '156-1137-8442',
    email: 'sakurai46@icloud.com',
    url: 'https://auth.xiaoming1223.cn/Beauty',
    staffnum: 82,
    regcapital: 250.4023,
    paidcapital: 186.6815,
    insuranceratio: 95.41,
    isenable: true,
    status_singleselect: '3',
    range_multipleselect: ['F3134', 'I3008', 'I3010', 'I3011', 'I3012', 'I3027', 'I3034', 'L2032', 'P1029'],
    status_radio: '7',
    range_check: ['I3006', 'I3007', 'I3010', 'I3011', 'I3014', 'I3027', 'I3032', 'I3034', 'L2032', 'O3010'],
    establishdate: '2022-11-27T14:02:00.000Z',
  },
  {
    createdAt: '2023-04-03T11:41:58.000Z',
    updatedAt: '2023-07-25T07:08:25.000Z',
    createdById: 4,
    updatedById: 5,
    testdataid: 53,
    orgcode: '420053',
    orgname: '睿工业有限责任公司',
    address: '914 Xiaoping E Rd, Baiyun ',
    phone: '20-439-3781',
    email: 'lucimoreno@outlook.com',
    url: 'https://www.lanwang1968.biz/BaggageTravelEquipment',
    staffnum: 138,
    regcapital: 176.73,
    paidcapital: 238.5599,
    insuranceratio: 0.32,
    isenable: true,
    status_singleselect: '8',
    range_multipleselect: ['I3008', 'I3014', 'I3027', 'I3032', 'L2032', 'O3010'],
    status_radio: '1',
    range_check: ['I3007', 'I3008', 'I3010', 'I3011', 'I3027', 'M2070', 'O3010', 'P1029'],
    establishdate: '2021-04-22T09:11:54.000Z',
  },
  {
    createdAt: '2023-04-12T12:17:12.000Z',
    updatedAt: '2023-08-28T05:02:11.000Z',
    createdById: 3,
    updatedById: 5,
    testdataid: 70,
    orgcode: '620070',
    orgname: '唐有限责任公司',
    address: '449 East Alley',
    phone: '614-258-2477',
    email: 'swtong17@gmail.com',
    url: 'https://www.colesherr5.us/FilmSupplies',
    staffnum: 42,
    regcapital: 232.1782,
    paidcapital: 280.8157,
    insuranceratio: 20.07,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: ['F3134', 'I3007', 'I3008', 'I3012', 'I3034', 'L2095', 'P1029'],
    status_radio: '6',
    range_check: ['I3010', 'I3027', 'I3032', 'O3010'],
    establishdate: '2023-10-29T00:43:12.000Z',
  },
  {
    createdAt: '2023-04-13T23:19:56.000Z',
    updatedAt: '2023-08-08T03:08:04.000Z',
    createdById: 3,
    updatedById: 6,
    testdataid: 9,
    orgcode: '630009',
    orgname: '刘有限责任公司',
    address: '917 Alameda Street',
    phone: '213-743-6878',
    email: 'huimeiwong1127@icloud.com',
    url: 'https://image.rinkond1.co.jp/IndustrialScientificSupplies',
    staffnum: 190,
    regcapital: 264.891,
    paidcapital: 249.9184,
    insuranceratio: 54.71,
    isenable: true,
    status_singleselect: '8',
    range_multipleselect: null,
    status_radio: '3',
    range_check: ['F3134', 'I3007', 'I3010', 'I3012', 'I3014', 'I3032', 'I3034', 'L2095', 'P1029'],
    establishdate: '2023-08-22T00:30:45.000Z',
  },
  {
    createdAt: '2023-06-21T00:53:37.000Z',
    updatedAt: '2023-05-24T10:30:53.000Z',
    createdById: 1,
    updatedById: 10,
    testdataid: 24,
    orgcode: '140024',
    orgname: '秀英有限责任公司',
    address: '208 North Michigan Ave',
    phone: '312-222-4942',
    email: 'sitls@outlook.com',
    url: 'https://auth.ededward.co.jp/ClothingShoesandJewelry',
    staffnum: null,
    regcapital: 11.602,
    paidcapital: 289.2036,
    insuranceratio: 19.3,
    isenable: true,
    status_singleselect: '4',
    range_multipleselect: ['I3007', 'I3010', 'I3011', 'I3012', 'I3032', 'I3034', 'L2095', 'O3010'],
    status_radio: '2',
    range_check: ['F3134', 'I3007', 'I3012', 'I3014', 'I3027', 'L2032', 'O3010', 'P1029'],
    establishdate: '2023-04-14T10:31:47.000Z',
  },
  {
    createdAt: '2023-08-20T18:44:46.000Z',
    updatedAt: '2023-07-09T19:37:01.000Z',
    createdById: 7,
    updatedById: 4,
    testdataid: 15,
    orgcode: '150015',
    orgname: 'Doris LLC',
    address: '海珠区江南西路205号',
    phone: '152-0553-7772',
    email: 'onkay5@icloud.com',
    url: 'http://auth.yuningshi.cn/Baby',
    staffnum: 51,
    regcapital: 62.3067,
    paidcapital: 243.9179,
    insuranceratio: 65.83,
    isenable: true,
    status_singleselect: '1',
    range_multipleselect: ['I3007', 'I3014', 'I3032', 'L2095', 'M2070', 'P1029'],
    status_radio: '8',
    range_check: ['I3007', 'I3010', 'I3011', 'I3014', 'I3032', 'M2070', 'O3010'],
    establishdate: '2021-12-17T06:08:17.000Z',
  },
  {
    createdAt: '2023-07-08T09:41:18.000Z',
    updatedAt: '2023-08-04T03:25:41.000Z',
    createdById: 9,
    updatedById: 8,
    testdataid: 11,
    orgcode: '230011',
    orgname: 'Alexander Trading Inc.',
    address: '21 4th Section  Renmin South Road, Jinjiang District',
    phone: '28-964-2526',
    email: 'stewart324@gmail.com',
    url: null,
    staffnum: 74,
    regcapital: 205.9118,
    paidcapital: 252.7905,
    insuranceratio: 32.57,
    isenable: true,
    status_singleselect: '4',
    range_multipleselect: ['F3134', 'I3006', 'I3010', 'I3011', 'I3014', 'I3027', 'I3032', 'L2095', 'M2070'],
    status_radio: '4',
    range_check: ['F3134', 'I3006', 'I3010', 'I3011', 'I3034', 'L2032', 'L2095', 'M2070', 'O3010', 'P1029'],
    establishdate: '2022-10-25T03:57:53.000Z',
  },
  {
    createdAt: '2023-06-10T01:08:46.000Z',
    updatedAt: '2023-08-18T23:07:28.000Z',
    createdById: 6,
    updatedById: 8,
    testdataid: 58,
    orgcode: '340058',
    orgname: 'Frank Software LLC',
    address: '天河区天河路330号',
    phone: '165-2214-3345',
    email: 'crystortiz8@icloud.com',
    url: 'https://auth.richardlee7.co.jp/ClothingShoesandJewelry',
    staffnum: 180,
    regcapital: 17.056,
    paidcapital: 156.8234,
    insuranceratio: 67.28,
    isenable: null,
    status_singleselect: '1',
    range_multipleselect: ['I3006', 'I3007', 'I3008', 'I3011', 'I3032', 'L2032', 'L2095', 'O3010'],
    status_radio: '6',
    range_check: ['F3134', 'I3006', 'I3010', 'I3011', 'I3012', 'I3014', 'I3027', 'I3032', 'I3034', 'L2032', 'L2095'],
    establishdate: '2021-02-13T02:03:49.000Z',
  },
  {
    createdAt: '2023-07-01T15:28:34.000Z',
    updatedAt: '2023-07-10T02:25:11.726Z',
    createdById: null,
    updatedById: 5,
    testdataid: 39,
    orgcode: '510039',
    orgname: '龚电讯有限责任公司',
    address: '365 Central Avenue',
    phone: '838-434-4757',
    email: 'barnicholas@gmail.com',
    url: 'http://www.lokht1970.info/CDsVinyl',
    staffnum: 83,
    regcapital: 21.7374,
    paidcapital: 120.7973,
    insuranceratio: 47.33,
    isenable: true,
    status_singleselect: '1',
    range_multipleselect: ['F3134', 'I3007', 'I3011', 'I3034', 'L2095'],
    status_radio: '7',
    range_check: ['I3006', 'I3032', 'L2095', 'P1029'],
    establishdate: '2022-01-02T12:32:43.000Z',
  },
  {
    createdAt: '2023-04-03T15:27:23.000Z',
    updatedAt: '2023-05-29T18:51:28.000Z',
    createdById: 1,
    updatedById: 10,
    testdataid: 18,
    orgcode: '120018',
    orgname: '胡工程有限责任公司',
    address: '279 Zhongshan 5th Rd, Zimaling Shangquan',
    phone: '190-4080-9140',
    email: 'guanqi@icloud.com',
    url: 'http://www.nancyha5.us/CDsVinyl',
    staffnum: 112,
    regcapital: 144.4962,
    paidcapital: 145.2596,
    insuranceratio: 14.18,
    isenable: null,
    status_singleselect: '1',
    range_multipleselect: null,
    status_radio: '5',
    range_check: ['F3134', 'I3008', 'I3014', 'M2070', 'O3010'],
    establishdate: '2023-04-21T15:43:03.000Z',
  },
  {
    createdAt: '2023-06-18T11:53:30.000Z',
    updatedAt: '2023-08-03T11:12:45.000Z',
    createdById: 0,
    updatedById: 7,
    testdataid: 79,
    orgcode: '130079',
    orgname: '卢記工业有限责任公司',
    address: '坑美十五巷202号',
    phone: '769-6447-1441',
    email: null,
    url: 'https://www.simmons96.org/Appliances',
    staffnum: 144,
    regcapital: 207.3241,
    paidcapital: 128.2934,
    insuranceratio: 13.18,
    isenable: true,
    status_singleselect: '6',
    range_multipleselect: ['I3007', 'I3008', 'I3011', 'I3014', 'I3027', 'I3034', 'L2032', 'L2095'],
    status_radio: '7',
    range_check: ['I3006', 'I3008', 'I3010', 'I3014', 'M2070', 'O3010'],
    establishdate: '2021-05-10T10:57:24.000Z',
  },
  {
    createdAt: '2023-08-16T10:27:52.000Z',
    updatedAt: '2023-07-10T02:25:11.725Z',
    createdById: null,
    updatedById: 4,
    testdataid: 63,
    orgcode: '320063',
    orgname: 'Perry Pharmaceutical LLC',
    address: '房山区岳琉路956号',
    phone: null,
    email: 'kwyin302@hotmail.com',
    url: 'http://video.wingfatcha7.us/ComputersElectronics',
    staffnum: 169,
    regcapital: 143.6456,
    paidcapital: 17.5641,
    insuranceratio: 49.78,
    isenable: true,
    status_singleselect: '5',
    range_multipleselect: ['I3007', 'I3008', 'I3027', 'I3034', 'L2095', 'M2070'],
    status_radio: '2',
    range_check: ['F3134', 'I3006', 'I3014', 'I3027', 'I3032', 'I3034', 'L2032', 'L2095'],
    establishdate: '2021-04-22T09:53:07.000Z',
  },
  {
    createdAt: '2023-07-10T19:32:09.000Z',
    updatedAt: '2023-07-10T02:25:11.732Z',
    createdById: null,
    updatedById: 7,
    testdataid: 94,
    orgcode: '430094',
    orgname: 'Aguilar LLC',
    address: '710 1st Ave',
    phone: '718-677-4225',
    email: 'mchic@outlook.com',
    url: 'http://www.wtimothy.info/AppsGames',
    staffnum: 158,
    regcapital: 229.4348,
    paidcapital: 248.8399,
    insuranceratio: 82.42,
    isenable: true,
    status_singleselect: '3',
    range_multipleselect: ['I3006', 'I3008', 'I3010', 'I3011', 'I3012', 'I3014', 'L2032', 'P1029'],
    status_radio: '1',
    range_check: [
      'I3006',
      'I3007',
      'I3008',
      'I3012',
      'I3014',
      'I3027',
      'I3032',
      'L2032',
      'L2095',
      'M2070',
      'O3010',
      'P1029',
    ],
    establishdate: '2021-04-20T23:37:47.000Z',
  },
  {
    createdAt: '2023-07-01T04:31:41.000Z',
    updatedAt: '2023-05-06T19:48:18.000Z',
    createdById: 6,
    updatedById: 4,
    testdataid: 90,
    orgcode: '640090',
    orgname: 'Maria Communications LLC',
    address: '東城区東直門內大街160号',
    phone: '10-4155-4174',
    email: 'keithug@gmail.com',
    url: 'https://auth.ayahasegawa9.net/Others',
    staffnum: 195,
    regcapital: 24.8107,
    paidcapital: 92.6292,
    insuranceratio: 67.85,
    isenable: true,
    status_singleselect: '8',
    range_multipleselect: ['I3007', 'I3008', 'I3014', 'I3027', 'I3032', 'I3034', 'L2095', 'M2070', 'P1029'],
    status_radio: '2',
    range_check: ['I3006', 'I3007', 'I3011', 'I3012', 'I3014', 'I3027', 'L2032', 'M2070', 'P1029'],
    establishdate: '2023-06-19T18:33:58.000Z',
  },
  {
    createdAt: '2023-04-30T13:55:12.000Z',
    updatedAt: '2023-07-10T02:25:11.739Z',
    createdById: 10,
    updatedById: null,
    testdataid: 61,
    orgcode: '640061',
    orgname: '秀英工程有限责任公司',
    address: '锦江区人民南路四段972号',
    phone: '28-8229-5494',
    email: 'kkakeung@hotmail.com',
    url: 'https://image.daisauman.co.jp/BaggageTravelEquipment',
    staffnum: 169,
    regcapital: 8.3627,
    paidcapital: 267.237,
    insuranceratio: 41.35,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: ['F3134', 'I3007', 'I3011', 'I3012', 'I3014', 'I3027', 'I3032', 'L2032', 'L2095', 'P1029'],
    status_radio: '1',
    range_check: ['F3134', 'I3006', 'I3008', 'I3010', 'I3014', 'I3027', 'I3034', 'L2032', 'P1029'],
    establishdate: '2021-09-20T16:20:50.000Z',
  },
  {
    createdAt: '2023-04-04T06:33:16.000Z',
    updatedAt: '2023-06-11T22:08:56.000Z',
    createdById: 3,
    updatedById: 4,
    testdataid: 1,
    orgcode: '320001',
    orgname: '子异有限责任公司',
    address: '锦江区人民南路四段831号',
    phone: '188-8510-5193',
    email: 'yamadaayato907@yahoo.com',
    url: 'http://video.merichardson5.biz/SportsOutdoor',
    staffnum: 87,
    regcapital: 224.0227,
    paidcapital: 288.8069,
    insuranceratio: 0.75,
    isenable: true,
    status_singleselect: '1',
    range_multipleselect: [
      'I3006',
      'I3007',
      'I3010',
      'I3011',
      'I3012',
      'I3027',
      'I3034',
      'L2032',
      'L2095',
      'M2070',
      'O3010',
    ],
    status_radio: '1',
    range_check: ['F3134', 'I3007', 'I3010', 'I3012', 'I3032', 'L2032', 'M2070'],
    establishdate: '2022-12-03T17:27:35.000Z',
  },
  {
    createdAt: '2023-04-02T06:15:56.000Z',
    updatedAt: '2023-06-20T05:21:38.000Z',
    createdById: 3,
    updatedById: 4,
    testdataid: 38,
    orgcode: '450038',
    orgname: "Mills's Toy Inc.",
    address: '成华区二仙桥东三路904号',
    phone: '28-3185-9051',
    email: 'moria@icloud.com',
    url: 'https://video.sekat.net/ComputersElectronics',
    staffnum: 39,
    regcapital: 173.4007,
    paidcapital: 38.577,
    insuranceratio: 84.44,
    isenable: true,
    status_singleselect: '5',
    range_multipleselect: ['I3011', 'I3032', 'L2095', 'M2070', 'P1029'],
    status_radio: '7',
    range_check: null,
    establishdate: '2022-05-21T01:29:54.000Z',
  },
  {
    createdAt: '2023-03-13T10:50:26.000Z',
    updatedAt: '2023-06-01T14:40:47.000Z',
    createdById: 6,
    updatedById: 6,
    testdataid: 14,
    orgcode: '420014',
    orgname: '安琪制药有限责任公司',
    address: '986 1st Ave',
    phone: '718-947-3048',
    email: 'powm@yahoo.com',
    url: 'http://drive.chengsw.co.jp/AutomotivePartsAccessories',
    staffnum: 174,
    regcapital: 142.9716,
    paidcapital: 80.9565,
    insuranceratio: 23.87,
    isenable: null,
    status_singleselect: '4',
    range_multipleselect: ['I3006', 'I3007', 'I3008', 'I3010', 'I3011', 'I3012', 'I3027', 'I3032', 'L2032', 'L2095'],
    status_radio: '1',
    range_check: ['F3134', 'I3006', 'I3010', 'I3014', 'I3027', 'I3032', 'L2032'],
    establishdate: '2022-04-02T23:44:50.000Z',
  },
  {
    createdAt: '2023-05-01T17:47:37.000Z',
    updatedAt: '2023-07-11T11:57:11.000Z',
    createdById: 1,
    updatedById: 6,
    testdataid: 7,
    orgcode: '120007',
    orgname: 'Dawn Consultants Inc.',
    address: '西城区西長安街924号',
    phone: '10-368-3528',
    email: 'onkayya@yahoo.com',
    url: 'http://auth.che7.us/PetSupplies',
    staffnum: 18,
    regcapital: 118.5422,
    paidcapital: 26.4247,
    insuranceratio: 3.54,
    isenable: true,
    status_singleselect: '2',
    range_multipleselect: ['I3006', 'I3007', 'I3014', 'I3027', 'I3032', 'L2095', 'O3010', 'P1029'],
    status_radio: '2',
    range_check: ['F3134', 'I3010', 'I3012', 'I3014', 'I3027', 'I3034', 'L2032', 'P1029'],
    establishdate: '2023-08-06T11:29:00.000Z',
  },
  {
    createdAt: '2023-06-27T04:27:55.000Z',
    updatedAt: '2023-07-22T04:49:39.000Z',
    createdById: 5,
    updatedById: 3,
    testdataid: 96,
    orgcode: '210096',
    orgname: 'Samuel Electronic Inc.',
    address: '53 Dong Zhi Men, Dongcheng District',
    phone: '169-1441-5592',
    email: 'xuyuning@mail.com',
    url: 'https://auth.wu76.cn/Baby',
    staffnum: 195,
    regcapital: 39.701,
    paidcapital: 207.4876,
    insuranceratio: 94.68,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: ['F3134', 'I3007', 'I3010', 'I3027', 'I3032', 'L2095', 'M2070', 'P1029'],
    status_radio: '8',
    range_check: ['I3006', 'I3034', 'L2032', 'L2095'],
    establishdate: '2021-05-01T13:11:18.000Z',
  },
  {
    createdAt: '2023-06-12T05:43:49.000Z',
    updatedAt: '2023-07-17T08:15:07.000Z',
    createdById: 6,
    updatedById: 5,
    testdataid: 40,
    orgcode: '230040',
    orgname: 'Johnny LLC',
    address: '788 Shennan Ave, Futian District',
    phone: '755-593-2881',
    email: 'hjo@mail.com',
    url: null,
    staffnum: 137,
    regcapital: 24.5385,
    paidcapital: 250.5823,
    insuranceratio: 52.25,
    isenable: true,
    status_singleselect: '2',
    range_multipleselect: ['F3134', 'I3007', 'I3010', 'I3011', 'I3014', 'I3032', 'L2032', 'L2095', 'M2070'],
    status_radio: '4',
    range_check: ['I3007', 'I3010', 'I3014', 'I3027', 'I3032', 'I3034', 'L2095', 'M2070', 'O3010', 'P1029'],
    establishdate: '2023-11-17T21:46:14.000Z',
  },
  {
    createdAt: '2023-06-15T06:41:01.000Z',
    updatedAt: '2023-06-17T00:51:49.000Z',
    createdById: 8,
    updatedById: 5,
    testdataid: 42,
    orgcode: '630042',
    orgname: 'Griffin Brothers Logistic Inc.',
    address: '197 Grape Street',
    phone: '213-126-7992',
    email: 'zitaowan1971@outlook.com',
    url: 'https://image.parkerdiane.net/Others',
    staffnum: 26,
    regcapital: 251.177,
    paidcapital: 292.6008,
    insuranceratio: 79.99,
    isenable: true,
    status_singleselect: '6',
    range_multipleselect: ['F3134', 'I3006', 'I3007', 'I3011', 'I3034', 'L2032', 'L2095', 'O3010', 'P1029'],
    status_radio: '1',
    range_check: ['F3134', 'I3011', 'I3012', 'I3027', 'I3032', 'P1029'],
    establishdate: '2022-10-06T20:13:52.000Z',
  },
  {
    createdAt: '2023-03-04T18:03:54.000Z',
    updatedAt: '2023-08-07T00:59:00.000Z',
    createdById: 5,
    updatedById: 1,
    testdataid: 52,
    orgcode: '650052',
    orgname: '杜工程有限责任公司',
    address: '924 Wooster Street',
    phone: '212-656-1851',
    email: 'sakamoto303@icloud.com',
    url: 'http://drive.waiman1946.biz/ClothingShoesandJewelry',
    staffnum: 149,
    regcapital: 85.2739,
    paidcapital: 47.86,
    insuranceratio: 5.24,
    isenable: true,
    status_singleselect: '1',
    range_multipleselect: ['I3007', 'I3008', 'I3012', 'I3014', 'I3027', 'I3032', 'I3034', 'L2095', 'M2070', 'P1029'],
    status_radio: '8',
    range_check: ['I3007', 'I3010', 'I3012', 'I3027', 'I3032', 'I3034', 'L2032', 'L2095', 'M2070'],
    establishdate: '2021-04-22T06:26:45.000Z',
  },
  {
    createdAt: '2023-04-01T16:13:17.000Z',
    updatedAt: '2023-05-22T09:24:17.000Z',
    createdById: 8,
    updatedById: 10,
    testdataid: 6,
    orgcode: '630006',
    orgname: 'Jacob Telecommunication LLC',
    address: '东泰五街968号',
    phone: '769-163-0302',
    email: 'ikiw@gmail.com',
    url: 'http://www.wmsiu.cn/ClothingShoesandJewelry',
    staffnum: 129,
    regcapital: 28.8318,
    paidcapital: 17.9883,
    insuranceratio: 87.39,
    isenable: true,
    status_singleselect: '1',
    range_multipleselect: ['I3006', 'I3007', 'I3008', 'I3012', 'I3014', 'I3032', 'L2032', 'O3010', 'P1029'],
    status_radio: '3',
    range_check: ['F3134', 'I3006', 'I3010', 'I3011', 'I3027', 'I3032', 'L2095', 'M2070'],
    establishdate: '2023-04-11T23:07:14.000Z',
  },
  {
    createdAt: '2023-03-15T15:19:15.000Z',
    updatedAt: '2023-07-10T02:25:11.742Z',
    createdById: 8,
    updatedById: null,
    testdataid: 83,
    orgcode: '440083',
    orgname: 'Linda Software LLC',
    address: '福田区景田东一街36号',
    phone: '193-8669-1290',
    email: 'mio10@outlook.com',
    url: 'https://video.joe306.org/Handcrafts',
    staffnum: 123,
    regcapital: 127.33,
    paidcapital: 209.9907,
    insuranceratio: 96.56,
    isenable: true,
    status_singleselect: '2',
    range_multipleselect: ['I3006', 'I3012', 'I3027', 'L2095', 'M2070', 'O3010', 'P1029'],
    status_radio: '3',
    range_check: [
      'F3134',
      'I3006',
      'I3007',
      'I3008',
      'I3011',
      'I3012',
      'I3027',
      'I3032',
      'I3034',
      'L2095',
      'M2070',
      'O3010',
    ],
    establishdate: '2021-10-24T08:00:26.000Z',
  },
  {
    createdAt: '2023-08-08T08:50:58.000Z',
    updatedAt: '2023-07-30T08:37:12.000Z',
    createdById: 1,
    updatedById: 4,
    testdataid: 64,
    orgcode: '650064',
    orgname: '范物业代理有限责任公司',
    address: '459 Xue Yuan Yi Xiang, Longgang',
    phone: '755-2232-1934',
    email: 'leswhite229@outlook.com',
    url: 'http://auth.nakanaoshi1954.biz/BeautyPersonalCare',
    staffnum: 167,
    regcapital: 50.6322,
    paidcapital: 253.2695,
    insuranceratio: 74.53,
    isenable: null,
    status_singleselect: '4',
    range_multipleselect: ['F3134', 'I3006', 'I3007', 'I3008', 'I3010', 'I3012', 'I3027', 'I3034', 'O3010'],
    status_radio: '5',
    range_check: ['F3134', 'I3006', 'I3007', 'I3008', 'I3032', 'I3034', 'L2032', 'L2095', 'M2070', 'P1029'],
    establishdate: '2021-09-28T05:02:42.000Z',
  },
  {
    createdAt: '2023-03-13T20:21:27.000Z',
    updatedAt: '2023-07-10T02:25:11.738Z',
    createdById: 7,
    updatedById: null,
    testdataid: 50,
    orgcode: '420050',
    orgname: 'Helen Electronic Inc.',
    address: '891 028 County Rd, Yanqing District',
    phone: '10-8030-8564',
    email: 'linr1955@icloud.com',
    url: 'http://www.rossanna908.xyz/SportsOutdoor',
    staffnum: 149,
    regcapital: 143.496,
    paidcapital: null,
    insuranceratio: 40.72,
    isenable: true,
    status_singleselect: '5',
    range_multipleselect: ['F3134', 'I3006', 'I3007', 'I3011', 'I3027', 'I3032', 'I3034', 'L2032', 'O3010'],
    status_radio: '5',
    range_check: ['I3006', 'I3008', 'I3010', 'I3011', 'I3012', 'I3032', 'M2070', 'O3010'],
    establishdate: '2023-11-27T06:02:42.000Z',
  },
  {
    createdAt: '2023-08-22T10:45:00.000Z',
    updatedAt: '2023-07-06T22:27:32.000Z',
    createdById: 5,
    updatedById: 9,
    testdataid: 2,
    orgcode: '120002',
    orgname: 'Black Technology LLC',
    address: '71 Pedway',
    phone: '312-418-7933',
    email: 'aarimura9@outlook.com',
    url: 'http://auth.patel03.biz/HealthBabyCare',
    staffnum: 95,
    regcapital: 70.7385,
    paidcapital: 96.6588,
    insuranceratio: 74.47,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: [
      'F3134',
      'I3006',
      'I3007',
      'I3008',
      'I3010',
      'I3011',
      'I3012',
      'I3014',
      'I3027',
      'I3032',
      'M2070',
      'O3010',
      'P1029',
    ],
    status_radio: '7',
    range_check: ['F3134', 'I3007', 'I3011', 'I3012', 'I3014', 'I3032', 'I3034', 'L2095', 'P1029'],
    establishdate: '2021-02-18T13:44:21.000Z',
  },
  {
    createdAt: '2023-07-24T13:03:58.000Z',
    updatedAt: '2023-08-15T23:59:06.000Z',
    createdById: 0,
    updatedById: 10,
    testdataid: 76,
    orgcode: '340076',
    orgname: 'Jeff Inc.',
    address: '珊瑚路357号',
    phone: '769-5438-3615',
    email: 'ikk53@gmail.com',
    url: 'http://www.chimingyeung.info/CellPhonesAccessories',
    staffnum: 127,
    regcapital: 12.9611,
    paidcapital: 177.9844,
    insuranceratio: 98.28,
    isenable: true,
    status_singleselect: '2',
    range_multipleselect: ['F3134', 'I3007', 'I3010', 'I3027', 'L2032', 'L2095', 'O3010', 'P1029'],
    status_radio: '4',
    range_check: [
      'I3006',
      'I3007',
      'I3008',
      'I3011',
      'I3012',
      'I3027',
      'I3032',
      'I3034',
      'L2032',
      'L2095',
      'O3010',
      'P1029',
    ],
    establishdate: '2021-02-20T10:36:54.000Z',
  },
  {
    createdAt: '2023-06-01T02:52:18.000Z',
    updatedAt: '2023-07-05T07:00:07.000Z',
    createdById: 0,
    updatedById: 10,
    testdataid: 16,
    orgcode: '410016',
    orgname: '卢系统有限责任公司',
    address: '670 Figueroa Street',
    phone: '213-571-5321',
    email: 'shiry@gmail.com',
    url: 'https://video.yishihan64.xyz/Food',
    staffnum: 63,
    regcapital: 127.0708,
    paidcapital: 25.9392,
    insuranceratio: 89.87,
    isenable: null,
    status_singleselect: '5',
    range_multipleselect: ['I3006', 'I3008', 'I3010', 'I3011', 'I3027', 'I3032', 'L2095'],
    status_radio: '3',
    range_check: ['I3008', 'I3014', 'I3032', 'L2095', 'P1029'],
    establishdate: '2022-03-30T15:28:56.000Z',
  },
  {
    createdAt: '2023-08-09T02:19:55.000Z',
    updatedAt: '2023-07-10T02:25:11.730Z',
    createdById: null,
    updatedById: 3,
    testdataid: 98,
    orgcode: '350098',
    orgname: 'King Brothers Telecommunication Inc.',
    address: '582 Diplomacy Drive',
    phone: '614-785-5349',
    email: 'lindadams922@gmail.com',
    url: 'https://www.luwu.biz/ToolsHomeDecoration',
    staffnum: 146,
    regcapital: 237.6197,
    paidcapital: 295.3728,
    insuranceratio: 92,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: ['I3007', 'I3014', 'I3027', 'L2095', 'O3010'],
    status_radio: '7',
    range_check: ['I3006', 'I3008', 'I3010', 'I3012', 'I3027', 'L2095', 'O3010', 'P1029'],
    establishdate: '2022-09-04T04:56:52.000Z',
  },
  {
    createdAt: '2023-03-03T14:11:21.000Z',
    updatedAt: '2023-06-10T07:54:44.000Z',
    createdById: 10,
    updatedById: 8,
    testdataid: 82,
    orgcode: '650082',
    orgname: '子韬食品有限责任公司',
    address: '889 Alameda Street',
    phone: '213-000-0993',
    email: 'goto1004@icloud.com',
    url: 'https://drive.kucy84.org/FilmSupplies',
    staffnum: 164,
    regcapital: null,
    paidcapital: 75.458,
    insuranceratio: 56.84,
    isenable: true,
    status_singleselect: '5',
    range_multipleselect: ['I3006', 'I3007', 'I3012', 'I3014', 'I3027', 'I3034', 'L2095', 'M2070', 'P1029'],
    status_radio: '1',
    range_check: ['F3134', 'I3010', 'I3012', 'I3014', 'I3027', 'I3032', 'L2032', 'L2095'],
    establishdate: null,
  },
  {
    createdAt: '2023-05-04T12:45:29.000Z',
    updatedAt: '2023-05-29T15:14:39.000Z',
    createdById: 4,
    updatedById: 4,
    testdataid: 32,
    orgcode: '210032',
    orgname: '杰宏贸易有限责任公司',
    address: '695 Fifth Avenue',
    phone: '212-301-3048',
    email: 'tianrui6@mail.com',
    url: 'https://video.fat12.us/AppsGames',
    staffnum: 196,
    regcapital: 253.0279,
    paidcapital: 201.5416,
    insuranceratio: 26.71,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: ['F3134', 'I3006', 'I3007', 'I3014', 'I3032', 'I3034', 'M2070'],
    status_radio: '7',
    range_check: [
      'I3006',
      'I3007',
      'I3008',
      'I3010',
      'I3011',
      'I3012',
      'I3032',
      'I3034',
      'L2032',
      'L2095',
      'O3010',
      'P1029',
    ],
    establishdate: '2022-11-10T12:49:21.000Z',
  },
  {
    createdAt: '2023-06-20T22:11:44.000Z',
    updatedAt: '2023-07-01T22:29:47.000Z',
    createdById: 0,
    updatedById: 6,
    testdataid: 88,
    orgcode: '550088',
    orgname: 'Thomas Engineering LLC',
    address: 'No.24, Dongsan Road, Erxianqiao, Chenghua District',
    phone: '178-8435-2008',
    email: 'rachelj5@icloud.com',
    url: 'http://www.dulan.us/VideoGames',
    staffnum: 49,
    regcapital: 57.0454,
    paidcapital: 79.007,
    insuranceratio: 84.03,
    isenable: null,
    status_singleselect: '7',
    range_multipleselect: ['I3007', 'I3010', 'I3011', 'I3032', 'I3034', 'L2095', 'M2070', 'O3010'],
    status_radio: '1',
    range_check: ['I3007', 'I3012', 'I3027', 'I3034', 'L2095'],
    establishdate: '2022-02-23T22:58:42.000Z',
  },
  {
    createdAt: '2023-03-28T22:03:18.000Z',
    updatedAt: '2023-06-03T12:48:50.000Z',
    createdById: 4,
    updatedById: 8,
    testdataid: 3,
    orgcode: '430003',
    orgname: '彭記电子有限责任公司',
    address: '白云区小坪东路412号',
    phone: '155-6842-3057',
    email: 'ctingfung@mail.com',
    url: 'http://image.yan2007.org/FilmSupplies',
    staffnum: 158,
    regcapital: 15.9472,
    paidcapital: 73.3624,
    insuranceratio: 67.76,
    isenable: true,
    status_singleselect: '6',
    range_multipleselect: ['F3134', 'I3010', 'I3011', 'I3012', 'I3027'],
    status_radio: '5',
    range_check: ['I3006', 'I3007', 'I3008', 'I3012', 'I3027', 'M2070', 'O3010', 'P1029'],
    establishdate: '2023-02-18T04:02:21.000Z',
  },
  {
    createdAt: '2023-06-15T07:46:48.000Z',
    updatedAt: '2023-05-01T19:53:39.000Z',
    createdById: 6,
    updatedById: 3,
    testdataid: 92,
    orgcode: '510092',
    orgname: '邱記贸易有限责任公司',
    address: '930 Rush Street',
    phone: '312-781-8750',
    email: 'xiujia1961@mail.com',
    url: 'http://image.lamwingfat.us/ComputersElectronics',
    staffnum: 76,
    regcapital: null,
    paidcapital: 282.6866,
    insuranceratio: 76.86,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: ['I3006', 'I3007', 'I3011', 'I3027', 'L2032', 'P1029'],
    status_radio: '7',
    range_check: ['F3134', 'I3010', 'I3014', 'I3032', 'L2095', 'P1029'],
    establishdate: '2023-11-18T06:44:55.000Z',
  },
  {
    createdAt: '2023-06-09T17:49:18.000Z',
    updatedAt: '2023-07-10T02:25:11.735Z',
    createdById: null,
    updatedById: null,
    testdataid: 68,
    orgcode: '450068',
    orgname: '汪記有限责任公司',
    address: '坑美十五巷585号',
    phone: '769-680-4019',
    email: null,
    url: 'https://video.lok2000.org/IndustrialScientificSupplies',
    staffnum: 139,
    regcapital: 90.9114,
    paidcapital: 284.0137,
    insuranceratio: 21.71,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: ['I3006', 'I3010', 'I3012', 'I3027', 'I3032', 'L2095', 'O3010'],
    status_radio: '7',
    range_check: ['F3134', 'I3006', 'I3007', 'I3008', 'I3012', 'I3014', 'I3027', 'I3034', 'L2032', 'L2095'],
    establishdate: '2021-11-03T10:14:09.000Z',
  },
  {
    createdAt: '2023-06-16T05:04:59.000Z',
    updatedAt: '2023-06-16T07:08:56.000Z',
    createdById: 6,
    updatedById: 9,
    testdataid: 54,
    orgcode: '310054',
    orgname: "Martinez's Logistic LLC",
    address: '476 Yueliu Rd, Fangshan District',
    phone: '10-971-0570',
    email: 'yogoto@hotmail.com',
    url: 'https://www.kimuram2.co.jp/CDsVinyl',
    staffnum: 132,
    regcapital: 80.6528,
    paidcapital: 87.4258,
    insuranceratio: 87.58,
    isenable: true,
    status_singleselect: '7',
    range_multipleselect: ['I3006', 'I3010', 'I3011', 'I3032', 'I3034', 'L2032', 'L2095', 'M2070', 'O3010', 'P1029'],
    status_radio: '3',
    range_check: ['I3006', 'I3007', 'I3008', 'I3012', 'I3032', 'L2095', 'M2070', 'O3010', 'P1029'],
    establishdate: '2021-08-11T08:08:33.000Z',
  },
  {
    createdAt: '2023-07-17T20:41:03.000Z',
    updatedAt: '2023-06-24T05:16:44.000Z',
    createdById: 5,
    updatedById: 9,
    testdataid: 93,
    orgcode: '350093',
    orgname: '安琪有限责任公司',
    address: '712 Huaxia St, Jinghua Shangquan',
    phone: '138-0406-7670',
    email: 'zhennanm@yahoo.com',
    url: 'http://drive.akinamori803.cn/Food',
    staffnum: 186,
    regcapital: 37.836,
    paidcapital: 53.0391,
    insuranceratio: 74.62,
    isenable: true,
    status_singleselect: '2',
    range_multipleselect: ['I3006', 'I3012', 'I3014', 'I3027', 'O3010'],
    status_radio: '2',
    range_check: ['I3007', 'I3010', 'I3011', 'I3012', 'I3034', 'M2070', 'O3010'],
    establishdate: '2023-06-28T09:10:55.000Z',
  },
  {
    createdAt: '2023-05-11T11:07:26.000Z',
    updatedAt: '2023-06-20T19:16:26.000Z',
    createdById: 9,
    updatedById: 9,
    testdataid: 13,
    orgcode: '310013',
    orgname: 'Evans Brothers LLC',
    address: '黄浦区淮海中路683号',
    phone: '164-7632-7777',
    email: 'aokon@gmail.com',
    url: 'http://image.shawm9.us/CellPhonesAccessories',
    staffnum: 97,
    regcapital: 182.859,
    paidcapital: 105.8903,
    insuranceratio: 63.85,
    isenable: true,
    status_singleselect: '3',
    range_multipleselect: ['F3134', 'I3006', 'I3008', 'I3012', 'I3014', 'L2032', 'O3010'],
    status_radio: '5',
    range_check: ['I3008', 'I3010', 'I3012', 'L2095'],
    establishdate: '2021-09-05T13:35:37.000Z',
  },
  {
    createdAt: '2023-03-10T11:27:35.000Z',
    updatedAt: '2023-06-28T11:16:47.000Z',
    createdById: 6,
    updatedById: 4,
    testdataid: 91,
    orgcode: '240091',
    orgname: '郑顾问有限责任公司',
    address: '872 Canal Street',
    phone: '212-005-7205',
    email: 'rink4@mail.com',
    url: 'https://image.jimenezt7.jp/Others',
    staffnum: 167,
    regcapital: 288.5176,
    paidcapital: 33.3404,
    insuranceratio: 55.87,
    isenable: true,
    status_singleselect: '3',
    range_multipleselect: ['I3011', 'I3027', 'I3034', 'L2032'],
    status_radio: '8',
    range_check: ['I3008', 'I3011', 'I3034', 'L2032', 'L2095', 'M2070', 'O3010'],
    establishdate: null,
  },
  {
    createdAt: '2023-08-11T17:06:24.000Z',
    updatedAt: '2023-06-24T22:14:32.000Z',
    createdById: 3,
    updatedById: 7,
    testdataid: 17,
    orgcode: '620017',
    orgname: 'Patrick Technology LLC',
    address: '505 Ridgewood Road',
    phone: '330-035-6552',
    email: 'nommis58@icloud.com',
    url: 'http://auth.suitsuki304.net/Handcrafts',
    staffnum: 196,
    regcapital: 163.7404,
    paidcapital: 14.0622,
    insuranceratio: 25.51,
    isenable: null,
    status_singleselect: '3',
    range_multipleselect: null,
    status_radio: '2',
    range_check: ['F3134', 'I3006', 'I3011', 'I3012', 'I3014', 'I3027', 'I3034', 'L2032', 'M2070', 'O3010', 'P1029'],
    establishdate: '2022-08-06T12:16:45.000Z',
  },
  {
    createdAt: '2023-03-09T02:32:46.000Z',
    updatedAt: '2023-07-02T05:29:43.000Z',
    createdById: 3,
    updatedById: 6,
    testdataid: 46,
    orgcode: '220046',
    orgname: "Schmidt's Inc.",
    address: '503 2nd Zhongshan Road, Yuexiu District',
    phone: '20-272-7867',
    email: 'ylla@outlook.com',
    url: 'https://www.seikokudo.cn/Books',
    staffnum: 131,
    regcapital: 259.642,
    paidcapital: 172.6541,
    insuranceratio: 38.83,
    isenable: true,
    status_singleselect: '3',
    range_multipleselect: ['I3006', 'I3007', 'I3010', 'I3012', 'I3027', 'I3034', 'L2095', 'M2070', 'P1029'],
    status_radio: '7',
    range_check: ['I3010', 'I3011', 'I3012', 'I3014', 'I3027', 'I3032', 'I3034', 'M2070'],
    establishdate: '2023-06-16T08:57:56.000Z',
  },
  {
    createdAt: '2023-04-02T02:35:24.000Z',
    updatedAt: '2023-07-22T21:27:03.000Z',
    createdById: 7,
    updatedById: 5,
    testdataid: 48,
    orgcode: '530048',
    orgname: "Long's LLC",
    address: '成华区双庆路872号',
    phone: '28-2897-5944',
    email: 'koonwy@outlook.com',
    url: 'https://auth.zj2017.org/Beauty',
    staffnum: 22,
    regcapital: 81.2412,
    paidcapital: 89.2826,
    insuranceratio: 22.58,
    isenable: true,
    status_singleselect: '8',
    range_multipleselect: ['I3008', 'I3011', 'I3012', 'I3034', 'L2032'],
    status_radio: '1',
    range_check: ['F3134', 'I3006', 'I3008', 'I3011', 'I3012', 'I3034', 'L2032', 'M2070', 'O3010'],
    establishdate: '2023-08-06T15:31:45.000Z',
  },
  {
    createdAt: '2023-08-06T02:38:26.000Z',
    updatedAt: '2023-07-01T01:03:02.000Z',
    createdById: 4,
    updatedById: 6,
    testdataid: 99,
    orgcode: '310099',
    orgname: 'Jamie Consultants LLC',
    address: '罗湖区清水河一路156号',
    phone: '755-7525-5040',
    email: 'ikki922@icloud.com',
    url: 'http://video.haki.info/ArtsHandicraftsSewing',
    staffnum: 140,
    regcapital: null,
    paidcapital: 234.484,
    insuranceratio: 55.76,
    isenable: null,
    status_singleselect: '1',
    range_multipleselect: ['F3134', 'I3010', 'I3011', 'I3012', 'I3014', 'I3027', 'I3032', 'M2070', 'O3010'],
    status_radio: '1',
    range_check: [
      'I3007',
      'I3010',
      'I3011',
      'I3014',
      'I3027',
      'I3032',
      'I3034',
      'L2032',
      'L2095',
      'M2070',
      'O3010',
      'P1029',
    ],
    establishdate: '2022-10-17T10:43:37.000Z',
  },
  {
    createdAt: '2023-06-12T16:34:38.000Z',
    updatedAt: '2023-08-03T11:31:10.000Z',
    createdById: 8,
    updatedById: 4,
    testdataid: 22,
    orgcode: '120022',
    orgname: '子韬有限责任公司',
    address: '442 Tangyuan Street 5th Alley, Airport Road, Baiyun',
    phone: '168-8185-4258',
    email: 'anqihu5@gmail.com',
    url: 'http://auth.ontang10.xyz/CDsVinyl',
    staffnum: 150,
    regcapital: 191.8842,
    paidcapital: 282.3322,
    insuranceratio: 37.26,
    isenable: true,
    status_singleselect: '5',
    range_multipleselect: ['F3134', 'I3007', 'I3014', 'I3034', 'M2070', 'P1029'],
    status_radio: '8',
    range_check: [
      'I3006',
      'I3007',
      'I3008',
      'I3010',
      'I3011',
      'I3012',
      'I3014',
      'I3027',
      'I3032',
      'L2032',
      'L2095',
      'M2070',
      'O3010',
      'P1029',
    ],
    establishdate: '2022-05-31T12:34:11.000Z',
  },
  {
    createdAt: '2023-04-26T08:00:47.000Z',
    updatedAt: '2023-06-26T22:25:14.000Z',
    createdById: 7,
    updatedById: 7,
    testdataid: 97,
    orgcode: '120097',
    orgname: '嘉伦有限责任公司',
    address: '天河区大信商圈大信南路342号',
    phone: '165-5165-2865',
    email: 'gonzalez1221@gmail.com',
    url: 'https://auth.kwansw901.biz/HouseholdKitchenAppliances',
    staffnum: 34,
    regcapital: 38.0234,
    paidcapital: 180.0297,
    insuranceratio: 10.93,
    isenable: true,
    status_singleselect: '5',
    range_multipleselect: ['I3008', 'I3010', 'I3011', 'I3014', 'I3034', 'L2095', 'M2070', 'O3010', 'P1029'],
    status_radio: '6',
    range_check: ['F3134', 'I3007', 'I3011', 'I3012', 'I3014', 'I3027', 'L2032', 'M2070', 'O3010', 'P1029'],
    establishdate: null,
  },
  {
    createdAt: '2023-04-20T07:20:03.000Z',
    updatedAt: '2023-05-22T03:28:57.000Z',
    createdById: 7,
    updatedById: 3,
    testdataid: 67,
    orgcode: '240067',
    orgname: 'Betty Telecommunication Inc.',
    address: '351 Figueroa Street',
    phone: '213-187-2091',
    email: 'marthaferguson6@gmail.com',
    url: 'https://www.taniguchihina1110.us/SportsOutdoor',
    staffnum: 58,
    regcapital: 188.0791,
    paidcapital: 127.5241,
    insuranceratio: 36.21,
    isenable: null,
    status_singleselect: '5',
    range_multipleselect: ['I3006', 'I3010', 'I3014', 'I3027', 'L2032', 'O3010'],
    status_radio: '1',
    range_check: ['F3134', 'I3006', 'I3007', 'I3008', 'I3011', 'L2032', 'L2095', 'O3010'],
    establishdate: '2023-01-11T04:04:18.000Z',
  },
];
export default module.exports = {
  appendJsonCollectionName,
  builtinGeneral,
  builtinFile,
  builtinExpression,
  generalWithNoRelationalFields,
  generalWithNoRelationalFieldsData,
};
