import React, { useContext, useEffect, useRef } from 'react';
import {
  GeneralSchemaDesigner,
  gridRowColWrap,
  SchemaSettingsBlockTitleItem,
  SchemaSettingsDivider,
  SchemaSettingsItem,
  SchemaSettingsRemove,
  useAPIClient,
  useCollection_deprecated,
  useDataSource,
  useDesignable,
} from '@tachybase/client';
import { useField, useFieldSchema } from '@tachybase/schema';

import { Empty, Result, Spin, Typography } from 'antd';
import { ErrorBoundary } from 'react-error-boundary';

import { ChartDataContext } from '../block/ChartDataProvider';
import { useChart } from '../chart/group';
import { ChartConfigContext } from '../configure';
import { useData, useFieldsWithAssociation, useFieldTransformer } from '../hooks';
import { useChartsTranslation } from '../locale';
import { createRendererSchema, getField } from '../utils';
import { ChartRendererContext } from './ChartRendererProvider';

const { Paragraph, Text } = Typography;

export const ChartRenderer: React.FC & {
  Designer: React.FC;
} = (props) => {
  const { t } = useChartsTranslation();
  const ctx = useContext(ChartRendererContext);
  const { config, transform, dataSource, collection, service, data: _data } = ctx;
  const fields = useFieldsWithAssociation(dataSource, collection);
  const data = useData(_data, dataSource, collection);
  const general = config?.general || {};
  const advanced = config?.advanced || {};
  const api = useAPIClient();
  const chart = useChart(config?.chartType);
  const locale = api.auth.getLocale();
  const transformers = useFieldTransformer(transform, locale);
  const chartProps = chart?.getProps({
    data,
    general,
    advanced,
    fieldProps: Object.keys(data[0] || {}).reduce((props, name) => {
      if (!props[name]) {
        const field = getField(fields, name.split('.'));
        const transformer = transformers[name];
        props[name] = { label: field?.label || name, transformer, interface: field?.interface };
      }
      return props;
    }, {}),
    ctx,
  });
  const C = chart?.Component;

  if (!chart) {
    return <Empty image={Empty.PRESENTED_IMAGE_SIMPLE} description={t('Please configure chart')} />;
  }
  if (!(data && data.length) && !service.loading) {
    return <Empty image={Empty.PRESENTED_IMAGE_SIMPLE} description={t('No data')} />;
  }

  return (
    <Spin spinning={service.loading}>
      <ErrorBoundary
        onError={(error) => {
          window?.Sentry?.captureException(error);
          console.error(error);
        }}
        FallbackComponent={ErrorFallback}
      >
        <C {...chartProps} />
      </ErrorBoundary>
    </Spin>
  );
};

ChartRenderer.Designer = function Designer() {
  const { t } = useChartsTranslation();
  const { setVisible, setCurrent } = useContext(ChartConfigContext);
  const { removeChart } = useContext(ChartDataContext);
  const { service } = useContext(ChartRendererContext);
  const field = useField();
  const schema = useFieldSchema();
  const { insertAdjacent } = useDesignable();
  const dataSource = useDataSource();
  const { name, title } = useCollection_deprecated();
  return (
    <GeneralSchemaDesigner disableInitializer title={title || name}>
      <SchemaSettingsItem
        title="Configure"
        key="configure"
        onClick={async () => {
          setCurrent({ schema, field, dataSource: dataSource.key, collection: name, service, data: service.data });
          setVisible(true);
        }}
      >
        {t('Configure')}
      </SchemaSettingsItem>
      <SchemaSettingsItem
        title="Duplicate"
        key="duplicate"
        onClick={() => insertAdjacent('afterEnd', gridRowColWrap(createRendererSchema(schema?.['x-decorator-props'])))}
      >
        {t('Duplicate')}
      </SchemaSettingsItem>
      <SchemaSettingsBlockTitleItem />
      <SchemaSettingsDivider />
      <SchemaSettingsRemove
        // removeParentsIfNoChildren
        breakRemoveOn={{
          'x-component': 'ChartV2Block',
        }}
        confirm={{
          onOk: () => {
            removeChart(schema['x-uid']);
          },
        }}
      />
    </GeneralSchemaDesigner>
  );
};

const ErrorFallback = ({ error }) => {
  const { t } = useChartsTranslation();

  return (
    <div style={{ backgroundColor: 'white' }}>
      <Result status="error" title={t('Render Failed')} subTitle={t('Please check the configuration.')}>
        <Paragraph copyable>
          <Text type="danger" style={{ whiteSpace: 'pre-line', textAlign: 'center' }}>
            {error.message}
          </Text>
        </Paragraph>
      </Result>
    </div>
  );
};
