import React, { useCallback, useContext } from 'react';
import {
  DataBlockInitializer,
  DEFAULT_DATA_SOURCE_KEY,
  SchemaInitializer,
  SchemaInitializerItem,
  useACLRoleContext,
  useSchemaInitializer,
  useSchemaInitializerItem,
} from '@tachybase/client';
import { uid } from '@tachybase/schema';

import { BarChartOutlined, LineChartOutlined } from '@ant-design/icons';

import { ChartConfigContext } from '../configure';
import { FilterBlockInitializer } from '../filter';
import { lang } from '../locale';

const ChartInitializer = () => {
  const { setVisible, setCurrent } = useContext(ChartConfigContext);
  const { parseAction } = useACLRoleContext();
  const itemConfig = useSchemaInitializerItem();
  const filter = useCallback(
    (item) => {
      const params = parseAction(`${item.name}:list`);
      return params;
    },
    [parseAction],
  );

  return (
    <DataBlockInitializer
      {...itemConfig}
      filter={filter}
      filterDataSource={(ds) => {
        return ds.key === DEFAULT_DATA_SOURCE_KEY || ds.getOptions().isDBInstance;
      }}
      icon={<BarChartOutlined />}
      componentType={'Chart'}
      onCreateBlockSchema={async ({ item }) => {
        setCurrent({
          schema: {},
          field: null,
          collection: item.name,
          dataSource: item.dataSource,
          service: null,
          data: undefined,
        });
        setVisible(true);
      }}
    />
  );
};

export const chartInitializers = new SchemaInitializer({
  name: 'charts:addBlock',
  icon: 'PlusOutlined',
  title: '{{t("Add block")}}',
  items: [
    {
      name: 'chart',
      title: lang('Chart'),
      Component: ChartInitializer,
    },
    {
      name: 'otherBlocks',
      type: 'itemGroup',
      title: lang('Other blocks'),
      children: [
        {
          name: 'filter',
          title: lang('Filter'),
          Component: FilterBlockInitializer,
        },
      ],
    },
  ],
});

export const ChartV2BlockInitializer: React.FC = () => {
  const itemConfig = useSchemaInitializerItem();
  const { insert } = useSchemaInitializer();
  return (
    <SchemaInitializerItem
      {...itemConfig}
      icon={<LineChartOutlined />}
      onClick={() => {
        insert({
          type: 'void',
          'x-component': 'CardItem',
          'x-component-props': {
            name: 'charts',
          },
          'x-designer': 'ChartV2BlockDesigner',
          properties: {
            [uid()]: {
              type: 'void',
              'x-component': 'Grid',
              'x-decorator': 'ChartV2Block',
              'x-initializer': 'charts:addBlock',
            },
          },
        });
      }}
    />
  );
};
