import React, { useCallback } from 'react';
import { css } from '@tachybase/client';

import { InputNumber, Select } from 'antd';
import { Cron } from 'react-js-cron';

import { useTranslation } from '../locale';
import CronZhCN from './locale/Cron.zh-CN';

const languages = {
  'zh-CN': CronZhCN,
};

const RepeatOptions = [
  { value: 'none', text: 'No repeat' },
  { value: 60_000, text: 'By minute', unitText: 'Minutes' },
  { value: 3600_000, text: 'By hour', unitText: 'Hours' },
  { value: 86400_000, text: 'By day', unitText: 'Days' },
  { value: 604800_000, text: 'By week', unitText: 'Weeks' },
  // { value: 18144_000_000, text: 'By 30 days' },
  { value: 'cron', text: 'Advanced' },
];

function getNumberOption(v) {
  const opts = RepeatOptions.filter((option) => !isNaN(+option.value)).reverse() as any[];
  return opts.find((item) => !(v % item.value));
}

function getRepeatTypeValue(v) {
  if (!v) {
    return 'none';
  }
  if (v && !isNaN(+v)) {
    const option = getNumberOption(v);
    return option ? option.value : 'none';
  }
  if (typeof v === 'string') {
    return 'cron';
  }
}

function CommonRepeatField({ value, onChange }) {
  const { t } = useTranslation();
  const option = getNumberOption(value);

  return (
    <InputNumber
      value={value / option.value}
      onChange={(v) => onChange(v * option.value)}
      min={1}
      addonBefore={t('Every')}
      addonAfter={t(option.unitText)}
      className="auto-width"
    />
  );
}

export function RepeatField({ value = null, onChange }) {
  const { t } = useTranslation();
  const typeValue = getRepeatTypeValue(value);
  const onTypeChange = useCallback(
    (v) => {
      if (v === 'none') {
        onChange(null);
        return;
      }
      if (v === 'cron') {
        onChange('0 * * * * *');
        return;
      }
      onChange(v);
    },
    [onChange],
  );

  const locale = languages[localStorage.getItem('TACHYBASE_LOCALE') || 'en-US'];

  return (
    <fieldset
      className={css`
        display: flex;
        flex-direction: ${typeValue === 'cron' ? 'column' : 'row'};
        align-items: flex-start;
        gap: 0.5em;

        .react-js-cron {
          width: 100%;
          padding: 0.5em 0.5em 0 0.5em;
          border: 1px dashed #ccc;

          .react-js-cron-field {
            margin-bottom: 0.5em;

            > span {
              margin: 0 0.5em 0 0;
            }

            > .react-js-cron-select {
              margin: 0 0.5em 0 0;
            }
          }

          .react-js-cron-week-days {
            > span {
              min-width: 2em;
            }
          }
        }
      `}
    >
      <Select
        value={typeValue}
        onChange={onTypeChange}
        className="auto-width"
        options={
          RepeatOptions.map((option) => ({
            value: option.value,
            label: t(option.text),
          })) as any
        }
      ></Select>
      {typeValue && !isNaN(+typeValue) ? <CommonRepeatField value={value} onChange={onChange} /> : null}
      {typeValue === 'cron' ? (
        <Cron
          value={value.trim().split(/\s+/).slice(1).join(' ')}
          setValue={(v) => onChange(`0 ${v}`)}
          clearButton={false}
          locale={locale}
        />
      ) : null}
    </fieldset>
  );
}
