import { SupabaseClient } from "@supabase/supabase-js";
import { cache } from "react";

export const getUser = cache(async (supabase: SupabaseClient) => {
  const {
    data: { user },
  } = await supabase.auth.getUser();
  return user;
});

export const getSubscriptions = cache(async (supabase: SupabaseClient) => {
  try {
    const { data: subscription } = await supabase
      .from("subscriptions")
      .select("*, prices(*, products(*))")
      .in("status", ["trialing", "active"])
      .throwOnError();
    return subscription;
  } catch (error) {
    console.error("Error:", error);
    return null;
  }
});

export const getProducts = cache(async (supabase: SupabaseClient) => {
  const { data: products, error } = await supabase
    .from("products")
    .select("*, prices(*)")
    .eq("active", true)
    .eq("prices.active", true)
    .order("metadata->index")
    .order("unit_amount", { referencedTable: "prices" });
  if (error) {
    console.log(error.message);
  }
  products?.sort((a, b) => {
    const priceA = a.prices?.[0]?.unit_amount || 0;
    const priceB = b.prices?.[0]?.unit_amount || 0;
    return priceA - priceB;
  });
  return products ?? [];
});

export const getUserDetails = cache(async (supabase: SupabaseClient) => {
  const { data: userDetails } = await supabase
    .from("users")
    .select("*")
    .single();
  return userDetails;
});

export const setUserQuotas = cache(
  async (supabase: SupabaseClient, id: string, quota: number) => {
    try {
      const { error } = await supabase
        .from("users")
        .update({ write_gpt4_quota: quota })
        .eq("id", id);
      if (error) throw error;
    } catch (error) {
      console.error("Error:", error);
      return null;
    }
  },
);
