module Ant.Button exposing
    ( Button
    , button, onClick, ButtonType(..), withType, withIcon, ButtonSize(..), disabled, withHtmlType, HtmlButtonType(..)
    , toHtml
    )

{-| Button component

@docs Button


# Customizing the Button

@docs button, onClick, ButtonType, withType, withIcon, ButtonSize, disabled, withHtmlType, HtmlButtonType

@docs toHtml

-}

{-
   A NOTE TO DEVELOPERS.

   This component is themable.

   See styles at src/Ant/Button/Css.elm
-}

import Ant.Css.Common exposing (..)
import Ant.Icons as Icon exposing (Icon)
import Css exposing (..)
import Html exposing (Html)
import Html.Styled as H exposing (fromUnstyled, text, toUnstyled)
import Html.Styled.Attributes as A exposing (css)
import Html.Styled.Events as Events


{-| The type of the button
-}
type ButtonType
    = Primary
    | Default
    | Dashed
    | Text
    | Link


{-| Determines the size of the button
-}
type ButtonSize
    = Large
    | DefaultSize
    | Small


{-| The "type" attribute of a HTML button as defined in the button spec:

<https://developer.mozilla.org/en-US/docs/Web/HTML/Element/button>

-}
type HtmlButtonType
    = Button_
    | Reset
    | Submit


type alias Options msg =
    { type_ : ButtonType
    , htmlType : HtmlButtonType
    , size : ButtonSize
    , disabled : Bool
    , loading : Bool
    , href : Maybe String
    , onClick : Maybe msg
    , icon : Maybe (Icon msg)

    -- size : Size (Small, Medium, Large)
    -- etc etc
    }


defaultOptions : Options msg
defaultOptions =
    { type_ = Default
    , htmlType = Button_
    , size = DefaultSize
    , disabled = False
    , loading = False
    , href = Nothing
    , onClick = Nothing
    , icon = Nothing
    }


{-| Represents a button component
-}
type Button msg
    = Button (Options msg) String


{-| Create a Button component.

    button "Click Me!"
        |> toHtml

-}
button : String -> Button msg
button label =
    Button defaultOptions label


{-| Change the default type of the Button

    button "submit"
        |> withType Dashed
        |> toHtml

-}
withType : ButtonType -> Button msg -> Button msg
withType buttonType (Button options label) =
    let
        newOptions =
            { options | type_ = buttonType }
    in
    Button newOptions label


{-| Add an icon to the button

    button "Search"
        |> withIcon searchOutlined
        |> toHtml

-}
withIcon : Icon msg -> Button msg -> Button msg
withIcon icon (Button options label) =
    let
        newOptions =
            { options | icon = Just icon }
    in
    Button newOptions label


{-| Specify the "type" attribute of the button.
-}
withHtmlType : HtmlButtonType -> Button msg -> Button msg
withHtmlType htmlType (Button opts label) =
    let
        newOpts =
            { opts
                | htmlType = htmlType
            }
    in
    Button newOpts label


{-| Make your button emit messages. By default, clicking a button does nothing.

    button "submit"
        |> onClick FinalCheckoutFormSubmitted
        |> toHtml

-}
onClick : msg -> Button msg -> Button msg
onClick msg (Button opts label) =
    let
        newOpts =
            { opts | onClick = Just msg }
    in
    Button newOpts label


{-| Make the button disabled. If you have a `onClick` event registered, it will not be fired.

    button "You can't click this"
        |> onClick Logout
        |> disabled True
        |> toHtml

-}
disabled : Bool -> Button msg -> Button msg
disabled disabled_ (Button opts label) =
    let
        newOpts =
            { opts | disabled = disabled_ }
    in
    Button newOpts label


iconToHtml : Icon msg -> H.Html msg
iconToHtml icon =
    icon
        |> Icon.toHtml
        |> fromUnstyled


{-| Turn your Button into Html msg
-}
toHtml : Button msg -> Html msg
toHtml (Button options label) =
    let
        animatedButtonTypes =
            [ Default
            , Primary
            , Dashed
            ]

        animationAttribute =
            if List.member options.type_ animatedButtonTypes then
                [ A.class "elm-antd__animated_btn" ]

            else
                []

        onClickAttribute =
            case options.onClick of
                Just msg ->
                    [ Events.onClick msg ]

                Nothing ->
                    []

        htmlButtonType =
            case options.htmlType of
                Button_ ->
                    A.type_ "button"

                Submit ->
                    A.type_ "submit"

                Reset ->
                    A.type_ "reset"

        buttonTypeClassName =
            case options.type_ of
                Default ->
                    btnDefaultClass

                Primary ->
                    btnPrimaryClass

                Dashed ->
                    btnDashedClass

                Text ->
                    btnTextClass

                Link ->
                    btnLinkClass

        commonAttributes =
            [ A.class btnClass
            , A.class buttonTypeClassName
            , A.disabled options.disabled
            , htmlButtonType
            ]

        attributes =
            commonAttributes ++ onClickAttribute ++ animationAttribute

        iconContent =
            case options.icon of
                Nothing ->
                    H.span [] []

                Just icon ->
                    H.span
                        [ css
                            [ marginRight (px 8)
                            , position relative
                            , top (px 2)
                            ]
                        ]
                        [ iconToHtml icon ]
    in
    toUnstyled
        (H.button
            attributes
            [ iconContent
            , H.span [] [ text label ]
            ]
        )
