package sootup.codepropertygraph.benchmark;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Tag;
import org.junit.jupiter.api.Test;
import sootup.codepropertygraph.BenchmarkTestSuiteBase;
import sootup.codepropertygraph.ast.AstCreator;
import sootup.codepropertygraph.cdg.CdgCreator;
import sootup.codepropertygraph.cfg.CfgCreator;
import sootup.codepropertygraph.cpg.CpgCreator;
import sootup.codepropertygraph.ddg.DdgCreator;
import sootup.codepropertygraph.propertygraph.PropertyGraph;
import sootup.core.model.SootMethod;
import sootup.core.signatures.MethodSignature;
import sootup.core.types.ClassType;

@Tag("Java8")
public class CpgBenchmarkTest extends BenchmarkTestSuiteBase {
  private final ClassType IfElseStatement = getClassType("IfElseStatement");
  private final ClassType TryCatchFinally = getClassType("TryCatchFinally");
  private final ClassType SwitchCaseStatement = getClassType("SwitchCaseStatement");
  private final ClassType WhileLoop = getClassType("WhileLoop");

  private CpgCreator cpgCreator;

  @BeforeEach
  public void setUp() {
    cpgCreator =
        new CpgCreator(new AstCreator(), new CfgCreator(), new CdgCreator(), new DdgCreator());
  }

  @Test
  public void testCpgForIfStatement() {
    runTest(
        "ifStatement",
        IfElseStatement,
        "int",
        Collections.singletonList("int"),
        getExpectedDotGraphForIfStatement());
  }

  @Test
  public void testCpgForIfElseStatement() {
    runTest(
        "ifElseStatement",
        IfElseStatement,
        "int",
        Collections.singletonList("int"),
        getExpectedDotGraphForIfElseStatement());
  }

  @Test
  public void testCpgForIfElseIfStatement() {
    runTest(
        "ifElseIfStatement",
        IfElseStatement,
        "int",
        Collections.singletonList("int"),
        getExpectedDotGraphForIfElseIfStatement());
  }

  @Test
  public void testCpgForIfElseCascadingStatement() {
    runTest(
        "ifElseCascadingStatement",
        IfElseStatement,
        "int",
        Collections.singletonList("int"),
        getExpectedDotGraphForIfElseCascadingStatement());
  }

  @Test
  public void testCpgForIfElseCascadingElseIfStatement() {
    runTest(
        "ifElseCascadingElseIfStatement",
        IfElseStatement,
        "int",
        Collections.singletonList("int"),
        getExpectedDotGraphForIfElseCascadingElseIfStatement());
  }

  @Test
  public void testCpgForIfElseCascadingElseIfInElseStatement() {
    runTest(
        "ifElseCascadingElseIfInElseStatement",
        IfElseStatement,
        "int",
        Collections.singletonList("int"),
        getExpectedDotGraphForIfElseCascadingElseIfInElseStatement());
  }

  @Test
  public void testCpgForTryCatch() {
    runTest(
        "tryCatch",
        TryCatchFinally,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForTryCatch());
  }

  @Test
  public void testCpgForTryCatchNested() {
    runTest(
        "tryCatchNested",
        TryCatchFinally,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForTryCatchNested());
  }

  @Test
  public void testCpgForTryCatchFinallyNested() {
    runTest(
        "tryCatchFinallyNested",
        TryCatchFinally,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForTryCatchFinallyNested());
  }

  @Test
  public void testCpgForTryCatchFinallyNestedInFinally() {
    runTest(
        "tryCatchFinallyNestedInFinally",
        TryCatchFinally,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForTryCatchFinallyNestedInFinally());
  }

  @Test
  public void testCpgForTryCatchFinallyCombined() {
    runTest(
        "tryCatchFinallyCombined",
        TryCatchFinally,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForTryCatchFinallyCombined());
  }

  @Test
  public void testCpgForTryCatchFinallyNestedInCatch() {
    runTest(
        "tryCatchFinallyNestedInCatch",
        TryCatchFinally,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForTryCatchFinallyNestedInCatch());
  }

  @Test
  public void testCpgForTryCatchFinally() {
    runTest(
        "tryCatchFinally",
        TryCatchFinally,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForTryCatchFinally());
  }

  @Test
  public void testCpgForTryCatchNestedInCatch() {
    runTest(
        "tryCatchNestedInCatch",
        TryCatchFinally,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForTryCatchNestedInCatch());
  }

  @Test
  public void testCpgForTryCatchCombined() {
    runTest(
        "tryCatchCombined",
        TryCatchFinally,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForTryCatchCombined());
  }

  @Test
  public void testCpgForSwitchCaseGroupedTargetsDefault() {
    runTest(
        "switchCaseGroupedTargetsDefault",
        SwitchCaseStatement,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForSwitchCaseGroupedTargetsDefault());
  }

  @Test
  public void testCpgForSwitchWithSwitch() {
    runTest(
        "switchWithSwitch",
        SwitchCaseStatement,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForSwitchWithSwitch());
  }

  @Test
  public void testCpgForSwitchCaseStatementInt() {
    runTest(
        "switchCaseStatementInt",
        SwitchCaseStatement,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForSwitchCaseStatementInt());
  }

  @Test
  public void testCpgForSwitchCaseGroupedTargets() {
    runTest(
        "switchCaseGroupedTargets",
        SwitchCaseStatement,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForSwitchCaseGroupedTargets());
  }

  @Test
  public void testCpgForSwitchCaseStatementEnum() {
    runTest(
        "switchCaseStatementEnum",
        SwitchCaseStatement,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForSwitchCaseStatementEnum());
  }

  @Test
  public void testCpgForSwitchCaseStatementCaseIncludingIf() {
    runTest(
        "switchCaseStatementCaseIncludingIf",
        SwitchCaseStatement,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForSwitchCaseStatementCaseIncludingIf());
  }

  @Test
  public void testCpgForSwitchCaseWithoutDefault() {
    runTest(
        "switchCaseWithoutDefault",
        SwitchCaseStatement,
        "void",
        Collections.emptyList(),
        getExpectedDotGraphForSwitchCaseWithoutDefault());
  }

  @Test
  public void testCpgForWhileLoop() {
    runTest(
        "whileLoop", WhileLoop, "void", Collections.emptyList(), getExpectedDotGraphForWhileLoop());
  }

  private void runTest(
      String methodName,
      ClassType classType,
      String returnType,
      List<String> parameters,
      String expectedDotGraph) {
    MethodSignature methodSignature =
        getMethodSignature(classType, methodName, returnType, parameters);
    Optional<? extends SootMethod> optionalMethod = getMinimalTestSuiteMethod(methodSignature);
    assertTrue(optionalMethod.isPresent(), "Method should be present");

    SootMethod method = optionalMethod.get();
    PropertyGraph graph = cpgCreator.createCpg(method);

    String actualDotGraph = normalizeDotGraph(graph.toDotGraph());
    String expectedNormalizedDotGraph = normalizeDotGraph(expectedDotGraph);

    assertEquals(
        expectedNormalizedDotGraph, actualDotGraph, "DOT graph should match the expected output");
  }

  private String normalizeDotGraph(String dotGraph) {
    return dotGraph.replaceAll("\\s+", "").replaceAll("[\\r\\n]+", "");
  }

  private String getExpectedDotGraphForIfStatement() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"0\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"1\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"42\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"@parameter0: int\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"@this: IfElseStatement\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"7\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"8\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"9\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"10\" [label=\"if l1 &gt;= 42\", fillcolor=\"lightblue\"];\n"
        + "	\"11\" [label=\"ifStatement\", fillcolor=\"white\"];\n"
        + "	\"12\" [label=\"int\", fillcolor=\"lightgray\"];\n"
        + "	\"13\" [label=\"int\", fillcolor=\"lightgray\"];\n"
        + "	\"14\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"15\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"16\" [label=\"l1 := @parameter0: int\", fillcolor=\"lightblue\"];\n"
        + "	\"17\" [label=\"l1 &gt;= 42\", fillcolor=\"white\"];\n"
        + "	\"18\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"19\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"20\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"21\" [label=\"l2 = 0\", fillcolor=\"lightblue\"];\n"
        + "	\"22\" [label=\"l2 = 1\", fillcolor=\"lightblue\"];\n"
        + "	\"23\" [label=\"return l2\", fillcolor=\"lightblue\"];\n"
        + "	\"24\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"this := @this: IfElseStatement\", fillcolor=\"lightblue\"];\n"
        + "	\"10\" -> \"17\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"22\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"10\" -> \"22\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"10\" -> \"23\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"11\" -> \"13\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"6\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"7\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"9\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"10\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"16\" -> \"14\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"21\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"16\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"15\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"3\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"10\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"21\" -> \"18\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"23\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"22\" -> \"19\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"22\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"22\" -> \"23\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"22\" -> \"23\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"23\" -> \"20\"[label=\"ast_op\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"16\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"25\" -> \"24\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"10\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"16\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"21\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"22\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"23\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"25\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"8\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"12\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForIfElseStatement() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"0\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"1\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"2\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"42\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"@parameter0: int\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"@this: IfElseStatement\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"8\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"9\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"10\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"11\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"12\" [label=\"if l1 &gt;= 42\", fillcolor=\"lightblue\"];\n"
        + "	\"13\" [label=\"ifElseStatement\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"int\", fillcolor=\"lightgray\"];\n"
        + "	\"15\" [label=\"int\", fillcolor=\"lightgray\"];\n"
        + "	\"16\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"17\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"18\" [label=\"l1 := @parameter0: int\", fillcolor=\"lightblue\"];\n"
        + "	\"19\" [label=\"l1 &gt;= 42\", fillcolor=\"white\"];\n"
        + "	\"20\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"21\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"22\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"24\" [label=\"l2 = 0\", fillcolor=\"lightblue\"];\n"
        + "	\"25\" [label=\"l2 = 1\", fillcolor=\"lightblue\"];\n"
        + "	\"26\" [label=\"l2 = 2\", fillcolor=\"lightblue\"];\n"
        + "	\"27\" [label=\"return l2\", fillcolor=\"lightblue\"];\n"
        + "	\"28\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"this := @this: IfElseStatement\", fillcolor=\"lightblue\"];\n"
        + "	\"10\" -> \"14\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"27\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"12\" -> \"11\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"12\" -> \"19\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"25\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"12\" -> \"25\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"12\" -> \"26\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"12\" -> \"26\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"13\" -> \"10\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"13\" -> \"15\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"13\" -> \"7\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"13\" -> \"8\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"12\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"18\" -> \"16\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"24\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"19\" -> \"17\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"19\" -> \"4\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"12\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"24\" -> \"20\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"11\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"25\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"21\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"27\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"26\" -> \"22\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"26\" -> \"27\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"26\" -> \"27\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"26\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"23\"[label=\"ast_op\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"18\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"29\" -> \"28\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"6\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"11\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"12\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"18\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"24\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"25\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"26\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"27\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"29\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"8\" -> \"9\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForIfElseIfStatement() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"0\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"1\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"123\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"2\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"3\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"42\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"@parameter0: int\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"@this: IfElseStatement\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"10\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"11\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"12\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"13\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"14\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"15\" [label=\"if l1 &lt;= 123\", fillcolor=\"lightblue\"];\n"
        + "	\"16\" [label=\"if l1 &gt;= 42\", fillcolor=\"lightblue\"];\n"
        + "	\"17\" [label=\"ifElseIfStatement\", fillcolor=\"white\"];\n"
        + "	\"18\" [label=\"int\", fillcolor=\"lightgray\"];\n"
        + "	\"19\" [label=\"int\", fillcolor=\"lightgray\"];\n"
        + "	\"20\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"21\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"22\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"l1 := @parameter0: int\", fillcolor=\"lightblue\"];\n"
        + "	\"24\" [label=\"l1 &lt;= 123\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"l1 &gt;= 42\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"28\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"30\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"31\" [label=\"l2 = 0\", fillcolor=\"lightblue\"];\n"
        + "	\"32\" [label=\"l2 = 1\", fillcolor=\"lightblue\"];\n"
        + "	\"33\" [label=\"l2 = 2\", fillcolor=\"lightblue\"];\n"
        + "	\"34\" [label=\"l2 = 3\", fillcolor=\"lightblue\"];\n"
        + "	\"35\" [label=\"return l2\", fillcolor=\"lightblue\"];\n"
        + "	\"36\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"37\" [label=\"this := @this: IfElseStatement\", fillcolor=\"lightblue\"];\n"
        + "	\"10\" -> \"11\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"18\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"13\" -> \"35\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"14\" -> \"35\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"15\" -> \"14\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"15\" -> \"24\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"33\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"15\" -> \"33\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"15\" -> \"34\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"15\" -> \"34\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"16\" -> \"13\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"16\" -> \"15\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"16\" -> \"15\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"16\" -> \"25\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"32\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"16\" -> \"32\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"10\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"12\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"19\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"9\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"15\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"23\" -> \"16\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"23\" -> \"20\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"31\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"7\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"22\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"3\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"21\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"6\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"16\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"31\" -> \"26\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"13\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"32\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"27\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"35\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"33\" -> \"14\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"33\" -> \"28\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"33\" -> \"35\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"33\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"34\" -> \"29\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"34\" -> \"35\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"34\" -> \"35\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"34\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"30\"[label=\"ast_op\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"23\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"37\" -> \"36\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"8\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"13\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"14\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"15\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"16\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"23\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"31\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"32\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"33\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"34\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"35\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"37\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForIfElseCascadingStatement() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"0\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"11\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"12\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"3\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"42\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"42\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"@parameter0: int\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"@this: IfElseStatement\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"10\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"11\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"12\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"13\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"14\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"15\" [label=\"if l1 &gt;= 42\", fillcolor=\"lightblue\"];\n"
        + "	\"16\" [label=\"if l1 &gt;= 42\", fillcolor=\"lightblue\"];\n"
        + "	\"17\" [label=\"ifElseCascadingStatement\", fillcolor=\"white\"];\n"
        + "	\"18\" [label=\"int\", fillcolor=\"lightgray\"];\n"
        + "	\"19\" [label=\"int\", fillcolor=\"lightgray\"];\n"
        + "	\"20\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"21\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"22\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"l1 := @parameter0: int\", fillcolor=\"lightblue\"];\n"
        + "	\"24\" [label=\"l1 &gt;= 42\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"l1 &gt;= 42\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"28\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"30\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"31\" [label=\"l2 = 0\", fillcolor=\"lightblue\"];\n"
        + "	\"32\" [label=\"l2 = 11\", fillcolor=\"lightblue\"];\n"
        + "	\"33\" [label=\"l2 = 12\", fillcolor=\"lightblue\"];\n"
        + "	\"34\" [label=\"l2 = 3\", fillcolor=\"lightblue\"];\n"
        + "	\"35\" [label=\"return l2\", fillcolor=\"lightblue\"];\n"
        + "	\"36\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"37\" [label=\"this := @this: IfElseStatement\", fillcolor=\"lightblue\"];\n"
        + "	\"10\" -> \"11\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"18\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"13\" -> \"35\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"14\" -> \"35\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"15\" -> \"16\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"15\" -> \"16\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"15\" -> \"24\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"34\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"15\" -> \"34\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"16\" -> \"13\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"16\" -> \"14\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"16\" -> \"25\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"32\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"16\" -> \"32\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"16\" -> \"33\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"16\" -> \"33\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"10\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"12\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"19\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"9\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"15\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"23\" -> \"16\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"23\" -> \"20\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"31\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"7\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"21\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"5\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"22\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"6\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"15\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"31\" -> \"26\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"13\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"32\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"27\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"35\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"33\" -> \"14\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"33\" -> \"28\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"33\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"33\" -> \"35\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"34\" -> \"29\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"34\" -> \"35\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"34\" -> \"35\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"34\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"30\"[label=\"ast_op\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"23\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"37\" -> \"36\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"8\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"13\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"14\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"15\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"16\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"23\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"31\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"32\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"33\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"34\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"35\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"37\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForIfElseCascadingElseIfStatement() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"0\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"11\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"12\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"123\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"13\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"2\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"42\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"42\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"@parameter0: int\", fillcolor=\"white\"];\n"
        + "	\"10\" [label=\"@this: IfElseStatement\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"12\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"13\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"14\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"15\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"16\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"17\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"18\" [label=\"if l1 &lt;= 123\", fillcolor=\"lightblue\"];\n"
        + "	\"19\" [label=\"if l1 &gt;= 42\", fillcolor=\"lightblue\"];\n"
        + "	\"20\" [label=\"if l1 &gt;= 42\", fillcolor=\"lightblue\"];\n"
        + "	\"21\" [label=\"ifElseCascadingElseIfStatement\", fillcolor=\"white\"];\n"
        + "	\"22\" [label=\"int\", fillcolor=\"lightgray\"];\n"
        + "	\"23\" [label=\"int\", fillcolor=\"lightgray\"];\n"
        + "	\"24\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"28\" [label=\"l1 := @parameter0: int\", fillcolor=\"lightblue\"];\n"
        + "	\"29\" [label=\"l1 &lt;= 123\", fillcolor=\"white\"];\n"
        + "	\"30\" [label=\"l1 &gt;= 42\", fillcolor=\"white\"];\n"
        + "	\"31\" [label=\"l1 &gt;= 42\", fillcolor=\"white\"];\n"
        + "	\"32\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"33\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"34\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"35\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"36\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"37\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"38\" [label=\"l2 = 0\", fillcolor=\"lightblue\"];\n"
        + "	\"39\" [label=\"l2 = 11\", fillcolor=\"lightblue\"];\n"
        + "	\"40\" [label=\"l2 = 12\", fillcolor=\"lightblue\"];\n"
        + "	\"41\" [label=\"l2 = 13\", fillcolor=\"lightblue\"];\n"
        + "	\"42\" [label=\"l2 = 2\", fillcolor=\"lightblue\"];\n"
        + "	\"43\" [label=\"return l2\", fillcolor=\"lightblue\"];\n"
        + "	\"44\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"45\" [label=\"this := @this: IfElseStatement\", fillcolor=\"lightblue\"];\n"
        + "	\"11\" -> \"15\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"16\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"17\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"18\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"19\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"20\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"28\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"38\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"39\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"40\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"41\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"42\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"43\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"45\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"13\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"22\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"43\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"16\" -> \"43\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"43\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"16\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"18\" -> \"17\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"18\" -> \"29\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"40\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"18\" -> \"40\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"41\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"18\" -> \"41\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"19\" -> \"20\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"19\" -> \"20\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"19\" -> \"30\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"19\" -> \"42\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"19\" -> \"42\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"15\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"20\" -> \"18\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"20\" -> \"18\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"31\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"20\" -> \"39\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"20\" -> \"39\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"21\" -> \"11\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"12\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"14\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"23\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"28\" -> \"18\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"28\" -> \"19\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"28\" -> \"20\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"28\" -> \"24\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"28\" -> \"38\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"28\" -> \"9\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"27\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"4\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"30\" -> \"25\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"30\" -> \"7\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"26\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"8\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"19\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"38\" -> \"32\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"15\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"39\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"33\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"43\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"40\" -> \"16\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"40\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"40\" -> \"34\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"40\" -> \"43\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"41\" -> \"17\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"41\" -> \"35\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"41\" -> \"43\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"41\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"42\" -> \"36\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"42\" -> \"43\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"42\" -> \"43\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"42\" -> \"6\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"43\" -> \"37\"[label=\"ast_op\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"45\" -> \"10\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"45\" -> \"28\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"45\" -> \"44\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForIfElseCascadingElseIfInElseStatement() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"0\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"1\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"123\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"21\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"22\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"23\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"42\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"42\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"@parameter0: int\", fillcolor=\"white\"];\n"
        + "	\"10\" [label=\"@this: IfElseStatement\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"12\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"13\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"14\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"15\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"16\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"17\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"18\" [label=\"if l1 &lt;= 123\", fillcolor=\"lightblue\"];\n"
        + "	\"19\" [label=\"if l1 &gt;= 42\", fillcolor=\"lightblue\"];\n"
        + "	\"20\" [label=\"if l1 &gt;= 42\", fillcolor=\"lightblue\"];\n"
        + "	\"21\" [label=\"ifElseCascadingElseIfInElseStatement\", fillcolor=\"white\"];\n"
        + "	\"22\" [label=\"int\", fillcolor=\"lightgray\"];\n"
        + "	\"23\" [label=\"int\", fillcolor=\"lightgray\"];\n"
        + "	\"24\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"28\" [label=\"l1 := @parameter0: int\", fillcolor=\"lightblue\"];\n"
        + "	\"29\" [label=\"l1 &lt;= 123\", fillcolor=\"white\"];\n"
        + "	\"30\" [label=\"l1 &gt;= 42\", fillcolor=\"white\"];\n"
        + "	\"31\" [label=\"l1 &gt;= 42\", fillcolor=\"white\"];\n"
        + "	\"32\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"33\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"34\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"35\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"36\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"37\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"38\" [label=\"l2 = 0\", fillcolor=\"lightblue\"];\n"
        + "	\"39\" [label=\"l2 = 1\", fillcolor=\"lightblue\"];\n"
        + "	\"40\" [label=\"l2 = 21\", fillcolor=\"lightblue\"];\n"
        + "	\"41\" [label=\"l2 = 22\", fillcolor=\"lightblue\"];\n"
        + "	\"42\" [label=\"l2 = 23\", fillcolor=\"lightblue\"];\n"
        + "	\"43\" [label=\"return l2\", fillcolor=\"lightblue\"];\n"
        + "	\"44\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"45\" [label=\"this := @this: IfElseStatement\", fillcolor=\"lightblue\"];\n"
        + "	\"11\" -> \"15\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"16\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"17\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"18\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"19\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"20\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"28\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"38\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"39\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"40\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"41\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"42\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"43\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"45\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"13\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"22\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"43\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"16\" -> \"43\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"43\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"17\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"18\" -> \"29\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"41\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"18\" -> \"41\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"42\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"18\" -> \"42\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"19\" -> \"15\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"19\" -> \"20\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"19\" -> \"20\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"19\" -> \"30\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"19\" -> \"39\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"19\" -> \"39\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"16\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"20\" -> \"18\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"20\" -> \"18\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"31\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"20\" -> \"40\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"20\" -> \"40\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"21\" -> \"11\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"12\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"14\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"23\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"28\" -> \"18\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"28\" -> \"19\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"28\" -> \"20\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"28\" -> \"24\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"28\" -> \"38\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"28\" -> \"9\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"27\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"3\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"30\" -> \"25\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"30\" -> \"7\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"26\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"8\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"19\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"38\" -> \"32\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"15\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"39\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"33\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"43\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"40\" -> \"16\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"40\" -> \"34\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"40\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"40\" -> \"43\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"41\" -> \"17\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"41\" -> \"35\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"41\" -> \"43\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"41\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"42\" -> \"36\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"42\" -> \"43\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"42\" -> \"43\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"42\" -> \"6\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"43\" -> \"37\"[label=\"ast_op\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"45\" -> \"10\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"45\" -> \"28\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"45\" -> \"44\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForTryCatch() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"\\\"try\\\"\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"$stack3\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"$stack3\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"$stack3 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"7\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"$stack4 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"10\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"12\" [label=\"$stack5 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"13\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"15\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"16\" [label=\"@this: TryCatchFinally\", fillcolor=\"white\"];\n"
        + "	\"17\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"18\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"19\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"20\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"21\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"22\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"24\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"l1 = \\\"\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"28\" [label=\"l1 = \\\"catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"29\" [label=\"l1 = \\\"try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"30\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"31\" [label=\"l2 = $stack4\", fillcolor=\"lightblue\"];\n"
        + "	\"32\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"33\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"34\" [label=\"this := @this: TryCatchFinally\", fillcolor=\"lightblue\"];\n"
        + "	\"35\" [label=\"tryCatch\", fillcolor=\"white\"];\n"
        + "	\"36\" [label=\"virtualinvoke $stack3.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"37\" [label=\"virtualinvoke $stack3.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"38\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"39\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"40\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"12\" -> \"10\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"14\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"38\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"12\" -> \"38\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"17\" -> \"12\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"21\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"27\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"28\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"29\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"31\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"32\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"34\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"36\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"38\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"6\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"9\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"19\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"32\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"27\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"22\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"29\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"28\" -> \"12\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"28\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"28\" -> \"25\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"28\" -> \"38\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"29\" -> \"23\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"36\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"29\" -> \"6\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"29\" -> \"9\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"31\" -> \"28\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"31\" -> \"30\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"8\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"34\" -> \"16\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"34\" -> \"27\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"34\" -> \"33\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"17\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"18\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"20\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"40\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"21\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"36\" -> \"37\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"9\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"37\" -> \"24\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"5\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"32\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"38\" -> \"39\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"11\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"26\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"13\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"36\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"6\" -> \"36\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"6\" -> \"4\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"9\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"9\" -> \"15\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"31\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"9\" -> \"31\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"9\" -> \"7\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForTryCatchNested() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"1catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"\\\"1try\\\"\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"\\\"2catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"\\\"2try\\\"\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"$stack3\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"$stack3\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"$stack3 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"9\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"10\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"$stack4 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"12\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"13\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"$stack5 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"15\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"16\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"17\" [label=\"$stack6 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"18\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"19\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"20\" [label=\"$stack7 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"21\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"22\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"$stack8 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"24\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"28\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"30\" [label=\"@this: TryCatchFinally\", fillcolor=\"white\"];\n"
        + "	\"31\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"32\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"33\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"34\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"35\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"36\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"37\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"38\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"39\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"40\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"41\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"42\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"43\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"44\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"45\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"46\" [label=\"l1 = \\\"\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"47\" [label=\"l1 = \\\"1catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"48\" [label=\"l1 = \\\"1try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"49\" [label=\"l1 = \\\"2catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"50\" [label=\"l1 = \\\"2try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"51\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"52\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"53\" [label=\"l2 = $stack5\", fillcolor=\"lightblue\"];\n"
        + "	\"54\" [label=\"l2 = $stack7\", fillcolor=\"lightblue\"];\n"
        + "	\"55\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"56\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"57\" [label=\"this := @this: TryCatchFinally\", fillcolor=\"lightblue\"];\n"
        + "	\"58\" [label=\"tryCatchNested\", fillcolor=\"white\"];\n"
        + "	\"59\" [label=\"virtualinvoke $stack3.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"60\" [label=\"virtualinvoke $stack3.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"61\" [label=\"virtualinvoke $stack4.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"62\" [label=\"virtualinvoke $stack4.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"63\" [label=\"virtualinvoke $stack6.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"64\" [label=\"virtualinvoke $stack6.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"65\" [label=\"virtualinvoke $stack8.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"66\" [label=\"virtualinvoke $stack8.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"67\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"11\" -> \"20\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"11\" -> \"25\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"61\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"11\" -> \"61\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"11\" -> \"9\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"12\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"28\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"53\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"14\" -> \"53\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"17\" -> \"15\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"26\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"63\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"63\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"20\" -> \"14\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"18\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"20\" -> \"29\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"20\" -> \"54\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"54\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"23\" -> \"14\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"21\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"27\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"65\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"65\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"31\" -> \"11\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"14\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"17\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"20\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"23\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"35\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"36\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"46\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"47\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"48\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"49\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"50\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"53\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"54\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"55\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"57\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"59\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"61\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"63\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"65\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"8\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"33\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"14\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"35\" -> \"36\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"36\" -> \"55\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"46\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"46\" -> \"37\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"46\" -> \"48\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"47\" -> \"17\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"47\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"47\" -> \"42\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"47\" -> \"63\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"48\" -> \"14\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"48\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"48\" -> \"38\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"48\" -> \"59\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"48\" -> \"8\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"49\" -> \"14\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"49\" -> \"23\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"49\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"49\" -> \"44\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"49\" -> \"65\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"50\" -> \"11\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"50\" -> \"20\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"50\" -> \"40\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"61\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"53\" -> \"13\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"53\" -> \"47\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"53\" -> \"51\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"54\" -> \"14\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"54\" -> \"19\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"54\" -> \"49\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"54\" -> \"52\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"57\" -> \"30\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"57\" -> \"46\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"57\" -> \"56\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"58\" -> \"31\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"58\" -> \"32\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"58\" -> \"34\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"58\" -> \"67\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"59\" -> \"14\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"59\" -> \"50\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"59\" -> \"60\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"60\" -> \"39\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"60\" -> \"7\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"61\" -> \"20\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"61\" -> \"35\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"61\" -> \"62\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"62\" -> \"10\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"62\" -> \"41\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"63\" -> \"55\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"63\" -> \"64\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"64\" -> \"16\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"64\" -> \"43\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"65\" -> \"14\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"65\" -> \"36\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"65\" -> \"66\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"66\" -> \"22\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"66\" -> \"45\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"8\" -> \"14\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"8\" -> \"24\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"8\" -> \"59\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"8\" -> \"59\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"8\" -> \"6\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForTryCatchFinallyNested() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"1catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"\\\"1finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"\\\"1finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"\\\"1finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"\\\"1try\\\"\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"\\\"2catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"\\\"2try\\\"\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"$stack10\", fillcolor=\"white\"];\n"
        + "	\"10\" [label=\"$stack10\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"$stack10 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"12\" [label=\"$stack11\", fillcolor=\"white\"];\n"
        + "	\"13\" [label=\"$stack11\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"$stack11 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"15\" [label=\"$stack12\", fillcolor=\"white\"];\n"
        + "	\"16\" [label=\"$stack12\", fillcolor=\"white\"];\n"
        + "	\"17\" [label=\"$stack12 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"18\" [label=\"$stack13\", fillcolor=\"white\"];\n"
        + "	\"19\" [label=\"$stack13\", fillcolor=\"white\"];\n"
        + "	\"20\" [label=\"$stack13 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"21\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"22\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"$stack4 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"24\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"$stack5 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"27\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"28\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"$stack6 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"30\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"31\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"32\" [label=\"$stack7 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"33\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"34\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"35\" [label=\"$stack8 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"36\" [label=\"$stack9\", fillcolor=\"white\"];\n"
        + "	\"37\" [label=\"$stack9\", fillcolor=\"white\"];\n"
        + "	\"38\" [label=\"$stack9 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"39\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"40\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"41\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"42\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"43\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"44\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"45\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"46\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"47\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"48\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"49\" [label=\"@this: TryCatchFinally\", fillcolor=\"white\"];\n"
        + "	\"50\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"51\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"52\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"53\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"54\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"55\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"56\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"57\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"58\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"59\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"60\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"61\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"62\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"63\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"64\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"65\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"66\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"67\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"68\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"69\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"70\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"71\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"72\" [label=\"l1 = \\\"\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"73\" [label=\"l1 = \\\"1catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"74\" [label=\"l1 = \\\"1finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"75\" [label=\"l1 = \\\"1finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"76\" [label=\"l1 = \\\"1finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"77\" [label=\"l1 = \\\"1try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"78\" [label=\"l1 = \\\"2catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"79\" [label=\"l1 = \\\"2try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"80\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"81\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"82\" [label=\"l2 = $stack12\", fillcolor=\"lightblue\"];\n"
        + "	\"83\" [label=\"l2 = $stack9\", fillcolor=\"lightblue\"];\n"
        + "	\"84\" [label=\"l3\", fillcolor=\"white\"];\n"
        + "	\"85\" [label=\"l3\", fillcolor=\"white\"];\n"
        + "	\"86\" [label=\"l3 = $stack7\", fillcolor=\"lightblue\"];\n"
        + "	\"87\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"88\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"89\" [label=\"this := @this: TryCatchFinally\", fillcolor=\"lightblue\"];\n"
        + "	\"90\" [label=\"throw l3\", fillcolor=\"lightblue\"];\n"
        + "	\"91\" [label=\"tryCatchFinallyNested\", fillcolor=\"white\"];\n"
        + "	\"92\" [label=\"virtualinvoke $stack10.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"93\" [label=\"virtualinvoke $stack10.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"94\" [label=\"virtualinvoke $stack11.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"95\" [label=\"virtualinvoke $stack11.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"96\" [label=\"virtualinvoke $stack13.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"97\" [label=\"virtualinvoke $stack13.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"98\" [label=\"virtualinvoke $stack4.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"99\" [label=\"virtualinvoke $stack4.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"100\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"101\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"102\" [label=\"virtualinvoke $stack6.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"103\" [label=\"virtualinvoke $stack6.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"104\" [label=\"virtualinvoke $stack8.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"105\" [label=\"virtualinvoke $stack8.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"106\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"100\" -> \"101\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"100\" -> \"17\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"100\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"100\" -> \"54\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"101\" -> \"25\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"101\" -> \"61\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"102\" -> \"103\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"102\" -> \"56\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"103\" -> \"28\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"103\" -> \"71\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"104\" -> \"105\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"104\" -> \"90\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"105\" -> \"34\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"105\" -> \"67\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"11\" -> \"41\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"9\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"92\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"11\" -> \"92\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"14\" -> \"12\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"42\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"94\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"14\" -> \"94\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"17\" -> \"15\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"38\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"48\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"82\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"82\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"20\" -> \"18\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"20\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"38\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"44\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"20\" -> \"96\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"96\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"23\" -> \"21\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"38\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"39\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"98\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"98\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"26\" -> \"100\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"26\" -> \"100\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"26\" -> \"17\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"26\" -> \"24\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"26\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"26\" -> \"40\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"102\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"29\" -> \"102\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"29\" -> \"27\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"45\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"30\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"47\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"86\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"32\" -> \"86\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"35\" -> \"104\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"35\" -> \"104\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"35\" -> \"33\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"43\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"38\" -> \"36\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"46\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"83\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"38\" -> \"83\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"50\" -> \"100\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"102\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"104\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"11\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"14\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"17\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"20\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"23\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"26\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"29\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"32\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"35\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"38\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"54\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"55\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"56\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"72\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"73\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"74\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"75\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"76\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"77\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"78\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"79\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"82\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"83\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"86\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"87\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"89\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"90\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"92\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"94\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"96\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"98\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"51\" -> \"52\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"54\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"54\" -> \"38\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"54\" -> \"76\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"55\" -> \"87\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"56\" -> \"87\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"72\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"72\" -> \"57\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"72\" -> \"77\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"73\" -> \"11\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"73\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"73\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"73\" -> \"62\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"73\" -> \"92\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"74\" -> \"14\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"74\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"74\" -> \"64\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"74\" -> \"94\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"75\" -> \"104\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"75\" -> \"35\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"75\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"75\" -> \"66\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"76\" -> \"102\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"76\" -> \"29\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"76\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"76\" -> \"70\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"77\" -> \"23\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"77\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"77\" -> \"38\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"77\" -> \"58\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"77\" -> \"6\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"77\" -> \"98\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"78\" -> \"20\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"78\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"78\" -> \"38\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"78\" -> \"68\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"7\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"96\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"79\" -> \"100\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"79\" -> \"17\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"79\" -> \"26\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"79\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"79\" -> \"60\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"79\" -> \"8\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"82\" -> \"16\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"82\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"82\" -> \"38\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"82\" -> \"78\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"82\" -> \"81\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"83\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"83\" -> \"37\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"83\" -> \"73\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"83\" -> \"80\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"86\" -> \"31\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"86\" -> \"75\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"86\" -> \"84\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"86\" -> \"90\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"89\" -> \"49\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"89\" -> \"72\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"89\" -> \"88\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"90\" -> \"85\"[label=\"ast_op\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"91\" -> \"106\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"91\" -> \"50\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"91\" -> \"51\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"91\" -> \"53\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"92\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"92\" -> \"74\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"92\" -> \"93\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"93\" -> \"10\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"93\" -> \"63\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"94\" -> \"55\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"94\" -> \"95\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"95\" -> \"13\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"95\" -> \"65\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"96\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"96\" -> \"38\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"96\" -> \"76\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"96\" -> \"97\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"97\" -> \"19\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"97\" -> \"69\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"98\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"98\" -> \"38\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"98\" -> \"79\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"98\" -> \"99\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"99\" -> \"22\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"99\" -> \"59\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForTryCatchFinallyNestedInFinally() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"1catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"\\\"1finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"\\\"1finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"\\\"1finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"\\\"1try\\\"\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"\\\"2catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"\\\"2catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"\\\"2catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"10\" [label=\"\\\"2try\\\"\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"\\\"2try\\\"\", fillcolor=\"white\"];\n"
        + "	\"12\" [label=\"\\\"2try\\\"\", fillcolor=\"white\"];\n"
        + "	\"13\" [label=\"$stack10\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"$stack10\", fillcolor=\"white\"];\n"
        + "	\"15\" [label=\"$stack10 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"16\" [label=\"$stack11\", fillcolor=\"white\"];\n"
        + "	\"17\" [label=\"$stack11\", fillcolor=\"white\"];\n"
        + "	\"18\" [label=\"$stack11 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"19\" [label=\"$stack12\", fillcolor=\"white\"];\n"
        + "	\"20\" [label=\"$stack12\", fillcolor=\"white\"];\n"
        + "	\"21\" [label=\"$stack12 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"22\" [label=\"$stack13\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"$stack13\", fillcolor=\"white\"];\n"
        + "	\"24\" [label=\"$stack13 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"25\" [label=\"$stack14\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"$stack14\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"$stack14 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"28\" [label=\"$stack15\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"$stack15\", fillcolor=\"white\"];\n"
        + "	\"30\" [label=\"$stack15 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"31\" [label=\"$stack16\", fillcolor=\"white\"];\n"
        + "	\"32\" [label=\"$stack16\", fillcolor=\"white\"];\n"
        + "	\"33\" [label=\"$stack16 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"34\" [label=\"$stack17\", fillcolor=\"white\"];\n"
        + "	\"35\" [label=\"$stack17\", fillcolor=\"white\"];\n"
        + "	\"36\" [label=\"$stack17 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"37\" [label=\"$stack18\", fillcolor=\"white\"];\n"
        + "	\"38\" [label=\"$stack18\", fillcolor=\"white\"];\n"
        + "	\"39\" [label=\"$stack18 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"40\" [label=\"$stack19\", fillcolor=\"white\"];\n"
        + "	\"41\" [label=\"$stack19\", fillcolor=\"white\"];\n"
        + "	\"42\" [label=\"$stack19 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"43\" [label=\"$stack20\", fillcolor=\"white\"];\n"
        + "	\"44\" [label=\"$stack20\", fillcolor=\"white\"];\n"
        + "	\"45\" [label=\"$stack20 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"46\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"47\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"48\" [label=\"$stack5 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"49\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"50\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"51\" [label=\"$stack6 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"52\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"53\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"54\" [label=\"$stack7 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"55\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"56\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"57\" [label=\"$stack8 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"58\" [label=\"$stack9\", fillcolor=\"white\"];\n"
        + "	\"59\" [label=\"$stack9\", fillcolor=\"white\"];\n"
        + "	\"60\" [label=\"$stack9 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"61\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"62\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"63\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"64\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"65\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"66\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"67\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"68\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"69\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"70\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"71\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"72\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"73\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"74\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"75\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"76\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"77\" [label=\"@this: TryCatchFinally\", fillcolor=\"white\"];\n"
        + "	\"78\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"79\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"80\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"81\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"82\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"83\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"84\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"85\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"86\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"87\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"88\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"89\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"90\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"91\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"92\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"93\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"94\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"95\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"96\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"97\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"98\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"99\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"100\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"101\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"102\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"103\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"104\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"105\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"106\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"107\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"108\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"109\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"110\" [label=\"l1 = \\\"\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"111\" [label=\"l1 = \\\"1catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"112\" [label=\"l1 = \\\"1finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"113\" [label=\"l1 = \\\"1finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"114\" [label=\"l1 = \\\"1finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"115\" [label=\"l1 = \\\"1try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"116\" [label=\"l1 = \\\"2catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"117\" [label=\"l1 = \\\"2catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"118\" [label=\"l1 = \\\"2catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"119\" [label=\"l1 = \\\"2try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"120\" [label=\"l1 = \\\"2try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"121\" [label=\"l1 = \\\"2try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"122\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"123\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"124\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"125\" [label=\"l2 = $stack13\", fillcolor=\"lightblue\"];\n"
        + "	\"126\" [label=\"l2 = $stack15\", fillcolor=\"lightblue\"];\n"
        + "	\"127\" [label=\"l2 = $stack19\", fillcolor=\"lightblue\"];\n"
        + "	\"128\" [label=\"l3\", fillcolor=\"white\"];\n"
        + "	\"129\" [label=\"l3\", fillcolor=\"white\"];\n"
        + "	\"130\" [label=\"l3 = $stack10\", fillcolor=\"lightblue\"];\n"
        + "	\"131\" [label=\"l4\", fillcolor=\"white\"];\n"
        + "	\"132\" [label=\"l4 = $stack8\", fillcolor=\"lightblue\"];\n"
        + "	\"133\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"134\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"135\" [label=\"this := @this: TryCatchFinally\", fillcolor=\"lightblue\"];\n"
        + "	\"136\" [label=\"throw l3\", fillcolor=\"lightblue\"];\n"
        + "	\"137\" [label=\"tryCatchFinallyNestedInFinally\", fillcolor=\"white\"];\n"
        + "	\"138\" [label=\"virtualinvoke $stack11.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"139\" [label=\"virtualinvoke $stack11.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"140\" [label=\"virtualinvoke $stack12.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"141\" [label=\"virtualinvoke $stack12.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"142\" [label=\"virtualinvoke $stack14.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"143\" [label=\"virtualinvoke $stack14.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"144\" [label=\"virtualinvoke $stack16.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"145\" [label=\"virtualinvoke $stack16.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"146\" [label=\"virtualinvoke $stack17.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"147\" [label=\"virtualinvoke $stack17.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"148\" [label=\"virtualinvoke $stack18.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"149\" [label=\"virtualinvoke $stack18.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"150\" [label=\"virtualinvoke $stack20.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"151\" [label=\"virtualinvoke $stack20.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"152\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"153\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"154\" [label=\"virtualinvoke $stack6.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"155\" [label=\"virtualinvoke $stack6.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"156\" [label=\"virtualinvoke $stack7.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"157\" [label=\"virtualinvoke $stack7.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"158\" [label=\"virtualinvoke $stack9.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"159\" [label=\"virtualinvoke $stack9.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"160\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"110\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"110\" -> \"115\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"110\" -> \"87\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"111\" -> \"144\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"111\" -> \"15\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"111\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"111\" -> \"33\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"111\" -> \"96\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"112\" -> \"154\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"112\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"112\" -> \"51\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"112\" -> \"90\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"113\" -> \"146\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"113\" -> \"36\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"113\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"113\" -> \"98\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"114\" -> \"104\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"114\" -> \"138\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"114\" -> \"18\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"114\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"115\" -> \"15\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"115\" -> \"152\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"115\" -> \"30\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"115\" -> \"48\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"115\" -> \"6\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"115\" -> \"88\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"116\" -> \"150\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"116\" -> \"45\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"116\" -> \"7\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"116\" -> \"94\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"117\" -> \"102\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"117\" -> \"142\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"117\" -> \"27\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"117\" -> \"8\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"118\" -> \"108\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"118\" -> \"158\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"118\" -> \"60\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"118\" -> \"9\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"119\" -> \"10\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"119\" -> \"156\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"119\" -> \"42\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"119\" -> \"54\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"119\" -> \"92\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"120\" -> \"100\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"120\" -> \"11\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"120\" -> \"148\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"120\" -> \"24\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"120\" -> \"39\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"121\" -> \"106\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"121\" -> \"12\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"121\" -> \"140\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"121\" -> \"21\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"121\" -> \"57\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"125\" -> \"117\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"125\" -> \"124\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"125\" -> \"23\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"126\" -> \"111\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"126\" -> \"123\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"126\" -> \"15\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"126\" -> \"29\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"127\" -> \"116\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"127\" -> \"122\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"127\" -> \"41\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"130\" -> \"114\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"130\" -> \"128\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"130\" -> \"136\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"130\" -> \"14\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"132\" -> \"118\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"132\" -> \"131\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"132\" -> \"56\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"135\" -> \"110\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"135\" -> \"134\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"135\" -> \"77\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"136\" -> \"129\"[label=\"ast_op\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"137\" -> \"160\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"137\" -> \"78\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"137\" -> \"79\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"137\" -> \"81\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"138\" -> \"121\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"138\" -> \"139\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"139\" -> \"105\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"139\" -> \"17\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"140\" -> \"141\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"140\" -> \"57\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"140\" -> \"86\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"141\" -> \"107\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"141\" -> \"20\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"142\" -> \"143\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"142\" -> \"85\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"143\" -> \"103\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"143\" -> \"26\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"144\" -> \"113\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"144\" -> \"145\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"144\" -> \"15\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"145\" -> \"32\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"145\" -> \"97\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"146\" -> \"120\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"146\" -> \"147\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"147\" -> \"35\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"147\" -> \"99\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"148\" -> \"149\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"148\" -> \"24\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"148\" -> \"84\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"149\" -> \"101\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"149\" -> \"38\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"13\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"130\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"15\" -> \"130\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"15\" -> \"75\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"150\" -> \"151\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"150\" -> \"83\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"151\" -> \"44\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"151\" -> \"95\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"152\" -> \"112\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"152\" -> \"15\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"152\" -> \"153\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"152\" -> \"30\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"153\" -> \"47\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"153\" -> \"89\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"154\" -> \"119\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"154\" -> \"155\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"155\" -> \"50\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"155\" -> \"91\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"156\" -> \"157\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"156\" -> \"42\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"156\" -> \"82\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"157\" -> \"53\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"157\" -> \"93\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"158\" -> \"136\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"158\" -> \"159\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"159\" -> \"109\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"159\" -> \"59\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"138\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"138\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"18\" -> \"16\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"69\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"140\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"21\" -> \"140\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"21\" -> \"19\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"57\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"21\" -> \"70\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"125\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"24\" -> \"125\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"24\" -> \"22\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"74\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"142\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"27\" -> \"142\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"27\" -> \"25\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"68\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"30\" -> \"126\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"30\" -> \"126\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"30\" -> \"15\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"30\" -> \"28\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"30\" -> \"73\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"33\" -> \"144\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"33\" -> \"144\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"33\" -> \"15\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"33\" -> \"31\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"33\" -> \"65\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"146\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"36\" -> \"146\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"36\" -> \"34\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"66\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"148\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"39\" -> \"148\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"39\" -> \"24\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"39\" -> \"37\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"67\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"42\" -> \"127\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"42\" -> \"127\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"42\" -> \"40\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"42\" -> \"72\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"45\" -> \"150\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"45\" -> \"150\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"45\" -> \"43\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"45\" -> \"64\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"48\" -> \"15\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"48\" -> \"152\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"48\" -> \"152\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"48\" -> \"30\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"48\" -> \"46\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"48\" -> \"61\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"51\" -> \"154\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"51\" -> \"154\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"51\" -> \"49\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"51\" -> \"62\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"54\" -> \"156\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"54\" -> \"156\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"54\" -> \"42\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"54\" -> \"52\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"54\" -> \"63\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"57\" -> \"132\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"57\" -> \"132\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"57\" -> \"55\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"57\" -> \"76\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"60\" -> \"158\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"60\" -> \"158\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"60\" -> \"58\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"60\" -> \"71\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"110\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"111\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"112\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"113\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"114\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"115\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"116\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"117\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"118\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"119\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"120\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"121\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"125\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"126\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"127\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"130\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"132\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"133\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"135\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"136\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"138\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"140\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"142\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"144\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"146\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"148\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"15\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"150\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"152\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"154\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"156\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"158\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"18\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"21\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"24\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"27\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"30\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"33\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"36\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"39\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"42\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"45\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"48\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"51\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"54\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"57\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"60\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"82\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"83\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"84\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"85\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"86\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"79\" -> \"80\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"82\" -> \"133\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"83\" -> \"133\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"84\" -> \"133\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"85\" -> \"133\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"86\" -> \"136\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForTryCatchFinallyCombined() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"\\\"finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"\\\"finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"\\\"finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"\\\"try\\\"\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"$stack10\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"$stack10\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"$stack10 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"10\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"12\" [label=\"$stack4 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"13\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"15\" [label=\"$stack5 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"16\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"17\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"18\" [label=\"$stack6 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"19\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"20\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"21\" [label=\"$stack7 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"22\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"24\" [label=\"$stack8 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"25\" [label=\"$stack9\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"$stack9\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"$stack9 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"28\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"30\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"31\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"32\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"33\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"34\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"35\" [label=\"@this: TryCatchFinally\", fillcolor=\"white\"];\n"
        + "	\"36\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"37\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"38\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"39\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"40\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"41\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"42\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"43\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"44\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"45\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"46\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"47\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"48\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"49\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"50\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"51\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"52\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"53\" [label=\"l1 = \\\"\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"54\" [label=\"l1 = \\\"catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"55\" [label=\"l1 = \\\"finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"56\" [label=\"l1 = \\\"finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"57\" [label=\"l1 = \\\"finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"58\" [label=\"l1 = \\\"try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"59\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"60\" [label=\"l2 = $stack8\", fillcolor=\"lightblue\"];\n"
        + "	\"61\" [label=\"l3\", fillcolor=\"white\"];\n"
        + "	\"62\" [label=\"l3\", fillcolor=\"white\"];\n"
        + "	\"63\" [label=\"l3 = $stack6\", fillcolor=\"lightblue\"];\n"
        + "	\"64\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"65\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"66\" [label=\"this := @this: TryCatchFinally\", fillcolor=\"lightblue\"];\n"
        + "	\"67\" [label=\"throw l3\", fillcolor=\"lightblue\"];\n"
        + "	\"68\" [label=\"tryCatchFinallyCombined\", fillcolor=\"white\"];\n"
        + "	\"69\" [label=\"virtualinvoke $stack10.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"70\" [label=\"virtualinvoke $stack10.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"71\" [label=\"virtualinvoke $stack4.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"72\" [label=\"virtualinvoke $stack4.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"73\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"74\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"75\" [label=\"virtualinvoke $stack7.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"76\" [label=\"virtualinvoke $stack7.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"77\" [label=\"virtualinvoke $stack9.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"78\" [label=\"virtualinvoke $stack9.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"79\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"12\" -> \"10\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"12\" -> \"24\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"12\" -> \"28\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"71\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"12\" -> \"71\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"15\" -> \"13\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"29\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"73\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"15\" -> \"73\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"18\" -> \"16\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"34\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"63\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"63\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"21\" -> \"19\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"32\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"75\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"21\" -> \"75\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"24\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"24\" -> \"22\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"33\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"60\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"24\" -> \"60\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"27\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"27\" -> \"25\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"30\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"77\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"27\" -> \"77\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"36\" -> \"12\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"15\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"18\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"21\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"24\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"27\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"40\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"41\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"53\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"54\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"55\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"56\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"57\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"58\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"60\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"63\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"64\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"66\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"67\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"69\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"71\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"73\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"75\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"77\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"9\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"38\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"40\" -> \"64\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"41\" -> \"64\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"53\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"53\" -> \"42\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"53\" -> \"58\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"54\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"54\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"54\" -> \"27\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"54\" -> \"47\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"54\" -> \"77\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"55\" -> \"15\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"55\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"55\" -> \"45\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"55\" -> \"73\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"56\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"56\" -> \"49\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"56\" -> \"69\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"56\" -> \"9\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"57\" -> \"21\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"57\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"57\" -> \"51\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"57\" -> \"75\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"58\" -> \"12\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"58\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"58\" -> \"24\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"58\" -> \"43\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"58\" -> \"6\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"58\" -> \"71\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"60\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"60\" -> \"23\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"60\" -> \"54\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"60\" -> \"59\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"63\" -> \"17\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"63\" -> \"57\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"63\" -> \"61\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"63\" -> \"67\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"66\" -> \"35\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"66\" -> \"53\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"66\" -> \"65\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"67\" -> \"62\"[label=\"ast_op\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"68\" -> \"36\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"68\" -> \"37\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"68\" -> \"39\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"68\" -> \"79\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"69\" -> \"41\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"69\" -> \"70\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"70\" -> \"50\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"70\" -> \"8\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"71\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"71\" -> \"24\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"71\" -> \"55\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"71\" -> \"72\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"72\" -> \"11\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"72\" -> \"44\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"73\" -> \"40\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"73\" -> \"74\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"74\" -> \"14\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"74\" -> \"46\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"75\" -> \"67\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"75\" -> \"76\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"76\" -> \"20\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"76\" -> \"52\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"77\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"77\" -> \"56\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"77\" -> \"78\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"26\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"48\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"31\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"69\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"9\" -> \"69\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"9\" -> \"7\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForTryCatchFinallyNestedInCatch() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"1catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"\\\"1finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"\\\"1finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"\\\"1finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"\\\"1try\\\"\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"\\\"2catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"\\\"2try\\\"\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"$stack10\", fillcolor=\"white\"];\n"
        + "	\"10\" [label=\"$stack10\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"$stack10 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"12\" [label=\"$stack11\", fillcolor=\"white\"];\n"
        + "	\"13\" [label=\"$stack11\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"$stack11 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"15\" [label=\"$stack12\", fillcolor=\"white\"];\n"
        + "	\"16\" [label=\"$stack12\", fillcolor=\"white\"];\n"
        + "	\"17\" [label=\"$stack12 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"18\" [label=\"$stack13\", fillcolor=\"white\"];\n"
        + "	\"19\" [label=\"$stack13\", fillcolor=\"white\"];\n"
        + "	\"20\" [label=\"$stack13 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"21\" [label=\"$stack14\", fillcolor=\"white\"];\n"
        + "	\"22\" [label=\"$stack14\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"$stack14 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"24\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"$stack5 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"27\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"28\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"$stack6 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"30\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"31\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"32\" [label=\"$stack7 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"33\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"34\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"35\" [label=\"$stack8 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"36\" [label=\"$stack9\", fillcolor=\"white\"];\n"
        + "	\"37\" [label=\"$stack9\", fillcolor=\"white\"];\n"
        + "	\"38\" [label=\"$stack9 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"39\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"40\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"41\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"42\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"43\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"44\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"45\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"46\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"47\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"48\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"49\" [label=\"@this: TryCatchFinally\", fillcolor=\"white\"];\n"
        + "	\"50\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"51\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"52\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"53\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"54\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"55\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"56\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"57\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"58\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"59\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"60\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"61\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"62\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"63\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"64\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"65\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"66\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"67\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"68\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"69\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"70\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"71\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"72\" [label=\"l1 = \\\"\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"73\" [label=\"l1 = \\\"1catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"74\" [label=\"l1 = \\\"1finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"75\" [label=\"l1 = \\\"1finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"76\" [label=\"l1 = \\\"1finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"77\" [label=\"l1 = \\\"1try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"78\" [label=\"l1 = \\\"2catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"79\" [label=\"l1 = \\\"2try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"80\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"81\" [label=\"l2 = $stack12\", fillcolor=\"lightblue\"];\n"
        + "	\"82\" [label=\"l3\", fillcolor=\"white\"];\n"
        + "	\"83\" [label=\"l3 = $stack9\", fillcolor=\"lightblue\"];\n"
        + "	\"84\" [label=\"l4\", fillcolor=\"white\"];\n"
        + "	\"85\" [label=\"l4\", fillcolor=\"white\"];\n"
        + "	\"86\" [label=\"l4 = $stack7\", fillcolor=\"lightblue\"];\n"
        + "	\"87\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"88\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"89\" [label=\"this := @this: TryCatchFinally\", fillcolor=\"lightblue\"];\n"
        + "	\"90\" [label=\"throw l4\", fillcolor=\"lightblue\"];\n"
        + "	\"91\" [label=\"tryCatchFinallyNestedInCatch\", fillcolor=\"white\"];\n"
        + "	\"92\" [label=\"virtualinvoke $stack10.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"93\" [label=\"virtualinvoke $stack10.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"94\" [label=\"virtualinvoke $stack11.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"95\" [label=\"virtualinvoke $stack11.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"96\" [label=\"virtualinvoke $stack13.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"97\" [label=\"virtualinvoke $stack13.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"98\" [label=\"virtualinvoke $stack14.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"99\" [label=\"virtualinvoke $stack14.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"100\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"101\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"102\" [label=\"virtualinvoke $stack6.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"103\" [label=\"virtualinvoke $stack6.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"104\" [label=\"virtualinvoke $stack8.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"105\" [label=\"virtualinvoke $stack8.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"106\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"100\" -> \"101\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"100\" -> \"17\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"100\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"100\" -> \"74\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"101\" -> \"25\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"101\" -> \"59\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"102\" -> \"103\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"102\" -> \"54\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"103\" -> \"28\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"103\" -> \"61\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"104\" -> \"105\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"104\" -> \"90\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"105\" -> \"34\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"105\" -> \"67\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"11\" -> \"44\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"9\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"92\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"11\" -> \"92\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"14\" -> \"12\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"45\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"94\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"14\" -> \"94\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"17\" -> \"15\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"46\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"81\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"81\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"20\" -> \"18\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"20\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"41\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"20\" -> \"96\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"96\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"23\" -> \"21\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"38\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"42\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"98\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"98\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"26\" -> \"100\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"26\" -> \"100\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"26\" -> \"17\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"26\" -> \"24\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"26\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"26\" -> \"39\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"102\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"29\" -> \"102\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"29\" -> \"27\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"40\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"30\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"32\" -> \"47\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"86\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"32\" -> \"86\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"35\" -> \"104\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"35\" -> \"104\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"35\" -> \"33\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"43\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"38\" -> \"36\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"48\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"83\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"38\" -> \"83\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"50\" -> \"100\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"102\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"104\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"11\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"14\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"17\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"20\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"23\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"26\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"29\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"32\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"35\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"38\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"54\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"55\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"56\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"72\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"73\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"74\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"75\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"76\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"77\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"78\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"79\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"81\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"83\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"86\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"87\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"89\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"90\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"92\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"94\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"96\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"98\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"51\" -> \"52\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"54\" -> \"87\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"55\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"55\" -> \"76\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"56\" -> \"87\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"72\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"72\" -> \"57\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"72\" -> \"77\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"73\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"73\" -> \"20\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"73\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"73\" -> \"62\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"73\" -> \"96\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"74\" -> \"102\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"74\" -> \"29\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"74\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"74\" -> \"60\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"75\" -> \"104\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"75\" -> \"35\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"75\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"75\" -> \"66\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"76\" -> \"14\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"76\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"76\" -> \"70\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"76\" -> \"94\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"77\" -> \"100\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"77\" -> \"17\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"77\" -> \"26\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"77\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"77\" -> \"58\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"77\" -> \"6\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"11\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"78\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"78\" -> \"68\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"7\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"92\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"79\" -> \"23\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"79\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"79\" -> \"38\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"79\" -> \"64\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"79\" -> \"8\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"79\" -> \"98\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"81\" -> \"16\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"81\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"81\" -> \"73\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"81\" -> \"80\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"83\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"83\" -> \"37\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"83\" -> \"78\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"83\" -> \"82\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"86\" -> \"31\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"86\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"86\" -> \"75\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"86\" -> \"84\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"86\" -> \"90\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"89\" -> \"49\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"89\" -> \"72\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"89\" -> \"88\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"90\" -> \"85\"[label=\"ast_op\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"91\" -> \"106\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"91\" -> \"50\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"91\" -> \"51\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"91\" -> \"53\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"92\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"92\" -> \"76\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"92\" -> \"93\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"93\" -> \"10\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"93\" -> \"69\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"94\" -> \"56\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"94\" -> \"95\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"95\" -> \"13\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"95\" -> \"71\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"96\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"96\" -> \"79\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"96\" -> \"97\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"97\" -> \"19\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"97\" -> \"63\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"98\" -> \"32\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"98\" -> \"38\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"98\" -> \"55\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"98\" -> \"99\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"99\" -> \"22\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"99\" -> \"65\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForTryCatchFinally() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"\\\"finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"\\\"finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"\\\"finally\\\"\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"\\\"try\\\"\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"$stack10\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"$stack10\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"$stack10 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"10\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"12\" [label=\"$stack4 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"13\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"15\" [label=\"$stack5 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"16\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"17\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"18\" [label=\"$stack6 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"19\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"20\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"21\" [label=\"$stack7 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"22\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"24\" [label=\"$stack8 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"25\" [label=\"$stack9\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"$stack9\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"$stack9 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"28\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"30\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"31\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"32\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"33\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"34\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"35\" [label=\"@this: TryCatchFinally\", fillcolor=\"white\"];\n"
        + "	\"36\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"37\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"38\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"39\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"40\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"41\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"42\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"43\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"44\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"45\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"46\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"47\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"48\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"49\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"50\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"51\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"52\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"53\" [label=\"l1 = \\\"\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"54\" [label=\"l1 = \\\"catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"55\" [label=\"l1 = \\\"finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"56\" [label=\"l1 = \\\"finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"57\" [label=\"l1 = \\\"finally\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"58\" [label=\"l1 = \\\"try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"59\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"60\" [label=\"l2 = $stack8\", fillcolor=\"lightblue\"];\n"
        + "	\"61\" [label=\"l3\", fillcolor=\"white\"];\n"
        + "	\"62\" [label=\"l3\", fillcolor=\"white\"];\n"
        + "	\"63\" [label=\"l3 = $stack6\", fillcolor=\"lightblue\"];\n"
        + "	\"64\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"65\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"66\" [label=\"this := @this: TryCatchFinally\", fillcolor=\"lightblue\"];\n"
        + "	\"67\" [label=\"throw l3\", fillcolor=\"lightblue\"];\n"
        + "	\"68\" [label=\"tryCatchFinally\", fillcolor=\"white\"];\n"
        + "	\"69\" [label=\"virtualinvoke $stack10.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"70\" [label=\"virtualinvoke $stack10.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"71\" [label=\"virtualinvoke $stack4.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"72\" [label=\"virtualinvoke $stack4.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"73\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"74\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"75\" [label=\"virtualinvoke $stack7.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"76\" [label=\"virtualinvoke $stack7.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"77\" [label=\"virtualinvoke $stack9.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"78\" [label=\"virtualinvoke $stack9.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"79\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"12\" -> \"10\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"12\" -> \"24\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"12\" -> \"28\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"71\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"12\" -> \"71\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"15\" -> \"13\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"29\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"73\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"15\" -> \"73\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"18\" -> \"16\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"34\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"63\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"63\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"21\" -> \"19\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"32\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"75\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"21\" -> \"75\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"24\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"24\" -> \"22\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"33\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"60\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"24\" -> \"60\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"27\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"27\" -> \"25\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"30\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"77\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"27\" -> \"77\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"36\" -> \"12\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"15\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"18\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"21\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"24\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"27\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"40\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"41\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"53\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"54\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"55\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"56\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"57\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"58\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"60\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"63\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"64\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"66\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"67\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"69\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"71\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"73\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"75\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"77\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"9\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"38\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"40\" -> \"64\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"41\" -> \"64\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"53\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"53\" -> \"42\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"53\" -> \"58\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"54\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"54\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"54\" -> \"27\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"54\" -> \"47\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"54\" -> \"77\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"55\" -> \"15\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"55\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"55\" -> \"45\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"55\" -> \"73\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"56\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"56\" -> \"49\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"56\" -> \"69\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"56\" -> \"9\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"57\" -> \"21\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"57\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"57\" -> \"51\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"57\" -> \"75\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"58\" -> \"12\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"58\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"58\" -> \"24\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"58\" -> \"43\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"58\" -> \"6\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"58\" -> \"71\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"60\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"60\" -> \"23\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"60\" -> \"54\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"60\" -> \"59\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"63\" -> \"17\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"63\" -> \"57\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"63\" -> \"61\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"63\" -> \"67\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"66\" -> \"35\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"66\" -> \"53\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"66\" -> \"65\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"67\" -> \"62\"[label=\"ast_op\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"68\" -> \"36\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"68\" -> \"37\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"68\" -> \"39\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"68\" -> \"79\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"69\" -> \"41\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"69\" -> \"70\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"70\" -> \"50\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"70\" -> \"8\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"71\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"71\" -> \"24\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"71\" -> \"55\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"71\" -> \"72\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"72\" -> \"11\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"72\" -> \"44\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"73\" -> \"40\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"73\" -> \"74\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"74\" -> \"14\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"74\" -> \"46\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"75\" -> \"67\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"75\" -> \"76\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"76\" -> \"20\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"76\" -> \"52\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"77\" -> \"18\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"77\" -> \"56\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"77\" -> \"78\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"26\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"78\" -> \"48\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"31\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"69\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"9\" -> \"69\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"9\" -> \"7\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForTryCatchNestedInCatch() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"1catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"\\\"1try\\\"\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"\\\"2catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"\\\"2try\\\"\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"$stack4 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"9\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"10\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"$stack5 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"12\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"13\" [label=\"$stack6\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"$stack6 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"15\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"16\" [label=\"$stack7\", fillcolor=\"white\"];\n"
        + "	\"17\" [label=\"$stack7 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"18\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"19\" [label=\"$stack8\", fillcolor=\"white\"];\n"
        + "	\"20\" [label=\"$stack8 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"21\" [label=\"$stack9\", fillcolor=\"white\"];\n"
        + "	\"22\" [label=\"$stack9\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"$stack9 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"24\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"28\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"30\" [label=\"@this: TryCatchFinally\", fillcolor=\"white\"];\n"
        + "	\"31\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"32\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"33\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"34\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"35\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"36\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"37\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"38\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"39\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"40\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"41\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"42\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"43\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"44\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"45\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"46\" [label=\"l1 = \\\"\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"47\" [label=\"l1 = \\\"1catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"48\" [label=\"l1 = \\\"1try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"49\" [label=\"l1 = \\\"2catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"50\" [label=\"l1 = \\\"2try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"51\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"52\" [label=\"l2 = $stack7\", fillcolor=\"lightblue\"];\n"
        + "	\"53\" [label=\"l3\", fillcolor=\"white\"];\n"
        + "	\"54\" [label=\"l3 = $stack5\", fillcolor=\"lightblue\"];\n"
        + "	\"55\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"56\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"57\" [label=\"this := @this: TryCatchFinally\", fillcolor=\"lightblue\"];\n"
        + "	\"58\" [label=\"tryCatchNestedInCatch\", fillcolor=\"white\"];\n"
        + "	\"59\" [label=\"virtualinvoke $stack4.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"60\" [label=\"virtualinvoke $stack4.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"61\" [label=\"virtualinvoke $stack6.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"62\" [label=\"virtualinvoke $stack6.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"63\" [label=\"virtualinvoke $stack8.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"64\" [label=\"virtualinvoke $stack8.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"65\" [label=\"virtualinvoke $stack9.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"66\" [label=\"virtualinvoke $stack9.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"67\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"11\" -> \"29\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"54\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"11\" -> \"54\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"11\" -> \"9\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"12\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"27\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"61\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"14\" -> \"61\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"17\" -> \"15\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"28\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"52\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"52\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"20\" -> \"18\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"20\" -> \"25\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"20\" -> \"63\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"63\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"23\" -> \"11\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"21\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"26\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"65\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"65\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"31\" -> \"11\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"14\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"17\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"20\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"23\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"35\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"36\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"46\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"47\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"48\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"49\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"50\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"52\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"54\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"55\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"57\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"59\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"61\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"63\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"65\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"8\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"33\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"55\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"36\" -> \"55\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"46\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"46\" -> \"37\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"46\" -> \"48\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"47\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"47\" -> \"20\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"47\" -> \"40\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"47\" -> \"63\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"48\" -> \"17\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"48\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"48\" -> \"38\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"48\" -> \"59\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"48\" -> \"8\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"49\" -> \"14\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"49\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"49\" -> \"44\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"49\" -> \"61\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"50\" -> \"11\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"50\" -> \"23\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"50\" -> \"42\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"50\" -> \"65\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"52\" -> \"16\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"52\" -> \"47\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"52\" -> \"51\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"54\" -> \"10\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"54\" -> \"49\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"54\" -> \"53\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"57\" -> \"30\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"57\" -> \"46\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"57\" -> \"56\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"58\" -> \"31\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"58\" -> \"32\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"58\" -> \"34\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"58\" -> \"67\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"59\" -> \"17\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"59\" -> \"35\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"59\" -> \"60\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"60\" -> \"39\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"60\" -> \"7\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"61\" -> \"55\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"61\" -> \"62\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"62\" -> \"13\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"62\" -> \"45\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"63\" -> \"50\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"63\" -> \"64\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"64\" -> \"19\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"64\" -> \"41\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"65\" -> \"11\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"65\" -> \"36\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"65\" -> \"66\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"66\" -> \"22\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"66\" -> \"43\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"8\" -> \"17\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"8\" -> \"24\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"8\" -> \"59\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"8\" -> \"59\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"8\" -> \"6\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForTryCatchCombined() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"catch\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"\\\"try\\\"\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"$stack3\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"$stack3\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"$stack3 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"7\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"$stack4 := @caughtexception\", fillcolor=\"lightblue\"];\n"
        + "	\"10\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"12\" [label=\"$stack5 = &lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"13\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"&lt;java.lang.System: java.io.PrintStream out&gt;\", fillcolor=\"white\"];\n"
        + "	\"15\" [label=\"@caughtexception\", fillcolor=\"white\"];\n"
        + "	\"16\" [label=\"@this: TryCatchFinally\", fillcolor=\"white\"];\n"
        + "	\"17\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"18\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"19\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"20\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"21\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"22\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"24\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"l1 = \\\"\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"28\" [label=\"l1 = \\\"catch\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"29\" [label=\"l1 = \\\"try\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"30\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"31\" [label=\"l2 = $stack4\", fillcolor=\"lightblue\"];\n"
        + "	\"32\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"33\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"34\" [label=\"this := @this: TryCatchFinally\", fillcolor=\"lightblue\"];\n"
        + "	\"35\" [label=\"tryCatchCombined\", fillcolor=\"white\"];\n"
        + "	\"36\" [label=\"virtualinvoke $stack3.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"37\" [label=\"virtualinvoke $stack3.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"38\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"lightblue\"];\n"
        + "	\"39\" [label=\"virtualinvoke $stack5.&lt;java.io.PrintStream: void println(java.lang.String)&gt;(l1)\", fillcolor=\"white\"];\n"
        + "	\"40\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"12\" -> \"10\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"14\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"38\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"12\" -> \"38\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"17\" -> \"12\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"21\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"27\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"28\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"29\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"31\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"32\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"34\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"36\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"38\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"6\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"9\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"19\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"32\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"27\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"22\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"29\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"28\" -> \"12\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"28\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"28\" -> \"25\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"28\" -> \"38\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"29\" -> \"23\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"36\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"29\" -> \"6\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"29\" -> \"9\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"31\" -> \"28\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"31\" -> \"30\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"8\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"34\" -> \"16\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"34\" -> \"27\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"34\" -> \"33\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"17\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"18\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"20\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"40\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"21\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"36\" -> \"37\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"9\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"37\" -> \"24\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"5\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"32\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"38\" -> \"39\"[label=\"ast_invoke\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"11\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"26\"[label=\"ast_arg\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"13\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"36\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"6\" -> \"36\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"6\" -> \"4\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"9\"[label=\"cfg_except\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"9\" -> \"15\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"31\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"9\" -> \"31\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"9\" -> \"7\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForSwitchCaseGroupedTargetsDefault() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"first\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"other\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"\\\"second\\\"\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"8\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"@this: SwitchCaseStatement\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"7\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"8\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"9\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"10\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"11\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"12\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"13\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"l1 = 8\", fillcolor=\"lightblue\"];\n"
        + "	\"15\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"16\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"17\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"18\" [label=\"l2 = \\\"first\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"19\" [label=\"l2 = \\\"other\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"20\" [label=\"l2 = \\\"second\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"21\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"22\" [label=\"switch(l1) \\{     case 1:     case 2:     case 3:     default:  \\}\", fillcolor=\"lightblue\"];\n"
        + "	\"23\" [label=\"switchCaseGroupedTargetsDefault\", fillcolor=\"white\"];\n"
        + "	\"24\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"this := @this: SwitchCaseStatement\", fillcolor=\"lightblue\"];\n"
        + "	\"26\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"10\" -> \"21\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"11\" -> \"21\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"14\" -> \"12\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"22\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"14\" -> \"22\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"14\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"10\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"15\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"19\" -> \"17\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"19\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"19\" -> \"21\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"11\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"16\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"20\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"22\" -> \"10\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"22\" -> \"11\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"22\" -> \"13\"[label=\"ast_switch_key\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"22\" -> \"18\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"22\" -> \"18\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"22\" -> \"19\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"22\" -> \"19\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"22\" -> \"20\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"22\" -> \"20\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"26\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"6\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"7\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"9\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"14\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"25\" -> \"24\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"10\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"11\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"14\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"18\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"19\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"20\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"21\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"22\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"25\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"8\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForSwitchWithSwitch() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"-1\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"11\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"12\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"2\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"2\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"220\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"230\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"240\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"3\", fillcolor=\"white\"];\n"
        + "	\"10\" [label=\"@this: SwitchCaseStatement\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"12\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"13\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"14\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"15\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"16\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"17\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"18\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"19\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"20\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"21\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"22\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"24\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"l1 = 2\", fillcolor=\"lightblue\"];\n"
        + "	\"26\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"28\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"30\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"31\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"32\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"33\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"34\" [label=\"l2 = -1\", fillcolor=\"lightblue\"];\n"
        + "	\"35\" [label=\"l2 = 11\", fillcolor=\"lightblue\"];\n"
        + "	\"36\" [label=\"l2 = 12\", fillcolor=\"lightblue\"];\n"
        + "	\"37\" [label=\"l2 = 2\", fillcolor=\"lightblue\"];\n"
        + "	\"38\" [label=\"l2 = 220\", fillcolor=\"lightblue\"];\n"
        + "	\"39\" [label=\"l2 = 230\", fillcolor=\"lightblue\"];\n"
        + "	\"40\" [label=\"l2 = 240\", fillcolor=\"lightblue\"];\n"
        + "	\"41\" [label=\"l2 = 3\", fillcolor=\"lightblue\"];\n"
        + "	\"42\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"43\" [label=\"switch(l1) \\{     case 10:     case 20:     default:  \\}\", fillcolor=\"lightblue\"];\n"
        + "	\"44\" [label=\"switch(l1) \\{     case 1:     case 2:     case 3:     default:  \\}\", fillcolor=\"lightblue\"];\n"
        + "	\"45\" [label=\"switch(l1) \\{     case 20:     case 30:     case 40:     default:  \\}\", fillcolor=\"lightblue\"];\n"
        + "	\"46\" [label=\"switchWithSwitch\", fillcolor=\"white\"];\n"
        + "	\"47\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"48\" [label=\"this := @this: SwitchCaseStatement\", fillcolor=\"lightblue\"];\n"
        + "	\"49\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"11\" -> \"15\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"16\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"17\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"18\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"19\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"20\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"25\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"34\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"35\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"36\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"37\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"38\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"39\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"40\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"41\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"42\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"43\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"44\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"45\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"48\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"13\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"16\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"16\" -> \"42\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"19\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"19\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"19\" -> \"42\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"42\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"25\" -> \"21\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"43\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"25\" -> \"44\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"25\" -> \"44\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"25\" -> \"45\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"34\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"34\" -> \"33\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"34\" -> \"42\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"35\" -> \"15\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"35\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"26\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"16\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"36\" -> \"27\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"36\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"28\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"45\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"37\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"17\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"38\" -> \"29\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"6\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"18\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"39\" -> \"30\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"39\" -> \"7\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"40\" -> \"19\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"40\" -> \"31\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"40\" -> \"8\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"41\" -> \"20\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"41\" -> \"32\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"41\" -> \"9\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"43\" -> \"15\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"43\" -> \"16\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"43\" -> \"23\"[label=\"ast_switch_key\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"43\" -> \"35\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"43\" -> \"35\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"43\" -> \"36\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"43\" -> \"36\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"44\" -> \"16\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"44\" -> \"19\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"44\" -> \"20\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"44\" -> \"22\"[label=\"ast_switch_key\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"44\" -> \"34\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"44\" -> \"34\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"44\" -> \"37\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"44\" -> \"37\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"44\" -> \"41\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"44\" -> \"41\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"44\" -> \"43\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"44\" -> \"43\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"44\" -> \"45\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"45\" -> \"17\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"45\" -> \"18\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"45\" -> \"19\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"45\" -> \"24\"[label=\"ast_switch_key\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"45\" -> \"38\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"45\" -> \"38\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"45\" -> \"39\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"45\" -> \"39\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"45\" -> \"40\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"45\" -> \"40\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"46\" -> \"11\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"46\" -> \"12\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"46\" -> \"14\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"46\" -> \"49\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"48\" -> \"10\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"48\" -> \"25\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"48\" -> \"47\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForSwitchCaseStatementInt() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"invalid\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"one\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"\\\"three\\\"\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"\\\"two\\\"\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"5\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"@this: SwitchCaseStatement\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"8\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"9\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"10\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"11\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"12\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"13\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"14\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"15\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"16\" [label=\"l1 = 5\", fillcolor=\"lightblue\"];\n"
        + "	\"17\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"18\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"19\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"20\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"21\" [label=\"l2 = \\\"invalid\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"22\" [label=\"l2 = \\\"one\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"23\" [label=\"l2 = \\\"three\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"24\" [label=\"l2 = \\\"two\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"25\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"26\" [label=\"switch(l1) \\{     case 1:     case 2:     case 3:     default:  \\}\", fillcolor=\"lightblue\"];\n"
        + "	\"27\" [label=\"switchCaseStatementInt\", fillcolor=\"white\"];\n"
        + "	\"28\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"this := @this: SwitchCaseStatement\", fillcolor=\"lightblue\"];\n"
        + "	\"30\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"11\" -> \"25\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"12\" -> \"25\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"13\" -> \"25\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"16\" -> \"14\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"26\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"16\" -> \"26\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"16\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"20\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"25\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"22\" -> \"11\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"22\" -> \"17\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"22\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"13\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"19\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"12\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"24\" -> \"18\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"26\" -> \"11\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"26\" -> \"12\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"26\" -> \"13\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"26\" -> \"15\"[label=\"ast_switch_key\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"26\" -> \"21\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"26\" -> \"21\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"26\" -> \"22\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"26\" -> \"22\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"26\" -> \"23\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"26\" -> \"23\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"26\" -> \"24\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"26\" -> \"24\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"27\" -> \"10\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"30\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"7\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"27\" -> \"8\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"16\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"29\" -> \"28\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"6\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"11\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"12\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"13\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"16\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"21\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"22\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"23\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"24\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"25\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"26\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"29\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"8\" -> \"9\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForSwitchCaseGroupedTargets() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"first\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"second\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"7\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"@this: SwitchCaseStatement\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"6\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"7\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"8\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"9\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"10\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"12\" [label=\"l1 = 7\", fillcolor=\"lightblue\"];\n"
        + "	\"13\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"15\" [label=\"l2 = \\\"first\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"16\" [label=\"l2 = \\\"second\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"17\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"18\" [label=\"switch(l1) \\{     case 1:     case 2:     case 3:     default:  \\}\", fillcolor=\"lightblue\"];\n"
        + "	\"19\" [label=\"switchCaseGroupedTargets\", fillcolor=\"white\"];\n"
        + "	\"20\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"21\" [label=\"this := @this: SwitchCaseStatement\", fillcolor=\"lightblue\"];\n"
        + "	\"22\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"12\" -> \"10\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"18\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"12\" -> \"18\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"12\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"13\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"9\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"16\" -> \"14\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"17\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"16\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"11\"[label=\"ast_switch_key\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"15\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"18\" -> \"15\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"16\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"18\" -> \"16\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"17\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"9\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"19\" -> \"22\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"19\" -> \"5\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"19\" -> \"6\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"19\" -> \"8\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"12\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"21\" -> \"20\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"12\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"15\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"16\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"17\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"18\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"21\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"9\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"7\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"17\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForSwitchCaseStatementEnum() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"green\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"\\\"invalid\\\"\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"\\\"red\\\"\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"$stack3\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"$stack3 = &lt;SwitchCaseStatement$1: int[] $SwitchMap$SwitchCaseStatement$Color&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"7\" [label=\"$stack3[$stack4]\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"$stack4\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"$stack4 = virtualinvoke l1.&lt;SwitchCaseStatement$Color: int ordinal()&gt;()\", fillcolor=\"lightblue\"];\n"
        + "	\"10\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"11\" [label=\"$stack5\", fillcolor=\"white\"];\n"
        + "	\"12\" [label=\"$stack5 = $stack3[$stack4]\", fillcolor=\"lightblue\"];\n"
        + "	\"13\" [label=\"&lt;SwitchCaseStatement$1: int[] $SwitchMap$SwitchCaseStatement$Color&gt;\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"&lt;SwitchCaseStatement$Color: SwitchCaseStatement$Color RED&gt;\", fillcolor=\"white\"];\n"
        + "	\"15\" [label=\"@this: SwitchCaseStatement\", fillcolor=\"white\"];\n"
        + "	\"16\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"17\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"18\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"19\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"20\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"21\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"22\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"24\" [label=\"l1 = &lt;SwitchCaseStatement$Color: SwitchCaseStatement$Color RED&gt;\", fillcolor=\"lightblue\"];\n"
        + "	\"25\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"28\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"l2 = \\\"\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"30\" [label=\"l2 = \\\"green\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"31\" [label=\"l2 = \\\"invalid\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"32\" [label=\"l2 = \\\"red\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"33\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"34\" [label=\"switch($stack5) \\{     case 1:     case 2:     default:  \\}\", fillcolor=\"lightblue\"];\n"
        + "	\"35\" [label=\"switchCaseStatementEnum\", fillcolor=\"white\"];\n"
        + "	\"36\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"37\" [label=\"this := @this: SwitchCaseStatement\", fillcolor=\"lightblue\"];\n"
        + "	\"38\" [label=\"virtualinvoke l1.&lt;SwitchCaseStatement$Color: int ordinal()&gt;()\", fillcolor=\"white\"];\n"
        + "	\"39\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"12\" -> \"10\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"12\" -> \"34\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"12\" -> \"34\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"12\" -> \"7\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"12\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"20\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"21\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"24\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"29\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"30\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"31\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"32\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"33\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"34\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"37\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"6\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"9\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"18\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"20\" -> \"33\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"21\" -> \"33\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"24\" -> \"14\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"22\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"29\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"24\" -> \"9\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"29\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"25\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"29\" -> \"6\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"30\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"30\" -> \"21\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"30\" -> \"27\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"28\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"33\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"32\" -> \"20\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"32\" -> \"26\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"34\" -> \"11\"[label=\"ast_switch_key\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"34\" -> \"20\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"34\" -> \"21\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"34\" -> \"30\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"34\" -> \"30\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"34\" -> \"31\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"34\" -> \"31\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"34\" -> \"32\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"34\" -> \"32\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"35\" -> \"16\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"17\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"19\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"39\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"15\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"24\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"37\" -> \"36\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"23\"[label=\"ast_base\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"12\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"6\" -> \"13\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"5\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"9\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"9\" -> \"12\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"9\" -> \"12\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"9\" -> \"38\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"8\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForSwitchCaseStatementCaseIncludingIf() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"-1\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"1\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"11\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"12\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"2\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"2\", fillcolor=\"white\"];\n"
        + "	\"7\" [label=\"3\", fillcolor=\"white\"];\n"
        + "	\"8\" [label=\"666\", fillcolor=\"white\"];\n"
        + "	\"9\" [label=\"@this: SwitchCaseStatement\", fillcolor=\"white\"];\n"
        + "	\"10\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"11\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"12\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"13\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"14\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"15\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"16\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"17\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"18\" [label=\"if l1 != 666\", fillcolor=\"lightblue\"];\n"
        + "	\"19\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"20\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"21\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"22\" [label=\"l1 != 666\", fillcolor=\"white\"];\n"
        + "	\"23\" [label=\"l1 = 2\", fillcolor=\"lightblue\"];\n"
        + "	\"24\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"26\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"27\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"28\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"29\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"30\" [label=\"l2 = -1\", fillcolor=\"lightblue\"];\n"
        + "	\"31\" [label=\"l2 = 1\", fillcolor=\"lightblue\"];\n"
        + "	\"32\" [label=\"l2 = 11\", fillcolor=\"lightblue\"];\n"
        + "	\"33\" [label=\"l2 = 12\", fillcolor=\"lightblue\"];\n"
        + "	\"34\" [label=\"l2 = 2\", fillcolor=\"lightblue\"];\n"
        + "	\"35\" [label=\"l2 = 3\", fillcolor=\"lightblue\"];\n"
        + "	\"36\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"37\" [label=\"switch(l1) \\{     case 1:     case 2:     case 3:     default:  \\}\", fillcolor=\"lightblue\"];\n"
        + "	\"38\" [label=\"switchCaseStatementCaseIncludingIf\", fillcolor=\"white\"];\n"
        + "	\"39\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"40\" [label=\"this := @this: SwitchCaseStatement\", fillcolor=\"lightblue\"];\n"
        + "	\"41\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"10\" -> \"14\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"15\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"16\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"17\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"18\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"23\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"30\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"31\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"32\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"33\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"34\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"35\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"36\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"37\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"40\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"11\" -> \"12\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"36\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"15\" -> \"36\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"16\" -> \"36\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"36\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"14\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"18\" -> \"15\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"18\" -> \"22\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"32\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"18\" -> \"32\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"33\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"18\" -> \"33\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"22\" -> \"21\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"22\" -> \"8\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"18\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"23\" -> \"19\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"37\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"37\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"23\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"30\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"30\" -> \"29\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"30\" -> \"36\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"31\" -> \"18\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"31\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"31\" -> \"24\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"14\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"32\" -> \"25\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"32\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"33\" -> \"15\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"33\" -> \"26\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"33\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"34\" -> \"16\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"34\" -> \"27\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"34\" -> \"6\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"17\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"35\" -> \"28\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"35\" -> \"7\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"16\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"37\" -> \"17\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"37\" -> \"18\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"37\" -> \"20\"[label=\"ast_switch_key\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"37\" -> \"30\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"37\" -> \"30\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"37\" -> \"31\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"37\" -> \"31\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"37\" -> \"34\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"37\" -> \"34\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"37\" -> \"35\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"37\" -> \"35\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"38\" -> \"10\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"11\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"13\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"38\" -> \"41\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"40\" -> \"23\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"40\" -> \"39\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"40\" -> \"9\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForSwitchCaseWithoutDefault() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"\\\"one\\\"\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"\\\"three\\\"\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"\\\"two\\\"\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"6\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"@this: SwitchCaseStatement\", fillcolor=\"white\"];\n"
        + "	\"6\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"7\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"8\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"9\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"10\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"11\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"12\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"13\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"l1 = 6\", fillcolor=\"lightblue\"];\n"
        + "	\"15\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"16\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"17\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"18\" [label=\"l2 = \\\"one\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"19\" [label=\"l2 = \\\"three\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"20\" [label=\"l2 = \\\"two\\\"\", fillcolor=\"lightblue\"];\n"
        + "	\"21\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"22\" [label=\"switch(l1) \\{     case 1:     case 2:     case 3:     default:  \\}\", fillcolor=\"lightblue\"];\n"
        + "	\"23\" [label=\"switchCaseWithoutDefault\", fillcolor=\"white\"];\n"
        + "	\"24\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"25\" [label=\"this := @this: SwitchCaseStatement\", fillcolor=\"lightblue\"];\n"
        + "	\"26\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"10\" -> \"21\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"11\" -> \"21\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"14\" -> \"12\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"14\" -> \"22\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"14\" -> \"22\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"14\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"1\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"10\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"18\" -> \"15\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"19\" -> \"17\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"19\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"19\" -> \"21\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"11\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"20\" -> \"16\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"20\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"22\" -> \"10\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"22\" -> \"11\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"22\" -> \"13\"[label=\"ast_switch_key\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"22\" -> \"18\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"22\" -> \"18\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"22\" -> \"19\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"22\" -> \"19\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"22\" -> \"20\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"22\" -> \"20\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"22\" -> \"21\"[label=\"cfg_switch\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"23\" -> \"26\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"6\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"7\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"23\" -> \"9\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"14\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"25\" -> \"24\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"25\" -> \"5\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"10\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"11\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"14\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"18\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"19\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"20\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"21\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"22\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"25\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"7\" -> \"8\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "}\n";
  }

  private String getExpectedDotGraphForWhileLoop() {
    return "digraph merged_graph {\n"
        + "	rankdir=TB;\n"
        + "	node [style=filled, shape=record];\n"
        + "	edge [style=filled]\n"
        + "	\"1\" [label=\"-1\", fillcolor=\"white\"];\n"
        + "	\"2\" [label=\"0\", fillcolor=\"white\"];\n"
        + "	\"3\" [label=\"10\", fillcolor=\"white\"];\n"
        + "	\"4\" [label=\"@this: WhileLoop\", fillcolor=\"white\"];\n"
        + "	\"5\" [label=\"Body\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"6\" [label=\"Modifiers\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"7\" [label=\"PUBLIC\", fillcolor=\"lightgray\"];\n"
        + "	\"8\" [label=\"Parameters\", fillcolor=\"darkseagreen2\"];\n"
        + "	\"9\" [label=\"goto\", fillcolor=\"lightblue\"];\n"
        + "	\"10\" [label=\"if l1 &lt;= l2\", fillcolor=\"lightblue\"];\n"
        + "	\"11\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"12\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"13\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"14\" [label=\"l1\", fillcolor=\"white\"];\n"
        + "	\"15\" [label=\"l1 + -1\", fillcolor=\"white\"];\n"
        + "	\"16\" [label=\"l1 &lt;= l2\", fillcolor=\"white\"];\n"
        + "	\"17\" [label=\"l1 = 10\", fillcolor=\"lightblue\"];\n"
        + "	\"18\" [label=\"l1 = l1 + -1\", fillcolor=\"lightblue\"];\n"
        + "	\"19\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"20\" [label=\"l2\", fillcolor=\"white\"];\n"
        + "	\"21\" [label=\"l2 = 0\", fillcolor=\"lightblue\"];\n"
        + "	\"22\" [label=\"return\", fillcolor=\"lightblue\"];\n"
        + "	\"23\" [label=\"this\", fillcolor=\"white\"];\n"
        + "	\"24\" [label=\"this := @this: WhileLoop\", fillcolor=\"lightblue\"];\n"
        + "	\"25\" [label=\"void\", fillcolor=\"lightgray\"];\n"
        + "	\"26\" [label=\"whileLoop\", fillcolor=\"white\"];\n"
        + "	\"10\" -> \"16\"[label=\"ast_condition\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"10\" -> \"18\"[label=\"cfg_false\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"10\" -> \"22\"[label=\"cdg_next\", color=\"dodgerblue4\", fontcolor=\"dodgerblue4\"];\n"
        + "	\"10\" -> \"22\"[label=\"cfg_true\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"15\" -> \"1\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"15\" -> \"14\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"12\"[label=\"ast_op1\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"16\" -> \"20\"[label=\"ast_op2\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"10\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"17\" -> \"11\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"17\" -> \"18\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"17\" -> \"21\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"17\" -> \"3\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"10\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"18\" -> \"13\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"15\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"18\" -> \"9\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"21\" -> \"10\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"21\" -> \"10\"[label=\"ddg_next\", color=\"firebrick\", fontcolor=\"firebrick\"];\n"
        + "	\"21\" -> \"19\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"21\" -> \"2\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"17\"[label=\"cfg_next\", color=\"black\", fontcolor=\"black\"];\n"
        + "	\"24\" -> \"23\"[label=\"ast_leftOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"24\" -> \"4\"[label=\"ast_rightOp\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"26\" -> \"25\"[label=\"ast_returnType\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"26\" -> \"5\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"26\" -> \"6\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"26\" -> \"8\"[label=\"ast_params\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"10\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"17\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"18\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"21\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"22\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"24\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"5\" -> \"9\"[label=\"ast_stmts\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"6\" -> \"7\"[label=\"ast_modifiers\", color=\"darkseagreen4\", fontcolor=\"darkseagreen4\"];\n"
        + "	\"9\" -> \"10\"[label=\"cfg_goto\", color=\"black\", fontcolor=\"black\"];\n"
        + "}\n";
  }
}
