/*
 * SLD Editor - The Open Source Java SLD Editor
 *
 * Copyright (C) 2016, SCISYS UK Limited
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.sldeditor.ui.detail.config;

import com.sldeditor.common.undo.UndoActionInterface;
import com.sldeditor.common.undo.UndoEvent;
import com.sldeditor.common.undo.UndoInterface;
import com.sldeditor.common.undo.UndoManager;
import com.sldeditor.common.xml.ui.FieldIdEnum;
import com.sldeditor.ui.detail.BasePanel;
import com.sldeditor.ui.detail.config.symboltype.SymbolTypeConfig;
import com.sldeditor.ui.widgets.FieldPanel;
import com.sldeditor.ui.widgets.ValueComboBox;
import com.sldeditor.ui.widgets.ValueComboBoxData;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import org.apache.log4j.Logger;
import org.geotools.filter.LiteralExpressionImpl;
import org.opengis.filter.expression.Expression;

/**
 * The Class FieldConfigEnum wraps a drop down GUI component and an optional
 * value/attribute/expression drop down,
 *
 * <p>Supports undo/redo functionality.
 *
 * <p>Instantiated by {@link com.sldeditor.ui.detail.config.panelconfig.ReadPanelConfig}
 *
 * @author Robert Ward (SCISYS)
 */
public class FieldConfigEnum extends FieldConfigBase implements UndoActionInterface {

    /** The key list. */
    private List<String> keyList = new ArrayList<>();

    /** The value map. */
    private Map<String, String> valueMap = new HashMap<>();

    /** The combo data map. */
    private Map<String, ValueComboBoxData> comboDataMap = new HashMap<>();

    /** The field map. */
    private Map<Class<?>, Map<FieldIdEnum, Boolean>> fieldMap = new HashMap<>();

    /** The default value. */
    private String defaultValue = "";

    /** The combo box. */
    private ValueComboBox comboBox = null;

    /** The old value obj. */
    private Object oldValueObj = null;

    /** The logger. */
    private static Logger logger = Logger.getLogger(FieldConfigEnum.class);

    /**
     * Instantiates a new field config enum.
     *
     * @param commonData the common data
     */
    public FieldConfigEnum(FieldConfigCommonData commonData) {
        super(commonData);
    }

    /**
     * Adds the value.
     *
     * @param key the key
     * @param value the value
     */
    private void addValue(String key, String value) {
        if ((key == null) || key.isEmpty()) {
            keyList.add("");
            valueMap.put("", value);
        } else {
            keyList.add(key);
            valueMap.put(key, value);
        }
    }

    /** Creates the ui. */
    /*
     * (non-Javadoc)
     *
     * @see com.sldeditor.ui.detail.config.FieldConfigBase#createUI()
     */
    @Override
    public void createUI() {
        if (comboBox == null) {
            List<ValueComboBoxData> dataList = new ArrayList<>();

            for (String key : keyList) {
                dataList.add(new ValueComboBoxData(key, valueMap.get(key), getPanelId()));
            }

            if (!dataList.isEmpty()) {
                defaultValue = dataList.get(0).getKey();
            }

            int xPos = getXPos();
            comboBox = new ValueComboBox();
            comboBox.initialiseSingle(dataList);
            comboBox.setBounds(
                    xPos + BasePanel.WIDGET_X_START,
                    0,
                    isValueOnly()
                            ? BasePanel.WIDGET_EXTENDED_WIDTH
                            : BasePanel.WIDGET_STANDARD_WIDTH,
                    BasePanel.WIDGET_HEIGHT);

            FieldPanel fieldPanel = createFieldPanel(xPos, getLabel());
            fieldPanel.add(comboBox);

            if (!isValueOnly()) {
                setAttributeSelectionPanel(
                        fieldPanel.internalCreateAttrButton(String.class, this, isRasterSymbol()));
            }

            for (ValueComboBoxData data : dataList) {
                this.comboDataMap.put(data.getKey(), data);
            }

            comboBox.addActionListener(
                    new ActionListener() {
                        public void actionPerformed(ActionEvent e) {
                            valueStored();
                        }
                    });
        }
    }

    /**
     * Attribute selection.
     *
     * @param field the field
     */
    /*
     * (non-Javadoc)
     *
     * @see
     * com.sldeditor.ui.iface.AttributeButtonSelectionInterface#attributeSelection(java.lang.String)
     */
    @Override
    public void attributeSelection(String field) {
        if (this.comboBox != null) {
            this.comboBox.setEnabled(field == null);
        }
    }

    /**
     * Sets the enabled.
     *
     * @param enabled the new enabled
     */
    /*
     * (non-Javadoc)
     *
     * @see com.sldeditor.ui.detail.config.FieldConfigBase#setEnabled(boolean)
     */
    @Override
    public void internalSetEnabled(boolean enabled) {
        if (comboBox != null) {
            comboBox.setEnabled(enabled);
        }
    }

    /**
     * Generate expression.
     *
     * @return the expression
     */
    /*
     * (non-Javadoc)
     *
     * @see com.sldeditor.ui.detail.config.FieldConfigBase#generateExpression()
     */
    protected Expression generateExpression() {
        Expression expression = null;

        if (comboBox != null) {
            ValueComboBoxData value = comboBox.getSelectedValue();
            if (value != null) {
                expression = getFilterFactory().literal(value.getKey());
            }
        }

        return expression;
    }

    /**
     * Checks if is enabled.
     *
     * @return true, if is enabled
     */
    /*
     * (non-Javadoc)
     *
     * @see com.sldeditor.ui.detail.config.FieldConfigBase#isEnabled()
     */
    @Override
    public boolean isEnabled() {
        if ((attributeSelectionPanel != null) && !isValueOnly()) {
            return attributeSelectionPanel.isEnabled();
        } else {
            if (comboBox != null) {
                return comboBox.isEnabled();
            }
        }
        return false;
    }

    /** Revert to default value. */
    /*
     * (non-Javadoc)
     *
     * @see com.sldeditor.ui.detail.config.FieldConfigBase#revertToDefaultValue()
     */
    @Override
    public void revertToDefaultValue() {
        populateField(defaultValue);
    }

    /**
     * Populate expression.
     *
     * @param objValue the obj value
     */
    /*
     * (non-Javadoc)
     *
     * @see com.sldeditor.ui.detail.config.FieldConfigBase#populateExpression(java.lang.Object)
     */
    @Override
    public void populateExpression(Object objValue) {
        if (comboBox != null) {
            if (objValue instanceof String) {
                String sValue = (String) objValue;

                populateField(sValue);
            } else if (objValue instanceof LiteralExpressionImpl) {
                String sValue = objValue.toString();

                populateField(sValue);
            }
        }
    }

    /**
     * Gets the value.
     *
     * @return the value
     */
    @Override
    public ValueComboBoxData getEnumValue() {
        if (comboBox == null) {
            return null;
        }
        return (ValueComboBoxData) comboBox.getSelectedItem();
    }

    /**
     * Gets the string value.
     *
     * @return the string value
     */
    @Override
    public String getStringValue() {
        ValueComboBoxData enumValue = getEnumValue();
        if (enumValue != null) {
            return enumValue.getKey();
        }

        return null;
    }

    /**
     * Adds the config.
     *
     * @param configList the config list
     */
    public void addConfig(List<SymbolTypeConfig> configList) {
        if (configList != null) {
            for (SymbolTypeConfig config : configList) {
                if (config != null) {
                    fieldMap.putAll(config.getFieldMap());

                    Map<String, String> optionMap = config.getOptionMap();

                    for (Entry<String, String> entry : optionMap.entrySet()) {
                        addValue(entry.getKey(), entry.getValue());
                    }
                }
            }
        }
    }

    /**
     * Gets the field enable state.
     *
     * @return the field enable state
     */
    public Map<FieldIdEnum, Boolean> getFieldEnableState() {
        ValueComboBoxData value = getEnumValue();
        if (value == null) {
            return null;
        }
        Class<?> panelId = value.getPanelId();

        return fieldMap.get(panelId);
    }

    /**
     * Undo action.
     *
     * @param undoRedoObject the undo redo object
     */
    /*
     * (non-Javadoc)
     *
     * @see com.sldeditor.undo.UndoActionInterface#undoAction(com.sldeditor.undo.UndoInterface)
     */
    @Override
    public void undoAction(UndoInterface undoRedoObject) {
        if ((comboBox != null) && (undoRedoObject != null)) {
            if (undoRedoObject.getOldValue() instanceof String) {
                String oldValue = (String) undoRedoObject.getOldValue();

                ValueComboBoxData valueComboBoxData = comboDataMap.get(oldValue);
                if (valueComboBoxData != null) {
                    comboBox.setSelectedItem(valueComboBoxData);
                }
            }
        }
    }

    /**
     * Redo action.
     *
     * @param undoRedoObject the undo redo object
     */
    /*
     * (non-Javadoc)
     *
     * @see com.sldeditor.undo.UndoActionInterface#redoAction(com.sldeditor.undo.UndoInterface)
     */
    @Override
    public void redoAction(UndoInterface undoRedoObject) {
        if ((comboBox != null) && (undoRedoObject != null)) {
            if (undoRedoObject.getNewValue() instanceof String) {
                String oldValue = (String) undoRedoObject.getNewValue();

                ValueComboBoxData valueComboBoxData = comboDataMap.get(oldValue);
                if (valueComboBoxData != null) {
                    comboBox.setSelectedItem(valueComboBoxData);
                }
            }
        }
    }

    /**
     * Sets the test value.
     *
     * @param fieldId the field id
     * @param testValue the test value
     */
    @Override
    public void setTestValue(FieldIdEnum fieldId, String testValue) {
        populateField(testValue);

        valueUpdated();
    }

    /**
     * Populate field.
     *
     * @param value the value
     */
    @Override
    public void populateField(String value) {
        if (comboBox != null) {
            ValueComboBoxData valueComboBoxData = comboDataMap.get(value);
            if (valueComboBoxData != null) {
                comboBox.setSelectedItem(valueComboBoxData);
            } else {
                logger.error("Unknown ValueComboBoxData value : " + value);
            }
        }
    }

    /**
     * Sets the default value.
     *
     * @param defaultValue the new default value
     */
    public void setDefaultValue(String defaultValue) {
        this.defaultValue = defaultValue;
    }

    /**
     * Creates a copy of the field.
     *
     * @param fieldConfigBase the field config base
     * @return the field config base
     */
    @Override
    protected FieldConfigBase createCopy(FieldConfigBase fieldConfigBase) {
        FieldConfigEnum copy = null;

        if (fieldConfigBase != null) {
            copy = new FieldConfigEnum(fieldConfigBase.getCommonData());
        }
        return copy;
    }

    /**
     * Sets the field visible.
     *
     * @param visible the new visible state
     */
    @Override
    public void setVisible(boolean visible) {
        if (comboBox != null) {
            comboBox.setVisible(visible);
        }
    }

    /**
     * Checks if supplied value is valid option.
     *
     * @param optionToTest the option to test
     * @return true, if is valid option
     */
    public boolean isValidOption(String optionToTest) {
        return keyList.contains(optionToTest);
    }

    /** Value stored. */
    protected void valueStored() {
        if (comboBox.getSelectedItem() != null) {

            if (!isSuppressUndoEvents()) {
                Object newValueObj = comboBox.getSelectedValue().getKey();

                if ((oldValueObj == null) && comboBox.getItemCount() > 0) {
                    oldValueObj = comboBox.getFirstItem().getKey();
                }

                UndoManager.getInstance()
                        .addUndoEvent(new UndoEvent(this, getFieldId(), oldValueObj, newValueObj));

                oldValueObj = newValueObj;
            }
            valueUpdated();
        }
    }
}
