﻿using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Mvc;
using SSCMS.Configuration;
using SSCMS.Core.Utils;
using SSCMS.Enums;
using SSCMS.Utils;

namespace SSCMS.Web.Controllers.Admin.Cms.Contents
{
    public partial class ContentsSearchController
    {
        [HttpPost, Route(RouteTree)]
        public async Task<ActionResult<TreeResult>> Tree([FromBody]TreeRequest request)
        {
            if (!await _authManager.HasSitePermissionsAsync(request.SiteId,
                    MenuUtils.SitePermissions.ContentsSearch))
            {
                return Unauthorized();
            }

            var site = await _siteRepository.GetAsync(request.SiteId);
            if (site == null) return this.Error(Constants.ErrorNotFound);

            var channel = await _channelRepository.GetAsync(request.SiteId);
            
            var enabledChannelIds = await _authManager.GetContentPermissionsChannelIdsAsync(site.Id);
            var visibleChannelIds = await _authManager.GetVisibleChannelIdsAsync(enabledChannelIds);

            var firstEnabledChannelId = 0;
            var root = await _channelRepository.GetCascadeAsync(site, channel, async summary =>
            {
                var visible = visibleChannelIds.Contains(summary.Id);
                if (!visible) return null;

                var count = await _contentRepository.GetCountAsync(site, summary);
                var disabled = !enabledChannelIds.Contains(summary.Id);
                if (firstEnabledChannelId == 0 && !disabled)
                {
                    firstEnabledChannelId = summary.Id;
                }

                return new
                {
                    Count = count,
                    Disabled = disabled
                };
            });

            if (!request.Reload)
            {
                var channelIds = new List<int>();
                var firstChannel = await _channelRepository.GetAsync(firstEnabledChannelId);
                if (firstChannel.ParentsPath != null && firstChannel.ParentsPath.Count > 0)
                {
                    channelIds.AddRange(firstChannel.ParentsPath);
                }
                channelIds.Add(firstChannel.Id);

                var siteUrl = await _pathManager.GetSiteUrlAsync(site, true);
                var groupNames = await _contentGroupRepository.GetGroupNamesAsync(request.SiteId);
                var tagNames = await _contentTagRepository.GetTagNamesAsync(request.SiteId);
                var checkedLevels = ElementUtils.GetCheckBoxes(CheckManager.GetCheckedLevels(site, true, site.CheckContentLevel, true));

                var columnsManager = new ColumnsManager(_databaseManager, _pathManager);
                var columns = await columnsManager.GetContentListColumnsAsync(site, channel, ColumnsManager.PageType.SearchContents);
                var permissions = new Permissions
                {
                    IsAdd = await _authManager.HasContentPermissionsAsync(site.Id, firstChannel.Id, MenuUtils.ContentPermissions.Add),
                    IsDelete = await _authManager.HasContentPermissionsAsync(site.Id, firstChannel.Id, MenuUtils.ContentPermissions.Delete),
                    IsEdit = await _authManager.HasContentPermissionsAsync(site.Id, firstChannel.Id, MenuUtils.ContentPermissions.Edit),
                    IsArrange = await _authManager.HasContentPermissionsAsync(site.Id, firstChannel.Id, MenuUtils.ContentPermissions.Arrange),
                    IsTranslate = await _authManager.HasContentPermissionsAsync(site.Id, firstChannel.Id, MenuUtils.ContentPermissions.Translate),
                    IsCheck = await _authManager.HasContentPermissionsAsync(site.Id, firstChannel.Id, MenuUtils.ContentPermissions.CheckLevel1),
                    IsCreate = await _authManager.HasSitePermissionsAsync(site.Id, MenuUtils.SitePermissions.CreateContents) || await _authManager.HasContentPermissionsAsync(site.Id, channel.Id, MenuUtils.ContentPermissions.Create),
                };

                var titleColumn =
                    columns.FirstOrDefault(x => StringUtils.EqualsIgnoreCase(x.AttributeName, nameof(Models.Content.Title)));
                columns.Remove(titleColumn);

                var bodyColumn = new ContentColumn
                {
                    AttributeName = nameof(Models.Content.Body),
                    DisplayName = "内容正文",
                    InputType = InputType.TextEditor,
                    IsSearchable = true,
                };

                return new TreeResult
                {
                    Root = root,
                    ChannelIds = channelIds,
                    SiteUrl = siteUrl,
                    GroupNames = groupNames,
                    TagNames = tagNames,
                    CheckedLevels = checkedLevels,
                    TitleColumn = titleColumn,
                    BodyColumn = bodyColumn,
                    Columns = columns,
                    Permissions = permissions
                };
            }

            return new TreeResult
            {
                Root = root,
            };
        }
    }
}
