const withJigsawLayoutResult = (expectedResult, layouts) => {
  return layouts.map(l => ({ input: l, solution: expectedResult }));
};

const EASY_INVALID_JIGSAW_LAYOUTS = withJigsawLayoutResult(null, [
  // http://forum.enjoysudoku.com/jigsaw-layouts-generate-test-t35712.html#p274998
  '000000001223411101223415111223455556223444566233334566777374566787774566788888888',
  '000000001222304111222344111222344411555333444665553777666553777666583777688888888',
  // Partial.
  '.NoBoxes.Jigsaw~000000000000001000000011000000011100222000111332220444333220444333250444355555555.',
]);

const FAST_INVALID_JIGSAW_LAYOUTS = withJigsawLayoutResult(null, [
  // From Mathimagics
  // http://forum.enjoysudoku.com/jigsaw-layouts-generate-test-t35712-30.html?sid=6523d33401485a4b2414c1d7cc15a060#p310543
  'AAAAAAAABACCCCCDBBCCEEDDDBBCCFEEDDBBGGFFEDDBBGGFFEEDHHGGFFFEEHHGGFHHHHHIGIIIIIIII',
  'AAAAAAAABCCDDDDBABCCDDDBBBBCCDEDEBFBCCEEEEEFFGCGEHEHFFGGGGHHHFFGIGHHHHFFGIIIIIIII',
  'AAAAAAAABCDDDDDBABCCDDBBBBBCCDDEBEFFCCEEEEEFFCCEGEHHFFGGGGGHHFFGIGHHHHHFGIIIIIIII',
  'AAAAAAAABCDDDDDBABCCDEEDBBBCCFEDDBGGCCFEEEBGGCCFHHEBGGFFFHEEHGGFIFHHHHHGFIIIIIIII',
  'AAAAAAABBAACCCCDBBCCCEFFDBBCCEEFDDBBGEEFFFDDBGGEEFDDHHGGEFFDHHHGGEHHHHIIGGIIIIIII',
  'AAAAAAABBACCCCCABBCCDEEEEBBCCDDDEEBBFGGGDEEEBFFGGDDDHHFFGGGGDHHFFIHHHHHIFFIIIIIII',
  'AAAAAAABBCCCCCCAABDDDCCCBBBDDEEBBBFFDDEEEEEFFDDGGGEEFFGGGHHHFFFGIIHHHHHHGGIIIIIII',
  'AAAAAAABBCCDDDBAABCCDDDBBBBCCDDEEEBFCCDEEEGFFCHEEEGGFFHHHHGGGFFHIIHGGGFFHHIIIIIII',
  'AAAAAAABBCCDDDDAABCCDDDBBBBCDDEEBBFFCCEEEEEFFCCGGEEHHFGGGGHHHFFGIIHHHHFFGGIIIIIII',
  'AAAAAABBBCCCDEAABBCCDDEAFBBCCDEEFFGBHCDDEFFGBHCDDEEFGGHHDIEFFGGHHIIEFGGGHHHIIIIII',
  'AAAAAABBBCCDDEAABBCCDEEFABBCCDDEFBBGCCDDEFFGGCHHDEFFGGHHIDEEFGGHHIIEFFGGHHHIIIIII',
  'AAAAAABBBCCDDEABBBCCDDEAABFCCDDEGGBFCHDEEEGBFCHDDEGGFFCHIIEGGFFHHHIEGGFFHHHIIIIII',
  'AAAAABBBBCAAAABBBBCDDDDDDDBCDEDFFFFFCEEEEEEEFCCCCCGEGFHGGGGGGGFHHHHIIIIFHHHHIIIII',
  'AAAAABBBBCCCCAAAABCDDCEEFFBCDDCEEFBBGDDCEHFFBGGDEEHFFHGDDEEHFFHGIIIIHHHHGGGGIIIII',
  'AAAAABCCCAAAABBCCCBBBBBCCDDBEEEECDDDFFFFEDDDDFFFGEEEEHFFGGHHHHHGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCBBBBBCCDDBEEEFCFDDEEFFFFFDDEEFGFDDDHEEGGHHHHHGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCBBBBBCCDDBEFFFCDDDEEEFFFDDDEEEGFFFDHEEGGHHHHHGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCBBBBBCCDDEBFFFCDDDEEEFFFDDDEEEGFFFHDEEGGHHHHHGGGHHIIIIGGGHIIIII',
  'AAAAAAAABACCDBBBBBCCCDDBBEBFFCCDGEEEFFFCDGEEEFFFCDGGEEHFHHDDGGGHHHHHDGGIHIIIIIIII',
  'AAAAAAAABCCCDABBBBCCCDDBBBBEECCDFFFFEEECDGFFFEEEEDGGFFHHHHDDGGGHHHHIDGGGHIIIIIIII',
  'AAAAAAAABCCCDBBABBCCCDDBBBBEECCDFFFFEEECDGFFFEEEEDGGFFHHHHDDGGGHHIHHDGGGHIIIIIIII',
  'AAAAAAABBCCCDAABBBCCCDDBBBBEECCDFFFFEEECDGFFFEEEEDGGFFHHHHDDGGGHHHIIDGGGHHIIIIIII',
  'AAAAAAABBCCCDAABBBECCDDBBBBEECCDFGGGEEECDFGGGEEECDFFGGHHHHDDFFGHHHIIDFFFHHIIIIIII',
  'AAAAAABBBAACDDDDBBACCDDBBBBCCDDEEEFFCCDEEEGFFCCEEEGGFFHHHHGGFFIHHGGGGFIIHHHIIIIII',
  'AAAAAABBBACCACCBDBACCCCCBDDEEEEFFBBDEGFFFFFBDEGGFFDDDDEEGHHHHHIGEGHHIHHIGGGIIIIII',
  'AAAAAABBBACCDAABBBCCCDDBBBEFFCCDGEEEFFFCDGEEEFFFCDGGEEFHHHDDGGGHHHIIDGGIHHHIIIIII',
  'AAAAAABBBCCACCABDBCCCCCABDDEEEEFFBBDEGFFFFFBDEGGFFDDDDEEGHIIIIIGEGHIIHIIGGGHHHHHH',
  'AAAAAABBBCCADAABBBCCCDDBBBEFFCCDGEEEFFFCDGEEEFFFCDGGEEFHHHDDGGGHHHIIDIGGHHHIIIIII',
  'AAAAAABBBCCADAABBBCCCDDBBEBFFCCDGEEEFFFCDGEEEFFFCDGGEEHFHHDDGGGHHHIIDIGGHHHIIIIII',
  'AAAAAABBBCCCDAAABBCCCDDBBBBEECCDFFFFEEECDGFFFEEEEDGGFFHHHHDDGGGHHIIIDGGGHHHIIIIII',
  'AAAAAABBBCCCDAAABBCECDDBBBBEECCDFGGGEEECDFGGGEEECDFFGGHHHHDDFGFHHIIIDFFFHHHIIIIII',
  'AAAAAABBBCCCDABBBBCCCDABEEEFCCDABEEEFFCDDDGEEFFFHIDGGEFFFHIDGGGHHHHIDGGGHHHIIIIII',
  'AAAAAABBBCCCDABBBBCCEDABFFFCCEDABFFFCCEDDDFGGEEEHIDFGGEEEHIDFGGHHHHIDGGGHHHIIIIII',
  'AAAAAABBBCCCDABBBBECCDABFFFECCDABGFFEECDDDGFFEECHIDGGFEEEHIDGGFHHHHIDGGGHHHIIIIII',
  'AAAAAABBBCCCDABBBECCCDABBEEFFCDABGEEFFCDDDGEEFFCHIDGEEFFHHIDGGGFHHHIDGGGHHHIIIIII',
  'AAAAAABBBCCCDABBEBFCCDABBEEFCCDABEEEFCCDDDGGEFFFHIDGGEFFHHIDGGEHFHHIDGGGHHHIIIIII',
  'AAAAAABBBCCCDABBEECCCDABBBEFCCDABEEEFCFDDDEGEFFFHIDGGEFHHHIDGGGFFHHIDGGGHHHIIIIII',
  'AAAAAABBBCCCDABBEECCCDABBEECCFDABBEECFFDDDEEGFFHHIDEGGFFHHIDGGGFFHHIDGGGHHHIIIIII',
  'AAAAAABCCAAABBBBCCDDBBBECCFDDGEBECFFDDGEEECFFDDGEHECFFDGGEHHHFFGGHHHHIIIGGHIIIIII',
  'AAAAAABCCAAABBBBCCDDBBEBCCFDDGBEECFFDDGEEECFFDDGEEHCFFDGGHEHHFFGGHHHHIIIGGHIIIIII',
  'AAAAABBBBAACADBBBBACCDDDDDBCCDDEEEFFCCDEEEGFFCCEEEGGFFHGGGGGFFIHHHHGIFIIHHHHIIIII',
  'AAAAABBBBAACADDBBBACCDDDDBBCCDDEEEFFCCDEEEGFFCCEEEGGFFHHGGGGFFIHHHGGIFIIHHHHIIIII',
  'AAAAABBBBAACADDDBBACCDDDBBBCCDDEEEFFCCDEEEGFFCCEEEGGFFHHHGGGFFIHHGGGIFIIHHHHIIIII',
  'AAAAABBBBAACCBBDBBAACCCCDDBEEFFFCCDDEEGFFFCDDEEGGFFFDDHEEGGGGIIHHEHHGGIIHHHHIIIII',
  'AAAAABBBBAACDADBBBACCDDDDBBCCDDEEEFFCCDEEEGFFCCEEEGGFFHHGGGGFFIHHHGIGFIIHHHHIIIII',
  'AAAAABBBBAACDADDBBACCDDDBBBCCDDEEEFFCCDEEEGFFCCEEEGGFFHHHGGGFFIHHGGIGFIIHHHHIIIII',
  'AAAAABBBBACAAADBBBCCDDDDDBBCDDEEFFFFCDEEEEEGFCCCCEEGGFHHGGGGGFFHHHGIIIFIHHHHIIIII',
  'AAAAABBBBACCCABBBBCCDEABFFFCCDEAEFFFCCDEEEFGGDDDEHEFGGDDDIHEFGGIIIIHGGGHIIIIHHHHH',
  'AAAAABBBBACCCABBBBDCCEABFFFDCCEAEGFFDDCEEEGFFDDCEHEGGFDDDIHEGGFIIIIHGGGHIIIIHHHHH',
  'AAAAABBBBACCCABBBBDDCEABFGGDDCEAEFGGDCCEEEFFGDDCEHEFGGDDCIHEFGGIIIIHFFFHIIIIHHHHH',
  'AAAAABBBBACCDAABBBACCDDEFBBGGCCDEFFFGGGCDEFFFGGGCDEEFFHHGCDDEEIHHHIIDEEIHHHHIIIII',
  'AAAAABBBBACCDAABBBCCCDDABBEFFCCDGEEEFFFCDGEEEFFFCDGGEEFHHIDDGGGHHHIIDGGIHHHHIIIII',
  'AAAAABBBBACCDAABBBCCCDDAEBBFFCCDGEEEFFFCDGEEEFFFCDGGEEHHFIDDGGGHHHIIDGGIHHHHIIIII',
  'AAAAABBBBACCDAEBBBAACDDEBFBGGCCDEFFFGGGCDEFFFGGGCDEEFFHGHCDDEIIHHHCIDEEIHHHHIIIII',
  'AAAAABBBBCAADAEBBBCCCDAEBBFCCCDEEFFFCCDDEFFGGDDDEEFGGGDHHEIFGGGHHHEIFIIGHHHHIIIII',
  'AAAAABBBBCACDAABBBCCCDDABBEFFCCDGEEEFFFCDGEEEFFFCDGGEEFHHIDDGGGHHHIIDGIGHHHHIIIII',
  'AAAAABBBBCCADAAABBECCDDFBBBEECCDFGGGEEECDFGGGEEECDFFGGHHHCDDFFGHHIIIDIFFHHHHIIIII',
  'AAAAABBBBCCADAABBBCCCDDAEBBFFCCDGEEEFFFCDGEEEFFFCDGGEEHHFIDDGGGHHHIIDIGGHHHHIIIII',
  'AAAAABBBBCCADABBBECCCDABEEEFCCDABEEEFCCDDDGGEFFFHIDGGEFFFHIDGGGFHHHIDIGGHHHHIIIII',
  'AAAAABBBBCCCDAAABBCECDDABBBEECCDFGGGEEECDFGGGEEECDFFGGHHHIDDFGFHHIIIDFFFHHHHIIIII',
  'AAAAABBBBCCCDAAABBECCDDABBBEECCDFGGGEEECDFGGGEEECDFFGGHHHIDDFFGHHIIIDFFFHHHHIIIII',
  'AAAAABBBBCCCDAABBBCCCDDAABBEECCDFFFFEEECDGFFFEEEEDGGFFHHIIDDGGGHHHIIDGGGHHHHIIIII',
  'AAAAABBBBCCCDAABBBCECDDAABBEECCDFGGGEEECDFGGGEEECDFFGGHHIIDDFGFHHHIIDFFFHHHHIIIII',
  'AAAAABBBCAAAABBBCCDDEBBFCCGDDEFBFCGGDDEFFFCGGDDEFHFCGGDEEFHHCGGEEHHHIIIIEHHHIIIII',
  'AAAAABBCCAAAABBBCCDDBBBECCFDDGBEECFFDDGEEECFFDDGEEHCFFDGGEHHHFFGGHHHIIIIGGHHIIIII',
  'AAAAABBCCAAAABBBCCDDEEBBCCFDDGEBBCFFDDGEEECFFDDGHHECFFDGGHHEEFFGGHHHIIIIGGHHIIIII',
  'AAAAABBCCAAAABBCCCBBBBBCCDDEEFFFCDDDEEFGFCFDDEEEGFFFDDEEGGHHHHHGGGHHIIIIGGHHIIIII',
  'AAAAABBCCAAAABBCCCDBBBBCCEEDBFFFCEEEDDFGFCFEEDDDGFFFHEDDGGHHHHEGGGHHIIIIGGHHIIIII',
  'AAAAABCCCAAAABBCCCBBBBBCCCDBEEEEEFFDEEFFFFFDDEFFDDDDDGEHHHGGGGGHHHGGIIIIHHHGIIIII',
  'AAAAABCCCAAAABBCCCBBBBBCCDDEEBFFCDDDEEFFFFFDDEEEGFFHDDEEGGHHHHHGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCBBBBBCCDDEEEBFCFDDEEFFFFFDDEEFGFHDDDEEGGHHHHHGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCBBBBBCCDDEEEFBCDDDEFFFFFFFDEEEGHFDDDEEGGHHHHHGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCBBBBBCCDDEEEFBCFDDEEFFFFFDDEEFGHFDDDEEGGHHHHHGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCBBBBBCCDDEEFBFCDDDEEFFFFFDDEEEGFHFDDEEGGHHHHHGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCBBBBBCCDDEEFFBCDDDEEFFFFFDDEEEGHFFDDEEGGHHHHHGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCBBBBBCCDDEFFFBCDDDEEEFFFDDDEEEGHFFFDEEGGHHHHHGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCDBBBBCCEEDBBFFCEEEDDFFFFFEEDDDGFFHHEDDGGHHHHEGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCDBBBBCCEEDBFBFCEEEDDFFFFFEEDDDGFHFHEDDGGHHHHEGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCDBBBBCCEEDBFFBCEEEDDFFFFFEEDDDGHFFHEDDGGHHHHEGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCDBBBBCCEEDDBFBCEEEDFFFFFFFEDDDGHFHEEDDGGHHHHEGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCDBBBBCCEEDDBFBCFEEDDFFFFFEEDDFGHFHEEDDGGHHHHEGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCDBBBBCCEEDDDBBCFEEDFFFFFFFEDDFGHHEEEDDGGHHHHEGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCDBBBBCCEEDDFBBCEEEDFFFFFFFEDDDGHHFEEDDGGHHHHEGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCDBBBBCCEEDDFBBCFEEDDFFFFFEEDDFGHHFEEDDGGHHHHEGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCDDBBBCCEEDBBBFCFEEDDFFFFFEEDDFGFHHHEDDGGHHHEEGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCDDBBBCCEEDBBFBCFEEDDFFFFFEEDDFGHFHHEDDGGHHHEEGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCDDBBBCCEEDDBBBCFEEDFFFFFFFEDDFGHHHEEDDGGHHHEEGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCCCDDBBBCCEEDDBBBCFFEDDFFFFFEEDFFGHHHEEDDGGHHHEEGGGHHIIIIGGGHIIIII',
  'AAAAABCCCAAAABBCDCBBBBBCCDDBEFFFCDDDEEFGFCFDDEEEGFFFDHEEGGHHHHHGEGHHIIIIGGGHIIIII',
  'AAAAABCCCAABBABCDCABBBBBCDDEEEEFFCCDEGFFFFFCDEGGFFDDDDEEGHHHHHIGEGHIHHIIGGGHIIIII',
  'AAAAABCCCAADABBBCCADDBBCCCCDDBBEEEFFDDBEEEGFFDDEEEGGFFHHHHGGFFIHHGGGIFIIHHHGIIIII',
  'AAAABBBBCAAAAABBCCDDEFBBCCGDDEFBFCGGDDEFFFCGGDDEFHFCGGDEEHHFCGGEEHHIIIIIEHHHHIIII',
  'AAAABBBBCAAAAABBCCDDEFFBCCGDDEFBBCGGDDEFFFCGGDDEHHFCGGDEEHFFCGGEEHHIIIIIEHHHHIIII',
  'AAAABBBBCAAABBDBBCAAEBDDCCCEEEFDCCGGEEEFDCGGGEEFFDCGGGFFFDDHGIIFHHDHHIIIFHHHHIIII',
  'AAAABBBBCAAABBDBBCAEABDDCCCEEEFDCCGGEEEFDCGGGEEFFDCGGGFFFDDHIGIFHHDHHIIIFHHHHIIII',
  'AAAABBBBCAAABBDBBCEAABDDCCCEEEFDCCGGEEEFDCGGGEEFFDCGGGFFFDDHIIGFHHDHHIIIFHHHHIIII',
  'AAAABBBBCAAABBDBCCAABBDDCCCEEEEDCCFFEEEGDCFFFEEGGDFFFFGGGDDHHIIGGHDHHIIIGHHHHIIII',
  'AAAABBBBCAAABBDBCCAABBDDCCEFFFGDCCEEFFFGDCEEEFFGGDCEEEFGGDDHHIIGGHDHHIIIGHHHHIIII',
  'AAAABBBBCAABBBDBBCEAAADDCCCEEEFDCCGGEEEFDCGGGEEFFDCGGGFFFDDHHHGFIIDIIIHHFIIIIHHHH',
  'AAAABBBBCAABBBDBCCAAABDDCCCEEEEDCCFFEEEGDCFFFEEGGDFFFFGGGDDHIIIGGHDHHHIIGHHHHIIII',
  'AAAABBBBCAABBBDBCCAAABDDCCEFFFGDCCEEFFFGDCEEEFFGGDCEEEFGGDDHIIIGGHDHHHIIGHHHHIIII',
  'AAAABBBBCAABBBDCBCAAABDDCCCEEEEDCCFFEEEGDCFFFEEGGDFFFFGGGDDHIIIGHGDHHHIIGHHHHIIII',
  'AAAABBBCCAAAAABBCCDDEEBBCCFDDGEBBCFFDDGEEECFFDDGHHECFFDGGHHEEFFGGHHIIIIIGGHHHIIII',
  'AAAABBBCCAABBBDBBCAAABDDCCCEEEEDCCFFEEEGDCFFFEEGGDFFFFGGGDDHIIIGHHDHHHIIGGHHHIIII',
  'AAAABBBCCABBBBDBBCAAAADDCCCEEEEDCCFFEEEGDCFFFEEGGDFFFFGGGDDHHHHGIIDIIIIHGGIIIHHHH',
  'AAAABBCCCAAABBBCDCAABBBBCDDEEEEFFCCDEGFFFFFCDEGGFFDDDDEEGHHHHIIGEGHHHIIIGGGHHIIII',
  'AAAABBCCCAABABBCDCAABBBBCDDEEEEFFCCDEGFFFFFCDEGGFFDDDDEEGHHHHIIGEGHHIHIIGGGHHIIII',
  'AAAABBCCCAABBBBCDCAAABBBCDDEEEEFFCCDEGFFFFFCDEGGFFDDDDEEGHHHIIIGEGHHHHIIGGGHHIIII'
]);

const HARD_INVALID_JIGSAW_LAYOUTS = withJigsawLayoutResult(null, [
  // From Mathimagics
  // http://forum.enjoysudoku.com/jigsaw-layouts-generate-test-t35712-30.html?sid=6523d33401485a4b2414c1d7cc15a060#p310543
  //
  // These are the ones which take more than 1 second.
  'AAAAAABBBCCADAEBBBCCCDAEFBBCCCDEEFFBCDDDEFFFGHDDEEFGGGHHDEIFGGGHHHEIFIGGHHHIIIIII',
  'AAAAABBBBAAAACBBBBDDDDCCCCBDEEEECCCCDDDDEFFFFGGGGEEEEFHGGGGFFFFHHHHGIIIIHHHHIIIII',
  'AAAAABBBBAAAACBBCBDDDDCCBCBDEEEECCCCDDDDEFFFFGGGGEEEEFHGHGGFFFFHGHHGIIIIHHHHIIIII',
  'AAAAABBBBAAAACCBBBDDDCCCCBBDCCCEEEEFDDDDEFFFFDEEEEGGGFHHGGGGFFFHHHGGIIIIHHHHIIIII',
  'AAAAABBBBAAAACCBBBDDDDCCCBBDEEEECCCCDDDDEFFFFGGGGEEEEFHHGGGFFFFHHHGGIIIIHHHHIIIII',
  'AAAAABBBBAACADDDBBCCCADDBBBCEEEEDDDDCCCCEFFFFGGGGEEEEFHHHGGIFFFHHGGGIFIIHHHHIIIII',
  'AAAAABBBBAACCBBBBBAACCCCDDDEFFFFCCCDEEEEFDDDDEGGGFFFFDEEEGGGGHHIIIIIGGHHIIIIHHHHH',
  'AAAAABBBBACCCABBBBCCCDABEEECFCDADEEECFFDDDEEGFFFDHDGEGFFFIHDGGGIIIIHGGGHIIIIHHHHH',
  'AAAAABBBBACCCABBBBCCCDABEEEFFCDADGEEFFCDDDGEEFFCDHDGEEFFFIHDGGGIIIIHGGGHIIIIHHHHH',
  'AAAAABBBBCAAADDBBBCCCADDDBBCDDDDEEEFCCCEEEFFFCEEEGGGGFHHGGGIFFFHHHGGIIIFHHHHIIIII',
  'AAAAABBBBCCADAEBBBCCADAEFBBCCCDEEFFFCCDDEFFGGDDDEEFGGGHHDEIFIGGHHHEIFIGGHHHHIIIII',
  'AAAAABBBBCCADAEBBBCCCDAEFBBCCDDAEFFFCCDEEEFGGDDDEHFFGGIIDEHFGGGIIIEHFHGGIIIIHHHHH',
  'AAAAABBCCAAAABBCCCDDDBBCCCCDBBBEEEEFDDDDEFFFFDEEEEGGGFHHHHGGFFFHHHGGIIIIHHGGIIIII',
  'AAAAABBCCAAAABBCCCDDDDBCCCCDBBBBEEEFDDDEEEFFFDEEEGGGGFHHHHGFFFFHHHGGIIIIHHGGIIIII',
  'AAAAABBCCAADAABCCCDDDDBBCCCDBBBBEEECDDDEEEFFFGEEEHHHHFGGGHHFFFFGGGHIIFIIGGHHIIIII',
  'AAAAABBCCADAAABCCCDDDBBBCCCDBBBEEEECDDDDEFFFFGEEEEHHHFGGGHHHFFFGGGHIIIFIGGHHIIIII',
  'AAAAABBCCADAABBCCCDDABBCCCCDBBBEEEEFDDDDEFFFFDEEEEGGGFHHHHGGIFFHHHGGIIFIHHGGIIIII',
  'AAAAABBCCDAAABBCCCDDABBCCCCDBBBEEEEFDDDDEFFFFDEEEEGGGFHHHHGGIFFHHHGGIIIFHHGGIIIII',
  'AAAAABBCCDADEABBCCDDDEABCCCDDDEABBCCDFFEEEBBGHHFFIEGGGHHHFIEGGGHHFFIEGIGHHFFIIIII',
  'AAAAABCCCAAAABBBCCDDDBBCCCCDBBBEEEEFDDDDEFFFFDEEEEGGGFHHHHGGFFFHHGGGIIIIHHHGIIIII',
  'AAAAABCCCAAADABCCCDDDDBBBCCDBBBBEEECDDDEEEFFFGEEEHHHHFGGHHHFFFFGGGHIFIIIGGGHIIIII',
  'AAAAABCCCADDAABBCCDDDABBCCCDBBBBEEECDDDEEEFFFGEEEHHHHFGGGHHIFFFGGHHIIFFIGGGHIIIII',
  'AAAABBBCCDAAABCCCCDAABBCCCEDBBBFFFFEDDDDFEEEEDFFFFGGGEDHHHGGIIEHHHHGIIIEHHGGGIIII',
  'AAAABBCCCAAAABBCCCDDABBCCCEDBBBFFFFEDDDDFEEEEDFFFFGGGEDHHHGGIEEHHHGGIIIIHHHGGIIII',
  'AAAABBCCCAAAABBCCCDDDABCCCEDBBBBFFFEDDDFFFEEEDFFFGGGGEDHHHGIEEEHHHGGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCCDABBBCCEEDBBBFFFFEDDDDFEEEEDFFFFGGGEDDHHGGGIEHHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCCDDABBBCCEDBBBFFFFEDDDDFEEEEDFFFFGGGEDHHGGGIEEHHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCCDDABBCCEEDBBBBFFFEDDDFFFEEEDFFFGGGGEDDHHGGIEEHHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCCDDABBCCEEDBBBFFFFEDDDBFGEEEDFFFFGGGEDDHHGGIEEHHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCCDDABBCCEEDFFFBBBBEDDDFFFEEEDGGGGFFFEDDHHGGIEEHHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCCDDDABBBCCDEEEEBBBFDDDDEFFFFDGGGEEEEFHHGGGIFFFHHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCCDDDABBBCCDEEEEFBBBDDDDEFFFFGGGDEEEEFHHGGGIFFFHHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCCDDDABBCCEDFFFBBBBEDDDFFFEEEDGGGGFFFEDHHGGIEEEHHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCCDDDABBCCEDFFFFBBBEDDDGFBEEEDGGGFFFFEDHHGGIEEEHHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCDEABBBCCCDEBBBFFFFDEEEEFDDDDEFFFFGGGDEHHHGGGIDEHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCDEBABBCCCDEBBBFFFFDEEEEFDDDDEFFFFGGGDEHHHGGIGDEHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCDEEABBCCCDEBBBBFFFDEEEFFFDDDEFFFGGGGDEHHHGGIDDEHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCDEEABBCCCDEBBBFFFFDEEEBFGDDDEFFFFGGGDEHHHGGIDDEHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCDEEABBCCCDEFFFBBBBDEEEFFFDDDEGGGGFFFDEHHHGGIDDEHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCDEEBABCCCDEBBBBFFFDEEEFFFDDDEFFFGGGGDEHHHGIGDDEHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAABBCCCCDAABBCCEEDBBBFFFFEDDDDFEEEEDFFFFGGGEDDHHGGIIEHHHHGGIIIHHHGGIIII',
  'AAAABBCCCAAABBCCCDEAABBCCCDEBBBFFFFDEEEEFDDDDEFFFFGGGDEHHHGGIIDEHHHGGIIIHHHGGIIII',
  'AAAABBCCCAAABBCCCDEEAABCCCDEBBBBFFFDEEEFFFDDDEFFFGGGGDEHHHGIIDDEHHHGGIIIHHHGGIIII',
  'AAAABBCCCADAABCCCCDDAABBBCCDEEEEBBBFDDDDEFFFFDGGGEEEEFHHGGGIIFFHHHHGIIFIHHHGGIIII',
  'AAAABBCCCADAABCCCCDDAABBCCEDFFFFBBBEDDDGFBEEEDGGGFFFFEDHHGGIIEEHHHHGIIEIHHHGGIIII',
  'AAAABBCCCADAABCCCCDDAABCBCEDFFFBBBBEDDDFFFEEEDGGGGFFFEDHGHGIIEEHHHHGIIEIHHHGGIIII',
  'AAAABBCCCDAAABBCCCDAABBCCCEDBBBFFFFEDDDDFEEEEDFFFFGGGEDHHHGGIIEHHHGGIIIEHHHGGIIII',
  'AAAABBCCCDAAABBCCCDDAABCCCEDBBBBFFFEDDDFFFEEEDFFFGGGGEDHHHGIIEEHHHGGIIIEHHHGGIIII',
  'AAAABBCCCDAAABCCCCDAABBBCCEDFFFFBBBEDDDDFEEEEDGGGFFFFEDHHGGGIIEHHHHGIIIEHHHGGIIII',
  'AAAABBCCCDAAABCCCCDAABBCCEEDBBBBFFFEDDDFFFEEEDFFFGGGGEDDHHGGIIEHHHHGIIIEHHHGGIIII',
  'AAAABBCCCDAAABCCCCDAABBCCEEDFFFBBBBEDDDFFFEEEDGGGGFFFEDDHHGGIIEHHHHGIIIEHHHGGIIII',
  'AAAABBCCCDAAABCCCCDABABCCEEDBBBBFFFEDDDFFFEEEDFFFGGGGEDDHHGIGIEHHHHGIIIEHHHGGIIII',
  'AAAABBCCCDAAABCCCCDDAABBBCCDEEEEBBBFDDDDEFFFFDGGGEEEEFHHGGGIIFFHHHHGIIIFHHHGGIIII',
  'AAAABBCCCDAAABCCCCDDAABBBCCDEEEEFBBBDDDDEFFFFGGGDEEEEFHHGGGIIFFHHHHGIIIFHHHGGIIII',
  'AAAABBCCCDAAABCCCCDDAABBCBCDEEEEBBBFDDDDEFFFFDGGGEEEEFHGHGGIIFFHHHHGIIIFHHHGGIIII',
  'AAAABBCCCDAAABCCCCDDAABBCCEDFFFFBBBEDDDGFBEEEDGGGFFFFEDHHGGIIEEHHHHGIIIEHHHGGIIII',
  'AAAABBCCCDAABBCCCCDAAABCCEEDBBBBFFFEDDDFFFEEEDFFFGGGGEDDHHGIIIEHHHHGGIIEHHHGGIIII',
  'AAAABBCCCDDAABCCCCDAAABBCCEDBBBBFFFEDDDFFFEEEDFFFGGGGEDHHGGIIIEHHHHGIIEEHHHGGIIII',
  'AAAABCCCCAAAABBBCCDDDABCCCEDFFFBBBBEDDDFFFEEEDGGGGFFFEDHHHGIEEEHHGGGIIIIHHHHGIIII',
  'AAAABCCCCAAAABBCCCDABBBCCEEDBBBFFFFEDDDDFEEEEDFFFFGGGEDDHHGGGIEHHHGGIIIIHHHHGIIII',
  'AAAABCCCCAAAABBCCCDDABBBCCEDBBBFFFFEDDDDFEEEEDFFFFGGGEDHHGGGIEEHHHGGIIIIHHHHGIIII',
  'AAAABCCCCAAAABBCCCDDDABBCCEDFFFBBBBEDDDFFFEEEDGGGGFFFEDHHGGIEEEHHHGGIIIIHHHHGIIII',
  'AAAABCCCCAAAABBCCCDDDABBCCEDFFFFBBBEDDDGFBEEEDGGGFFFFEDHHGGIEEEHHHGGIIIIHHHHGIIII',
  'AAAABCCCCAAAABBCCDEEABBCCCDEBBBBFFFDEEEFFFDDDEFFFGGGGDEHHHGGIDDEHHGGIIIIHHHHGIIII',
  'AAAABCCCCAAAABBCCDEEABBCCCDEBBBFFFFDEEEBFGDDDEFFFFGGGDEHHHGGIDDEHHGGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCCABBBBCDDDEBBBFFFFDEEEEFDDDDEFFFFGGGDEEEHGGGGIHHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCCDABBBBCEEDBBBFFFFEDDDDFEEEEDFFFFGGGEDDHGGGGIEHHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCCDABBBCEEEDFFFBBBBEDDDFFFEEEDGGGGFFFEDDDHGGGIEHHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCCDDABBCEEEDBBBFFFFEDDBBFGGEEDFFFFGGGEDDDHGGIEEHHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCCDDABBCEEEDFFFBBBBEDDGFFFBEEDGGGGFFFEDDDHGGIEEHHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCDABBBBCCDDEBBBFFFFDEEEEFDDDDEFFFFGGGDEEHHGGGGIEHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCDEABBBBCCDEBBBFFFFDEEEEFDDDDEFFFFGGGDEHHGGGGIDEHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCDEABBBCCDDEBBBBFFFDEEEFFFDDDEFFFGGGGDEEHHGGGIDEHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCDEEABBBCCDEBBBBFFFDEEEFFFDDDEFFFGGGGDEHHGGGIDDEHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCDEEABBCCBDEFFFBBBBDEEEFFFDDDEGGGGFFFDEGHHGGIDDEHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCDEEABBCCDDEBBBBFFFDEEBFFFGDDEFFFGGGGDEEHHGGIDDEHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCDEEABBCCDDEBBBFFFFDEEBBFGGDDEFFFFGGGDEEHHGGIDDEHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCDEEBABBCCDEBBBBFFFDEEEFFFDDDEFFFGGGGDEHHGGIGDDEHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCDEEBABCCDDEBBBBFFFDEEBFFFGDDEFFFGGGGDEEHHGIGDDEHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCDEEEABBCCDEBBBBFFFDEEBFFFGDDEFFFGGGGDEHHGGIDDDEHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCDCABBBBCCDDBBBEFFFFDEEEEFDDDDEFFFFDGGGEEHHGGGGIHEHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCDCABBBBCCDDBBEBFFFFDEEEEFDDDDEFFFFGDGGEEHHGGGGIHEHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCDCEBBABCCDDEBBBBFFFDEEEFFFDDDEFFFGGGGDEEHHGIGGDHEHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCDDABBBBCCCDEBBBFFFFDEEEEFDDDDEFFFFGGGDEHHHGGGGIEEHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCDDEABBBCCCDEBBBBFFFDEEEFFFDDDEFFFGGGGDEHHHGGGIDEEHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCDDEEABBCCCDEBBBBFFFDEEBFFFGDDEFFFGGGGDEHHHGGIDDEEHHGIIIIHHHHGIIII',
  'AAAABCCCCAAABBBCCCDAABBCCEEDBBBFFFFEDDDDFEEEEDFFFFGGGEDDHHGGIIEHHHGGGIIIHHHHGIIII',
  'AAAABCCCCAAABBBCCDEEAABCCCDEBBBBFFFDEEEFFFDDDEFFFGGGGDEHHHGIIDDEHHGGGIIIHHHHGIIII',
  'AAAABCCCCAAABBCCCDAABBBCCDDBBBEFFFFDEEEEFDDDDEFFFFDGGGEEHHGGGIIEHHHGGIIIHHHHGIIII',
  'AAAABCCCCAAABBCCCDEAABBBCCDEFFFFBBBDEEEEFDDDDEGGGFFFFDEHHGGGIIDEHHHGGIIIHHHHGIIII',
  'AAAABCCCCAAABBCCCDEAABBCCDDEBBBBFFFDEEEFFFDDDEFFFGGGGDEEHHGGIIDEHHHGGIIIHHHHGIIII',
  'AAAABCCCCAAABBCCCDEAABBCCDDEFFFBBBBDEEEFFFDDDEGGGGFFFDEEHHGGIIDEHHHGGIIIHHHHGIIII',
  'AAAABCCCCAAABBCCDCAABBBCCDDEBBBFFFFDEEEEFDDDDEFFFFGGGDEEHHGGGIIHEHHGGIIIHHHHGIIII',
  'AAAABCCCCAAABBCCDCEAABBCCDDEBBBBFFFDEEEFFFDDDEFFFGGGGDEEHHGGIIDHEHHGGIIIHHHHGIIII',
  'AAAABCCCCAAABBCCDCEAABBCCDDEBBBFFFFDEEEBFGDDDEFFFFGGGDEEHHGGIIDHEHHGGIIIHHHHGIIII',
  'AAAABCCCCAAABBCCDCEAABBCCDDEFFFBBBBDEEEFFFDDDEGGGGFFFDEEHHGGIIDHEHHGGIIIHHHHGIIII',
  'AAAABCCCCAAABBCCDDAABBBCCCDEBBBFFFFDEEEEFDDDDEFFFFGGGDEHHHGGGIIEEHHGGIIIHHHHGIIII',
  'AAAABCCCCAAABBCCDDAABBCCDDDBBBBCEEEDFFFEEEDDDFEEEGHHHHFFFGGHHIIFFGGHHIIIGGGGHIIII',
  'AAAABCCCCAAABBCCDDBAABCCCDDBBBBEEEEDFFFFEDDDDFEEEEGGGGFFHHHGIIGFFHHGGIIIHHHHGIIII',
  'AAAABCCCCAAABBCCDDEAABBCCCDEBBBFFFFDEEEBFGDDDEFFFFGGGDEHHHGGIIDEEHHGGIIIHHHHGIIII',
  'AAAABCCCCAABABCCDCAABBBCCDDEBBBFFFFDEEEEFDDDDEFFFFGGGDEEHHGGGIIHEHHGIGIIHHHHGIIII',

  'AAAAABBBBAAAACBCBBDDDCCCCBBDCCCEEEEFDDDDEFFFFDEEEEGGGFHHGGGGFFFHHGHGIIIIHHHHIIIII',
  'AAAAABBBBAAAACCCBBDDDDCCBBBDCCCCEEEFDDDEEEFFFDEEEGGGGFHHHGGFFFFHHGGGIIIIHHHHIIIII',
  'AAAAABBBBACAADBBBBCCCADDDDBCEEEEDDDDCCCCEFFFFGGGGEEEEFHGGGGIFFFHHHHGIIFIHHHHIIIII',
  'AAAABBBCCADAABCCCCDDAABCCCEDFFFBBBBEDDDFFFEEEDGGGGFFFEDHHHGIIEEHHHHGIIEIHHGGGIIII',
  'AAAABBCCCAAAABCCCCDDABBBCCEDFFFFBBBEDDDDFEEEEDGGGFFFFEDHHGGGIEEHHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCCDDBABCCEEDBBBBFFFEDDDFFFEEEDFFFGGGGEDDHHGIGEEHHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCCDDDABCCEEDFFFBBBBEDDGFFFBEEDGGGGFFFEDDHHGIEEEHHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAAABCCCDEEEABCCCDEBBBBFFFDEEBFFFGDDEFFFGGGGDEHHHGIDDDEHHHGIIIIHHHGGIIII',
  'AAAABBCCCAAABBCCCCDDAABCCEEDBBBBFFFEDDDFFFEEEDFFFGGGGEDDHHGIIEEHHHHGGIIIHHHGGIIII',
  'AAAABBCCCADAABBCCCDDAABCCCEDFFFBBBBEDDDFFFEEEDGGGGFFFEDHHHGIIEEHHHGGIIEIHHHGGIIII',
  'AAAABBCCCDAAABCCCCDDAABBCCEDFFFBBBBEDDDFFFEEEDGGGGFFFEDHHGGIIEEHHHHGIIIEHHHGGIIII',
  'AAAABCCCCAAAABBCCCDDABBBCCEDFFFFBBBEDDDDFEEEEDGGGFFFFEDHHGGGIEEHHHGGIIIIHHHHGIIII',
  'AAAABCCCCAAAABBCCCDDABBCCEEDFFFBBBBEDDDFFFEEEDGGGGFFFEDDHHGGIEEHHHGGIIIIHHHHGIIII',
  'AAAABCCCCAAAABBCCCDDBABCCEEDBBBBFFFEDDDFFFEEEDFFFGGGGEDDHHGIGEEHHHGGIIIIHHHHGIIII',
  'AAAABCCCCAAAABBCCCDDDABBCCEDBBBBFFFEDDDFFFEEEDFFFGGGGEDHHGGIEEEHHHGGIIIIHHHHGIIII',
  'AAAABCCCCAAAABBCCDEABBBCCCDEBBBFFFFDEEEEFDDDDEFFFFGGGDEHHHGGGIDEHHGGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCBCCDDDABCBCEDFFFBBBBEDDDFFFEEEDGGGGFFFEDHGHGIEEEHHGHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCCDABBBCEEEDBBBBFFFEDDDFFFEEEDFFFGGGGEDDDHGGGIEHHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCCDDABBBCEEDBBBBFFFEDDDFFFEEEDFFFGGGGEDDHGGGIEEHHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCCCDDABBCEEEDBBBBFFFEDDBFFFGEEDFFFGGGGEDDDHGGIEEHHHHGIIIIHHHHGIIII',
  'AAAABCCCCAAAABCCDDEBABBCCCDEBBBFFFFDEEEBFGDDDEFFFFGGGDEHHHGGIGDEEHHGIIIIHHHHGIIII',
  'AAAABCCCCAAABBCCCDAABBBCCDDEBBBFFFFDEEEEFDDDDEFFFFGGGDEEHHGGGIIEHHHGGIIIHHHHGIIII',
  'AAAABCCCCAABBBCCCDAAABBCCDDEBBBFFFFDEEEEFDDDDEFFFFGGGDEEHHGGIIIEHHHGGGIIHHHHGIIII',
  'AAAABCCCCAABBBCCDDAAABBCCCDEBBBFFFFDEEEEFDDDDEFFFFGGGDEHHHGGIIIEEHHGGGIIHHHHGIIII',
  'AAAABCCCCDAAABCCCEDAABBCCBEDFFFBBBBEDDDFFFEEEDGGGGFFFEDGHHGGIIEDHHHGIIIEHHHHGIIII',
  'AAABBBBBCAAABBDBBCAEAADDCCCEEEFDCCGGEEEFDCGGGEEFFDCGGGFFFDDHHGHFIIDIIHHHFIIIIIHHH',
  'AAABBBBBCAAABBDBBCEAAADDCCCEEEFDCCGGEEEFDCGGGEEFFDCGGGFFFDDHHHGFIIDIIHHHFIIIIIHHH',
  'AAABBBBBCAABBBDBCCAAAADDCCCEEEEDCCFFEEEGDCFFFEEGGDFFFFGGGDDHHHHGGIDIIIHHGIIIIIHHH',
  'AAABBBBBCAABBBDCBCAAAADDCCCEEEEDCCFFEEEGDCFFFEEGGDFFFFGGGDDHHHHGIGDIIIHHGIIIIIHHH',
  'AAABBBBBCADABBBCCCDDABCCCCCDAAEEFFFFDAEEEEEGFDDDDEEGGFHHHHHIGFFHHHIIIGFGHIIIIIGGG',
  'AAABBBBBCADABBCCCCDDABBCCCCDAAEEFFFFDAEEEEEGFDDDDEEGGFHHHHIIGFFHHHHIIGFGHIIIIIGGG',
  'AAABBBBCCAAABBBCCDEAABBCCDDEEACCCFFDEEFFFFFDDEFFGGGHDDEEGGIIHHDEGGIIIHHHGGIIIIHHH',
  'AAABBBBCCAAABCCBBCAADBECCCCDADBEEFFFDDDEEEFFFDDDEEGFHFIIIIEGFHHIGGIIGHHHIIGGGGHHH',
  'AAABBBBCCAAABCCBBCADABECCCCDDABEEFFFDDDEEEFFFDDDEEGHFFIIIIEGHFHIGGIIGHHHIIGGGGHHH',
  'AAABBBBCCAAABCCBBCDAABECCCCDADBEEFFFDDDEEEFFFDDDEEGFHFIIIIEGHHFIGGIIGHHHIIGGGGHHH',
  'AAABBBBCCAAABCCBBCDAABECCCCDDABEEFFFDDDEEEFFFDDDEEGHFFIIIIEGHHFIGGIIGHHHIIGGGGHHH',
  'AAABBBBCCAAABCCBBCDDABECCCCDAABEEFFFDDDEEEFFFDDDEEGHHFIIIIEGHFFIGGIIGHHHIIGGGGHHH',
  'AAABBBBCCAABBBDBBCAAAADDCCCEEEEDCCFFEEEGDCFFFEEGGDFFFFGGGDDHHHHGIIDIIIHHGGIIIIHHH',
  'AAABBBBCCAADBCCBBCAADBECCCCDADBEEFFFDADEEEFGFDDDEEHFGFIIIIEHFGGIHHIIHFGGIIHHHHGGG',
  'AAABBBBCCDAABBBCCEDAABBCCEEDAACCCFFEDDFFFFFEEDFFGGGHHEDDGGIIHHEDGGIIIHHEGGIIIIHHH',
  'AAABBBCCCAAAABCCCCDAABBCCEEDBBBFFFFEDDDDFEEEEDFFFFGGGEDDHHGGIIEHHHHGIIIIHHHGGGIII',
  'AAABBBCCCAAAABCCCDEAABBCCCDEBBBFFFFDEEEEFDDDDEFFFFGGGDEHHHGGIIDEHHHGIIIIHHHGGGIII',
  'AAABBBCCCAAABBBCCDEAABBCCDDEEABCCFFDEEFFFFFDDEFFGGHIDDEEGGHHIIDEGGHHHIIIGGGHHHIII',
  'AAABBBCCCAAABBBCDCEAABBCCDDEEABCCFFDEEFFFFFDDEFFGGHIDDEEGGHHIIDGEGHHHIIIGGGHHHIII',
  'AAABBBCCCAAABBBCDCEEABBCCDDEAABCCFFDEEFFFFFDDEFFGGHIIDEEGGHHIDDGEGHHHIIIGGGHHHIII',
  'AAABBBCCCAAABBBCDDEAABBCCCDEEABCCFFDEEFFFFFDDEFFGGHIDDEGGGHHIIDEEGHHHIIIGGGHHHIII',
  'AAABBBCCCAAABBBCDDEABBBCCDDEAACCCFFDEEFFFFFDDEFFGGGHHDEEGGIIIHDEEGIIIHHHGGGIIIHHH',
  'AAABBBCCCAAABBBCDDEEABBCCCDEAABCCFFDEEFFFFFDDEFFGGHIIDEGGGHHIDDEEGHHHIIIGGGHHHIII',
  'AAABBBCCCDAAABCCCEDAAABCCCEDBBBBFFFEDDDFFFEEEDFFFGGGGEDHHHGIIIEDHHHGIIIEHHHGGGIII',
  'AAABBBCCCDAABBBCEEDDAABBCCEDFFAABCCEDGFFFFFCEDGGHIIFFEDGGHHIIEEDDGHHHIIEGGGHHHIII',
  'AAABBBCCCDAAABCCCEDAAABCCCEDBBBBFFFEDDDFFFEEEDFFFGGGGEDHHHGIIIEDHHHGIIIEHHHGGGIII',
  'AAABBBCCCDAABBBCEEDDAABBCCEDFFAABCCEDGFFFFFCEDGGHIIFFEDGGHHIIEEDDGHHHIIEGGGHHHIII'
]);

const VALID_JIGSAW_LAYOUTS = withJigsawLayoutResult(true, [
  // From https://www.sudokuwiki.org/js/jigsaw3.js?ver2.12e
  '111222333111222333111222333444555666444555666444555666777888999777888999777888999', // 0 = SUDOKU
  '111222233111222233114452333144455633444555666774556669777856699778888999778888999',
  '111222223111222233111555333444453336444456666477756666777555999778888999788888999',
  '111222233111442233111442233554444233555466663755566699777566999777888889778888999',
  '111122233111222333411222333444555663444555666744555666777888996777888999778889999',
  '111222223411222323441155333441155333444555599664665999666677799888677799888888777',
  '112222222112333332111133336441555566444456666445555966477779999877777899888888899',
  '115558888411555888441155588444117777944213377994223377996622337996662233996666223',
  '122223444112223344111223444151133384555663388555667778559667778999966788999967788',
  '112222233111222333411123336441555366444555666447555966477789996777888999778888899', //  9, moonlotus, moonlotus1016@hotmail.com, 25/01/2006, and Leonid Kreysin, Rebecca Schwartz
  '111112222113345522133444552134444452637777752633777559638878859668888899666699999', // 10, Onion, Andrew Smith, andrewgsmith@ntlworld.com, 15/01/2006
  '111152222111555222175555582177999882777999888477999883476666683444666333444463333', // 11, Double Mirror, Fer van Nieuwenhuizen, TeachOffice@Wanadoo.nl
  '111222333111123333411222336445525566444525666444555666777789999777888999778888899', // 12, cyndie.smith@netzero.com
  '111123333111224333122244443125544443225555566788885569788886669777866999777769999', // 13, Tornado, Kathleen R Nicol, kath@nic01.com
  '188888996118889966411899666441199667444119677443322777433352277333555227355555222', // 14, Cross, Andrew Smith, andrewgsmith@ntlworld.com
  '112222333111225333112225633114455663444456666744556699774588899777588999777888899', // 15, Worm, Leonid Kreysin, lkreysin@yahoo.com
  '444455555444435556994233566991223336911222336911122366998112766988817777888887777', // 16, Cabbage, Tim Cieplowski, tjciepl@bgnet.bgsu.edu
  '111222333111222333411522633441552663444555666744855966774885996777888999777888999', // 17, Stripes, leob, leob54321@yahoo.com
  '111223333112223633112225633411525666444555666444585996774588899774788899777788999', // 18, H, Bob & Debbie Scott, krydak@yahoo.com
  '111222223111122233441152333444555633744456663774555666777859966778889999788888999', // 19, Wednesday, G�rard Coteau, coteau41@wanadoo.fr
  '111123333114122333144522236144552236447555266487755669487775669888779699888879999', // 20 zigzag, G�rard Coteau, coteau41@wanadoo.fr
  '112223344111233334112223344155255344165258449665558899666788889667778899677777999', // Legoman Offside
  '111112222166511132665544432655448832654478332654778392654788399657789339777889999', // Anon 1
  '112222233112444233112444233116444833516678839556777899556678899556777899556678899', // Tim Tang
  '112222334112223334111233344115237444555667744555667779855667779886669999888888999', // 24
  '111122222113344422133344442133555442133555669788555669788886669778886699777779999', // 25
  '111123334111523334115523334555522444556622244666662994777668999777888899777888899', // 26
  '111122333111122233144442233445452233445555599665658999666668899677778889777778889', // 27
  '122223333112222333112456663114455663144455566174455669777458899777888999778888999', // 28
  '111112222133411152334466652344667752346687552346887592346877599348879559888779999', // 8 clue jigsaw

  // From Mathimagics
  // http://forum.enjoysudoku.com/jigsaw-layouts-generate-test-t35712-30.html?sid=6523d33401485a4b2414c1d7cc15a060#p310543
  '111123333141123353441122355466662325477762225478766665448779955848879959888879999',
  '111111122333334122344334122344444522655555552665777778669788778669788888669999999',
  '111112222111222323415552333444453333444456666777756666777855596787888999888899999',
  '111111122133444142333554442363554722663854722663855727688855777689888779669999999',
  '111112333414112233444512333446552223446555277866655277888695777886699797888699999',
  '111111112313444442333454242336554222336555277666855277686858777688888797699999999',
  '111111233111222223445522623744556633774456633774455663784885566788888999778999999',
  '111122333144123333444122233441155226475555526477558866997778666999978668999778888',
  '111111222133144222133342225333645555636645575666645777688847779888449779888999999',
  '111112222311112422335662427335564477355666447335564477385866477885899997888899999',
  '111122223111122333145223333445266663445555566744448566777788569777889999788889999',
  '111112222131222444331122445366644455337666455337776665377889955777888959888899999',
  '111112233114422233144225533144225336447555366477588669775588669778886699778899999',
  '111112222133322442133332444156332644156666647556899647555899997855889997888877777',
  '111111123114222223444225533464255733664457733664558737665588777688888799689999999',
  '111112222113112244333333224555363244566666664557868444577888888557799899777799999',
  '111222223111122233414125336444455536477753336475556666477589696778889999788888999',
  '111111122133412222333422442334444456355555556357777766877887666888897669889999999',
  '111111223111422233444422233444425333655555553666578888667778888667778999677999999',
  '111222333141255336441257336411257366418257396448257996488257966488557969888777999',
  '111112222133322224113325444133355544666357444665557778666597788699997778999988888',
  '111111112344444412334544222333552262337555266737755666777885866798888886799999999',
  '111112233111222223144442233445445333645555573666577577668877779688888999668899999',
  '111111223144112233444522233445522336477555336477885566778885666778899669788999999',
  '111112223144122233144125233444425333465555537666587777668589779668889779688899999',
  '111112222113222444133244444133245556333555666355578669777778669777888699888899999',
  '111111223141551223444452233644452233664752833667758883667758888677955989677999999',
  '111122333141122223141222333444453366455555556447756666777888969788889969777889999',
  '111112222131445222331145552633347552663347755866347775866649977888644979888899999',
  '111111122134444442135544222333554222333356666777855666777885569788888869779999999',
  '111112223111222333412255533444256633474456636774458666775558896777888999788899999',
  '111222223441112323445112333445512336447555366477895566777899566787899966788888999',
  '111111223111422223444425233464455233664555733668557737668587777688887999688999999',
  '111111122331441225336444425376444225376666625377888625378888655377988955779999999',
  '111111222111333324555333224556633224576666624577886644577888444578888999777999999',
  '111122223411112233444122333454423336455555556477786656777889666778899996788889999',
  '111112222111322244155322444555366644775333644775553666777883669778883999888899999',
  '111112233141522233441522333441522366447555366447885966777885966778885969778899999',
  '111122233111222233144425233144455333444555666777556668779596668779999888779998888',
  '111112222333311222333441225344461655346666657446865557499855777999887777999988888',
  '111111123441441223444445233677755233667555233667552223667588888677988988679999999',
  '111111222131133222333332242555366644555666444556667444858877777888779979888999999',
  '111112222331422225311426655314466775344467775344667785336697885399997855999988888',
  '111112223444112233445162237455166237458666237458669337458869377558899777588899999',
  '111112222111122322444443332456663377455666337445566637855577777885889999888899999',
  '111111222331122222334125555334555565334444466737777466777789466888889966888999999',
  '111111112333331222443335262744535662744555662744585662747588866777988888799999999',
  '111111122133145222334445262334455762834457762834557766838577766888579669889999999',
  '111111222344152222344152662344157766334555766334458776933958776999958776999888888',
  '111111123114222223444425233464425333465555537666587737668587777688888799689999999',
  '111112222133332222113334552163344455666645555664447758966477788999977778999988888',
  '111111222333122224333152244663155744663555744663558744669958777699998777999888888',
  '111111122333344112533364442553366442577666442577668822577768882599778888559999999',
  '111112223141212333441222333455552363444456666747855556777888966777898969788899999',
  '111111222113122224533223644533333644556666644556777774556788774588889799888999999',
  '111111222133122224133332244553336444566666664555677744558877779588889779888999999',
  '111111233114412233444522233444555233647752283667555888667775888667798899667999999',
  '111112222113122422133334424536333444566666664555777674585577779885889799888899999',
  '111112222113342252113442555633342555633347775666847775666844799868847799888899999',
  '111111122133341222335546627385544627385546627385446627385546677888947779889999999',
  '111111122333311422333444442335545442666555222677575588677777888667998888669999999',
  '111112222341112252341442255344446665334666755366677775338877975838899975888899999',
  '111112222133222224113334444513666644533367774556666784555577788599999778999988888',
  '111112222311414422313444422333345426355555556378586666778888696778898996777799999',
  '111112222311144422333154442333554462738555462738855666788859666778889996777799999',
  '111111233114122223444222533444425533466555337665587777665888777688889799668999999',
  '111112222331111224335667724355667724385567724385566774385566744388999944888899999',
  '111111222331441522333641552336645522733645882776645588766945888776944988777999999',
  '111111122113442222333342225336345555636645575666647577688847777888844799889999999',
  '111112223111222233441223333444455663745555563744556666777788966778888999788899999',
  '111111123441422123444422223545466623555666333576668383577778888579778988579999999',
  '111111122313222224313255244333335444666555444666577777668558797688888797889999999',
  '111112223111422223414422533444455533466653337665557777665887797688887999688899999',
  '111111122133444422133454422633355442633555772688557772668858779668888779669999999',
  '111222233141222223441113333444133566455555556445778666777788866799999868779999888',
  '111112222133122224333112444533336644556666644556677774555899777588889779888899999',
  '111112233112222233414452333414552363444555666747855696777856696778888899778899999',
  '111111122333311222334332252444632655446666655446786555747788588777998888779999999',
  '111111122133333122334433222444445522445555566775566666777886688779888889779999999',
  '111111112344441222334444222333545526373555626375585666777888866777988886799999999',
  '111122233411122333411222333445552636444555666474855566777888996777889996778889999',
  '111112222311444422311546662335544462738555462738885566733385996778888996777799999',
  '111122223111224333512244333515246363555546666757548696777448896777488999788889999',
  '111111122311344422333344422335445422635555572668588577668887777668887997669999999',
  '111111222311142225333442625377746625377746665387746665387844555388849995888999999',
  '111111222133122224553126244553366674533367774536667744558689744588889779888999999',
  '111222223111233333411235533412266555446666655444667785994497885999997888977777888',
  '111112222131222244331154244333354444335555566777756666778759966778888969888899999',
  '111111233114222233414422233444525336475555536477585666778886696778888699778999999',
  '111111233141422223144452223444555233466653337668555777688857779688887979668999999',
  '111112222133312222113333324556634444556666644555576644587777799888897779888899999',
  '111112223114222323114225333444445536477555336475566666777588699787888699788899999',
  '111122233412225236411225336411755366447753366447553886477599886479599986779998888',
  '111111222313144422313555442333654422373654828776654888766555898776669898777999999',
  '111111112331444422335554222333654227386654427388654777888655577886666977899999999',
  '111111233142222233142422533144455533644457773665557778665997978669999978669888888',
  '111111222133113244533333224556636244577666224557686644577888884557899889777999999',
  '111112222331422252331425555331455565331444766838884766888894766989994766999977777',
  '111111122331441222333442225364444425366666665378888865377788555777988955779999999',
  '111112222113122425333122445367774445367674745366677745366889555386889599888899999',
  '111111222333141222553142226553344666533347776555447766588849766888949777888999999',
  '111111112133332222434332222444355566445357566445557666888877676888877779899999999',
  '111111222333341222553441122553346662553347766855547766889944766888947777888999999',
  '111112222113313332144333222444435552444555666755586666777888669788898899777799999',
  '111112222133322222113344444133334554665555544655677778666667788999997778999988888',
  '111111222133332242113335442666335542766555442765588444766588899767788889777999999',
  '111111122113333322443355332444556622774456622774455666788558866778888899779999999',
  '111112222111133322333333224556664424555666444575566644577888888778889999777799999',
  '111112222111314422555344442555336442785536642788533666788883666778893999777799999',
  '111112222131112244533362224553667744533666774553366744588867774558899979888899999',
  '111111222113322244133552644773552664773852644733855644773855669778886699888999999',
  '111112222133322224111324444333355544366555447665557777666687999688887779888899999',
  '111111112133333322333445522667745552677444552677745522667744888668888889699999999',
  '111111222133333222114553332444455322444456666778556666788855699777888889777999999',
  '111111122331455122333456662773456682733456882734456822744456888779556988779999999',
  '111111222331441122344444442333355552663357722655557777688888887669988977666999999',
  '111111222131145552333445552363347552668347522688347727688844777688849979666999999',
  '111112222111322442155322444555336644775536644775533666777883669877883999888899999',
  '111111222133222242113332544333355544666654444665557777665877799868888779888999999',
  '111112222313312222333112444335666644355564447556666477555899777888897797888899999',
  '111112222311444442331144422333355462735555562738556666778889966788888996777799999',
  '111112222331122442331155422333354444663555744666657777886559977866889977888899999',
  '111111122334441122333445522334445222334555677888566677885566777889966677889999999',
  '111111223114222223144423333445525363744555663747585566777786669788888699788999999',
  '111111122344444112333344222333445552663555722655588777666887777699888887669999999',
  '111122223444112233444122333456112637456666637456899637555889777558899777588889999',
  '111111222134441122334544226344554266337555266337855886377885866779988869777999999',
  '111112333114452233144552333144652273446652277846652779888655779886657799888699999',
  '111111222311145552333445522333645527388645227386645777886644777866649997888999999',
  '111111222313311242333552242336552444636654474666855477868855777868997797888999999',
  '111112222131122222333144444353344446355555556377776656777778666999998868999988888',
  '111112222113412552333412552363445552363347757866644757866894777866894799888899999',
  '111111222133142255336142255336442557336842577366844577668849577668849779888999999',
  '111111222133122242333155244336655247366654447368554477668559777868889779888999999',
  '111112222331452522331455522311445666377745556333744886997774866997974866999988888',
  '111111122333344122333554122336554442776654422766655488779655888779668888779999999',
  '111112222133332222113344442513666644553367744556666784955557788999977778999988888',
  '111111222113331244533322244533226644536666674556688774558887774558977799888999999',
  '111112222133411222133444422333444456355555556357777666887777669888997669888899999',
  '111111222313441122333444442334455222335555566777558866788888666779988696777999999',
  '111111222311144442335554422633754882633754882633754882667755588677779998666999999',
  '111112222113222332143333352146663555444666555444766658947777758977999788999988888',
  '111112233441222233411122333444552336445555566477855666777889996778888966778899999',
  '111222233111223333141122353444462355466666665447865555747889959777788999778888999',
  '111111222133414222134444425334466225336666655377668855378888859777898559777999999',
  '111122223144123223114553223144453333444555666777756668799755688799798668799998888',
  '111112223411212333414222353444662353746666653747866555747888595777898995788899999',
  '111122223141123333441123335446222355466666665447888655477789955777789959788889999',
  '111111222311145552333344552367345522367348528667748528677448888677749998666999999',
  '111111222333311122343332222344444556445555566455666667888877767889997777888999999',
  '111112222133122222113345555663344775663444755633447755666647788999998778999988888',
  '111112222311412552333415552333455652783444652783884666788894666788794996777799999',
  '111111222331142222331344442533336642556666644756688884755558988777759988777999999',
  '111112222113112222333333244555363444566666664555767444558777777888899799888899999',
  '111112222111332222133332455666333455664444455664777555664877779888877999888899999',
  '111111112334444122333344442553336422566666662557688822577778888559777788599999999',
  '111122223111124233515524333555224366577444336557488666777486696778489999788889999',
  '111111122134144222334442225344662555346666675333866775388877755888779759889999999',
  '111111122314444222313444422333455526373555626375558666778888696777888896779999999',
  '111111222131142222333445522633345555633645775666647775886644777888849979888999999',
  '111111122311322222333322455636344455666347555664447575664887777888887997889999999',
  '111112233144412333144512233446552237486555237486655277886695779888697779886699999',
  '111111222331112244333522446375522446377555446377885546377885666778899966888999999',
  '111111222311132222333333424566663424555564444575866664575888888777789998777999999',
  '111111122131455222333452226343457726344457776384457676388857666888557969889999999',
  '111112222113122442133225444333355544366357447665557777666588779866889799888899999',
  '111112222131112224335552444333652474336654477636854777666855577688899979888899999',
  '111112222113422442113444442333555542333356666785555666788888699788778699777799999',
  '111111222113442222133444425336664425336764655387766655387777559888877599888999999',
  '111122333141123335441126635442226335422666775488677755486679955488879959888779999',
  '111112222331114425331444225344446625376666625376688885377888955378899955777799999',
  '111111222311312222333333244555563244566666664557864444557888888777798998777999999',
  '111112223141512323441522333444552633744852663774855666777885966787895969788899999',
  '111112222113112224333333224555336664555666444566677444588777777588899799888899999',
  '111111233111222223444444223455644623556666633576886338577888888577777999557999999',
  '111112223111122233442223333444445536475555536475566666777788866778889999788899999',
  '111112222133311222334414422354444466355555556337777756887797766888996669888899999',
  '111111122313314552333344552367745522366745228667745528677448888677498898669999999',
  '111112223114422333144422233145523336445555566477785569778886669777886699788899999',
  '111111222334441112334455222334455226334457766388557766888557766899977766888999999',
  '111112222113442222133546267133546667335546677355546778359546778999944788999988888',
  '111112222113334522163344522163347525666347555686347759886447759886477799888899999',
  '111112333111122333444223335442222665446666655466777755488877555888779999888799999',
  '111112333111422333515442233555642237555642777586642777886644797888664999888699999',
  '111111123441152223444452223446452323666555333676758388677758888677759988679999999',
  '111111222111333232444453332644455732684555732684557772688857777686888999666999999',
  '111112233144122223441123323445525333444555666777585566787789966788889669778899999',
  '111111222311441222333544442333544462735555562738885666788885666777988996777999999',
  '111111222344441122334441225333644625376666625376886555377988855779988885777999999',
  '111112333411152333416552733446652773446652277844652277884655297888659997888699999',
  '111122233141222223141255333144553363444456666747755668777559868799999868779998888',
  '111112222331142222331142444335444466355555556337777566777879966888879966888899999',
  '111111222111222233445225553445555563466666663467777733477788733448888999888999999',
  '111112222333311244333511244363555244667752244667555848667995888667998888777799999',
  '111112223111422233441422333444442553665555533655788888666778988667778999677799999',
  '111112222131333222133343252136644555666444555666445578969747778999777878999988888',
  '111111222133332244153332224153362644556666644556768849577788849557788889777999999',
  '111122223411112223444122333454466633456467637556667737555889777588899997588889999',
  '111112222311122224333152444363555744663357744663555747666859777688889997888899999',
  '111111233111455233444452233444652233476652238776652888776658888776558999776999999',
  '111122233111222333144222333144455536444555666475556668777999668777999888779998888',
  '111112222113444442113445442333555622733356662773555666788588699788888699777799999',
  '111112222133312242133315242336555444636654474666555477868597779868897779888899999',
  '111112222133112222333314442333544544665555544665665777866697777888899779888899999',
  '111122223141222333441252336411553336444555666477755886477959866777999868799998888',
  '111111112334444412355464442356667772355567772355566672388868772398888822399999999',
  '111111122133333224133322244553226644556666644556677844557778889577888889779999999',
  '111112222131122222431333555433336555466666665444677775444777875999998878999988888',
  '111111122113333322433455332444455222446457277666557777688557887668888899669999999',
  '111112223141122333441522233444552633744555663774855666778885966777889969788899999',
  '111111122113333322443353322444455522644457772665557777668858877668888899669999999',
  '111112233114122333144222233444452533445555566775856666778888669777889699778899999',
  '111111112331442222335442222335544444333555666777775566888877566888877966899999999',
  '111111222331122222331455266334455667334456677344556677448556977888889977888999999',
  '111112223114412233414522233444555233664457733668555777668885797668897799688899999',
  '111111112134442222333444422333544526375555526375885666778888666777788869799999999',
  '111112222133312242156332442156333344556666644557777648955977648959987778999988888',
  '111112222112222332114333333444444553445555566755666666777777688977999988999988888',
  '111111223144222233144222335146663335447666355477766658477999558779999558799888888',
  '111123333111222334551126334522226344556666644557688884577689944577888999777789999',
  '111111223414412233444212533464222533465555537665888737665898777668897797688999999',
  '111111222313441222313455522333456662334456677844456777885556797888966797888999999',
  '111111222133331224135366244335662244355762448557766488557668489577988889777999999',
  '111111222311144442333445442333355542666357222685557777688588777688889997666999999',
  '111111222133222222113344444333355544663555744665557777666667788999999778999888888',
  '111112233112222334511222344566633344557666344557776664557888994577888899778899999',
  '111112222111333222133343322535544666555444666555446676887747779888777999888899999',
  '111112223114411223444522233464525333464555737666585737668885777688997799688899999',
  '111111222113144422333554422333544462738555462738885666778855666778889699777999999',
  '111112222134411122335444422335444662335555566733888566778888566779998869777799999',
  '111112222112222233415555333416655553476666653477776683444777783449999988999988888',
  '111112222133322222133344455136645555133645778666645578664447778999997778999988888',
  '111111222331441252344461255344662225344666775388866775338967775838977955888999999',
  '111111222131455222331445266337745226377745556388745566338744966888774969888999999',
  '111112222133312442533311442553336642756666642756688844755998884755798889777799999',
  '111112223114122333144522366445522336477555336477885566447885669777889699788899999',
  '111112223111422323144442333445552333445652577888655577888677779868667999866699999',
  '111111122333341122553342222533442666535546676555844776888847766889947777889999999',
  '111112333141122335441222335466222335446666655477888665477888955477889959777899999',
  '111111222311142255333446225773346255778346255778346655788344666778849996888999999',
  '111111222131122224533224444533666674533367774536666774555588774588889979888999999',
  '111112223141122233144223335444626635477666335476686555477788559778889959788899999',
  '111111112333332212334442222533546662555546666755546886777744488797788888799999999',
  '111111112333333122443353222444455266477555226447556666777858866779888888799999999',
  '111112222331311122333344522633644552666645555766445885776448888779998988777799999',
  '111111112333332212343332222445566662445555566744445566777788868797788888799999999',
  '111112233411112223444422323455523333444555666777785556787886666788899996778899999',
  '111111122334441122334552222334456627334555677384456677888855677889966677889999999',
  '111111222333331112334335662774455562744456662745556622744588688799988888777999999',
  '111112333444112223444512333445512233466555227886695577888695777866699777888699999',
  '111111122333334122335664122735644442735666482755556482779566488779588888779999999',
  '111111222311445552313645222333645527383645727386645777888645797866644997888999999',
  '111111233114422223414425223444555233466653337668555777688587797688887799668999999',
  '111112222111132442556334422556634727555634777585634477886633477866839999888899999',
  '111112222331422222311444445316664455336764655337766685377777885999997855999988888',
  '111111222113341552333445552363345552663745822677748828677744888677948899666999999',
  '111112222131222224133334444133355554666357444655557778666677778699999878999988888',
  '111111233141152233444452333444552323466652227868655777888657777886659979886999999',
  '111111223111452233444452236477552236477852336478855336478856666778856999788999999',
  '111111222333144422333145442633155442673555842677559882677579888667779888666999999',
  '111111122134442222133345522334445672334555677834566677885567779888866679889999999',
  '111112222333112224533311224555366664553367744566667444588997774588899777888899999',
  '111111122333314122353344622753346662755546662755548862775448868779498888779999999',
  '111111122134444222133354442633555442633357772688555772688857779666888879669999999',
  '111112222311222424315552444313354444333356666777756686777955586797999886999988888',
  '111111222133112242333352442365552447366654447366855547866857777868899779888999999',
  '111112333112222333412452223414455633444555666774556686799956986777999988777988888',
  '111112333141512333444512233446522273446555277846665277886695777888695979888699999',
  '111111222113332242513362442533362444536666674555867774855867794858877799888999999',
  '111112222133112444135522444333522464733555664737885666777885569777899669888899999',
  '111122223111123333444122235446626335447666355477686655478889555777789999788889999',
  '111111223144112233445522333444522233464555737668885777666885577668899779688999999',
  '111111222113332224133552224633354444663555744666657774688855779688877799888999999',
  '111112222133122242133332444156633344556666644557776648555977778959998778999988888',
  '111111233414411233444422223445552223666555333677755588677778888667998898667999999',
  '111111223142222223144555233144453333446457377666657778669555778699999978699888888',
  '111111222311144442333544422635544772633555772633885572668885777688889997666999999',
  '111112222133142252331144552336645552333645777866645577866449977868849779888899999',
  '111111233441121233444222233445552233644555773668855577668888777668989977668999999',
  '111111233414441223444541223466552223667555233677755338677985888677988898667999999',
  '111112222314122222314155566334457766344457776334457766335558786999998786999988888',
  '111112222133122444331152444635552274633357774638855574666859977666889779888899999',
  '111112222113342225166342275166344775663347755633447558639947558699947788999988888',
  '111123333111123333442123566422225566442555766445577776445879766888879999888879999',
  '111111222131444422133334422335554422336555477886655577886677779886666979888999999',
  '111112222113412552133422555333442556337745566377844666777884669877894699888899999',
  '111111223144512223144552323444452333664752833666758888676755889677795889677999999',
  '111122233114122223114455223444455333466457337666557777688557799688889799668889999',
  '111112222111312422333322444335554446377555446377755566777886666887896999888899999',
  '111112222133142222113342555333644555336645577666445777666847799888849779888899999',
  '111112222111133232444433332445553632744555662784855566788886666787889999777799999',
  '111111223114422223414445223444555323666653333676555888677588898677778899677999999',
  '111111222313144422333154422336655472836555472836554477886659777886669797888999999',
  '111111112314442222334544222335554462333754666837755566888775766888877796899999999',
  '111111122133222222133344255663345555633444775666647755668447779888888779889999999',
  '111111222311312222333344522666344555663347755666447555886447777888897997888999999',
  '111111223411222223441523333444525633744555663774585666777785966788888996788999999',
  '111112222313412252313452552313455555333444666777774686977974686979984686999988888',
  '111112223444111223444122233456626337455666337455686637558889777588999777588899999',
  '111122233111222233141445223444455333446555377666557777688577979668888999668889999',
  '111111222113442252133342555663342775663842755633847755666847779868844799888999999',
  '111111122113322222333334552666334452666444555764488555766488888777778899779999999',
  '111111122113333322444435332664455532644457772685557722688587777668888899669999999',
  '111112333411122353412222355466622355477666335447886665447888895747889995777899999',
  '111111123142222223145552333644452773664457733644857773666855579688888879689999999',
  '111112222111133322433333222445553666444555666444755566888777776887779999888899999',
  '111112222311222242311444444333445556333555666355577666777777886979999886999988888',
  '111111122331144442533344442533646622533666772556686772588887772588889977559999999',
  '111111222311444442314455222333455266337458266337558666777558896788888996777999999',
  '111111223141222223441233323444445333455555556777566666787778966788888969788999999',
  '111122233141422233141452233144452333445555566777856669778856969778886969778889999',
  '111111222312222224311544444333554464333555666737755666777775886799999986999888888',
  '111111112344122222344455522333456672333456777834456777885556667888889667899999999',
  '111112223114222333114452233444552633744555663774855666778856699777888699788899999',
  '111111222331114442333444422333355442665555522677558888667777888677799988666999999',
  '111122223111442223155542233155643337555643777566643778669947778699944888699998888',
  '111112222331222224331544444311554644333555666773755886777775866799999866999988888',
  '111112222331415262371445266371745226377745556388745956338744956838794966888899999',
  '111111222133311222333344442334445542675555542675577788677778888666998889666999999',
  '111111222333114422333315422336655422666555444776554488776598888776699888777999999',
  '111111122313144422333344222355444426375555526378888556777886666778889696779999999',
  '111122233444112223444122333451123336455555556477789956777889666788899666778889999',
  '111111222111342222553342662533447766553347766553344776855847766888847999888999999',
  '111111222133122242133325444333325544666555444665578888666578889767779889777999999',
  '111111233114122223444425523444225333466555337666588777685587777688889799668999999',
  '111112223111422223144444223445433333655555553666667577688777779688887999688899999',
  '111222333111223334512263434512263444516666674555869974585869974588899777888999777',
  '111111112133344422333444442553334662556666622566788822577777888557778889599999999',
  '111111222331422225331425255331445555333645777666644877669694877699994877999888888',
  '111112233141222233141452223444455533664457733665557777688857979668888979668899999',
  '111112223411122323441222333444445333455555556777566666777888966787889996788899999',
  '111112223111222333144252333444452663745555563744856666777858669777888999788899999',
  '111112222111122223444526773445556673455666773456677733455687333488889999888899999',
  '111112223111412223444422333445423333455555556777786566777886666788896999788899999',
  '111112222131114252333644552333645522336645577886645777866445777868499979888899999',
  '111111122344412222344412552346442555366666665333788685733798885777798885779999999',
  '111111122333444122335444122335554422633754882667755588669777588669777888669999999',
  '111111222334411122344444522334664552736666652733866855773888885779998855777999999',
  '111111223114452223414552323444452333664752833666758888676755898677758899677999999',
  '111111122331444122533664472538664772538864472533866472538866772559888977559999999',
  '111111233414122223414222333444523336455555556477785666777888696788889696778999999',
  '111111222133122222333124444335554446375754546377755566777789666888889669888999999',
  '111111222331412225331446725344466725384666725384667775384677955388897955888999999',
  '111112222133222322133333344115566644555666444556664477558888887998999887999977777',
  '111111222314444252316444252316442255336666655337788695737888695737888895777999999',
  '111111233142222223144256233144556633774555633774455668779459668799999968779888888',
  '111222223411122233444112333444513336455555556477785666777988666779998886799999888',
  '111222233111122223144152333444555336474456636477555666777859669788889999778888999',
  '111112223122222333111245553666444533776645533776444555766648999777888889788899999',
  '111111122333312224335312224355366624555666444576668448577798488577798888779999999',
  '111112233111222233414242333444442553645555573655877777666878797668888999668899999',
  '111112333111422333155442233555542236557742266587746666887744669888774999888799999',
  '111111222113444442133444222333455262335555566737558666777888669788888699777999999',
  '111111122331451222344456222334556277334456677338455677888456667888956977889999999',
  '111111122313445522313345522333645222366645557888645777886647797886644797889999999',
  '111111112133332222333442222355546666355444667555546667888844777888877779899999999',
  '111111122113222222333344425367747445366777445366767745386665555888888599889999999',
  '111112333411222353411233353462222555466666665444777765848887995848777995888799999',
  '111111222133332224135222444135554444633357774666655578666999578699977778999888888',
  '111111222113441422333344422335444562735555562735888566778886666778988699777999999',
  '111111222111222222334455555333445666377745556333744666777774466888888999888999999',
  '111111122312222224313254444333355544663357744665557777666658797688888897889999999',
  '111112222131222242331144444335444666355555556333777566777778866979999868999988888',
  '111111233114122233444552223444522333464555737666885777688855777668889799668999999',
  '111112222131444222134442255133466665334467755366667558339977758999777858999988888',
  '111112222131222424133552444133655477336654477336554778666955778696999878999988888',
  '111112223112222333142333353142666655444465555446666758949999758999777788977788888',
  '111112222111312224333333224355366444556666644555667447588777777588897999888899999',
  '111112222131442222133546627135546677335546677335546678395546778999944878999988888',
  '111112222131122242133444442335544466335555566337775566877777669878889969888899999',
  '111112222311122244331522444335554444333356666777755566777885966778889996888899999',
  '111112222131444222131444422333454666355555556333757666887777969888777969888899999',
  '111112222111334522167734552666734452867734452867734555866734459886733999888899999',
  '111111233144222233114522236444552336477555336477855666478885699778888669778999999',
  '111112333111422233144455223444455223667458233677558888677558889667778999666799999',
  '111111222113122244333225546337725446377555446377584466375588666778889699888999999',
  '111222223141122223444113333444155536477555336475558666777788666799998868799999888',
  '111112333122112233142222333444556663445555566744455666777888869778899889777899999',
  '111111222333441112333444222355444422335555566778888556777888666799988666777999999',
  '111111222333311122344445522334445226344555776388577766885577776889996666888999999',
  '111111222311322224313335424333555424666654444676555888676588898677778998777999999',
  '111112222133312224153332244153636644553666744556676748559977748599987778999988888',
  '111112222131222224331154444333355446377555446377556666777758866799999868999988888',
  '111111122133334122533344662555346622775348622775548666755448886779488889779999999',
  '111111222333341122333441225366642255366645557668845557688944777889947777888999999',
  '111111223144222233144552333144522633744555663774885669777855669778888669788999999',
  '111112222331111422335664422335664472335564477835566477885566477885999977888899999',
  '111112222113122244333122445336444445366666665377777655377889555778889599888899999',
  '111112222333111224333152224363555444366654447666555747688859777688999777888899999',
  '111112223411112333445222363445223363445555566747788566747888566777899996788899999',
  '111111122133333422135334422635554222666654444777655584776688589776888889779999999',
  '111112222131122422335166427355556477335666477335644447385669477885889979888899999',
  '111111112333444412534464442533366222533666772555667772588868872598888777599999999',
  '111122233111222333141522333445552663444456666744855566777885969777888999778889999',
  '111111222111344222333354442363355422366555447886557747866657777888667999888999999',
  '111111122133444422133344422536664772536864672533866672558887779558888779559999999',
  '112222223111142233511144236555543336577546336577746666578449996778849999788888899',
  '111111222133144442133344422333445522365555527665588777668887779688889779666999999',
  '111112222331122244311524444335524466337555466337785566777785996778889966888899999',
  '111111122333331122435536772435636772455666772455686782455687782449988888449999999',
  '111122233444122335411122235416623335476666635477786695478889995477889555778889999',
  '111112233114122223414422533444425533664555733665587777665887797688889799668899999',
  '111111222113332222133333224555366644555666444556667444588777779888877799888999999',
  '111111122311444222344444222333455562333555666735558666777888886777888996779999999',
  '111111233144112223444222533444225533664555733665588777665888777688899779668999999',
  '111112222331122224333115524336655444336654477666554477685599777688889977888899999',
  '111111233144122223145552223444452333664457733666857777688855579688889779668999999',
  '111222233111122333114225333444525366444555666447585666777588699777889999778888999',
  '111111222133311244333352244365552244366752448667755548667758888667998889777999999',
  '111111223122222223144444433144453333655555553666657778667777778699999998699888888',
  '111111222113313222433333322444555562444456666745555666778888886777898899777999999',
  '111112222331442555331142555333142255633742885667749888666749988666744988777799999',
  '111111222133333222134443225136643525666645555676845589677844489777888889777999999',
  '111111122133331422333355422366665442776555422766544448776558888776988889779999999',
  '111111222111333224553322224533666644553367744556666774588887744588777999888999999',
  '111112222113332422113532442333554444636555474666655777866875799886877799888899999',
  '111111223144222223144422333144555533664457733665555778666997778699999778699888888',
  '111122333111222334115622334555622334556666644577886444577886499577888999777889999',
  '111111112314442222344442522333445562333555666735588666775788886777788896799999999',
  '111111222134441222134452226333455566344457776335557666388857769888977769888999999',
  '111112222131122244331552444333552446337754466377855666777855966778889969888899999',
  '111111222111334255333344255333642255777642555776642888776448888776488999666999999',
  '111112333144112233444152233445552236478852236478855566778859666778899669777899999',
  '111122333122222344111233334551263644556666644556768944577778999557888889777889999',
  '111111222311222244331255444333254464733555664737758666777558966778888996888999999',
  '111111222341441225344441255346646225337666255377686685337988885377988985777999999',
  '111112222133114222333144422353646447356666647355656747885559777888599779888899999',
  '111111112313322222333422555336424555366444557666484577666884777888887797899999999',
  '111111112333333122345536662744536622774555622774485662744485568779888888799999999',
  '111112222333114422333314422355514442366555447866695557886697777886699777888899999',
  '111112222333111122334445226377745256377745556378745556388744466889999666888899999',
  '111122223111112233444442233456642337456666637455876637558877777558899999588889999',
  '111122233114442223114522233414553333444555666777755686779995688799966688779998888',
  '111112222111122333444223333445525663744555663744585566777788666777889999888899999',
  '111122223111223323114233333554266666554444466555557466888887499878877999877779999',
  '111111122333341122335442222355542667355546667355846667888844677889947777889999999',
  '111112222131144222133444425334464625336666655376868855378888559777889959777799999',
  '111111122133334422133444442553336642576666642576688822577777889557788889559999999',
  '111112222341111255346222255346777775346666675344444675338888675338999975888899999',
  '111111223144422223144225233144555333446653377666555778669599778699997778699888888',
  '111111222131122224335562724355667744355666774335566774385867744388889949888999999',
  '111112222133111222443333352446636352444666555748686655748888855777999889777799999',
  '111111122133451222333452226344452776344555776344857776388857666888957669889999999',
  '111112222131412552333412255336442557336645577366844577668894777866894979888899999',
  '111112222113312442333512442336554442336555477866655477866895777866897799888899999',
  '111112222131112442333322244555366664533367774566667444558887777855899979888899999',
  '111122223144422333114522233114555333444456666777555688779995688777996668799998888',
  '111111112133333322444445322464455322467555328667558828667588888667777779699999999',
  '111112222131112242333335442633355442666555444766558884766588888767799989777799999',
  '111112222333312244335312244365112444365555547666899547668897577668897777888899999',
  '111122333141122233144555233144452233446457277886657779886555779886669979888669999',
  '111112222111122223444556273445566673455567773456667733458677333488889999888899999',
  '111112333411222223411552333444552633444752666884755666888755996877777996888799999',
  '111111122131422222333445252333645555336645577666645777868644777888884979889999999',
  '111112233114422233411422223444455533466555337665557777688887997668887799668899999',
  '111111222111333244333322224533662444556666644555766884577778888557888999777999999',
  '111111222331122242331325244633335444665555544666577774668587977868889977888999999',
  '111111222331312222333312424536666444535564474555666674585897777888897977888999999',
  '111112333411122233444122333445552263445752566847755566888779666887779996888799999',
  '111112222113312442133332444533322664556666644566778884555788889755798899777799999',
  '111111233411212223444222233444553336475555536477755666778888666788898996778999999',
  '111112233141222235441262235441666335477763335477666855479969855479999858779988888',
  '111111122131332222333322244355544444366555447666665557668887777888877979889999999',
  '111111222133342222113442562333455567344456667345556777845866799888867779888999999',
  '111112222113122425163722455663777455633374445663777455663887459683889499888899999',
  '111111222134444442133456642173455662773555622733558629783358669788888869777999999',
  '111111122113333322433443322444445522645555572665577777668877887668888899669999999',
  '111112222111122333444552223464557333464457737666455737688855777666889999888899999',
  '111112233144512233114522223444552633774456633774855666788885699778895669778899999',
  '111112222113132242313332244333555544666654444665555777668877797868879799888899999',
  '111111223144122223144425223446455533466653337665557377688587779688889779688999999',
  '111111222131422222331445526337745566377745556337745566387744966888884969888999999',
  '111111122333331122443332222445535662444555666744585566777788866779988888779999999',
  '111111112314452222334552222344456666344457776333357776888855766888857796899999999',
  '111111222333441122334441222334544426355555556378885866777988866779988666777999999',
  '111111222333311122443322225463335555466666665444477765488887755889997777888999999',
  '111111222333331122334335162744445662745555562744566662748599699778899999777888888',
  '111111222331113322433333222455566662444555666744445556777888886778899988777999999',
  '111111222113414222333444442333545542633555772685585777688888777666898799666999999',
  '111112333112222233114425333444225563444555666745588666777586699778888899777899999',
  '111112222111312242533322444533366664553367744566667774555887774858897999888899999',
  '111122223122222333114453333114455536444456666475556688777756688777999998799998888',
  '111122223122223333112455333411455366444456666447556886777556988777799998799998888',
  '111112233114552333114452333444452326477752226478756666888756699888755699887799999',
  '111111223144422223114223323444553333445555566777755666787788699788886669788999999',
  '111111222133141222334444422334455462335555566738558866778888866777989669777999999',
  '111111233114222233144522233444552336474555636477855666778885669778888699778999999',
  '111112233411222333411255333444255366442257766448557666888557996888777996887799999',
  '111111222334441122344451222344455626333555666373558886777958886779988866777999999',
  '111111122133222222133345552663344755663444755663447755866647779888888779889999999',
  '111111122134144425334444425366662225337762255377766665378888855378889859779999999',
  '111112222311422225361477255361477755366444775336664875339664875399994885999988888',
  '111111122133314422333444422344452226355555556377758886778888666778896669779999999',
  '111112223144112233444152233444455223666555333677558888667759888667799889677799999',
  '111111222133442252133342555133342255666347555668847779666847779868844779888999999',
  '111111122314415222333445222334455672334456677834556677888566777888596697889999999',
  '111122233111223333411233535466222555446666655444777665484887995888877999887779999',
  '111111222112222233414455233444453333446555377666657777668557797668888899888999999',
  '111222333111112344551222334566223334556666644577788664577888944557899999777888999',
  '111222233111112333144422333444522536445555566475885666777886669777899999778888999',
  '111112222311112442333444422355554442333356666788855556778888666788799996777799999',
  '111111122333331122343455222344456622344555667884456667888556767889977777889999999',
  '111111233414152233444152333444552223647752283677755888666759888667759898667999999',
  '111112333144412223444112233456622237456666637458886637558899777588897779555899999',
  '111111222333112222333142442355544446377555446377775556877879666888899666888999999',
  '111112222133322322143333325144466655144666557446665557489999957889889997888877777',
  '111122233111222223114445323444545333446555377666575777686577799688888999668889999',
  '111112222131144442331544422333555442633357772688555777668885977688889979666699999',
  '111122223112222333411152333444453366445555566447756666777859996777888899788889999',
  '111112223412222333411123333445523666445555566444785566777789996777888896788899999',
  '111112222131122442133342442335444466355555556337777566877876669877889969888899999',
  '111111122133412222333422555336442555336645577666844577666884777888894779889999999',
  '111111222133312222443313255466333255446666655447888665447898855777798889777999999',
  '111112222133142442133344442133555542673358842675555889677778889677679889666699999',
  '111112222133332222133332444155636444156666647555686447555988887999988887999977777',
  '111111233144122233444155233444455233677752223667558888667559888667779889667999999',
  '111111222133331124536332224556633224555666444577886644577788684579988889777999999',
  '111122223141122223141552333444453336447555366477756666777855969788889969788889999',
  '111112333112122233412223333455526666444555666444475556888877796887779799888799999',
  '111112222333111244333312244556336224556666644577688644557798888557999888777799999',
  '111122223141222233441552336411553366447753366447755886477955866779999868799998888',
  '111111112133442222333542267355542667355842667355846667358846777888844779899999999',
  '111111222113442222133452267333455567344456667345556777348856779888866799888999999',
  '111112222131113222433333552436663552444666555744866685744888885777899989777799999',
  '111111222344451112334552222334456772334456677833456677888855677899956667888999999',
  '111111222133112224333522444333525444663555744666585777666885777688899779888999999',
  '111112223411122223441225533444455333466457337666557777665588977688889997688899999',
  '111111222134441122334444522336446552736666652733688655773888855779988859777999999',
  '111111222111322224333322544633335544666555444665577774665887777688887999888999999',
  '111112222133122442333112544363332544665555544665788848665799888766779889777799999',
  '111111122113442222533342622533346666535347676555547776885847776888844799889999999',
  '111112333112222344122223334156633444556666644555776648577799998557999988777988888',
  '111112333111222333441222223455566633444555666774445556788888966777888999777899999',
  '111111112344444122333344422335542226375555526377785566778886666779888886799999999',
  '111112223444412323444112333455512233467555238667795558666799888676798888677799999',
  '111122233111223333141223566442233566445555566445778866445788969777788999778889999',
  '111222223411122223445116333445166377445666377445668377555688377599998887599999888',
  '111112222331111222334442255344466675346467675346667775338877755888999955888899999',
  '111111223411152233444552333644752283644752883647752883666755888667759998677999999',
  '111111223114522223144552233444452333466457337666857777668855779688885799688999999',
  '111112222131342222131342555333345555663444755666647777666847979888847979888899999',
  '111112233114222233414422333414525533444555666775585696777886696778888699778899999',
  '111111222133321224133322244335554444365654547666655577668887779688987779888999999',
  '111111222311441222333344442333444552635555572655888777688887777666988997666999999',
  '111112222131312224133333244556633244556666644557886644557888889577798989777799999',
  '111112222113312424133222444335554446337555466377755566777888669787896699888899999',
  '111111222133331122334433322555466662755444662755554666778884488779988889777999999',
  '111111122333114442333344442553446662553666722566688722588887777588899777559999999',
  '111111112133442222333542222335546667355546667355546677888846777888844779899999999',
  '111111222313122244333122445367772445366777445366877745366889555668889595888999999',
  '111111223414411233444555233466752238467752238467752338667555888667998898677999999',
  '111111222113422222513442666553347776553347766533347766555844796888884799888999999',
  '111111223111222233444223335444263335466666665477768555477788555778888999788999999',
  '111112222333311112443333222445553662444555666744855566777888866799998888777799999',
  '111112222331142222311444255344464655346666675336767775338777995888879955888899999',
  '111112222311122442333134422533336444566666664555677774885579777855889997888899999',
  '111112222113342552133445552133345552673348852677748889677744889677648899666699999',
  '111112222113344442113344542333445522363555727665588777685887799688887799666699999',
  '111111112333333312443566622473556662477555662477755862447775822498888888499999999',
  '111111222331142222335142627355544677355546667335446667385849677888849977888999999',
  '111112222311112422335554422333354446377754446377756666887755566887899996888899999',
  '111111122333114422335444426375554226378854426377855526378888566778899666779999999',
  '111112222131122242333142445366644455377666445337776665377879555878889959888899999',
  '111111122344141222334444422335554662733555662733855566778888866777989886779999999',
  '111111122334444112334442222334545562733555662735585866777788866799888866779999999',
  '111111222131133322333334225666444525676645525676444555677488888778889989777999999',
  '111112333144222233114255233144552233444456666778855669778558699778888669777899999',
  '111112233144122333114452223444452233645555573668857777688857799666889779668899999',
  '111123333411123335411222235446662235447666255487766655487777995488879995888879999',
  '111111222333111422553666422553367442853367742855367744885666744885999777888999999',
  '111111222333331112433532222445532666444555666444785566777785886799988888777999999',
  '111112222131122442133522444663555774633357774633555744666885779866889979888899999',
  '111111122333314522333314552666344522776645522776448555766498888776498888779999999',
  '111111222344411222334512226344555666344457776333555776388895766888997776888999999',
  '111111122334142122334442222334545566344555776335575766888877766889879766889999999',
  '111122223114112323444122333455552333444456666777855556777889666787899699788889999',
  '111112222133312222113333342566664344555564444557566664857777799888897779888899999',
  '111112333111122334555223344522223364566666664567788884557788444577889999777899999',
  '111112333111122233444422333442225553444555666755588866777886666778889999777899999',
  '111112222333122242311125444336555447336654477366555477666589997868889777888899999',
  '111112222311415522331445522333345552663347755866647777886644977886694997888899999',
  '111112222133311422133444442335544422335555566778885566788888669778996669777799999',
  '111111222331112222433555226433356666443357766444457776488555776888899977888999999',
  '111112222131422222131445555336645577333645777336645577666644878999994878999988888',
  '111111122334144125334444425366477725366777225367778225368888855369889855669999999',
  '111112222311142552331442252633347755666347555663347775868844977866849997888899999',
  '111112233114222233144422333144525533444555666775585669777886669778888699778899999',
  '111111222331144442331444422333355422635555572668557777668888977688889977666999999',
  '111111122113322222433325555436326575436666675434687675444487775888887799889999999',
  '111112222133445522113645522133645527333645777386645779886645799886644779888899999',
  '111122333111222233114522663444552633474555636774855666744885699778888999777889999',
  '111111122341441222344422252334466655334666755336667755838887775888977975889999999',
  '111111122133334442133444422333454222655555552666757888667777889677788889669999999',
  '111111122311222224333355224363355444666357444666557747688557777688888997889999999',
  '111112222333311122343335162444355562744456662745558666749588868779998888777799999',
  '111112333122222233144552233114456633444555666774456688779955668779999998777988888',
  '111122333112224433112522433162555433666654444776555849776885899776688899777889999',
  '111112222111122233444422333546444433566666663557777673555887777558889999888899999',
  '111111122314442122344442222345555666343356676333555576888877776889877796889999999',
  '111111223441511223446522233446552337446852377466855377668885377688995977688999999',
  '111111112333444122334452222334556672834555672834455677888856677889666777899999999',
  '111112222133222224113355444163357744663555744663357748666557788699999778999988888',
  '111111122313314442333344442355554222366754228666755558677778888677798898669999999',
  '111111122133444222133444425334462225366666665377768855378888559777888559779999999',
  '111112333412222223411123333445525366444555666447585566777789996788888896777899999',
  '111112333441112333442212233445222536445555566475888566778898866777899966777899999',
  '111111222331312244333312224536662444535666474555866674588897777558897977888999999',
  '111111222333112242333512242355566647355666447356664447858894777858899777888999999',
  '111112222112222233411555553455556633446666633446677773477777883449999988999988888',
  '111112333411222223441122233444555533464457737665555777668889977688888997666899999',
  '111112222311144442315444422335554662733356662733855566778888596788889996777799999',
  '111122233144522333114552233114452336444752666488756699887755699888775669887779999',
  '111112222311114422334444442333455552633357772655558777688888877668899997666699999',
  '111112222111222332415556732445566733455666773445667733845677793844888999888899999',
  '111112222333112444333512244633515244637555284667595884667795888666799888777799999',
  '111111222331411225344422225346667755344666775334466675388887775388997955888999999',
  '111111112133333442335444442335554222666754222666755588677777588677888889699999999',
  '111111223441112223445552323444452333664752833666758888676755588677799988677999999',
  '111112222341122552346112755346662775344467775344866675334899675833889975888899999',
  '111112222331444422311144522334445562333555666735588866775889996778888966777799999',
  '111123333411223355411122335466662235447762255487766665488779995448877995888879999',
  '111112222311122452316772455366672455366777445336874445336877495836889995888899999',
  '111111233111222233444442233544662273546666673548866773558877777558888999558999999',
  '111111233114412223444422323445522333465555537666885577686887777688897799668999999',
  '111111233112222233144425533444225633474555636774588666775586669778888899778999999',
  '111112222331112224335152254365555554366666664377777764378879744388899944888899999',
  '111112333411122333441222233455552236444456666478855556778888966777889996777899999',
  '111111222133333322133435662144455662774555622744556668744596998779999998777888888',
  '111112222112222234155333334155336634556666644576677448577777448579999988999988888',
  '111112222111122332444425532446655337446653377466553377865587777866889999888899999',
  '111111233144212233145222236445552336447753366477855566478888569778898669778999999',
  '111111223141122223444442523444655523666653333675553888675788888677779989677999999',
  '111111112133333322334444222344455222445555566777556668777666688778888889799999999',
  '111122233411223333411222355416626335444666555477686695447888995777788995778889999',
  '111112222111332342155333342556663342755666442758866644758888449758788999777799999',
  '111122233412222235411123335416663355447666355447766685477798885479999985779998888',
  '111112222131132222133333442553636644555666444556676744855777779888879979888899999',
  '111112223111122323444423333445223536445555566475788566777786666787889999788899999',
  '111112222313112422333315442366335542666555444765588448766598888776799898777799999',
  '111111112344444212333454222335554262337854266737855566777858666797888886799999999',
  '111112222311122442315526477355626477335666477335686447335684497855889997888899999',
  '111112222111322324156333324556633324555666444578886644578888649578778999777799999',
  '111111222133141422333444442333354542665555522675758888677777888667979889666999999',
  '111111222133344422133334442133444522655555552665777889677788889667778889666999999',
  '111111222111222233455672233456677733455666773445556673448856773448888999888999999',
  '111112233144122233114422333444422553645555573655887777666887799668889779668899999',
  '111112222333311112333444442334445552664555722655577788677777888699998888666699999',
  '111222333111122344511222344567727344566777334556787734556888994556889999666888999',
  '111112223114422233114225333444255533464457737665558777666588799668887799688899999',
  '111122223441112233444122333455123336445555566477789556777889666778899966788889999',
  '111112223112122333412225366444455336477555336477556666447588896777889899788899999',
  '111111112333331442553664442553364442573666842577768822577766822577988888599999999',
  '111111112333441222533342262555347662855347662855347666858847776888944777899999999',
  '111111222314112222334422555344666675344467775346666775333887755888899795888999999',
  '111111112314444222334455622334445622733555662773588866773558866777888896799999999',
  '111122233111222333112233334555266644555666444556667444588887799888777999887779999',
  '111122233111222223141523333444555636474456636474555666777785969788888999778889999',
  '111111122311444222344454622334554662733555662733855866773858886777888996779999999',
  '111111122344412122344442225336442655336666655336788655377788885779798885779999999',
  '111111123444412123442222223445553333646555373666655577688888877689897777689999999',
  '111111222133322222114333332444535566444555666445575666877777699888887779888999999',
  '111222233441222335411233355411233665416666675466889775448889775488999755889999777',
  '111112222131444225131644225336644425337666455387776655388776959388777959888899999',
  '111112222331112422351662447355562477335666477335864447355866947885899977888899999',
  '111112222331412252331412555633442555633347775666844775666894977868894977888899999',
  '111112333114122233444122333444222356455555556457888666777889666778889699777899999',
  '111111122333313122443333522444455522644457772665557777665888877669898888669999999',
  '111122223144112333144422333144522533445555566775885669777886669777899669788889999',
  '111111122333144442333145422363545442663555722688585727668589777688889777669999999',
  '111112222311114252336444452336644452738666452738886655738888655737899995777799999',
  '111111122333341122334444442533466422536666672558668772588888877559987777559999999',
  '111112233114155223144452223444452333646752383666758888677758889677559899667799999',
  '111111122113322224553667224553677724553666744583336744588366744588887799889999999',
  '111111222133222244113244445333266445376666645377668555377778599778888559888999999',
  '111111222113312222333345552366444552336645577866444557866647777888897799888999999',
  '111112222113122222133344445336747745366777445367767455366665558999998588999988888',
  '111111122311444422344456442333556622773555622773355666788358886778888996779999999',
  '111111222133411442333444422333455522365457528665557888667777888677997889666999999',
  '111112222111222332414233335444333655466666665446777555477778595877888999888899999',
  '111123333111122333144522336445552266447752266447755566488775669888779999888879999',
  '111123333111223443112233344522266664555267444566667774558887799855877999888879999',
  '111111222333122244331122245367744445367777745366667745368889955668889555888999999',
  '111111122133331222443552226433355626444357666484557776488855766888977779889999999',
  '111122233111222333411222335446663355444666555447766655477888995777888999778889999',
  '111111123112222223444425533444425333466555337666587777665587777688888899689999999',
  '111112222131411255633442225633347255663347755668347775688844775668994979888899999',
  '111122223114122333114223333444525563444555666745585666777788699777889699788889999',
  '111111122311444422334442225366644255337666255337886665377788855778888995779999999',
  '111112333414112223444512233464522237465555537468885737668895777688899797666899999',
  '111112222113333222113432255664433355666444555667774455668874799888777799888899999',
  '111122333114122333114223335444422265466666665467775555488877599888779599888779999',
  '111111222311222224331554424633355444633357774666557774686655977688888997888999999',
  '111112222111332442133322444335352444635555574666857577666887779866877999888899999',
  '111112223144122333441122355467772335467872735466877735446889955666889559688899999',
  '111112222113312424133522444333524446375555546377785666777885669787896699888899999',
  '111111122333122222333142552633344555666444555666447775866849777888889777889999999',
  '111112223411222333441252336441255336447555366477558966477858966777888996788899999',
  '111111122133331222334422252344446657346666657346655557848885577888977779889999999',
  '111112222113145552133345552333344522677348552667448888677748889677749899666699999',
  '111111123114442223444422223445566623555666333576663388577778888577788899579999999',
  '111111123141222223444452223644453333645555573666657773688857777688888979689999999',
  '111112222131452267331452267331455267344456667348556977348856977348856979888899999',
  '111122233444112233411122333455223366445555566447788556777889996778899666778889999',
  '111122233141123333144422233144225563445555566745588669778886669777789969778889999',
  '111111122333333112333442222444445562745555562745566666777766888799888888779999999',
  '111111222133311222333445252363345552666444555766648858767644888777998889777999999',
  '111112222131311222333334422533544666555546666555446776884477777888997979888899999',
  '111112222331412252311422655344446675344666775346677775336887995838897955888899999',
  '111112222331455222331445526311745526337745566387745996387744966888774966888899999',
  '111112333141112335442222235442263335466666665477768855478888855477899959777899999',
  '111112222311112222333334525663434555663444755666474755686477777888899997888899999',
  '111111112333333122444433322546666722544467772554666672558887777559888888599999999',
  '111111122344412122334442222333444556335555566355777666888877766889897776889999999',
  '111111222333122245366122245366177445367777745366778445369998445369998555999888888',
  '111111222111332222443332552443333655466666665446777755844877755888877999888999999',
  '111111112344444122344445222333335522633555772665577777666588887669888887699999999',
  '111112222311112242333344442333354542665555544765658888766668888767799998777799999',
  '111111112333341222355342222553344626555347666585447766888847667888947777899999999',
  '111111222113144222333342252366345555336645577666647557868847777888449799888999999',
  '111112222311112244333522444363552744363555747663855747666885777668899997888899999',
  '111122223144122223441123333455133663445555566744778556777798866799998668799998888',
  '111111233144412223444512223444552323666555333676755888677795888677798889667999999',
  '111111222133441122333445522343455226344457776388557676885577666889977669888999999',
  '111122223111222333415223336415553666444753666444755586477799586777999888799998888',
  '111111222333111224333322244355326644556666644556678448557778888577999888777999999',
  '111112233111122333444552233444455236478852236478556666778855666777889999778899999',
  '111111112133444442334454222334555226337752266377555866777858866788888669799999999',
  '111112222331112242331552244333354444366555447666657777668855977868899977888899999',
  '111111233441222223441125533444225333464555737666588777665589977688888977668999999',
  '111111223141415233444445233647775233647555283667522283667588888667598989677999999',
  '111122233114223333114223533414225566444555666445577686995977688999977688997778888',
  '111111222331441242331444442335545622733555662773585566788888966787988966777999999',
  '111111222133131222333334552636644522666444555776445585766488888777989889777999999',
  '111222233111123333144122233445552636444555666474855566778889669777789999778888999',
  '111112222333112442333312442335616642555666444756686499755789999755788999777788888',
  '111112222111322422515334442555534444556537477666637777866633797886883999888899999',
  '111111222333122244333122444355124444335555566777789556777889666778889666888999999',
  '111111222333112224333312224353566664555564444566664747588897777588899777888999999',
  '111112222333114442333511422335544422366555447886665577886995777866699777888899999',
  '111111222334561222334561122334566627334555677384445677889945677888945677888999999',
  '111112222133312244335512246355712746357777746357897446358894466558896669888899999',
  '111111222133412222133442555333642557366645557366845777666844779888894779888999999',
  '111111112333331422333536422755566422755666442775664442775684888775988888799999999',
  '111111233122222233142533336142553666444555666444755869477775869778888889778999999',
  '111112222113142222333144442355554644333754666773755556877779666888879699888899999',
  '111111223112222233144425336444525336477555336477585666477586669778888899788999999',
  '111111122133344442134442222334545522333555666775585866777788869788886669779999999',
  '111111122333311442553344442533664442536666672588866772588887722588997777559999999',
  '111111222133331442133355442336654422776555422776654488766558889766988889777999999',
  '111111122122222223145666663445556633447756633447755533477777538499999998998888888',
  '111111112313422222333425526377425556377444556377784556387784666888884696899999999',
  '111122333114152223114452233444452233464555737668857777668857799688859799666889999',
  '111111222133222222113334444556633334556666644577776644555577788999999778999888888',
  '111112222111312242333322444356664477355666447335566647555887777858897999888899999',
  '111112223111144223444442223455552333466653337666855557688877777688779999688899999',
  '111112222113222224113354444333555466377754446337555666777756688799999688999988888',
  '111222223122223333111133334555166444556666644555667444588887777888899997899999777',
  '111112222113112422333332424355344444365555547666667557686877777886899799888899999',
  '111111222331441122333544422335554422336654477886655577886665777889966977888999999',
  '111111222311222242315544246333544446337555466377775666378775596878888996888999999',
  '111111122113344422333334522344444522365555527665888887665877777668887799669999999',
  '111122233112223333122425336114455536444456666475556688477596998777799988779998888',
  '111112222111122232455555532466555333446666633444777663847777773848889999888899999',
  '111112222133314442113354442333355442667555422677558888677758899677798889666699999',
  '111111222311122242333322444353664447356666647355566747555887777858889997888999999',
  '111112222111122333442223333444553366445555566447755666777788866777889999888899999',
  '111122233112224333122244355116643355666743555667745588667449998777499988779998888',
  '111111222311441225336647225366647225336647755388647775388647755388944995888999999',
  '111111122133321224333222244335666644355564447556666477558888777588987779889999999',
  '111112222131112442335622477355662447335666477355866447335886477855899979888899999',
  '111111222134411225333462275344466275348666275348667775348867555388997759888999999',
  '111112222333111422333331442355566642756564642756664448755988888775999888777799999',
  '111112222113114242333544442333555422336654477886555777866665777868699799888899999',
  '111112233141511233444522233464552237464852737468855737668885777668995979668899999',
  '111112222111132444333332444533362224566666664577768884555788888555789999777799999',
  '111112222133311224155332224553336664553666744566677744588877449588997779888899999',
  '111112222331112444531222464533277764533777664537778664535888964555899966888899999',
  '111222223141122333141122355446633335446666655477776655447889959777889959788888999',
  '111111222131341222333344242335444442355555556788888566787886666777986969777999999',
  '111112222111322442513322644553366674533367774536667744556887794855887999888899999',
  '111112222131132222133332444335554444366555447666655577666877779888879979888899999',
  '111111122344444112334344522634355522633357772665557872665887877699888887669999999',
  '111112222111322242515344442555334644755536664775733666877773696878883999888899999',
  '111111112133455552633445552633345722663845722663847772688844772688884779699999999',
  '111112222333312224331114224333554444665555544666655777688699977688897777888899999',
  '111112222333111224333312244553362444566666664555768844557798888577999888777799999',
  '111111122341444122344445522333345622733555662773586666775588886779888986779999999',
  '111112222131112222334445552333345555633645775666647777866644477888899979888899999',
  '111111222133313222144333225444633625446666655476886555477888559777898889777999999',
  '111111122133333122334355222444355226444456666477558666777558688779888889779999999',
  '111111222133334242135534442133554422637754482667755889677785589676788889666999999',
  '111112333441112333455612373445662273445666277845566277848596227888599977888599999',
  '111111122313144222333544222336554442336555477866655477888665777888669797889999999',
  '111112222133111422333335422363555424666654444676555848776588888776999889777799999',
  '111111233144422223154426223154446623554666733586677739588687739588887779558999999',
  '111111122113333224553622224553626774553666744533686744588886744588777799889999999',
  '111111222131222224133332444333554444365555547666655777666877779688888979888999999',
  '111111122131444422333345422633555472638854472638555772668587777668888979669999999',
  '111111112333321222333322244355626644555666444556676448557778888777978888799999999',
  '111112222133312444133222444133325544635555574665587779666888779666897779888899999',
  '111111112334422122334442225344462655346666675336867775388877755889887755899999999',
  '111112222133322444133322244113335554666555444655577788669997778666997778999988888',
  '111111222133312224555316224533366724553666744583667774588697444588897779888999999',
  '111111222333441122344444122334545226335555566377585866779888886779988666777999999',
  '111111222331114225334444225344466625374666825376668885377888855377899955777999999',
  '111112333122222333111223343555526644556666644556674444858877999888777779888799999',
  '111111112333314552367344552367345522667348522667748528677448528677498888699999999',
  '111111122134444422135444226335542266337555266337785566377888569778888869779999999',
  '111112223144212233114222533444455536477753336475556666775888699778898669788899999',
  '111112223111222233144432235444633335466666665477776555478875559778888999788899999',
  '111111222333131122433332222434555566444456666445555676888877776889979777888999999',
  '111112222311114425333444425366444225376666625377888665378888555378899995777799999',
  '111122333111223333114422235444622635446666655476886555478885599777788999777889999',
  '111111222133331122333444422335545422666555444776565588776666888779988889777999999',
  '111112223411122223444122333454443333455555556777766656777889666788889996788899999',
  '111111222341112252344422255344466655344666775336667775338887775838899975888999999',
  '111111122133333122444533322444553622474555626774855666778885666779888889779999999',
  '111112223141122333441222533444255533644457773665558777665888977666889979688899999',
  '111111122133322222144322566444355576433357776435557666445887669888887779889999999',
  '111111122133314222333444442333455542665457522675557888677777888666798889669999999',
  '111111123441122223442222333444455533664555733665557777666888877688889977689999999',
  '111111222311122222333324445367444745367777745367666745366685555888889995888999999',
  '111112222134512226134577226134577766334555766334445768399445768399985768999988888',
  '111111123414122223444422233445626633455666337556686377558887777588889797589999999',
  '111111222313144422333142222335544444335555566777775566888879666887779696888999999',
  '111111222331122244333122445366772445367777745366877445366889555668889955888999999',
  '111111122131444442333445522363445772663555722633588727665588777688888979669999999',
  '111122233111222333411255333444225636444555666474588666777558996777888999778889999',
  '111111122133331222334444522344444522335555566775888886775888866777966669779999999',
  '111111123144221223444422223545466663555467333566667373588887777588988779589999999',
  '111111112333334122335534226555534226577444226577486666577486688779488888799999999',
  '111111112133444422533466622534467882533467882533467782556667882557777889599999999',
  '111111122333334122333554122535544222555546666777446686779466888779488888779999999',
  '111112233411222333441122333444522366455555556447885666777889966777888996778899999',
  '111111122333444112334444522333455522663457822665557888665777788699777888669999999',
  '111111222311144442334442222355446662335555566733388556777788866788889996777999999',
  '111122333112222344112333334512266644555666444556667784599999788559777788999778888',
  '111112223114122233444152233444452333645555573666857777668859777668889799688899999',
  '111111223114222233144442223445545333466555337666575577688877779668888799688999999',
  '111111112133444442553644472553666472533367772538666722538886722588888779599999999',
  '111112222133312224113352424333554444663555744666655777686857799688897779888899999',
  '111111122133144422333444222355554422333654777886655557888666777886669779889999999',
  '111112222131411225336442225366647775336647755366647775388844755388994959888899999',
  '111111233142211233442222235446663335447666355477766655478888855778998859778999999',
  '111111222133314442135544222333554442673358842677755888666775589677798889666999999',
  '111112222333312244553611244533616274538666274538696774558996744558897777888899999',
  '111112222133332224133552424113355444666357444666557788696955778699977778999988888',
  '111112222111222323144423333445444433655555553667777577666687779686888999888899999',
  '111111122334441122344444425333662225376666625377766555378888885779988855779999999',
  '111111122134412222334552666334552677334852677334855677444855677888896679889999999',
  '111112222133345222133445225133645555133645778666645778699644778999647778999988888',
  '111112222311112244333552244333355244366654447668557777668855777668899997888899999',
  '111112222311312424331322444533332644566666664556788884555778988575798998777799999',
  '111111222334411252344461252334667752834666752834466755838967775838997755888999999',
  '111112222111122334566227344556277344566777334556778344556788334566889999888899999',
  '111111122131322222333322455666333455664444455664777555664887777888887979889999999',
  '111112233411122223444123333455522363444555666747885556777789666788889996778899999',
  '111111223141122223444222333454466337456666637455667737555888777588889979588999999',
  '111111112313222222333425525333445555663444755666644777686684777888888797899999999',
  '111111223122222233142443333144455363445555566747556668777766968779999998799888888',
  '111122233411222233411252333414555336444456666477555686777959886779999886779998888',
  '111111112331444442355544442355566772356666672355667772388887772388888922399999999',
  '111111222113333222144355262443355266443357766448557766848557669888777799888999999',
  '111112222311122244355125244355555564366666664367777774338789774338889994888899999',
  '111111122311442222333444425336444625376666625376888655378888555777788995779999999',
  '111111112133333332434553222444556622774456622774455666777855686788888889799999999',
  '111111222311444242331344442333355542663555722685557777688887977686888997666999999',
  '111122333112222334112533344612535444662555744666585794668885799688777799888779999',
  '111111222331441122333442225344466275346666675348667775388877555889977955888999999',
  '111112222333142442311144422355554462333754666837755556887779996877879666888899999',
  '111112333144122333114552236444755236487752236487552666487755699888779669888799999',
  '111112222131122424133322444533336644536666674556677774555887779585889979888899999',
  '111112222331222244311554244331554444333356666777755866779755886779999866999988888',
  '111222333112223344511123334555122634566666664576889444577789994557788899777888999',
  '111111223144222233144552233144452333446555377666857779668855779668888779688999999',
  '111112222131422222133444444133345546335555566375576668777777668999997868999988888',
  '111112222333122242331152444331555446377754446377555866777958866979998666999988888',
  '111111112331444222333544426377544426375555526378885226378885666777888966799999999',
  '111122223122222333114443333511546366555546666557546886777744488777999998799998888',
  '111112222311412255331442665377742665377842665377846665377844955338894995888899999',
  '111111122333311222343555552643377752647378752647778852644444858666998888669999999',
  '111111222133331122333444422335544442675555542677775588667777888669988889666999999',
  '111112222311412222331442555366645577333645777336645557666844977888894997888899999',
  '111112223111222223144455323444556333774456633777455666787556669788888999788899999',
  '111111112344441552334644452376645552377666552377786652378886822377988882399999999',
  '111111222134144442134442222333455266335555566337558666777788869788889869777999999',
  '111111222133322244113222444333355444365555547666557777666888799668887779888999999',
  '111122223141223333141255336441255336444257666488557966488557969888877969877779999',
  '111111222133342222133342552136644555336645577666445578966947778999947778999888888',
  '111112222133114222133544224335544444333555666777775566788775669888799669888899999',
  '111111122333314422333315442637755442667555422677554482677598888667798888669999999',
  '111111122113333332443345532644445222644555772666577772685578877688888899669999999',
  '111111223441112223444422233456462333456666637555867637558887777588899977588999999',
  '111111222344111222344442255334442655366666665336788855337788885777999885777999999',
  '111112333112222334112553334662255344666257444668557744688855799688777799888799999',
  '111222223111222233411333335414366655444666555446667585477777885779999888799999888',
  '111111112333332122433332222444555562444456666745555666777788886779788888799999999',
  '111111222331114252333344252363645552763645852766645858767448888767499988777999999',
  '111112222111332222144333562443335566443555766445577766845777669888877999888899999',
  '111112222133312244335311244365332244665555544667788548667998588667798889777799999',
  '111111222311444222331354442333355442637555482677558888677758988666777998666999999',
  '111222233141122223141123333445533666445555566444775566777789969788889969778888999',
  '111111112133332222443533332444556622744555662774455666788885866777788889799999999',
  '111111233411122233444444233456646233755666223775686628775888888775559998775999999',
  '111112222131411222335442266335544446377555446377775566338877566888997969888899999',
  '111112222113332222113442555333445555633645775666644777666844799888877799888899999',
  '111112333122222333114423335166422555666444555666774558699974488999777778999788888',
  '111122333411222223411122333455553366444456666447755556777889996788888996777889999',
  '111112223141122233441222533444455533664457733665557777665888977668889979688899999',
  '111111122133444222135554222333654277366654447338654777888655579888666779889999999',
  '111112333114412333414442233444552223665555522766655888776688898777698899777699999',
  '111111122333334112333344222444444252655555552656777777666778888699788888669999999',
  '111111112344444122344522222334556662334555766833355766888885776889777776899999999',
  '111112223144112333145522333444552263744852663748855666777885569777899669788899999',
  '111122223111122333441225333444525336444555666477585666777588966777889999788889999',
  '111122233441223235441223335411123635466666665476789995477788955478788955778889999',
  '111112233112212334515222234555633334566666664577776444578888494577898899778899999',
  '111111222111333322433332222444355566444555666445557666888877776887777999888999999',
  '111122233144112223114442323444552333465555537666855777686877799688899779668889999',
  '111111222333331222333411522364445552666645555766644458776994888777988888777999999',
  '111111112333144422334445422334545222336555277666585877668588877668889777699999999',
  '111111223112212233444222333444455536477555336475556666777888666778898899788999999',
  '111112223111222323142253333444555633444456666774555666777758869787888999788899999',
  '111111222334441225334461125374466825374666825374668825379968855377988855777999999',
  '111112233411112223444422323455423333445555566777786556787886666788899996778899999',
  '111111223414112223444442333456662233457666238557766638555788888577799898577999999',
  '111112222113314422333514442335554462738854462738855566788895666778896699777799999',
  '111122333412122333412223355411266355446666655447668995447788895777889895777889999',
  '111111122333411225344462225344466625374468825376668885377768885377998555779999999',
  '111111222113342225133442225633345555663347755666647775688844779688847799888999999',
  '111111122133333222144453322444455326478555326478556666778856669777888889779999999',
  '111111222314122222314444552366664455337764455337766665833777795888889795888999999',
  '111111222133131242533332242555633642556666644756886444757788884757989889777999999',
  '111111222333111422333444422355544426378555426378885556778888666778999666777999999',
  '111112222112222333112433335664443555666743555666744455677774899777888899888899999',
  '111111222113312224333552444336652247366555447368854477666855777688897799888999999',
  '111111122331334142533334442567737422566777442556787742566688882569688988559999999',
  '111112222333314242331114442333444542655555552675777888677799988676798888666699999',
  '111112333111142223444442333445222233455555556778888566777866666788869999777899999',
  '111112222131222244131325544333355444336654477666557777665587979668888979888899999',
  '111222223141122223441553333411533366444555666447775886777755866799998868799999888',
  '111111223414122233414522233444555233664457733668555777668885797668889797688999999',
  '111111222113333332414453332444555222464457727666555777688857797688888899666999999',
  '111111112333322212344333222444355562444555666745558666777888668797778888799999999',
  '111111112344441222333444522363344527365555527365887727665888777666988887699999999',
  '111112233144112223445122333445552366477852336447855566777889566788899669778899999',
  '111112222333312242333314442355511442666555444766885559766689999767789999777788888',
  '111111223111222233444252633744255633744555663774558663774858666778888999788999999',
  '111112222112222332411455333444455633774456633774556666777556886977999988999988888',
  '111112333114222233144225333144255536444257666485557669888577669887777699888799999',
  '111112222333122244333122544336155544366157448665557488665997888669997888999977777',
  '111112222113112442333322442356664477355666447335566647855887777855899799888899999',
  '111112222122222333144445633114455633774555633774556688774566668777999998999988888',
  '111111122333312122334522226334555666434456676444555677488885677889897777889999999',
  '111111233114422233144555223444452233664752833667758888677555889667778899667999999',
  '111122333112123334112223344566622344555666444557886664557788899577789899777889999',
  '111122223141122333141552223444553333444456666777755666788855969777889969788889999',
  '111111222113412222333422566344477556347777756344775556334885666888895699888999999',
  '111112222333311224335331224635551244666654444667855594677899599677889999777788888',
  '111111233111422223444425223644445233665555533667588883677578888677778999667999999',
  '111111122331444122333334422355554222366654447888655557886677777889666977889999999',
  '111111122311444422333333442536634442556666622577786682577888888557777998559999999',
  '111122333112222333111442233555446236555546666578544666778844999777888899777889999',
  '111112222333112222333114244335554444635654574666655577668699777888899777888899999',
  '111112222314111422334444422333345426355555556378586666778888866778999896777799999',
  '111112233414511233414552233444652233474652838776652888776655898776995898776699999',
  '111112222134112225134462255334466675344467775346667755338867759388899759888899999',
  '111112222112222324115633334555666334557763344577666444577776488597999988999988888',
  '111111222131222244133322444335544464335555566737775566777886669778888969888999999',
  '111111223441222233441122333444525363745555563747585666777889966778888966788999999',
  '111111122113332224333552224663355244666357444668557744688855777688877799889999999',
  '111112233111222233144442333444522533445555566775885666777866669778888999778899999',
  '111122233111124533122224533666244533676645535776448555776488889776489999778889999',
  '111222233141252336441255366411253336417258396477758996447558966477858969778888999',
  '111122333441112335411222335446222635476666635476888655477888995477899955777889999',
  '111111222344441122334444122333545526373555626375585666779888866779988886777999999',
  '111111122344451122333455222333456627384456627384456777888556777889956667889999999',
  '111111112134444222333442222333444256355555556357888666777788666777888869799999999',
  '111122223141112323444412333444522533465555537665885777666897777686899979688889999',
  '111112222113332224513666224513367744555367444553367784599666784599977788999988888',
  '111112222111222233144256633774455633744555663774556633774458669778888999888899999',
  '111111223444112233444512223464522333465555537666885737688895777668899777688999999',
  '111112222111132222333334442335444444355555556777777566877766666888869999888899999',
  '111122223114222223114445333144555363447753366747555668777566688799999688799998888',
  '111111122133444422133444522633455572633458772635558772665888779668888779669999999',
  '111111122131444225334444255364422225366666665377778865337888855377888959779999999',
  '111112222333311112333344522666345552766444552766648555776448888799998888777799999',
  '111112222314115622314455622333456672333456777834456777884556697884599697888899999',
  '111111222333311244353661224553362224553666744588867744588966747558997777888999999',
  '111222223111123333145122336445523366444555666447785566477889569777789999788888999',
  '111112222133114422134444442333545522333555666775585666788888869778899669777799999',
  '111111222331114442333444222335454422335555566778858566777888666788899966777999999',
  '111112222111133332444332222443335556444555666455577766888877666877779999888899999',
  '111111222131442222133546226333546666333546777555546777588546779888844979888999999',
  '111111222331451222331455226377445266377745556338744556388774966888974966888999999',
  '111111122333445122344445122334455222333456777888556677889566667889566777889999999',
  '111112222333311244536612224533612744553666744553896774588896674558997777888899999',
  '111112222131112444335222466335552446377754446377855566337888566777899969888899999',
  '111122223114123333444122223444155533464555737665558777699998777666698788699998888',
  '111111222331222224311566244377556664377555664377755664338775994388888944888999999',
  '111111222113133324553333324556622224556666644577776644578888844578889899777999999',
  '111111222344155262344152266334152776344852776344859766338859776838559776888999999',
  '111112222111222324551623334555633334576666634577776444577786944587888999888899999',
  '111111222333122224531122644533366644553367744556667774556889974588889777888999999',
  '111112222133114422133334222335544444635555574666665577888677779886699779888899999',
  '111111112344422212333444222334445526335555566375588866777888666797778886799999999',
  '111111233112222234155222334556663334556763644577766644577888449578888899778999999',
  '111122233111222333441122355446623335446666655477786655447889955777888999778889999',
  '111112222133412252133415552633447552663347755866344775866694779868894779888899999',
  '111112222344512222334512667334511667334555677344885677344985677999985667999988888',
  '111112222133332222133445552113344555666347555666447788966644778999977778999988888',
  '111112222333122224353126444353166647355168447356668747555698747599998777999988888',
  '111112222311314422313354422333355424666555444676558888776658898776698998777799999',
  '111111222133331122333422225334455555666444555666664477688884777889977779888999999',
  '111111222133332222113444522663345555633645775666647755886444799888877779888999999',
  '111122333411122333414122335444222365466666665467888555477889595777889995777889999',
  '111112222133122224113524444333555644773356644773555666777785699788889669888899999',
  '111112222111334422313344442333345542675555542675578888677778898667788999666699999',
  '111111222311441225366642255366642755366842775336847775338847775388944995888999999',
  '111112222111122333444442335466662335477862335477866665477855555777889999888899999',
  '111112222333144422335111422335444442335555566788888566778999566778889666777799999',
  '111111222111344222333342255363342555366645557666847757668847777888447999888999999',
  '111111233144512223444512333444552233467752238667755888666795888677795889667999999',
  '111222233111112233144452333445552366447852366447855566777856669778899999778888999',
  '111111122113344222333444222355444462335555566738888556777888666777886699779999999',
  '111112333411222233441152233444552233444555666778855666778859966778888996777899999',
  '111112222131142252333144552363345552363645757866647757866449777868849979888899999',
  '111112222133122244133555224133354444336654477666657778699555778669998778999988888',
  '111111122331444122333344422333544422655555552667775888667778888669777988669999999',
  '111123333111123433122224443222555543666654444765555888766688889776789999777789999',
  '111112222311142222313342555333445555363645757666644777666847797888849997888899999',
  '111111222113222222133445555333345565737444565737746666777744668999999688999888888',
  '111222233112223333411223535411663555446666655444766885474799885777799988779999888',
  '111222333111122233414225336414255336444555666477558696477588696778889999777888999',
  '111112233111412333444442233455552233446652277886655557886677777888697999886699999',
  '111123333111224353161244355666247335622247775688247555668447959868477999888879999',
  '111111222311222224331444244335554446333555666377755566778777966788888996888999999',
  '111111222331112222333452266333455566344456667445556777448856777888899977888999999',
  '111111112334415222344445226347455266337458266337558286377588886777598866799999999',
  '111112222113122442133442442335544466335555566337775566877877669877889699888899999',
  '111111222333314442333114442353444662556666622566777828577799888577798888555999999',
  '111111222333311122334333222444445526445555566475566666777888688779998888777999999',
  '111111122333311224333552244366655244367752248667554448667755888677998888779999999',
  '111111222333331222333451122434455662444555666744556686779956888777988888777999999',
  '111111222113332222414333322444355366445555566447557666887777696888877799888999999',
  '111112222113312444133222444333332554665555544655788888666777889666798899777799999',
  '111122233141225223141125233444455333446457377666557777668589979688588979668889999',
  '111111112331222222332244444335545464733555664737575566777778866888888966899999999',
  '111112222344111255334162275344462275346666675348867775348869755338999775888899999',
  '111111222331144422333144425364442225366666665377788865378889555778889955777999999',
  '111122233111124233144444233445555233467752238667555588667888889667879999667779999',
  '111112222134112222334412445366444455336666655337777665377897755888899759888899999',
  '111112333411112233444552333446522273446555277846665277888655777886699997888699999',
  '111112222311444222314454442314556662333555666733355896788858896777888996777799999',
  '111111223111422223444422333454462633456666637556867737555887777588887999588999999',
  '111112223141222333141252236445553336444753666477755566478858969777888969788899999',
  '111112222131452222133455526137455566337444566337774568397774668999974868999988888',
  '111111222113312242533662444553662774533862774533866744555866774858897799888999999',
  '111122223114123333144223353146223655446666655446788659747788559777789599788889999',
  '111122223112222333114253333144556663444555666744455668777759688777999988799998888',
  '111111222344411122344444522334355522333356666775556866775888886779998886777999999',
  '111112222132222424133333444133355444165555547666558887666888887696999987999977777',
  '111111222113333332133344522666645522766444552776645555776448889788888899777999999',
  '111111122333333112344332222444522566445555566445775666777788668799888888779999999',
  '111111222113312222333444522363345555666347555666647757886444777888897799888999999',
  '111112222113334422113444422333455552334456677855556777886666799886677799888899999',
  '111112222314112225314667225334667755344467775334466755388466795388899795888899999',
  '111112222133311242155333442556633342756666642758886644755888449757998889777799999',
  '111111222311144442333345422336355422736555482776558488776568888766669998777999999',
  '111112233144122333441123333442222556445555566455777766888879966888779669887799999',
  '111111222111332222444435522466437555466437557666437557886637777888833999888999999',
  '111112333411112333444222335464222235466666665478888565477888555777899995777899999',
  '111111222133112222433332526434335566444555666445577676485877776888899779888999999',
  '111112222111333232144433332445553662744555662744855566788886669787888999777799999',
  '111112223444412233441112233455522336477555336477885556778899966778896666788899999',
  '111111233114442223414422233444225533665555533665577888667778898677788899667999999',
  '111112222331112424333312424533662444536666674555866774585897777585899977888899999',
  '111112222331111222344444442335555442333356666788555566788888886777999966777799999',
  '111112222133333322114453332144455522444456666775556668799956688779999998777788888',
  '111111222133112244333332444335552244667555244667755588666788888667799889777999999',
  '111112222311144422315544422333556442373356646877355666887775596887779996888899999',
  '111111112333322212334452222344556667344555667344455667888856677898887777899999999',
  '111112222111332222144432555444435555646437575666637777666837779888833999888899999',
  '111111122113222224333225444333355444636555474666557777666588777688888799889999999',
  '111111122344411225333446225344466225374666825377668885377688555377998885779999999',
  '111111222333411122344444422345545622333555666773585586778888886779998666777999999',
  '111112222313415222313415225333455555633444775666664777688694797888694797888899999',
  '111111222113333332414443322444453522465555527665857777668877797688888899666999999',
  '111111223441122223441552223444553333464653737666655777688855977688889977688999999',
  '111111222131222222331345555633344555633444775666447775666647877999999878999888888',
  '111111222131222244133224444333335444355555556777566666777788669778888969888999999',
  '111112222111222322144566337445563377445666377445563377455663778995999888999988888',
  '111111123142222223144444223544463333566666663555567773588777779588888879589999999',
  '111112222111322244551322664577366664557333644577773664577883944558883999888899999',
  '111111222133322242133552444133552444663357744666855779666855779868887779888999999',
  '111112222133132222113334552663444555663347755666444755866477799888879779888899999',
  '111222233111223333411122335444426635446666655476685555477889995777788999778888999',
  '111112233141222233144255336144255336447258366477558669477558669778888969778899999',
  '111112222133111442333555422333654422366654447886654777886555777866999779888899999',
  '111111122133333422133444422536644442536666672588886672558888779558777779559999999',
  '111111122133333332134452222434552266444555666447755686777756689788888889779999999',
  '111112222131142225336147725366644725336647755386447775386649755388849959888899999',
  '111123333111122333142222353446662555446762655444766655848777759888779999888879999',
  '111111222333112222334122445364444445366666665377777765377889755888899555888999999',
  '111112222311415522313466522333466527334465577384665777884665797884495997888899999',
  '111122223411111223444452223444655533466653337665553777688857777688999997688889999',
  '111112222333111244333315224666335524676555424675588444677598888667999888777799999',
  '111111222311422225314426775334626775334666755344686755344687795388887995888999999',
  '111112222333111242533331244566633224555666444577886664557988884757999888777799999',
  '111111222113444442133354442333554222633555772666855777688857779688888799666999999',
  '111112222113142242313342442333444456355555556357777666877876696878879699888899999',
  '111111122112222223444445233444555333466653337666555777668577777688888899889999999',
  '111112222111222332145552333644457733644457773664457773666855579866888999888899999',
  '111112222314112255314662225344667755344467775334466775388866795338899795888899999',
  '111112222313311222333341522636444555636645575666444575886947777888997797888899999',
  '111112223144522323114522333414552333444456666777855696777885699787885669788899999',
  '111111222113451522333455522333445522336645577886644777886664777886964799888999999',
  '111111122344412222344512622344555666334457766333555776883895776888897776889999999',
  '111111233411222223441222333444433356455555556457776666777888966788888996778999999',
  '111111122131442222335546222355546677333546777335546667888546677888844979889999999',
  '111111222133411222553422266553447766533347776553344766558884766888994779888999999',
  '111111122133333122333455522434452226444456666477756686775556888779888889779999999',
  '111112222111132322443333322443535666445555566444575766887777766887879999888899999',
  '111111122331341442333344442536344622556666622556778682577778888577978988559999999',
  '111111222311312222333322444356644447356666647355556647555887777888897997888999999',
  '111112222134411225134446625334466225334467755388667755386677759388997759888899999',
  '111111222133311442333354422637555442637754482677555482667758888677998889666999999',
  '111111222333411222333451522336455552366444557866664577886964777888994777888999999',
  '111112222331222244331255544331654447331654877366654877665559877669999877999988888',
  '111111223412211233444222233445555333644457773666555577668888777668998897688999999',
  '111112222133322442113334442513336642556666644756688894755588899755778889777799999',
  '111111222133341222133445252333645552366645557866645777868644779888947779888999999',
  '111111222311122224333325244336655547366654447365554477668587777688889997888999999',
  '111111222131422225331462275334466775344666775344667755348867955388887959888999999',
  '111112223414112233444412233454622633456666637556886737558897777558899797588899999',
  '111112222131142222133445525633347555663347755666347775686644779888849979888899999',
  '111112222133132222113333424555363444566666664555767444585777799888879779888899999',
  '111111122113322224555326224553366774533666774533667744588687444588887799889999999',
  '111111222314441225314666255344467255388467225338467775338666795388977795888999999',
  '111122223141122333144523333145522663444555666744885569777785669777889969788889999',
  '111111222134441255334441225344466225376666625377668885377988855337988859777999999',
  '111111123444411223445222233445552233466653337668855577668888577688997777689999999',
  '111111222113414422333444442333554222335555566777855666788888666778898699777999999',
  '111112222313312222333415525336414555366444557666484577696684777999987787999988888',
  '111112222113312422333514442333544442365555547866665777866695777886897799888899999',
  '111112222113114442333335422663535442763555842766585844776588888766699899777799999',
  '111122233112222234511163234556663334555763444577766644578769994578888899778889999',
  '111112233112222333142333355142266555446666655444667758449999758999777788997788888',
  '111222223441552233441152336411152366477753336447859996477859966778855966788888999',
  '111111222341451122344452226334452766334555766334857766388857776889957976888999999',
  '111112222341222552341116752344666755344467775334666775834699975833888975888899999',
  '111112222111312244555332224555534444655634774666637777688833777668893999888899999',
  '111112222311142252314446252334446655344666775336677755838677795838879995888899999',
  '111112222131122444135662477335622447335666477355886477335866479555889979888899999',
  '111112222331442422311144422355556444333356666777355556887779996887877966888899999',
  '111111222314412222344512627334556677334555677334455677384895667888896697888999999',
  '111112233111212333444222235444266335476666635477668555478888555777898999778899999',
  '111111222311122242355226644357776644355777664335577764335588664838889994888999999',
  '111111222311444222331444252336644452336666655738886655737888955777888995777999999',
  '111111123444122223445126623445226333445666377555688377586689377588889777589999999',
  '111111222133331122333442225344466255346666657448665557488855777889977779888999999',
  '111112222311113222333333224555553664556666644566744444588777777888799997888899999',
  '111111122133333332145667332445667772455567772455566722488566729488888889449999999',
  '111111122333334112333444442356666422557764422557666628577777888599788888559999999',
  '111111112133455522333455222333445522367745528667744888666774888667774889699999999',
  '111112223441122233444123333445122533445555566775889566777789666778889966788899999',
  '111111223144442223114552233444552333466752338666755888667755899677788889677999999',
  '111122233112222234512633334511663344555666444557766884577776984579999988779998888',
  '111222233411222335441112235441666635477763335476666855479988855477999885779999888',
  '111111233111422223444425233444525233664555733668585777668587777688887999668999999',
  '111112222133322422133324444153366644153666748556667748555597778995997778999988888',
  '111112222113412552333416522334466527334666577384665577884695777844895799888899999',
  '111111222113312244553362274553662274533666774538866744538867744558897799888999999',
  '111122333411112333444512233446555237446652277486555277886695777888699997888669999',
  '111112222111332222144533236445533336444555666477775566478775669888877999888899999',
  '111111112334442122344442222335442666355555556333788566777788886779788866799999999',
  '111112333144112223144442223444552233665555533667755888677788889677799889666799999',
  '111112222333112222333411562334445566344555667445566677845996777888899777888899999',
  '111111222113331322444433332444445322655555552667588888677778888667977799666999999',
  '111111112344444212344455222333456622733555662773358666777558886797888886799999999',
  '111112222333412552333412522633411555636444575666884775996984777966984777999988888',
  '111112222311422252314424555314444465366666665367777785333797785939997885999988888',
  '111111222334112225334122265344477765347476765347776665348889655388899655888999999',
  '111111222133314222133554422333355442666654444766558888776655889777698889777999999',
  '111111222111222223445627733455666733455567773445666773445586733488888999888999999',
  '111112222314512222334512667344516677334555677334485667344985677999985687999988888',
  '111112233114112333444552223444455233647752283667558888677755888666799899667799999',
  '111111233142222233142225333144455363444555666747556668777599968779999968779888888',
  '111111222333314225333114225366664255367444258667455558677499888677498888777999999',
  '111111222133444442113444422333335522635555572665577777668888799688888779666999999',
  '111112223441115233444415233466755238467752238467552338667598888667599988677799999',
  '111111112333312222343552222343356666444555666444457767888855767888897777899999999',
  '111111122113333224333322224536624444556666644555576684577778888577888899779999999',
  '111111222131122242333524244336554447336555477366655477668685777868889979888999999',
  '111111112334422122334442222334544466355555556337775766888877766889887766899999999',
  '111111112344455212344456222334456622733458662773358866777358886797558886799999999',
  '111111222131122222333345526377444556377745556377444556387746666888889969888999999',
  '111111222131144442333333422536664442537666482577766682557888888577779989555999999',
  '111222233111222223411155333444453336444555666477756666777558886799999888779999888',
  '111111122313415522333455552633445522633745882667744882677774888667794898669999999',
  '111122223111224333551224366551244336557743366577448966557488966777488999788889999',
  '111111122314444122344454422333355522633357772665557777668858887669888897669999999',
  '111111222333331112434433222434455522444555666775556686777886686799988888777999999',
  '111111112333344122334334222344444252655555552656777778666788788669778888699999999',
  '111112222113122242333156244773556644773555644773355644778359666878889699888899999',
  '111111222111322242333325244663355544663357744665557744668587777868887999888999999',
  '111111222333142222355142666335142677355444667335849677555849667888849777888999999',
  '111111122313142222333544422366545427336555477386565447886665777888869797889999999',
  '111111222111322224533332244536336444536666674555677674558877774588887999888999999',
  '111111112333445122336645522333645522773645822776645888776645588779644888799999999',
  '111111222113344222513342266553342666533347776555847766558847796888447799888999999',
  '111112333112222333144523363145522266444555666447775568949975668999777788999788888',
  '111111122331445226331445226344445226347555286377588886377588966377588966779999999',
  '111111222313112222333322444353366444556666644555667747555887777888899797888999999',
  '111112222311442422311544442333545462333555666738585666788885996778788996777799999',
  '111112222331222224331155444633155444633357774666558774666558877699999877999988888',
  '111112223111222233414455236444755236477753336478553666478556696778888999788899999',
  '111111222133331222133422255634455555633444775666664475668884779888977779888999999',
  '111111122313122222333344442333555542666654444765555888766668888777779898779999999',
  '111111222133332222133333424156664424555666444575566648575999998777799998777888888',
  '111112222111334252166334252663344752663347755863447755868477559868477999888899999',
  '111222333111122223144152333445552336444456666477855566777859669788889999777888999',
  '111111122334414122333444522333455522364457728665557888665777888669797788669999999',
  '111112223141222233145223333145553366444456666447755568777799568779999868799988888',
  '111112222133114252333314252336344255666645555667448588767498888767499889777799999',
  '111111122134444222133444422333455662735555562733558666778888669777888869779999999',
  '111112233114122235144222235446677735466777335467773355468888559468889599668899999',
  '111112223111122333444422333444442253655555553657788888666778888666779999677799999',
  '111112333114222233114452223444552333444555666777855666788856699778888699777899999',
  '111112222113311244553662274553662274533367774538866744538866744558997799888899999',
  '111111222333114242333144442533344642566666662576778882577779888575799888555999999',
  '111111222131444222133554242333554442363357747866655777868655779888666979888999999',
  '111112333114222233144223335144223665446666655466788559477788559778888599777899999',
  '111112222331122422331124444333335554666555444655577777666689977886889977888899999',
  '111112222113322222113344444333344554635555574655667777666667788999997788999988888',
  '111111122133333122443532226443532266445555566447785866477785866779888889779999999',
  '111111222331112244333333224356662244555666444557766648577888888557799988777999999',
  '111112222113312244155362224553366674533367774536667744588867449558897799888899999',
  '111111122334144122344444222333342256355555556357786666777888886779889866779999999',
  '111122233111122233414442333444422353455555556757886666777866696778889999778889999',
  '111112333111142355677442335667742335668742355688742255688744225668749999888799999',
  '111112222131222425331166425377776445337666455377644445387669955387888959888899999',
  '111222223111223233411123333444443556445555566455766666777789996778788999788888999',
  '111112333144552333114652233414652273444652777846652797886652799888655779888699999',
  '111111122313314222333444422344455526374555826375558886778888666777896696779999999',
  '111111222113333332143355632444456622774555622774456666784558869788888899777999999',
  '111111222131412222333452525333455555336444577666664777686864777888894979888999999',
  '111122333114112333144422235446662235478666235478866655478885559777899599777889999',
  '111111233144112233444552233444655233746652283776552888776655888776699889776999999',
  '111112233411152223441552333444652337448652377488652777888655977866659997886699999',
  '111122223111223333141523366445522366447555366447885566447785969777788999788889999',
  '111112222133122442135124442335554466337754466337755566877789569877889669888899999',
  '111112222131411552333466522333466522334465577884665777884665777844995979888899999',
  '111112233144122223114225333444525363447555366747585666777588699788889669778899999',
  '111111222334411125334462225344662725344666775384866775388867755389997755888999999',
  '111112222333112422333144442335166642355666447856669477855559777885899777888899999',
  '111111112333333212344532222345522666444555666444775568777785668797888888799999999',
  '111123333111123444122223544622333544625555574665888774665877779666879999888879999',
  '111112222133312422113554422333355444666357444666557777886655799886897779888899999',
  '111112222131311224333352224363355544666654444665557747688857777688997979888899999',
  '111111122344441122333344222334445226355555556377588866777886666779988886779999999',
  '111111222314412222344415262333455566344457776335557666838597776888897796888999999',
  '111111123144552223144452623774852633774852633774852633784856669788855669789999999',
  '111111222111342222553342677553342667553444677533846677553846677888846999888999999',
  '111112222311222224315544444313544664333555666733775686777775586799999886999988888',
  '111111233112222223144225533444445333446555377666577777665588779688888899668999999',
  '111111222133144242333154442633355542677358442675558882677759888676779889666999999',
  '111111223411421223444422233554446233566666663557688833557778888577978998577999999',
  '111112222333111124333522224633554444636555474666655774688885777689999777888899999',
  '111111112133444222333455222334445627384555627384566677888556777888666779899999999',
  '111111233141522233144522223444526633774555633774485666788885669778885969778999999',
  '111111123141222223444425523644425333664555733666587773685587777688888979689999999',
  '111111122134445222134455226334457226333457666388457766388557769888577769889999999',
  '111111112333334122333354442677755422667555422667554442677758888669788888699999999',
  '111111112333341442333444442536642222536666672555586672588888777588987777599999999',
  '111112222111312242555322444565337444565537747666533747666883777868893999888899999',
  '111112233441222333441112333445222563445555566745888566777899966777888966778899999',
  '111111122311222222333444442335544446335555566377775566877777666888888996889999999',
  '111122333144122233145122233145552633444456666774855569778889569778889669777889999',
  '111111223411222223441422333444433353455555556757776666777886966788888996788999999',
  '111112222311412422333414425366444425336666655378888665378898555778798995777799999',
  '111122223111112223444442233455553336447753366477755556778866666788899999788889999',
  '111222223141223233441123335461133555466666665444778865477798855779799858799999888',
  '111112222111322244155324246555344446557333466577773666578783669778883999888899999',
  '111112222311444422311544422333545462333555666738585666778885996778888996777799999',
  '111111222133331222144433322445535362444555666748585566778886669777988889777999999',
  '111111122113333322445533332444556222474456626777455666788885566778888899779999999',
  '111111112134444442333334222356364422356666627558867627555877777588888879599999999',
  '111111122334412125334442225344646625374666825376686885377788855379798855779999999',
  '111112223441512223441512233446552337446653377466855377668895977688895977688899999',
  '111111112314444222333444522633445572633555772635588772665888777666888897699999999',
  '111112222113333332114453332444555622744456662774555666788856699788888899777799999',
  '111111122133441222553342262553347266553347766558347766858847766888944779889999999',
  '111111122133333122443533222445553262444456666747855566777885866779888889779999999',
  '111111122333144122333334442553644442566666662577776822577788888559779888559999999',
  '111112223111222233414252333444455336474555636477556666777858696778888999788899999',
  '111111112333133222443335226444535226474555626477585666477588866777889888799999999',
  '111112233111222334551622334555662234577666334578866444577886944577888999778899999',
  '111111122331444122334452222333455526344457776385557666888857766889777966889999999',
  '111111122333331122443335222443455522444456666775556866777588866779988888779999999',
  '111111123144422223145422333445522336477555336477885566777886569788886669789999999',
  '111112222113313222414333332444444352455555556758666666788888696777898899777799999',
  '111112222331414442311444222355446662335555566733388556777888996788898966777799999',
  '111111233441112233444552223464552223664457733688855737688855777668899977668999999',
  '111111223414411233444455233467775233467555238667522238667558888667998898677999999',
  '111111222314112252344444255346446225336666655377688685337888885737799895777999999',
  '111112222311114222344444522346665555344666775333366675883777775888799995888899999',
  '111111222314441122334445222334455662334555766833557766888577766889977796888999999',
  '111112222131222442133552444133554466337754466337755668777955668977999868999988888',
  '111111122133333322144433222444435226455555556477586666777886669778888889779999999',
  '111112222331111224335555224355656274356666674358676774388777744388999944888899999',
  '111111222113442222133342555633342555663347755666847775666847779888844799888999999',
  '111111222111333244553333224553362224566666664577768844577888844557888999777999999',
  '111112222111122223444442333455553333444653777666655557666877777688889999888899999',
  '111112222113412255663412775633422775663444755633884775633894755668894799888899999',
  '111112233144152223144552333144652333444652777888652779888655779866659779886699999',
  '111111122313144422333355422366354422376555428776658448776558888776669898779999999',
  '111112222133334222113334552133644552666645555766445889766488899777488889777799999',
  '111112222133114222133544422333545444363555747666565777886665779888699779888899999',
  '111112222313112222333412555333445555366444557666644777666894777888899797888899999',
  '111111222314441122344445226334455266337555266337558866377588886779988896777999999',
  '111112222133122244153667224153667744553367744553366748599366748559998778999988888',
  '111111112333144442333444242335452222635555572666658577686888777688889777699999999',
  '111111222331414422331444222335554442333555666788855566777888966778898966777999999',
  '111112222111122322444442333554446663554666733566677733555877777885889999888899999',
  '111111122134441222334444422355546622333555666773385556778888866777988869779999999',
  '111111112333312222343332522446636552444666555744686655774788858777798888799999999',
  '111112222111332244133332444333522444655555554666775888666788889667788999777799999',
  '111112333414152333414152233444552223464652727866655777886659797888659797888699999',
  '111112233114122333144522333445552366447752266448755566888775669888779699887799999',
  '111111222113144222333554442333354442673555842677758888677755888666779899666999999',
  '111111222131122244333322244353366644355666447556667747558887777558889979888999999',
  '111122333112223333412222533416665555414666575444466675884999975888899977888997777',
  '111112333144122233144425233146455233446555277886557279886567779886669779888699999',
  '111112222311142522316647522366447555366647775333644775883647795883849995888899999',
  '111111112133444222333444422536644772536666672533886672558888777555888779599999999',
  '111111112344444212333442222334455226335555566377558866777788666797888886799999999',
  '111112222311222445316624455317624445337666455377786495337786695377888995888899999',
  '111112222113142252333144255363345257366645557368647757668449777868849799888899999',
  '111111122333112222334442225344466775346666675344667775388877755888899555889999999',
  '111111122133345222133345552336445222366645557888644577866647779888647779889999999',
  '111111233111222233444222333445552633444555666774855566777888666778888999778999999',
  '111111112334442212344455222334445662733555662733588866777558886797788866799999999',
  '111112222133112222143333325444443365466666665467755555487777759888899779888899999',
  '111122223114222323114455323144455333446457377666557778696557788696999788699998888',
  '111112333444512233446512233446511233446555277886995277886695277886695777888699999',
  '111111223141152223444452323444552333646752383666755888676758888677759989677999999',
  '111122233441122223441522333411553366447555366447755886777995866799998866779998888',
  '111111122133414222333444422335445422335555566778588566778888666777898669779999999',
  '111112222133332444133322244133524444155555556777785996778889996777899996888866666',
  '111122333114112233144522333445552236444752666487755566888775669887799699888779999',
  '111112222113322224113354424333555444663357744666555777686657799688887799888899999',
  '111112233112222333412225333414455563444456666745556686777599986777999988779988888',
  '111111122134412222334444442334355552633357772655557877688888877666698879669999999',
  '111111122313344122333354222633554422666555444776655884777658888779668898779999999',
  '111112222113333332144434532144444532665555522675888889675878889677777799666699999',
  '111111122334414122533444462533477462537777762538778662538888662559898866559999999',
  '111112222131122222133445555336645577336444577336645577666644778999998878999988888',
  '111112222333111422333414422354444426355555556378888856778898666778999666777799999',
  '111112333111422223414422333444225533445555566775588666777886696788886999777899999',
  '111111122333122222333122444335666644355564447556666477555889777888889777889999999',
  '111111222313451122333455622344456622333456777884456667884556777889956797888999999',
  '111112222131412442331444442333545662735555562733585666788888966788798969777799999',
  '111111122333444122334444122334455522633555772665558877669888877669888777669999999',
  '111111233414211223444222233454266637455666337456668737558888777588998797558999999',
  '111122233141112333144452233444555233446652277886555777886657779888699979886669999',
  '111111112334441222344544226347554266337555266337855286377885886777988866799999999',
  '111112222111314252333334552336634552766444552766485588766488888767498999777799999',
  '111122223111123323415223333445523666444555666444785566777788596787789999788889999',
  '111111122333333142556663142577763442577863442577864442579866622579888888559999999',
  '111112222133114222133334422335444444355555556777777566887766669888799669888899999',
  '111112222111122232455677732445666733445567733445666733845556773848889999888899999',
  '111112222311122224331352444333355444366555447666557777666857977688889997888899999',
  '111112222311422225311446625377746665333746555377746665387744995388884995888899999',
  '111111112333321222334222255334555555364444457666666477668888477888987777899999999',
  '111111122333444112333344422335444522665555522665777588667778888699777888669999999',
  '111112222313112242333414442334444552335555566755888866788898666787799696777799999',
  '111222233412223333411123335441626655441666755446686755499987775999988875998888777',
  '111222223111222333441112355466663335444763555477766665447899955777888999788888999',
  '111112222113144222133544442335544462333555666738885566788885669777889699777799999',
  '111222233112222333411233353441166655444666555446667755848889775888999977889999777',
  '111112233114122333444152333446552237446555277486655277888659777888669799886699999',
  '111111222331142222331344442533366642553666744856667774855557977888859977888999999',
  '111111222314141222334442225336644445336666655377776655388877755888979795888999999',
  '111112222311414222331444452336664452337666455837766655837777955888797995888899999',
  '111112222333112422331122444355444464335555566737777556777889966887899666888899999',
  '111222223112233333111224333551664444556666644555566744888599777888889977899999777',
  '111112222311112242335442442335544466337555466337775566877877566878899996888899999',
  '111111222111344255333342255663342225663347755688847755668847777668447999888999999',
  '111111122133452222133455552333445562738845562738844666788884669777784669779999999',
  '111111122133344422135344422635554472633854772638855572668887579668887779669999999',
  '111112222111333222144433322445553366445753566447755566887776669888777999888899999',
  '111112222113112244333552244333352444663555744666857777668855777668899799888899999',
  '111111122313345122333345222336445225666645555677644588777648888779648898779999999',
  '111111122333311222333422225336444555666645555666444577688884777888997777889999999',
  '111122333441122235441226235471266335471666835477669835479699855479998855777998888',
  '111111222113331222333433225636445525666444555676644585677884888777988899777999999',
  '111122333144422223114522233114555333444456666777555688779995688799996668777998888',
  '111112222133112422135544422333554466377754446337755666887775569887899669888899999',
  '111111112133334222335444442335444222365555527666888577688888577666877779699999999',
  '111112222113322244311325544333325444366555447666587777665587997668887799888899999',
  '111111222113322222133452666334455676334555677434556677444856779888887799888999999',
  '111112222341111222345556772344456772334456677833456667833455567888999967888899999',
  '111111222133112224335555524366665524378865524378866664378888844377799449777999999',
  '111112222311112224333332244533366444556666644555667774558877777588899997888899999',
  '111112222133444225134462225134667755134666758334466758399967758399777558999988888',
  '111111122333414122344444225366644225337666255377886665377888885779898555779999999',
  '111111223441152233441552233444652333744652883777652888776655988776659988766999999',
  '111112222113122422133355424336354444336555477666657477686557779886889799888899999',
  '111111223414411223444522223664552333644555773666855733688885777688997797688999999',
  '111111112333333312344453222644455722644555772664557772666857778698888888699999999',
  '111222223112243333512243333511446666515546676555544776888849976888849977899999777',
  '111112222311222444331255544331254446333754666377758966775558966777888996888899999',
  '111111123114222223444425323444525333664555733666585777686587777688888799689999999',
  '111112222131112442533662742553367742553666744853367744853866774855899979888899999',
  '111112233141222333141552263444552663774852633744855666748855969777888969778899999',
  '111111122133334125663344225666347225688347225688347555688447755689477779889999999',
  '111111222341122252341446255344446255346666675338677775338677975838889975888999999',
  '111111122344421125334422225334466625344666775386667755388887755389987775889999999',
  '111111122133314222335554422336654427386654427386654477886655577888697779889999999',
  '111122223111112233444445223444555233466653337668555777688577777668899999688889999',
  '111111122331455552331455552333452222336444277666684777688884977688884977669999999',
  '111112233112222223411553323444556333444456666777455666787755996788888899778899999',
  '111112333441222323441152223444152333445555566777859666788859966787888966777899999',
  '111111122133333332145532322444552226444456666477755666778785569788888889779999999',
  '111111112313344422333544422635554422633754882667755582667775888667778898699999999',
  '111111122334112222334444225366664255377764445338766665388777755888899755889999999',
  '111112223114411233444222233445525333644555773666585577668888777668997799688899999',
  '111111222133122222333142555633344455666347555664447775666849777888889779888999999',
  '111222233411122333411223355441233665446666655466778955447788995777889995778888999',
  '111111223441211233444222235466677235467777735468773335468888555668998955688999999',
  '111112222344411222334115262344455766334555766334557776838599766888997776888899999',
  '111112222111122223445556723455656733445666733445676773485677733488889999888899999',
  '111122223111222323411253333444555636474456636474555666777758996787888999788889999',
  '111111222133311222444332225433366555436666675444667775488877555888997779888999999',
  '111222233411222233411155236444153336447555366477758666479558886779999886779999888',
  '111111222331114222344444552336464452736666652738868655733888885777899955777999999',
  '111111122133344122333554422367554428367754428367755428667755888669778889669999999',
  '111111222313441122333455226344445226344555776388577776388557666889977696888999999',
  '111111122113342222333544222335546667355546667355546677888446777888847799889999999',
  '111112222311122422333152444663355744633555774663557744666859777886889997888899999',
  '111111222111332224555532244655532444665537744666837774668837777688833999888999999',
  '111122333122223343111223343156624444556666644555576648959977888959977778999778888',
  '111112333114152333144452223446455233446652277886557277866657779888659799888699999',
  '111112222333312222344112556344717556347777756344787556344988556999986666999988888',
  '111111122333112222343445226344445526344555776385577776388577676888899666889999999',
  '111112222133122222134156666334555677334456677334555677444458678999998778999988888',
  '111112222111132222443332555433636575436666675434676775444877755888879999888899999',
  '111122223144112233144152333444452233465555537668857777666859779668899779688889999',
  '111111222331122224351555224355555664336666644366777774388777974388889944888999999',
  '111112233111222333441522333445522663444555666744885566777885966777888999778899999',
  '111122233111142223414445223444555233466653337668555777688577797688879999668889999',
  '111122223114112223144442233444553333465555537666655777668877779688899799688889999',
  '111222233411112235446112335446622335477666335477886655477899655478899995778888999',
  '111112222333111222334412245364444445366666665377777765378897755888999555888899999',
  '111111222314422222311444452336464455336666655337767655837777995888887795888999999',
  '111111222311422225331444225344466665334467755366667775388777955388887995888999999',
  '111122223441225223441155323441153333444653777666658877696558877699599877699998888',
  '111122223114222333114552333144452336444555666477856669777855699777888699788889999',
  '111122233112223233122433334115444444655555554666666577688886997889899977889997777',
  '111111222113144222333354442333554442637555482677755888677758888666779899666999999',
  '111111222313311244333355244633755244667752244667552884667558888667998898777999999',
  '111111122131334222333334252366644552766645552766445558767488888777488989779999999',
  '111111223114422223144442333444522533665555533665775888666788889677778899677999999',
  '111111222131444222134444425333466225336666655377668555378888859777888959777999999',
  '111111123144442223154444423556662223557666233577766633578888839577788889579999999',
  '111112222331114225336174425366677425386777425386774445386679455388699955888899999',
  '111112222133334222133334224155344444165555547666668557699688887999688887999977777',
  '111112222333122242335116247355166647355564447356668447359688477959998777999988888',
  '111112222131122242133332444335544464735555564737775566777866669878889969888899999',
  '111112222111312224333333244355566244356666647558664447558777777588897999888899999',
  '111112222113112222443556662433577766433555776443335776844455766888899799888899999',
  '111112333111122223444445233464555237468852237468555737668577777688889999666899999',
  '111111112133444442333455222363455227364458827366558727666558777688888779699999999',
  '111111222331441122344444425346662225333666555377766685378888885779988955777999999',
  '111222223111242333511142233555443336555546666577744666778849996777848999788888999',
  '111111222113422252133452552333455556337444566377774666877874669878884699888999999',
  '111111122113222222333324455336664557366444557366455577664487777888888799889999999',
  '111111222131442222135442266335444446335555566377777566338877569888877969888999999',
  '111122333111223333411222233444255566444555666445557666887777996888877999888779999',
  '111111222133122222333124455336644557366645557366445577664489777888889779888999999',
  '111111222333441112334444522334445522633555772665588877665888877699988777666999999',
  '111112222133122224113333244553336644556666644556677744558777799588889779888899999',
  '111111122333441122333544222366554427386555427386655447888665777889966777889999999',
  '111112222311412225331422665377446655377746665337744665377884955388894995888899999',
  '111112222113322222114333333444535666445555566444575666777777688999997788999988888',
  '111111122133314442333354422335554442666754222677755588667758888677798889669999999',
  '111111122331441222334445226344455266347555286337558886377588866777988966779999999',
  '111122333111222233411522333444552366444555666447855666777885996778888999777889999',
  '111111222133441522133345522333645552763645852766645888776648889776944889777999999',
  '111112333441512233444511233446552237486652237486655277886995777886695977888699999',
  '111111222133312222134445522333345555663645755666647777886644479888897779888999999',
  '111111112333441222334422222354444466355555556337777756888887766888977666899999999',
  '111111122313144422333334442356644422356666627558886627588877777558889797559999999',
  '111222233411222333412233355411136555466666665444678885447779985777999885779999888',
  '111111222131132222333332244356666444355564447555666647558877777888879979888999999',
  '111111223114422223144422233444455533666555333665557777668887779688887799688999999',
  '111111122333114222355564222335664472335666477835566477888564447888599777889999999',
  '111112222111122322445553332444453633744753663774756666877755566887889999888899999',
  '111111223114421223444422233454426633556666633556678838557778888577978899577999999',
  '111112222113342422113342442333444456355555556357777666877876699887876699888899999',
  '111111122134414422333444222355544622333555666773885556777888666778898869779999999',
  '111112233112222333114252633444552636474555636474855666774858699777888899778899999',
  '111222223141112233441512336447552336447555366477855366477895966778899969788888999',
  '111112333111222333414452233444552263447555266847755666887756696888777999888799999',
  '111111222133333222114322255444333665446666655466777555448887599888777779888999999',
  '111112222333311122333445152336645552766444552766645588769644888779998888777799999',
  '111111112333333312334444222444544522665555522665775777666777788698888888699999999',
  '111111222111222233444445233464455233664555733668557737668577777668888999888999999',
  '111111223111222223444223333445553366445753566447755566777788666788888999788999999',
  '111112222311122222314556666344457776344555776344457776333355786999998886999988888',
  '111111222311442252314442252344646655334666755336676775838877795838877995888999999',
  '111112222313411222333412255363455557366444557366664757668894777888994797888899999',
  '111112222313122222333114556377744456377745556374445556377488666999998686999988888',
  '111112333111122333444522233444522236445555566478885666778885666777889999777899999',
  '111111112133332222333435522366445522666444555776644558776684888777788889799999999',
  '111111122333141222334444222344554266345555576338755776888777766888979666889999999',
  '111111222133341122333542222633544444635555574666665774888865777889967779888999999',
  '111112233111222333144552233444522633444555666774885666778855669777888999778899999',
  '111111122131444222333444225334442265366666665367788855377888555777888959779999999',
  '111111222334111222334455262333455662334456677844556777848556677888999677888999999',
  '111112222311122224315562774335667744355666774335566744355867794388889994888899999',
  '111111222131414425334444425334666625377762225376666855378888855378898959777999999',
  '111112222333314522333114552336144522666645555776448599766488999776489999777788888',
  '111111122331441422334444442533333622566666662556777772588888877558988977559999999',
  '111112222331112244331522244333555446377754446377555666778885966778899966888899999',
  '111122223141152233144452333144552363447852366747855669777856669778859969788889999',
  '111111222311122224333552444663352474633555774636857744666855777688889997888999999',
  '111111222133122222133445552333455555336444577666664777866644779888889779888999999',
  '111222223111122333441122355446666335447763355477666655447889955777889999788888999',
  '111112223441112333444122235466772235467777735468877335468889555666899955688899999',
  '111222233111233333141223355441266665444267555466667855449977858999997888997777888',
  '111111222311122222333344452636347555666347555666347575864447777888889997888999999',
  '111112222113313322145333226445533626444555666474885566477888569778898899777799999',
  '111111122344441125334662225344662725334862755384866775388866755389977775889999999',
  '111111222131441222333452226344455566344457776335557776388857666888977969888999999',
  '111111222334151226334155226344452276344457776348857776388559766388959766888999999',
  '111112333142122333142223355142626355446666655447686859447788859777889859777899999',
  '111112222113222442133526644173555644773356644773555648773395668977999688999988888',
  '111111122333311222443333255446636255477666225447686655447888855777998888779999999',
  '111122233144112233144122333444225533445555566775588666777889669778899669778889999',
  '111111222131222242133255446333554446377754446377755666377558669878888969888999999',
  '111112233111412223444422233445425333465555537666587577668887777688897999668899999',
  '111112333411152233414552336444752236487752236487752666488755696887759996888799999',
  '111111112133332222334322225344466655346465657446665557488887577888877779899999999',
  '111111112333333412533673422556677442566777442566774422556874882596888888599999999',
  '111111122134444442133334222335344222355555556777886566777866669788888869779999999',
  '111111122113333442333335442677555422667754422667555442677588888677888899669999999',
  '111112223111222333441252336447552366447555366447855366477858966777888999788899999',
  '111111122131322224333352244336552444636555474666855477668857777688887979889999999',
  '111112233441122233445112333445552336447752266488755566888799566887779966887799999',
  '111112233412212333411222333455526663444555666744485556777888996777898896778899999',
  '111122333441122233441225236411525336447555366477585996478588966778889966777889999',
  '111112233111142223555442323555542333556742388666748888676744888677749999667799999',
  '111111122331451222333455226333452266344456667448856777488556777888956977889999999',
  '111111222311451262337455562837445662837745562833744562837774566838974996888999999',
  '111111112334444412355554442355566642355666772386667772388877772398888822399999999',
  '111112222133311242133555442336654442733654882766654488766555889767998889777799999',
  '111111122311422225333442665377742665377842665377846665377844555388884995889999999',
  '111111222133311442334444422334554222335555566777855866778888866788996669777999999',
  '111112222133334242113554442133655472836654472836554779866655799868677779888899999',
  '111111122333114442333344442355366442556666622577668228577778888577799888559999999',
  '111111222111222233455262733445566773455666773455667733445868773448888999888999999',
  '111111222131142225633442755663342775663842755633847755663844775688849979888999999',
  '111111222133144222333142225367744445367777745366667745388869555888669559888999999',
  '111111112133444422333445422363455427366555227368558727668588777668888779699999999',
  '111222233111122334551622234551666334557763344577666844579996844577998888779999888',
  '111111222314516622334516772834556772834456672833455672833495677884495697888999999',
  '111111122344114422334444222335545226333555666377585566777888866778899886779999999',
  '111122333441522333441522366411552336417752286499755886449775866999775866999778888',
  '111122233144122223114555223144453333446457377666657778699555788699998778669998888',
  '111111112314444442333334542663334522665555522665788822675788888677777798699999999',
  '111112223411122333441225333444225663745555563744588666777588966777889996788899999',
  '111111112133322222334332225364455555366444557666664457688877477888887779899999999',
  '111112222331112422536612444533662744533367774553866774555896674885899977888899999',
  '111111233111222233444233335442226355466666665447688855477778555778888999778999999',
  '111111222131122232433333332444556662445555566744455666788888886787779989777999999',
  '111111122341441222344445222343545266333555666337585686777588886777988986779999999',
  '111112223141122223144445223446455533666457333665557377688577779688889979688899999',
  '111111222313122224313322444333554444635555574666655777666887797688889797888999999',
  '111111122113333322443355332444455222644457772666557777688558877668888899669999999',
  '111112333141122223144222333444255363445555566747558666777888669788889969777899999',
  '111111222133322222134333255144336655446666655446677558449777578999997778999888888',
  '111112222113222424133233444133334454655555554656677778666779778696999788999988888',
  '111111222113332244313322444333522444655555554666775888666778898667788899777999999',
  '111111222111222233444252333464455337464555737466557737666858777668888999888999999',
  '111112222113112342333333342555666642755564442756666444758888888758799899777799999',
  '111111222113312224333355244336352244666555444667758488667558888677798899777999999',
  '111112333414152323414152223444552333446652277888655777866659797868659797888699999',
  '111112222313112244333312244355562647356666647356864447558897777558899797888899999',
  '111111222334111442334444422333455222335555566777558666778888866788999866777999999',
  '111112222311122224313524444333554664773555644733755666777785696788889996888899999',
  '111122333111122344512222344556263334576666634577768644557888894557889999777889999',
  '111112222111222324155562334557766334577666334577663344577864449587888999888899999',
  '111112222113222242113355444333354444633555774666657777666557788969999788999988888',
  '111112222133112222333145266334445567344555667345566677448569777888899779888899999',
  '111112222133133222133334225166634555664444455666475558699477778999778778999988888',
  '111112222133122224553116244553366674533367774536667744558699744588889779888899999',
  '111112233144122223444152233444152333465555537666859777668859777688889779668899999',
  '111112222344115262344115262334455266344457776338557766838599776838599776888899999',
  '111112222131114442333355422336655427386654427386554477886557777866699979888899999',
  '111112222331114442335144422335544422333555666778885566778889566788899966777799999',
  '111111222133122222133333244556334444566666664555577644558777779888889779888999999',
  '111111222344411122334442222344545666335555566333575776888877766889997776888999999',
  '111112222311122242315542446333544446373555646377775666377875596878889996888899999',
  '111111222331412222331444422335445446335555566377577566887777966888897966888999999',
  '111111122133322224133222444533366444536666674555667774555888779588887779889999999',
  '111111222333111242353222246357774446355777446355577746358888646858999666888999999',
  '111112222333314252633114552663144552763348852766449855766499885767498888777799999',
  '111112222131122222331445555633455775633444775633664775666644877999998878999988888',
  '111112222113333322411333222455553666444456666444755556888777996887777799888899999',
  '111112222311442555331142555633742885633742885633742885666749988666744998777799999',
  '111112333114122233144422233444522533445555566775885666778886669778889699777899999',
  '111112222133144442333114422335554462735854562738855566778899666788889669777799999',
  '111112222311311242333332244556332644556666644556788644557788888757998998777799999',
  '111112222311412222313445552333645557336645577366645777866644797888894997888899999',
  '111122233414112223444112233455552336447753366477855556778899666788899696778889999',
  '111112222133332422113334442153634642556666644756586849755588899775788889777799999',
  '111122233141122223144223333145533666445555566444775568777799668799998868779998888',
  '111112222311444222311445622344455666334555766333557776883577996888777996888899999',
  '111112222111312442333332442536332444566666664555788684755788888755798999777799999',
  '111122223141112233144552333444522336474555636477885666777855669778899969788889999',
  '111111122333311222333322224556344444566666664555557644588887777888997777889999999',
  '111111112344444122333542222335544462733555662738885566777785666779888886799999999',
  '111111222134112225334446225344666725334467755384666775388677755388899759888999999',
  '111112222333312242333112244355616644355666447556686447559988777959987777999988888',
  '111111233444112233444122333455222236475555536478888556777889666778899666778999999',
  '111111223441222233411252333444552363447555366747855666777858996778888966788999999',
  '111112223114412223144442323445545323666555333676585588676788889677798899677799999',
  '111112223114112333444552333444552263748852263748855666777855666777899699788899999',
  '111112222131112242333335542333655442666654444766554888765588888767799989777799999',
  '111111122113333224533322224556332644556666644556788644577778884577888899779999999',
  '111111222333112242333122244553666644553367744556666744558889777858899777888999999',
  '111111222331112242333332442355444442365555547866666557866877777868899977888999999',
  '111112222113311242333322244335554444635654574666655577668887777868997799888899999',
  '111111223114122223444425233664555233644457773668555733668587777688889799688999999',
  '111112222133322242113566642313564444333564777555564787956664788959997778999988888',
  '111111122133333332134322222444425566445555566445576666777778689788888889779999999',
  '111112222113311222333334422335554444666555444666655577886677777888997799888899999',
  '111111122331142222333542226355547776355444776355547776388847666888849966889999999',
  '111122233111223333141223553444266653446267655846667555844877959888877999887779999',
  '111112222133122222133145555334445565337745566737744466777748668999998668999988888',
  '111111122133441422334444422333455226375555526377558666778888866778988669779999999',
  '111222223141111233441552236444552336477753336477855666478855966778999969788888999',
  '111111222311455222316445272336645572333645777836645577838644597888664997888999999',
  '111111112133333222334443222344545522444555666775565668777866688777888889799999999',
  '111112222133415222134415226334455776334457766344557766388597769888597669888899999',
  '111122223411122323411225333444555336474456636477555666777588996787889996788889999',
  '111111222131222224331254444633355774663555744633557774666658977688888979888999999',
  '111111223411222223441233323445555333444456666777555566787778966788888996788999999',
  '111122333111422233115442236555542236577546336578846666578844699778884999777889999',
  '111112222331112424331322444533326644536666674556687774555887977585899977888899999',
  '111111122314444122334455222334455226333457666388557766888557766889777796889999999',
  '111111222131144422333334442563777742563378842567777842566688888556669989555999999',
  '111111112134445552334465552344666552374468852377666882377768822377788829399999999',
  '111122333144122333114522223144552633444555666774855669788885699777889669777889999',
  '111112222314122255334116225344446675344666775346677775388699755338889795888899999',
  '111112222113134222133334525333644525666645555676445888676488889777489899777799999',
  '111112222113412222133442566337745556377745556377745566337844669888894699888899999',
  '111111222334112252344412252346444455366666665337777675838897775838899755888999999',
  '111111123444441123445222223445552333666752333666755588677777588679988888679999999',
  '111111122133333122333344222555554626575444626575466666777448888779888889779999999',
  '111112222133312224333312424356614444356666647555586647595987777599987778999988888',
  '111112222133322242113333342156634444556666644555576648957777788959997778999988888',
  '111111223414152233444152236477752236478555236478853336478859666778859696788999999',
  '111123333111224355661124335662224735682444735682477755688479955668477999888879999',
  '111111233441512233446512333446552337446652277488655277888695277886695977886999999',
  '111112333411512233441552333446652237444652777486652277888655977886695997888699999',
  '111111233114522223144522333444552333444652777888655777888665779866665799886999999',
  '111111223112222223144423333444445533465555537665577777666687779688888899688999999',
  '111111112133342222333444422355544422336555477886665557886666777888867779899999999',
  '111111222311444225331444225336644425376666625378886655377888955377888995777999999',
  '111111223144122223444122333554426663544666773566687733555889777588889779588999999',
  '111112222131122422133526444335566477335666477335664477555684779885889979888899999',
  '111111233414122233444122333444222553645555573655888777666889777668889797668999999',
  '111111112133442222333342252663345552666347555866647755868847777888844779899999999',
  '111112333141222233441255223441255333444456666777558966788558966778888969777899999',
  '111122333111223343511223444522233464566666664565778884555788994757788999777889999',
  '111111112133344422533444422536464772536666672533868672558888772558887779599999999',
  '111111222113222242133334442533334644566666664556577774855577779858888799888999999',
  '111111223144521233144522233444522336477555336477885666778885669778985669788999999',
  '111111222133121244536622224533667444533367774555366774588886674558989779888999999',
  '111112222131312422333312444553332464566666664565788844555798888775798989777799999',
  '111111222113341222333445552333645552763645852766645888766644888777948899777999999',
  '111111112331444442344442222335552662335752566733755566777788886788888966799999999',
  '111111222113332222133452662334455667344555667344556677844856779888877799888999999',
  '111112222113333322113334322555544626555444666575446666778488899778888899777799999',
  '111112222131411225334446225344466725334666755384667775388677755388997959888899999',
  '111111223414112223444422223454333333455555556777777656788886666788899696788999999',
  '111112223112222323144253333144553636444555666474755668777759668797999988799988888',
  '111111222133114422333354442336554422736555482776655488766658888776699889777999999',
  '111112223141122333441223353466622353444666555747886665747788955777889959788899999',
  '111112222311122224331325544333355444636654474666557777665587977688889997888899999',
  '111222223111112333144552366447552336447852366477855366447855669777899999788888999',
  '111112222133222242133355444133354444136555478666657778666557778969999778999988888',
  '111111223411222223441225333644455533644457773665557773666588977688888997688999999',
  '111111222131442222134442255334444665336666655366777755338877759888877959888999999',
  '111111122133333222134335262444435562744555662745586666747588689777888889779999999',
  '111111222333111244333552224336552244366654447668855477688855777668999777888999999',
  '111111222113444422313454422333455462733555662738558666778858696778888699777999999',
  '111111233441211223444222233454626633454666737556686737558888777588998977558999999',
  '111111222333311222334331252444636652444666555746686555747988588777998888777999999',
  '111122333111223343511233443552266644522666774556667744855887994858877999888779999',
  '111112223114522223114552333444452336477456336477856666777855699788885699788899999',
  '111111222113332222145532266444533366477555336447775666448875569888877799888999999',
  '111112333111412233444422333455222263445555566748888556777886666778896999777899999',
  '111122233111122334515226634555226334577666334577688444576688494577889999778889999',
  '111112222113312442133552244336652447336555477366854477668855779866897799888899999',
  '111112222111122322444523333664557773644555773644455733666685777886889999888899999',
  '111112222133112442333144442333445542635555572685588777688889777688699779666699999',
  '111112223112222333144552236144533336444555666477775668479955668777999988799988888',
  '111111222113332222144333352446363552446666655744868655748888559777788899777999999',
  '111112222331122222311444444333354446355555556377756666777777886999998866999988888',
  '111111233141455233144452233444652233746652283776652888776658889776558989776999999',
  '111112222313411122333455522344456672334456677834456667885556777889996797888899999',
  '111111222111222233442255553446665553476765653477766633477778833448888999888999999',
  '111112223141222223145523333144553336444456666477755668777795568799999868799988888',
  '111112222333311124355336124553366224557666244577668844579688448579998888777799999',
  '111111122333331124356332224356662224555564444577766648577788648579988888779999999',
  '111112222111222233144452333444553333444555666777755666777856669778888999888899999',
  '111112222311114422335544442333544462733555662738885666788885566778899996777799999',
  '111111122331444122333452222344455266344555776338557776888857666889777966889999999',
  '111112222113222242513333644513333644556666644556777784556777784959999788999988888',
  '111112222331414225331414425366444425376666625378888665378898955377898955777799999',
  '111111233444111233442222233444522333455555556777885666778888866778999666778999999',
  '111111122333331122444333225446362225446666655477768655477888555779988888779999999',
  '111111222113314442333344442356364422356666627558867627588887777588897799555999999',
  '111112223114222223144523333144553366447555366447755668777795668799999688799988888',
  '111111222331411222333452252333455555636444575666664777868864777888994977888999999',
  '111111112133222222334256667344255667334555677344558667344458677888888779899999999',
  '111111222133121224333522244336555447366654447366555477668885777688989779888999999',
  '111111222112222223144444433556644433556666633557776633557777778599999988999888888',
  '111112222331122242531124444533366664533367774566667774555589974858889977888899999',
  '111222233111223333141123355446226355446666655447688655447789959777788999778888999',
  '111111122311322224333322524663355544666357444665557744685887777688887997889999999',
  '111122233141125223444155233444152236477753336478859666778559666788589969778889999',
  '111111112333122222332224444333354444655555554666657777666688877888889777899999999',
  '111111122333312222333415252663445552663444755866644755868694777888897777889999999',
  '111111112333333122334442222344455662745555562744556668777766688779888888799999999',
  '111112222331112444331222444333255544363555747665558777666888977666899977888899999',
  '111111222131222224331244444355554466333754666337755556777778966788888969888999999',
  '111112222314411222314555622344457666334457766333457776883555796888997796888899999',
  '111111112333331222433533222445566662744555662744445566777885886777988888799999999',
  '111111112334415222334455222344456277338456277338456667888556677888596677899999999',
  '111112222331412225344411225344446655346666675336677775388997775388897955888899999',
  '111111122333331142333554442367755422667754422667554428677755888679988888669999999',
  '111112222314112222314455662334457766344555776334457766833557796888899796888899999',
  '111111122313314222333344225363642255666645555667745858677448888777498898779999999',
  '111112222331111242533336242533366442555666444755668884757688884757999988777799999',
  '111111222333341122333341225666344225666645555677448555677948888779948888777999999',
  '111111222331142222331445252333645555633645775666645777868644977888849977888999999',
  '111111222111333322433353622443556662744555662744455866774858886778888999777999999',
  '111222333111122333411222235444466635474666535476665555478888995777889999777888999',
  '111122333141122233145522233144556233444456666778455669778885569778889969777889999',
  '111111222313411222333442252336645555336645577666645577868844777888994797888999999',
  '111112222331111422333444422335444426355555556378888566778888666778999966777799999',
  '111112222111312442333355422336665442336555477866544477886557777866897999888899999',
  '111111222333112222333312442355444444365555547666666557866897777888899777888999999',
  '111111233441411233444452223446655233776555233776552288766658888776998988776999999',
  '111112222133322224113333324153664444556666644555566748597777788599997778999988888',
  '111112222133112242133554242333554444633654774666655777868655779868899779888899999',
  '111112222131122244135555224335555564366666664367777744388777749338889949888899999',
  '111111222314441122334444222355544266333555666337885556777888866779988896777999999',
  '111112223111422233144442223445453333465555537666657577688877779668887999688899999',
  '111112222311114252333334552636344522666347555886447575866477777868499997888899999',
  '111112222111312422533366422533364444533666774555567774885667774885897999888899999',
  '111111112333334122533444422533664442576666642577766882557777882559788888599999999',
  '111112223114422333114222333444425553644555773655587777666888799666887799688899999',
  '111112222113322224113324444533366474536666674535667774555587799588887799888899999',
  '111111222333441112335544422336554222336654477888655477886665577899966777888999999',
  '111111122133313222434333322444445526475555526475566666778888686777898889779999999',
  '111111122113333222444353362743355662744555662744558862748858666777888899779999999',
  '111112233141212333144222333444522366455555556447885666777888669777898969778899999',
  '111112333111222233144222233444445553644555773655577777668888779668888999666899999',
  '111111122313331442333444442553346642556666622576678822577777888577988898559999999',
  '111112222331222444311225446317555466337754466337555486377599886777999866999988888',
  '111111222314411222344555622344457626333457666383457776883555776888997796888999999',
  '111112222133311422133344422335554442366555447866655577886667779886997779888899999',
  '111111222113312222444332332445533366445555566447775566877877666888897799888999999',
  '111112333141122233442123335442223665446666655466788855477789855778889959777899999',
  '111111222133312242134444442333334542655555552675788888677777789676698889666999999',
  '111122333444123333441122233444115222455555556777588666997778866999978666999778888',
  '111111223411122223444422523466445533466555337665577337685887777688889997688999999',
  '111112223111142333444442233445545233677555223667585588667788888666789999677799999',
  '111112222111322242155344442555366644775333644775553666877773669878883999888899999',
  '111111122333314552663314552677344552673348852677448552677498822677498888669999999',
  '111112222133412522134416572334666572334465577834666577834695579884895779888899999',
  '111111233114122233444422333445452223645555573688857577666887777668889799668999999',
  '111111233114522233414522233444552233464457737668855777668885797668885799668999999',
  '111111222331141222533446772553346772855346772855346677855344667888949966888999999',
  '111111122333112224333322444355552444366752448666755558666778888677799888779999999',
  '111112222133312222333314424536614444536666674555586674595587777999987778999988888',
  '111111233141142233444445223467455223667555233677558238677588888667789989667999999',
  '111111223114412233444522233444552233466555337668855777668885777668897799688999999',
  '111112222111122232444455532446453333466457337666657377865557777868889999888899999',
  '111112223141122233444123333444522336455555556477885666777789666778889969788899999',
  '111122333114122333444112335446222235466666665478888655477899555777889599777889999',
  '111111233144422223154422223154446663554666733566677738599997738599997778559888888',
  '111112222113122422333144442353336644356666647556677747855559777885889799888899999',
  '111111222131422255331422256333455556377444556377774666378884966778884969888999999',
  '111111112133322222333442555366422557366444557366884557666844777888887779899999999',
  '111111222113322242133224444335554464337555466737755566777788669878886699888999999',
  '111111122133333222133422255334442555666645555666744488667774889777888889779999999',
  '111112333111122223444223333455226663445555566744488556777788666788889999777899999',
  '111111122333341122335344226355342226555546666577748668577448688779948888779999999',
  '111112333414112333444212233444222253455555556758888666778898666777899696777899999',
  '111222223112223333111423555614433355666444555667774485666794888777799988799999888',
  '111111223144411233444522223444525323666555333676575888677775888667998889677999999',
  '111112222111333322413333222455553666444456666444755556888777796887777999888899999',
  '111112223441152333411552336444752336487752236488752666488755996888759966877799999',
  '111111222314412222344516272334556677334555677334455677838495667888896697888999999',
  '111112222113112242553322444533336664533666774566677774555887744858899799888899999',
  '111111123112222223444255623774455633774456633774556633784558666788888899789999999',
  '111111112313332222333422225364455555366444557666664457688884777888877797899999999',
  '111122223111144223144442223445666633455563337556666377588877779588779999588889999',
  '111111122134444422134454226334552266337555266337755866377858869778888869779999999',
  '111111222331222242311424444335444566335555566335777566777787996878888966888999999',
  '111111122131444422334444225366422255336666655337778665377888855778888959779999999',
  '111111122133444222133344422335445426375555526378588566778886669777888669779999999',
  '111111122333114222334444425344466625377666225376668885378888855777899555779999999',
  '111111223414122223444122233445566637455666337456663377558889777588889797588999999',
  '111111222134142222134442425336644455336666655337776655387877759888879759888999999',
  '111111222133333242133332242155566642555666444756664448757799998757999998777888888',
  '111112222113222224513677744513666744553367744553666784553336784599999788999988888',
  '111111222113334422133555422333354442677754442677758888667555889667788899666999999',
  '111112223111222333144423333445225366445555566447588566777786669777888999788899999',
  '111111123114222223444252323445553333444653777666655577686858777688888799689999999',
  '111122233114125233114425236444225336477555336477588666478586699778589699778889999',
  '111111112333333122443553222644553772644457772644855772666855877669888888699999999',
  '111111233114522233414522233444552336474852636477855666778885696778885699778999999',
  '111111123444121223454422223554442633566666663556788833577778838577979888579999999',
  '111112222111322242551344442555344646575333646575773666877773966878883999888899999',
  '111122233141122235441123335446226335476666635477688655477789955478889959778889999',
  '111112222311142522331445552336645572333645777836645577866644977886849997888899999',
  '111112222133332222113334442513364644556666644556567784955577788999977778999988888',
  '111112222133411125334444225344446225366666665377688885377888855379998559777799999',
  '111112222122222333114333333414455556444456666455556676888888677888999997999977777',
  '111111122333341122555344226553347226558347266588347766588447666889947777889999999',
  '111112222334411252344612255344616275344666775348696775338896775838997755888899999',
  '111112222133222422113334444133355444635555574666557778666677788996999778999988888',
  '111112222133112442333155422663355442663357744866557744886559777866899779888899999',
  '111112222313111244333552224363355244666357444668557747688855777668999797888899999',
  '111111122333344112335346222755346662755348662755548662777548688799448888779999999',
  '111111222112222233144256633774556663774555633744455633774458669778888899888999999',
  '111111222333311122334355622334455662744456662744556688774558688779998888777999999',
  '111112222331122222311444445334466645334666755376667755377777885999998855999988888',
  '111111112333333122433355662447755662477756662477556622477558882449888888499999999',
  '111111222133441122334444222334454562335555566735858866777888866779988669777999999',
  '111123333111223433151233444552266674522666774526667744555887949885877999888879999',
  '111122223411112223415522333444533366447555366447775666777885596788899996788889999',
  '111112222131344442131334442333344552665555522655778888677788989677778989666699999',
  '111111222113444242313354442333354422635555572668857777688857797686888799666999999',
  '111112233112222234122333334155566344556666644557664448577777998579999988779988888',
  '111112222311144242331554442333554662738854462733855666788855966787889996777799999',
  '111111112334452122334552226344457266348457276338457776388855766889857766899999999',
  '111111122314441222344422225346665555344666775333366675388887775888977795889999999',
  '111111222334441112334422222344445566335555566335577776888887766899977766888999999',
  '111111222122222233144553333144453366447555366447756668777755668779999998999888888',
  '111112222311442222331142556377745566337745566337745556377849966888844996888899999',
  '111112222133133222113332244556334444566666664555577644558877799888779779888899999',
  '111111222131444225134422225334444665336666655366777755388887759388777959888999999',
  '111111122334112222344422555344426665344666775366687775333887775888899755889999999',
  '111111222331341142333344442335444222655555552666777588677778888679978988666999999',
  '111112222131444422133344222135444566335555566335777568999776668997777868999988888',
  '111112222311112222333445526377744556337745566377445556387744666888899996888899999',
  '111111223111444223444452233445552333667752233666755588667758888677888999677999999',
  '111111222133122224133442244335544446335555566377775566778877669788889669888999999',
  '111112222334111122334455522344456772334456677833456667885556677889999677888899999',
  '111111122313314222333554262355544267358546267358446667858466777888497797889999999',
  '111111122333312224333315244366655224667555244677554448667598888677798888779999999',
  '111111233411122233422225333424555363444456666747555686777588886778889996778999999',
  '111111222131444242133444442533366672533666772536667772588888779585888979555999999',
  '111122233111112333444552333446552377446652277448655277888655777888699999886669999',
  '111112222314444422311144522366445552336666655733388665773889995778888895777799999',
  '111111223114222233414255333444255633744258663774558666777558696778888699788999999',
  '111111222113222224133524444663555774633357774633555744666685779688888799888999999',
  '111112222112222344512333344516663334556763644577766684557777984557999988999988888',
  '111112222131112224333322444335366644355666447556667477555887777588899979888899999',
  '111112222131112444533662224553362774533666774533867744588866774555899979888899999',
  '111111222133114252336644452738664452738864452738866452738886655737899559777999999',
  '111111233114222233414252233444552336474555636477855666778858696778888699778999999',
  '111112222311311222333334442566633442555666444755886664755588888777998998777799999',
  '111111122314441225344622255346662255344467775338866675338886775388977795889999999',
  '111111122133333122434353322444455522644457772665557777668858787669888889669999999',
  '111111122133444125334466225344662225374468825377766885377668855379888559779999999',
  '111111222333331112344356622344555622374456628774555668774458668799988888777999999',
  '111112222133332244113222444133355544663555744665557778666999788669977778999988888',
  '111122233111122233444155223464557233464457737668455737688559777668889999668889999',
  '111112222333122424311152444366552477336555477336855447666859997686889777888899999',
  '111111112333344122333542222355546262355546667858546667888846777889447777899999999',
  '111112222112222334511662334555663344577763334557766444577866994577888899888899999',
  '111112222131122422333124444333355544663555744665557777666689777886889979888899999',
  '111111222133112222443333332455556362444456666748455556788888866777799889777999999',
  '111111223144211223444222233454466333456666637555667737558888777588998779588999999',
  '111222333141252336441552366411752336481752936488752996448755966488757969888777999',
  '111112222111312422556337422566637444566637774555637774885633744885893999888899999',
  '111122233111122233412225336444455336474555636477556666477588896778889999778889999',
  '111112222333311222333312244556316444556666644555678644559978888999778888999977777',
  '111122223111223323411255333444255636474456636474558666777558996787788999788889999',
  '111112222311142552333142555633442575633842775636844775666849777866849997888899999',
  '111111222333411222355412226355447766355547776335544776388894776888994666888999999',
  '111112222344412222344511162334555666344457776333555766839995776888897776888899999',
  '111111222333112242353612447355662247355666447358866447355896747858899777888999999',
  '111111222113441422333444442333455552663357722655558777688888777668988799666999999',
  '111111222111334222333344442353346642756666642756658848755558888777588999777999999',
  '111111122333141422333444442353664422356666627558866727588888777558989777559999999',
  '111112222113322244113322444533366474536666674535667774555887799558887799888899999',
  '111112222133111222333444422335554444336555477666655577886666777888999779888899999',
  '111112222113142522313345522333644555336645577666445777886647797886849799888899999',
  '111122233411112333444122335446662235476862635478866655477889555777899995778889999',
  '111112333141122323441452223444452233465555537668857777688857977686889979666899999',
  '111112222333112224333114244355554246377754446378755556778799666788899666888899999',
  '111111122333341122344444522334455522633457882665557788665777778669978888669999999',
  '111122233411123223441223333441255366445555566447558966777788966788789996778889999',
  '111112333411122333444122233445552263447555266847755566887779666888779996888799999',
  '111111222111222223444455233444553333644653773666655777668557777688888999888999999',
  '111112222134414425133444425136664225337666255377866659378888559378898859777799999',
  '111112222131112444533322224553333364566666664567777744588887774555899979888899999',
  '111111222334511222334516622334566627334555677384445677884495677888995677888999999',
  '111111222133144422133345422336655472836555472836554477886567779886669779888999999',
  '111122333111123333144222223444555523444456666785555666788888669777789999777889999',
  '111111223114222233144452333445552633744852663774855566777856669778888699788999999',
  '111111222131444225133464225336664225377764445388766655388767559388777959888999999',
  '111111222111222232444445532666455333644457773666557333865577777868888999888999999',
  '111112222113122222313445555333344555663347755666447777666644787999998788999988888',
  '111111222131332242133335442636535422666555444776585484766588889767788989777999999',
  '111111122331344122333334442356664442556764622577766628577788888559778988559999999',
  '111112223111122333444422333445552363445752566848755566888776666888779999877799999',
  '111111112344444122333344222333554422635555572668855777666887777669888887699999999',
  '111111122333314122333544422335544422677555442667775588667775888669798888669999999',
  '111112222131122224133324444333335444655555554666577777666687779688889979888899999',
  '111111222113333332133455322444456622474555626774456666778556889788888899777999999',
  '111112222113333244155322224153333664556666644566777748599997448559777788999988888',
  '111111222133444442113554422333554622733854662773855666778855699788888669777999999',
  '111112222111312444333332224533632444566666664555786884577788888555798999777799999',
  '111112233141122233144445233446455233746555283776558288776588889776669989776699999',
  '111112222311122242315526644337556664377756664377755644337785594838889994888899999',
  '111112222131112224335522664337556664377756664377755644377885544388899949888899999',
  '111111222334441122344544122335544662733555662733885566779885886779988866777999999',
  '111112222134411225334441255344442265366666665367788885337988855377998859777799999',
  '111112222133112442333512444663552274633555774638855744666895777866899779888899999',
  '111111222333122222333112455633344455666645555664447775664899777888889777888999999',
  '111112333414122223414552233414552333444652777888655797886655797866669797888699999',
  '111112333112222333112442235444462335466666665477865555478855899777888899777899999',
  '111112223112222223144445533146445333446555377666577378665577778699999988699988888',
  '111111122334114222334444442536647772536666672533386672588888877555899877559999999',
  '111111122133333222134433352644477752644777552647775552648885589666888889669999999',
  '111112222311144222333144442333545542363555747865565777866669777888669997888899999',
  '111112222133342522163344522163347555166347558666347758996447758996947778999988888',
  '111112222131422525133446525133346525777346555787346669787344669787884969888899999',
  '111122223411122233441123333445523663445555566744785566777789966778889996788889999',
  '111122333111123333144222233442256666455555556444457766887777669888879999888779999',
  '111111222313141552333445522336645527388645227386645577886644777866949797888999999',
  '111111222113222224153333324553636344556666644557676744587777749588888799888999999',
  '111112333114112223444442233455542233467555238667785558667788888677799899666799999',
  '111111222113412225333442265377742665377746665377846665378844555388894599888999999',
  '111111222311422222314444442355554646333356666373755556877777796888887996888999999',
  '111111122333314122333554222633554422666654444776655884777655888779698888779999999',
  '111111122334144122334444222335554226335754566388755566888777766889779766889999999',
  '111111122133144422334444222335454662735555562733858566777888866778889669779999999',
  '111111222131312224533322444533332644566666664556788884555778884577798989777999999',
  '111122333111112223444442333445222336455555556477888566777866666788899999777889999',
  '111112222131122242331324444333355444635555574666557777666687977868889979888899999',
  '111111222133322224133335224136355544666654444665557478699577778699997778999888888',
  '111122223111122233144523336444525336474555636477585666477785669778889999788889999',
  '111112222133312244135552444133352444633752884666758889666755589667798889777799999',
  '111111222333312222333415226344415666345555567444586667499586777999987777999888888',
  '111112233141122333441222333445252663445555566744858566777888966777889969778899999',
  '111111222133311244553662244533662274533862774538866774558866744558997779888999999',
  '111112222311122244315522544365555554366666664377777764337887794338889994888899999',
  '111112222113114222333334442335366642555666444756668488755588888777599899777799999',
  '111112222131122422133524446335554466377754446337755566377785669887889969888899999',
  '111112222133112242133552442333544444363555747666665777866855779868899779888899999',
  '111111222333141552333341552336644522776645522776445588766948888766949888777999999',
  '111111222311144422344444222333345266355555556337586666777888886778889996777999999',
  '111111233141122223444425233466455237468555237468557337668587777688889979668999999',
  '111111222133145522133445552336645722833645772883645577866644779886649779888999999',
  '111112222311122222314445666344555766344457776334555776333577786999998886999988888',
  '111112222333112222433511662433556666433555776444455776844995776888899777888899999',
  '111112222333314222351114226355554776355444776355477776388499976888496666888899999',
  '111122233111222233141422333444445336455555556477566666777886969778888999778889999',
  '111222333111123333112222343555626644555666444556676444858777799888879999888777999',
  '111111222131341222333342252636344555666347555666447575868847777888947979888999999',
  '111111222133311242333322242536644444536666674555556674858887777858997779888999999',
  '111111222133322222134445525133346555777346555777346668797744468999996668999888888',
  '111111222331222244311524444335524664733555664733785566777785996778888966888999999',
  '111112222113333244113332224555332664556666644566788444577788899557888899777799999',
  '111112222333311224533311244566332244556666644557788664557998884577998888777799999',
  '111112222133312422113332442553336644556666644556677744855877799885897779888899999',
  '111112222113142222313445525333346555777346555777346666787744696888849699888899999',
  '111112222331122442311552466377752446337555466377854446337855996877889966888899999',
  '111112222311222422311554466377755466337754466337554446337755886997999886999988888',
  '111112222111122322455525333455555363466666663464777773444787773884889999888899999',
  '111111122113333222444453332444453322645555572668857777688857777666888899669999999',
  '111111122333341122333544442633555422677754442667555882677775888669978888669999999',
  '111111122333114442334444442333552222365555527666655777688888877688899777669999999',
  '111111122333311222343355222344556266344456667448455667888557767888997777889999999',
  '111111112333334122333544422366545422776555422776565448776665888779688888799999999',
  '111112233141152223441552223444456323777456333787456666788855966788859969778899999',
  '111112233141122233144222333444425536475555536475586666777888669778889969778899999',
  '111112233141152233444155233444652237488652337486652777886559777886659979886699999',
  '111111233144222223114222333444455333445555566777556666777888699788888669778999999',
  '111112223441122323411522333445523366447555366447785566777885996787889966788899999',
  '111112222111222223144555553444666653477765553476666333477777338499999888999988888',
  '111112222133422552133455552113445652333745666873744699877774669877884669888899999',
  '111111122333311222344552262344452667334555677344856667848855667888997777889999999',
  '111112333414122233444112333442222553445555566755888866777899666778889696777899999',
  '111111112333144442333444422335466622355666227556668277558888777588889777599999999',
  '111111123441412223444422333455522233466555337668885557666887777688897977689999999',
  '111112222333111222333341242355444444365555547666666557868967777888999777888899999',
  '111111122331444122344454222334455522333356666775558866777858886779888966779999999',
  '111112333141522223141522223444566633774555633774445666788885969788885969777899999',
  '111112222311122242331544442333554466377555446337755666877775966878889996888899999',
  '111111112333331222443332222455536626444555666474485556777788866777988888799999999',
  '111111222314151222344556662334457762834457762834457766833355776888959796888999999',
  '111111122333114422333544422335554222366654447888655577886665777886699777889999999',
  '111111122313441422333444422333445222355555556777588666778888666778988696779999999',
  '111111112333322212334332222354446666555546666555544467888877477898887777899999999',
  '111111222113122222433355662443355666433357776444557766844557776888889799888999999',
  '111111222311312242333332444536662444536762684555766684555788888757798998777999999',
  '111112222131422222131455555633447575633347775636344775666664878999994878999988888',
  '111111112333456612333455622373456622773456822774456828774556888794456888799999999',
  '111111122133444422135444222335544622333555666773885566777888569778888669779999999',
  '111111222113142255333442555363342225663347755688847757666844777668849799888999999',
  '111111222113314222333334522363644525666645555676445858776488888777498899777999999',
  '111112222113444222113444442333455562333555666735558666788888699777888699777799999',
  '111112222133112242333132444553332664556666644566788844555789888757799889777799999',
  '111111222113441552333645552333645722883645722883645777866645777866944799888999999',
  '111111122113332224333322444556332244566666664557788644555778888577788899779999999',
  '111122223114122323414152333444552363447555366747855666777859696787889699788889999',
  '111112222133332222113455552613444755663347755663444785966664788999977778999988888',
  '111111122311332222333332444556362244556666644557768644555788888777788998779999999',
  '111112222311122442333125442333335544666555444665577777866589777866889997888899999',
  '111111233112222233145522333444526633444555666774485666777885569778888899778999999',
  '111112222314142252314444252314664455336666655337766785939777785939978785999988888',
  '111112222111122233445552253446655553476666653477776633478877733448889999888899999',
  '111112222113142522613342555633442755633347775663844775666847795886849799888899999',
  '111122233144112223144422233145453333445555566777756568779996668799988668779998888',
  '111111222133444422133344422135345462735555562738586569778886669778888669777999999',
  '111111222133312242153322244553333664556666644566777744558887749858897779888999999',
  '111111233144112233444222233445525233644555773668585577668888777668899779668999999',
  '111112222314122556314772566314772556334872566344877596334877596344889596888899999',
  '111112222333311122343551622443556662443357766844455766884955767889997777888899999',
  '111123333141122235141222335446622335446666655477886655477888959478889959777789999',
  '111122333144123333441123235441622225466666665477776855497978855999978558999778888',
  '111122223141225323144225333114445333444555666777566688777599668797599868799998888',
  '111112222111312242555336242577736644577736664557736664858733444858893999888899999',
  '111111112334444412333445442333355222666555222666557777688588777698888877699999999',
  '111122233441222233441252333411553366441555766448855776888959766889999766889997777',
  '111111222344441122344455122333445626373555626373588666779558886779988886777999999',
  '111111112314444422333344422335342222355555556777786566778886666778888896799999999',
  '111111122333333122444355122464335727464555727464588727669558777669888888669999999',
  '111111122333112222333445522633345552666444555766648885776644888777799888779999999',
  '111111122333341122533346662577344662577348662577448662577748882559948888559999999',
  '111112222113312242133542244335544444333555666777775566778875669878896699888899999',
  '111122223414123223414153223444153333445555566777758666799758686799798686799998888',
  '111111222334451222344551122344456772333456777833456667889955667888956677888999999',
  '111122233441222233441125233444115536477555336475588666779598866779999866779998888',
  '111112222113144422133555422336654427333654777386654477886555779886669799888899999',
  '111112222131222422133355444133554444633357774666655778666557778996999878999988888',
  '111112223144111223444552233464552733664852733664855737668855777688999779688899999',
  '111111222113333322144443322444433522455555556775886666778866669778888899777999999',
  '111111122113444442333333442536636422557666422557686682577888888577777899559999999',
  '111111223111222223444526233744556633774555633774455663778485666788888999788999999',
  '111111222133222222143335566143335666443555766444577768445577768999999778999888888',
  '111122233114122223144523333144523366445555566447785669777785669788889699778889999',
  '111112222133222242133354444136655477136555478336554478666657778969999778999988888',
  '111111222133331242156333242556633244557666244557886644757888649757988889777999999',
  '111111222133122222133334425663344555666347555666447755684477779888889779888999999',
  '111111112341422222344445522333455662334457766833557666885577776888887976899999999',
  '111111123444221123444222223454333333455555556777777656788888666789988666789999999',
  '111112223114222333114225533444425633474555636774586666775588699777888699788899999',
  '111112222133334442113444442133545222335555566777585669788888699788866669777799999',
  '111112223111412333444442233445222533465555537665888577668877777666897999688899999',
  '111111122333331122443332225443622655446666655446776855477788855779988888779999999',
  '111112222113412522133455526337445526337745566387744566387774669887894699888899999',
  '111111222333111222433332252436636555444666555444676675848877775888999777888999999',
  '111112223441152223441152333446552373446852377646855377666859977688859977688899999',
  '111111233412222223411225333444445336475555536477566666777588996788888896778999999',
  '111111122134144222334444252336644252336666655737886655737888855777889859779999999',
  '111112222333311112334344522344445522344555667885566667885667677899997777888899999',
  '111111222122222233114455336444453336477555336477756666477556688779999998999888888',
  '111111222313112222333452526374455556377444556377774456387874666888899696888999999',
  '111112223111444223144452223446555233466653337668555377688857779688777999688899999',
  '111111122333341122334442222335444442355555556788888566777788866779986666779999999',
  '111222223111112323414452333444452336475555536477856666777856696787899999788888999',
  '111112222311122244331225444333335544663555744665577777666588977668889997888899999',
  '111111112344442122334422222334445556333555666355577766888887766889877776899999999',
  '111112222113342552133442255133346255777346555778346669778844669877846699888899999',
  '111112333111122344552222344567722334567777734566887734556888844556889999666899999',
  '111112222131144222331445226334455766344555776334557766388577966888779969888899999',
  '111112222131411222333455226344455776334457766344557776388557666888997969888899999',
  '111111112333332212444333222444355226445555566477558666777888666797788888799999999',
  '111122223114422233114532236144533336445555566477775668479975688779996688799998888',
  '111122333114152333144552233144652273444652777846652779886655779888659799888669999',
  '111111222331114442335544222333544426378555426378885666777885566788899966777999999',
  '111112233111422233441422333444442253655555553657788888666778988667778999667799999',
  '111111122333411422344444442333355222635555572666557777688888887668998777669999999',
  '111112222133222242153336444153366744153666748553667748555677748959999778999988888',
  '111112222313122442333112442333554444635555574666655777866899777866889797888899999',
  '111111233411222233412223333444255563444555666745558666777788896778888996778999999',
  '111111122333311224335222224335554444366654447666655577688888577688997777889999999',
  '111111123141222223444442223456463333456666637555567637588877777588888979589999999',
  '111111112313445522333645222336645522733645882776645588777645888776644898799999999',
  '111111222131133222433333522466663552444465555744866665774888885777889989777999999',
  '111112222333122242335111244335444444335555566777777566778999566878889666888899999',
  '111111222113334252313344252333645252666645555767645888767448898767488899777999999',
  '111122333114112333144222533444255536442257766485557666885777669888799699888779999',
  '111111222334141225344441225366444225336666655377888665377988885377989855777999999',
  '111112222113415252333415552333445552633745882677744888677794888676794899666699999',
  '111111222331441222331546622335546672335546677835546677885546977888944977888999999',
  '111111112314444442344452222333555622733356662773555666777758886788888896799999999',
  '111111112313344422333444442353466222356666627555668727588888777558887797599999999',
  '111111112333144222334444225364747725364777825367787825366888855666889555699999999',
  '111222333412222355411122335466113335446666655477788665477998885447999985777999888',
  '111112222311112242333554442363557442663357744866355747866655777868899997888899999',
  '111111122333341122355346662335346622775444622775548688755548668779948888779999999',
  '111112222113332442133322542163335544666555444665577748965997778966977788999988888',
  '111122233414122233444122333441152336455555556477859966777889666778889696778889999',
  '111111233144422233145422233145552333444652777888655579886667579886667779886999999',
  '111111222111334222555634227555634247566634447568634777588634777888633999888999999',
  '111112222113332224133332244155366664555367444566667448559977778599977788999988888',
  '111112222133322244133554224133544444133555667888885667899855667889996667999977777',
  '111111222311421225334422245336444445366666665377777655378887755388987995888999999',
  '111112222111133322444443322454433332455555556788886656778866666778889999777799999',
  '111111112133444422553336442533336442576666642577688882577688822557777889599999999',
  '111112233144122223144122333444525336475555536477585666777889669788889669778899999',
  '111111222331112222333334242536664444536564674555566674858577777888899977888999999',
  '111112222112222323144444333145444333655555553666777578666777778696999988999988888',
  '111111233141212233444222333444552263745555563748855666777888666778898969778999999',
  '111111122334441122333455522333456622734456682774456888775556888779966688779999999',
  '111111222333441225344461225344661225344666775388966775388967775388977555888999999',
  '111222233141122333444112235464177235467777735468779535468899555666889959668888999',
  '111123333111222334512223334512626644555666444556676784599977784599777888999978888',
  '111222333111223344511223334551223664556666644566788944577788994557788999777888999',
  '111112222311222224311555644377756644377756664337756664337555884399999884999988888',
  '111112233111122223444423233445423336455555556477786566778786666788889999778899999',
  '111112333114512223144522233444552633744852663774855666778885669788895699777899999',
  '111111222113333222413332252446335555466666665444477655848877795888777799888999999',
  '111112223122222323114553333144553666444753666444755668777755688797999998799988888',
  '111111223144112223444422333544422363566666663565778883555778888577799889577999999',
  '111111233441122223441225233446255337446555377466558377668588977688889977668999999',
  '111111222133122244135522444633552774663357744633855774666885579668889779888999999',
  '111112233114122223444152223464455733664555733664557737688859777688889799668899999',
  '111111122333341152363345552363445552667745522677744828677748828679948888669999999',
  '111111112344444212344455222333455226373356626377558666777558886797888886799999999',
  '111111233111222233422233335426665535444666555474466685477778885778888999778999999',
  '111112222134112225134466725333466725344467775384667555384667759388899759888899999',
  '111222233411112233441522336441552336447555366477855966477885966778899996778888999',
  '111111222112222233142553333444553663444753666744755666777755869778888899888999999',
  '111111222311441222335444422335544426333555666378885566778888566777988996777999999',
  '111112222113113224333333244356662244555666444557766648557888888577899899777799999',
  '111112333114122233144223335444622235466666665478886555477788559778889599777899999',
  '111112222133322244133332244113535444665555544666575788669977778669997778999988888',
  '111111122333334112333434522644444522645555572665777772665787888699788888669999999',
  '111111122331442225331444225344466225346666675388667775388777955388877955889999999',
  '111111112334442212333444222333455552634457782655557888666777888696677788699999999',
  '111112223114122223414523233444553336447555366477755666778785696788889699788899999',
  '111111112313334422333444442533446622556666622556677882577777888557788898599999999',
  '111112222111322244513323244563333344566666664557777764558787794558887999888899999',
  '111112222113322222113334555663334455663444755664477755666477788999997788999988888',
  '111112222311442442313444422313345552333555666755586696778888696788788996777799999',
  '111112222111122223444555523455566553446666633477667773487777333488889999888899999',
  '111111222333311242335541442333544422677555442667775888677975588676998888666999999',
  '111111122133344442134444222335554262337555266737855566777888869788886669779999999',
  '111111222311222224315256644335556774355666774355677744336678794388888994888999999',
  '111111112313344442333344442355646222356666627555686227588887777588887797599999999',
  '111111122113444222333342225344442665346666675366877775388875555888777599889999999',
  '111112222113452222133455552133456655733444665773374668977774668999974688999988888',
  '111111222333311142333544442336545422776555422776565488766665888769998888777999999',
  '111112222133122224133352444135554444633357774666655578666957778699998778999988888',
  '111111122133332224133355224336652244666555444667754488677558889677788889779999999',
  '111111112333442212333342222555346662555347666855547666888847777898844777899999999',
  '111112333114222233144222333145552636444555666474855569777888669778888699777899999',
  '111111223114422233144452233444552233466555337668855777668857779668887799688999999',
  '111112222133411222333441526344444526345555576385777776385977666888997669888899999',
  '111111122134442222134444422333455526333555666375558666778888869777788869779999999',
  '111112222111122322444444332556444333566666663555777633855777777885889999888899999',
  '111111222134441222133344522334445562334555766835577766885776669888977769888999999',
  '111112222113222424113325444663355574633357774635557744666587799686888799888899999',
  '111112222311112442333444442333455522335456577885556777866666777866899997888899999',
  '111111123144444223145644223555664223577666223577866333577886389577888889579999999',
  '111112222111122222333334444536333644556666644556777674555577777888889999888899999',
  '111111222331444422331144422335554226335754566388755566887779966887777966888999999',
  '111111233122222233114423333444525366445555566447585666777786699778888889778999999',
  '111122223111123233144523333445522663444555666744885566777785669778789999788889999',
  '111112222133422225133442555133346555177346558777346668777944668799994668999988888',
  '111111122333144552633145522633745582637745582637745882667749882677449888669999999',
  '111111222133331322144433322444535522444555666775585666778886669778988889777999999',
  '111122223114112333144222335446662335447666355477866655477888559777899599788889999',
  '111122333144112333144522233145522263444555666748885569778885669777899669777889999',
  '111112223141512223441552233444452336477753336477856666778855966788895969788899999',
  '111111222333112224536612244533662444533367774555866774558896674588899777888999999',
  '111111222131122222333333324355566444556666644555664447587777777888889979888999999',
  '111111122113333422333344442355544422667555422667775558677778888667888899669999999',
  '111112222111332244551532444555532464757832464757836666777836966778833999888899999',
  '111111222333122222334125555334155556374444456377778466777798466999998666999888888',
  '111122333141123333441122233442225556444555666455577766887779966888879969888779999',
  '111112222333114242333114442553366642753666842756668844755599888757599888777799999',
  '111111222131344442133334442335554422366555227668855577688877779688887979666999999',
  '111112222131422255331425255331455556333444666377774866779794866779994868999988888',
  '111111122113344422333444422533366442536666672588667772558888777558887799559999999',
  '111112222113333222114433332444444352455555556758666666788886699777888899777799999',
  '111111112333142222344442222345445666345555576333577576888877776888879666899999999',
  '111111122113222224333555524355556624356666674386677774387777444388888499889999999',
  '111111222111333222444443332445533322445555566778885566788866666777888999777999999',
  '111112222113322322113333324555636644555666444556676444587777799887887799888899999',
  '111111222133311442333444442533366442556666622577668882577777888577998889555999999',
  '111112233111222333144423353442226653446666655746688855747785559777888999778899999',
  '111112223144412233114452223444455233666555333667558888677758899667798889677799999',
  '111111112344441222334422255366444225336666655377888665337778855777988885799999999',
  '111112223411112323444422333444225553664555733655588777666887777686899997688899999',
  '111111112333334122534444422536464472536666672538868672558888872559877777599999999',
  '111111122313144522333645522336645227386645527388645577886645777886449797889999999',
  '111111222113332222133435252334445555636645575666644477868674779888877799888999999',
  '111112222113332242511332442533336644556666644556677774855877994858877799888899999',
  '111111222341411222344442255346662555344666775333866675338877775888997975888999999',
  '111112222344111222344451262344455566333457666335557776838957776888999776888899999',
  '111111122333311222343522266443525666433555776444585766448885767888997777889999999',
  '111111122344441122334544426335554226337752266377855566378885866779988886779999999',
  '111112222333112242333112244355544444336555477666665557668899777868899777888899999',
  '111111222111333242333322244355566244356666647558664447558887777858777999888999999',
  '111111122133344122335346662735346622735546682775548682755548688779448889779999999',
  '111112222131122222133333444335566644355666447556664477555777778999998878999988888',
  '111112222111332442155532242555534446577734446577736666878836669877833999888899999',
  '111111112333331222334445222344455266347555268447556668777566688777988888799999999',
  '111111222131122242333325244336655547366654447365554477668587777868889979888999999',
  '111123333111222223114225333444525366444555666447585666777588699788888999777789999',
  '111111222111332242553322444533662774536666674533866774555887744858877999888999999',
  '111111222133322242133325244136655444336555477666554478669597778969997778999888888',
  '111111122334441122334455526334452226374458826377758866375558866779988866779999999',
  '111111112333333122344352222444355622444555666774558666777758668779888888799999999',
  '111112222113122442333125442333355444366555447666557777866589777866889799888899999',
  '111112223141122333144552333444552236474852636478855666777855669777889969788899999',
  '111111122133312222444332322445533366445555566447775566887877666888897779889999999',
  '111111223441112233444222233445552336477555336477855566778888666778899966788999999',
  '111112222111222333142233333444443556445555566455766666777778869777888999888899999',
  '111112222133332222143333325144466655144666557446665557489999957888899997888877777',
  '111111222311144442333445222334445226335555566377588866777588666788889996777999999',
  '111111222111222232433333332444444536455555556475666666877777776878888999888999999',
  '111222223111112233444122335444633335466666665477776555477889555778899999788888999',
  '111112222113333322114433322444455326445555566478556666778886699778888899777799999',
  '111112222113122444513662224533667744533367774553366774588866794555889799888899999',
  '111111112133333322434553322444452222444555666777756666778855686778888889799999999',
  '111112333111212344522222344567727344566777334556787734556888884556898999666899999',
  '111112222112222333114552333444556363744456663747455666777855699777888899888899999',
  '111112222331442222331145256377145556337745566377748556379748866999944866999988888',
  '111111222133145522133445222333645552336645577866645777888644779886649779888999999',
  '111112222111314552333334522366644527386645527386445557886477777866497999888899999',
  '111111112334444412334444222333552262735555562737755666777888866798888866799999999',
  '111111222131145222333445225333345525666347555686647777688644777888649979888999999',
  '111111112333331442334444442335554222666555222666755588677777788677988888699999999',
  '111112222333311422355116427355566427385666427385664447385699447885997777888899999',
  '111112333114112235444222235466662335477862335477866665478888555478899599777899999',
  '111112222133411222133455225333445555363645757666644777688664779888994779888899999',
  '111112223444112223441152233445552336477753336477855566778859966788899666788899999',
  '111111122314444122334454422634555772633357772633555872668858877669888897669999999',
  '111111222133311442333444422335445422365555527668588577668888777688997779666999999',
  '111112223122222333112444333155554433666654444776655558777666988777999998799988888',
  '111112222133322422115332444615533334666555444677775584666977588996997778999988888',
  '111111122131334222333344422354444422355555556778888856778886666777866969779999999',
  '111122333144123333114222223144552663445555566744855669788888699777789669777889999',
  '111112222131444422131542226335544466337555466337775566388875969887777969888899999',
  '111122333441112233441522333441552236447555266487755966888775966887799966888779999',
  '111111223144444223144454223166552223666555333677755338677959998677999998677888888',
  '111112222133332442113332244153366244556666644557668849557788899755788889777799999',
  '111111122133444422133554442663554772633854772633855722688855779668888779669999999',
  '111112222131444255131664225337664225377764445388766455388766959338777959888899999',
  '111111222333141222355142262355544766355547776335447776838849776888949666888999999',
  '111112222111222322144433333445533663445555566744775566777776668997999888999988888',
  '111111112334444122333442222333544422355555556778885666777788666779888866799999999',
  '111222223111122333141523363445523366444555666447785566747785969777889999788888999',
  '111111122333311224353366624555362224557368244577768444576668848577998888779999999',
  '111112222113312244133552664773522644773555644773885644733855669778896699888899999',
  '111122223411123333414222333414255563444555666745558696777888696777789996788889999',
  '111112233114211223444222233454443333455555556777766656778888666788998699778899999',
  '111112233144422223111425223444455333664555733666557777688587999688887779668899999',
  '111111122344142122344442222343455566333555666335557676888877776889879776889999999',
  '111111122311422222333444422355544466337555466337775556887777666888887996889999999',
  '111112222311112222333455266333455666334456677444556777448556777888899997888899999',
  '111111233444111233544444233556664233755666223775866622775888882775999888775999999',
  '111111122133333322134343222444445562745555562745566666777868689778888889779999999',
  '111111222314122225314442255366644445337666455377776665338877795388889795888999999',
  '111111122334441122334452222334555672334456677834555677888856677889966677889999999',
  '111111222333112242333122244355566447356666647355664447558889777858899777888999999',
  '111112223141152233144552233444456233778456233778456666778855669778859969788899999',
  '111111122113334442553366442533336442577666442577688882577668822577788899559999999',
  '111111233441112233444422233445525233467555238667575588667778888667799988667999999',
  '111111222331112242333222544363335544666555444665577747665888777868899977888999999',
  '111112222331112222331444255333645555336645577666645777668444977888899977888899999',
  '111112222113312442155333342556636342758666342758686644758888449755798899777799999',
  '111112222333112442333115442663355542766358442765558844766599888766799888777799999',
  '111112222331222244331124444333155464735555564737558666777798866779999866999988888',
  '111122333114222233144552233114552633444456666774855699778855669778888699777889999',
  '111112333141112333444422323455522223444555666788885556787886666777899969777899999',
  '111122333112223333112222453166644453666645555764445558764999988777799988777998888',
  '111111112133444442334442222335545226337555266377585566777788866788888669799999999',
  '111111112313334442333544442363554422666555222667755828677775888677788898699999999',
  '111123333111123343522122344552226344566666664557688844557889884757789999777789999',
  '111112233141222223144442233145533336445555566477775568779966668799999868779988888',
  '111112222133112442333155422336665427366555447386544477886559777866899779888899999',
  '111111122113333322433335526444455226447752266477556666475588886778888899779999999',
  '111111122333441122334444422536644772536666672533886672558888877559988777559999999',
  '111123333111222223441125233444525636474555636474585666778589966788888999777789999',
  '111111222133311222434352226444355566433357776445557666488857676888997779888999999',
  '111111122134444222133554222333554426337754466387755666888755669888777769889999999',
  '111111233412222233415222336415553366444753666447755586477999586779999986779888888',
  '111111222113322222133334442335366644555666444556667477855577779888887799888999999',
  '111112222131134442133335442333555442667754422677555888677588889677789989666699999',
  '111111122113333222433355262444355266447358266447558666747558886777888899779999999',
  '111111122133412222333455522344456627334456677384456667885556777888896779889999999',
  '111112222133122442135524444133526644733555664773385669777785569877889669888899999',
  '111111233141212223444222523464444533665555533665777737685888777688898979668999999',
  '111112222131122242133444442333455556334457766355557666877777669878889969888899999',
  '111111223144222233114455233444455233667458233667558888667558899667777889677999999',
  '111111112314444222334522262334555762334457766834555766838885766888777796899999999',
  '111111222133334422133334222135554424666555444676655589777688889776688889777999999',
  '111112222131112242333344442344445562335555566735588886788886666787799969777799999',
  '111111112333444412333454422633355422667555422667558882667757888697777888699999999',
  '111112222113445222133345252133345555633645775666647778969647778999644788999988888',
  '111112222311122444312224455336664445337666455377766655337788895777889995888899999',
  '111111222311144442335554442333356422778356422778356666788855566788889996777999999',
  '111111222131133322433335322444555662744456662744555666778588886778889989777999999',
  '111111112334441222344445266347755266347555286337552286337588886777988866799999999',
  '111111222111344442333334442533364642556666622576768882577788888577778999555999999',
  '111111233141551223444452223446452233466752338667758388677758888677955989667999999',
  '111111223122222233124444443154466433556666633557667738577777798559999998599888888',
  '111111222334441122333441526344445526377555226375588886375988666779988866777999999',
  '111112222311145522333145222333445556377745556377744666888749666887749996888899999',
  '111111123411222223442255523444456333777456333777456666785558866788888996789999999',
  '111112222331114222333144225367744445367777745366667745388669555888699955888899999',
  '111111122113334422333344442335444222655555552666777588677778888667788899669999999',
  '111112222133222442133256444173556644173555648773355648777359668977999668999988888',
  '111111122311444442344445222333355522363357727665557777666588887688888997669999999',
  '111111122334512222334512666334552676334852677434855677444895677888895677889999999',
  '111112222134442522134444522134466555136666657333668857993888857993988857999977777',
  '111112233111412333555442333566642773555642777855642227888644777888694999886699999',
  '111111223444111233444222223445552333466555337666855577688888777668999777688999999',
  '111112222113222422113554444633557774663357744633355774666655788996999788999988888',
  '111112222113422252113445552333345555333645777666647777866644799868884799888899999',
  '111111122333342112333342222553446626555546666575544866777748888799748888779999999',
  '111111122313122224333335224336655244666555444667554488677588888677779898779999999',
  '111112223111212333442222533444445536477555336475566666775888866777898999788899999',
  '111111222311414422333444425336664225376864625377866655378888555778898995777999999',
  '111122333112122234112225334666655334667555344677554444677588899678889899777889999',
  '111111233144422233114422223444452333655555553666758888677778899667778889667999999',
  '111111112313342222333342222335544444665555544666665577888867777888867797899999999',
  '111111222333314422333511422365554422366654447886655547886995777886697777888999999',
  '111122223411122233411225333444445333445555566777566666777588996778889996788889999',
  '111112233141122333441522223444525333444555666777585666788885966777889969778899999',
  '111111122133122224333322444553332464566666664565788844555778888577779889779999999',
  '111111112314442222334442555344462225366666665377768885333788855777788895799999999',
  '111111222331411222333452252333455555636444575666664777868864777888994977888999999',
  '111111222133112222433333225466663355444465555447766665488777775888899779888999999',
  '111111222131331244533333244556663224577666224577866644557888884557988989777999999',
  '111112233114112233444455236477552236478852236478855336478556666778899699778899999',
  '111112222131122242331524444333554464733555664737755666777785966878889969888899999',
  '111111222133332224133522244133555444636654474666555778669995778699977778999888888',
  '111111222311144422344445422366665422333365555778366665778388885778889995777999999',
  '111111222331112224335522424333554444336654477666655777686885577688899977888999999',
  '111112222131122224133524444333554444333555666777755666777785669788889969888899999',
  '111112222311112442333562422335566447335666477355664477885864777855899997888899999',
  '111112222133222442133333442153366444156666647555668847955888887955999887999977777',
  '111111222333111222433355522444356772444356777844356777885556667888999666888999999',
  '111111222334441222334441122334555562334457766835555766889977766888977766888999999',
  '111112222311311224333332224356644444356666647555556647588877777588997997888899999',
  '111111122113442222333542666335542667335842677355846677555846777888844799889999999',
  '111112333111142223555442233565542237568547237568847737668844777688849999666899999',
  '111111122134412222334455222344445666343555676333577776888557766888897769889999999',
  '111111222111322222444352666443355677443555677443556677333856777888886999888999999',
  '111112222133322444132224444133355464135555567838556667888899967888996667999977777',
  '111112233112222333144452233144452336445555566477856669778856669777888899778899999',
  '111222223141222323141113333444553366445555566447755666777788868797999868799999888',
  '111111222331122222333142555334445565737745565737744466777849666888889966888999999',
  '111111222131322224133322444333355444665555544666557777666887779688887979888999999',
  '111112222331442222311144245366664445333764555377766665378779995888877955888899999',
  '111111222113331222443322255444366665433367775466667555448887755888977799888999999',
  '111111222313314442333514442335554222367754428666755588677795888677798898666999999',
  '111112222331111442333444442333455542663555722685558777688888777688999977666699999',
  '111122223111223223441153323444555333444456666777555666787759966788788999788889999',
  '111111222331122244331552246337755246377754446378554466378855966778889966888999999',
  '111111222111332222433332255463335555466666665444477765448877775888877999888999999',
  '111112223114411233444222223445533333445555566777775566788888666778996699788899999',
  '111111122331212224333222444335554446377555446377755566777888666788898966889999999',
  '111111222313311422333344442563347742567777742567768842566668888556998898555999999',
  '111111122133331222334335526444355226474456626477558666475588688777988889779999999',
  '111112222313312222333311244355556644356666647556644447558997777888897797888899999',
  '111111112133445222334445222334455262334457766838557766888577766888577669899999999',
  '111111222311144422333554442333554422633754882667755888677755888667779998666999999',
  '111112223114112223444422533446625537466555337465583377665887777688899799688899999',
  '111112222133333322143332225144466655144666557446665557488899957889999997888877777',
  '111111122131422222333422555336445557336645577366644577666884777888884979889999999',
  '111112222333411112344445222334455526334457766385557766888577776899997666888899999',
  '111112222131122222133455555334455665733444665733774466777774668999998868999988888',
  '111111112344444212344442222333333352655555552657777777666688887696888887699999999',
  '111112222333311122433335122443555662444456666744555866779588886779998888777799999',
  '111111222333331112433555222433456622444456666774456886777555886799988888777999999',
  '111111122333331122333435552674455522677444552667774452677784888669988888669999999',
  '111122333114442333114452233144555223446652277866555779886657799888677799888669999',
  '111111223141222223441253333444255336447555366477558666777758966788888969788999999',
  '111111122112222223444444233544466633555666333556667773558777777588888899889999999',
  '111112222341152262344152266344155776334457766344558776339958776939958876999988888',
  '111112222311311442333355422663355422666357444886557744886557777866997997888899999',
  '111112222344112225334411425366444425336666655378888665378998855377799885777799999',
  '111111122134441422334544422333545222333555666777585666778885866778988869779999999',
  '111112222111334222156634227555634477556634477556634777588634479888633999888899999',
  '111112222314411252334461255344662275344666775348866775338967755838997795888899999',
  '111112222333122422333112444355616447356666647355686447555988777995998777999988888',
  '111111112333333122345533222444566622744555662774445666777885568779888888799999999',
  '111111223444155233444152233446152333746652283777659288776659888776559888766999999',
  '111112222131442425131444425334466225336666655377668855378888959378788959777799999',
  '111111112314444442333422222335445262335555566737588566777778666788888896799999999',
  '111112233411222233411425233444425336475555536477586666778586996778888996778899999',
  '111112222331312224533311224556333644556666644556777644588997774588897977888899999',
  '111111122333132122343332222445356666445555566444457566888877767889879777889999999',
  '111112223111212323444222333444455533464555737665557777666888777686898999688899999',
  '111111222111222232445553236444453336477753336477756666478755566878888999888999999',
  '111111122133334122335554222633354422666654444776658884777655588779688889779999999',
  '111112333122222333112443334122544444655555554666665778699966788999777778999788888',
  '111112233144112333444122223444225533465555537665588777688889777666899779668899999',
  '111111112333122222334444225344446625346666675386677775388777755888889555899999999',
  '111112222331112424333122444333552444635555574666855777666889777686899977888899999',
  '111222333141123355441223335411222635466666665476888995477788955447789959777888999',
  '111112222113442222113444425333444655366666665336777555387777599888877599888899999',
  '111112222133412222133415255333455555363444757666664777668694779888894779888899999',
  '111222223111142333511442333555542236575546636578846666777844996777849999788888999',
  '111111233144522233144552333144452263748852263748856669777855669778885669778999999',
  '111111123122222223144244333544444633566666663556777773555778779588888889589999999',
  '111122223111222333411223353444633655446666655446776555747788995777888999788889999',
  '111112222133144422133144422335554462337555466837755566887779669887779669888899999',
  '111112223111412233444422533466422537465555537465887337665887777668897999688899999',
  '111111222333312222334411522336445555636645575666644577886994477888897777888999999',
  '111112222133332224133332244115366644555666444556667488559977778599977778999988888',
  '111112223411122333412222333444444353455555556757666666777888896777889996788899999',
  '111111222333111222334444442335444522335555566775888566788888866777999666777999999',
  '111112223111222223144555233444456333477456336777456666778555669788888999788899999',
  '111112222331112242333512442635554444633357774666655574866895777868899977888899999',
  '111112222311442222311442455366644445333666555377776665337877995888877995888899999',
  '111112222111333222143333332444445562445555566745566666788888869777888999777799999',
  '111112222111122232444445532644455333666457333666557773865577777868889999888899999',
  '111122233114125223114455233444452233466457337668857777668557799688589799668889999',
  '111111122113334422333554442363554422367754428667755828677755888667788899669999999',
  '111112223111222223144442333445555333644457773666555577666877779688888999688899999',
  '111122233111142223155544233555542233566547337668847777668447779688849999668889999',
  '111222223141123223141123333445553636444555666474755566777789969788789969788888999',
  '111111112344444212343454222333554262733555662737855666777858686797888886799999999',
  '111112333141222233141522233444552336444555666477855666778885969778888969777899999',
  '111112333112222233112444223555546663775546633755546666788444899778888899777899999',
  '111122223111123233445123333445522366444555666447885566777789566778789999788889999',
  '111112222333311112333455522334445522666645555776644488776664888799998888777799999',
  '111112223114122233444152333464452237465555537468857737666859777668889799688899999',
  '111111122314152222334552627334452677344456667334856677384855677888859697889999999',
  '111111112333334412533344222536664442576764642577766682555778882597788888599999999',
  '111111222133322244113322244333355544666555444665557777668887799668887779888999999',
  '111112222331122242311566244375556664377756664377755564338775994838889944888899999',
  '111111233411122233444445233446555233476652238776555288776588888776669998776999999',
  '111111233122222223142533333144556663444555666744455668777775968799999998779888888',
  '111111233414122223414552223444452333646555373666857777688855797688889797668999999',
  '111122223122222333111145533444445336447555366477566666775568888777999998799998888',
  '111111222112222223144443333546666733544467773554666673555577778599999988999888888',
  '111122223114112233414222333444455336475555536477556666777888696778899699788889999',
  '111111222111333342333355442673554422677754442667755842677558888678888999666999999',
  '111112222111222322144553333444455333464457737666557777666655778996999888999988888',
  '111112222331142525331142555333442655773842655773844666777849966787849966888899999',
  '111111222113122242333444242333454446355555556377757666878777666878889699888999999',
  '111112222111322442513326642533366444533367774555667774856687794855887999888899999',
  '111111223144452233114552333444652233474652838776652888777655899776658889766999999',
  '111112222314112526314772566344772566344872556334877556334877596384899596888899999',
  '111112222133322224153333424156336444156666647555688647595888847599998887999977777',
  '111111222113415222333455522336455572336444577836664577886664777888694799888999999',
  '111112222111122332444422333455444333465555537666777557666887777866889999888899999',
  '111111122113222222333244556337745556377444556377745566377448666888888699889999999',
  '111122333144112233144222533142255533444456666775558869775888669778899669777889999',
  '111111122313122224333322244356664447356564647355566647558887777588889797889999999',
  '111112222311112242335555242367775544366777554336677754838666644838899994888899999',
  '111112222131114222333544442363545742363555747863565747866665777888699979888899999',
  '111111112331444442333454442333552222663555722666655777688858777688888977699999999',
  '111111122344441122344455522334456622733356662773358866775558886779988886779999999',
  '111112222314115622344145622344445662334555766833577776883579776883599796888899999',
  '111112222131112422335552462337754466377754446337754466837855566887899969888899999',
  '111111222113312222444333322445353366445555566447757566887777666888897799888999999',
  '111111222311412222344422555344444565366666665363777775333887775888897995888999999',
  '111111112314444222333354442663557742683555742683355722688857777666888897699999999',
  '111111222113313242553333244556633224576666624577886644557888844757898899777999999',
  '111122223111122223444152333445553363444753666747755566777859666788889999788889999',
  '111122223144155233441152236447152336477852336477859366478859966778559669788889999',
  '111112233412222235411162235447663335447666355477766355478869995478888895778899999',
  '111111233114222233414422533444422533465555537665887777665887797668888799668999999',
  '111111222341441122344444422333355562733555662735556666778888886779988986777999999',
  '111112222111122332444445322664555333644457773666555733886577777866889999888899999',
  '111111122133331224533633324556662224555762444577766644578886884577988889779999999',
  '111111122331444122344444222333554262335555566737855666777888886779888966779999999',
]);

const SLOW_INVALID_JIGSAW_LAYOUTS = withJigsawLayoutResult(null, [
  // From Mathimagics
  // http://forum.enjoysudoku.com/jigsaw-layouts-generate-test-t35712-30.html?sid=6523d33401485a4b2414c1d7cc15a060#p310543
  //
  // These take 1s-1m to invalidate.
  'AAAAAABBBCCADAEBBBCCCDAEFFBCCCDEEFFBGCDDEFFHBGDDEEFHHHGDDEIFHHHGGGEIFIHHGGGIIIIII',
  'AAAAAABBBCCADAEBFBCCCDAEBFBCCCDEEFFBGCDDEFFHBGDDEEFHHHGDGEIFHHHGDGEIFIHHGGGIIIIII',
  'AAAAABBBBAACDAEFBBCCCDAEFBBCCCDEEFFBCCDDEFFGGHDDEEFGGGHHDEIFGGGHHDEIFGIIHHHHIIIII',
  'AAAAABBBBAACDAEFFBCCCDAEFBBCCCDEEFBBCCDDEFFGGHHDEEFGGGHHDEIFGGGHDDEIFGIIHHHHIIIII',
  'AAAAABBBBACADAEBFBCCCDAEFFBCCCDEEFBBCCDDEFFGGHHDEEFGGGHDDEIFGGGHDHEIFIGIHHHHIIIII',
  'AAAAABBBBACADAEFBBCCCDAEFBBCCCDEEFFBCCDDEFFGGHDDEEFGGGHHDEIFGGGHHDEIFIGIHHHHIIIII',
  'AAAAABBBBACADAEFBBCCCDAEFFBCCCDEEFBBCCDDEFFGGHHDEEFGGGHDDEIFGGGHHDEIFIGIHHHHIIIII',
  'AAAAABBBBACADAEFFBCCCDAEFBBCCCDEEFBBCCDDEFFGGHHDEEFGGGHHDEIFGGGHDDEIFIGIHHHHIIIII',
  'AAAAABBBBACCAABBDBCCCADDDDBCDDDDEEEBCCCEEEFFFGEEEHHHHFGHHHHIFFFGHGGIIFFIGGGGIIIII',
  'AAAAABBBBACCAABDBBCCCADDDDBCDDDDEEEBCCCEEEFFFGEEEHHHHFGHHHHIFFFGGHGIIFFIGGGGIIIII',
  'AAAAABBBBACCAADBBBCCCADDDDBCDDDDEEEBCCCEEEFFFGEEEHHHHFGHHHHIFFFGGGHIIFFIGGGGIIIII',
  'AAAAABBBBACCAADDBBCCCADDDBBCDDDDEEEBCCCEEEFFFGEEEHHHHFGGHHHIFFFGGHHIIFFIGGGGIIIII',
  'AAAAABBBBACCAADDDBCCCADDBBBCDDDDEEEBCCCEEEFFFGEEEHHHHFGGGHHIFFFGHHHIIFFIGGGGIIIII',
  'AAAAABBBBACCDAEBBBACCDAEFFBCCCDEEFFBCCDDEFFGGHDDEEFGGGHDDEIFGGIHHHEIFGGIHHHHIIIII',
  'AAAAABBBBACCDAEBFBACCDAEBFBCCCDEEFFBCCDDEFFGGHDDEEFGGGHDHEIFGGIHDHEIFGGIHHHHIIIII',
  'AAAAABBBBACCDAEFBBACCDAEFBBCCCDEEFFBCCDDEFFGGHDDEEFGGGHHDEIFGGIHHDEIFGGIHHHHIIIII',
  'AAAAABBBBACCDAEFBBACCDAEFFBCCCDEEFBBCCDDEFFGGHHDEEFGGGHDDEIFGGIHHDEIFGGIHHHHIIIII',
  'AAAAABBBBACCDAEFFBACCDAEFBBCCCDEEFBBCCDDEFFGGHHDEEFGGGHHDEIFGGIHDDEIFGGIHHHHIIIII',
  'AAAAABBBBCAAAADBBBCCCDDDDDBCDDDEEEEBCCCCEFFFFGEEEEHHHFGHHHHHFFFGGGHIIIIFGGGGIIIII',
  'AAAAABBBBCAADAEBFBCCCDAEFFBCCCDEEFBBCCDDEFFGGHHDEEFGGGHDDEIFGGGHDHEIFIIGHHHHIIIII',
  'AAAAABBBBCAADAEFBBCCCDAEFBBCCCDEEFFBCCDDEFFGGHDDEEFGGGHHDEIFGGGHHDEIFIIGHHHHIIIII',
  'AAAAABBBBCAADAEFBBCCCDAEFFBCCCDEEFBBCCDDEFFGGHHDEEFGGGHDDEIFGGGHHDEIFIIGHHHHIIIII',
  'AAAAABBBBCAADAEFFBCCCDAEFBBCCCDEEFBBCCDDEFFGGHHDEEFGGGHHDEIFGGGHDDEIFIIGHHHHIIIII',
  'AAAAABBBBCACAABBDBCCCADDDDBCDDDDEEEBCCCEEEFFFGEEEHHHHFGHHHHIFFFGHGGIIFIFGGGGIIIII',
  'AAAAABBBBCACAADBBBCCCADDDDBCDDDDEEEBCCCEEEFFFGEEEHHHHFGHHHHIFFFGGGHIIFIFGGGGIIIII',
  'AAAAABBBBCACAADDBBCCCADDDBBCDDDDEEEBCCCEEEFFFGEEEHHHHFGGHHHIFFFGGHHIIFIFGGGGIIIII',
  'AAAAABBBBCACDAEBFBCCCDAEBFBCCDDAEFFBCCDEEEFGGHDDEIFFGGHDHEIFGGGHDHEIFGIGHHHHIIIII',
  'AAAAABBBBCACDAEFFBCACDAEFBBCCCDEEFBBCCDDEFFGGHHDEEFGGGHHDEIFGIGHDDEIFGIGHHHHIIIII',
  'AAAAABBBBCCAAABBDBCCCADDDDBCDDDDEEEBCCCEEEFFFGEEEHHHHFGHHHHIFFFGHGGIIIFFGGGGIIIII',
  'AAAAABBBBCCAAABDBBCCCADDDDBCDDDDEEEBCCCEEEFFFGEEEHHHHFGHHHHIFFFGGHGIIIFFGGGGIIIII',
  'AAAAABBBBCCAAABDDBCCCADDDBBCDDDDEEEBCCCEEEFFFGEEEHHHHFGGHHHIFFFGHHGIIIFFGGGGIIIII',
  'AAAAABBBBCCAAADBBBCCCADDDDBCDDDDEEEBCCCEEEFFFGEEEHHHHFGHHHHIFFFGGGHIIIFFGGGGIIIII',
  'AAAAABBBBCCAAADDBBCCCADDDBBCDDDDEEEBCCCEEEFFFGEEEHHHHFGGHHHIFFFGGHHIIIFFGGGGIIIII',
  'AAAAABBBBCCAAADDDBCCCADDBBBCDDDDEEEBCCCEEEFFFGEEEHHHHFGGGHHIFFFGHHHIIIFFGGGGIIIII',
  'AAAAABBBBCCADAEBBBCCADAEFFBCCCDEEFFBCCDDEFFGGHDDEEFGGGHDDEIFIGGHHHEIFIGGHHHHIIIII',
  'AAAAABBBBCCADAEBFBCCADAEBFBCCCDEEFFBCCDDEFFGGHDDEEFGGGHDHEIFIGGHDHEIFIGGHHHHIIIII',
  'AAAAABBBBCCADAEFBBCCADAEFBBCCCDEEFFBCCDDEFFGGHDDEEFGGGHHDEIFIGGHHDEIFIGGHHHHIIIII',
  'AAAAABBBBCCADAEFBBCCADAEFFBCCCDEEFBBCCDDEFFGGHHDEEFGGGHDDEIFIGGHHDEIFIGGHHHHIIIII',
  'AAAAABBBBCCADAEFFBCCADAEFBBCCCDEEFBBCCDDEFFGGHHDEEFGGGHHDEIFIGGHDDEIFIGGHHHHIIIII',
  'AAAAABBBBCCCAABBDBCCAADDDDBCDDDDEEEBCCCEEEFFFGEEEHHHHFGHHHHIIFFGHGGIIFFFGGGGIIIII',
  'AAAAABBBBCCCAABDBBCCAADDDDBCDDDDEEEBCCCEEEFFFGEEEHHHHFGHHHHIIFFGGHGIIFFFGGGGIIIII',
  'AAAAABBBBCCCAABDDBCCAADDDBBCDDDDEEEBCCCEEEFFFGEEEHHHHFGGHHHIIFFGHHGIIFFFGGGGIIIII',
  'AAAAABBBBCCCAADBBBCCAADDDDBCDDDDEEEBCCCEEEFFFGEEEHHHHFGHHHHIIFFGGGHIIFFFGGGGIIIII',
  'AAAAABBBBCCCAADDBBCCAADDDBBCDDDDEEEBCCCEEEFFFGEEEHHHHFGGHHHIIFFGGHHIIFFFGGGGIIIII',
  'AAAAABBBCDAABBBBBCDAADDDBCCDDDDEEEECFFFFECCCCFEEEEGGGGFFHGGGIIGFHHHHHIIGFHHHIIIII',
  'AAAAABCCCDAEFABCCCDAEFABGCCDDEFFBGGCDDEEFBBGGHDDEFFBGGHHDEIFBIGHHHEIFBIGHHHEIIIII'
]);

const OLD_VERY_SLOW_INVALID_JIGSAW_LAYOUTS = withJigsawLayoutResult(null, [
  // From Mathimagics
  // http://forum.enjoysudoku.com/jigsaw-layouts-generate-test-t35712-30.html?sid=6523d33401485a4b2414c1d7cc15a060#p310543
  //
  // These used to take over a 1 minute to invalidate, but are now fast.
  'AAAAAABCCADDEAFBCCDDDEAFBCCDDDEFFBCCDGEEFBBCHGGEFFBHHHGGEFIBHHHGGEFIBHHIGGEIIIIII',
  'AAAAAABCCDDDEAFBCCDDDEAFBCCDDEEAFBCCGDEFFFBHCGGEFIBBHHGGEFIBHHHGGEFIBHHHGGEIIIIII',
  'AAAAABBBBAAAABBBBBCCCCCCDDDEFFFFCCCDEEEEFDDDDEGGGFFFFDEEEGGGGGGHHHHHIIIIHHHHIIIII',
  'AAAAABBBBAAAABBBBBCCCCCDDDDEFFFCCCCDEEEFFFDDDEGGGGFFFDEEEEGGGGGHHHHHIIIIHHHHIIIII',
  'AAAAABBBBAAAABBBBBCCCCDDDDDCDDDDEEEFCCCEEEFFFCEEEGGGGFGGGGGFFFFHHHHHIIIIHHHHIIIII',
  'AAAAABBBBAAAABBBBCDDDDDBCCCEFFFDDDDCEEEFFFCCCEGGGGFFFCEEEHGGGGGEHHHHIIIIHHHHIIIII',
  'AAAAABBBBAAAABBBCCDDDDDBBCCEFFFDDDDCEEEFFFCCCEGGGGFFFCEEHHGGGGGEEHHHIIIIHHHHIIIII',
  'AAAAABBBBAAAABBCBBDDDDDBCCCEFFFDDDDCEEEFFFCCCEGGGGFFFCEEEHGGGGGHHEHHIIIIHHHHIIIII',
  'AAAAABBBBAACAABBBBDCCCCBEEEDFFFCCCCEDDDFFFEEEDGGGGFFFEDDDHGGGGEHHHHIIGIIHHHHIIIII',
  'AAAAABBBBAACAABBBBDCCCCCBEEDFFFFCCCEDDDDFEEEEDGGGFFFFEDDHGGGGGEHHHHIIGIIHHHHIIIII',
  'AAAAABBBBAACAABBDBECCCCBBDDEFFFCCCCDEEEFFFDDDEGGGGFFFDEEHHGGGGDHEHHIIGIIHHHHIIIII',
  'AAAAABBBBAACABBBBBCCCADDDDDCDDDDEEEFCCCEEEFFFCEEEGGGGFGGGGGHFFFIIIIIHFHHIIIIHHHHH',
  'AAAAABBBBACAAABBBBDCCCCBEEEDFFFCCCCEDDDFFFEEEDGGGGFFFEDDDHGGGGEHHHHIIIGIHHHHIIIII',
  'AAAAABBBBACAAABBBBDCCCCCBEEDFFFFCCCEDDDDFEEEEDGGGFFFFEDDHGGGGGEHHHHIIIGIHHHHIIIII',
  'AAAAABBBBACAAABBBDECCCCCBBDEFFFFCCCDEEEEFDDDDEGGGFFFFDEHHGGGGGDEHHHIIIGIHHHHIIIII',
  'AAAAABBBBACAABBBBBCCCADDDDDCDDDDEEEFCCCEEEFFFCEEEGGGGFGGGGGHFFFIIIIIHHFHIIIIHHHHH',
  'AAAAABBBBCAAAABBDBCCCCDDDDBCDDDDEEEBCCCEEEFFFGEEEHHHHFGHHHHFFFFGHGGIIIIFGGGGIIIII',
  'AAAAABBBBCAAABBBBBCCCADDDDDCDDDDEEEFCCCEEEFFFCEEEGGGGFGGGGGHFFFIIIIIHHHFIIIIHHHHH',
  'AAAAABBBBCAAABBDDBCCADDDDBBCDDDEEEEFCCCCEFFFFCEEEEGGGFHHGGGGIFFHGGHHIIIFHHHHIIIII',
  'AAAAABBBBCCAABBDBBCCAADDDDBCDDDDEEEFCCCEEEFFFCEEEGGGGFHGGGGIIFFHHGHHIIFFHHHHIIIII'
]);

