<?php
/**
 * IsBizMail tells you whether a given email address
 * is free e.g. gmail.com, yahoo.es, yandex.ru etc or not.
 * The list of emails used by IsBizMail is taken from here:
 * http://svn.apache.org/repos/asf/spamassassin/trunk/rules/20_freemail_domains.cf
 * All credits for the list itself go to SpamAssasin authors and contributors
 *
 * @category PHPUnit
 * @package  IsBizMail
 * @author   Zhmayev Yaroslav <salaros@salaros.com>
 * @license  MIT https://opensource.org/licenses/MIT
 * @link     https://github.com/salaros/free-mailchecker
 */

namespace Salaros\Email;

/**
 * Tells you whether a given email address is free  or not
 */
class IsBizMail
{
    private static $freeMailDomains;
    private static $freeMailPatterns;

    /**
     * Tells you if a given email is valid in terms of structure
     * and if it does not belong to one of known free mail box providers
     *
     * @param string $email Email address
     *
     * @return bool
     */
    public function isValid($email)
    {
        if (false === filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        return (isset($this) && $this instanceof self)
            ? !$this->isFreeMailAddress($email)
            : !self::isFreeMailAddress($email);
    }

    /**
     * Returns true if a given email belong to one of known free mail box providers
     *
     * @param string $email Email address
     *
     * @return bool
     */
    public function isFreeMailAddress($email)
    {
        $parts = explode("@", $email);
        $emailDomain = strtolower(end($parts));

        if (2 !== count($parts) || empty($emailDomain)) {
            throw new \InvalidArgumentException("You have supplied an invalid email address", 100);
        }

        if (false !== stripos($emailDomain, '*')) {
            throw new \InvalidArgumentException(
                sprintf(
                    "Domain part of the the following email contains a wildcard, which is not allowed: '%s'",
                    $email
                ),
                200
            );
        }

        // Check if the giveb domain is among known free providers
        if (in_array($emailDomain, self::getFreeDomains())) {
            return true;
        }

        // Check if patterns match the given domain
        foreach (self::getFreeDomainPatterns() as $pattern) {
            if (fnmatch($pattern, $emailDomain)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Gets the list of all known free mail box providers
     * @codeCoverageIgnore
     *
     * @return array
     */
    public static function getFreeDomains()
    {
        if (is_null(self::$freeMailDomains)) {
            self::init();
        }

        return self::$freeMailDomains;
    }

    /**
     * Gets the list of wildcards matching some free mail box providers
     * @codeCoverageIgnore
     *
     * @return array
     */
    public static function getFreeDomainPatterns()
    {
        if (is_null(self::$freeMailPatterns)) {
            self::init();
        }

        return self::$freeMailPatterns;
    }

    /**
     * Initializes the array of known free email domains
     * contains both full domains and widlcards
     *
     * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
     * @codeCoverageIgnore
     *
     * @return void
     */
    public static function init()
    {
        // phpcs:disable Generic.Files.LineLength
        self::$freeMailPatterns = array(
            // free email patterns start
            "aol.*", "aol.co*.*", "*.att.ne.jp", "excite.*", "excite.co*.*", "fastmail.*",
            "fastmail.co*.*", "freemail.*", "freemail.*.*", "gmx.*", "hotmail.*", "hotmail.co*.*",
            "live.*", "lycos.*", "lycos.co*.*", "mail2*.com", "ms*.hinet.net", "outlook.*",
            "strompost.*", "tiscali.*", "tiscali.co*.*", "vodafone.*", "xemail.*", "yahoo.*",
            "yahoo.co*.*", "yandex.*", "runbox.*", "*.onmicrosoft.com",
            // free email patterns end
        );

        self::$freeMailDomains = array(
            // free email providers start
            "020.co.uk", "027168.com", "0815.ru", "0815.ru0clickemail.com", "0815.ry", "0815.su",
            "0845.ru", "0clickemail.com", "0-mail.com", "0wnd.net", "0wnd.org", "0x207.info",
            "100likers.com", "10mail.com", "10mail.org", "10minut.com.pl", "10minutemail.cf", "10minutemail.com",
            "10minutemail.co.uk", "10minutemail.co.za", "10minutemail.de", "10minutemail.ga", "10minutemail.gq", "10minutemail.ml",
            "10minutemail.net", "10minutesmail.com", "10x9.com", "111mail.com", "123box.net", "123.com",
            "123india.com", "123iran.com", "123mail.cl", "123mail.org", "123-m.com", "123qwe.co.uk",
            "126.com", "12hosting.net", "12houremail.com", "12minutemail.com", "12minutemail.net", "12storage.com",
            "138mail.com", "139.com", "140unichars.com", "141.ro", "147.cl", "14n.co.uk",
            "150mail.com", "150ml.com", "163.com", "16mail.com", "188.com", "189.cn",
            "1-8.biz", "1963chevrolet.com", "1963pontiac.com", "1ce.us", "1chuan.com", "1fsdfdsfsdf.tk",
            "1mail.ml", "1netdrive.com", "1pad.de", "1st-forms.com", "1stpd.net", "1st-website.com",
            "1to1mail.org", "1zhuan.com", "20after4.com", "20email.eu", "20email.it", "20mail.eu",
            "20mail.in", "20mail.it", "20minute.email", "20minutemail.com", "2120001.net", "21cn.com",
            "24h.co.jp", "24horas.com", "24hourmail.com", "24hourmail.net", "263.net", "271soundview.com",
            "2die4.com", "2emea.com", "2ether.net", "2fdgdfgdfgdf.tk", "2-mail.com", "2mydns.com",
            "2net.us", "2prong.com", "3000.it", "30minutemail.com", "30wave.com", "33mail.com",
            "36ru.com", "37.com", "3ammagazine.com", "3d-painting.com", "3email.com", "3l6.com",
            "3mail.ga", "3trtretgfrfe.tk", "3xl.net", "418.dk", "420blaze.it", "420email.com",
            "42o.org", "444.net", "4degreez.com", "4email.com", "4email.net", "4gfdsgfdgfd.tk",
            "4mail.cf", "4mail.ga", "4-music-today.com", "4newyork.com", "4-n.us", "4warding.com",
            "4warding.net", "4warding.org", "4xn.de", "5005.lv", "50mail.com", "55mail.cc",
            "5ballov.ru", "", "5fm.za.com", "5ghgfhfghfgh.tk", "5gramos.com", "5mail.cf",
            "5mail.ga", "5oz.ru", "5x25.com", "5x2.de", "5x2.me", "60minutemail.com",
            "6210.hu", "672643.net", "675hosting.com", "675hosting.net", "675hosting.org", "6hjgjhgkilkj.tk",
            "6ip.us", "6mail.cf", "6mail.ga", "6mail.ml", "6paq.com", "6sens.com",
            "6url.com", "702mail.co.za", "7110.hu", "75hosting.com", "75hosting.net", "75hosting.org",
            "7days-printing.com", "7mail.ga", "7mail.ml", "7tags.com", "80665.com", "8127ep.com",
            "8848.net", "8chan.co", "8mail.cf", "8mail.ga", "8mail.ml", "8m.com",
            "8m.net", "8u8.com", "8u8.hk", "8u8.tw", "8x.com.br", "99experts.com",
            "99pubblicita.com", "9.cn", "9mail.cf", "9me.site", "9ox.net", "a45.in",
            "abakiss.com", "abcmail.email", "a-bc.net", "abha.cc", "about.com", "abusemail.de",
            "abuser.eu", "abv.bg", "abyssmail.com", "ac20mail.in", "academiccommunity.com", "acatperson.com",
            "acceso.or.cr", "access4less.net", "accessgcc.com", "accountant.com", "acdcfan.com", "acentri.com",
            "acmemail.net", "actingbiz.com", "activist.com", "add3000.pp.ua", "adexec.com", "adiga.com",
            "adinet.com.uy", "adiq.eu", "adobeccepdm.com", "adogperson.com", "adpugh.org", "adres.nl",
            "adrianou.gq", "adsd.org", "advalvas.be", "advantimo.com", "adwaterandstir.com", "aegia.net",
            "aegiscorp.net", "aeiou.pt", "aelo.es", "aeneasmail.com", "aeonpsi.com", "aeterna.ru",
            "", "africamail.com", "afrik.com", "afrobacon.com", "afropoets.com", "agadir.cc",
            "agedmail.com", "agger.ro", "aggies.com", "agtx.net", "ag.us.to", "ahaa.dk",
            "ahk.jp", "ahsa.ws", "aichi.com", "aim.com", "aircraftmail.com", "airpost.net",
            "airsi.de", "airsport.top", "aiutamici.com", "ajaxapp.net", "ajman.cc", "ajman.us",
            "ajman.ws", "akapost.com", "akerd.com", "aklan.com", "aknet.kg", "aktiefmail.nl",
            "alabama.usa.com", "alaska.usa.com", "alavatotal.com", "albafind.com", "albaha.cc", "albawaba.com",
            "alburaq.net", "aldeax.com", "aldeax.com.ar", "alex4all.com", "alexandria.cc", "algeria.com",
            "algerie.cc", "alice.it", "aligamel.com", "alisongamel.com", "alivance.com", "aliyun.com",
            "all4theskins.com", "alldirectbuy.com", "allergist.com", "allhiphop.com", "allmail.net", "allowed.org",
            "allsportsrock.com", "allthegoodnamesaretaken.org", "alph.wtf", "al-qaeda.us", "alriyadh.cc", "alskens.dk",
            "altavista.se", "altbox.org", "alternativagratis.com", "alum.com", "alumni.com", "alumnidirector.com",
            "alunos.unipar.br", "alvilag.hu", "alwaysgrilling.com", "alwaysinthekitchen.com", "alwayswatchingmovies.com", "alwayswatchingtv.com",
            "amail4.me", "a.mailcker.com", "amail.com", "ama-trade.de", "ama-trans.de", "amazon-aws.org",
            "amelabs.com", "amenworld.com", "america.hm", "americamail.com", "amilegit.com", "amiriindustries.com",
            "amiri.net", "amman.cc", "amnetsal.com", "amorous.com", "ampsylike.com", "ananzi.co.za",
            "anappfor.com", "anappthat.com", "anatomicrock.com", "andthen.us", "anet.ne.jp", "anfmail.com",
            "angelfire.com", "angelic.com", "animail.net", "animeone.com", "animesos.com", "anit.ro",
            "aniverse.com", "anjungcafe.com", "ano-mail.net", "anonbox.net", "anon-mail.de", "anonmails.de",
            "anonymail.dk", "anonymbox.com", "anonymized.org", "anonymousness.com", "another.com", "ansibleemail.com",
            "antedoonsub.com", "anthony-junkmail.com", "antichef.com", "antichef.net", "antireg.com", "antireg.ru",
            "antispam24.de", "antispam.de", "antispammail.de", "antwerpen.com", "anunciador.net", "anytimenow.com",
            "aon.at", "a.org.ua", "apexmail.com", "apfelkorps.de", "aphlog.com", "apkmd.com",
            "apollo.lv", "aport2000.ru", "", "aport.ru", "", "appc.se",
            "appinventor.nl", "appixie.com", "appraiser.net", "approvers.net", "apps.dj", "aprava.com",
            "apropo.ro", "aqaba.cc", "arab.ir", "arar.ws", "archaeologist.com", "arcor.de",
            "arcticmail.com", "arduino.hk", "argentina.com", "arizona.usa.com", "arkansas.usa.com", "armmail.com",
            "army.com", "armyspy.com", "arnet.com.ar", "aroma.com", "aron.us", "arrl.net",
            "arroisijewellery.com", "artlover.com", "artman-conception.com", "aruba.it", "arurgitu.gq", "arvato-community.de",
            "aschenbrandt.net", "asdasd.nl", "asdasd.ru", "asheville.com", "ashleyandrew.com", "asia.com",
            "asia-links.com", "asia-mail.com", "asiamail.com", "asiancutes.com", "assala.com", "assamesemail.com",
            "ass.pp.ua", "astroempires.info", "asu.mx", "asurfer.com", "asu.su", "aswan.cc",
            "asylum.com", "at0mik.org", "atech5.com", "a-teens.net", "atheist.com", "atlas.cz",
            "atlas.sk", "atl.lv", "a-topmail.at", "atozasia.com", "atreillou.com", "atrus.ru",
            "", "att.net", "atvclub.msk.ru", "a.ua", "", "aubenin.com",
            "auctioneer.net", "aufdrogen.de", "auf-steroide.de", "augmentationtechnology.com", "au.ru", "aus-city.com",
            "ausi.com", "aussiemail.com.au", "australiamail.com", "auti.st", "autoindia.com", "autopm.com",
            "autorobotica.com", "autotwollow.com", "avasmail.com.mv", "aver.com", "avls.pt", "avpa.nl",
            "a.vztc.com", "awatum.de", "awiki.org", "axarnet.com", "axiz.org", "ayna.com",
            "azcomputerworks.com", "azet.sk", "azmeil.tk", "b1of96u.com", "b2cmail.de", "baalbeck.cc",
            "babbalu.com", "badgerland.eu", "badgers.com", "badoop.com", "bahraini.cc", "bakpaka.com",
            "bakpaka.net", "balochistan.org", "baluch.com", "bama-fan.com", "bancora.net", "banha.cc",
            "bankersmail.com", "bareed.ws", "barlick.net", "barriolife.com", "barryogorman.com", "bartdevos.be",
            "bartender.net", "basketball-email.com", "basscode.org", "bauwerke-online.com", "baxomale.ht.cx", "bazaaboom.com",
            "b-boy.com", "bcast.ws", "bcb.ro", "bccto.me", "beabookworm.com", "beagolfer.com",
            "beahealthnut.com", "bearsarefuzzy.com", "beautifulboy.com", "beddly.com", "beeebank.com", "beefmilk.com",
            "beehive.org", "been-there.com", "begoz.com", "beirut.com", "believeinliberty.com", "belizehome.com",
            "belizemail.net", "belizeweb.com", "bellair.net", "belljonestax.com", "bellsouth.net", "benipaula.org",
            "berlin.com", "berlin.de", "besser-als-du.de", "bestchoiceusedcar.com", "bestcoolcars.com", "bestjobcandidate.com",
            "bestmail.us", "besure2vote.com", "betr.co", "bflomail.com", "bgay.com", "bgnmail.com",
            "bgx.ro", "bharatmail.com", "bicycledata.com", "bicycling.com", "bidourlnks.com", "big1.us",
            "bigboss.cz", "bigfoot.com", "bigger.com", "bigheavyworld.com", "bigmailbox.com", "bigmailbox.net",
            "bigmir.net", "big-orange.com", "bigpond.com", "bigprofessor.so", "bigstring.com", "bigtimecatperson.com",
            "bigtimedogperson.com", "bigtimereader.com", "bigtimesportsfan.com", "bigwhoop.co.za", "bij.pl", "bikerheaven.net",
            "bikerider.com", "bikermail.com", "billssite.com", "binkmail.com", "bio-muesli.info", "bio-muesli.net",
            "bip.net", "birdlover.com", "bitwiser.com", "biz.by", "bizerte.cc", "bizhosting.com",
            "bk.ru", "", "blackandchristian.com", "blackburnmail.com", "blackcity.net", "blackglobalnetwork.net",
            "blackmarket.to", "black-sea.ro", "blackvault.com", "blackvoices.com", "blader.com", "bladesmail.net",
            "blida.info", "blink182.net", "blip.ch", "blogmyway.org", "bluebottle.com", "blue.devils.com",
            "bluedumpling.info", "bluemail.ch", "bluewerks.com", "blumail.org", "blvds.com", "bmx.lv",
            "bmxtrix.com", "boardermail.com", "boarderzone.com", "boatnerd.com", "bobmail.info", "bobmurchison.com",
            "bodhi.lawlita.com", "bofthew.com", "bolando.com", "bolbox.com", "bol.com.br", "bollywood2000.com",
            "bollywoodz.com", "bombka.dyn.pl", "bonbon.net", "bongmail.com", "bonobo.email", "bookthemmore.com",
            "boom.com", "bootmail.com", "bootybay.de", "borda.ru", "", "borged.com",
            "borged.net", "borged.org", "bostonoffice.com", "bot.nu", "boun.cr", "bouncr.com",
            "bowl.com", "box.az", "boxbg.com", "boxemail.com", "boxformail.in", "boximail.com",
            "boxtemp.com.br", "brain.com.pk", "brainsurfer.de", "brandallday.net", "brasilia.net", "brasx.org",
            "bravanese.com", "brazilmail.com", "brazilmail.com.br", "breakthru.com", "breathe.com", "brefmail.com",
            "brennendesreich.de", "brestonline.com", "brew-master.com", "brew-meister.com", "brfree.com.br", "briggsmarcus.com",
            "britishintelligence.co.uk", "br.mintemail.com", "broadbandninja.com", "brujula.net", "bsdmail.com", "bsnow.net",
            "bspamfree.org", "bspooky.com", "bst-72.com", "btb-notes.com", "btcc.org", "btc.email",
            "btizet.pl", "buffaloes.com", "buffemail.com", "bugmenever.com", "bugmenot.com", "bulgaria.com",
            "bulldogs.com", "bulrushpress.com", "bumerang.ro", "bu.mintemail.com", "bum.net", "bumpymail.com",
            "bunchofidiots.com", "bundes-li.ga", "bund.us", "bunsenhoneydew.com", "buraydah.cc", "burnthespam.info",
            "burntmail.com", "burstmail.info", "businessbackend.com", "businesssuccessislifesuccess.com", "buspad.org", "butch-femme.net",
            "butch-femme.org", "buymoreplays.com", "buyordie.info", "buyusedlibrarybooks.org", "buzy.com", "buzzjakkerz.com",
            "byebyemail.com", "byespm.com", "byke.com", "byom.de", "c3.hu", "c4.com",
            "c51vsgq.com", "cachedot.net", "cadinfo.net", "calcfacil.com.br", "calcware.org", "californiafitnessdeals.com",
            "californiamail.com", "california.usa.com", "calle22.com", "callnetuk.com", "cam4you.cc", "camaroclubsweden.com",
            "cameroon.cc", "canada-11.com", "canada.com", "canal21.com", "cane.pw", "cannabismail.com",
            "canoemail.com", "capsfanatic.com", "capshockeyfan.com", "capsred.com", "car101.pro", "caramail.com",
            "cardblvd.com", "card.zp.ua", "care2.com", "care-mail.com", "caress.com", "carioca.net",
            "car-nut.net", "caseedu.tk", "cash4u.com", "cashette.com", "casino.com", "casinomail.com",
            "casualdx.com", "cataloniamail.com", "catalunyamail.com", "cataz.com", "catcha.com", "catholic.org",
            "caths.co.uk", "catlover.com", "catlovers.com", "catpeoplerule.com", "cat-person.com", "cavi.mx",
            "caxess.net", "cbair.com", "c-box.cz", "cbrmail.com", "cc.liamria", "cc.lv",
            "cdfaq.com", "cdpa.cc", "ceed.se", "cek.pm", "cellurl.com", "cemelli.com",
            "ce.mintemail.com", "centermail.com", "centermail.net", "centoper.it", "centralpets.com", "centrum.cz",
            "centrum.sk", "centurylink.net", "cercaziende.it", "certifiedbitches.com", "cetpass.com", "cgac.es",
            "chacuo.net", "chaiyo.com", "chaiyomail.com", "chammy.info", "championboxing.com", "chance2mail.com",
            "channelonetv.com", "charter.net", "chattown.com", "chatway.com", "chat-with-me.com", "cheaphub.net",
            "cheatasrule.com", "cheatmail.de", "checkitmail.at", "cheerful.com", "chef.net", "chelny.com",
            "chemist.com", "cheshiremail.com", "chewiemail.com", "chickenkiller.com", "chielo.com", "childsavetrust.org",
            "chil-e.com", "chilkat.com", "chillaxer.de", "chillimail.com", "chillymail.com", "china.com",
            "chinamail.com", "chithinh.com", "chogmail.com", "choicemail1.com", "chong-mail.com", "chong-mail.net",
            "chong-mail.org", "christianmail.org", "ch.tc", "chumpstakingdumps.com", "ciaoweb.it", "cigar-auctions.com",
            "cine.com", "ciphercom.net", "circlemail.com", "cititrustbank1.cjb.net", "citromail.hu", "citynetusa.com",
            "ciudad.com.ar", "civx.org", "ckiso.com", "cl0ne.net", "clandest.in", "claramail.com",
            "classicmail.co.za", "classprod.com", "classycouples.com", "cl-cl.org", "clerk.com", "cliffhanger.com",
            "clipmail.eu", "clix.pt", "clixser.com", "close2you.net", "cloud.me", "",
            "clovermail.net", "clrmail.com", "clubmember.org", "cluemail.com", "clujnapoca.ro", "cmail.com",
            "cmail.net", "cmail.org", "cnamed.com", "cnew.ir", "cnmsg.net", "cnsds.de",
            "cobarekyo1.ml", "cock.li", "codeandscotch.com", "codivide.com", "coieo.com", "coldemail.info",
            "collector.org", "collegeclub.com", "colombia.com", "colorado.usa.com", "columnist.com", "com.ar",
            "combcub.com", "comcast.net", "comfortable.com", "comic.com", "company.org.ua", "compaqnet.fr",
            "compareshippingrates.org", "completegolfswing.com", "compuserve.com", "computer4u.com", "computermail.net", "computer.net",
            "computhouse.com", "comwest.de", "conevyt.org.mx", "congiu.net", "connect4free.net", "connecticut.usa.com",
            "consultant.com", "consumerriot.com", "contractor.net", "coolandwacky.us", "cool.fr.nf", "coolgoose.com",
            "coolimpool.org", "coolkiwi.com", "coollist.com", "coolmail.com", "coolmail.net", "coolmail.ru",
            "coolsend.com", "coolshit.com", "coolsite.net", "cooltoad.com", "cooperation.net", "copacabana.com",
            "copticmail.com", "coreclip.com", "corporateattorneys.com", "corporation.net", "corpusmail.com", "correios.net.br",
            "correo.blogos.net", "correomagico.com", "cosmo.com", "cosmorph.com", "cosmosurf.net", "cougars.com",
            "counsellor.com", "count.com", "countrybass.com", "couple.com", "courriel.fr.nf", "courrieltemporaire.com",
            "coxinet.net", "coza.ro", "crankhole.com", "crankmails.com", "crapmail.org", "crastination.de",
            "crazespaces.pw", "crazy4baseball.com", "crazy4homeimprovement.com", "crazy4mail.com", "crazyaboutfilms.net", "crazycarfan.com",
            "crazyforemail.com", "crazymailing.com", "crazymoviefan.com", "creazionisa.com", "criticalpath.net", "critterpost.com",
            "crosspaths.net", "crossroadsmail.com", "crosswinds.net", "crotslep.tk", "crusthost.com", "cryingmail.com",
            "cs.com", "csh.ro", "csucsposta.hu", "cszbl.com", "ctos.ch", "cubiclink.com",
            "cu.cc", "cumallover.me", "cumbriamail.com", "cuoly.com", "curio-city.com", "curryworld.de",
            "cust.in", "custmail.com", "cutey.com", "cuvox.de", "cwazy.co.uk", "cwazy.net",
            "cww.de", "cyberaccess.com.pk", "cyberdude.com", "cybergal.com", "cybergirls.dk", "cyberguys.dk",
            "cyberkriminell.de", "cybernet.it", "cyberservices.com", "cyberunlimited.org", "cyber-wizard.com", "cycledata.com",
            "cylab.org", "cymail.net", "d1yun.com", "d3p.dk", "dab.ro", "dabsol.net",
            "dacoolest.com", "dada.net", "dadanet.it", "daemsteam.com", "dailypioneer.com", "daintly.com",
            "dallasmail.com", "dammexe.net", "damuc.org.br", "dandikmail.com", "danneben.so", "dansegulvet.com",
            "darkfear.com", "darkforces.com", "darkharvestfilms.com", "darkhorsefan.net", "daryxfox.net", "dash-pads.com",
            "data54.com", "dataarca.com", "datafilehost", "datarca.com", "datazo.ca", "daum.net",
            "davegracey.com", "davidkoh.net", "davidlcreative.com", "dayrep.com", "dayzers.com", "dbmail.com",
            "dbunker.com", "dbzmail.com", "dcemail.com", "dcsi.net", "ddcrew.com", "deacons.com",
            "deadaddress.com", "deadchildren.org", "deadfake.cf", "deadfake.ga", "deadfake.ml", "deadfake.tk",
            "deadlymob.org", "deadspam.com", "deagot.com", "dealja.com", "deal-maker.com", "dealrek.com",
            "de-a.org", "dearriba.com", "deekayen.us", "defomail.com", "degoo.com", "degradedfun.net",
            "delajaonline.org", "delaware.usa.com", "delayload.com", "delayload.net", "delfi.lv", "delhimail.com",
            "delikkt.de", "deliveryman.com", "demon.deacons.com", "denmark.ir", "der-kombi.de", "derkombi.de",
            "derluxuswagen.de", "descriptivemail.com", "desertonline.com", "desidrivers.com", "deskpilot.com", "despam.it",
            "despammed.com", "detik.com", "devils.com", "devnullmail.com", "dexara.net", "deyom.com",
            "dfgh.net", "dhahran.cc", "dharmatel.net", "dhmail.net", "dhm.ro", "dhofar.cc",
            "dialogus.com", "diapaulpainting.com", "dicksinhisan.us", "dicksinmyan.us", "didamail.com", "differentmail.com",
            "digitalmariachis.com", "digitalsanctuary.com", "digitaltrue.com", "dildosfromspace.com", "dingbone.com", "dino.lv",
            "diplomats.com", "direccion.com", "director-general.com", "diri.com", "dirtythird.com", "discard.cf",
            "discard.email", "discard.ga", "discard.gq", "discardmail.com", "discardmail.de", "discard.ml",
            "discard.tk", "disciples.com", "discofan.com", "discos4.com", "discoverymail.net", "disinfo.net",
            "dispo.in", "dispomail.eu", "disposableaddress.com", "disposable.cf", "disposable.com", "disposableemailaddresses.com",
            "disposable-email.ml", "disposable.ga", "disposableinbox.com", "disposable.ml", "disposeamail.com", "dispose.it",
            "disposemail.com", "dispostable.com", "di-ve.com", "divermail.com", "divismail.ru", "djibouti.cc",
            "djmillenium.com", "dlemail.ru", "dmailman.com", "dm.w3internet.co.uk", "dndent.com", "dnses.ro",
            "dnsmadeeasy.com", "dob.jp", "doctor.com", "dodgeit.com", "dodgemail.de", "dodgit.com",
            "dodgit.org", "dodsi.com", "doglover.com", "dogmail.co.uk", "dogpeoplerule.com", "dog-person.com",
            "doiea.com", "doityourself.com", "dolphinnet.net", "domaindiscover.com", "domainmanager.com", "domforfb18.tk",
            "domforfb19.tk", "domforfb1.tk", "domforfb23.tk", "domforfb27.tk", "domforfb29.tk", "domforfb2.tk",
            "domforfb3.tk", "domforfb4.tk", "domforfb5.tk", "domforfb6.tk", "domforfb7.tk", "domforfb8.tk",
            "domforfb9.tk", "dominican.cc", "domozmail.com", "doneasy.com", "donemail.ru", "do.net.ar",
            "dontexist.org", "dontreg.com", "dontsendmespam.de", "dopefiends.com", "doquier.tk", "doramail.com",
            "dores.com", "dostmail.com", "dot5hosting.com", "dotcom.fr", "dotman.de", "dotmsg.com",
            "dotnow.com", "dotslashrage.com", "dott.it", "doubt.com", "douchelounge.com", "dozvon-spb.ru",
            "dp76.com", "dplanet.ch", "dr69.site", "draac.com", "dragoncon.net", "dragonfans.com",
            "drakmail.net", "dr.com", "drdrb.com", "drdrb.net", "dr-dre.com", "dreamstop.com",
            "dred.ru", "drevo.si", "drivetagdev.com", "droolingfanboy.de", "dropcake.de", "droplar.com",
            "dropmail.me", "dropzone.com", "dserver.org", "dsiay.com", "dspwebservices.com", "duam.net",
            "dubaiwebcity.com", "dublin.com", "dublin.ie", "dudmail.com", "duk33.com", "dukedish.com",
            "dumoac.net", "dumpandjunk.com", "dump-email.info", "dumpmail.de", "dumpyemail.com", "durandinterstellar.com",
            "duskmail.com", "dustdevil.com", "dutchmail.com", "dyceroprojects.com", "dynamitemail.com", "dyndns.org",
            "dz17.net", "e3z.de", "e4ward.com", "e-apollo.lv", "earthalliance.com", "earthling.net",
            "eastmail.com", "eastrolog.com", "easy.com", "easydoesit.com", "easyinfomail.co.za", "easy-pages.com",
            "easypeasy.com", "easy-trash-mail.com", "easytrashmail.com", "ebeschlussbuch.de", "ecallheandi.com", "echina.com",
            "eclub.lv", "ecn.org", "ecplaza.net", "edgex.ru", "edinburgh-airporthotels.com", "edsamail.com.ph",
            "educacao.te.pt", "edumail.co.za", "edu.my", "edu.sg", "edv.to", "ee1.pl",
            "ee2.pl", "eeism.com", "eelmail.com", "efxs.ca", "ego.co.th", "egypt.ir",
            "egypt.net", "e-hkma.com", "einmalmail.de", "einrot.com", "einrot.de", "eintagsmail.de",
            "eircom.net", "ekolay.net", "elearningjournal.org", "electro.mn", "elforotv.com.ar", "elitemail.org",
            "elitevipatlantamodels.com", "elsitio.com", "eltimon.com", "elvis.com", "elvisfan.com", "email2me.com",
            "email60.com", "emailacc.com", "emailaccount.com", "emailaddresses.com", "emailage.cf", "emailage.ga",
            "emailage.gq", "emailage.ml", "emailage.tk", "e-mail.am", "e-mailanywhere.com", "email.bg",
            "e-mailbox.ga", "email.cbes.net", "emailchoice.com", "e-mail.com", "email.com", "email.com.br",
            "emailcorner.net", "e-mail.cz", "email.cz", "emaildienst.de", "emailengine.net", "emailengine.org",
            "email-fake.cf", "email-fake.ga", "email-fake.gq", "email-fake.ml", "email-fake.tk", "emailfast.com",
            "emailgaul.com", "emailgo.de", "emailgroups.net", "emailhut.net", "emailias.com", "emailigo.de",
            "emailinfive.com", "emailisvalid.com", "email.it", "email-jetable.fr", "emaillime.com", "email.lu",
            "email.lviv.ua", "emailmiser.com", "emailn.de", "email.net", "email.nu", "e-mail.org",
            "e-mail.ph", "emailpinoy.com", "emailplanet.com", "emailplus.org", "emailproxsy.com", "emailresort.com",
            "email.ro", "email.ru", "", "emailsensei.com", "emails.ga", "email.si",
            "emailsingularity.net", "emailspam.cf", "emailspam.ga", "emailspam.gq", "emailspam.ml", "emailspam.tk",
            "emailtemporanea.com", "emailtemporanea.net", "emailtemporario.com.br", "emailtemporar.ro", "emailthe.net", "emailtmp.com",
            "emailto.de", "emailure.net", "emailuser.net", "emailwarden.com", "emailx.at.hm", "emailxfer.com",
            "emailz.cf", "emailz.ga", "emailz.gq", "emailz.ml", "ematic.com", "embarqmail.com",
            "embroideryforums.com", "emeil.in", "emeil.ir", "emeraldwebmail.com", "emil.com", "e-milio.com",
            "emkei.cf", "emkei.ga", "emkei.gq", "emkei.ml", "emkei.tk", "eml.cc",
            "eml.pp.ua", "emltmp.com", "emoka.ro", "emptymail.com", "emz.net", "enel.net",
            "enelpunto.net", "engineer.com", "england.com", "englandmail.com", "enterate.com.ar", "enterto.com",
            "entryweb.it", "entusiastisk.com", "enusmail.com", "envirocitizen.com", "epatra.com", "epb.ro",
            "ephemail.net", "ephemeral.email", "epix.net", "epomail.com", "epost.de", "eprompter.com",
            "eqqu.com", "eramail.co.za", "eresmas.com", "ericjohnson.ml", "eriga.lv", "eritrea.cc",
            "ero-tube.org", "ertelecom.ru", "escapeartist.com", "escapehatchapp.com", "esc.la", "esde-s.org",
            "esemay.com", "esfera.cl", "esgeneri.com", "esprity.com", "estadao.com.br", "e-tapaal.com",
            "etllao.com", "etranquil.com", "etranquil.net", "etranquil.org", "euaqa.com", "euromail.net",
            "europe.com", "europemail.com", "euroseek.com", "euskalmail.com", "evafan.com", "evanfox.info",
            "everyday.com.kh", "everymail.net", "everyone.net", "evopo.com", "evyush.com", "e-webtec.com",
            "example.com", "execs2k.com", "execs.com", "executivemail.co.za", "exitstageleft.net", "expertrenovator.com",
            "explodemail.com", "expn.com", "expressivemail.com", "expressmail.dk", "express.net.ua", "extremail.ru",
            "eyepaste.com", "ezfill.com", "ezilon.com", "ez.lv", "ezrs.com", "ezstest.com",
            "ezsweeps.com", "f4k.es", "facebook-email.cf", "facebook-email.ga", "facebook-email.ml", "facebookmail.gq",
            "facebookmail.ml", "facilmail.com", "fadingemail.com", "fadrasha.net", "fadrasha.org", "fag.wf",
            "failbone.com", "faithhighway.com", "faithkills.com", "faithmail.com", "fake-box.com", "fakedemail.com",
            "fake-email.pp.ua", "fakeinbox.cf", "fakeinbox.com", "fakeinbox.ga", "fakeinbox.info", "fakeinbox.ml",
            "fakeinbox.tk", "fakeinformation.com", "fake-mail.cf", "fakemail.fr", "fake-mail.ga", "fakemailgenerator.com",
            "fake-mail.ml", "fakemailz.com", "falasteen.cc", "familymailbox.com", "familyroll.com", "familysafeweb.net",
            "fammix.com", "famous.as", "fanaticos.com", "fan.com", "fangoh.com", "fan.net",
            "fanofbooks.com", "fanofcomputers.com", "fanofcooking.com", "fanoftheweb.com", "fansworldwide.de", "fantasymail.de",
            "faroweb.com", "farrse.co.uk", "farts.com", "fastacura.com", "fastchevy.com", "fastchrysler.com",
            "fast-email.com", "fastemailer.com", "fastemail.us", "fastem.com", "fastermail.com", "fasternet.biz",
            "fastest.cc", "fastimap.com", "fastkawasaki.com", "fastmailbox.net", "fastmail.fm", "fast-mail.org",
            "fastmazda.com", "fastmessaging.com", "fastmitsubishi.com", "fastnissan.com", "fastservice.com", "fastsubaru.com",
            "fastsuzuki.com", "fasttoyota.com", "fastwebmail.it", "fastyamaha.com", "fatflap.com", "fawz.net",
            "fax.ru", "", "fdfdsfds.com", "fea.st", "federalcontractors.com", "fedxmail.com",
            "feelingnaughty.com", "feelings.com", "female.ru", "fepg.net", "fer-gabon.org", "fettometern.com",
            "ffanet.com", "fiberia.com", "fictionsite.com", "fieldmail.com", "fightallspam.com", "figjs.com",
            "figshot.com", "fiifke.de", "filbert4u.com", "filberts4u.com", "filipinolinks.com", "film-blog.biz",
            "filzmail.com", "financesource.com", "financier.com", "findmail.com", "findu.pl", "fireman.net",
            "firemyst.com", "fir.hk", "fiscal.net", "fit.lv", "fivemail.de", "fixmail.tk",
            "fizmail.com", "flashmail.com", "fleckens.hu", "fleetmail.com", "flemail.ru", "flipcode.com",
            "floridagators.com", "florida.usa.com", "flowu.com", "flurred.com", "flyinggeek.net", "flyspam.com",
            "fly-ts.de", "fmailbox.com", "fmail.co.uk", "fm.com.ua", "", "f-m.fm",
            "fmgirl.com", "fmguy.com", "fnmail.com", "focusedonprofits.com", "focusedonreturns.com", "foobarbot.net",
            "footard.com", "footballer.com", "forecastertests.com", "forfree.at", "forgetmail.com", "fornow.eu",
            "forspam.net", "forsythmissouri.org", "fortuncity.com", "forum.dk", "fotoplenka.ru", "",
            "foxja.com", "foxmail.com", "foxtrotter.info", "fr33mail.info", "frapmail.com", "freeaccess.nl",
            "freealtgen.com", "freebabysittercam.com", "freeblackbootytube.com", "freecat.net", "free.com.pe", "freedompop.us",
            "free-email.cf", "free-email.ga", "freefattymovies.com", "free.fr", "freegates.be", "freeghana.com",
            "freehosting.nl", "freei.co.th", "freeler.nl", "freeletter.me", "freemailen.de", "freemail.globalsite.com.br",
            "freemail.ms", "freemailn.de", "freemails.cf", "freemails.ga", "freemails.ml", "freemuslim.net",
            "freenet.de", "freenet.kg", "free.net.nz", "freeola.net", "freeonline.com", "freepgs.com",
            "freeplumpervideos.com", "freesbee.fr", "freeschoolgirlvids.com", "freeservers.com", "freesistercam.com", "freestart.hu",
            "freesurf.ch", "freesurf.fr", "freesurf.nl", "freeteenbums.com", "freeuk.com", "freeuk.net",
            "freewebemail.com", "freeweb.it", "freeyellow.com", "freundin.ru", "friendlymail.co.uk", "frisurf.no",
            "fr.nf", "fromru.com", "", "front14.org", "frontiernet.net", "front.ru",
            "", "fsmail.net", "fsnet.co.uk", "ftml.net", "ftpinc.ca", "ftp.sh",
            "fuckedupload.com", "fuckingduh.com", "fudge.com", "fudgerub.com", "fuelie.org", "fuirio.com",
            "fujairah.cc", "fujairah.us", "fujairah.ws", "fulvie.com", "fun.21cn.com", "fun64.com",
            "fun-greetings-jokes.com", "funkytimes.com", "funnycodesnippets.com", "furzauflunge.de", "fusemail.com", "futboladdict.com",
            "fut.es", "fux0ringduh.com", "fxnxs.com", "fxprix.com", "fyii.de", "g4hdrop.us",
            "gabes.cc", "gafsa.cc", "gafy.net", "gaggle.net", "gala.net", "galaxyhit.com",
            "galaxy.tv", "gally.jp", "galmail.co.za", "gamebox.net", "gamecocks.com", "gamegregious.com",
            "gamerssolution.com", "games.com", "garbagecollector.org", "garbagemail.org", "gardener.com", "gardenscape.ca",
            "garizo.com", "garliclife.com", "garrifulio.mailexpire.com", "garrymccooey.com", "gav0.com", "gawab.com",
            "gay.com", "gaymailbox.com", "gazabo.net", "gaza.net", "gazeta.pl", "gci.net",
            "gdi.net", "geekforex.com", "geeklife.com", "geew.ru", "gehensiemirnichtaufdensack.de", "geldwaschmaschine.de",
            "gelitik.in", "gemari.or.id", "genderfuck.net", "genxemail.com", "geologist.com", "geopia.com",
            "georgia.usa.com", "germanymail.com", "geronra.com", "geschent.biz", "get1mail.com", "get2mail.fr",
            "getairmail.cf", "getairmail.com", "getairmail.ga", "getairmail.gq", "getairmail.ml", "getairmail.tk",
            "geteit.com", "getintobooks.com", "get-mail.cf", "get-mail.ga", "get-mail.ml", "getmail.no",
            "getmails.eu", "get-mail.tk", "getonemail.com", "getonemail.net", "get.pp.ua", "ggaweb.ch",
            "ghosttexter.de", "giaiphapmuasam.com", "giantmail.de", "giga4u.de", "ginzi.be", "ginzi.co.uk",
            "ginzi.es", "ginzi.net", "ginzy.co.uk", "ginzy.eu", "girlsindetention.com", "girlsundertheinfluence.com",
            "gishpuppy.com", "giza.cc", "gjk.dk", "glay.org", "glendale.net", "glitch.sx",
            "glittergrrrls.com", "globalfree.it", "globalpinoy.com", "globalsite.com.br", "globaltouron.com", "globalum.com",
            "globetrotter.net", "globomail.com", "glucosegrin.com", "gmail.com", "gmal.com", "gmatch.org",
            "gmial.com", "gnctr-calgary.com", "go2net.com", "go2usa.info", "go4.it", "go.aggies.com",
            "go.air-force.com", "goatrance.com", "goat.si", "go.badgers.com", "go-bama.com", "go.big-orange.com",
            "go.blue.devils.com", "go.buffaloes.com", "go.bulldogs.com", "go-cavs.com", "go-chargers.com", "go.com",
            "go.cougars.com", "go-dawgs.com", "goddess.com", "go.dores.com", "goemailgo.com", "gofree.co.uk",
            "go.gamecocks.com", "go-gators.com", "gohip.com", "go-hogs.com", "go.huskies.com", "go-irish.com",
            "golfemail.com", "goliadtexas.com", "go.longhorns.com", "gomail.com.ua", "gomail.in", "go.mustangs.com",
            "gonowmail.com", "gonuts4free.com", "googlemail.com", "goplay.com", "go.rebels.com", "gorillaswithdirtyarmpits.com",
            "go.ro", "gorontalo.net", "go.ru", "go-spartans.com", "gospelcity.com", "go.terrapins.com",
            "gothere.biz", "gothicgirl.com", "go-tigers.com", "gotmail.com", "gotmail.net", "gotmail.org",
            "gotomy.com", "gotti.otherinbox.com", "govzone.com", "gowikibooks.com", "gowikicampus.com", "gowikicars.com",
            "gowikifilms.com", "gowikigames.com", "gowikimusic.com", "gowikinetwork.com", "gowikitravel.com", "gowikitv.com",
            "go.wildcats.com", "go.wolverines.com", "go.yellow-jackets.com", "grad.com", "graduate.org", "graffiti.net",
            "grandmamail.com", "grandmasmail.com", "grapemail.net", "graphic-designer.com", "gratisweb.com", "greatautos.org",
            "great-host.in", "greenmail.net", "greensloth.com", "greggamel.com", "greggamel.net", "gregorsky.zone",
            "gregorygamel.com", "gregorygamel.net", "grish.de", "grokleft.com", "groupmail.com", "grr.la",
            "gs-arc.org", "gsredcross.org", "gsrv.co.uk", "gtechnics.com", "guate.net", "gudanglowongan.com",
            "guerillamail.biz", "guerillamailblock.com", "guerillamail.com", "guerillamail.de", "guerillamail.info", "guerillamail.net",
            "guerillamail.org", "guerrillamail.biz", "guerrillamailblock.com", "guerrillamail.com", "guerrillamail.de", "guerrillamail.info",
            "guerrillamail.net", "guerrillamail.org", "guessmail.com", "guinea.cc", "gustr.com", "guy.com",
            "gwalla.com", "gynzi.co.uk", "gynzi.es", "gynzy.at", "gynzy.es", "gynzy.eu",
            "gynzy.gr", "gynzy.info", "gynzy.lt", "gynzy.mobi", "gynzy.pl", "gynzy.ro",
            "gynzy.sk", "gzb.ro", "h8s.org", "haberx.com", "habitue.net", "hacccc.com",
            "hacker.am", "hackermail.com", "hackthatbit.ch", "hahawrong.com", "hail2theskins.com", "hailmail.net",
            "hairdresser.net", "haitisurf.com", "halejob.com", "haltospam.com", "hamptonroads.com", "hamra.cc",
            "handbag.com", "hanmail.net", "happemail.com", "happycounsel.com", "happyhippo.com", "harakirimail.com",
            "haribu.com", "haribu.net", "hartbot.de", "hasakah.com", "hateinthebox.com", "hatespam.org",
            "hat-geld.de", "hawaii.com", "hawaii.usa.com", "hawrong.com", "hayahaya.tg", "hazelnut4u.com",
            "hazelnuts4u.com", "hazmatshipping.org", "headstrong.de", "heathenhammer.com", "heathenhero.com", "hebron.tv",
            "hecat.es", "hedgeai.com", "heesun.net", "hellodream.mobi", "helloricky.com", "helpinghandtaxcenter.org",
            "heremail.com", "heroulo.com", "herpderp.nl", "herp.in", "hetnet.nl", "hg8n415.com",
            "hi5.si", "hiddentragedy.com", "hidemail.de", "hidzz.com", "highbros.org", "highveldmail.co.za",
            "hilarious.com", "hildebrands.de", "hingis.org", "hiphopfan.com", "hispavista.com", "hitmanrecords.com",
            "hitthepuck.com", "h-mail.us", "hmail.us", "hmamail.com", "hmh.ro", "h.mintemail.com",
            "hoanggiaanh.com", "hochsitze.com", "hockeyghiaccio.com", "hockeymail.com", "holapuravida.com", "homelocator.com",
            "homemail.com", "homemail.co.za", "homenetmail.com", "home.no.net", "home.ro", "home.se",
            "homestead.com", "homosexual.net", "homs.cc", "hong-kong-1.com", "hongkong.com", "hopemail.biz",
            "hopthu.com", "horsefucker.org", "hosanna.net", "hostlaba.com", "hotbot.com", "hotbox.ru",
            "hotcoolmail.com", "hotdak.com", "hot.ee", "hotfire.net", "hotinbox.com", "hotmai.com",
            "hot-mail.cf", "hot-mail.ga", "hot-mail.gq", "hot-mail.ml", "hot-mail.tk", "hotmial.com",
            "hotpop.com", "hot-shot.com", "hotvoice.com", "hour.com", "housat.com", "housemail.com",
            "houseofhorrors.com", "howling.com", "hpc.tw", "hs.vc", "ht.cx", "hugkiss.com",
            "huhmail.com", "hulapla.de", "hullnumber.com", "humaility.com", "human.lv", "humanoid.net",
            "humn.ws.gy", "humour.com", "hungpackage.com", "hurra.de", "hush.ai", "hush.com",
            "hushmail.com", "huskies.com", "huskion.net", "hutchcity.com", "hvastudiesucces.nl", "hwsye.net",
            "i12.com", "i2828.com", "iaoss.com", "ibatam.com", "ibest.com.br", "ibizdns.com",
            "ibnuh.bz", "ibra.cc", "icafe.com", "icantbelieveineedtoexplainthisshit.com", "ice.is", "icestorm.com",
            "icloud.com", "icq.com", "icq.ir", "icqmail.com", "icrazy.com", "icx.in",
            "icx.ro", "idaho.usa.com", "id.au", "idea-mail.com", "idigcomputers.com", "idigelectronics.com",
            "i-dig-movies.com", "idigvideos.com", "idirect.com", "idncafe.com", "id.ru", "idunno4recipes.com",
            "ieatspam.eu", "ieatspam.info", "ieg.com.br", "ieh-mail.de", "iespalomeras.net", "iespana.es",
            "i-france.com", "ifrance.com", "ig.com.br", "ige.es", "ignazio.it", "ignoremail.com",
            "ihatenetscape.com", "ihateyoualot.info", "iheartspam.org", "ikbenspamvrij.nl", "ilike2helpothers.com", "ilike2invest.com",
            "ilike2workout.com", "ilikeelectronics.com", "ilikeworkingout.com", "illinois.usa.com", "illistnoise.com", "ilovehomeprojects.com",
            "iloveourteam.com", "i-love-restaurants.com", "ilovespam.com", "iloveworkingout.com", "ilse.net", "ilse.nl",
            "imailbox.com", "imail.ru", "imails.info", "imap.cc", "imap-mail.com", "imapmail.org",
            "imel.org", "imgof.com", "imgv.de", "imstations.com", "in2autos.net", "iname.acom",
            "iname.com", "inbax.tk", "inbound.plus", "inbox2.info", "inboxalias.com", "inboxclean.com",
            "inboxclean.org", "inbox.com", "inboxdesign.me", "inboxed.im", "inboxed.pw", "inbox.ge",
            "inbox.lv", "in-box.net", "inbox.net", "inboxproxy.com", "inbox.ru", "inbox.si",
            "inboxstore.me", "incamail.com", "inclusiveprogress.com", "incognitomail.com", "incognitomail.net", "incognitomail.org",
            "in.com", "incq.com", "indamail.hu", "", "indexa.fr", "india.com",
            "indiamail.com", "indiana.usa.com", "indiatimes.com", "indieclad.com", "indirect.ws", "ind.st",
            "induquimica.org", "ineec.net", "inet.com.ua", "inet.ua", "", "infinito.it",
            "infoapex.com", "infocom.zp.ua", "infohq.com", "infomail.es", "infomart.or.jp", "infosat.net",
            "infovia.com.ar", "inggo.org", "inicia.es", "inmail24.com", "inmail.sk", "innocent.com",
            "inorbit.com", "inoutbox.com", "inoutmail.de", "inoutmail.eu", "inoutmail.info", "inoutmail.net",
            "insanumingeniumhomebrew.com", "insorg-mail.info", "instantemailaddress.com", "instant-mail.de", "instruction.com", "instructor.net",
            "insurer.com", "intelnet.net.gt", "intelnett.com", "interblod.com", "interestedinthejob.com", "interfree.it",
            "interia.pl", "interlap.com.ar", "intermail.hu", "internetegypt.com", "internet-e-mail.com", "internetemails.net",
            "internet.lu", "internetmailing.net", "internet-mail.org", "internetoftags.com", "interstats.org", "intersteller.com",
            "intimatefire.com", "intomotors.com", "inwind.it", "iobox.com", "iobox.fi", "iol.it",
            "iol.pt", "iowa.usa.com", "iozak.com", "ip3.com", "ip4.pp.ua", "ip6.li",
            "ip6.pp.ua", "i-p.com", "ipermitmail.com", "iphon.biz", "ipoo.org", "ipsur.org",
            "iqemail.com", "iquebec.com", "ir.ae", "iran.com", "irangate.net", "iraq.ir",
            "irbid.ws", "irc.so", "ire.ir", "ireland.ir", "irelandmail.com", "irish2me.com",
            "irj.hu", "", "iroid.com", "ironiebehindert.de", "irow.com", "irr.ir",
            "irssi.tv", "is.af", "iscool.net", "islandmama.com", "ismailia.cc", "ismart.net",
            "isonews2.com", "isonfire.com", "isp9.net", "ispey.com", "israelmail.com", "ist-der-mann.de",
            "ist-der-wahnsinn.de", "ist-echt.so", "istecht.so", "ist-genialer.de", "ist-schlauer.de", "ist-supersexy.de",
            "isukrainestillacountry.com", "it7.ovh", "italymail.com", "itelgua.com", "itloox.com", "itmom.com",
            "itunesgiftcodegenerator.com", "ivenus.com", "iwan-fals.com", "iwatchrealitytv.com", "iwi.net", "iwon.com",
            "ixp.net", "ixx.io", "jadida.cc", "jadida.org", "jafps.com", "japan.com",
            "jaydemail.com", "jazzemail.com", "jdmadventures.com", "jdz.ro", "jedrzejow.pl", "jellyrolls.com",
            "jeoce.com", "jerash.cc", "jetable.com", "jetable.fr.nf", "jetable.net", "jetable.org",
            "jetable.pp.ua", "jetemail.net", "jingjo.net", "jippii.fi", "jizan.cc", "jmail.co.za",
            "jmail.ro", "jnxjn.com", "job4u.com", "jobbikszimpatizans.hu", "jobposts.net", "jobs-to-be-done.net",
            "joelpet.com", "joetestalot.com", "jojomail.com", "jopho.com", "jouf.cc", "journalist.com",
            "jourrapide.com", "jovem.te.pt", "joymail.com", "jpco.org", "jpg.ir", "j-p.us",
            "jsrsolutions.com", "juanitabynum.com", "jubii.dk", "jubiipost.dk", "jumpy.it", "jungkamushukum.com",
            "junk1e.com", "junkmail.ga", "junkmail.gq", "junk.to", "juno.com", "justemail.net",
            "justmailz.com", "jwork.ru", "kaazoo.com", "kabissa.org", "kairouan.cc", "kaixo.com",
            "kakadua.net", "kalapi.org", "kalluritimes.com", "kalpoint.com", "kamsg.com", "kann.so",
            "kanoodle.com", "kansas.usa.com", "kaovo.com", "karak.cc", "karenkey.com", "",
            "kariplan.com", "kartvelo.com", "kasmail.com", "kaspop.com", "katamail.com", "kataweb.it",
            "kayafmmail.co.za", "kazelink.ml", "kcrw.de", "keepmymail.com", "keinhirn.de", "keipino.de",
            "kekita.com", "keko.com.ar", "kemptvillebaseball.com", "kennedy808.com", "kentucky.usa.com", "keptprivate.com",
            "keromail.com", "khaimah.cc", "khartoum.cc", "khobar.cc", "kiani.com", "kickboxing.com",
            "kidrock.com", "killmail.com", "killmail.net", "kimo.com", "kimsdisk.com", "kingsq.ga",
            "kinkyemail.com", "kiois.com", "kir.ch.tc", "kismail.ru", "kissfans.com", "kisstwink.com",
            "kitnastar.com", "kittymail.com", "kiwitown.com", "klassmaster.com", "klassmaster.net", "klik.it",
            "klikni.cz", "kloap.com", "kludgemush.com", "klzlk.com", "kmhow.com", "kmtn.ru",
            "koko.com", "kolozsvar.ro", "kombud.com", "kommunity.biz", "kon42.com", "kook.ml",
            "kool-things.com", "kopagas.com", "kopaka.net", "koreamail.com", "koreanmail.com", "kosmetik-obatkuat.com",
            "kostenlosemailadresse.de", "koszmail.pl", "kotaksuratku.info", "k.ro", "krovatka.su", "",
            "krunis.com", "krypton.tk", "ksa.ir", "ksiskdiwey.cf", "kuhrap.com", "kukamail.com",
            "kulturbetrieb.info", "kuronowish.com", "kurzepost.de", "kuwait.ir", "kuwaiti.tv", "kvhrr.com",
            "kwift.net", "kwilco.net", "kyal.pl", "kyokodate.com", "kyokofukada.net", "kyrgyzstan.cc",
            "l0real.net", "l33r.eu", "labetteraverouge.at", "lackmail.net", "lackmail.ru", "ladymail.cz",
            "lagoon.nc", "lags.us", "lahaonline.com", "lain.ch", "lakelivingstonrealestate.com", "lamalla.net",
            "lancsmail.com", "landmail.co", "land.ru", "laoeq.com", "laoho.com", "laposte.net",
            "larjem.com", "lastmail.co", "lastmail.com", "latakia.cc", "latchess.com", "latinabarbie.com",
            "latinmail.com", "latinogreeks.com", "lavabit.com", "lawlita.com", "lawyer.com", "lawyersmail.com",
            "lawyerzone.com", "lazyinbox.com", "l-c-a.us", "ldop.com", "ldtp.com", "lebanese.cc",
            "lebanonatlas.com", "leeching.net", "leehom.net", "lee.mx", "leesville.com", "legislator.com",
            "lellno.gq", "lemondrop.com", "leonardo.it", "leonlai.net", "letmeinonthis.com", "letsjam.com",
            "letterboxes.org", "letterbox.org", "letthemeatspam.com", "levele.com", "lexpress.net", "lez.se",
            "lhsdv.com", "liamcyrus.com", "libero.it", "liberomail.com", "libertysurf.net", "libre.net",
            "lifebyfood.com", "lifetotech.com", "lightwines.org", "ligsb.com", "lillemap.net", "lilo.me",
            "lindenbaumjapan.com", "link2mail.net", "linkedintuts2016.pw", "linkmaster.com", "linuxfreemail.com", "linuxmail.org",
            "linuxmail.so", "lionsfan.com.au", "list.ru", "", "litedrop.com", "livedoor.com",
            "lkgn.se", "llandudno.com", "llangollen.com", "llogin.ru", "lmxmail.sk", "loadby.us",
            "loan101.pro", "lobbyist.com", "locomodev.net", "loggain.net", "loggain.nu", "login-email.cf",
            "login-email.ga", "login-email.ml", "login-email.tk", "logular.com", "loh.pp.ua", "loin.in",
            "loketa.com", "lolfreak.net", "lolmail.biz", "lolnetwork.net", "lol.ovpn.to", "london.com",
            "london.ir", "longhorns.com", "look.com", "looksmart.com", "looksmart.com.au", "looksmart.co.uk",
            "lookugly.com", "lopl.co.cc", "lortemail.dk", "losemymail.com", "loteria.net", "lotonazo.com",
            "louisiana.usa.com", "louiskoo.com", "love2exercise.com", "love2workout.com", "loveable.com", "lovecat.com",
            "loveemail.com", "lovefantasysports.com", "loveis.lv", "lovemail.com", "lovemeleaveme.com", "loves.dicksinhisan.us",
            "loves.dicksinmyan.us", "lovetoexercise.com", "lovingjesus.com", "lowrider.com", "lpemail.com", "lpfmgmtltd.com",
            "lpo.ddnsfree.com", "lr78.com", "lr7.us", "lroid.com", "lru.me", "lubnan.cc",
            "lubnan.ws", "lucky7lotto.net", "luckymail.com", "luckymail.org", "luisgiisjsk.tk", "lukecarriere.com",
            "lukemail.info", "lukop.dk", "luso.pt", "lusoweb.pt", "luukku.com", "luv2.us",
            "luvfishing.com", "luvgolfing.com", "luvsoccer.com", "lv-inter.net", "lycosmail.com", "lyfestylecreditsolutions.com",
            "m21.cc", "m4ilweb.info", "ma1l.bij.pl", "maboard.com", "mac.com", "machinecandy.com",
            "mac.hush.com", "macmail.com", "macosnine.com", "macromaid.com", "madcrazy.com", "madeniggaz.net",
            "madinah.cc", "madonnafan.com", "madonno.com", "madrid.com", "mad.scientist.com", "mag2.com",
            "magamail.com", "maghreb.cc", "magicbox.ro", "magicmail.co.za", "magik-net.com", "mag-spam.net",
            "maidlow.info", "mail114.net", "mail15.com", "mail1a.de", "mail21.cc", "mail2rss.org",
            "mail2world.com", "mail3000.com", "mail333.com", "mail4me.com", "mail4trash.com", "mail666.ru",
            "mail707.com", "mail72.com", "mail8.com", "mailandftp.com", "mailandnews.com", "mailas.com",
            "mailasia.com", "mail-atlas.net", "mail.austria.com", "mail-awu.de", "mail.az", "mailback.com",
            "mail.be", "mail.bg", "mailbg.com", "mailbidon.com", "mailbiz.biz", "mailblocks.com",
            "mailbolt.com", "mailbomb.com", "mailbox92.biz", "mailbox.as", "mailbox.co.za", "mail-box.cz",
            "mailbox.gr", "mailbox.hu", "mailbox.sk", "mailbucket.org", "mail.bulgaria.com", "mail.by",
            "mailcan.com", "mailcat.biz", "mailcatch.com", "mail-center.com", "mail-central.com", "mailchop.com",
            "mailcircuit.com", "mailcker.com", "mailclub.fr", "mailclub.net", "mailc.net", "mail.com",
            "mail.co.za", "mail.de", "mailde.de", "mailde.info", "mail.dk", "maildozy.com",
            "maildrop.cc", "maildrop.cf", "maildrop.ga", "maildrop.gq", "maildrop.ml", "maildu.de",
            "maildx.com", "maileater.com", "mailed.in", "mailed.ro", "mail.ee", "maileimer.de",
            "maileme101.com", "mailexpire.com", "mailfa.tk", "mail-filter.com", "mailfly.com", "mailforce.net",
            "mailforspam.com", "mailfree.ga", "mailfree.gq", "mailfree.ml", "mailfreeonline.com", "mailfs.com",
            "mailftp.com", "mailglobal.net", "mail.goo.ne.jp", "mail.gr", "mailguard.me", "mailhaven.com",
            "mailhazard.com", "mailhazard.us", "mailhz.me", "mailimate.com", "mailin8r.com", "mailinatar.com",
            "mailinater.com", "mailinator2.com", "mailinator.com", "mailinator.co.uk", "mailinator.gq", "mailinator.info",
            "mailinator.net", "mailinator.org", "mailinator.us", "mailinbox.cf", "mailinbox.ga", "mailinbox.gq",
            "mailinbox.ml", "mailincubator.com", "mailingaddress.org", "mailingweb.com", "mailisent.com", "mailismagic.com",
            "mailita.tk", "mailite.com", "mail-jp.org", "mailjunk.cf", "mailjunk.ga", "mailjunk.gq",
            "mailjunk.ml", "mailjunk.tk", "mail.lawguru.com", "mailmate.com", "mail.md", "mailme24.com",
            "mail-me.com", "mailme.dk", "mailme.gq", "mailme.ir", "mailme.lv", "mailmetrash.com",
            "mail.mezimages.net", "mailmight.com", "mailmij.nl", "mail.mn", "mailmoat.com", "mailms.com",
            "mailna.co", "mailna.im", "mailna.in", "mailna.me", "mailnator.com", "mailnesia.com",
            "mailnew.com", "mailnull.com", "mailonaut.com", "mail-online.dk", "mail-on.us", "mailops.com",
            "mailorc.com", "mail.org", "mailorg.org", "mail-owl.com", "mail-page.com", "mailpanda.com",
            "mailpersonal.com", "mail.pf", "mailpick.biz", "mailproxsy.com", "mail.pt", "mailquack.com",
            "mailrock.biz", "mailroom.com", "mail.ru", "mailru.com", "mailsac.com", "mailscrap.com",
            "mails.de", "mailseal.de", "mailsent.net", "mailserver.dk", "mailservice.ms", "mailshell.com",
            "mailsiphon.com", "mailslapping.com", "mailslite.com", "mailsnare.net", "mailsurf.com", "mailtemp.info",
            "mail-temporaire.com", "mailtemporaire.com", "mail-temporaire.fr", "mailtemporaire.fr", "mailtome.de", "mailtothis.com",
            "mailtrash.net", "mailtv.net", "mailtv.tv", "mailup.net", "mailvault.com", "mailworks.org",
            "mail-x-change.com", "mail.yahoo.co.jp", "mailzilla.com", "mailzilla.org", "mailzilla.orgmbx.cc", "mailzi.ru",
            "mail.zp.ua", "maine.usa.com", "majorana.martina-franca.ta.it", "majorgolfer.com", "majorshopaholic.com", "majortechie.com",
            "makemetheking.com", "maktoob.com", "malahov.de", "malayalamdtp.com", "malayalamtelevision.net", "malayalapathram.com",
            "male.ru", "manager.de", "manama.cc", "manifestgenerator.com", "manlymail.net", "mansiondev.com",
            "mansoura.tv", "mantrafreenet.com", "mantramail.com", "mantraonline.com", "manybrain.com", "marchmail.com",
            "marihuana.ro", "marijuana.nl", "marillion.net", "marketweighton.com", "markmurfin.com", "marrakesh.cc",
            "maryland.usa.com", "mascara.ws", "masrawy.com", "massachusetts.usa.com", "mauimail.com", "mbox.com.au",
            "mbx.cc", "mcache.net", "mciek.com", "mcom.com", "mcrmail.com", "md5hashing.net",
            "mdhc.tk", "me.by", "me.com", "medicinatv.com", "meepsheep.eu", "meetingmall.com",
            "megamail.pt", "megarave.com", "mega-schlau.de", "mega.zik.dj", "meinspamschutz.de", "meknes.cc",
            "meltmail.com", "memori.ru", "", "menara.ma", "merseymail.com", "mesra.net",
            "messagebeamer.de", "messagez.com", "messwiththebestdielikethe.rest", "metacrawler.com", "metalfan.com", "meta.ua",
            "", "mexico.com", "mexicomail.com", "mezimages.net", "mfsa.ru", "miaferrari.com",
            "miaoweb.net", "michigan.usa.com", "micro2media.com", "midcoastcustoms.com", "midcoastcustoms.net", "midcoastsolutions.com",
            "midcoastsolutions.net", "midlertidig.com", "midlertidig.net", "midlertidig.org", "mierdamail.com", "miesto.sk",
            "mighty.co.za", "migmail.net", "migmail.pl", "migumail.com", "mijnhva.nl", "milacamn.net",
            "milmail.com", "mindless.com", "mindviz.com", "minister.com", "ministry-of-silly-walks.de", "minnesota.usa.com",
            "minsmail.com", "mintemail.com", "mississippi.usa.com", "missouri.usa.com", "misterpinball.de", "mixmail.com",
            "mji.ro", "mjukglass.nu", "mkpfilm.com", "ml1.net", "ml2clan.com", "ml8.ca",
            "mlanime.com", "mm5.se", "mmail.com", "mm.my", "mm.st", "moakt.com",
            "moakt.ws", "mobileninja.co.uk", "mobimail.mn", "mobi.web.id", "mobsters.com", "mobstop.com",
            "moburl.com", "mockmyid.com", "modemnet.net", "modomail.com", "moeri.org", "mofa.com",
            "mohmal.com", "mohmal.tech", "moldovacc.com", "moldova.com", "momentics.ru", "monadi.ml",
            "monarchy.com", "moncourrier.fr.nf", "monemail.fr.nf", "moneypipe.net", "monmail.fr.nf", "montana.usa.com",
            "montevideo.com.uy", "monumentmail.com", "moomia.com", "moonwake.com", "moose-mail.com", "moot.es",
            "mor19.uu.gl", "moreawesomethanyou.com", "moreorcs.com", "morsin.com", "mosaicfx.com", "moscowmail.com",
            "motique.de", "motley.com", "motormania.com", "motor-nut.com", "mountainregionallibrary.net", "movemail.com",
            "moviefan.com", "mox.pp.ua", "moza.pl", "mozej.com", "mr24.co", "mr.outblaze.com",
            "mrspender.com", "msa.minsmail.com", "msb.minsmail.com", "mscold.com", "msgos.com", "msk.ru",
            "msn.com", "msn.co.uk", "msnzone.cn", "mspeciosa.com", "mswork.ru", "msxd.com",
            "mt2009.com", "mt2014.com", "mt2015.com", "mtmdev.com", "muathegame.com", "muchomail.com",
            "mucincanon.com", "mundo-r.com", "munich.com", "muscat.tv", "muscat.ws", "music.com",
            "musician.net", "musician.org", "musicsites.com", "muslim.com", "muslimsonline.com", "muss.so",
            "mustangs.com", "mutant.me", "mvrht.com", "mvrht.net", "mwarner.org", "mx0.wwwnew.eu",
            "mxfuel.com", "mxs.de", "my10minutemail.com", "mybitti.de", "myblue.cc", "mycabin.com",
            "mycapitalsmail.com", "mycard.net.ua", "mycatiscool.com", "mycity.com", "mycleaninbox.net", "mycommail.com",
            "mycool.com", "mycorneroftheinter.net", "mydemo.equipment", "mydomain.com", "myecho.es", "myemailboxy.com",
            "myeweb.com", "myfantasyteamrules.com", "myfastmail.com", "myfunnymail.com", "mygamingconsoles.com", "mygrande.net",
            "myiris.com", "myjazzmail.com", "mykickassideas.com", "mykolab.com", "mymacmail.com", "mymail.dk",
            "mymail-in.net", "mymailoasis.com", "mymail.ph.inter.net", "mymail.ro", "mynet.com", "mynet.com.tr",
            "mynetstore.de", "myopang.com", "myopera.com", "myotw.net", "myownemail.com", "mypacks.net",
            "mypartyclip.de", "mypersonalemail.com", "myphantomemail.com", "myplace.com", "myrealbox.com", "mysamp.de",
            "myself.com", "myspace.com", "myspaceinc.com", "myspaceinc.net", "myspaceinc.org", "myspacepimpedup.com",
            "myspamless.com", "myteamisbest.com", "mytemp.email", "mytempemail.com", "mytempmail.com", "myt.mu",
            "mytrashmail.com", "mywarnernet.net", "myway.com", "myzx.com", "mzgchaos.de", "n1nja.org",
            "n2business.com", "n2.com", "n2mail.com", "n2software.com", "nabble.com", "nabeul.cc",
            "nabeul.info", "nablus.cc", "nabuma.com", "nador.cc", "najaf.cc", "nakedtruth.biz",
            "name.com", "nameplanet.com", "nanamail.co.il", "nanaseaikawa.com", "nandomail.com", "nanonym.ch",
            "narod.ru", "naseej.com", "nastything.com", "nate.com", "national-champs.com", "nationalgardeningclub.com",
            "national.shitposting.agency", "nativeweb.net", "naveganas.com", "naver.com", "nebraska.usa.com", "negated.com",
            "nemra1.com", "nenter.com", "neomailbox.com", "nepwk.com", "nerd4life.de", "nerdshack.com",
            "nervhq.org", "nervmich.net", "nervtmich.net", "net4b.pt", "net4jesus.com", "net4you.at",
            "netbounce.com", "netbroadcaster.com", "netbusiness.com", "netcabo.pt", "netcape.net", "netcourrier.com",
            "netexecutive.com", "netfingers.com", "netfirms.com", "net.hr", "netkushi.com", "netmails.com",
            "netmails.net", "netmongol.com", "netpiper.com", "netposta.net", "netricity.nl", "netris.net",
            "netscape.com", "netscape.net", "netscapeonline.co.uk", "net-shopping.com", "netsquare.com", "net-surf.com",
            "nettaxi.com", "netti.fi", "net.ua", "netviewer-france.com", "networld.com", "netzero.com",
            "netzero.net", "netzidiot.de", "neustreet.com", "nevada.usa.com", "neverbox.com", "nevermail.de",
            "newhampshire.usa.com", "newjersey.usa.com", "newmail.com", "newmail.net", "newmail.ok.com", "newmail.ru",
            "newmexico.usa.com", "news-fanatic.com", "newspaperemail.com", "newspaperfan.com", "newyorkcity.com", "newyork.com",
            "newyork.usa.com", "nextstopvalhalla.com", "nezzart.com", "nfast.net", "nfmail.com", "nguyenusedcars.com",
            "nh3.ro", "nice-4u.com", "nicegal.com", "nicknassar.com", "nickrizos.com", "nigge.rs",
            "nightimeuk.com", "nightly.com", "nightmail.com", "nightmail.ru", "nincsmail.hu", "ninfan.com",
            "niwl.net", "nm7.cc", "nmail.cf", "nm.ru", "", "nnh.com",
            "nnot.net", "noavar.com", "nobitcoin.net", "noblepioneer.com", "nobugmail.com", "nobulk.com",
            "nobuma.com", "nocharge.com", "noclickemail.com", "nodezine.com", "noemail.com", "nogmailspam.info",
            "noicd.com", "nokiamail.com", "nomail2me.com", "nomail.pw", "nomail.xl.cx", "nomorespamemails.com",
            "nom.za", "nonomail.com", "nonpartisan.com", "nonspam.eu", "nonspammer.de", "nonze.ro",
            "noolhar.com", "noref.in", "norih.com", "norseforce.com", "northcarolina.usa.com", "northdakota.usa.com",
            "nospam4.us", "nospamfor.us", "nospammail.net", "nospamme.com", "nospamthanks.info", "no-spam.ws",
            "nospam.ze.tc", "nothingtoseehere.ca", "notmailinator.com", "notrnailinator.com", "notsharingmy.info", "no-ux.com",
            "nowhere.org", "now.im", "nowmymail.com", "nowzer.com", "n.spamtrap.co", "ntlhelp.net",
            "nubescontrol.com", "nullbox.info", "null.net", "nurfuerspam.de", "nus.edu.sg", "nutpa.net",
            "nuts2trade.com", "nwldx.com", "nwytg.net", "ny7.me", "nyc.com", "nycmail.com",
            "ny.com", "nypato.com", "nz11.com", "nzoomail.com", "o2.pl", "o2stk.org",
            "o3enzyme.com", "o7i.net", "oath.com", "obfusko.com", "objectmail.com", "obobbo.com",
            "obxpestcontrol.com", "oceanfree.net", "ocsnet.net", "odaymail.com", "oddpost.com", "odeon.pl",
            "odmail.com", "odnorazovoe.ru", "oerpub.org", "offcolormail.com", "offshore-proxies.net", "offshorewebmail.com",
            "ofir.dk", "ohaaa.de", "ohio.usa.com", "ohi.tw", "ohne-drogen-gehts.net", "oicexchange.com",
            "okclprojects.com", "oklahoma.usa.com", "okrent.us", "ok.ru", "okzk.com", "ole.com",
            "oleco.net", "olympist.net", "olypmall.ru", "omail.pro", "omaninfo.com", "omani.ws",
            "omdurman.cc", "omen.ru", "", "omnievents.org", "onatoo.com", "ondikoi.com",
            "one2mail.info", "onebox.com", "onenet.com.ar", "oneoffemail.com", "oneoffmail.com", "one-time.email",
            "onet.pl", "onewaymail.com", "ongc.net", "oninet.pt", "onlatedotcom.info", "onlineidea.info",
            "online.ie", "online.ms", "online.ru", "onlinevideosrock.com", "onlinewiz.com", "onobox.com",
            "onqin.com", "on-steroids.de", "ontyne.biz", "onzmail.com", "oolus.com", "oopi.org",
            "opayq.com", "openbg.com", "open.by", "openforyou.com", "openmail.cc", "opentransfer.com",
            "opentrash.com", "operamail.com", "operationivy.com", "oplusnet.com", "opp24.com", "optician.com",
            "oran.cc", "orange.es", "orange.fr", "orangehome.co.uk", "orange.jo", "orange.pl",
            "orbitel.bg", "orcon.net.nz", "ordinaryamerican.net", "oregon.usa.com", "oreka.com", "organizer.net",
            "orgio.net", "org.ua", "oroki.de", "orthodontist.net", "orthodox.com", "oshietechan.link",
            "osite.com.br", "oso.com", "otherinbox.com", "oued.info", "oued.org", "oujda.biz",
            "oujda.cc", "ourbrisbane.com", "ourklips.com", "ournet.md", "ourpreviewdomain.com", "ourprofile.net",
            "ourwest.com", "outgun.com", "outlawspam.com", "ovpn.to", "owlpic.com", "ownmail.net",
            "ownsyou.de", "oxfoot.com", "oxopoha.com", "ozu.es", "ozyl.de", "p.9q.ro",
            "pa9e.com", "pacer.com", "pacific-ocean.com", "pacificwest.com", "pagamenti.tk", "paginasamarillas.com",
            "paidoffers.net", "pakistani.ws", "pakistanmail.com", "palmyra.cc", "palmyra.ws", "paltalk.ir",
            "pancakemail.com", "pandawa.com", "pando.com", "pandora.be", "paplease.com", "paris.com",
            "parsimail.com", "parspage.com", "pastebitch.com", "patmail.com", "pattayacitythailand.com", "pc4me.us",
            "pcbee.com", "pcpostal.com", "pcusers.otherinbox.com", "pediatrician.com", "penguinmaster.com", "penisgoes.in",
            "pennsylvania.usa.com", "peoplepc.com", "peopleweb.com", "pepbot.com", "persian.com", "personales.com",
            "personal.ro", "peru.com", "peterdethier.com", "petlover.com", "petml.com", "petrofind.com",
            "petrzilka.net", "pfui.ru", "photofile.ru", "", "photographer.net", "photomark.net",
            "phpbb.uu.gl", "phreaker.net", "phunkybitches.com", "physicist.net", "pigeonportal.com", "pii.at",
            "pikaguam.com", "piki.si", "pilu.com", "pimagop.com", "pimpedupmyspace.com", "pinehill-seattle.org",
            "pingir.com", "pinkcity.net", "pinoymail.com", "pipni.cz", "pisem.net", "pisls.com",
            "pitbullmail.com", "pi.vu", "pjjkp.com", "planetaccess.com", "planetmail.com", "planetmail.net",
            "planetout.com", "planet-school.de", "planetsmeg.com", "plasa.com", "playersodds.com", "playful.com",
            "plexolan.de", "plhk.ru", "pluno.com", "plusmail.com.br", "plw.me", "pmail.net",
            "pnetmail.co.za", "pobox.ru", "pobox.sk", "pochta.com", "", "pochtamt.ru",
            "pochta.ru", "poczta.fm", "poczta.onet.pl", "poetic.com", "pogowave.com", "poh.pp.ua",
            "pojok.ml", "pokemail.net", "pokiemobile.com", "polandmail.com", "polbox.com", "politician.com",
            "politikerclub.de", "pooae.com", "poofy.org", "pookmail.com", "poop.com", "poopiebutt.club",
            "poormail.com", "pop3.ru", "pop.co.th", "popesodomy.com", "popgx.com", "popmail.com",
            "poppymail.com", "popsmail.com", "popstar.com", "porsh.net", "portafree.com", "portaldosalunos.com",
            "portsaid.cc", "portugalmail.com", "portugalmail.pt", "postaccesslite.com", "postacin.com", "posta.ge",
            "post.com", "post.cz", "post.expart.ne.jp", "postiloota.net", "postinbox.com", "postino.ch",
            "postino.it", "postmaster.co.uk", "postonline.me", "post.pl", "postpro.net", "post.sk",
            "potsmokersnet.com", "poutineyourface.com", "powdermail.com", "powered.name", "powlearn.com", "ppetw.com",
            "pp.ua", "praize.com", "presidency.com", "press.co.jp", "priest.com", "primabananen.net",
            "primetap.com", "primposta.com", "printesamargareta.ro", "privacy.net", "privatdemail.net", "private.21cn.com",
            "privy-mail.com", "privy-mail.de", "privymail.de", "probemail.com", "procrackers.com", "profesional.com",
            "profession.freemail.com.br", "programmer.net", "proinbox.com", "project420.com", "projectcl.com", "prolife.net",
            "promessage.com", "prontomail.com", "propscore.com", "pro-tag.org", "protestant.com", "protonmail.ch",
            "protonmail.com", "provincial.net", "proxymail.eu", "proxyparking.com", "prtnx.com", "prtz.eu",
            "psh.me", "publicaccounting.com", "publicist.com", "puertoricowow.com", "punkass.com", "puppetweb.com",
            "puppy.com.my", "purcell.email", "purelogistics.org", "put2.net", "putthisinyourspamdatabase.com", "pwrby.com",
            "pyjgoingtd.com", "qassem.cc", "qasti.com", "qatar.io", "q.com", "qc.to",
            "qibl.at", "qipmail.net", "qip.ru", "", "qisdo.com", "qisoa.com",
            "qlmail.com", "qoika.com", "qq.com", "qq.my", "qrio.com", "qsl.net",
            "qsl.ro", "quadrafit.com", "qualityservice.com", "quds.cc", "qudsmail.com", "queerplaces.com",
            "quepasa.com", "quick.cz", "quickinbox.com", "quickmail.nl", "quickwebmail.com", "qvy.me",
            "qwickmail.com", "r320.hu", "r4nd0m.de", "r8r4p0cb.com", "ra3.us", "raakim.com",
            "rabat.cc", "rabin.ca", "racingseat.com", "radicalz.com", "radiojobbank.com", "radiologist.net",
            "raetp9.com", "rafah.cc", "ragingbull.com", "raisingadaughter.com", "raketenmann.de", "rallye-webmail.com",
            "ramallah.cc", "rambler.ru", "rancidhome.net", "randomail.net", "ranmamail.com", "rapstar.com",
            "rapworld.com", "raqid.com", "rastamall.com", "ratedx.net", "ravearena.com", "ravemail.com",
            "ravemail.co.za", "ravermail.com", "rax.la", "raxtest.com", "razormail.com", "rbb.org",
            "rbcmail.ru", "rbox.co", "rbox.me", "rcpt.at", "rcs.gaggle.net", "realbookfan.com",
            "realemail.net", "realhealthnut.com", "realitytvaddict.net", "realitytvnut.com", "reallyfast.biz", "reallyfast.info",
            "reallyintomusic.com", "reallymymail.com", "real.ro", "realtravelfan.com", "realtyagent.com", "realtyalerts.ca",
            "rebels.com", "reborn.com", "receiveee.com", "recife.net", "recipeforfailure.com", "recme.net",
            "recode.me", "reconmail.com", "recursor.net", "recyclemail.dk", "redchan.it", "redfeathercrow.com",
            "rediffmail.com", "rediffmailpro.com", "redseven.de", "redskinscheer.com", "redskinsfamily.com", "redskinsfancentral.com",
            "redskinshog.com", "redskinsrule.com", "redskinsspecialteams.com", "redskinsultimatefan.com", "redwhitearmy.com", "reftoken.net",
            "regbypass.com", "regbypass.comsafe-mail.net", "reggaefan.com", "re-gister.com", "registerednurses.com", "reincarnate.com",
            "rejectmail.com", "relapsecult.com", "reliable-mail.com", "relia.com", "religious.com", "remail.cf",
            "remail.ga", "remarkable.rocks", "remixer.com", "remote.li", "repairman.com", "representative.com",
            "reptilegenetics.com", "rescueteam.com", "revenue.com", "revolvingdoorhoax.org", "rexian.com", "rhodeisland.usa.com",
            "rhyta.com", "riddermark.de", "risingsuntouch.com", "ritmes.net", "rklips.com", "rma.ec",
            "rmqkr.net", "rnailinator.com", "rn.com", "roanokemail.com", "robertspcrepair.com", "rochester-mail.com",
            "rock.com", "rockeros.com", "rocketmail.com", "rocketship.com", "rockfan.com", "rockinghamgateway.com",
            "rojname.com", "rollin.com", "rol.ro", "ro.lt", "romance106fm.com", "rome.com",
            "romymichele.com", "ronnierage.net", "r-o-o-t.com", "rotaniliam.com", "rowe-solutions.com", "royaldoodles.org",
            "royal.net", "rpharmacist.com", "rppkn.com", "rt.nl", "rtrtr.com", "ruffrey.com",
            "rumgel.com", "runi.ca", "ru.ru", "rushpost.com", "russiamail.com", "rustydoor.com",
            "rvb.ro", "rxpost.net", "s0ny.net", "s33db0x.com", "saabnet.com", "sabrestlouis.com",
            "sacbeemail.com", "sackboii.com", "sacmail.com", "safat.biz", "safat.info", "safat.us",
            "safat.ws", "safe-mailbox.com", "safe-mail.net", "safersignup.de", "safetymail.info", "safetypost.de",
            "safrica.com", "saharanightstempe.com", "saigonnet.vn", "saintly.com", "saint-mike.org", "salalah.cc",
            "salesperson.net", "salmiya.biz", "samerica.com", "samilan.net", "samsclass.info", "sanaa.cc",
            "sandelf.de", "sandiego.com", "sandwhichvideo.com", "sanfinder.com", "sanfranmail.com", "sanim.net",
            "sanook.com", "sanriotown.com", "sanstr.com", "sapibon.com", "sapo.pt", "sast.ro",
            "satukosong.com", "saturnfans.com", "sausen.com", "sayhi.net", "saynotospams.com", "sbcglobal.com",
            "scatmail.com", "scay.net", "scfn.net", "schachrol.com", "schafmail.de", "scheint.so",
            "schmeissweg.tk", "schrott-email.de", "schweiz.org", "sciaga.pl", "scientist.com", "sci.fi",
            "scotlandmail.com", "scoutmail.com", "scrapbookscrapbook.com", "sd3.in", "seapole.com", "search417.com",
            "seark.com", "seatto.com", "sebil.com", "secmail.pw", "secretary.net", "secretemail.de",
            "secretservices.net", "secured-link.net", "securehost.com.es", "secure-jlnet.com", "secure-mail.biz", "secure-mail.cc",
            "seductive.com", "seeb.cc", "seek4wap.com", "seekapps.com", "seekjobs4u.com", "sejaa.lv",
            "selfdestructingmail.com", "selfdestructingmail.org", "sendfree.org", "sendingspecialflyers.com", "sendmail.ru", "sendme.cz",
            "sendspamhere.com", "senseless-entertainment.com", "sent.as", "sent.at", "sent.com", "serga.com.ar",
            "sermix.com", "server4free.de", "server.ms", "serverwench.com", "services391.com", "sesmail.com",
            "sexforswingers.com", "sexical.com", "sexmagnet.com", "sexriga.lv", "sezet.com", "seznam.cz",
            "sfamo.com", "sfax.ws", "shadango.com", "shalar.net", "sharedmailbox.org", "sharklasers.com",
            "sharm.cc", "she.com", "shhmail.com", "shhuut.org", "shieldedmail.com", "shieldemail.com",
            "shiftmail.com", "shinnemo.com", "shipfromto.com", "shiphazmat.org", "shipping-regulations.com", "shippingterms.org",
            "shitmail.de", "shitmail.me", "shitmail.org", "shitware.nl", "shmeriously.com", "shortmail.net",
            "shotmail.ru", "showslow.de", "shrib.com", "shuf.com", "shut.name", "shut.ws",
            "siamlocalhost.com", "siamnow.net", "sibmail.com", "sify.com", "siliwangi.ga", "simpleitsecurity.info",
            "sina.cn", "sina.com", "sinai.cc", "sinamail.com", "sinanail.com", "sin.cl",
            "sinfiltro.cl", "singalongcenter.com", "singapore.com", "singlespride.com", "singmail.com", "singnet.com.sg",
            "sinnlos-mail.de", "sino.tw", "siraj.org", "siria.cc", "sirindia.com", "sirunet.com",
            "sister.com", "sistersbrothers.com", "siteposter.net", "sizzlemctwizzle.com", "sizzling.com", "skeefmail.com",
            "sketchyfriends.com", "skins4life.com", "sky-inbox.com", "sky-ts.de", "slamdunkfan.com", "slapsfromlastnight.com",
            "slaskpost.se", "slave-auctions.net", "slayerized.com", "slickriffs.co.uk", "slingshot.com", "slomusic.net",
            "slo.net", "slopsbox.com", "slothmail.net", "slushmail.com", "sly.io", "smaakt.naar.gravel",
            "s-mail.com", "smapfree24.com", "smapfree24.de", "smapfree24.eu", "smapfree24.info", "smapfree24.org",
            "smartemail.co.uk", "smartstocks.com", "smashmail.de", "smellfear.com", "smellrear.com", "smtp99.com",
            "smtp.ru", "smwg.info", "snail-mail.net", "snakebite.com", "snakemail.com", "sndt.net",
            "sneakemail.com", "sneakmail.de", "snkmail.com", "snoopymail.com", "snowboarding.com", "socamail.com",
            "socialfurry.org", "socialworker.net", "sociologist.com", "sofimail.com", "sofort-mail.de", "sofortmail.de",
            "softhome.net", "softpls.asia", "sogetthis.com", "sohu.com", "soisz.com", "sol.dk",
            "solidmail.com", "solution4u.com", "solvemail.info", "solventtrap.wiki", "songwriter.net", "soodmail.com",
            "soodomail.com", "soodonims.com", "soon.com", "soon.it", "so-simple.org", "sos.lv",
            "soulja-beatz.org", "soundvillage.org", "sousse.cc", "southcarolina.usa.com", "southdakota.usa.com", "space.com",
            "spacetowns.com", "spain.ir", "spainmail.com", "spam4.me", "spamail.de", "spamarrest.com",
            "spamavert.com", "spam-be-gone.com", "spambob.com", "spambob.net", "spambob.org", "spambog.com",
            "spambog.de", "spambog.net", "spambog.ru", "spambooger.com", "spambox.info", "spambox.irishspringrealty.com",
            "spambox.org", "spambox.us", "spamcannon.com", "spamcannon.net", "spamcero.com", "spamcon.org",
            "spamcorptastic.com", "spamcowboy.com", "spamcowboy.net", "spamcowboy.org", "spamday.com", "spamdecoy.net",
            "spamex.com", "spamfighter.cf", "spamfighter.ga", "spamfighter.gq", "spamfighter.ml", "spamfighter.tk",
            "spamfree24.com", "spamfree24.de", "spamfree24.eu", "spamfree24.info", "spamfree24.net", "spamfree24.org",
            "spamfree.eu", "spamgoes.in", "spamgourmet.com", "spamgourmet.net", "spamgourmet.org", "spamherelots.com",
            "spamhereplease.com", "spamhole.com", "spamify.com", "spaminator.de", "spamkill.info", "spam.la",
            "spaml.com", "spaml.de", "spamlot.net", "spammotel.com", "spamobox.com", "spamoff.de",
            "spam.org.es", "spamsalad.in", "spamslicer.com", "spamspot.com", "spamstack.net", "spam.su",
            "spamthis.co.uk", "spamthisplease.com", "spamtrail.com", "spamtroll.net", "spartapiet.com", "spb.ru",
            "specialoperations.com", "speed.1s.fr", "speedgaus.net", "speedpost.net", "speed-racer.com", "speedymail.net",
            "speedymail.org", "spells.com", "spikio.com", "spils.com", "spinfinder.com", "spoofmail.de",
            "sportemail.com", "spray.net", "spray.no", "spray.se", "spr.io", "spritzzone.de",
            "s.proprietativalcea.ro", "spybox.de", "spymac.com", "squizzy.de", "srbbs.com", "srilankan.net",
            "sroff.com", "sry.li", "ssan.com", "ssl-mail.com", "ssoia.com", "staatsterrorist.de",
            "stade.fr", "stalag13.com", "stampmail.com", "stanfordujjain.com", "starbuzz.com", "stargate2.com",
            "stargateatlantis.com", "stargatefanclub.com", "stargatesg1.com", "stargateu.com", "starlight-breaker.net", "starline.ee",
            "starmail.com", "starmail.org", "starmedia.com", "starspath.com", "start.com.au", "startfu.com",
            "startkeys.com", "start.no", "statdvr.com", "stathost.net", "statiix.com", "steambot.net",
            "stexsy.com", "stinkefinger.net", "stop-my-spam.cf", "stop-my-spam.com", "stop-my-spam.ga", "stop-my-spam.ml",
            "stop-my-spam.pp.ua", "stop-my-spam.tk", "streetracing.com", "streetwisemail.com", "stribmail.com", "stromox.com",
            "stuckmail.com", "student.com", "student.ednet.ns.ca", "studmail.com", "stuffmail.de", "stumpfwerk.com",
            "subspacemail.com", "suburbanthug.com", "suckmyd.com", "sudanese.cc", "sudanmail.net", "sudolife.me",
            "sudolife.net", "sudomail.biz", "sudomail.com", "sudomail.net", "sudoverse.com", "sudoverse.net",
            "sudoweb.net", "sudoworld.com", "sudoworld.net", "suez.cc", "sugarray.com", "suioe.com",
            "suisse.org", "sunbella.net", "sunmail1.com", "sunpoint.net", "sunrise.ch", "sunumail.sn",
            "sunuweb.net", "suomi24.fi", "super-auswahl.de", "superbikeclub.com", "superdada.it", "supereva.com",
            "supereva.it", "super-gerissen.de", "supergreatmail.com", "superintendents.net", "supermailbox.com", "supermailer.jp",
            "superplatyna.com", "superposta.com", "superrito.com", "superstachel.de", "suremail.info", "surf3.net",
            "surfassistant.com", "surfguiden.com", "surfsupnet.net", "surfy.net", "surgical.net", "surimail.com",
            "surnet.cl", "sverige.nu", "svizzera.org", "svk.jp", "svxr.org", "sweb.cz",
            "sweden.ir", "swedenmail.com", "sweetwishes.com", "sweetxxx.de", "swift10minutemail.com", "swift-mail.com",
            "swissinfo.org", "swissmail.com", "swissmail.net", "switched.com", "switzerland.org", "sylvannet.com",
            "syom.com", "syriamail.com", "szerz.com", "t2mail.com", "tabasheer.com", "tabouk.cc",
            "tafmail.com", "tafoi.gr", "taglead.com", "tagmymedia.com", "tagyourself.com", "tajikistan.cc",
            "talk21.com", "talkcity.com", "talkinator.com", "tangiers.cc", "tangmonkey.com", "tanta.cc",
            "tanukis.org", "tapchicuoihoi.com", "taphear.com", "tatanova.com", "tattoodesign.com", "taxcutadvice.com",
            "tayef.cc", "taylorventuresllc.com", "tb-on-line.net", "teachers.org", "teamster.net", "tech69.com",
            "tech-center.com", "techemail.com", "techgroup.me", "techie.com", "technisamail.co.za", "technologist.com",
            "teenchatnow.com", "teenmail.co.uk", "teenmail.co.za", "teewars.org", "tefl.ro", "tejary.com",
            "telebot.com", "telecomix.pl", "telefonica.net", "telegraf.by", "teleline.es", "telenet.be",
            "teleosaurs.xyz", "teleworm.com", "teleworm.us", "telinco.net", "telkom.net", "telpage.net",
            "telstra.com", "telusplanet.net", "tempail.com", "tempalias.com", "tempemail.biz", "tempe-mail.com",
            "tempemail.com", "tempemail.co.za", "tempemail.net", "temp.emeraldwebmail.com", "tempinbox.com", "tempinbox.co.uk",
            "tempmail2.com", "tempmail.co", "temp-mail.com", "temp-mail.de", "tempmail.de", "tempmaildemo.com",
            "tempmailer.com", "tempmailer.de", "tempmail.eu", "tempmail.it", "temp-mail.org", "temp-mail.ru",
            "tempmail.us", "tempomail.fr", "temporarily.de", "temporarioemail.com.br", "temporaryemail.net", "temporaryemail.us",
            "temporaryforwarding.com", "temporaryinbox.com", "temporarymailaddress.com", "tempr.email", "tempsky.com", "tempthe.net",
            "tempting.com", "tempymail.com", "tenchiclub.com", "tennessee.usa.com", "terrapins.com", "testudine.com",
            "tetouan.cc", "texascrossroads.com", "texas.usa.com", "tfwno.gf", "tfz.net", "thai.com",
            "thaimail.com", "thaimail.net", "thanksnospam.info", "thankyou2010.com", "thc.st", "the5thquarter.com",
            "theaviors.com", "thebearshark.com", "theblackmarket.com", "thecloudindex.com", "thediamants.org", "the-fastest.net",
            "thegame.com", "thegamefanatic.com", "theinternetemail.com", "thelimestones.com", "thembones.com.au", "themostemail.com",
            "theoffice.net", "theplate.com", "thepostmaster.net", "the-quickest.com", "theracetrack.com", "therapist.net",
            "thereddoors.online", "thescrappermovie.com", "theserverbiz.com", "theteastory.info", "thewatercooler.com", "thewebpros.co.uk",
            "thex.ro", "thietbivanphong.asia", "thinkpost.net", "thirdage.com", "thisisnotmyrealemail.com", "thismail.net",
            "thisurl.website", "thnikka.com", "thraml.com", "thrma.com", "throam.com", "thrott.com",
            "throwam.com", "throwawayemailaddress.com", "throwawaymail.com", "throya.com", "thundermail.com", "thunkinator.org",
            "thxmate.com", "tic.ec", "tightmail.com", "tijdelijkmailadres.nl", "tilien.com", "timemail.com",
            "timgiarevn.com", "tim.it", "timkassouf.com", "timor.cc", "tinati.net", "tin.it",
            "tinoza.org", "tinyurl24.com", "tiscalinet.it", "tittbit.in", "tiv.cc", "tizi.com",
            "tjohoo.se", "tkcity.com", "tkitc.de", "tlcfan.com", "tlen.pl", "tlpn.org",
            "t-mail.com", "tmail.com", "tmailinator.com", "tmail.ws", "tmicha.net", "tmpboxnet.me",
            "tmpjr.me", "t-net.net.ve", "toddsbighug.com", "todito.com", "todoperros.com", "toiea.com",
            "toke.com", "tokem.co", "tokenmail.de", "tokyo.com", "tokyo.ir", "tombstone.ws",
            "t-online.de", "tonymanso.com", "toomail.biz", "toothandmail.com", "toothfairy.com", "top101.de",
            "top1mail.ru", "top1post.ru", "topchat.com", "topikt.com", "topmail.co.ie", "topmail.co.in",
            "topmail.com.ar", "topmail.co.nz", "topmail.co.uk", "topmail.co.za", "topmail.dk", "topofertasdehoy.com",
            "topranklist.de", "toprumours.com", "topsurf.com", "toquedequeda.com", "torba.com", "torchmail.com",
            "tormail.org", "torontomail.com", "toss.pw", "tosunkaya.com", "totalfoodnut.com", "totallyintobaseball.com",
            "totallyintobasketball.com", "totally-into-cooking.com", "totallyintocooking.com", "totallyintofootball.com", "totallyintogolf.com", "totallyintohockey.com",
            "totallyintomusic.com", "totallyintoreading.com", "totallyintosports.com", "totallyintotravel.com", "totalmail.com", "totalmoviefan.com",
            "totalsurf.com", "total-techie.com", "totalvista.com", "totesmail.com", "totonline.net", "tough.com",
            "toughguy.net", "tp-qa-mail.com", "t.psh.me", "tqoai.com", "tradermail.info", "tranceversal.com",
            "trash2009.com", "trash2010.com", "trash2011.com", "trash-amil.com", "trashcanmail.com", "trashdevil.com",
            "trashdevil.de", "trashemail.de", "trashinbox.com", "trash-mail.at", "trashmail.at", "trash-mail.cf",
            "trash-mail.com", "trashmail.com", "trash-mail.de", "trashmail.de", "trashmailer.com", "trash-mail.ga",
            "trash-mail.gq", "trashmail.me", "trash-mail.ml", "trashmail.net", "trashmail.org", "trash-mail.tk",
            "trashmail.ws", "trash-me.com", "trashymail.com", "trashymail.net", "trasz.com", "travel2newplaces.com",
            "trav.se", "trayna.com", "trbvm.com", "trbvn.com", "trbvo.com", "trevas.net",
            "trialmail.de", "trickmail.net", "trillianpro.com", "trimsj.com", "tripod-mail.com", "triton.net",
            "trmailbox.com", "troamail.org", "trollproject.com", "tropicalbass.info", "trungtamtoeic.com", "tryalert.com",
            "tsamail.co.za", "ttszuo.xyz", "tualias.com", "tunisian.cc", "tunome.com", "turbonett.com",
            "turkey.com", "turoid.com", "turual.com", "tushmail.com", "tut.by", "",
            "tvchannelsurfer.com", "tvchd.com", "tverya.com", "tvnet.lv", "tvstar.com", "twc.com",
            "twinmail.de", "twoweirdtricks.com", "txtadvertise.com", "tyhe.ro", "tyldd.com", "typemail.com",
            "u.0u.ro", "u2club.com", "u2tours.com", "uae.ac", "ua.fm", "",
            "ubbi.com", "ubbi.com.br", "ubismail.net", "ubm.md", "uboot.com", "ucche.us",
            "ucupdong.ml", "ucupkoh.ml", "uemail99.com", "ufacturing.com", "ugeek.com", "uggsrock.com",
            "uguuchantele.com", "uhhu.ru", "uk2.net", "uk2net.com", "ukr.net", "ukrpost.net",
            "ukrpost.ua", "uk.to", "uku.co.uk", "ulimit.com", "ultimateredskinsfan.com", "umail.net",
            "ummah.org", "umpire.com", "unbounded.com", "undo.it", "unendlich-schlau.de", "unicum.de",
            "unids.com", "unimail.mn", "unimark.org", "unit7lahaina.com", "unitedemailsystems.com", "universal.pt",
            "universiabrasil.net", "universia.cl", "universia.edu.ve", "universia.es", "universia.net.co", "universia.net.mx",
            "universia.pr", "universia.pt", "unmail.ru", "unofree.it", "uol.com.ar", "uol.com.br",
            "uole.com", "uolmail.com", "uomail.com", "uorak.com", "upliftnow.com", "uplipht.com",
            "uploadnolimit.com", "uraniomail.com", "urbi.com.br", "urdun.cc", "ureach.com", "urfey.com",
            "urfunktion.se", "uroid.com", "usa.com", "us.af", "usanetmail.com", "userbeam.com",
            "us.to", "utah.usa.com", "utiket.us", "uu.gl", "uwork4.us", "uyhip.com",
            "uymail.com", "uyuyuy.com", "v3mail.com", "vaati.org", "valemail.net", "valhalladev.com",
            "vankin.de", "vctel.com", "vda.ro", "vdig.com", "vegetarisme.be", "velnet.com",
            "velocall.com", "venompen.com", "vercorreo.com", "verdejo.com", "verizonmail.com", "vermont.usa.com",
            "verticalheaven.com", "veryday.ch", "veryday.eu", "veryday.info", "veryfast.biz", "veryrealemail.com",
            "veryspeedy.net", "vesa.pw", "vfemail.net", "victime.ninja", "victoriantwins.com", "vidchart.com",
            "videogamesrock.com", "viditag.com", "vietmedia.com", "viewcastmedia.com", "viewcastmedia.net", "viewcastmedia.org",
            "vikingsonly.com", "vinernet.com", "vip.126.com", "vip.163.com", "vip.188.com", "vip-client.de",
            "vipepe.com", "vip.gr", "vipmail.name", "vipmail.pw", "vip.qq.com", "vip.sina.com",
            "vip.sohu.com", "vip.sohu.net", "vipsohu.net", "vip.tom.com", "vipxm.net", "viralplays.com",
            "virgilio.it", "virginia.usa.com", "virgin.net", "virtual-mail.com", "visitmail.com", "visto.com",
            "vitalogy.org", "vivelared.com", "vixletdev.com", "vjtimail.com", "vkcode.ru", "vmailing.info",
            "vmani.com", "vmpanda.com", "vnn.vn", "vodamail.co.za", "voidbay.com", "voila.fr",
            "volkermord.com", "volunteeringisawesome.com", "vomoto.com", "vorga.org", "vosforums.com", "votiputox.org",
            "voxelcore.com", "vpn.st", "vps30.com", "vps911.net", "vrmtr.com", "v-sexi.com",
            "vsimcard.com", "vsnl.com", "vsnl.net", "vssms.com", "vubby.com", "vztc.com",
            "w3boats.com", "w3internet.co.uk", "w9y9640c.com", "wakingupesther.com", "walala.org", "walkmail.net",
            "walkmail.ru", "walla.co.il", "walla.com", "wallet.com", "wallm.com", "wam.co.za",
            "wanex.ge", "wants.dicksinhisan.us", "wants.dicksinmyan.us", "wapda.com", "wap.hu", "wapicode.com",
            "wappi.com", "warpmail.net", "washington.usa.com", "wassup.com", "wasteland.rfc822.org", "watchever.biz",
            "watchfull.net", "watch-harry-potter.com", "watchironman3onlinefreefullmovie.com", "waterloo.com", "waumail.com", "wayintocomputers.com",
            "wazmail.com", "wbml.net", "w.cn", "wearab.net", "web2mailco.com", "web2mail.com",
            "webaddressbook.com", "webbworks.com", "webcity.ca", "web.de", "webdream.com", "webemaillist.com",
            "webemail.me", "web.id", "webindia123.com", "webinfo.fi", "webjump.com", "webl-3.br.inter.net",
            "webm4il.info", "web-mail.com.ar", "webmail.co.yu", "webmail.co.za", "web-mail.pp.ua", "webmail.ru",
            "", "webmails.com", "webmailv.com", "webname.com", "web.nl", "webpim.cc",
            "webspawner.com", "webstation.com", "websurfer.co.za", "webtopmail.com", "webtribe.net", "webtrip.ch",
            "webtv.net", "webuser.in", "weedmail.com", "weekonline.com", "wee.my", "wef.gr",
            "wefjo.grn.cc", "wegwerfadresse.de", "wegwerf-email-addressen.de", "wegwerfemailadresse.com", "wegwerf-email-adressen.de", "wegwerfemail.com",
            "weg-werf-email.de", "wegwerf-email.de", "wegwerfemail.de", "wegwerf-email.net", "wegwerfemail.net", "wegwerfemail.org",
            "wegwerf-emails.de", "wegwerfmail.de", "wegwerfmail.info", "wegwerfmail.net", "wegwerfmail.org", "wegwerpmailadres.nl",
            "wegwrfmail.de", "wegwrfmail.net", "wegwrfmail.org", "weirdness.com", "welikecookies.com", "westvirginia.usa.com",
            "wetrainbayarea.com", "wetrainbayarea.org", "wg0.com", "wh4f.org", "whale-mail.com", "whatiaas.com",
            "whatifanalytics.com", "whatisthis.com", "whatmail.com", "whatpaas.com", "whatsaas.com", "when.com",
            "whiffles.org", "whipmail.com", "whoever.com", "who.net", "whopy.com", "whyspam.me",
            "wibblesmith.com", "wickmail.net", "widget.gg", "wild4music.com", "wildaboutelectronics.com", "wildcats.com",
            "wildmail.com", "wilemail.com", "willhackforfood.biz", "williams.net.ar", "will-keinen-spam.de", "willselfdestruct.com",
            "wimsg.com", "winemaven.info", "winning.com", "winningteam.com", "wins.com.br", "winwinhosting.com",
            "wisconsin.usa.com", "witelcom.com", "witty.com", "wmail.cf", "wolfsmail.tk", "wollan.info",
            "wolverines.com", "wooow.it", "worker.com", "workingaroundthehouse.com", "workingonthehouse.com", "workmail.com",
            "workmail.co.za", "worldcrossing.com", "worldemail.com", "worldmedic.com", "worldonline.de", "worldspace.link",
            "wowmail.com", "wpg.im", "wp.pl", "wprost.pl", "wralawfirm.com", "wrestlezone.com",
            "writeme.com", "writeme.us", "writesoon.com", "wronghead.com", "wrongmail.com", "wtonetwork.com",
            "wurtele.net", "wuzupmail.net", "wuzup.net", "www.com", "www.consulcredit.it", "www.e4ward.com",
            "www.gishpuppy.com", "www.mailinator.com", "wwwnew.eu", "wxnw.net", "wyoming.usa.com", "x24.com",
            "x5a9m8ugq.com", "xagloo.co", "xagloo.com", "xasa.com", "xcompress.com", "xcoxc.com",
            "xcpy.com", "xemaps.com", "xents.com", "xfreehosting.com", "xf.sluteen.com", "xing886.uu.gl",
            "x.ip6.li", "xjoi.com", "xl.cx", "xmail.com", "x-mail.net", "xmail.net",
            "xmaily.com", "xmasmail.com", "xmsg.com", "xn--9kq967o.com", "xnmsn.cn", "xoom.com",
            "xoxox.cc", "xoxy.net", "xpectmore.com", "xrea.com", "xrho.com", "xsmail.com",
            "xtra.co.nz", "xuite.net", "xwaretech.com", "xwaretech.info", "xwaretech.net", "xww.ro",
            "xyzfree.net", "xzapmail.com", "xzsok.com", "y7mail.com", "ya.com", "yahala.co.il",
            "yaho.com", "yahoomail.com", "yalla.com.lb", "yamal.info", "yam.com", "yanbo.cc",
            "yanet.me", "yapost.com", "yapped.net", "yaqp.com", "ya.ru", "yawmail.com",
            "ycare.de", "ycn.ro", "yeah.net", "yebox.com", "yedi.org", "yehey.com",
            "yellow-jackets.com", "yellowstone.net", "yemeni.cc", "yenimail.com", "yep.it", "yepmail.net",
            "ye.vc", "yhg.biz", "yifan.net", "yk20.com", "ymail.com", "ynmrealty.com",
            "yodx.ro", "yogamaven.com", "yomail.info", "yoo.ro", "yopmail.com", "yopmail.fr",
            "yopmail.gq", "yopmail.net", "yopmail.pp.ua", "yougotgoated.com", "youmail.ga", "youmailr.com",
            "youneedmore.info", "yourdomain.com", "yourewronghereswhy.com", "yourlms.biz", "your-mail.com", "yours.com",
            "yourwap.com", "you-spam.com", "ypmail.webarnak.fr.eu.org", "yroid.com", "yspend.com", "yugasandrika.com",
            "yui.it", "yunus.cc", "yuurok.com", "yxzx.net", "yyhmail.com", "z0d.eu",
            "z11.com", "z1p.biz", "z6.com", "z86.ru", "za.com", "zagazig.cc",
            "zainmax.net", "zain.site", "zambia.cc", "zasod.com", "zebins.com", "zebins.eu",
            "zednet.co.uk", "zeeman.nl", "zehnminuten.de", "zehnminutenmail.de", "zepp.dk", "zetmail.com",
            "zfymail.com", "zhcne.com", "zik.dj", "zipcad.com", "ziplip.com", "zipmail.com.br",
            "zipmax.com", "zippymail.info", "zipsendtest.com", "ziza.ru", "", "zmail.pt",
            "zmail.ru", "zoaxe.com", "zoemail.com", "zoemail.net", "zoemail.org", "zoetropes.org",
            "zoho.com", "zombie-hive.com", "zomg.info", "zona-andina.net", "zonai.com", "zoneview.net",
            "zonnet.nl", "zooglemail.com", "", "zoomshare.com", "zoznam.sk", "zp.ua",
            "zubee.com", "zu-geil.de", "zumpul.com", "zuvio.com", "zwallet.com", "zworg.com",
            "zxcvbnm.com", "zxcv.com", "zybermail.com", "zzn.com", "zzz.com",
            // free email providers end
        );
    }
}
