
#include <hz.h>

#define GLFW_INCLUDE_NONE
#include <GLFW/glfw3.h>

#include "hz_gl3.h"


#include <errno.h>
#include <stdlib.h>
#include <stdio.h>

#define WIDTH 1280
#define HEIGHT 720

#define ARRAYSIZE(x) (sizeof(x)/sizeof((x)[0]))

static int load_font_face(stbtt_fontinfo *fontinfo, const char *path)
{
    uint8_t *filedata;
    FILE *file;
    size_t length;

#ifdef _WIN32
    errno_t errno;
    if (errno = fopen_s(&file, path, "rb"), !errno) {
#else
    if ((file = fopen(path, "rb")) != NULL) {
#endif//_WIN32
        fseek(file, 0, SEEK_END);
        length = ftell(file);
        fseek(file, 0, SEEK_SET);

        filedata = (uint8_t*)malloc(length);
        fread(filedata, 1, length, file);
    } else {
        fprintf(stderr, "Error: failed to open file \"%s\" (errno %d)\n", path, errno);
        return 0;
    }

    if (!stbtt_InitFont(fontinfo, filedata,0)) {
        fprintf(stderr, "Error: failed to initialize font info!\n");
        return 0;
    }

    return 1;
}


//   (-1,1)       (1,1)
//   +-----------+
//   |\          |
//   |  \        |
//   |    \      |
//   |      \    |
//   |        \  |
//   |          \|
//   +-----------+
//   (-1,-1)      (1,-1)
//  
//
/*
GLuint create_fs_quad_shader()
{
    static const char fs_quad_vert[] = "#version 450 core\n"
    "#extension GL_ARB_separate_shader_objects : enable\n"
    "const vec2 fs_quad_verts[6] = vec2[6](vec2(-1,-1),vec2(-0.5,-1),vec2(-1,-0.5),vec2(-1,-0.5),vec2(-0.5,-1), vec2(-0.5,-0.5));\n"
    "const vec2 fs_quad_uvs[6] = vec2[6](vec2(0,0),vec2(1,0),vec2(0,1),vec2(0,1), vec2(1,0), vec2(1,1));\n"
    "out vec2 fragTexCoord;\n"
    "void main(){\n"
    "    gl_Position = vec4(fs_quad_verts[gl_VertexID],0.0,1.0);\n"
    "    fragTexCoord = fs_quad_uvs[gl_VertexID];\n"
    "}\n";

    static const char fs_quad_frag[] = "#version 450 core\n"
    "#extension GL_ARB_separate_shader_objects : enable\n"
    "in vec2 fragTexCoord;\n"
    "uniform int has_texture;\n"
    "uniform sampler2D uMainTex;\n"
    "layout(location=0) out vec4 outColor;\n"
    "void main(){\n"
    "    if (has_texture == 1){\n"
    "        outColor = vec4(texture(uMainTex,fragTexCoord).xyz,1.0);\n"
    "    }else{\n"
    "        outColor=vec4(1,0,0,1);\n"
    "    }"
    "}\n";

    GLuint vert_shader, frag_shader;
    vert_shader = hz_gl3_create_shader(fs_quad_vert, GL_VERTEX_SHADER);
    frag_shader = hz_gl3_create_shader(fs_quad_frag, GL_FRAGMENT_SHADER);
    
    GLuint program = hz_gl3_create_program(vert_shader, frag_shader);
    glDeleteShader(vert_shader);
    glDeleteShader(frag_shader);

    return program;
}
*/
#define GL_CHECKERR(id) { GLint err; while ((err = glGetError()) != GL_NO_ERROR) printf("%s: %d\n", id,err); }

static float min_zoom = 0.0f;
static float max_zoom = 10.0f;
static float zoom_factor = 0.0f;
static float cursorx = 0.0f;
static float cursory = 0.0f;
static float camerax = WIDTH/2;
static float cameray = HEIGHT/2;
static float half_height;


void scroll_callback(GLFWwindow *window, double xoffset, double yoffset)
{
    zoom_factor += yoffset/32.0f;
}

float get_linear_zoom()
{
    float log_min_zoom = log(1.0f + min_zoom);
    float log_max_zoom = log(1.0f + max_zoom);
    float log_zoom = log_min_zoom + (log_max_zoom - log_min_zoom) * zoom_factor;
    return exp(log_zoom);
}

void cursor_callback(GLFWwindow *window, double xpos, double ypos)
{
    double dx = xpos - cursorx;
    double dy = ypos - cursory;

    if (glfwGetMouseButton(window, GLFW_MOUSE_BUTTON_LEFT) == GLFW_PRESS) {
        float zoom = get_linear_zoom();
        camerax -= dx / zoom;
        cameray += dy / zoom;
    }

    cursorx = xpos;
    cursory = ypos;
}

int main(int argc, char *argv[])
{
    hz_config_t cfg = {
        .flags = 0,
        .log_severity = HZ_LOG_SEVERITY_ERROR,
        .ucd_version = HZ_MAKE_VERSION(15,0,0)
    };

    if (hz_init(&cfg) != 0) {
        fprintf(stderr, "Failed to initialize Hamza!\n");
        return EXIT_FAILURE;
    }
#if 1
    static const char *noto_sans_arabic_ui_woff2_base85 = "G>C[`!!*\'\"!!)rX!\"o83!!:r/!!)qB!!*\'\"zzzzz)N@!$JY#=NcsR&c5?(TO6%aXeP7@jP$(>]C#RZ=ZFG?\\f$^iIE!B(!)\"0t`++9d`o#D\'8b%\"TL)G,LK2eRE?,eN<d?W=\"8M[h/XJ2Kb<g*UeE04CYc\'`V2F6<nmh=W%_^g@,&$pI^b;\'Sm(idhk0lIs8W-!s)1S[KRYoOZ?`dk,SCfL&h6jRHe7*\\NS8^ePfT)Y[[X,%cT/tg-o!VS1ScA\\S7J3,.<YU[V-V\'][2Aj3$70SJf*ja7VsIA2@ZpJc\\!VIjo\'0f%0,YZCj`!VYXV$9@P^,ed&s?88V)ia3$2H]tm:Ai`6MISF,qhm>7/^:t;ij#s>2)q,kn,4D]2<(\"f9Q%NNg\'`F#&+tR\'K7pH?XY%4cJB[^OG\"l)X6,@YAs0-0U\'XCG8iqick@M8g&BMTQI#uWW0ekD8:%LPpIG4L&-u++1&jU@J&M\"G`0Yp-.2CU4WY&?P%PiQ/k%t.K6^-@\'1SuSdf6KdZ+I6H-EQ.L54,i0_dn6KHIKuHW^;7c\\9[q^B&c=Sl#I:[%A3^2l*DHT-X-&4#KL4&sdr[)inh[?jDeAP[7!Dd0i5P,]JjG&Iapmemd_nTs]dADD8Z-\'gB,1;VR3K@\"e`mF9eOG\"m*oEP4=L,3c7e^n(Bi4VuBX6V,s6+aXFiN`<T&F1.krUEYf4a3!Y0+P5I5B%u]%1\\)\"2$k;t:H7f()!^i;nosBKpH]FP13[RFLej,6b?I4OA)i/E<?d`BA!]C26p/JiII``opsRU),[6jolo)Pi?3uf?,UbrNYiF\\VDZASLm04dU-rOQE3+rJ\\CiNDk<.%(bCg\'+sUtQ/@ADk7[Jn.seplT6bQu#.$q0s3Z@2:[CL_YM\"_+H=Gr\':cYrR!e`hnDg^_e>q-TA5;\'IdNbOY?kjR/j8t1@%dm4U&cR%Oc^t?jWi1J<9/fPTXbf\\>MODV/YZ>LA&,*P6%r8DJq@*<A+QnVTYC-Sq<Ijk<H%QFhsCD^\\*r0V+YNAFAB`6LiN*dBYh!q=EU7_fQtM\\0cIcd*ZBNi/c*P\\u,9camZ<PJe>\\VUeHQY>B3`S49mPr\"$!W;,larq\'Meo!VMr-beVqs8B2PGNPh=OAg+!g?JD!G)iJ\'&`7LGb=\\Y43:4Z^TDN^Es`?h&\"[Z]/_4>JldbZ+;*mgu-lY&Ik^u7.n*VE6K@3UoV^V/d<?m8(F)\'.Z:YJp:@X1\\Z1lbe<\"(k1U4!%njn_E.n#Q_N-Cr>jTdsQXQJVAid=-BtSAS1b*-bs#.AZ.$;.Rfo&U8#1`,muYYT9e!%*9r^pN\'p9k\'K+7P*VbE)TDSKa$7H7+,\\9CN2EQ5f0==Vgl#84aX#WCO`h.7;.O9o-1?-,@g_``!ArtJnLko2H@^_(`.O$&\\Asjr2`9b[1Ha1c/QV6&dF^91DA_2\']\'Tlet&O]%R[bJQ71D%1B5g2J]/d&Y%STq;G(_403@I?YJfM](U>87X?BAM1Rc$O>9/<%t&?:^F!&b)J+rU7p1mlB\"5ba4`_<?>Mo`\\DRrFl1%MFO2d:I>8/dhiZ\")[%\'L<#q^Sl[?bGc]7m]IXT+>BldM&f38iGT;$N=HG&Se]b^r#4c5lsCjV85i+0R1)ea8c,q!hZKnTt2`!ZBDdchn\"7&fdO\"HtG63D%%8iCa;)<Rs(#oijk+?.BG\"V@!#B](qPFl4!4]jU4q3O4ANg!Jo8G#]CRj[Oru<>3`C=@%U8T/EX5Mhfo>5I\"fJ%-C]3](M:/%oP*G<57.8uif\\,\\pR,HO565$33h+J\"6!Eg=Ci,QrdT8kq8h4!D?L@!hQ:(*@aG$d,K$U?kiQjU@44*1G;779ODnSoii?50DWm@P>7=,73WZcH0oT_R92Y)u^IF$ql1`&!Q!,RE>:-Z3m23aik[@5s!m\"&E1>ao?6e=98W`Dq2.>0&@J7>f2pW.A)496[hao)l@F8GBk1An<s$h8+=8a17kMBiqk3k/^Xm^bk^e<ZG;s7k$\\sJ$^\\\\@^#:UcSh%OanEsi4bj)H5\"q[$pHU,kXqF\\1iJZ0;E<\")TkP:X]BM?ABaR2cKKrmr\\7S$`8p9BbeI<e)f!5ZQ@t!U[l^$me\"=6\'tA1f[df]4aJ>J]Sf19P<\'l`EU`_!l@7Q8)[+]a*!jafS\'mckqNYg:+6A>`P+n`Y0e[0n52@\"k[kAX$\\d9:+PXn\'&)>([8OHN%/f&K1$<.fj7=P!7=rq<u9\'+)gi7+-q!?4*E5MR%_f[XUo5@Z]Ki@Ie/J<b\\h-h5k\\QXf#bgb?8_(nG?26G!7XBGJL&6,os4MZ#`dAnl&]6_kSX\\3u:FQJP_)1W#a#2Y:L\\dG?3,EnC0,gT1s`_J@UcbUE\"uQ1Y_lfRA0Di-DW3:mZLKD-mT^>JcGb9s39,s1PG0f[Jr0k$\\rg`a03U5LgB:gWAdTbP0O\\n_q>_L/n?+[\"sGf9-Sk<iCSdTi>3s.p$_WIK)r]f;!l+V&]lQ\"R-6&rEkl8l\'m0kdlJcf/73/)JZVc]0FN]^C+8llc)B0KZV$n!VC[20Mi6HYRaQZLf%8<lMQGBW)BF_(Vk1><1RfI%MgZ<55WKP.?Q+:qL2%N7R9>O)9nScUFX`rN1H&utJX1&13Ap9XeshB\'NE$%tUT]<)\'IfMJJ:Z]:-;W%$0\\L@\"r<Y.)q8\\em#;UN#3SUs;]@+6TA?eY+qnX;.0/lluCio\'(u@.@#\'T\\LtSQghX+\\OqB-$AjTc]Q3u)g\\rK=4E.BS@F#GpgDakuO4F]^.Z3,nSHeAr!!X:FC,rYeP<Z1eihdmg9A=t>UT\\Peo$.kW3hNF`uI/k7TW1VpR(aI7m63?q+ro!J)5e@JZ(Q&>CTdqGaos*9.^?R2am*TN.8<=BN$7Xas`-!@/kAAIdrf-B$+2u^b]jH-dhG6O-,_Q^SLklINOJ*YAmZeXtajRS<OWp?RUBLB4Dtr7Amb?;_RP_kA0kH#Fl?cIAm>l)\\rs]!I7!SrEoWUlEF1),?IKg@bD5cr*XeotQj/0)gs.GNsRpLI,DVBE2TVA0TJ0:;Vn\'?B+der96cciPap%2mrXr4=WcSELn-q^Tu.Mmq)\"YErW-Pt3O11(\"<TDO*)E3\"N7?`&B77$JNTWu(.ZoCn^/pMZH*#bh;O!9-Kk8<E+#T_eIKTLLsTW/!hOBGseL[.%7M`3cBB@p\\bXZ)C+,GcbhUZ+U5^Z+GX]Z*,E\\m-c35I+Ba.ptUn/!>;/p\"(F4_5iS@0*9o-*r::2S+Ckk6EKC58s\'Q6@O6#O+J>g$k=tn5&<7Z?aLj]8jBIr:J(tjunB6Gn9d=MP-IB;K9X+iWk)DHk9kmXYU6SgWcWT)OG!W).qdUVSr^k%kiDb30P_.t.?\"ZSi;2JFkaWMI<H?c%B&WtG=^.\\*Oa)?mQt_:uj%$sf;#\'./Y+,p7O!HAl6r@M:A)Yq0\"38PEFA-r>Jj#,!e;&nhMU8<-k?@5A!4C\"l0E#!t^u*X&@M=->,pi1_MKZ$Yo`/f;-fL(6SL1j8!hh:53Njdf/*!ak6^LH\"/0mYGFFM93%\\5PKn1?Fk<YG=Vn,!T,^<+CLaVnpdV,5d5b@,Ha@\"O+Qkkfd@kr87jABUOPUq:htpikL31W<8s\"0C0qCs!KHi9/ckY3lM7=`J$T\"Khim/22(Q\\+9qaW%P9F,`\'Z@QUH>RKPlPU4&/U:p!#8-\'5Z\\5,4.RR*A6iLE$qMFr6j!/`1)afmqLuojcmb<r\"c-*9U\"c<;sLs+me)\'E/\\h_B``VutZZ6t\'<H-E]nB<,;.rZc2eq%%cpd;!X\'@@NN2ORb)t]@pF?.<h9Pge*R`Ud?Cm.S>,ET<PqIuVH7jkY8Q\'N0QhbRM`M*fF;Z*?*EV/7YK0@lep_Qelk`YV^M?L;b>;sOD^j^t)aEhL-P!iK<>OH]?Cc?87\"+=Y&L*5-95iK$JcMf!lTmFR`,l6Yqn<CC\"JT9(aF\".6+J=KS58\",Fh_93URC<FV!Xo_@dXs9jRT0Ujc6FM5fpB7^MS,2fC\\-oi,o?kp#X`Sf\'M\'MuG1g+EP^,IRd1C./<o20`P+CBQMq4Y)^CeeOq/\"JY+hVi;Jj<a8\'Fmp%.[4,-dEWd96fCRX^fL*#htYuV7i2soMTFt=3)i-6U(r2m^;K`Dq/LWk!^mWOZ5Be3Q$q,gB&]]t=b?]V?8o5@5$KTfKQjq8TC39LlqJmGnVhDCjaFl(CGt`(!71`<!#*-k:4dmacH91KHIf.RorVR\'qQ)tO2n1O&n1SFkD=3kik+R\':Ssl!EP7Lg?9%J\'@\',1s`\\#E9#6\'DdGjt\'W>[HVV5JHsL&P5q<Pq;&e^%=YKMD^22hA13[>NV\\c-?\"Q<aTBdl9YM8p;>LIS>BFjck2<adLd-jbgrm:O%51di$0,SBXA[JLFqKhB8c:pKGFSY\'*XM7.o_`C.O+rcHr#AS#;062Hoe>QTp]P/;@i4M17JZgjg$TpS:dHteQZ$K:eUS4oD-j;=QCK?1Y5G=@S=e29q@R(?l8j<bL67\'c!ak0L`Q>@J&3iuJm8\"(?SXUC*U\\1VnbU/\"(7h!L*>S99J0V8]L9(QCk>!`fQ$L\'201!P_/!e4pJ\\b\\ga\"gr\\=@/<iM1\\d*ML:u5d#GLj$IlZ!P3/Fb8;S[JBG:h5:[SV8fu)gEMN:H)skYiJ#OX,G\\l-5BKZBt^7;jk>p2+#&A584>nC!I\'MrZ413o21g#X?Ap\'rFST##=?mIRF`.dNJOfLS%Z5QJS\"+R_%Qe1DR9k7^cg-I\'p2nCWkrN/JoD.JW:JjIg$OAu>8MM%V]#YXM_hGc,o+P]]Ka@hSQ13+UB&n0P$YeGs\'2H6%1r5JgaY<\'\"Z$64r6;6iPJq:NKD?dUL>\"#?<XY9_V5bX\\]-LAY,1,1XtLUpiD-?rKC9Pf#lOs6&$-J6*s0AE!c%#hV@Y<DBpXnUE.@P\'_ORTg%O0=1j]#Tb0V;H08u*c:\\9KgA&OVO;QH9Pe8Vq`2MbcFLL9PAqMD1<N>9N7%V\\[t=iBouuSqGO-WlMcLW$h_3r_$U1Z.Ruh(_C?lsUA\"*Y?_,e][BKQ3e:eq;_QhI2(Ic[Ai=q\"ASkL_QL2i0hqaS%b?CI7X)JCUD`X1\"4GS)jmQme0i18]U5S=F9#++lDf6qXKW*]4BVQIr(aFPP?%T$Qpa$]2_.!:UQf/5YeC9SY8MT.&ZXZ$E\'s#`1d>2W%(_K4:bL3<(r\'ODD$EEclc5JR*?Ou#6)qQnj)<&Q8uU8&2-Zt!aePt#k2[V\"4!6iNNFb:#.E^f,GmRuP0K+BR\\N;XH-2`kF\"IN=rL_FSPE4so1TB1%ifPl(@=F;sa?es7elqNJSSQ#;^3L0K>*W>jmcLU!9d6+O2Oc)\"Oa3$Y7b0L]D6K7oOYL\"AL&Cmi?BTckhW8]oBP4IlRBPaR:*ktiI\'WW+HC#5=Zs8=(oVRIXjSZKQ]:>\"FE^^peZQ[Xj!6R(5f8VDojk3q:OE(s7^%0*cXn7\'f>>#`e]V,sV8gu[!_]?#qXn?8I664uVS&UOpVE!`Y^sB1]l4L;C+6)*Z;mpR\'M.8G6g00=gWMGp4`O5l/*//_tV5@\"`+RqM=UQ\\D75/(W6OtXXs@qXHIbIF6DU+O[Gju^=]I<*\\Pr=UPM&-4aur*>JmjU=!Y`@q[A)Te\"oPA5:j^fXhG1`786UXJ]>Zc5+q(HgN:O/L._QcR!c<8tXMrJ63j7;bq]^C,:D<E\"2.8=HP@3\"NN8[;jaUM[TW0O3<g6W\'Q,F,P/(AD)M+^KTDOrUpZ+\\^P]\'F!J5eHJj-1REjI;3pT+>LTAKU_l#h2^bl5ld]5P8tH1nb;Q2\'$/?Z#6\'R,l,[I#q7<?o3S<0GKHD2mO2c>b9mMF4\"GSck5Bn]ggUJ\"-)XTd\'VWP\\*cNu-M3F`;gKo;f?1*+DWhA)N<iB.$2p$[F*CmglTo73\\3&9PFZBTd3c&M3cY(9Vpe\\.?*Db\"(\"E*P14r8Ppo$r,bEQ;XAmNkJ,>sQehHAuD%lJ,c.chqS61b#X9/8]Wk,lLl=3.(Xi10MW=)E%T#qRNR,6UV$.#>.q!>\\U];\\`s_8%Wn*6Ro-@N557o:oE!n/B6LI\":\"@tuK0#_h,B]NL\"[dMIoGeu%2GK++RsC69C*BbG01X=6iKcS+qKK[8\"\')Ou/qV6t@4MPFX\\E\"-BL3ZF=Ut_1a/48`>CUZ20&qXjAr_CgIT\"RHc7\'^)1=imOmB5j^F2OnEp.-CJo/tXB\\+[!DqNIT%1o6C.11sG6LcQOgjT7o:iX<mu[6U.iSdi8HB`U11LTQBpOB\'`%di]_HLnE[U\":`5$?jW,XeD$]9Z-<d$@N^cWN.<<*kKftENbtf=!<ArPY!@4lW/UW4-?i&QJ/,.tI6bs$$!6S:/tHomD\\HM-\\J./g:Q5hrg`Grs8>r3fmdCQ`;FM&,fM=ZK%@1@1\\bG:KIe&R1LJ9bF*S`aK37%(<*H5)#At303pk[6fE!l<4TD5a5`-e.<_ZZ/KR6gif!W\"))(\"LR74@\\0)4F,?M\"^*i%n<O+$</8H(GjQD_Ou`!\"+-rLl6)u&tS81rID3@Ip[7gC>A>0&-PCuI,5OdqlF2AgtAs&^[WT=),fh_@=Q[5F2,5O@<!gC]<@$PSZo$t5*PN\"(tY_eEch/HW21KBl^Td?#:rkL+tX5.42\\uWT,;-RkEm4/\"X5/qEJe*cbBYb7O)r6un)o7WmQ-0Fuq,>NmCn.1cVK\\j,)l%])N!\\A)J7M4a=<7.KlrIM6tE(5kUG@YV%j`o((bf+\'=7pNU]rP@MWAd[TrSp+7lHoUhib3-toO^p&l2DQ@fPBl0>H\"XgW,J<lQm?X*jlA@*-D4;.[f@.S(YJ-K@P\'5F2MgXfjV@W$7&`@[([h\'gVotCNJn85%2.LRK2,$L!qY-^]q.pIcR%Hs0r6$]@pi*IUUXg0T&+\"ZLK;IpYBH[O#)p=W$..,H*m8n4Co]A_BbVG1$_\\/Y?BFMN=NM,`>P%i8P$<4u=A&[dKt9\'#8cH(3(UU6DAC\'Em\\,P_-Fa/@_==WtadAop$hk>!&6chVcNm((9\\7Y[3,$[-T[!REojJOH$\"l@4\']sA[1>8?o/g9-D@nrPbU1EW73nW;#+6]>UEH#UR%ZIBPVF5^IS@jK`Zs.cuI11M^`:JP1nJt_JH*$,m^G2C9%\"]HpR?b!SkuhP+;,<WH,=UD+\\\"o9CA\\,ji`h3F[3S\"K3+)T\'],ok@7`bY\"1;0)Hn;b:Fg8Y:*dGu*_!:W>S[R_m0,OReWM<UbJ%;Y,Z*E,q][k?gW1$PUJK6VbW+I#l$QGTVj-TSe=K3>Q]--N#N/;3#d\'[P$PttRq4%,l/%\\0khU=[db84V7PX3)S$\"pp<(\\7(9^FTdWH>HCe(3KrmS\'F=\"/RChE5kp3SmY#;&?AV<q))O&A`Ap)@\"d46Im<\\i7N\'\\dZ7qm60@bo<Z\"\'U4k5R1%)U<K&[B/PlIQL@.U-=K/D6Pobh*%;&qJL5!boV6tPMX-dteIhosFF4;0$`Iu*R)6XHbA=d[.HX</t\\n7FK@Q$fIp!,k-;0uNYiE3Xoc=$c=$Ak)IdpgYIkWXZZf(!\"]I_M<)(dHne*l`hq8*+DK>E_+i*T&+c@[YI4\':;dE@s,B3Yn.LZP=[:\'_i/Pa5BiVNKheilrT[8Vpmh)Fb[T8m\"@]R8\"N/n:QSVu(k!;@Up/-tt,\"cI)\\X1L$8g73D]#\\-:;E5qR;]S;K8(bM`(1MM<RDE=%&u)Yt,_6X&YdpdV.HNX`Ea\"K_nSV<YbB0N,jhe2R!c-DXF\'NS_d?V%%j`\\0mG([1Y!8eU/glT(0ZgA21>T#diLQY%XHcX3T2ksjJjcljnAMI2&20G1n%GmkkZp0qJ\\7>M-FBq?iOh*!$k#ugXVHNYKV7F8Pk+ZeDhJH)iRUapq9t<jGJjm%MkFLOe@nB6,M2!=hhNTV#]t\'`s:)2\\b(2&j>hV#acVp-8?F&J9jY\'=:gN9:HQ<cYoEJTc/KcQk7!2#+on\\+e-s\\97X5>p=C(=N%-B?VD\\F-+d$o?PG=@hR$N72B7-#!K^[:]a@pSD$,K/=!`-$;gDT<;[1kM2QG,U,1u;c#egfT<bII\"=o@c]q:3Y3]]Jm0ID9dHk^QEqj`WcIM=O71JQ`CD?0WE#Pb(/D+DBpW\\rL(c+[J9@!+>j/\'\\7sD)!l5#k4?^P^k5.5%#lBa`\'V:^W^YM@>/46kS_]ugfu$nS(p+=n-.\"X>@/s$hEY]Etd%q)mlu^.?EU?U7bR8q+I*M;NZTXE=\'6lW5\\1WtYL=D!06)/?olj9UR3XO]Q&og1@!?j6o2Wc\"%ak5W70TH7F*]\"W625_gl#W`9FCg.i/YI36MM%g6j\"H^#:G$lTcORF[:42qS5rbs\'X2=QIVWd1&0At=j&Q%B4!hU6`[FfKMC;_%<,!?%gb!4[Or5S18gPLsm8i;lYR!TS6SL+>.]4$H#\'_e#dOh[PSG6r+!G\\;JmoHTchtj5&g)f>.J]i2NS0VHd3t5mH2;)\'t?oPZ(\"&_=X[,@_>Hl-Mu8<N;\"1M)b.?UN+e*/`L#X`!Y!W$.\\dQ9n\'HNT4!pl<$JIH\"SYD!U8K$]5YP(\'(QW@4(S`-*L_cpuc\"!TnrbB^@qiXP[[+-btgB-obgnpp+T_Gd%G_IR>J_@?:CM!nY]8/\"A_`hK^BM_hj6.,i+.cB$C8Q!7*#<L$UCLl;!)3$nIFq7TUF(4ZXGJL3Y*?o$Ul:4i(t\"W?YsG/kgAje5B*WN^4@cZV!Wp[EL2mi7r$Xi65c;bJ.YZP*@!U*uk<>!I[!Y*6P!c[?LoUeMjul60q9*lYPu@(Fk=LO-l!:gn9m>]t=.J2^E+?Q=DmH^;WFNVMCFhoK=&?97P<@s.YVnNe/4pkl\\Y#DLG+DquY#E,s9l%W?U_;?g3>g$3#>YGI/$\\,EQmpSI,*WN%EhGNGrr(*5akN:\\0ZP):;Z7+q#4^.[\"o[NsbpH\\X;,Cu,3(r**pmX=\\QE/\'m0%DbJX:/c^7q=Jk_YV8D!P*Te]07VfPu$5DId[8[7WR$UG\"!e8ps\'0%kE+niBdA/\'s;Rl1V=):!E/*b;^[iqh3[a,-;Mkj#@!Y0=WGHP_T/j7UKbE\'\\mB@naOZ?:LC9#.`gMfb1:/P-XhqE!XImMhTdp>d%`1-Ng030jMNR\"@\'Ksm2]u[X*(TgeV<G2@cg_=fLj9t,-GMgT^I\'(MNQ4>\\),.r]V$]IUZfZUhoXJkm9f5[0(spjIsF.)U\'Z!S/B8<OPKi/0L$Z776M2Z)jfhB_*H6^[*uI,>Ge9-]_jh$t8^gcM?*%0ri,P1@s.\"S\"\\m3u\\?\\`rJnVTs\\MAJiRR-3GXjoa$HmTh5W,f5d\\g:bg-6gDH/Ar`m;A_\'o\"eCE)cUTe]2^uM,_QlKBHPHU!pdOAL\']?lC$<CW.>q&FXLSlA\\OH=cA(e&SD7KM_JJZh\\t%p45SoV2=>\"abe+$J\'OBACN%Vr]!qOYe95:;s$e^]mmDIBSVlPbD^jpbNQnBE\'i@0!\\0UHnfik#Y8t^g/a.RZShJ_tjQgn=>KfdQe#`q\"C!B.qh3obSH8@\'NFP@M^@ZBf51A`dG8C[Y%EOmRok2fC_NSHV\"rZeIS+Qqabpl3NSUp9J,`MN(No@^;AT3agX^-1bFrF6EVaP!LurlSL`*D?s!p+n=LQVc>/\"<]beM8:k#F!$b94,ikFokJI`Fk+W7mJ8.F:@Zh\'pQsX5hpsTN8A>51UNe4S.W(purKL9RTp+Wt&)!;d&ek@\\3ei/&i,*T.<=;P\\[p:h\\3QC8uU@MoZPK+F;A1^eY<$)!uL:FjgiTc`jip)9reWk?g=Vn-ni20LUu/p:Y3#@2M!jXZ^-Ru8UXmHC]rnbk!k4t<Y@Bc,m2/\\O_PfeNYYSg\'\'RB(\"nX)O7)NVA\"[N-#>eZfXC:XnSj)m)`9R&<jBMSUq$<QB7hU8,u^Z\"<RH_*.mWes6\\UTD(8\'m<Gg.=uUpKhGg?U>]5XuV!dt#p8WIm/W10,/3W$K(%hESR_<g`/#/\'FOkRsZm3=lh6F%Og(GS=g[NJ/gS1Z/OcXT$8U9A\')5\\@d3uG*Hs/.;CEG$7>ZCTd\'7^fL$pkg\\AuCNNB3?+7&L2`Cb#rH0Vl$);TCVWq(dg]A>5J;-^A/c)N^?\"+84u^?tnM\"a9a?9DaU,*+bn_iXFPS*$mM%+iSUVklkg]VZF=pV(+=+gs4c.I<DVC?qY/J>IQeNIan7?M=+\\N*,4bO_Lf2F=1>p\'@drigg$-bC+9]\'flc\"L7fYd.2VD]-D<2[W\'c+5+uaP0?^(:@22Ml>=rMD<9lbRY3A@1.s0Pn[h,9[8EXEZ]04bb#_/(0l\"&1B?FMUQ<oINWd%*ck5/P&CWSRiWLl\"@eX:>2pZi0aU2[!d,UW*LSCPu#!8u=TmG9pIm9(3+(bpoq4&r%^ofgphH<Qj(UL@iME^Vc@;dCNd?^[2PlH@&7f`Uk3kXBtHJk$HoK,Mn@r>mLb-`9D0ic:,kbLrYc1X\'BU&$j\'BTB%VK:Eb1e`G\\K`07-$;<<[@`Q0/P%dt<Fj!d&L@\'it!Mb;FSKf-\"O\'4;u+rctO=e9J2<Ort]$.IHDT.!]Gm-$pA-\"+6coaEtC_\":u#a4!/=A2@c)M+aq2O49IL&\\+t0*N(EgNGJO;C!#Q/gsYIKUt,Q8B!dN+;l%ECM2raL@dWA=8m\"@$r9R7ds@Vm<#\\@V<q*E6[B>ft6]U\"@_6]R5b\'0$`Yj*^H1+PZ=.1_iR4_@Ln7-^(ai3NPQ#A(S/jVQPATWrloj@RgpSaPr#l]hF\"#RE3p(WZ@k$M(Y[pEM%=LS/6m\'m`X\\%9,^j\"\"FEr=&Ci9(P.eg0<<\"u8=\\+VL#Qd:)[t:De.O$T9![`4f(;.H6:S8O2i>-B8QOGblmJ1j]e1%DAiIY<h,+hM?7AT9Rio4DC>,UR8GgZ_*_#VM-q^l)\'4H(ub[2R,SY(22NiGUu*c=67R+L01;b?\\gWna\"nZ%AoN;K_qI>g%/[X:bm]b!V&$t!1g#]WpDR]Rnl3YMDj>CS6QfEmrLal[r@gNiJX>pr<\"TPdNICM4s6KIGJA*110bNa@!<E@RL6%t\"WNpZ%4ID+7uI9OLghj,YFY?g@QW=E/7&jQr6!*qip#7\"j*(g^M]NAZ*f&s3#(`SP4ufOB\"U]0f-A*?Od1%^53kOb?ZY\\7/UKBYT@-,e0/H[Rf.\\qFp;A8E[RQVrpL;.hrg6[,OUGGHnAS?E?Wjm<rII`<(dXg\\;\'e--dDnUiYUEEOZu#8&;a799!4D_W(\"VGbWhRIq6bf]=^O7@b=1/0ri3#km*!HZ5G+>2dM!H)=gl,c)<A8)s4?OG>7,+3Muu<>Vl/@?3\\GHeY0/Ws2c@:B=,_:?O=*[hd3<gqbOkRW8%\'@qb#\'FpB&45LQQ%l_g11#2qOo<8Gf=/-/a3;%jH;BP@eGjZhj\\1O,*Anad(i^Pk\\-6MK/e`ML)%<%(]Wc1N\"IuB.F<0ce*d0r$_HVrk,cN!cfp/=W0\"HHYGn*R@*[O;*/.P-;)%saK/0s8;BfQgBckdW)f*Cl1)XpM)2l7ro@;dh&?G%[FI]scQMoAVgh\\eo1\\n-[,JZeGidb:p>f\\Pan(1B&4s4;nj!^u8q_Z9\'QqZchL^;-HOkuMCK)E%D<.M;45<:`7RJSP<!sK&)JtBUA>%@(n(Pp1]8eqA`<%e%o2=(@0*`B@$r-\"CH]\'+aTOALL\\dp>D?_FmfY99QAUd\'V/BrkH$#09LL=0FVu15?091s\\%%Y(06A:>ZDc/P8lsJ:)7mc<+:s6JBb3_A*hN<f#(YlCcmR-h#U<Ss6(S2@*qO5S16%0(o\'$8*HB/P4mB<G)0d\"0D34ScYF=sO7b5LYI;ciU[Zm47M2aS1gaUe8)fdggGr,2(R,E/OW^p!GLs9PPcib.B]7W<@q#=,-,\'!Ndo6D>Gt\",_4(P:Qi8)-=(K%8Mheh$s<j>Hg3If,`(Rj]A:4(/W[QYtK3hm&b3h8_4?W!Su$M>e7kftlC4>,djLJh^(IH;jp/p,9im\\CJk\\k8-$b7+qTnX@tkENFMR@\"6.kO>Br4DIp@s+?bU\'TeLJU0a.lpQrt0#qiKrTqE>>\"Wq%\"[\'$sCSCfkCr\'-:(A!dl]6$$+guTP-A9\"Ap)1TMn#W-I*`VHN%9O\\en_QQMZMESYbgM]MHaN3qC\\aW*3W_S\"4XIH^g<N>J`YG5TY)iXs5p_,3D3PQNE<%]i:`Sonj:rmTkF:1&edC_,M_ZZO9S_2/DWE8h,nd\\9-+;;J\"o]=8_qTn0NTf>`Hk[TMu=SY8W/^bLjC37PWS\'aD;_%mOMYi(&4ttG&.74F<Tk08i##MeU;@\'3cmsIJqs68)/2O02I5J%csuE6_&97*2/09aWqN;j%kO3gjEoA;pm@X[Jh<Ck4kYCU@Xd?cl0@2r!Y8_:8/7RgEnYCQ\"=8.*P,*<$nXa1*\"34\'2fAOG_R&Vh&Y#Xr0\"Lhn5\\@*ZZ./NeZ\"+Ej354;)i24]%pkkZF_$KYu`qB8KE*sJ+/)t2eA_b3SLF%7l6NPA*\"(FogZXQEWjWOPDI&A)\'U@hu#um$JEpci[8&-,n1eT9E6l^KeYQE)Ra\'h-3rgKG8\";%P0q7eHOTcS.G+JiG_;_Z@%W,(LQm:@ah#oj>KT?i6?[J+OZUH,W>Yf&tKR\"gIQT:f-1<_BN,Elid)1c&r\'Qt^T;Qi[!qmH\'lJm\")\"_,1FG<T4>bR7;pIIepWCle\\\",_@qi`s2n,V_c*5#`RH`cm0+XW]X<gpjCm1e]b6WsWh.=5ke,->5YDP6:RU3FjA*JU^rJm%L-kQ9Kc=>6!(\\p\\sb]1&4PK.VKk\'*@NDTDG?[[2<HI%1PXgS9;bi+!@F/3ogj0eRiBS^o\\(iY=6iA;hijbjVAR[*MC?;[i\\4P7:i9$7RO][1IM;BoIhNa*LQ.d3.BlY\'[@?&UJI\\5NXnrk&HEJKL-_V00d>`:Q2&F#=2H/FIlsTO)X#[T!`HQ_HGjF)H%sb?XTW`j&]1mG8]6%Mu!>bF1gt>AX=N\'[&?e<OmmgUbShIInc2FtPZK/YSe]]-`*WJbUa#c`mnqZ)bUhku+K<qhqr?,Bei6&9Db%KU.5.r@Vt\"=6,&!5VdTIFVS\\?9fF2qDWOc=%<Y[jfk66fSCE+K0l>WfEkM6Wr3k@7U&Oth\\]ALd]qmC$19e\"!fHA`BX.;:K*mcO&XVAc$m8N*.33E2qd<Qu&nfpMYS=+U\"chFN!50,3WRG(cBVWg1JVrdW$p-q82=YOGIE3rSMF<^e)><d,@OU;$#*=Aq;5_Fg5jNkJ.Vba\"MftFX2S)8#W+EZ*!cN%1MW^-6h*HWK[Ep<6an5)Ic5DPlIV1L3(Kr.[D_r[/1qSpGok@A[-e3c>=;oKVK))2\\D4JA\\\"bZLIe;F(;A6f4$8sOU(f\\82)h\'t\\d@^QQAZn&]#L&5bEN#nf36(\'qTTrESOI5aYj(Ap**g6Cm+6:=[/Mi9[\"a0AR7NP;N.SNp1UX1F5+5TW9Id8$+c4$c/FJF8l&\\i/Cn?>-M\"\\U\"IQX%lT&l=#O2IE4&-:A/B32#g#sS$-V]_q$*a*aY40\\h(I.P>J1_b#\\6eP0Ht6LpjeLd5r%_)H`6g5=#PcU2Yd]CE^)oG`F;4bL$A\"Ef\\(/&6^T_j-MO&:D2SU=<>sf!VD#SBR%aC8PDXc7\"Pn>lYmOO`JZs*$K@N*Gn`\\nOI>N*3X-X)l8_AZr5#t\\9j2B6Yr;R%#_3>N\\\'@V9Y<oM,5^qUVQmi3SBuqI6S`:o8>i@^Wh;U$tGg%u6dR1n0%Ia,2=+9Y.Iq&&rYN)#&e\"fq\"Ac*Mn*P%X`p]RnjN%&[3l9#<&+E@KS\\\"?b)35I8Yd>NYj3!h2-)ap\'Q_>?M7\'W/e2=Ub4o!.uuC\"t$Ktj%D\'<*9&q4,`t\\7q35ofRoI;\"M\'mDl`sAmD^h77]iWk!=l8\"IUP/NYANk;KgYem)@cEakBI?!cJ]T,n:E?(5#O1@_sg867@nPGM^rTOG^+$7$.34Jc*@s9tS-XKB9oIF!)6\'utUB!fTKWA4XKL=]<W2>qem7+hIuKDq5LjQd5.]-?_=<\\(oq$6U$*[fMlR)M,SKij,CRN\'%f,(Rp@G@JG4fe133/>.W;+#<qa=TD[Y4@nlA/ZK@^Y^d.dr?9RtQ#@X:a(9;#JA.@k8bU0`_*UMpTqP*_HEJ3ABiiKj&q>t_Ee&(4DYf--n_0mBhVs&A6Gg/=j8%?\\8GNJbQ,BJjrD3b?(LI+@`^\'<mcQG^sOjo%p5_d-F4@4\\3;XC`g=Gj<tRRV&\'hR7B\\fB#))4:h]rLm5*![=^%,\"ol=&F=Jq].3f1eEBM0uElBHg00g\\)G<\"PohUl#p#M8>XQ1nj*-U-KiLRLoVC-!l5+l7_q\\LjoV,U&a0OFVN;qI`AXD72itWi^ED2!Q:t=$tTuL%5&b@s.9glJba24RD3@pN1]O+2%\'J9kfONq6:iO)^tp#0(Ws-tCI\\Mf_@jr>7M#VkEHDr:1a\\Csi)hau0==mJX8a_ST`XEY#S+KMM3Ff-NAa.9*jatP\'oG5_0JO3:_T@R7WCEX@SVTf`bZV63im;UVIYW7Ml2]i\"%7%\'s)9@Rs!5b`cORd8mno8u+UfS)J85\"Im2]bN^e,VL83?ZOc\"1\"s_5k:cT85>!AOtV]0M8[Xe7=@fs063rYM-5M5%T,D1DQJH<E?Fl0IR\\3(>H\')?N&5F]>prgtRuYJ-;#_c@X+XH&)ZBFTP>p+q,.m.5*2a.,=6W#M6CE(7;DWu-aiL$S3Y5?cc*\').X\'RbnN$-JM#k<$\\VD;9Lh=FnA)]>a,P.->f]e]KH:/p>u80tj\'[Q6OOfg,$Gdfoi%3mMD8=0=fTe6\'AaFV9BT2&\"?!?Bqt$5)aj;S+Pq(<gSP\\C6kRVY1;Uf0Gf$;7p7!p`;9Xmf2$Q!`r\":Lac;<Gn$.]5GXsR:nggDc#$U3kmJ,]]\"9q%\"hjq?d^Qk[VY=#ef0/Ws@p]\"R/JAD7NW<AUeHcH>,^?bYW_#1;`BYa:R1j7gUkJb/Jn(?uc_6st;(QF?q\"a3b8*%fcN1R>Z5,Q1L6g;F?YV\"@.Fs\'gos_B9K/,D$1G`]ADm(,cZ+))U]JQeluMK23ilmu-F\'SZaSHp?aWt^At2DK/aQq,U``5i&X_(\'14<j`3[HY]A(N@^!FFdGG%;Ck5W*O#CXP2I*uXIkHdp\'WZQT7\'41<o=%;kus3Qga]\'V.AR[*Lj)s?l]Jmrt-eLeg\':\"B./jT_+D_`4iTfj]HLBi.oR`s\\X&>W(>V\"/@hI)oo9(pZ!X.@)nb6)l??l3lY:??C9BTpscZK)44OY1uUH7(tI#p@<?!!e!S02<ZmG7+X)Fs*<Vgsh(/=2)=cK4DnYnm/3(k4=nD@(GKtF4n0&\'78:X)V#_ep\"\'Gj3Bk[KE$DMq*52\"rTXSeDo9r5Y?5\'Hd=G6$q:\\?k5A>LCn#[qr5CQY:R0_SOHjCh+1A$W5\\!T/K`mrbt3e)g?rrc^$hW,hPp1\".\'*4/I!<@S5uKPootraBlaFQ_&<*:S_o[bHiF@bSU:bC@d@HPl-_KJ^\'bo3L2md;NTlXT%?@Uo*`4i/(leh9WcejN%s\'9f?n*fdo]g)PShs5Q;^:tahqCkec[[[p0+q>:/!(<4DPTdhq17$J)bgaY7B8)Gie&FElPr?@!9!kkKB=7`kdP]eI4>DfDDI1n2(oLN2iSCA.a_pcs^YY$:f!Oh7#VgJ4e.uN9V6;I>0\"(Aj](-1<-Ipa0\'-MW=L;K(O>s_?j)H3<Ug#.,?l)+gdJtCt[imjNB?bZKs34X;0pdITi,UB3sAE=MjPh\"8@ZupI\"%#mqSbq(EJn,spQU%i7p@k_&%\'XP3RGLT!Torc[P!!GSrE$PdM_\\:**.`9*MfnJ\\5k4&-IO]Wfk/(>K>hI&0EXPO26Hjk`)>GKIC%4\\3Z#Si\"@U)7I)U;?A5#t8st$AN:);nWK1Esm/!CJ6heAh\'A1JO%qA`UP7;\"OZGM=DX+IJO<n\'A\'PFZ2-Ou(A\"Ei(-Z2g`?fX<P.Kb6dNQ\'!eP*F>W.NG!a@C\"\\HE8;7deB6k;<GILS]gjEU()oM#(nq7d1k<C\\+V;>[:q^9kIGf(($\\TiQ;DU5>(*\"<B<@IgolB5p<Qi$a^-]]B(KRht`&4(t;(_K&%M$$$e#n8q0fPnpL7t?U&.0^-D1sog+N>W*Ed`Glf(>17,ZX=.DN6.*.3cbAU/jOQH/cu/Q2CtAG6mTaMBYBo<7pP2.klqX#3s:4O84_9.:jPAolQHA*(,c.5+J=$\'+?`(Yp\'M8b\"<imV13.I&fN[@Rff)MQ^M)l51,:/:<CI$Z;%/Ht;#qqA#Se\\DMP9%N;MAtL+d7j)\\L7e!U+9)uTgC%GF7dl1Tmu=dFThkQ64@uE1Xb<PZ!K@eKLK.rnp9uE`qDB#fOE\"iCsLl4e\\X`Q:\'gn6$ph[JN(=d!_*kKJT`t]/.T$j+EJ/2qG1X7kH;8LE-Ntqm;6L?Xok1q[HnOP8h[`SO+@6Y$Ir;9n19d_+FKfH2/0K\'`>l;Iud\\cQ9flQ.j2]i&er9&UJN]ACdP4LG?.o>MoCMp+LX\'V!%XZHAVL7P4r:@fc(XjPS:!J\\IT)\'#>LWO!T%kK+-F=.WNf0j&f\'7H$coX77u$4\"$<MU-:7r$SVS)!%UgWpkb(X7u8]9L.j]@.7pO53=;!!pJ-\\P9e,qn?@sTL>.%qnJRdkl3?T8`/[`uWm6t1Rr,.!H+m@1IqBagb[0ZnA1gd-s\'>%V#hIi*/&EpM[e\'I`6+X9L;:KO3&.>ls3CaaB*K#KYkE>/JkOI6\\Ck7`q:gcgfZ:a/FM.L@BoG5ro&\"T_+Qp4C5F%Y;R7>s\"(T:5u\"PWBD/t?-eYhSP_m6cDiZ=kL=j1Gp?5bSLsrMeAXq[`V*l2AsOmj?]Z.$\"0\\\\JXZ\\q&lLE^@D?JrG*PWSd[::/mTS)bm+R#IVmU-@L_&+)_8MO07OE[0Uf$E0#qumHEl&]R+`CQ4;_aXAn*7sXaQ:?qBB\\2X\\dN-$>.%Wu\'0j%Q46]j?kB7$=?Teun`hfGTiH^GJ)CWt)*ok*Zfb:Zm@eY.D4WNDpnBD-C&4^%U]O5#<m7]+g6^BOf7R!Us,RK>!D^1([1W`JS`<pb2A(cGm?#)_76S`Zin?3:Ed*LIEI.9iM-/UrI)?Hm4b]/nKE?M.=?8BbU5>9sRY$05dnp$VHV6Za4L,@[Br!(/>-_]A9d/oFPi6:>>i7#38<-fh,FW)$neSDSsWb\"kpdo_s-\'4a&\'mAL&XF&]b`D(._K/?;`JAJsFQ?\\\"Q\'i5cfc@JP^&1\"l2Va\\=^cW3;8%58X.eCDKob<&ksT#YE5]I*);V:#Hk`+jX3Tdr=1mRKJO5d9:$r#YmORfE(>mKC`H6Xl13#4)1$js*$WC<IIDgU+MEg^de\\GXm>IMm#1q\\o++Y`d*17e?9VjPu^^(@\"_Vb\\F2@^\'GM(aD[C>_-l\"1X1.2a0RHaA?iB3do$9HSddP/45^c77pR[;Q!A\"Jo:OA9UiM/a#DnHqVdJc1GCDJ!R(sa/SB?T]0[755n.U/j5-h.9q0N#[WJY+h<RW<!\\kZ._\'GaI:\',E0D@4;(;K!*\\C#Irmj-s!Ba.&k6$+SqM6?JjL5jW]m+F++;P-PDt[[:<Eg;@!b:_BEAESP9uc:N(CXs2s[]C*+BpHA#m@o%7,g(!o/?XWbl:<O/5#Z`\'HktE4(-THiGAb&60[GFpXqL`8`<!O5J5cW<PXZhE5nRO\\M0:(A)BT1BM\"pBGd?j%V)E4JE!W&pfs;[0-oNtF/YZ-S6/2s%Gh34TCT^%#F2&\"7q;^Fme:%.\'K>#D0]k>-i(lTbA5kqc<bk\"/Z;ZJQ^,mmdaA^U/gX;W`6B@Kibr+KZN3LOI+?<D_>,m%5fj#a41t_^`k+AS5O*8$RB+\'5Tp+N_80YW$Bj#G@,b/r3&\"NE\"tZ&c91-0aU=b\"Rg%14?Nebsa\"TdW][ja59.[3kc@\\s`EH*rOW>_&Bl/Ll!?<5XBZC-7XLA00i6BTtFiXFR#`M!@K!>)8#XTM[R(&qpGs\"UG83-27cu\"fm*E4[D#W4/3O\"c=1cK/\"e&\"F@ks,E!-bEB!j]i(\\*FAMj$EBJ3_XQ4>n?=m-@IsC(RP;c[9*p+.2H7%>TYMLJ(UXGs!B+q\'nqOD\\ZrZ<=f.X@c)0b^T^2hXNh\\^3.F=Q`M=Z182XBoY\\*hQduKU@\\dRsc&)L_VnDi,\"NbKIHRt[O[bOG_@:_pLL[KaeR`MZgC^Hi/Rr+a*Ue1!JSal_%Jo#VUn6G@3LC:tCq?4?srprPfOeSOsXd6O];!YIf>+g[\'N(W\\q&#Gklb-7OQFg2ml`Rb8/7R=i`-/+f64=YAjgpdPfd<_bhe$#\'R>`e%%D3\"RR.1&CN!kt$23+JB*(B]<DYR5`IjL+&2E&XJS]_W5SFJ\"\\&,Bp#DF^3M@W9]G.59u^/fA*53ie_\\!aUb][E+3.BuhB:a9lEsn>#lf7i\\V)5=$1,20IY<pES<Mf=:I7\\qXVdHD]SKZ1lue4JECXs*r*jT(LK\\):$<=2sb#,.\\F@D\\U2i=Q)]uMNb<:XJj8f(\"Ncc=n;Yh4BJC8ga+eCo0UIGAdtno)gt^V;#DGM91U$uY*:H53QoTjuL;l+@JXkDABaNXje,L=>@HkV^%45HX8(9%Mn4L<AdtR4e.^nU7Yf>\'l^Hg$\"LP`d2<^E],5]%!/VW6IT)\"S(+Bj)J<]7g!4ql[,icc9j[\"W/`lY.UZQi(@$*+D[/HW]E2(1%mRm!lYlV[r`XMZ!loF\\(j?VG9+I3i2Vg81m=S%4aqV+`XPgWh\"a5?A\".b6O@\'?`HnId[5E6I5PPD#5r)rm-&\"kB!M\'jTV9EAY,0*r\'`NF^mU;pmdDtWW(YK`/3LF?)(VYiGZa3lUp)PSE*/7,jJbt:.?Kt3O4&4.AY6aQS@\'3F`54mREIf,U6n^>Rmm%FNP]adr&d/CsY/nOkjh%LS75a2o,rRI8Dt4$S*01u)65XP\'L^EYfa6F_\"87L(k+Fkq13Kt%UHYZ9OK1(1np*]S,H^;np/<gu-/)-oN)VXDLLG=h_9_Fc\")]H<011r&SYo_Y:RD_Vl%?#_Ap;<-`Y,7Fb0.R3.bJsA9H=E)C(0onf]Z&f,>UiiaK1RLi><FO+F=IfIHM1I+3rE6u`u/1Hfl7r%=PteWdtT;S]Qkt`1.(Eki$!#8p=YmWPAddP3OIR.`\"sU0[P]CnSKNMuA7U6W1@qGc]DJ%0*%.XDo>Y+\'UZphN#Dh@n7lX!,/+(tG=;&b48R=I8INSA\\n$?KaLfKu:;nErpMnM%Uq%iRfLQ]###_mACQh)K<XohOW\'l[is/Ze=lQ7mohq]=^j[\\S\\Ib=.gS*>arX!uYWh;;#tp:Hg;C]OohEXa:YaaAg4jVP86]IXNS[Za\'5$ipH^Xq$s^CD[G.R)7J;CYe7k\\3@B9Z%Cg7]Nco=LlsOa>Y:Z,:cED$0)\'jBtmVgprK\"4h$A:r76KY9PQ[V\'jO:e/(</N\'\\=52\\3He8V#^m5[q1,6InV`s=:^8ZOGnf\':_!)sshsICipNopN[TB%S\'+Nm>7<=/=76h+JeZlo=V!59bS&g1^jkABIoRnsCUYiY8T`BQk1LA.>/C(#*dFH[@=O\\_X:<F0\"qPbii\\i::_9N<%krQNu!d_e\"pCkGs:K<GA9N^\\mZqC`-fMKnf96qZB]QC?]EG,Np/Vk^*!B&isYZDm\'jHPAX:*`1CJ$3FhMrlqB&X]bQhaIAUtr[n<.0rkJcQQ:4IOk\'c6l,N@*JTSE1oOrjUCXYM$8;%rP>pg58>(AM[V*XV,s39&Q6iED/)4\'F9K.Hh<WT^e^!#9(gaH%Nuh6ct#m2`,Gjr&_\\XX48GUlH2ZmhKBZpW_%;Pt>`qo/<JEW`T$\\dZl:[q^oiUYdc#7K8EO%F?-H?#Hj`j8ed?B+(kn]i\'eH?SNi?0KOrQr5/<hhnud\\upm/ebW?J]c\"H99Y+MH46(%bfQVRGUs%E=eQJ/Te@TC!+c2P$:%Ae@KK@=J;aIb\"<\\fe]`8<A9Dko\"L\'f.4\'9[m55Z9m(n#)]Co&b11MSE(A!Af-UJcO0@lj[@ok;t)=TcH\'WCBObb?jM?15^g.\"!)OZ-NF+Zu68Ln&:XJn:*[%b!g-q(rp1;Gfm\'O;kN62%eDc]nj%=9Ha%c(CZjp$a7h9UW8L*t]k;Q),2]\'8nE5f7C,]cEkVB&&2(e9O4t#=&iC_6;XgP5\\NA<+geXrh7egm=-C5n,me@\"suemF#q&Z6ra+=h)E@pOH9,I6)Itf!teA21664P%MPK?l$2ZcgctA.\"qTrnoTt4=8r5k5(rRUT0F8Tl#=re_gA_n6+_6r9DMC*C]T+d.3\\C_XU()DO/p_iW6J?.qAu*LF+;CS\\Duf09Sq3hk/M\"p&#*f0pOQ.9.63A9Z8^X\"HTI;_LZ#^Iqm$1UHT33*e6N@58\"q<./lY]]DeS3.-lLj&odgemcnbAJ22NH22Vm]Np7LVi4kP)&l.rDomCMSC\"<M!#nAj-2\'IX!>I_I=ktn:@WP7@]m/b6RohENC7O%mjfp+24Y>MqsZL[+9j\"R%dk4T*#Z%2(d]s/7!E(5l$>-gRe.Z>6(e;J5Q\']qU`mlEad[l)W:p\'LP<=i*7eZSaE\\usHIok,55Kh%=ha4>aUfCL3n^\'o:T\\J1a,?L\"0&-GW9G]/f:M16Oit`\'^nV(Gh!`h:H8C_YQb.[+fD]5UDT7A(VVEp#O(.@@bKV#JOV<[a;[qI\'%Bk3K9:@_G9,/b,KQ[IQ<UT&,Gl*bi&ZHtLr45O]>8@m>??XJhb0(hl(]@*lgm98,rg#@>-ZUn!!Uod0bF6<gLfW!:S=(gXr-1eJT),B<o`u%*PpfN/Y)^^;n\\V5^K$5JH5hNLP4IZ59,7GgC<j^ci?*ZgBS?\"#L0UpkapQhg=1YIW>3%cH%sU>_^17Q7s0SPm&R]tS5TkI]W(DYLBmH`soid]Y7h[sme,ra[L?ei\'`u%;n_SHI/jprJf<-7<GbJ/[?%p-Z5oSr+]X`@1BFE(^A+,U#hkCBebC@1EkmKW5DgtC#.FiF0r.#&LXF@\\g>R#H9oZ`c#8SM5.[2l2odSpA^ThgC2S*=Z.V:fPAFbA6&UR,#b([;qN@_O%,enXFlC?*#>a3hm)%i)+fBEc0fl(fb#YZc-rd!1YoL=FjZ2Xs\'Zj7LacPqu2dU`Cgk8PN_?`TGRYRQM/R?),T,L$Om\'kAY;JUC.(5\'.bMDSHkZtRpba+-?PB$B($2JO36\'>IDDd2Is)]ubYGWRTM2>I<9mPR<)8$PFT/$c0Oa(VuH1&>Aq3%HsFLH-3qM8Zj0bQD5<=?Qtaa<UMYTYmD?ca0E)l1RgG-R0hIaH,^0&_an4EDI4to\\0o]g^0SHXR5%Xq8],r/f#[`JYE*Z:*\\sK4P?fqJms1WQaVANK1BiG\"2/E!/Z_i:fAJUT_G-[C*:U,s+F1^6</\"C>ihKL>u6kC],]aD=[-BQKpL1E*3;PG0(I+!UW@V51SkuB@t?!_m\"Ua_fYVIaFNmiV7M0UA.t<_?1Tb6:71.s-J2?pJ0d@6e%)aR37k\'\"Z\"MG1BOi0Ca:7\'*K6W(PDmcD2#Z6IgBWY1F)gE%i=kd1B668gEEIoG!UeOZAPZ:[Z7L4M\'sj@)];l7=P;EaHs\\#pJMAc#hX(hF;?!^W>Cp9D__T\\Wgj<I;?BV[q^LfKb&1)^1P19)UbIPN.bg_`o\'\\CPE\\#AcmY(Q-05kgVlU8LmqDcY%N6Ofp,\\JA64g0)hhrGl*;CLtblMNtfrf5c4![Rf^Q,?Na-=ctpMlV\";l>Lli\\.i#KCpWe53=/jEH>1lMKC0.Ik%^4;?m.@\"VUUkIb;4\'483in8h\\;h-<l\"#HN$dK-.=a<\"B[W\\qi-XNMMA]e#Qd,9IECq]ZR(pmi,=(PNZ#!a<Kni3UB+nlGO#!4k3+_V8o9;0dJ>b6I!P:N/O:8QG7!g/RZ<skT##_:pOK:;j,_M\"k=e\"G!;UF=MrGqDI*KDM<Kor2m.lVZ/:p*5Q%1Q]n+IU.3rj2\\jZUU1O%ediJIqa^@#O5BE%kPg-u;j\\fCoi)k0#B$&kM2[G%`0&g&oa&Op.LOg(o83mBJtGRI55#YN3LC6=7;o/*:Hs+Tet`i7MrV:OWL=r#a#f>XMTkFD;bf^^hbMnV[oqqD+O#scia3D)0^4#>\'.*VH5_n<k5P#gDA]+n#jiM\"%N%g6e.^M1SIta@;]Dl\"mX:C[gV-#Peo\"r^JGsr/N5bi<+=rr#oEV.0eCR(&4)Mf,dBspeB\'&!_^<M:4X.hTLjLO*l%D`EmUs1]@%DjP=0E8nd?M5TW-Xb,\"X:LR`JWEQ\"//a0P);/A[-^fT=n,/\'-(3S3WJ`+\\o5KFcRRBYqkUj+`\"o85+X;.s14(\"mGoN.h:>Z4&@=>#;-i*Ll7+jRmt:EObS,m<T]a7\"5nNDST)^o*sXV>m-p>SkDc\\u^t<r]J]:C@a,OR#eqhjHQM*k3<?^pgKV\"&=RK+ImE]:1B`5JVW52]L%l`QUbi\'hi`LO[<\\Ug[l\\Jj=\'u`ofd2g]Dic5+?boGF3j?]7h0X:Aiu%[tq:A=0$<SRA.A#\'RLTA46EE\"ku[.jqM;I@.\")-(p3Q/:0[8??\\j;`Ig\"o>]rt7/&81C&)1g$/d`!d>MNdf-KpZB^SOcDbKfo,UfpN-cN)0P]/ZOhlm\'6/tbOrONLs),7]JJuA$DfW1Iiuc8$[T72hdjchjN$P>P.5W8`1b0G#-Wr8ZH)QsTZlfNc;$se65Zh?T?J7L8Jsp_L*5:.;R]rMjokhFO@U/SQZ*(=HeCoY(\\AE[`d6LTC8^_9#),s6EN?f$o,p+^3!LcPfH6$\\=.,)`JCaQON\"!=b\'-J/&M]SSnhR_\\q%6J2Do_>pGYY)E5%5?0Wo8))`b.OUM&A[?AlT<7Pkr\"Eq@Z-tRUVl+`iQW/1r@0\'%6X%Q?L%bnK-h-[g+fiqV*]A@5a$Y:Dn0VU^\"DMp1G.QThC]r?7D/PL.6Z$RN8.qcGORM03k6[Et)bt+0\"#X:46G#*,VTng>37l^,XUrrs1N36o]=H;2&Zlt?1g(^2Oa3_t#4).Z%)U\"2/!5FX@F&#O/:UOGF2Et2Dk3?LC5X;+gE`e(`j\"mL**9<TdHtH1`;k[fn[QTMq4SA3,\'0R*7;UGUb`cWL73)iRP^^@5!^^H#6%@@`R,&T8U3AVpo6ArA)5h;T\"cZ.J<F:=_GJ\\_r+!Q:d:^Ktnl0h;Z\\!D8\"G%%YG>%FLR0ZdsE^CG]K0rMG2o_^%]64VdAJ#[.bmE]P\'d?,h6Ti`0PPH.%<_OgDUtrXE5oY1F*SD9$SA+.H4%BXH\'SSeQaA=5W&;3M(P#/8sT^qRYamPrAD<aV.<q$i\'$AdrDf>1p\"6UN,DjAGr$;)*fGt\\XkE3VX.Va1l_H?*PW:.kb<F>6q#N50?t^(lYbuALh,n!?>,;heh840J&9dO\'l_7E&BaJW?^)u]8H*_.,#?G+#rk9rp7R&F[dU(nK1[lg:Nc+(N:!jseC(/QF1Wk)CNogTh5>qM=:Wkr<!O)UF5/VK05nM\'Fdg6\"j(eBAfSO!fhL9JS>(B@iK*J@fE07tAq_*jmr5_9SAlt$_2-b/_$QAbFBcnL;4#XZZfaT,_D:bk`0%\"ch]ZsH\\4-_hQKclal*gX%&_6+bJd4Mb+snekj)TB@>A[4XZ=Q(b,5k[/^+cT^k`)b`b\"i%6N7Z9M_F-r34n1cu+I?^1F\\<!>`s?o3AKkXCt1\\hqj5O^X\"^$]KuFk9Y=^=)%\"^NSUD8gE#:V^pl&E\"gHbQ-2r5n6e#p[l(?d=%sb>n+8<NqSm_\\N^jP6t+0Y-`?N1TtBKpe,+7`U]j@p`>M`hoe?P=F\\&3b(IJ:N.\\Vga3Dm+L#JmJ^6;ne>.R4;i_Bi\"5[iK&7`:9Q]q46(gJ<g\'.cu`;MheENra9?WWWgq\'qNM9=lET_&VQ2FTl=m1oq\"r+8;Z>7[qZ;VN8d$B3d_4.2sR6$&\"fKJ#r#j?IN5C4S?0Ms#(hU65i^o8T?d*aYY(]&W-RSd,Q*9:ca%k3/3oQ_=\"=>EkrQ%1X.)(XbqG7`@m4icjcI@X6nr,A;WXj\"m)k0L7-gr4\\\\#6Fg4lP2C_k2%jfA%$0O46g$VB:3!)*Em%2iND%UU`;r(YC8ThmB#QO7#haPI[_64Z/>MiYMaUm$7`HfFW421NBR$6o3C1D\\%brI.UHScDbfX3jL\\[!KARUn\\/&&\\D7G$ZdW:6p4+1Y6JlSZ<WF75JieRMB[SCL,-.-#RQ89f*@==)Ro.S9N`)p2]ce@^TjPAlJUDn?`F^JRXRF\\YmOO<63\\#VkL7OU>\"1W.]lq8S7gSl;s0<\'::QeTacL)\'NZ;R$!#T.T9A^flei3UAJh]?ap15AS)OL;t^*_o+b$ALjm!]!JMlZE8X%qW!W-i^FoNp]T(8-\'LE+0_S/V*O2mP!Hl5be$g9#$ZqG$2lroO3DkCn7orWYRR^PDMR>&`fJpGu`7(@K>V<[rl]ebfBgn.611u()8Sg5EDQ&C\'DS1bfF,=HUR@#qiD-)rb<VGc&O=IO.hqd7j*-jpkg7p\'!V\"YP](uDX@uV.kRcP\'..Yr?kC>r3X&9*CD/re>]0?#BmuJa4ZRQ64@iWc`-BB,Hc$+?@G&RGgp*X#HbHrC=S;ZoL*GM5#eZ0\"m6K9pcEJC/8D(;AS72uXW\",:U&\'NP3Y$$#I6*_iL*T[MstnBJQtgS9o3<=ki2l1TTL/j+tj(*V7u/-\'JbKUD!k(tceWjYNqE6F%IA@N7?dXFU)\"%KfM!FoB5Q%FX@]4(0ut,>`-Y-JoMi$@O4#O7`0QD7,1\\@->L=ibuDObTdctiE\\k#_tT7!,UqPCkNFDV7FBQ,VYJjq>irsk$?O[+K(R:$iPKUl<<;Z9+#K,:K2\'(Ej0N(^AG6RZOZWr2Uo%&%L,%9$BNL;3P?7b.h1#,J[,XpfKtCP6FbEO4GB\\:,iZO6\\WLCoF.;\'(r#>`LEOD=9$@1*ApGc<)`;BK4$f[A#W^&+n2#oDY@8/p-_C\"VaV+QbYL6JUY?8Eq^3\"]D6B@J&cPA`7=--[Fu3DSh2`K>M=pC8[/\"S*Va^JWHQ=^(csh;ZQ1_/cS>]^(k+.]Ur*@:S,U*SarE@eknXf6M8nS_Ii+^d6:l=r\'^KHN\"u&D5]ans,.nlFn^7fkUm]QdN&sf_iK<RH66,%7J=q7U38hKtBV/%[0L5_/2`9Ynj9*=+m%P6^ShQat5Y@XrRE0dLWL;o1%WfneH&,=i!,m+2*U.Q6[\'0c034<]joF1?q\"taMK+ZcUoFcolrff*3F^\'mnThrnt=B)(oP_-`bI=):tj3Yq6t*hIq?*laX]!K5s35O+5hb`C1]h1jM<a0/uRctB0NFQT!lEhE9rEm1CpE(HK(5jq5\\f%)rMH3L^B#aq0iL=&WloL_@d6h\'_X-T%oniUo)M!3a6C$!tM+2qN8drQ0A0.GX_^)2Es>03(t+OZ*9H^jU$,Gt*fRd:J1VHrF0ijS#*,`LV?.c10kAaW<=cEn,:??pB\'$Kf;0q\"YFN]NdN/C.(SjG)FE>B*.:coC*Gq!MhcR^W9DJ^DZk`E#:_k43tM?\".Bkq+2W=<NBucCrp(rJBZeE?2UJ]V3LC9J=>9BOpJ*`Q6Q$\'/F[%X=\\-]$B=rODJ>P^5JdI;%bI1[3tTBbl_>cY?Y,!O=?k$;]Y!+Ar;i_?Iac1_kM?f-aA2\'-5S\"(#u\"p\'Ud]%o44`]&&a_Pl(b>8/gbRh%L+ssXaFqE,D#h<\"+)De$3T?-a;a0\'+nDV.Dc?\'5([Qeo/ZR_Ba&unlN&/,)qE]JuV+A>p`M<aKgg+b7k-tE%Ku$Cil;.(0`2rJsVtV(B2jp^qFDS_6p4,j&;8QY##oXKV&E;;rli/UG>af<$_55fL%r7t^W[Z!%SQ7NWI2jiP?a\\rb@ccM-$A>lS+?]<c\"TPWKI0P6mr8rN*^YP!#STW!]r>!6]kS>AMa2t@,=%,Z.<l\"-3VY7%_aNi]3jD.;eA74eelFqT[YY_I8Gf?qOX\':k$h38Y\']**4*gZ9ElJJ\":P+ZOmLR=V0+0/S,ek^,-dB=[Tm,LA;F]VbjMI:rE!GlGtmMUD[_M@1M(3!mJ/aVn`IN<\\L]CNl)el.*V>Q`^R[930=uOHXhm^[]nL,::Lt]ZI&X]Lf[U<!5+9@0!Z,65IZXD0J-6@X&^Z)CeYqOW<mV7uoZT\"6pVcLDOo*[kpp97kQ8%!<e:$WLq0E(kI/6gLL^XnZOEs5q4<O,r!SmUg0pbi\\>#+\\0:6ZWa/\'[?\\t-g4Kdl<j!Pr$2]QT?\\rBqqk\"\"]BC!jHu\"iQq8cP9)D$1<TN1Oa;E?t%CPh&r*7R.?f^2U\\85?)fk%[$GlN)Un34V!c@C)/4R`]\\`HH33f$:o+MtCIr=d-($m?j\\W_(=TIaFKZWspm+QO`DV7:qYd4\\c\\;t-9DoohrRp.T=p\")?e:IrD:H$+`*f*BF^L!k&(\"m\"E`nNJ``\'FLs(^Y\\TNs]&R,ZH4Zi6.hit5YJk<\"B4V,[BH\"peOR\"mG4:j#MkL;uN-S#MQ)(;.JI@DChF$g\"Q5i.7&g]A%\'7VO5Vp\'_%YXri!^ip>l$?e\'pZAW+r\"91\'>.RYV\"m8Q]Ahmo/n!3_cIQacpq=l5)^nr]#f.^V\"Q1Vj_umS9Y$a\'8Ee1Sr,#\'c:cWcg!:QCZ!G7=>Y.X%8Y@sd4iJX*JY*\',(@_AW1Gl*iK6?6t]7`m_?*_HIEJBYZ;6J-I,Z,?#2:H3hB9\'!5E=ls*h@m=r;gaFhSKB\'_%8_!u6\"\\VUQ7KJ\\9H%Bu>RWRO8C@u@rN8VSb3n;AblgPR[RdOGJe\'7f@^YCPIgZgL4J-5P!#=a8W#8<CAPl<@Q;qV$-G\",@m[AE\\5YM4/%N;`VN0M6]Y!?Zrpb<p:r=LUM12jefGnu/\\E+2\'H?2>%jXKGKfO3h\"tZDdL1<679s\\]jsO297Xl$VBJ`SLRib\'J@Ne7AOlXCl0E\\S:i7$42BONd]r/]p6YKpALCg)ZZ_*>Wb+_T!m]<(#\\M5AMhkMDnK\"dQ0K,-go`^*\\j/FLTmZ,gCS/DP-g^+D(4S@Ymc]me\'2#FpjArhYuYLu%%/Es\'Yp8p/dY8gG<%rkiu9KHA[Sh18Ng,\\/%CL.8b)`?#\"UM\\r18U05JXTS3V6+C!%>JeKn-:pR7`>Xsn:EHI/\'[KYgBI[R1C@gE/%lPN_Q@fF2O\"1B8)*FTfM.GOik,V/R8V,j^LKMM.A1QA)@DCZLIZJ!NAoE\'5M@@q&OZ*-+\\^])O$F\\.^p#9QaJ7P\"8\'p%RI%XNb#YSYe@J76$(OMdFSOF.c<(n$Z0qImQ>3:@WNhPm3\"o\'0tEGKNqiS3Jh0.L\\,Z]g3b-o!P:cTJ=&b-djnghRc!<@>C=.hA6K@1c?Y8[K.=TVSsr3K;oQS\"h&N,P,UJ?a.AJDke7ff\\i%t#b`*9tkrdP3*>V%W$;a_tGA1)nUF6bjl6O%Ja1^rM94kjk4o`o].m,Y\\9o=n67:tPd.KUsU[N;?cempro&Qu$q>s!#-NWZ;%J?3cjO0(1RVtD@:Ek:QLQ[JfS_b*<U)q8.&*5l8g9$!GobOB?7(g%Fum`*&J&e_X:V42ZE_f6%ce5;qAH;H;1qi_P,+Yl6hkngD.??]Ja(1+EV%$\\K\'ncrbWa[bS\'AC`\'g5gp2XV[`K]\"ZgQ#AuE@QV^2*SI\\OO#G[RWVH;<u+AS4+SG=u\"8]HN<EM9;L\"btQgUKKuk:\'k\"G/js.<7cpfq=R-I%&kcl4Z[\\5:r`EK4,).kmtnF$\":`l]U@I$sAaqPMut\'%M5bFS[]:1I:_G;G5\";1J7e4Vp\"EHco\\+\\AdarQ8oc`6.t];\"b-2UQQE]\"U+eq<FoIV_ni(3(nnirJEL)Q3XPP&.iL$C&EnON[s!Dr[(f.jiAiSB09qpmBn]p.m$.#n>uG>eo-l]c)hO])OWf$k!s!R[$Q6HA9B!HQ,(!R0pJg`Y<o(37++h\\[pUaUNb3eKKVPZhf)@s+RjJFi\\7\"e5/MNiFstPRLdS56<TTUZ2EdqNJ$$i)nS./\"o$XPR\\/_QLY\'<a[$Vs<,\'Lu,[$R7fbd[9T6bu0c*C3%A)29TYF9oa5Y-coE3)\\(]ANeGZkkjQ/>-e)u`[L\'cX=\\1m5D@!K1>2i[m).+,95CZlXK&9>.i<:K=W*Q13]9S/7\"6(?/#,Mq,mP:`9biStA,pPGYc@+JSn$1*/(pebCS\\U*_4;*USD3X(c`HhL6?Gnj__n*r)pM3h\"Z=1-Bm/q3=lts4%WgC:]uuLm\"DM+W@H$BWeB=;5jrkS5TPs.Q3s3GT10Yr=\"a:coP%S!da@khQr&^Z[l0)cjEZ@A`RaQurOIjuMA@)=V\"\'`+(]-eJI\'Z,HhlDlg+KrS\\XBIkc!c`o6rI>,?:K5Zm\"#Pj^CB;Ei;98j_M^Q>nQK=A\\*p9GJ6&1_3Nl)O1-/,oXQYu@1\"]\"K&%KRo6dmW,E(N$sl&@P:8Zga^cj>I\\8&Umd#\"V9&93HX`8@qFg@]M^83=MKJGDki9,.Hn)K^?W97@%N&qA\"a@KL-V.iX1m@2h:B*o+dg;Rri/bd*Z4rYrU:dW,EV`Ik^Gdpqh\'p]se7S0uU$m0Ym\'$tWji@t*T_7+\\lef.K\"b9H.pGuW&\'HXLl/8m3DEH=huL>k6?2tO;\"0&@qmW)\"$lP#]P\'>U`0:LV<1,A?M^n_jX0@.`9O#2ek$N1NN5+I\'uhR%\'.a**28iIciCG]\\[HA<21/RJ7a2T]d6:fC4S9qs`dH?GLs:lgGjc@g\'9GK(q,7l\\(k.\',N^8[&DL(/ifCKOC-4O&\\/Mi^Mkhh?!3XUq<@fgE*VWDh#&7gE7kISW6ngHsG`_K\'00_:bG.\"TfIWbLsDn*6h%-G?s)o_T]7N0rO#1ug\\92td#/F!R]IimKUR\",s.QA.653+^oHL)+_`5H7pNAJ8,1tIXcel=4f:_+!j9+@Ki4&2Z4jfiCVJ&3QcA3Hc+OKD77nlJ*o9\'O.=1_W2WO),7FUgG/$\"67?7?c(JjI^DB+7(F7<m!GKu:-S9<-<l!Yec>C)(j&PRRVkQ7=B&cGemSh=,D$;3[(]KQ\\\"WofUD#CLKEc`T6L>EX3\\]Y%t/W@14_Z:ef<#%*pq[SD4lnd*13G2C%k:7YLXC;HD8$$.aKaEq37g]&do]_G5QX&]$?cYID(_+*eh27KhXM@T_IB>,hD$oT--N;K73=MA[A^<?@g\";#gKfiob4/HP8CG7-?\'p?g=KJ$14\"]m2JtG73jp\\(N2t3a?ZuDoFrn<l;7s1,U6*e[6=i;,D(Am2!F5DrL97T$PP96@qo[9fdH$UI_OBRL0gq8\\l+)>,%8L+tg)GhS<bP4&.hA++ZC)DdhTuM[rAtRV@7b7jrkW0L7:/==*R;p\"bUunNiW\'&/s?(QfS=Lq.%;m]8Y\\)Al(trWO^nk#]\\,&(*XjM[\\1(%K8?Q?_85cG9$o)HQGU`YhUY*X8lp@dHoM^cUONg>SRE+HN\"FkQVJ!bY,AdSmW,ctDEm\"=K[QM;kOo_pSeh\'RLg:8!Jf=[EJ^5aC>qLC=L>B!ab$eM.7GER=UlIBIu2@k&^!itr\\4YrC\';]Q=MhlRN,(h`F:H^_%[mq\'_9&C-f%Fek,Qiouc\"kq*o^>W)gUI$O%^1Yr$D[R)gB4sd,8fe>7LS%[uiFWQ^2<r9iDTq,k##S\\`4)lj=BhDc9*5L`\\&bZ^_`H#[JuISq+1^F3eh;j%8`[5X)\\7LN6OP.ZNII[>&<G$sFM%[jo0nZH@h/78mtMiIU%rNUaBb8Vd-ik9\"K>X=mtS-JRCAc/gn4U.DW*Eb*i.MEq;ScYHAMIT`q8?r\\#s!1p;O`W.MiP1m;Cf:$V!!:%/W.:.L.\"(<RSb]W\\og#.J\"\"UP\"@,Zf0]$%P\"l\'_ON-[1W@qK(2`DDagC+0RsS_R\"^=d-:;FEh$?$Rb\'?\"Qo-J(.9$+Z\\`6p3p)n\"_K0`D^F^%o/!T&rIdrI6=7?WHV0ZBI[@=ouj`\"r8MAM-4h=33(&qLi=bDXH^fA#DY;-4bA`SH!#D77B71k(!XB-9m>uGqY+T;f<lg)EDla\'Uu=\';;j0FmOp,GX\\eHPc0X4s[#bJLC/EqUK-KQ:od>L25$Gb:9_652>5dTHb[oL=dDD-,,h15I:\\ENB@:HaL4*-\'WaJ>0kN7k(5.f\"-;V<h_nn*.LW[V1^<Mo&\\JmX)jW]!^?30MHbG-W5\\_]2W:/[M2K\"?JBY\"UENLL9asV2A\\K3a$dPjm$14dh`ZldKWlM(sMM#PaXeY6G1*^U*hs`?UA`UlR\'WhL3ch\'PO#n4\"$X8_j7<YY5l14<\\>NNn\'ipI)8@Jt9b.QQuph$E26IF[Ob14FgMS\'\"(P*k8)?KY[dT9LF0kEP`_li`s;shVcis\\2tnnu`aZWsWYpOH:J!m<8XJ/1M5D5UcUaF]`5&2Nl^E%%h<Eqs/+lLG;+3hhW->-D#fP=tSOT;EAUupg^.d[(&egd/MT7p3PjWBVo-b\\+\',N33d(dLu=0akb\\L*\"K2=L%7^M08EMd>/q6>l#\"N\\VO]$cM$];FNgtQ`JFbaKQs\'8&h@#WIdaQlu]bWFbY(OUkX-2H3?4L%+q#a^RhNo*:Z!ZM*+!h@@6n@i`9m8?!5,7]b[jf.8JYW!n876@QqG5F5C0Fjoc!*h`3(o`n7emeWh<fZLp;(EY.g]jV,Kjph3dQR*4,bbR!ng@mXoOC]sl6.D3;U8%_,mq15hQ?DmCFLI4_ji!:_dK4F!faU)@.M3aj#T&P<s,[M@7gtlQ1P?L%`e8-Y_*ohc:cr#>%;!e6kRU%3^5m\"</FNqU5YN[;M1L/PR$3an?&JG_`[Lt!kbOr2EV\\\\CC`tMbU5_mFG._u!iB;_PLTDbXNABeE]NbB+P;qoo86!C$R\'_U7Vh7\'Bk]*X39QX3BUa4fY-NN(jG*Qd6,]u\\58h$/KY<B-T]/D1\'?Uei0mJ@R0.c>%&1p2d=BqMo&@g8J/o;%MH#?jcBZOLb.ck\\>-@r$B4*\\]d;AF9[jid,;pqb7IG;4\'s;\'g,[$Am^M>\\=&.Ymc`p[##/Vo2$+>n<CubEb,A5O(h&GVYl\\;$mHA_6rkr<rs9e4(inu0S`p]h\\5=sPs+Qa!oS=V#0DJrJ6h)0YC4@C9PKK\\[X!\"S>_-O\\eZCc[YE5p[3W_flXdXA8SQN-dX*]gebO+/SI&?#\'/h):bPKJ^P7TEB>rY=cb#83%.SpNMYT.-1:@8Q3qTFcZ1]#-pJf*K2gR1nDj7PCV<+%nT)4G(U8Xko8l.1G`5k>t6kJ<R`VJ!tnf*XL-7>\"<Mm.-7@L%F4GlOJ[;)0PLBp_d$H<?Ks9^]Eom[8GX:.&_J>:Ajffrb(6c$(&\"C^Xh#!GBS\'\\F7/\"?&g:4m)B!:dS%3r\\9nN.a++4(lMp$7$%aZ</-gSs=KpYE4=V*Yh0PI/;UWlnL)iou13%MPS9K]/-cPk?D>AS;%9)&4I6@%:;4sbSit)*+0[4\"cR`t?JBL-*@:%)cP],#siJ&t\"cX`>%CW<\';g&R^ofcN)(D`A@:QD,\'Qe(8j<Y2an=XDcC[<4!q2D$Wf$b7MeU1Ea<o]]77EE:)A73hj^.QcBR;tmC&`O;iQe#l\'[IH-CW7m`R8P6<*iVaAl>GJA1:o4,p0<)d*Mg0\"g*jOlbY:sBn_eH%h_`V[;%J;S#:kK5Zh]5d.[q8*?b:(d4+*>^*BulDg$YK#X;2\'\\F]u0_Q/Z>3*$nrMGTSP+MEZW\\sM`#`0MZ()_@7jV85Ik@oYOWQ:BCQ76jge-s#N-n@\'YieI@hZ\'K(,^+?nsX)?fFRar2l+!7/&*.$ZfHh-,T>d2e4D7jVie>;*,sXK>&OWZ,!faFj\';9olG-=R]m#Y4uR^US>h6V@`n1=2/qO;aJ0;.>j:YT`,8h$Hqct\"GbFg;9a=-Dbku^HKU#>J((R;n22,3%>M&Ua_a[r8E\\UeKqJ%ol#Wb%QSGtR)E#aj\\cYf_+d!r34a6],\\Mp@1UsbW9i,pIaqB\\1BS2mr]?S5Uu-^/r9G^V&*]]Nrum\\8Zsm&3V;(G>[R:P!o)3GX`EUWF_q2QH?\'=]m3JbQ3]b;SfLUN5<]:\\UUnseK$$hqR9bCVk!M.6O$oPn3hFnV@3e)!\'ZE652DRq2DQj9g(]]J9*tck^IUQ&3R(VScDYE^W8hVOG8)Z)qf@7\'e%%@PorlVZ4uO)uMRr*rn7%ZGs\'V^=HN0M(2`2WiS(#-#<dd<\\>Pkot;AKQ2C9m@_BY(,W,82(`,Us9^?Y66?nik,#)Gug1fP!JE/9XA+>P!)mUc]LX<JOC969\'E-,=W#niA-RWbGm8dbPJ463..<cq)\'1Mmk$q:o%2om:V5HTCt_s=4,7kpRE&\'^dDCF&]o8P_MXi_gH3e=E;P8JoIjJ![nO[T_$DWa<Ei*apTs%6A%Vj5V0\\Xra>\'7bkpXX2g&(F;2r>$\'+nAiQ75JZSW20#f]<40Q1U6mh%FO2M/qHcN6B2C_]CLa)[\"[GBbF,Q]1O>?SSOo#@[;b9o/;3rRb4c4)O?ekpud@`#A-@Dt8PrkT;$.@#3]-N2^C3L7:mj!47c.hGJ:/=QYA1+%:-Pe$qost=%+gkRAn@9ch!dBg7/dm\\-!=@5I$q_cGI<TO*EMV$T-[O*pI.j`K^f_rr9sH21F9S_X]@Yf.:s$2T_ro\'L0WggAT:Rh_&9@N:31O;cb/l39<*9WUEpf#Eh4(,\"l>i1Z`Use4<su.O.B*\\X[%<AlKWMS):sQ=nI8O/c=D\\-2B#p^#I_@\"S119`-]GrOd2@2`#q`VdDqP-6fdoHC%kZ6lq1?imUppD?U72!\"t:I]1jY>2d)MIs,PVI08ZAj1\\IFJUMQZFAKcP/9Y6EB^E*\"XL2\"d/0#eg$g$:jqf3:`q11nSpJc%>Be.OT3pD<94Sf=]<`:>0K=dUk-SB0\\Q+DDq.DPMDi$P5_bsa=cYD\'-dnGs],-Hq*qVh=ZS=l?7?aYH&([OQ8kFoV>K0?UU&I8<0%A\\D=cV%f\\jG_lTm`ErUIW;81_Asj:O+@,4_o[s%Jf^\\!,T<_X\"Xe;X-=77V2f$nq6><pSCJ$Asa3:O&:n\"[\\X-X!01C@\"_D0lA^G6rbO+3]7Yitt!1jY(@A<V638-?K%-!L[(19W!j#q+Oh!ER^@[B3\"*[eqX/A;h.tlp.Ph^2rp1-fTE?PJ)<kgUh7OY/EX>(XSHUh$BnqDoA=].Zrq?S,*:Z?qoI9qiFO=3\"#[rCm\'6X\'s.gbE\"_,knnstU2lOtk(%&cJ[%uNim*e>O`fHP;.$mr7fI\'TTUh#&->]Hc!X:Yb%cOfO,maRF^6YY<]oob`XV(<Bm.]H//NWC:kq:/7aAIja:0EGJAe@G^/kRA$O&V4O[n.5Qmu;N(H;D#m.]#)!dB1^3\"s3R(^BPJ$9$8Fj*7.XUE!Y.6ZIo5isjL@UGa\"^)[E2T7*%7/L(?\\(OT\"-C6s53tDLZ0=?s*nuJ,S4ugS(T-qNSZW(:.fe#9*_%+1t`(8NqPbb43Bm1ITdPmnI$is]:DkTIkPR_XJhb\\ri*-3/LhbZW2+T8.VJ&1\"4,5\"AJ!FL&\",g>D),[Z#/RfLrf3f6Dop5(XDUq.0s/_d+n+EUen;*Z>/+9jENR\'s_I7U**U(#*r@jNMW@`M-1ck/5+rUT-s\"%0D$\'3JH5sb\\r!uc8np!$-Vp83$QgA5^l77Lcn2@OR3ROl(,>tBW@8GAl,WJP7^jqeha%=GolpNlt1.7T[juZn:\'(/>2bpEi-/r%cn!=br`+YkQm.9O9cd\\ZWr!<9ld4qJDjiSj%D%o+^,_&h>KIodNS:c7p,cbPTOD$pI)K\".)cY%@=GPRd/DP?#WM*E4347s.oQ8fk>\\3Z=AG2pUaJ;\'I4O91[7L,hk&Yc39QuQr\'HnirEjup`3U4<BI@i1W[i\"`l.!hTFk.g[q0\'k[A=a)Urb.:p@fT[Ih$&cK6^)O4Z9gDZ@e+idNE:iTZ2!ILeqoFtr/[@S%):G(_&SXeBu.i+Uh7sWWja_k9\\!U9\';#ILc,o2,/\"c!\'4b]\\N&2\'*!Wl8<XO;:\"VXFU6,E\"ZE*fj\'go]+PqX`MhZWV=\'L/q4fI3DP)WGs$*Ct,^3[ECIGXu%6W^1OO11CH5aARL\\(2n4rbW`G_98Q8PXPf[jp=tnf9XbF3]A01aaU-*@6!->p;H`#\'(\\3rQqE,LnLL\'ET;^+/9b6>2]398cA:F[.%WDdc=D]S0Tf$]3!+IsR=s\'#6C2\\T%RJrBG6lFW9n)cMutoe\'qO9e6-2.\\\'1tkWP\'-3QJepCX)^hbQ<\\nP0?(s$>.5_2s\'n4X<]B@,JuBo.ipgmoSTtSIOAL>/#BgWs8BNB\\2*WKT>K)[\"h\\?7R1uZDoPfVa?V2nN:R0#h.@6:J!tmV<Uc]EY?.)L@(jO;VP)3n<d2DdoUEK.kb\"2,\'$8O*\"N9[oJ5E)$PebM1V(]QNB\'uAffF]i7D1!cOsH(5J5_mR#]AO`l\\FX]DIUrqD+BDbcr&XsC[PF?K,BZUod[8\\00Iu\\VNC=JMkaI)IZ.a:oGSaF=m6kZ:,s*C^;FSakkOcd,mFV&o05:.JZTn!8G8k+IKPHr@5#)U`mn(2T;CF0bT%JBOtTQjBe1+[OX<T]IYG$Kf:,<d3@Wp<\'+ILh@e?9>-;K=oCi.9U^6g;j^=k<F<Rk>0![jQh787EYZD$5[=?8T@R:bGrXd?2S!9YHKMp<Ni*R0\"kl3P?2&\",5-#0?BKL(\'6U<92s[hCr&Y-(\\a;o2\"L-462QpelTR`R*&7WiP67u_kJ9n@rPFe=LV9hShkje[.K>3\'s1nI=qhh:Ic9u8W>>C\"j(C&.XEh;l2HG&bbW^U1PiQ4\'UQakLOoI?^a^Js!PB\'rECVhGct.)k@Buh>QYM)3=4c:R7L(IE*kE5*^02(R$.o=f)-dmV_,>E7lb\'EYf%X]@d_5G^N+O51s912bK(DU6pn\'?SfD0#8^0N`K,+(Tn[;V(EthJO+k#7MtNNdPlVVG&snnm(XDp/p,SB4UKJ/66Z<Q9kBV@]5?o;21(\\O:nt!PQ`\"/[PS@Vtf>@\"h;/s:)@Q&hPr_3EF>dmKtrXGs4aE*d%BOqoP\\9\"rg/e$IPS@AK0n&ED\'Zs0OO^q!.YYJ#<-!W.n-:YiuLr22_3/M`65IqK:$EWtXY!8+Hb.#n6dsA,:jCb*C,&hc$9!X1(2*qCub&]WIRa9Lpu-b<?eJDChju0@X&=p)ETA;eOFIFaDdFcFbUcLgA/`aT>F6IZq\'m/+A#aO^ZdjZ)mjj>Aq13QtTA!5UArnJ0^HU^H.)>dlqlN+$!Z[YKs6ZJH[!Z)mP5e=WYj:3$^34h\"?8hqT4&q<s5o?05FUhUiD/;U!tP5?5ED;4<qtngpW$TT)JRV5?hVGXcC&\\C-+b<pST@,$GlWLbH9F8+bthkiZt%fOmVq[:dlS*kMe%.qp9>jefXO2nB9-e;-B=D4HFY#cOc)ekI]PRNj7dl9+W\"%C#\'=hZH$opkU!9tR(&oDfSuAZ;BH*MIccKa:TX=Im=Z62GT1o8hlgFb@HmU[[OF\'q*l?clm3WRCbU*JkJhVhAJF_k%QV18-P[j>;dfq0I9ouGM%tek&P\'!M(9n39Xg=ip?YUg;]2As<I2pr4`%j<=?$7(4.p86cVf@#ULYe,iV1U0DP_EGdY%XO9KSHR-;Cs(jf]U^>&i^]Cjb7t/f@kSTQiR;DMYm45>FhVag62u$r6[E?=W3U6aQEDs$GGFs/)eIoa,tdkcK$:N-SB@AHSse0$E4($K;LNP!1GL,LQ&=LE]$n4sg^*?`W`7[`FV:M)D-nXffQY0TmnKd#&SC#$/EZMHQj#^r_oQi9Jc^<m)/hUL^gcsheF$3,UkGDS\"lPH`nOKR6=jeLn\\29gnPTh23954Ib/PpQH_%c?J26_?.n9^Imk.T3K>Gp>i`@Go\"ZB[\"(5p1B\'XqL[)fVJ\"b!MH?anm\':RS<3=afdp7N4`\\NmmIATe@5+8GQarKh&qp22\\iJJri\\,4,2l2/pcFOW(>C<_nU\'[jcT>fP4SUM8;a]-EmLSFb`EgFpOo[lm;]\\ZD0[+W*DNXcad0AOo,\'7dA7K;qbd19%Y;XE=0W#kO@]=#oDag6?8M<A*Do$-t$29>MeJ2m$/9F3X*g!r6Q3DL-CrM.;C:?9k)\'Q\"BQNG+L1lQ\'VGa@?Wau@aX@-2+DX`-]]1JRYZ8])+\"UXTB\'/]PFe,&mbXMO@:\\_mgo3ELl0L=Al@DjL0[=7VX!V&/!\"$f)FM\\N6mtqpMKT^O?V\"T;#=OuVJ*PHiB+u@bLlC->EI7l+r]4-Zn=!m%Kn?Lb2q#HT,Z5sf6__:*mlPr)NZDP\":Ibsq[j)j;U>6-ZTpF\"UiQ!]cPf0Er#\\Hr*+O52e2SaE4IGSCp160\"%LJ=`-[V_^R>JV;..e\'_N(n.(l5>G#V<mkV.(&_7VK(o7$!p\'m+fH,J6/Ek6WK\'8pO(]2PT!lY-=V\"I]7lBs*@?\'J,cF,f%IY;MjIa2WWb*NPse\"QuC;IE<TMk18>G[hmYEK7]-3-F8s>]aC#>f.>sX99Q$[JM`D#@C;U/XPMA7Gbn:?fqKYEg\'\\isDN!>BE\"it,HKHcpX5^t?#I$C]=eWOJY;WAMs@AJ4LP6D7bYiq-:W,oHT&G!I6naS-jS-o]ji1pK8+QB]m7#>K@rLF>jf;?b9.b7IjZ^6I^d7$7D?T$sn/8Z9KfEN/:>MWEbHB`<qjnEfOmODHAM*U.>>S5f$VtWBU%1dh\\-2k*OkS\'6qLUg#)7psd*rdX4Z5Z/8&[<u@.PPDUkNa[i(PjtK@fs\"2#Ns?h3VK@tmSHh9J7HgoYW39nmlm<cX)BgZ)gd_\':ZI6AEQ4kso$GhqilP@mr\\lKCi-,2kC&Xr0bm\"lAK_X$Qr-;K+$\\-bbDqda(0`5AjRIh6EWTSo-FSu60<q(eQ\"/#h@dbVU0>G^_XG]7`5lpWD,7-[\\F<?DEoB7-*VCP$!8k/7dia_EAr,/Wo[>Atepjs/=o$8W4erj#VUgiLZ;a?S!oL_HOqt-%nops.=n$P6k&\'1\\N>YMQ]NtJ>>P[8)\"h)B>4>q7Fr+9W)F:X:m<Se@iUU`JHs=Y)T\\RRnB@_bn5J;O?\'rakcYnl2HHWGE&rS,8?6&W>3Eg?2MY(0^T*pg\\k@aO<1WAs;EA>b(qD\'fK)1&^cru1]%.P4Xdf-L<=rn1uUGO)5ai_AUFQ)-r=?]IAC8i<qX(/,<60J=XtfeG^ES6BOc\'_suSJ6@B*[Ar#.A5r?3b3CE;%*A:GM*bnQaU]o38MX=_A\'<,$*ZOZ=DWGZCpRQ0WDX??D;>D>uRCgI]<[9Q<4#VV:3a^q[#V=\\\'^Y:Ac&&0,3m7/q(P=YFb!>Y\\[\'$17p.R:[hU52>S&de#Y(F*Lcn48[+\'0,/d89g%XLP(WY\'cr:_n:uXg9>X-g=%-j0TQRDn0IJn]?\"t\\IQ;#[\'@ei!0hLB;3/$?-I^TjW2BYp-X]6[@62A[0h++%VrXsNMTM7@cWdR!?]Gf%-r>[RDeL<=eN3K7Q63jUBU_EJ5G5NrOlaC7WN0.Xr/-[GaX_01I?Tt@Qd6W]bUDHJrUCt3qa%`Sn46t@H;=;#T*WlJ>E1TJQPnh2gU$oN`I\\#1OYg)\':GQA;at^5T/;KP85ZW\\.#MX1dAb=PpEXBL?bbp7gP\\bGghoPTUi(B%$+n%%3>I*A]_p3YtPZP#4qJ.Bk]bos\'s)b[2\'eiWaqM\'C9sBS3t#^Z6GBBaC?mrO.R>(Ue-#pcp\"6%.WgsnUKNSZ9,O1!UUs^@-;JFJG\'KMbId2.<`J_CcQr!17[gP$e6;O%sY&qPQpiEX;?d68?@U)KU-?djd-t$*QN9*uBg3P)(<#CicO1@/%;!:YIO&RXrrX>F<,STQh=T-oe-D0DrI7qYN(WuEJ>7t6QlSfX3)mEaOed5\'7iPK<`NW*sh29KT@Tke@\'=(PH7F!Q0EPKbJ_?H;umQP3(-HJMH5_\'^0Ne;+Gokq;\"XD90AG_&_P*<e>;EPqP\'h-ZVHL)fT,1/0FeI8iV3.\\7fq+ZLJqUr</Ac#fm-&%U;7:?qgDX8lP7\\_A`e3d:;,g@,ns@ERQD;oT&PTO:c7oJ]P\\j,fTZ%K=I>j\"uY]/hTr3*>1S>QV_i%u%r:\'R;e/MO1)SFo*i9#20J\\5TLO8aN:Y=0P>6,6o)`nO4@63(2pFRtJSLn@2!O6s(bP:RHWnL_/*\\UoBEpH,Yq\'tZi#aUG[\"Cts(@@ANMS7=%BMZ@:p,Y\'uFPA?UjQC!lEq+Nd\"F8\"9o5k<p#9X7<]cE=-E.]_2Zl!gqK;q$<);b,;ka[_NZI^Y1XMnqrkL01(/BscibO[#Ki,Bm3=ImZLcdtdYqNh+%+N%Z[M+3ZnW(GJHd-[kjqr]G`ci8<YMB3Kcg>midNOmT;7VR9Lr5a!dCA*-P<2G%dm:1MUYq[n-<li13Lj$/4\"6,Fr;:3?-Ce@_SHf0#8*4D(pgIh:_*>=3\\GRF\'I@>d*fQSQXAJPUYpXS8#Bc#8%)tD?1Qilaa(4)!r0JE[o?\'\",!<A3UGa\\R*bi#+QinCIZsHHU.HX)i=];u\"WE@R!]QfcV%4bT:d$$XVh5Tcj!(6Vm+3Xa@p*l5>eBA81,WR`_=:?IcH&uQ)O631\\Fk8r3N4W0FXkWPYI\"VUR+psBHYk>eF\'__g!UYY./MF\\AbNn!UIT(?>5epnHfRiAb4`tRBO(P]BH<SD#`?_&,f-Ua5b4:_iMPEq111W*\'o($pkXLBL/Y]Vs+I>^W.:l?uFeAhF_Y)2t\\\'t7Y`c*?Yq>LVS,Q-i[I,-Y$*gRAgD8G,Em]-1]>>+[ej/#k,5Kd@sK=O4q#/$dKR*7Ubo-81ia\\=K5`9J,45FT.!,q\\VXhmMH&o!Xa8l&(K<&/FFa%p^A@I3;mb.BsSdl16s[rR\'R<%GX`JHgR.=@I>3NUnTEeM!S`J4G5HGec%25PC_j\'D0!/`j*-5[-QgFF\'O*8\"geP@(KHGH(M0>MJ8BO`C\"8!a?ofrVjL*b2GqdXW)*AMO;*S*80qrpN.kXVY=tfY.LFR/TkegFh?n\"R\\SbHT.qTn%M/5\\h4OXFF?+O$g_n]1j_,VT72Zi4kgM(e#2BAe[%IJLR0s\">$G19/73>Lo^/N8?Z4g!V.H=%[?UnF`[OYM%ugi!#<rU=-pNh_A[!R63!\"Z#r4Jl:g!\\[.cu?50h0WLcc2@_##hE/^naAY\"1]>/Ee))^l1kphY]r\\O>gWg`V]anb!O,:Ym86^>me@.%sgg_m1[eI?cd9K\\B<Q\"([qh,QYd=0-OB:Z.\'a8(FZ,JTRNk32g1^V870R$_*X&dH5(Ij9a12!PK%hHg.95$33YQ,M?2R\'c_.ZJGZ__<\"[UK*]Y<N,eT#\"Cl\"3n-Cgr()sdWcGK2SfR2b%oGV7+>#00/8h\"5hmHFcMgZT_m/80t[-(?m9dBQu^4<j_`#jP\'a6jA%gTu7TjULoab`hZ:\'c9ChOiB63LTi`:@E-\\]<!7!Qf-F\'/.)!>W_V[\"b;-5.WWc5F2uoRr,JqT!.#D`7(@<D4cT`1#<<W.&2n.G))`NQLmh0d$?4/\'^!kDTJmoI_l_`6\\S!V^4j(X$9fqB)jhU[ICHSk%&<:,)pX@6Eue:SXAE\\)$gFIL<6kK>MiBU#J4chsI7k-KeOqhVIQ9g2)e$;)?d495%ZMp^aG0-hh#8Ng`VJKg^E@U?XH(MV)kZHc8%lAg:L6?`=KdH[7SCcLk[mn0<NJ0T;Nd?4@nr\"Q/Ti&6ZA_a.-Z:Y*\"m6$UiHC816Bm_DpbGkD`rY`eMI%V+I!T\\MhMOq[JW/A%F@EO!DMt=X:91>GQ_\'Zf@4@Z\\WUJBE4lteV?.R*c4Ct^J?(%1r2nbr#Imtj\"KDt`9fWa4r)H>fKe%PTdKB6#AP0S6uR:[ednIrgV3DL1B`MOk=Z7:ht_qIL`^KU(81Bd9&,8SE*FIV-N3ZLp4ef#bsDpVPJYGOn+iZRKU(nGAOE?#mOK>TH\"kqNYg3AR0C2L+n8`<_pVB3]5\"B2ZCN4-tKf,WglT0qaZ7.l-p1.2!k!_&taZ9#1S\\:oqIrENUN%*`aN5#U\'BqI4S1@&J+=\\KQ8R[cWOgubpW.7[\'_I9=r;\'Se(Hl?ETbY(flr_oBj\\,I?Zd^C>:B,V1tiU(rTVr[3.1R9i2>`Ub^K6GQ)qAMY\\_KB,)[,N_/&M,?5m-0X2+L2,m\'8S`#u!ELrCOa8G-D#TlZdP;\\4!ZJH_lk3(PEQ\"C<A-+PW)03g\\JT#ha%*\\da>-bt8)g1Pkf\"0cB.=\"Wn\\--SYhioI\"FHd4\'F^>_93U%\\%bCI)0nY0#D%Jj-H$_Zt_>hiLO<,ps+oFp@2J@gg1B&GO@t>HY:AP%6K=ss%MqZk8\"ed_lLZ5BQRQ;YIo7Jcdsc.$ql?nR>^WR>h8o3*_te:B&?\")=^_6e+tGRVf6nl!dCgfFolFtbR/*%5p(A9Hb%*=rrG6D!To9YIC&2Lckbb)tS$8^!p-1`&2>`ZF;b#[GO1\'dU^@77AeMqa&:9(9mabPPNFI2eHZ+g:[.b[p6ErCc=SWEW?rKtqOcQW-Im/?/McD$`B]DX\\Nb`%6^knc+r]k?eiY1t4P,b45>gn-*#q>]RN.7M@(S:SJS4pNQOs!*O&N)!U#T=/+pP*6nX]=mf[U(;`WC/XBR;+%n`k[.>a4Mu6:.J*NN`UNY[Uj.V=IFbaK7mHW=7%9(1>?;MpB8FO/0-9>u!SM/;?Wk!H>^4tIBVn!Lk/8p$ouC]g#6G,1.34r[E.C=Z&Y-\\5%I_t&>_Fsl%)VW\"+XG$-bHC^:+V$F@V)Schn3-pHR\">M0MgkNK!m9s#bbP]*C=5u]&kA.[EC^^mRZpKRY_[NJLrkt1hPYn5ab6qT]\"_@eVH*F#C\\im8e5Tg*9pdYW8;J8F2GbIpj;RWp#]g23Ck4[kqu1pP0iT<N3dNZiK?Dn<L\\K47f310Sge0cg*.WN62*An!+k<3oa$BR\"%=9?bbm5Tl;\\I4Ch*%C9>Of3n0Gu\\&qh>j+@BVHEG7Y?cY0XE\\BQf1dqbNatFGK+i8(Bi#b.\"I?b9.dc0=k![;OD0GiI60D9qephl%DT+%sS6/$%8Q&.DD$;/)Tsu[_SE*\"tgbD9dI<J_hK)R1c0he9-GmO_)MGY7o<Pe9RTtQDhFMuNsfo&EO]X@i#*%!1d<c==LM`(&M2p\\o/kqS&BTFrY-o[5KUSqlH>WULJ,55\"4[V3s&g$1F&c)p=nJ;N05hTpC\'On\"-/G8ROmWoBs&cW`&j(!)3!^b.q\'udW\\`WsJ`kaq0dh>n\\o\\.Eui4)?(Uou\\WY##oW5SJ&]u`BD3#T&=QhEVO=H+1i[gbEQ`sWH3pZ_A,ChXSVP^0Sj*,nSTk61+EPTr:a-:7j.@ZYd3N\"Ig\'ZDO+W]DY]pW[`bJWj7D;N:Be>@fr8pMETiU2SC9UFiS0HDc?OJH.I.m92[uG9OrHs]SkDA:SQ%4:*>sqTdl(<#kp_Ol!95(\'skqG#RQ+pd\\NqUe[$tsM8=0!bQ=bF<BpsU82BqK!arYn,up9M@2D_!;_1&5eG2XCo)FupI!Nl@U:!0@6+)2m=E/P:2mPMQ8dYEt\'#U2gcE-HN)Zi9[.3?MoPCcZe/O/_C/+9\\DoVcd&@B3Q_cn)Q*,o)U;)k4!o?2!;s7c\"=$n[pD@JS;ia`Bij,a[8dH_M\'AJdG3\\j=,\'n$8Or%muk.hTnPK(&Ia(Mi,Xj^pO[Sg`#(@X$]-$!TS4Md:njPS3Y-7IL6.11J&q$3\\bl979T\'fek[!r)&2#m\'rTuMp[!M5s;UX1Jq[`eb:?LqDC3<,Poe,\"X&s.,!\\sm#0!Vr(=`fd^8I`8pTK$c6lZP[<\\mr6V\\poP_Wb8W`$ka2;^VH!`ZMrUWS2Um!S4oqdg5f!1@Bo-5`gW<>kD+G<YiHb<B5V!)=btt=OV21O[r\"RcDQ4[(AL%]Ce=Y:3rbF4p?<2Dm\\hOPJBocO8>9BsH(9;hDAi?m)2*d0=!h8;mE9!#&T5FlFUqe\'Qg9dYIu#E*6umK4@4[qRB]:kSP!d+RN<3a_[FM6KIWjTra9b8aYS\"PgOZ!-VQZkB\\6BaTm\"rbShQ`l*2l2Z8i!^HP.)r70?8/-\\!Kh/!TrlC1Bp\'l8,d=(kX%<)+.e-r].GPe)><I2JAWT4/dZ,\"<q9eO\"iAe<:@`1Q*UjNa!b\'Tj%N^A[Q[Elp6-qIN&Q7N]o]lhrZlbC\'%pCf)f;B+HCDl0tk!ZfocX*rUh?(N%CePFm3#`5B^Q%h<6^0Cm3:h(?FQg+2I6hI?.Z<@OGZ@O6_&UEIp>XM6c&8H4Yr\\1_RRcgBoQh>4tpZ.M5OIjN`GV:d9Ek`jMF\'ahGilDIH[P,KO[;J\\.SU3<E/G00irl^le\'j.0..71[WK\\;NZ*r0m*BZ71$gGN6ePRJ^-Nhga\\\\rQ,]tQ-.bm:L=1:!ZK-.k\"u%\"P:WI079RWOQOV1hH%5>`eqIK%,RC5eNn?.q!*D[=:XPQkFBX6QW,a=/YLN.V$lOtf\"7gra\"ufC-LEL8(c*\\iV@$+M61sTFc_Lf\\jBl2!_gGbDq?b7K(flqlkRE&L:KM=XJ^\\H,tMrn,V\\qfLq=EB,;4>&K7noEMdJ2c?S4NTC:bqfN@Oo+\'hOc(hCgGUl4E%Y,#4MMdTs\'bN\\pE4RMs85\'m%UJj_;oIW1E]AbX:6>TSjLUq$EIRtMa-+NWA9\'Dm<)bme23Q$T&EG4&I,V?5\\kM7?QK311&D<\\5S4UAL-6=i1Gq^A(ntU_`Mkt[#+=4bIRlUdE.a\'BdGR5tAU.PT.pL<jU9lHANEsBpo#a`AR0D6[b#JOqd0&bj#rQs;_8Up+5mR1^GaCj1bX)-(-8VYu[ork=eLP[.8+P\'st_s_]-E=S[f;L0dMm?!,)Y%WqGG@<-?54_c/MSpbKHAu&BEM<G2FsFOc5u<&E/jo%C;J0*UNT]=o/NRXG.=f(&JllbS/jVnDg5\'o-8^f\'AR%*KNla9l5@j?b2G;ac7c$@>ob@g4Jdof\"sR+6ptD>OO?+so<A-5%F&O^(-8+DiKkZ@`AMTMTjTX(aXEPasY2\':aK5iRK#9X1^Y^>t_3?o0N/\",(1P+h91K7].`P]m(d%Yc.QLCIY+G:L_I@<lki6WjE(+\"ath)WR:b-ZNWNdu69/AOC/HJ#eNlQe;22\"k\"-as57mOAI`ojY:a=QVL?DuX%UoNEB@Y.%&=U?9,cE_$TI?YM0HC.b?e=lon9kBHe-3ttV(X\')aBnLc.KnImokBL>s`G`A/PtYQSr\\m7/D8ra<:g>-?CX8Mu,)4b!fAEb-HFmp2r(.BKjlci^ak*!n&UYn5Sh$)lU``P2bI3(K9ts%K1OU@:&&Kj,h\".QQ!mg.c\"oPa.?=7Ma<(C\"kkGZe^mIPL/$Tgi:4$Dacn./XiA6qCpk8t7DKE%;<J\"nb-J(jp/EW-=QIYf0&C&J9j/lZTI;gX!emJCrK%anBZa>ci#J&:L)q_Z$Um_.-10_GR6\\q:?.=+\"tQ*If(d=0*&7X-@J\\Bb+!)9l0Xj[m]70rNT[GH6H,k&pHM4EYtLO_pCB!4<B)XaAB2E3+Ln<[i)s__L1p4VjeL]KR#oK>d8ODaV\"#(,qEs_anWhG`\\%gP_;E]kSD;:#+&1>MPcRa*!D+p_UO_G6TGBhLnUapSIn5DNf,%otf&ZC\\++B(U*.AS.%8Ah3UKtIaL-cidg7CYFL-h\"mLBf\"frLd6k)p5T,%D7,gcAL(Z3k+&_<j3ZX(%F^31og$rF^MTFO@m0c\'8(oUOl.lji?LDUGfTX`Sd(:lmI>,THg$@#)VXh\\hm<?6B[s?&nR__g,VHBZ\"eM-*E6;=JP6k*eZ&!B(`$Fp\"HKNb3msX=d?8E_-h3lJ7TiVi)pK*f]%N1EWq%SUrL;R<q?TU05g7jq%G(^)cqL6`hXnLl0W;\'e/qE$<2/s5sG;-qFX.HB2Df\':R8JE\\6aGSq@VZAit2+[/9/H_3Yo:>AX<6eLA/q4Ca__HcLMR#Ln.rCs=4GV]MI$4EB@iD/C:V_mg[!k6rXPH$<*?RuNX&ua\\AA_O.:9U:h5qD)?d70k6oTA@CY-dsfF;t-Q$^nsSoHToJ&p;tlWC!U)L*WV@VnKRm11mNgOM$6:KffcA,>IX;NaD6+_[F`Oem;J=?Yc\',R3a_Sd`^5h*@c>eRgUVFKM/h1Mq)$)p!7@!fkf2I)Z39rqnZAKD4(e^,Wl?6/;NZ$a2e`tM(X@S&rk:rof_hM,^:2%\'ATVQm:B8#hP]q]*+\\t=]>%HEFs$IDq)pK\'6ZPj5<`*dbc4]_!h,Rr&TFoOOm82@/%P*LeKRoa3h=1Bl(:9R3n?bin+A68=.Y=JO\\iOJTFnKQKBGU]&64#FG;\'8@;GPC]uMf`tXD=L8OU[hD]]PQSqE\'Q>N?3A)&S:iK/;<JNX`P\'GIMM1LiFNa@MZ<CTU^hVT=B-\"ZE!O@dd0^r%-8TK@>4+I$.tL+r3QfQe=X]Q\\_d7P\"0>%XSYC4,(YF>\'_U#9+G\"a*FZHsYer(-^[/X%jqdNABEi^0M\\,l,q\'Ra:H2jV+=X7flP=N;1&6J!o?771Q^OqbGIhQtG\\]GeMN-Uhlk<K17+7(YjV*!9\\-SWS=#BpTOG^:72%b\'XW8H<rI!)@u@@>E7t38?hOPk9g=PfdNZYE*2n6S-kPMnSE%+o[)u3-8-iSZRr$6u[\\0O^4;en;1K<8g+^/Le\"MDC+oq\"`;7q\"a3:W?0./?\"nBp,q>jpJ*Z[E[PL2Dh/h-jo)d:r`#!_AbXqd\"RObW[9gc\"IKHEP6Y\\L#Q+uYS,\'UQ$mT.?Taj6;3-MPoRoZ8`*SS+X%j*jpjg$-ES#E.1I\'E:LD#MUUi.$lk*f<p=ms)C`]\'/%LnYmu5rE$IQ\'hPNjl%C\"N`eab?Fr9\'du>Qr\"FtA@H%qVRccj##9QO4Ejel7D:U&dlHYAlGN,r1E^a@VckJ7!5*3Pb_^:]\"c90QarO?pc?,QI%Hl8kf,R3d!f)`F6d^Gn)d&WZT#6ScFld&#;8SH#_6U/E/YE?+tbM@N;[Z.#]DnA*t3aEpEi\\UF*]Yjf1kYZ65FA70rrk-Q>&IT(t:\"b)43Y9clk9qCD!Cg\'3?<4^b8_/E#Z.H:BQP+9IL+H>XtSgH]s)f?e->ER[Lr;L9K1cLbaEaTn<I^<h[8\'(AfaB%^W=E[CIO1L%85bVJQp`mV\"&0c.)&c:U%6,`8d`JJLtI&NPjea\"tY-8%o(I,?n]/#Dt]UHgF.iADK(Q,hGe&9_/>gB)d,D-B.)<m2!G78r1*<;6Si>q6mJBS!Y1e/^6Ur0jD=V:YqnKIll&ZV]Jai7/0%1cI?FjN=)d>t(Ru;5r;!e8;<(R5,I`\'l20P$9rG\'.XT)`g;n!W\"`@%cO&N#*@Hu.ODYl26HXQ\"BQY8m-B:p8[,_K<?n+qSuq=K`;cIP]OORt@]1>s39C\\%MHfrV(a-eYAk_P=dkU2OTVRILY&MpNGUW[>fd9c_[pJ5/K]^V9]l]Ru8W8_,U5<$H5W3trA[K`^X0ZD4ohk@d2]$Y/WJ3hCGkFMSJnYm.?_loNP@Yk^tCHh(=g03J!t=0tP7%qV?6X$[s5[-5CNfRmI+\'=@&cheuS\\j6&O2J3#$p0Q95F[5PqrFl<1J+epY\\NUfUZO-X_*E178#<6[2%/h1>&$3+Bmqs<@.Q`da-$jN&lAsRRV8I>+?oXjb!50$c=7)kSo237Jf(2Hh]c9\\m?;-o>f3s2;pYNVg\'0F9ep_<[4ir85`u\"U?-.gi[T#[M9O.F.r.NnS31$\\bYMP+J\"\"!B@`^VUb\"3EZ>5p+jE0j^-?M<l1Wt5p1,m\\b6+q4\"*i%O4ZLa>\'b\\M.@%#Ws<q$hL(\"?G/9[NZ<J/i(i\"T^LaDY&i4JI<N=IjSQ/7T=fZn1r3>cPVpOYWQ6DQ%OCZSaQ`(`bo70X^IcdoO`=1Rg?fkAJ\'ODUUs1lUnW.qMO4f1ZXeMVPd\\OE!Z:0O-])bq^*rgdhTpfp)O.R+&jDKTE;/\'FhFeUt.$fem&dlnAk2dWGRW5igPRPOnBMLo_.5FnCi_T`N%M9sZ*h@SGh\'XM\\/<i8X)NC0+G&D`r0.1S&INepBrb68G)ZusD`o7nek49_aV0AO&2\"fN%B%HPT(\\4s+RPt3+;U1`/fGO*T4p>Kc\']A/t+\\NNe+A\\LIa\\IfTF8jbhNNZ.=V=#iVH>9Z@@#U$p7ilBVCBq?=bcr7CT3i`NbO[o-Ag\\QdMBl\"?D?@cmL8\"NaRc81Y*Mro-lf,Vdfl-95oZl)c1Vj+dar%,uRN\\gBtk<\'i>BPGS+V<**P!Y)7ale0>f#LpI(C$i@^igofn4dY`q^gNq\\kf>l5XMdk;K<@uV#.Ke93Vkh]R(GBOA>`/aVXQ.80@G,<E*Js,X@+o<24.*MPA5\\ci965+RiIWe3+1=%-3;g\"S$I5e$VZ_Z?=nsFY/bj`:U8LnA]i,k]H\"TW.0k8Ag4=tn<XI#sj6gNcfc4q%=pdER9.Ep\";nlP&,Cd*+^DHcXa-pg2k6=\"\\\'Y*#ZGDH+(b8kE;*7M\'iM+=.Z(bl\\%cr(hV`uaRhKW@mN?t2Zpj8a9mEt3HWNSgs[!g8FRo`;_K.#VP[9=hBDrXt,fmU.7W?J#B$CRLQf0bX/LauisfgIZVYN0o\'WUH->YJr#`BbLh,kcssaQ-\":&G<cKWu-0X_S$-Ac)&L(1lNmQD/1SELF<`/_H9gV<tE\"G*EU[n$sF&6^QZ/QrKJqQ`qqA4#C*OmSOS,/6n-*u=3<P<sn$[9i/pm/QD>p0B%p/Zo+7@eYeDU&$Q9C%gp>,6e,mF(,M5YL6\"T$7-)[M_f:nD-s!/+Lo02HO(le.ioAYn7FSS#stF!SHPTp2h$^2!!DIpCI0HdI/6c_+\"pq&LX-V?fp;W4p(%WBOCiZH@g\'Kd:(Z-`$%$_-mEc<)dVge-HG2c3K-=+:=YccCoKQZK,`tpa43pu\\_027MU%\'\'QMt_@e=0\".\"82)Q.rRH>FmT1.N\\_d.&L6+V[2G6#l+\"\"@f0l.nFqYG`#O`R2ai>!s+WM2!bL5B4@qu?<eortqLN9KLj`hIQXP]A3iLn3me[(\\.l7Dd3+4&mSnEj7rc])[9KTCFc4i?ZA9oP&-\'\"0S3+p*,?hs.`UgSte?\\Dn\\(b(B#;\\;8l[HHo@n4_qhLFuAn\\][Gq@3gJc`/epHHN?\'@H.6\\O1%%9*Q>eDNB*,f_S*Pc.e5;]:6L,5drESUC93lfR`>YAqg3>QG_$jV^5`8W-0_\":4\"aCVe9B3;_0>!\"&D0M\'N-Tp,>A\":^f6RSLe(g./ug4FTUk6L7;al?sCQfC\'-,X,ju.K[U\\i;#b@K#4scaT@U*eft`pY^ch!GR;/90m@I8PI\";dN)?f\'+:#%3oPP_ccGPN*$%D$Gb#Ipo1l$Hr3=,T66)B3o+juB8q6Qmqt3NSV[D_@&-ai4Ct9ml,u%NMs$U;2$HD]0*rl\"\\k]e[BRS%=AF(V,Jl])]!91\'%,\\8SiCr+WnE6=dCM[\\Mi[uVQuHBpEpTGP.X+=[2laqQJCm2oOL3%s-a2&Hi8*Ym/oaJ9REGbrD?0mo#`5`r3q*o4<,^mN)H6e_V#][+6AZ,^CBXQATF<YF%S5N[8&\'i&F/%8Ggu1>u=`X*f*(g+YYc@#pUYVm(r4T0>W>CCm?8D2mN]+q`o.<mBl:&\"EYF><Q*O!9/oQOUrFerCiDL(*c7k)]gk=lrM+;%Nb>[j[84q]d@H8E].9fon,,/s+AHXo*X1.sZ&F:cU>Zb16lETDAt\"j5@/;0IS2b#n5=(#_<!8Vm^:fRlF;ZNX(3_+*%p@0!IYR1hRW,JI;L&Qt2_Yg:TTi+u-T1l;S658H3J+#JoVd#P=I\"Wt>7O?W@H&]utFm=l1?f>AO_5&_aT,<V!<NC%Fdn#_0r3\'8Z=j!)D:,:_PO3L-h<#D:[gBXg+fda/KY>WCep&@OSa_%VunB9rU(PeiQX&?hMX,+aG[BDogJ\"?/^M7O[(;h/>L4_,j&[lUX/ub2i0sS7o\'bPYUIgc%K[Ji`Tl7]U:\\hnQ`*^&jkFtl)HiqMF)d\'HT/u!`7^)D,gDLKI_B@8ij\\-DbQBSo\"HJEh&0+Oq39ARPlmp7uGe4DlF,;ARXhRd*$=$`qF)e`KiqJiO428lu^(Jr2TIGKjH#`mS9=$!2&&;&f4NTrCP5\";;WB(ej11^o3VFFf8Y//Z<$p(D,WV\'Mk\"2Muojc1peX&,:sqpaqN[gbnCHl%\"]:63Q@cu\\JNnk/UFI*OOu\\,.I)-nf7)9:d=-TTO?Ik2F1Tk&6u92n.nCTS=%bX^\\Obe2Y-KSql#j*0L!]9WDhlTd\\^iS*)N(I+O^c\\meV`AVi!>j:F.)nS7[(GrK%EY81DHUPMm6HT^Vrifd\\NKZ=q^d\';13R>7APZ5>jF.);cKGYS?81gHoT9MSh[BS\\:Egq+mCIo&@(jg[\'E-;p<VNd4CuV3K>9m$odU28K[5m8%d?fi83<BN7mP7nt4$N$o\\^^\\Oo90r(DML8tl_93C3#J.OXR$j\\C\"o8f@-rF?SnUt4M&8q2VE.gl0uX\"0\'\\1K1IX.HlfK-.;l/;14)Md6BBVU^CF#qUo\",]XXuiO_umCabX+H/\"R*5ca!]CD&&%,H&??N9b+]2f!W@u24q@Mbm!RWnGS>ag?ftY`??ka(OQB]UIu].SkEk7U$9@T,W^+];%B@lEN*0`O1j<@b_@.J\"#[#;4E_n_m>\"kEDlq0MRc0(:-Ej\'*[]Ri/M)8I#:;,G)`8&c>Nk<3KK]M6\\/=j\\qj\'>f%fkubMC[N\')%D`X\'*cH.G@hEoTc,Oulg`pDFM-U$,*U;G@Nm*9VeHk2Y.rUsq0l]Y+o.*Q8c<g6(J0Q+p4MX-WVa--;Cj(\'#k4`s`\"_\\e@<FAa+>:9QT[Te^EkP\'q%glkTDSt`FI9H.F/=B-UPMNf`c_!D&C-1V`^N$nq\\3HJH1\\(*DVf)&S4?LG,A$Q>$FYQ[_!J^$t<Q7EqQir!hZe&/5W=%=8t,/EcVe5ICVVg(O`/9C_gHq\\U)Mbu3P/A]26(@`Q>9rE3.O;Pgs\\D^KLbl\\XPY7b(uk?:^r_hd0i\\cmSlOb_L;NhdOi$r-9@\")Rf1JF,NLR\'tu(hG66AF^U(_7JlZ[)UW/h#UtHm&^Ks?Mes7@8>>Fpl[u*g(Xc(EU?)g*nJO?<=jFpAg3U1ka5V]B(p3Hag`i-g^qAgQ@guh1U4\\,]/bjiW](N<hqoe[g\\]$FtpoTLHZ*Ebj$q6ua`,\'NZOr_^cZH7ANc9$)R3\'^a/dH*Rr62CR0DZF`&ch&%P8/,qUjh[_Kklm!FC/M=1s4TGU$LZgD<Gu@WRWqB\"H#BbIo%TW*H3Lb/lrg@$Q,5NE5+&p>8C?)*r24Nr=U;2BWdd*_<W;m3OZ\"=Zb21^bQDT;:<&rFgJKG*4hRePk[\\H`erFW$2EUk+kfj2K!1NIeIng&t0K.i88>*Qg/YInbj@\'gqW:>O3A3FGS=rN$4dDUb-.KO^c2iDKL/nu*F3ETCCM+`&%OGMBmUqLYAfoAr@PUV1;7!4nP#EnrI/PU>`eX(Pm\'T2cm,#S+af!(!gO<Z<\'E:LuutSa.!S`U!T5aoJVMg<b]=rbg\"t2Oi+S?6mfSfd9q.i@$V7Rn[LYC2tDg-9!q>$DVR]D,F8bLtue[dnKm(6B?\'si8AD1p499D8b@\\%g4\"+EkH!U%Z,<<h>4d,p?`%AP\\<>\\D@M.]Z31VB-#<#,ol5>/9AWQ$jpOW`PhBI@5L$RM$?2%6D;tSo?RibPO<16n9@B1A1MUQl%2k[DjP0..oY@O6.+ai:OCO>i%kjc8ZNWI7J>JqQ=F>P#\'\'*]B8V$<L6T9;ZJD?B3X\'miF!=uNlR1Qp<#8mN;&9!;,\"5(WsU.*#,K/_\\\'qp>A;7J*3K=bXQk2Bltj^l%7>)e$`Lq;C/FtqD%hP.&G-mFIsY2Cc5/V+K9(01U2G\\\'h/i%@VoC<\'=roGG)_OHH9$<1\'pIJ*P<AU/[sc\'0O\'>NqTD]9>dF0E-\'&@Sn:mUn_;Z5>$hRXVISJ\'r[+#s\"ei\\1j]=7U-_g762.4&_(3,nSj!.!fat\\B`%J3oI^Q,(DHHZ`Rf<E!r\"Z\\\'%!@\\M_>i>pe&joDi[feP$*5Tok?H)o.7Yg4X`5do+!t0_]IQHWYnIME\"%%EPb(?6r5L:(4!sYRX7K)a:g&HIW)cLXVCNUE&>3U_Z./@2Gsnt&a4N;FH<Vk%k<0um_Wgccbh;\'m)t4V0#tFq])iI0^HNjS@sRLA,1p86qLTfcbnqp^c\"8B/qqcr=%m\\(5O&rs+.7*YA9I#%K`B@5>U$8\'0WBZsR[E$t\"G_siaR9n(\'qoaYVs-J,XHc!6#Hoc2,&^?.e5NH^Zh9W!tcWIib$@+<_j\"=2ak-+L-ZjYg.*4:.[^>#MB<ln`)(ej8Z8b0^G9Uj^`c3Xm7^[,&F*ZPBJa^?m6[&1h%@^3GMqIf\'X;%$ZCU&.)qc[@cDD5)@WmSCOeMIIN3B5EpPFr&4TdI#BETYXco&WY]U##!a&knn-DD0Q<siPkJ;<TnJ7L:,1L]O\'U>s,.hgPN?:`QW<F>\\C1i6\"0\"mddOW[[3RoI]2*KG8GL$)+e:T(-+=nT\\Of^]*&Xfh#T=$K?kDcqFQ*;bl,R_L[e2\'pp=agT\")MXb-&#(JXiai(?$1(7P6>tS4_l8Yf49>j.3Y:PO];=&m*1YB%%.+U\\+Uk?>Q%XkHD^UkL]LH82\\hS>keL&o&4WSq`)0$_XcE9=dG?WOj7A\\QK!VJQT(s3;7*m*G&`Nu\'C\\gjgC;3g48jMZS&DPO0(XFf.O-8Rt=*]:t#l>55-Q97;(o6JQK4\'S3q1\"!aOIf)&5$\"NZ4Z>6[+;3(IW+80l\"I=&4m3Z/-m5sJVZ`%\\8W,P3k=&]WL@4*d7,9GYeB\']Baeht_(iVohlVDb(WcMm#lQSJaa!S8k18p5F?Pj+bJ6Yckl<puj&^71o_h9JDLq-YCVp8;jQ8*5AhukpdP=r5L_Bj..[9-*VmXaZY;aTtA(gAY[_goO<.m[`I\"DGo)\\Xgh%#[NU,ucE2oH;>.NF`,e>0e3b+5is$\\GfNAnYd/rU`HY5++IGKC(3PIGaCj2g?X#HNhdj/Bl+\\&$(#YkqQ&G-]*ZA^Pue^#u>NZR\'m;AsqrVTXBJ7f&0/;-_h-+DS,9nN_lZa3A.=%JuVoTdBfmHBT2&3NAeBAJmb!gAHX`B[]OBL+gsmf1B1W4>F*c\"0o:>SV@uPRB)I\'rQT/4#?m-rH6!Qg9[82Jg`gM]IfaGkZ;s3A&SZkq/9crpK\\=4[Z3%N&2P&uk_%sY-nb>.]k.jmnLSYVJ\'5jUq@)2VhS!NbHe=Gt$qRDB;@RDFggCP8*0UPDRd6?,DJ$Nh9f+9jQF($nlViOZpLLMMFY(%EP-F$F6J^le/tNBNrG`.u3.S\'a[bN6D9^PJ[%kKoJ=X40(L@oL[2loq?MEg#r.sQ8lTkO4*4inNTCW+=@PZn5B#A7$ODf_fd\"UHr.ODd/Nmoil@li-Ps+;H6a<`gFTH17#`]QC8UF62(Hc38u\'?H+,(_Wo6oF)kVL)bT_;BDD\\C\"sDg&Gl?i9!b=RE@BH&;:Qf31CtVaoI(Y!aM,UJLCA6)uMfIINf*T)Od]0\"N.c4eK$^InI\"O=f#Yl7A]7Ob0-OsoLLg;Jl\\Wu)En-:>Xg68ehpKNi<!$fk2Os2#9UTq4<59[c1FDKiK6M\'\"5e+Ujhl@sd<t?9bkIdK3+sTkoPV@/&I-:_!ZHV^:[^6!rqtf+PHKp5r5XR,#=g8T.6iV%-rGl&7KEfZ-q<@A5gN%J^&<A0DnA`.<MTuOEF(g`.KHL]jA#MJIt*Rll*[WI_\\(Xa\'%NB>%Y,K_[:X\\m\"uFUoQien<\\GEp,^9AJNaYs_\'/%dCTL`Hi6l\'lG-^B74,AWM2pU#EcWoYGdG+!0&8,\"Oj87&5S\'!2tc+;Y/fPca<b*MuISK^Qos^M[(&/\'-WKQL7o9_D?6DW9)&7Y47bIOrL2Q8%PkkC\"66\"u6f5S5&_\\E`H^6jH_5LP#*XCXD8mJ.TD2uK=jrU%D=\'a^lN\\.S).SZ2K\'BJ,Z$0P,Rn/]BOX.i9%<W<20`t!)Z$A7f+oQ3*%Vl+CRs45ISI$A))mqOcJr]2HK\'C\'g@]U&BVloa\\0oaGOhYibB9?#;/&)AD;crl3m>cF^gRGt-:e@`8A[rGhF8qK!/)Pb3Xh0RM;KY\\*[D\\pgQ-[/8t@3rmNqe/LZb>raA,fE$5;q@L5p&Jok<<&]mo=58,0Hd(Xn\"mO+f&]s^2ManhL1J=E[\\DNZY\"eMKT;\"nPL_sH>f*ZmVO,66l#`=_HMGhGK9?#l#c<,\\QK?<_,45G<X2Vtsq)oSZf[3.[mpfEO@GpUPos.F!Zn;Sh*.\\LC*FSjh!K$3;ci#\\_t9+(\\0j,_,EJ1G[1B@Ca?s#/GM\\D\'cQck9S`EX!L;g>\\ER=195`JJ-Obi-7O@JXP=(J&;#N2BXP#d!!+\"b&kMU_<O7BoOO%bC4;5lH9&DFHJ41Q=3n?3dCclbfi=\'@Y/X`00B++C*OR`.-K:>25DPH19_)lS=E.#C+ao)I*L(Ff^JbAjK;f(%I!fY+j)$KKEmt8NWQN*^SNm+lFRT!?2KG!0Jj>FhC+:k2*&n/\"[:a)&Y\\Hi8[=;\'u*?`J=GT9+At-:poU:(T5pOV6T&UH5P)KQRF5\'NPT]La2YXE^tZk5EB-^`M*V@#j.8cYV?N#0se1iXeAc;*L?_#6Y?!312DVYVpk[;GGUuDLuC+sRM25bRt`sI`<<]>/s0El.YnF8%R:$1:\"DRf4=KAn/4)bqE8\'cQ47>1`<%R5[YnNmr8ajlFW^CAZ[-<c3XpO&9+N!<KG*LGOWi39^48QBnRhE\\Z2!SrVWQ287=e72aa=toOCh[mm8*$&`jXDZVM7<3GLB:[N#ut.451r`0F;mlbZ&dpI*@mM.>]N.dD%/uJYu`n.b_.K,V0M\\N]-MQ^GjU^H%P].>#%Ql<=BGWT&V]j\"kj=J)s/iSt\"Q\\p4D.p,K0#Q45]I1#2:eWGpB^BUF[7):4%JV<e9A%nA\\W6TG4<S@d!$8iKal8gE$ne[M=e>O\"332oJ/0s>7R$`n9qsG@$\'$n.(\\:%7s$*`@\"oI!YMoO8*d)%d!2&o?f\'V%=1\"j*<#7mNC[:;3<J&8H_g20\"&j@MZ>IU[6@2d=?+ELpkZLm(#A9]gB2S\\8FWdKT:t0`ne^YtO1KWD$Qg<L\"#%SkAs_@C.UQFhb,_U0T0\'!W[0\"F9W6.j>.5e2j!(*I7h%:nU#t^C5+VKoW,j,Z[24@6i^Y9h[+V?e.k9G_Ddd)aGWWaCHUH2<MnAq]$Ip,-NELRDLD^SZE]U;3,AYsXU+NK(rB1n/J=2JRj$Udo`74S\"lm6Ak-hO=%CXg&f8]\\1J%L)?\'+!/$X_`dl3#H:!M](6^D6==[Gs7coBqi[1(-\\VEf\\OP7YgT?5[d\"HWeqjt0uCXI+k.M\'>`9\':Fn%L_bi?HmN&b7j<1Mfi6;H<LTI8IEj`,fseS0L?9<hQ/S?[nWnM_lkR,7&_[k-#%6D*K;NI`ohcS7SmHsll^=obVp+kV\'OM4a\\@>hMW12J],9S@HhS-Z1q0(X(Zm\\OuoC,_U*o(b:Rjs/:rXXmU3^`_@RHJg\\e(iGM.ej&%<-,*eF./Z1-uofTJne.E&mT2!+]Gb>3\\7/(ok:d:,dWW#8*4FNA<J!ERPXKlbMP;[HbaDSWP$V^YQjV2L_aVCCqZ-<elna4XCR?2;Fhs^4S$Q=M56,.`ZFBnPa\"gN.M9kA<NaT9%=G3:?!_W1S208@re,kn0?Yp+:Z.V<:1iQWNhds:3Ee=sT;oVYmB@V9`p5Cq^h/KaOpJK\"%D$=fHh^_0;d@\\7BY/:@aa`K_oPmM[Ls*PYSf&i@Y6<6^\'+>]cA$!!:)8ku`*D2Ah`H7JZSUdSsBWOoG3VG9\'`[1giM6^RW4L\'.0Ci[\\@Zf#]#/3*aVBl8=t\'1.!75\"a)Wf^&1QVV_Qhdp@l3I?gi2*N.,\")sUeI10bu+pt=J^SgcBXZCNk%6HUYn#-9qndn_\"0:-TaRNOoRQdT;Vp!K_\"N%@neG<np-1heAp?&4jgqP6,Ym]JD;a\'QMMM7rM;Be5#r_\"(N`G0uu\\<8:l`KCS-6hhmIJ@!&\\gu\"KEGmjjk>t,4o:<Y.u.WZ/:%0G<P@dR6>SPK:J$%U:O:Z]sXr_>7AkD@p?F;Oc3frCU+!H>`<1[l\\IKGrm,?\',.Ru$=\'pST4-.9--YL_38^0E@Vf<k3dIl\'>m&dG7I4DJr[m*l>$L?BqedC7_#:m8Fq.`TI\"O>cAI4Z#KArO*S.r8if-(ZBhgZ\'o;DkB3bST0BG=5^UI8OH8UgdRmfL?]_XY?S\\8+m#l$&`N!&J!0^fpt^UA*Ca/5N3?:e&Kl4IQP0gl8BXXT:It%oe\\R$46ii10G#rCV=M])l8;@@iIe[bR\\8nHrKPZjO4OU(r#GHf9@/]\"`>nVK8R#ph$NJOr)-lio5oeo\\H\\)Z:`_?;LukBNiW?E-,/GD5Y>LKknj\'\">GNW^LWV`89p\\5[4-7ABN@DVc@NiH7,Q7Pucp.+F)#\'A2Its$JC#_C=M[\"2V-T*D0Wgn>\\55H%ejYP4JG<kTId/\"G+rE^2liJ,h7&3FN-[rlE\\BcfOe-#;Z!\\Q&gko3Enh\">A3fBBlUFOm?m\"3E4n8o`s#dOC<K1i1<:jZ!sZs&9pgWcB$D)/\'*5S`Pi8m35e1W-Q$(:UOAr^RT\'As.H`YRD<kB$qY%k.Ocdp=:^L[2Xm3-<oE]R\'e@RGSkjc&.!g^QLrb.cZA0Y7O>A[8F3WqAhk1_5YrkYFBiqH4iibuQWF\"JBd_sjQcLsQr_2SCehqW8T66&rmQ?3Vmu;:]1JEPlooh\'Nd#rb@.)=\'1!MLmWc[]E92`K4F6nnOVF@%)eBk`Y=Jftt_+W,fn(?NOe?`N+5!igI+-A<a]&J;/k.nd>b&>!l]GU/1-/ES2l$9a.iX`+fNVIknrcY3jLHYJ&?:<f.u)\\psEI!stFnkt\\0>*[dt/M\">YY66J*.+s!/)6P@\'$p\"Z!]E,Aa<9nq]I9-U(\"uWDGaO0i4h+dg1cZ3G#UUFbMR\"49[PU3Uno\'/2CS^t1&]bF:apW[+4<M#AUh_o]<.qQaM\"l8:9T^o!#L$,1+dn4%[Xqcf=_F5+L9qj.UOi36hO_E`DU>+I$*aJrUVUF)8Mk/cd6?kn\"6sYYoE6C).B+YDJJ7(]<W>52[aI*<Jd:7[S/67ho:CJim7^BUd<gChk.tj9^?^i%;;+Dh)j\'5me2thKrej>?^YQfTXN09s%,nAt,oi5OYf-F$@Tpm=IXJJ@DNln<#CDstoOW,Vfd5]Gj=IrH\\cd8lFm[_hhV-^(_g@-Ga(Q%DXa+hCAQ1dF*9GJ3t98PPE%Ghsu=2<9HThb\'N\\6n*Z:4(`d*G:I>9UW99a3AYSrX>APO1T4MU^d$koXfJ(gP_\\3*^Uu\'![Fp-!)kRT\\MG(\'??J:@eB&5@+1_.^I]#\'KOijgDVq%20;jD05rXO^#_`q/U+LbP*dj3#[gga!#9!i(rDahXq8[mjKd.*FgNbFJ:f6&.$?Z:Hhflq1MC@t>)YfKh8+Rq\"CfK4heElU7=n8(i9npVKBF1E86jLb\'C(+SfQYb-gTDa?!GjVd5aQJK(V8tHUH_g%]Umd$a*5BT#,V3A#^F-[0Bbc<D`Ce>+WA4=BY*!Og:Xbm7<(47XA)MW6J43mHP9PC_[4jfe,=$a%[11<$,Lc1j;g<PC>Vl3:3<=RjpVuUbO_-Sq`k7JqhC8P\'>^[2)f,`u\\@riW3`XfpGccdcGeZ?>8.pFZt_j?.KjCgSJu-W/P.k);!ZBX[_tO)mh:OkpsU7B;Ml<Ah0JqL>KU?>DPhWq@-Ne\\3*En50\'O(6BO]3#H1sih+XF6&%Sbl\'kEY5^1D?AY4S%j[eRtd4s=.:6AZSIj\"anb*p`B9f/(^_EQ]Y\"b/610\\\'_)q!Z8BKu#-uX:e\'1[ej:\'*jK`@b:Bo7TA7NWDqD0THNa3];U].P-DHN<\"l,7u6/_!$%i]R?Xh6q?(i8&DZ.lTg,\\)d%g9tEH<h_MTeoa(Mq%J_\'lbW?l@Di_gr!:8NPWV=.r/raL))=6^\'L\\=:3FOm5b!I;Dn$S[\'N)_&dhXZjjB\"@fd][S`<F01Ff<0Z!`KbDf:r4ta]M\'IlSQcgN9/\\[%15E%!M_o<bHA(M-^\"X_]$!$0.=*#UK#K#0o0[,fheC*NNuq%U^@o!\\6DiQ6T@(@-@0(\\3<p\"2oEZ\"8REgfghUk#XJXj&g??F&Y9q>p`-)KK0feaPC;Ce8T:pOHC?R5#Y6KI\\:WZ\'W5E<fj&1?QZ`^uF=gt[CcUOE,oQUG[n7[skoIJqX0d9_6!e\'^RT[)N[6f6=:s/Y3=mf(&UnC0ALnh#>*ifV1c_RAMZ*^O83#dcH,g,@T8/`g:NiKb@G?,hcc$J#,1>.*V*PD4PORKr>tQeC\\VJ.\"RE7m]IA7Ra++-NNJRj#rs/ftO]8$c*Fe\"r,0CF\"V.R66hYqhKI2]A#069c%5+nK\'ND_MHG\"#oJFC,!502:Xe!j:T,8T#$S!f^,VsrYkQ$uT:!Mo4Yl,SaH\'uB)!L3l1\'O\'jCi;\\mf%-Xf?KPNDCWj*oRs,TS:0>OP,Jj0YX&GF)lqZR_5ll_X*Z4fK`%VRI8K*P^nMF>DV\"DD\"S\"a^\\cn_k.q\'s<)AhG//D188O\'2\\:JF5Rj=gBA:]Oi3GeeYTG\\pRdgZ\'_fp(jIm_)[6EcO\"/&_&\\h&[2HMoYS%>nPa%#$\'<L0N[RZa3#V\\7*h3O9%j=@1;dQAV[hqlpPZi]>a$WTO5Y\'k=I[Rsh86uDG\'fi14s6\"i\\N\\nFa,5]P0h;VOT^p[i/@i5:>\\:_G3,o3/]Q_n*GtPuCO5Qe]_ooeUR7a7=]buRX![#Ea9-@8$&jcW]`fEs6$d,(XSYs_\\OY/i.k,edE/!\'+H5:/=NIn\\\"!IGeCOVb?qe*6C50o;l5C\'XtMM)!r`EJ1Aptb(dDPneuS>q!t@C%th^gK*?*X%YIFg^nrF#lq,jZEX0M$$Q9*h4LA(8IMDOF0I[`:fbO%#2@9QF//0nQ=J#4u8bKLugsH6h$atH\'dT,YMSneM2YU!V#=Na/pRmbX\"C?sXjBC-I=MQPB_:_.6iD1-9GXfO1=UsutC>W[d,p7\'h+omK+]_b;p>9Hg(bmAWT-^FS%\\CmWlsr$^)#(fs+#@q%0ekm,\\HhV:\"\\-q?Re`N56$]^Fn?Q\\b:XUFLbMa!\\[#G>F9s*t<E/[k4_u#&I!,`N1=-r6nGZ%UCYVTlHEhYDjNq*kN,SWCXsN<OEQRVH6d-K\\HRH(\\`W,Z-?VB2H^Z;UfZH/qirSYr@BNQ#QROr#EW[PQSM0K2\'\'0:^40<Wm=nTp:^FsQd!AnI)qmi*L4`jU3(]@oN&-G\'f:L#1P`V)e$f5t;e#9pgJQ6f]&7)Y;bTd6%>XB@GM,mEJGbX9I4RNs^TZjBVIHtmTq.ku7(0qje12MOHS`MBW><a\"4bd\'d6P#!gFhm-$sfS#1]\"#!nlQ3oX(5nKBO,.=d6le@9SHe6_*6_?cXMD+%K7BF2>k?(%`H[[%EOTd(T\"\'F]OXZC\"O\\5C*4_fCmOGNX)#@p/Crnk7Bifr\\u0iQtBW`=?%Dcl3Dd3=oXif^%hKci`-AY<,p@[OTXZ2-+\\$PSILmi(-]F?$U=fg/mp\\GL.(3f:[.`AfJ#8KF2l/$&NdPs,]39Erh_9?skO[Jp[:%8?tDJ98eJ=KS)!9gg6a:m3@JM@blQP_4?W^Ye(!bGhP=55NnOu*EKobY88R4mN/\\b,)t<4Qk;0R%Djbj=*&]ll/uf\"GD\"nmEC:F6Zj9c^-7)\'9&3Rp2OsVu;a%5S=]h8Ns^):i>E;(.8\"piO2^uHm`o?h3/Z:NW%h+\"?c0_AYNdm.)?!EH_]U=.R-Um2^\\k#Vg7Y5gE;#Ir4.(/%hNGM`?_Y,4$.*oAbs7:^1[\'l/0:Ft7R[0e_:8^te/*#jSJf66Vc-F7+d#a\\#j@EXGKKHItkEZ3cI7`!)^mlN$l@FmAlU]k\\k\"jkH2e+Ua]l8S;bWY/4>h!N9$.;iUC4THJWf\'*ZB,<;$XK_^,]s+A1=\\=;d=T@>W\\=kV>G*@iVHu4fW<s=7HurHM_V\\HaaIH-N5nth]WCa-L+k\"j!8<KRJ+DKQ?]b?7BNC6HN-+b]2!/mT#QeEcLCh,EkHN@L#8^n(.Br/7:?ognOC8WU(Wa().9&DI?E)/h!&=6H+P.%\'o:if?4\'6ef`CN6gaUrnRb\"hR[sp9GH@X<E7T/\'A%LrfqXQ?c<G6ee6!(aq@7e$bWZ(_$TOgrn(QH8IS;\"G\"4^o;BRcS,W!G`khqq$$4oDoXF!>e7d94AO<34$PlBVf[3\":1[@R6;SJ0(`RZ:<SItni?Li-8n=^=0;-GjOY5oc/l*GE\"[9NHej\\l7>(^JWoMAI(D;R(?Y59Zm:NZkr1Bj-^eF*SEDUS#8_WV/f\'EKrEO!59=B>i6\"S%[L7ISp3Ok\"8GH>fiIXfYh>DUah\'D#<<)B?V;-Z>-G,!d915-GR)#qLYj\"i.X:\\IY1&au\\nXt.qYfLe9:W@:73R7nR^t2k9[54YL5`:)6UZ55((6je[PU!aaL9h)hDoC*]1Z?E=V@ioei\\Rt3Fl2)$cN6K,IF4:Z9*7$_%1MPcSqm!@6Yo(q&6#iVS24c68h^l\'%)h<(9-XK&7.bkd7h`agcl!CPufte*bJE1=impr&ZWnsWdL4!?mUR]R-`$O?Fc=)152$Z/\\]B*DLa\'@kuu`>-C/S^9/B6.B\"9m6(Xl>W=G37qZY#LC\'s%>lIBu_2\\5Gu4W0JF(:c6#^BoQ?K=@c<-.cbJ<Kr1hN9rRAdh*RWuSh(_P8bAhq>nVCj`CVKccT:0fQd##jZL!n^aNik9$h[\\E/,nnAL%X;KVd5dqcg/gLicI&\\Ru0@?H)f8:7.iMXg`mDEl;.tgE3Z^LdN*qiS7Fe9_nM-e<NJ/*G#hh4]pc`7_pL\'k*BR%Cm@!G5\'Ff2Q-(1D4ZY\'E_\\_6eI4d9RWJh478VDd3)Wt+&16!s6E\'\'$Y$Xkh?N5D0QTPnG^p5<s\'srqbe9cSq;r&?#2hQ>=)3]<EAVs0qisY(%DH4;C/?a\\!a:+im+:j%e9\"\\`mVRT>b2/Q[sR!rt)!m^_jn@3^D!hrA<Y#im(:kdr&:ps(*8hrtIL_k.\\nXLr60$bAnEhgaY<uabqW?GH4qRAF_1rqIJo5/bo:].]IOSg].DdK^A5>A@:J*:o08L,+MS#&m#spH#0e25N^bd->@m8phWcG!Aq3t_Y.AlgaJ@]T7Zdr0Rr;J1dQh14L!RQ$D]?O878dA#&$G1hf`MQ+$2)to5F\\\'n4%P[Qhnc#9**U2\"GEu]Wn.`CN`PB./Ea1cC9+bGRei\",9])L>(<[hUAOs,**a3]q@R:>qGitKVm8haeM:=QN/A!u5BkD;/l]Cu5UWq.*gEYl],?Kce0\\J(@2JCoO^-`KS8KK\\1!)r\';D1Ct\'!/ZbrO=./8d=%Np&:o!qiQPbeVohZ=2\'IF(F9k#KRL0e.i4Q)R4K\\)P,6me6NAWnn^cRm^!08:eR>26793\'Vq^gfLYCn*s\"?lfbDpPV\"!/op98^<10;7DlnnOH\\8/X-lj5f(tpQCG2lq>Ge3NU[-&Q\\%9)Q!/W@47Kb&#DSGJl$oo3sTc64rN0&F#J5sc-;tn,HX&lE6U)&#92HLjH_Tc]%Oc^t%p%N/u^`YD,gY,^E9G/jjU0;]VjQ8bVG1D:DlHd$^>)(b@@1CctVl:?Z\"S\'kDej:,d:_\\3b1BWRgAqD>@j<.M0HRT%sM9lL*H-1!373(i\\f6jK8&$Gd=a).HAEnOB:hG(#L*gKUQ(3J5\"l4FWA_=gX.j1Zgq(^KP0bEQ7>?%THJVsB5tDiih/bqN9tN1\\N?)jqlq+em+`2P`]F7b&;QbaGB(jH`;6=8+\\RKL_;fEMs/lF\\hB:5YGKFgrD#\'rsfD8Q7]D/lWpKT_,su(_)pu:G-.hgG^I3n<`Rd0$+4,Pm6rQt)5#01i`&^X!1%[SgH>9X=!UKhE6cA%Xp\'2nF7gCA=Rar`k?91oBMB@h\"HViN0e/fTer^/GD\"U3[\'i0Tc\'din?#PcM^&I:\\PLO)uE%O\"fMh6:n_8r]=?Am84;\'9T=nFo37!ZW@abUGd!MY=b5%8BANNODWoNU@nct2,A@2on`d7E2\'gB5.43MGL(Y@.LRJ#U@WY;rIPET:ufI<4haI9Jr\'1+741&[Q,!T=)eA-(>Bb\",,#\'\'=BiN,1cg*oiD%^?](ICrHbuK<oT6DT9&ou$q5c=d:$faUGN;GUoeIr#+jK_=S\\&ts;[sJQem%ZJ;X&\\=&*h^@\"Kuo\"GG6k0o#2TDFF2Ko^IWZpj[:c,D>(P?2qY7MG\\sPE$pDam0@*gqLMAj+f>C\\qO>IkP4nLr/Z3ZCB@UC!7GF`YOh\"mmnmeII6*\"`7U!7LD\\]UY7Xnci)sg07*F0^&Ph8YE;,obMYn,E)=)PjW_60q?9L=N>sY04Ke;*V!%sMWL$?0H9\\&P=6>iPj[dDDVH\'B(0.2&KR+0\\;Mj5c5btsn/>nNW]a8]o6?64kK73+Lq@<!)%fU2cS:Yr\\B#Pq6gm\\92`(=f8GIh%/N^rF@U^m\',OWpg#V*sWW:EX1Ft<W_8f_:2BK,-VM\\=8G?^Qu4^hO!b)5I=)Jc4!!4uoFA1A_X@fn5t=)S,c\'NkQ2qW%YS:6j6*Rs1&?\'#m][l\'^ZDReVlUjtLIl`b&2F<b<(>Z!ljQe<GTI-\'S+nVAb6lU4I<nW,NQ:uD*/#mTOBmdlSSD/j-i+``cV/A5>=uJQS\"cYk#A[Y(7W-g#[C*P]9>^D<QG6EV0\"@IEEXHZ3^UqJXc&mT;7BE\"RS+%3AFm^`\\31[/Bjmj<5+QNf$lEDH\'0l2#^emPg+??\"b$PLg?8^8,3,R!+7]hk3j&QJp*bf;!:n.0Z:jr?A9%++u7aKrb;-7Ea-4!dnLPGotPYV]ag:8\'[t*EHh\'&4fY]JoHS`iI<S3(T5g,\'Jh$3f)\"8Ci.q8);r$DepA$%1an^M+MX`u8Y`Hs&>dKf1FM(E.B$Hs<A:[\'ZQHZD1iSpgs_R\\k/.iWYncV7+4\'Ca]1@kaqVOMU;KZicE$Jca.[&mfa[7X]jjtY1r23e3[@?t0#K*%];ePP.>*6?XO$d</DJ[^?;CKr\"e6sc)OO@>-eamgp2[`ZhIZ7i+Q#1ro>2+.a3ZoV;!P_W6bEG5!UAh\'aQ)SK%6\\4,=DkbKZ\'.0:?,`o_J9IHdL&6XnARGWsQ@2bAKS<7U.f)@a6qne+n<jJL*:EV7_6qd:#E^(=\':\"\"+EtXXNLGT;RLHN+R&qEL^m[U\'?XJoG\"rnY\']rD:3r8sZZ<\'%[-KG%M87F(te_(tc&j:i`W:mo!SU),_kA\\WRChN\\9F6!FZ@_b37EQ!JZ?-X$@pud``Ztl6T)B(_I?*I)Q,<P];Q260=ks+dK;nMDm1*6AW_q7=h\\.S2J(664mb/hgK)H\"\\2Bal(f.e^qaM;&7&0oU#V:O;]lbT3*OrS:mGq?=pEtA$3@7[Q^P94\'e)aEA3_C)(D2THr&Fr!8=EDn!/Q<S5t!ujM8@?O6q_\"u\'EK>\"@3Sq+He[!#A6p.\\B5rao;ss`1n)qtFXVRWB@4qQG,_bUB`l<#MFlgGn55hMm\"=i3Y+C\\0Z\\?X;g19XUc6[7fbqRJfW9p&P=7C5SF+;QUn\"Q\"E%g1UadbeKf63+u*i=h/AY#9rC^?@E_irM28\'!Q164&4!V`HFr]ale?e8h9b[^2I-$n`f`QSQb9gLSd[uMrX:$@=OR^Q`VI:sd1phPJ$EVV^9E(l+[JZ[i0t[fZ$kd2;IFSTk%jl0%)_\'!Ig/=tCPpe&W7okHl4p$M^R+ml[a8Ul2n&kU1OCn2VRgM:0l!k;0\\_;iR^Lro<jJ!7n&QDJ`mroegO!j09A[:+4S>*6%=di+LZ\"^Ni;4U\'SSpO`oTLZ-O$C>TRH/Dmn9j&5Y`ZWZCNsgfnT^sUWfg05+\'pXLf/A_8W7UoiZ\'fhDN>IAAi3-OPb)AOi*mHXdr%EKB?#@:Y0u2\':i=-DhWW$nFHOFHL8>\'^(lVeWBlu\'I;?;`?2%\"%S,?JCMNps$u?]r>n[NfQmhd7YfIN]6IX^At^mldlj6)s(q$TtMiJhRq[&P<oci-+JE.I6_QCBW&Tc0ng[!&q\\.uro\'3QY_T.6JHmE4p`bT6A&K!/b&gWLIJ43]X/\"]<FL+\'b4A8(82t6Mp=,d,h&td-7i@C,>e(&+\\^o*Xi1hfI4kQ>+e8>^lKjgZL(1$[\"3SJ%[uV_PRHRq`#]:-ZcHrY,P<q!YE&2gu2lIFflX!aYo6@\"-Re5p[36+=rqKKR;%WC,b\"NQLZkn7_uD;/3?Am2tVa9R>E;+\\Q$X_Bu&!E>%6H`!(7t3\\Gl?`]4Hn49q1Xo.kci\"f&ir4/XeiNK^G@Gf3NH#V)\"m)\'ZcOa<Jn*q`Am7PUCN.o-<Q?p!W9HKji!9=9?)!S;rXN\",U5GU63)akCf3\\iN9rdfMMXLWl4$MZn#t<1<0[E1?H_S/]_I7)V\"MJeL9.et/^q,j+fEN?Gd8oeMm<Tn<E$DVQ\"?a?+0oQTV;/.io45P02NM_D:\"Vn@^/h)JMffQj&?$/jh0cLVo;\'$29q-%*T7[Jr;WTN>fs\'tOkcuJJ0uE=H[oX;\\h2;WG.9i8PJaQEj.NLlE%JgrRl%qgfg^rZi06qe\'Ko66XIFcK:52tn#]C\">EY=;A#@F7/AB6Bk:;q0&QIo^9nDK-`J\\Gknja.*RaA$dK=570`E,MJq<4Dqe>;NLM33T\"U6TA$&E>k.b]!;\\D:^TAM;+e&*,Kn6qP`B!LqV8tXiK`0\':(Vq=sE_VRKCau<t@hLDF^tW^+A>t\"I`5U8+>>nC\"O9c)>?B6TubAKtZka0g(9VMK9Y07_gX0Z*bX!U5^?fUaZRc?]]/5#\'VK_ZrD^Z(E,M3jPp;Fm0[,<7%L,oP4V/h\'_15b&(<kT*m-@Wb8(Td@Zj\\dk%&\"!@EVU*iHJ2Lpj,2Z(aJ\\*Y)?,a$+YZ\"orJe)j*A0JGEl[FBBXgAF_8K$sZK,(@H_Obh:\\n(=4(W,s33^#H1aQ(goS4fIgbDQDPMTRG=n8T3,P\\dj[j==kE99]O\\*Ze&;TVK:A\"\"`ln\'hK$Eodr6EK4+Vkhk\"5JJ\'R&X/QfY&FQ1#_%Q2uQg9AbpZ*.ss*=hAOm4gqAEAm_\\5R`8[uQe\',p&$@H=3ih9h5_J^@?$Z,>eVg\\6pk>\\?Y3`GN-K=#n#GEh4Rlh$/#gBSlDg\"\'fb;\\N0M-/^O@Qj9;>gZCq8<d`eNNkSd/NBkX=\\(ir9T.=6(`nW0i\",!n*%F!)e4,\\^<+o\'=IcqZL3!Y?(b@RG)kX-Cf9B=g(bYN;3[MXSc$QQ$]*=AL_dHTldO*J[fDAjJkK%4NQ\\Dp#NVDj%`2u\"^(f\\,lngMj;$_t1C<`mo<A:o_^D^+W*%(Fu?j&M6T%\"U9bX994T1ZpVp%,EAJe<3YU?<0s\\M@g@0T9ZPCKPS=U0<iZT>&]k*_]15f-V%CJ5VPjBgilO5eo=F!<#D.D5)J)H+^58_/-Cn4_%M4H45YBZ;k\"s^9@A641_C5hCeYFf@J4O<Y$ipT]fL1_M5c91a(NaV367E1Xd9o?=FUL?PC%s;3CL0d;M/a)Ro3?\\M4pM(t*a\"Fk*2kEReGge*JX10A2f&3bm^XYnr\'%0#b6Yc]L*M8V%(C8&84GOTRkNTMe@$Q,OcVqUCr[0jTQ4o-FP&OA_mpN`@G!;Icp__eS<0P4dK0Y/\'Nh^qZP5]aLE!f5=a0Ol-F0L[)_(72`UMqKg]>L<-*uWbd/%@8=3WH],6<TAng1a3f;%uB06pZLC8YlN<:7sN6@F4t)g)`hNc9b8%C@bc43r-hkV>>.\'DRGA!CsPu_76sp^$cCl!2986+^aA,3RSK3C<LkM7\"Jpj%S1J_1mW^XU\'+c,T;ta\"#/\\C/2O<=!XCl,C]G7X]^HWq2V<AM=6a%gVP%fSi3c9\'QZad__;de.\\Gu)nP,OO#X<?j@7F_.N$e%/>XS](BoR\'$]LDQlpt7!u/$nkCgY!6e:Q2Cjf?>V.PX<\']0<U8R\\.o;j?Sa\'A4t)K/S+:lI7Yqd0eNd>h\'ij`f<.\\8c.]61r?Zm2X^>3)mF;r\"i#YIP!TbN<j&\"a8VNJ`U=iaVnE!km&SNPf=GADB6kJBem\'Hs)p`JoZUe*2q%T1GpT>NQab\\@]*c[EA%NjSlF[[\".-n*M/4i:Wi;-1<r>#5Z27TFVW?WAAsgM3Jq_EnKj[]We4\\uIPZcCLSrOQR+M*YhS.4>T]b%ds]MTnKr!q/H@Y^T\"M]^/!#j0upYf3eJXDKf0uU[QMGjJJot5=bJ%>lG`f.JB\"qj&Md8#QeSm/\"8\'!A&s[c./]W.,hKmDKbLV;F+,\'4.G^gCfKL2HdFe5nFGDo\\\\!OZ-)iL&n;eqAt(%dC;6Tnd45BDmEb=8$[gqlBZSGE@U&&QY@&\\]u@8LJ<o];99`FQuPUpUo>dPLHGL>j^:\"_;LLe<UI,e`4l#T#Ta@GB\"<h&9\"5NAYVtuPZC1;t1?D+GS8eFEg\'Oetgr<NS7Z\'`i\\`T<2JG#8tR\\fTVB;hOVQBa[>>aSkUr>(s((GV7Y@F=e;M\"Z`,F\'DR<*=_,urOI^W<;ROuWeSKF3k=(Ch=s3:1/\\/[)d^I@5Lkh#McW!QYIFubT3i?WmM9_G92/(0%Nh^acVGS(0bZQh,[U;afQ1a%1B]u<,#A_q-gm(f+MDH;m6S\"@j[,gg_$m=\\PB[0oYiB&?b`&\\W>7l*\"Nd]nm7EmP@e2SV!F*\"iVk=1guhq=NV%l*9kjM/?H5)d+N>_]Sn\'I0IH[jIF3IeGuY.hte612>:C&UGpdA_s-1N4^hD[I3@ha8p$N\'==0>>\\_,9\'^P2!NW&o<m)=$Q7@7=^?CP&O:DAc:HS<S>d:5\"j*,Z:[Wa]\'`=,\"ST[;:S5d90qZ[Qi]@5=O\"FEXIYJbc@WmOZhl+F-nWQ#3#u$mEme1Nio.`4L%G0Xc!E5Db(Nn2BD%:]i(\":Eg9GL9\\DBG!St,1+H`^VN`\'\'\"sffR7O=8016hZfZ^;%7H[>J\":YPKtB4#?(-Y+$322;?\"Nf(lJ(O4jQ?-=Yr;7\'\"i*L&!!5g(Z>::`!Vu+b(A_T9,ZlC(Te!i\"Zf%t)=*VJ\"=3%4\"QRMTG+_sO,mp`@kQR(=e0f]jBQHQ-2sLBmrnF+SLG/#N\\T)T[i9G?4-k9n%oj4CpX7.+?TZaV3O.BS$GeU7#,l&uP/JZI2V@CRcqOT\"1NK/L$r7S97-l388cgAEIpk2$BajuM,.^]LbWZ()L)0n8>Wr5!*\'OeBk6JLe:8;@eX]#pjA.u9-b\'TN)ZJ0\'\"LN!q@l\\C$Cf=M@j7XKQ\'.kl_[]%V?jkO=;A.6c](8q+t*NJ\'b%oMmj];JG65XpQ#!V@pN]\'eg/;*nE..2$4H!f%WRqg\'Yj0,[o8g2HPj=NZ/hLRJtfh!O>OUA\'IqkQD[oVr@D?]$cTtf>Re;[iMK77q`2WY0kcUHfJ+<E`$cak>fXLInXs7DZ?KJ`@q<GKXldBis\'qKH+[:iq![`nnE%0A*&(<*`cFs7WpE:<h?.LHjQlW?]*7\")o!gOsB[r#;p^6F\"/q&UVm2Koj^4[1#!7eEI3<@=_p_Gq24+cBXg2-,2Y$r:2r)GcB8.V).A+4nhXLg2=7**]kI%lCOSQ+/CqidL:!g$.gITB`G2Y*d&;DM\"/]>()=ug4\"j$]1J#Scnl!4A:rjUtEu0H3*!K\\ocP?l-#/>c:8B$DB+D_i\"/.`q#V33DS,_@*lka\\pLZS!Yq\\WR7eQT\'hRk[X4<7T?7$m?5/]&ho?+9Kj7aV4a2`L278<<rj/8$:?Z\\Y,Xod\'Fg\'8DN`pOYuN.nBetVRInfuP#c:C15]C@%;T!no1\'dkF:eOp64J+hD<S.\\AGJO4.YXTt.9+$\"L48Aj\"eAk9iD:fOa&!i\\UquC_s&U.5\"(^%#>nV(IiS[i:Kf]<\"k$S#sBJE>`j5T]n.A7M2p3+MMUI3a&\"aLC;G$HF;Jq0p2i#LaF@O>W23&\"XhfTJg?0gtbNiMjVE^G4@(N30IY//\\M$Tk/Pp5Y>eLN`kJ$Y,bf(eK=*q,B\"Mo_fJ]Pm+5(JRP_L+uF/>iFVmf=;9]IK.D^k7DT\'_5;E>&-=jr:roSmH\"VnG*A\'n3$aKoI]k9VLW%V_-nc757KQ/1#X[F#R?U\",>iG+58$Q9Bf]&I8R$(@n)EoMqu0;Tp)KK,2FBb(*PH>$]@\"shkNCI&?E9r%@`Yn8QG7WN?[o)0>VtG$s7l@mr>bYH75\'4+s,-3T95nMnD=@BBFA\\BE\\\'?]+Q,m>j*P=7[n]*_@\\f7Q(SIPBZoBq\\M7JbUVr>8>HX=?(FrhXF\\LC`g=_jb\'&I7TrHP7>6M:ChnK2hZ%L9t]/$&N5Emk)UQ]$#t6M?BL2\\X=%mBs2ud=31jLrJ)6_&qXcWbSrUFLqGBeHCI%/Bj^85SIo.uWOfj]4\"k#NBn`\'$3a7MoD<F\">:/flMQ#7!J!/&Ie7Q<J@p-\"/SbU;6OSa(P[9K*\'?<-WTF6GlJfU0Df[;$K$-tGkD6\\3`\'?L+`pYB=-U%eQ!Yln6SV;.,j5BZY<^),SuM)@p#ukK?IiZ\"iSXeS]m%X(p?ou$qs>m2(qseCI@ET88>2<I&t1J!g&u-tS.d9LoC,453.K`;`85uN5D+^,M#V7VqR^c<VUIu2@qsF(*EK,#\"TZ!H-9)Gj&7Su8a)A:r]_09NHVh/p*k3D_Fo^/@:Z[SZR9u:nnjjhE*BB4neZ,V8SJg1f.5\"XE+O?k4W\"`!ep3&>LZ%1#p;i,`)aS4Kk+79X65Uhm&1!=R\'/]k?:`QU:Ukf35nn,cqe.1TY%==)F`Wfg9n4V<TA^`ea#B#\"ZJae*i.<IF@BAhTQ(i6`7!MHM6ti<G`S=]h!P,J^g!JENQ^6,9*8kiTSshor)nrboXme(tT,l1LFT)WGtrP-J(&/!M=neNEGCd@s&\"kA*l4p>RA]\\8emFdjf\\S8Z#c!kF^L@9]]ij(eJ_S#e)i`F]8M\')%U\"A[nAkpSCSPFq&n:fS!Jb)%\\X5r!4\"\"n,WVq5n-hJ0r(\\];^uP_D2*i\"YD;\\&L\\d-\\-6jTR:\'^mIpKp:NPSOb7DdchQS^@$:FpitM5CF#\']s\';Kg=@R*)Kp^_\"[q(.C@d;@$s&[!i,Sf#`Lg%b*ha9]B.#$Em\'<<J3`aTt*.E*.R[*I)ZFR=pYO8B0ABuK\'UIM_+1Y^6e+dM=+I]Cb.l=:dc?=Crh:NWdtp*@ddV<+QeB<)1alC/\'))H$MOHG\\TIGWNL(1[Qo=_Q*b4S1OcrZm?ii+_6+ENZJOKe@DFR0kpe/<\'nhXURA=PYO/\\]t+W_H14*:f4??-^\\DUP]h12sJ8(;k2H*L_4the0Ah!!7DPPR;4Fe$2r]a\\hkZKb>YG]7CssYL6F:&7uUak?1:o>K+N7#?hLS:P0P/Hlh\'H;,3Cn`)I%A-\"bt!29bD@^m9_&@@(YPd5E^;>Vift[E^(QGaK>1jf>uNVbVD$itfUETuir6m3Q!G1aI_M.:BVe.cd#DPR;K!;];fU.G)E\'51^[G_VB?<0SMM.$=!YPg=8!b;*kGLiL0lfZ\'t!c2jtVa>s++K>:7WeM323+<\'[Ehjl%I&+:OI5Br%htNmB3@31Z0W\'HXbF2tLOhgFAl,SUu0)@(8Yem+X9$,\'\'!G@9OSK^lj`H&\'jp!1g$`jYa!#qWL5r0<#f&]I\\:ag!9FH#mFH[6#,oa9(qN.F\\%$B.C1NW70snq0NpN>Tm<#SpeZP::NH\'coon]99-c$A\"T]D$t6]%smmeQIL$0Ycb3f/;N#nUG.(Lqq9]$qk#[^&uiIa/&7/e<e%[pF2<DJ\'Mnjg%o%\"Et)SU+;Z5.ppcdKF6qp#`N4(A>P\'\\LS`nl5.-IZGed14_QGGeBbj8K\\9r[ajpd`A/eI>P$Bl%[&C_p9mKH^:ObWS7-#b_#bnQiL*VAQ!o#@I0JLhA,8l-UJ[nS0X3__q]PhTN)DjT3Yh<:8;]_Pfs:A<WYUpH!Ka:YC?\'\"IZO64n\"?V/0Fi)t\'c>Rk#g>Tc1BQP):k:cC73\\F=[G`r7#E4=BTeWiNIqf_:F)d\\?tEZ\"dTX_q0DmlT,D[&+=ScJ734`T/-*EH)u@Kb\'0W#)ZF&3;W\'n:8(=U&jCJ&nncQsq,1@..JHr17JjR*ln$#RLZkb4)]R5]peGlolZM5DoH2\"XmV;4u!f^dUd^I;fhQSk`Q8Z)\"V:^>e=<6]RWDi7!#hOV=^B;rpU/l\"ZLR!IR_gY=7G/31o!DWVVE^*(hiE`%jXee%:54S=no2\\u8O*MCK@k=.8JIUF:qY!\\\'#V&L_^b*SDtsak8:e>.,7G\'hhb@E\\40\'1p//t%%cfS?!SE(g].Map8hVH\\YW:YAkkLp]Sr^;EqZTrfla&YImh9\\doL\'k*gdf*)EQDNJTD-.%*7LqO_KJ\\a>qLOW,p.iJe(#Z$]^AYL\\alC#$6D#=BBVk/[P7L@clr-C>hpD$E@&K`Z9BL$1HBrD)f3;;T\\DBC(ag\'29]&5ru7o1(dd+8%B;9s_B/(*$:\'ZtR!Kr@kV@.rLmpenFAAr[9tiBI@(BVj8A!47gi0]cTAT+D3+N0)Yuu9\"g3^;/Kt$%.m@]<^&IY4&0!ou=k%HjTGb38lNsq%%8/[X:_GlH=85)B\\%)<]_S?\'i/,m^LngU)3^CJO5$Ie7t<oAWlXSJbEH(NKOD6T>_A=.9Cr6`u#7=E=D.E(Ghl2OEO\"d*f34!-a0^PfO#-TKgu<X5%LY#obQXfDKODU+/8EVhq/*.%XMQ]\\o$h-kqG7,*0(X=bNEe9&7?DeAABos-=Q74C(b`p1ID)Vq6MuLiTODU8*97+M*BN,9\\/?<5H0<G^LibnTo?_%ldr:jpb(^o1*r&jI?c,?\'a;Q/kJSPS0^\'n\'[CnbB\'<D3Y*FkV$Zr7EPgVR@.d5;j2<2)J#TOefR>.^.48uP((rk=5@Ro^rGbIAAl).]ZD`g<lLm?t;MM:g[Zq)WM^\'<Q<H$,r,s-LnJ60:u\"cK-\\0<:ZdqPnQa^l`AOhDdZZu.c=0<B>YqWf$<B8bsI4;bT-b-d>keLATcO3%:Qcj8kB>GhU+tud]\"\'2\\m\'K^b0uVc\\qctkPo?Q.dg>bgdqP;ghGj;PcltDYj:l5deFkQQ%M\\h[V:1\"jq0+>N=*G/NS_5<)VVA1RaoW3-L9=MV&<6QO\",W2t^a?C,Fh:.dm?Y7f2.Ab2-[!Ep=7;U6?nZ7qPuPj#eZ@P1F)=:(UiL*Jafi$Pa=h^5Xj3qTIj$(l`jpVlb[XP8O#Ctl]$Ot6\\T^EP=l)::C4(5o]BhG(>B+i[c<1BiXsQt:nDZI-X&>+u![0ZeiHnaNo:6_g142@,^K4kX64]*:f3X!^j6\'k-roesWr*)Ce:3Q:e<GA-r@1&Xga4%or\\kOX)S%$M.N>j/B&?cYX[GH$8fRmo*JqS>&@bJ0^Pm,3JL>T7!flaGX]89mAQ!;DkD.^IML&kG(\'\";C5YAN0KN3^T;f.nbHh2RZ!<SjdaW`BdL)Da\"(mV0B`_l\"[<p$&tr;>2W`lZm@\'<UAYsDg_<KY&XuuiHJsF2ZpC2+$_m5d*i5W&tEn*23j`\\=\"QM^KjS+cHEk!H>R.j\\?1nTk!f%gf0E&s+HSd5Zai`PQong\'+%MK<)$e\\qTO^F=FD<BZ;YdFbI[gMY^jF\"0\"QTFoteTI/YJChrZs+Kk<]&j$q)7mAFlcq8FH\'SS&@l8e>l>ZC\'X7sfL>oNT/:frGrnLR+<#dg*rT4X_!pY3!0J\"19:4?j/TgXOesDc)0WCi&Ft?GeFDTH4_7!P3JQ,#RDu\'S2t(J6]XfnF+ds.aG3&n0=0qgSkI+he]n!#*s>AHr2rXW\"sK,ED&\\BQUR=#leGIM/o-?-:r=HQa=Jr731nm5Q!Pt(Qu8/VBshWT9jL1[4U5=65_XpLp38XU/GU#:a)Tu<1Y;HDO3:k8M<=VL&.NMiRZ-ZpK)t8mqTlOUfGG[1j!YZ4Dg1?/kEi+.VWGgn`hnK_3H=E,Tm_92marL3MOeonR(g+P3pnqZs,R[tHCr1HQqOi%;GY/kRHJ:SB><P\'64Q32iF2b1I!/VI;]T2ILpKuV>I7eZ06RD5G2!<c_H/)EiU=&qdIHl:GFKOR[]%q%m]d4R(%%]KQaUDUX\"rR9>fPmdb?6_)#<&5\\B;K!dEQ3MpIn97e;;.C)[GdCiS0dnBOhC@fXCa;\'06<Qh8!6+`2I-*)%Yg9+r6Zo%@q&q[P.;E[ddU6A=B\\;.DQ$Hr%A,So^b;q@L8G[?`pkPjoR._>%bJ5\\-O\"BE4M&.XP9tp?[LI2&%l0G,[SjOeRB!.-%j>Q@;duBg+I?^cf-HA6f5h\'dI+C3@Aat.*@:,:\"1)3Zc?G6rsjY^o?aL7rm+sCY(8=DS`g>#75\\jj+#N9k7nD/*p%Y-\"iX0.r2R,N4/+gR`\"S.jYGm`\\nGKPONP0<`8I]?TN,rQpNTt817#(+U7kt=V;blPFU(X<*)X:!19Qj_F?m\"+W/nUN[#8ND<dS@CI=FLl6J5b2kBd[O?]7`m!u8:E&1MAZ4^<`RM?\"]b:$E:eY^aK_m!Xmi9]-FP<(dhPMQZP=\"d]>Hgrl9^k1fL-h[6,G>>4TdG*.$dTBj4U7s3-lZQiOL]i-<1o1hjj\"\'q?:0Tj;BM)2X4FsrOOC%h=pTWG\"RjnA:,/n\"#g@[@bl):\\d8Y>eHh)[6oYZFQgWi3S?<lBO\\PnhqKTkUR)UHmQag8d.(OAdfV+]Cd4hAW:o,pS8nl,$(OMQc;\\B%O?P\\:i$bWMXK(e5=d;mR:<0ORF9Pq/Fm-),k;.l]8).QeSobA\'jPKld[3%h:Z`B#Mp!879,[Bb01dOAI;.&p<H#f4WU$=n4u8XrHn]3>l9JTc!uEll$pRX[_jB5[.k=.N8E-/S)8kY2Fi\'MY`5sH1(kO[hWhUZUHInOW?.D@lYCR:456fn/oG^i\\TXHY8^6(anU@f\\AMf\\Kl\\kb\'FIHaAcgpoIfT_<=-(\\>:W0tdPF\\E@Qfk2XSXqBc1IOSj]M\\D+V\"m]J?E@=R\\SI<GLS&a0!o@E;\'9cEV[?1g9e5.uqiCb/)=Qm@sLBtbUq;mhbPC-m8nQ\"`G@0XP>\\3Qo`gr\'8q*@PLMIQV06LR;9)_m?fS>^BGrBqoF9IV5=,Up3RXr84Q.?<a\\ji7uoXhRBK\'e?H/Y6:@J+8jhFV6NBruBHOorEl/R*Eg2a=U(AE%@fXPAWRT1l(/:MMFqLW:I\\[2/[p.cPcX_hm>)Vs(G?Ufc[\\U5J\\Z7,#6\\o7BT]KaKbn6P@lpF7*/QC*bB8$\'`BT<!ZOBbiQO.bn[G/RMj\"EEt3Xb\"&+H\'aB>#8iZ6B=Y_QoaiJi>iF.6kO+TR=*NN0OmfqXqG#d,gEBDHl=BgAW;hE>T!L>9jDaiV%LNpm:;C*.1%bNI+nuR0G4+Wm-S8*2#.@u_#pHYM+O2\"C/-tf:A(;=.G\\n_p4\"4sfJ-/*f6`1SNS.IJa0jsk%QQ/krHM..r?T_DN8EH3C=UOGAR[V7,/IL#\\=%;t)Lo-<O8LBU^ZbHA$Qd^<P^rrG(uef?QPU/b2=!l8![5XAsoLnU:`juB,Uh,?O$KkP\\E`J-hm`/.#/%0?^DZSAfrZ3t39Ac;cFH`urP.A\\o`9KbIn.6>[$g8-\"]jKm`WIu_BK]:bUA*hOa>g*40hTjdQ39jE/SA$NN#`(@]BIVEeFVV)OUAE,Cs0!je%dT8Y@<W\'\\b=Aen2dT\"_&Krl%hcG_:>2/8IUO?uj30i/T^FTpibMhJkW4ob-$eQ)$d\'Xi6qmbUr:?a/!3I)(/l3KHOiKlpA(e#7$Ge3kV/JS10KP@*Md%UptSBn2s6qBr\"TE<6Z9UW?fQ*-\"hEHFXBr\\L$A:npCfQ&4]PRWQ6/\"<6,d%EACF(e\'`rY:8ad2\'Ym@sk%ASAVl<=YI[/;6J0/3bRq!&)aGQDocZm9n;c]l_TIh7i<A-\'S`][ue/E&X7i*oukRi^J6P;^7@+pG]PM@pBV_)s%S=ldFa_1s<uGR\"pdm+9dhBY\'=30/q<:&VWdhEa,:Yg_N>SC/:\'<,:q0KJ/m=\\WO/VS+RU1*r0kq+`&b4aQoI=O-LsM&EQBru!oDcEkOI>j?q(j?.8/tm@25boBeMZ##]aV]:r%ZYcf3\\MUON>5F9Va&J\'G\"oq!f_YZ>e\\c,/O?<+94Y,FOY5..pUb:,B\"Sj_J=np?uLE9.`$PQmB#ol:OL1mjD8\"0SktKpONk?p-nE5hs8OKmZbla\'B?u%0%iRm<.maoJ8t`K`jPZQ!J?h?e.02sY!+Gj]*d`takN#?9j\":pB\"dbap3c^V*gT@>l35jsE4!aj+HpK*]B1$]!XW\\IQ>HYOYdK\'csKr:nl!OuLS_Fi6uiFjm:P!@X\\jFP:6qEdK$rLH],m`aYNSEBPDJY\\#;\"opo#\"7.uq*MQ.)\"iq^K45iA1)@%<33kokTb&GD3HOTl(W;-GZ;/fWV3dTH\"Jh)3S>s.)C+Hs5D$jJ^1blrV]BS-I4N%=)4CeTC,ANrP%nqJCbJJ-h\"$\\jN$1Y3npcisi2jtJto]Y99mOl2+7^tBK$OLOJ\\_Q`bn]\'J;YW#q](jb+BV2S#Z^.$h1#kKL2jA[\\TK4_g&OYs7beo*:J,PY0%g,gDE*+]AcOZOHnuegI/aj3aXD*_DlkgO/PNYt/qI\\P[ME$hfWUZ;sZFbF^#I_gF14lTL%a\'^F#jJHQ!A=X\'\"+#\\?ro\"(,^[YfMd0hp1ll#mD>&r!^=\"&0k_+(]o.]=kWX=aJ0t(+\\`>l>82-opciWdpEtj[mR)Mq$-?Eq;.=mZ\'j?t\'a/$9cWdDB\'\".\\:,)4Rj9=*Ct6XGA\"LQa`W9^=SXnrHKjZYonc)cL&L;UZ55O$G2eeSM01=DPOBBij(OCg#4_@N1@[T0tPt%\"OO`/!6rnDme3DUeql9&^pSfdW/02sT87]7:GI%Dp&mY,W7M4/D_HD`_7MR5o>;4[)*QN:A6+lM;4mG4[[6C>]#_IF$Fh@BPPQ))mJq>Br]XTtajL4Z3\'J4;8^6S7j-iLHRAF%4I%GI;/3=&:W*p=\'9udSt3mrWqFBadcAcJq\"\"sG&HVaL%\"Zoob\"q)baq$\'O+RNUS1A!#cC(/75k1)lp9I*U8T:=uM7-CV4\"(*uQZNL+i#/E9NLUIe2?J]i-*)K!\\)p$-28AU\"jkujlcoO_Te)U3<Hr<SdGU@VWG-VNl0h#jmWu7.@uP\'K[_h;^OiblGNKWe7!\'TWI*d$f<Dpd9ADWM6!fChTE35fQb%8G)UV2?P^$J_Y@P0G0Zrk\'=g]Wn.065\"9.XGAQ&gNYnN$7FkBMV-2Hn-.iEfNFp]bS1.\'(Yh6C`.K2j+`6L\'i7\\fU!i]9LQ/)PQda#\'<K?]Bpc5Ro--$#6[?+9?>8/u;ch[n\'Jj2RT\"J-UYO[JaPIg)\"A1L:d]j\"o)NN3kR_NCLWkkUXu1OG)\\Nh#)CVLlHG#g$b(e::@-BJ<@3!liU`V\"BQ6;`/q?(YBiq:rGB)tjs05m0I\"+H,b<]]PHK-^V91.]$,F8aR`8%XqS*#ZN9o%bg82[_KV+pTntCh#5f=<0BK8*De8\'cl\")a@#0tPl#MV(Ohq$KRT2I9lO(L7ZAGNp+RAI/J1O^FK4[XNcc@^h&Q69658Pl2!\"gQX09`XWanG%,[W^iq7mME)X.1H\\E#c=1)Q(PFeb)s8rBPmT\\i*QWRkbPS7Y*_!q#JJ\'0`ZJP<UEM1FOe.7=sl=YQ%3\\V%(eeU*3?d:B&S($\"QH2n>%gq8el\'[\"r>n@;`!ctF<t3dk5=+jsOFa)IQtZ*Up^VOQI1We_j+2+)9HOJTV4-VY_q4PW<`7`L\'ff\\l72&]Xtn_\\$_\'*+k1LW\\`U8QmN18Ca>FXLHL,YSlHUip\\dBugd9<,T5UAFqj+trd2p]8mgde+(:rMglJC?+aC&rI3!%Zk\\4Y2f6i_9s-Tkjh1:[<&WY?&R^,;cb?][G@Xu3Ep%`dc1),f7IIdIs^(IR%ZeE?*gPJ@$(i)-DBi7&l0[VH*S?6a^rXGF3H9.G3\'V^Vnt5a`O8f<=-Vg+s)?4q6nj;I1^c%S>VM:2(Gk-Ju1oIo,PY(W\\tJ%N53bl8Y+W#.NP[Muhf!$c,\"lTe63sG`[9oqL3ni+,n\"N5u9HtM;`Ui\"*,[r2h7UqZ8M.PLip3ti\'A2I>;HJ<\"t^^.j>]6Jh.[;jcM#hKV9he`N+&ut6_:L@77XV]H4#\'G*5dlf7.P.>N\"0OVO[lME;krPq`W30`/;7V)d2cBY7c[A\'Zh<QRAIn#eDt_8^a_9^U%s.I9c66$68Qi%O#9\'ma/=S?QnFVC],ep5GTi8<pGZQ<<PQ,E\"4\'mra?3a5k%2[As[X.C5-3TDF@eJGn3<XXdOX:VQg%ZUb+rLGHTsAcWCq_1u_D@B2!:@Mr$6VZKo>Wde?k<Hk]`>AS_Q^Te6\'QG+;/G<r`R+.s=p!@h?&Q$HCY\'3(b0X;m</7b[&Gn1c9>Y.9ed>h@(Y9\"(<:j?+4nJ\\,r8IB^bk_.5`eP(cMGQ!gLPo@:<+I`r)@9AT&9+:!N<\\ZYE#LDZqBWLd@q$nD@\'6aPn%p/,[EO(;IkotTp6D8*&fVb%AfeHtCPXHTZJn5>W\"gl98J\"Q`dE1=`#F.*VAS/Ui=GPYhka*p.c,kVl7\\jZJGdD@%rX\"?!,dK,q3X]f7+;BTu#%ZBKGsJ&Y3q*$SAR+K\"qJ)Cubqb690sSUgFF4P+dcA6MqQ)R+F^U>n`CY2I_IALY9>j^j9?A[uL1DNmb`R5;,-7];0qEhf\'8Q@gD/N\"obrBqG(u*(USF5)cm\\7rT`e9]C)b)I\"_EYn8S=)fl@548!`]%c3eCC=9RMCuD51Y.db]hl4&kR+\\BNuINefrk]=QFWM>iG?uQ.:2daT=c8\\<`,(F\'_r7#\']QE<F#0&`od2A<,3_!*:NjPd^l2T@*bBV6Ep6O%A0=W,u&die!JY53E^uF=[%m(0=W@1\\Ni.ti?Tp+!1/Zi=]J4LCmRQW%(ClMSlIeKQCEP/V1X]gC24(.A<K`qV2Bj[,GfB`c\\-lnY7@]Q!cnTfU%Q65@=9s!L-_pt`83k4Z,t*_(fA#DD)N`b333i3%on\\=gi=PYbrlG+9\'@b6Q8IcH0eK\'3,iLRC?#p\'rrD@,4mK2eWnA@U^N6Yg?_313$CGtOaAKPJ;=_nQaj7*:>O]_2HPJDYq\'T*WdB#9%R-.5fOPdF,j[#3,[\\VZ>r.!4s@.:L0R7[R.U8l]udflr(7JV/YC#9T?H#%t+/pa)4o6,UJ>(_d`\"*\"j/**YB:%S=E6K0\"mr0eoiDF!42_1!&L_8\":a5W\'*s^h!]2p#SMG6-%0]6NWmk3/I?UGTW@sUs>4@n5N<_[NY[k_A%9n#Bl&#F_ilhSC\'1i.Cji6+T*^$-@n`I`U++F:m4X3`4qTV\\#jFj3S-W6l;*[;FiS9P>Q-$cmP4K6aQP8joX8\\\'_\'7h#nadItrGJ\"8+-e#P,+QP!]gB<9kKfPkG0ICA?t--K>:[CS@\\S$ldm\"-,D)(Ip@kF^!2tAb/;Fj_F8\"Z`W3VEA\'.s\\OS*,\'<uUPN6],NGe#Mu+=Waf3F37C=]%\'40JA:$S[]c\"KI+LAoV*Y]:##?Y\\D)\'TKA^G5[g9rlJu!(7j`j(UXHI\'Nh)HAdQGLb3EMAb(MWY4Q\'IRU\\M_\'M2[dbYfr@h\"<Z>KXJM\'ePo`GgF)BSPVn6\'PnKeT;MfI$X00Sf`X:8Ng3BT#ru4@t-J0c;Vg/Q[KM\'!e:7=C[3HjA.9)<.YG$*6\'dXoJ;%3F,Gu[,cfrL!q/X1)!CW;O`e@CG=QfKG(r](8j[E>&:##?)l\"n(9Xromham\\qYisA\"+S007a;Yc$#(IUhq<)#;FbB@.:&&K_idO[U!AQ:[+f+6nZIfk34N1_\"_O;,r50gA0a-u-:Lh)HBMkDO%[D9)>I+H0ciAMBTeY*(?n\\P:iO_`5oa\\cEoV,:J9[,Xk?X=7O.8=]%%%b0@H`oU=BP+H0d=jZo,Me=+5PY4FU3Hn!4<DVD68`\\!`QF@cV*ZYQob@Z2@,/2Wj4g>qkS(7VQ6#R(s%L5phAKoL_CLJL0+7(`SL(c_aGaRbU2B^)>\\C1Ldc6$[WcYd$<W?+(^_:(pP^EskVk&D5^Vf$<-gd0Tl.@7br`Osi-kMo-3a,dXFXUc[,u\"\\qi=Q:PS-B8\'6[enB2PK@.tNDG%^A#o#kY9I!CEd0,@h02N[OA5ObQnE)el3]Xf+=frij4f=r5F[cBU1g-dqU$s?4j(DQ@C>],?9gN.m\'F%&MiMOR<jV.Qh7p0Z@.IiXWd`;q?=A$\\fWuKRP4Q\\XA\"1+(..=>P56kUB!6D`\'I9GX+M170UNLPt]t!hi.9Oil_l>cCV:ZoYn._.[=oBCFdBb-dG>4\"bF+F`5e-5GL.l2)D_GZiYf\\1IYc0<$kZp<S)<#G4lE+:(Sf^N^0>c(b?.m+Hje[.ZIcQ`h.q(FdmhCAG9F4ItGl($\\T_/6n4B^/!MNk7]XuWN-=o4!\'h)4!enH&8/)tp`Sh+k&t)(_d2<FS:<kIcQ7JHBjm`L@O]WfmaV@F5>,0X4%7ZQ7@^Q5e+><p*K-QI+1^8lU\'8RnhKhA*rQn1.NbMrt`,-aNcPM=aDffM/S3P-](.^_aHdi45mZc,.Z(lTfJW5WMX4.Iq4B=V2a[_)LsGWK37[l]*G\'?[hH8fR(Y57\"#D8[H&XX1E#WhMK^gs(\\s*fmB.a&2R\'$\"Rjk!241h]8ra75kg_pEQ=U+8%/O<>p\\OLSk!/gRh9]O,dQdS2Dg^ni50]:B%\"=)DbQBF\\VIBUuYjoFYg$oMX[V/LTD\"DHIBe<2s_rp@1di6%7c#M<L4b(*cr80a2ADfX05KX(T0[J_FN$/n;MrTK5Ant\'D##Zr*\'X*\'\"adMNjNl_^G6:WVjWPP??!!5C34XLUo\'sM`>bf]#bEgaPp[A.;&!5>_L)Rg7=<c*7MQq@is2g[n].56WI!<D$n^f6dscZNZR+rEU09ipr<R,.Aof_=MS(Y<<74Cs*d.6j!0-3!o,Dm,XsVNZ<2=RC3]di]Nq#35O!qr;M5\\W)Bmj5]Rql=KIj@L)(Q:R9Kl_)apNb.g/t4\"FeA)p]ck%-_kQ+D\"KV#3.2KN^nmGRm@Gk?j&\\-h)#Dg?Nmo;^7eZY_(OG0d)[3Ce1b8c&;\'nV[+Vr/ejF/D&\\H&T@2Go&#aBLi`(1\"T\\]2[oB$Uc/UFJ%q^\\r!4LYq5Y#R2E2+,.au</:\'m$tFEP`ekLA)`WjR_f/Dj>-sN\"%@NrP41\'If^a(FB8Flr>GP+Ct,rE+PjEb/g2_`rNrJ.(AQ5eDEela)Idr6/XMbOr*Xr!RRho#`3cU4Y[3#`F^C^A)d<8[;IE-/Zt?YJI7$$et$X/6p#_Ddk9(S/[Zi[pU12@f]]\'N%S(NtF6_=J\\ikQ7SM.YkX]n-fX_(<C5WcH+OtNL?ZZ4TCgHYGYc$45(\\C9W#]_\"5(pc>MUNA7,K_M&:otam.^iBMOgU2m)jS=:Fu@PWY`m4WYRd,AdSs3.bt=f`2!pU=fe5L9dD(0DCocU/`1QiJ>5T*#&<Q0O`:<IUVbO\'797op^j%rUR)\"kY\\caGZXbOLL1c^lm*$Yc>-5LL\\sI-A=P4q%+/(gd`\"O9[mAV,\\2lWa(o-U\"2h04Op0A*AgZ^Y=VuKmiUT7eJWW<jKWc*6C=E%)V[f\'#lRstf9@a-LOQC?eg\\\"B@V]3enZd2;<P\'S!(XF[8.P[\\ZIdeb\\jT:gpA,nYE(T:qgEc2%M\"-MRWXTU`\'R\"gd7JPk]#mPUSQLH)SG&A1eh[=-,kQjcL2Mt.Q$b>sQrF3V@9Mf$3YH]AV6j9fkoG\',:XM]<<:nrgi4a,!s_-mL\'a6-ouAZ_m.hBC0nSZKbLEBq680k\\9OrH>^PsY3j!]VIMi$(\'Fb0D-obu%?)@Ja,&bBIP;n%(\':Sd9\'4-jWK(7n2sgu@&g8@t!lehHh.J@,hCY:H*Vtr!$\\DYL\\U0N!1G^g;4CMAV&iVNal=mQ\'Ri++?Co6%dK&_6\"`79qC`^I`o:@%6XC=.dk3SGL;$jggrT1r+h3=\\-`d\\O\\g2F!h>ZJ)\\GkkWSj`0()o)gDAV!ruC6$^M6\\$qE;1XZE3(3f8[.(f<db1/Bm6\'Gr-3j.b[Y2G,nAMcPd29q$ZjG12HfcdOn1kFt(\'>ArL4%!(fG\'B%0tR,/@OX$14Y3l&e#8&J#,/UJ&q<l%Bl-u*3kP4f_@ea?-B4id&RBG^0L6S<-^-$%0TSV-d@gEUraarK7@!/S7?1_o$Z\"9\\qh\'(c*jkN;K@htG@K#DYRi,hZB@4l%#_hW7V*Lq[;pZe:j3LY/2iq:c9M[GqH1)f\"OR.$u8NE9W(\\O:tNd#RYYR3c^cB+3T[_O/*JXe4FVj_/`XrMjsp;;c8%4bRdGW6NfZ3eAm`VW2$q;nG\\4(KCf*%$olf<!pBRh-=jM*@\\(<mh#hhlEJd<h6Rjojoh@4rs+cdM9[Q$V^FR?l,@a#?_l+#p9g2lH[pZ2RIJk%rGHkK;$cKp5i4U(32D?VJK>1cjQWI7P@gH`KW+MJK[+3U\'7ZOh8jt2l1VUPL\"WLqBufEBoHBD7(NlslBucnMh9Ctcs,dq,Lhb.L#+DP^0s4/!VE_^s=OVe2=W\\4^e,PcCE22_ch/I\'hUlGYUA8&t/hjHYVan/)<GkF;k:)FkN0rb)(o5Hp%F-oGbi[A\"U8g8JVSb=h\"4Y[(!qs2g$LI]6p\"+_7nIU2VHQ]SJ`3B.<Aog]qJ\"s#>_\"0bD`$.dnfN0YT\'!Fg,pp6HNKGd)(*U6*A8lbVpe/!!6^N*+_[+C!Cg$*/$n=o-!E6Ok3q#+(RHg-q:dmBrT*;*c+PqER-e0<o`U?<%@gr[(4[S2XX;.6PO$\'Qb4\"u7g$D3#Pb`#HBUDuGMD_3,^#cNj)Z*c_F5=M@U+LOAIGEc.mE)E)U.U-RWhS\\=LQ.d+.og5\\pZUC0IjN$83C&E-m/cn<*%_5flN0eRO6*qGC8=H9UB>U+-f@O\"eKlBN\"R-+qi::6)+fVs_TWC8W[eE5h8WS3Z\\-[N.+8YADEH6+e=4a54)Z\"3(et85G/K5H2o>>sb41-oGSKcb^@_H(+hT\\#UfT1sN`TM!o/q(lf^2UdRaoi%\",cIe>b\'g8Q!u29A%WMF[FL(W7LKn6(f3P$>M[7dYp*Ta9^C,8d:2W/%1^FeZn\"?e,r9l`o$/dRRj^7SdXA6B#(U4U1UB)\\J$q=.Ig!fQ3ro7:MC1p.4Tg%Nf_5p\'uTKtgl&+-Y%Ng&[QTk*^VcU#%maCB%\\#Qh(Wq\"qe;5<p`4YhC^*%st<D,7Ce$5UCU[!PDsD)ML#D_kSF^R;F3[Riq*@3lJ%)EFf00,1Nj/**,9\",X9rjhYLUSf,DAb_dE>sD?]MFSNUb\\RqR6VG1sa/m\'Gn^DRQN+F:c<-\".Q_dQrJtlXn)T*]:Z<1GK)6@FogjqY4iokE\"EDH+Dle^-K6Or,-4JnF^C2-:@3%<JS:5\"DZ(DSi2.(%T0Of4#Sct6&nV,q^^qnfi)UUZJA;_00qRDeJV+L;7aPrJ>s%7gd>so2L#K5d@bq%=GURA++0rCPJ$9YbWI>7pp*U%@iXae!VKbij>.saZ=Q[pa\\1\\bB6BtC5$broi*F8M&gg3#k)im-e$0B]Img!*Y$\\(6U`uKBXTDGn]m>[;(O=^*s5,a\\;a=5Gup=bg2*f1_oc_k];9fi3#NU3;3(LN\'cqT8Hr>Zd5_^jCVhRH)PNVC?3r@[u@H_>HAB;!&4E0c&\\13\'k[BRqFIn-EU=^EJn5>gR_\'XAj1#\\#Xj/;?+r@Y1Nh(Z345)I6Oo\'Do!eaVZm.o8l:5PjKP\":QZ)03LnXfTD/3AL9i^!ksPo9fN7Y7tc?,ac\\LT?.ALaFDU&m2!@eP]$cRA4Ja>CC:.MAHC@8<N!(=@<]E9N\'<=Xkj.`.RGRtbnfEB-F.Lmm4BV`\'PM2HK4#sOma`Qr_5XgVKc7@ZP$15[V3*eudaKo_[UuQP1aQ)$(Ma>5<\\nOqZ*oCrm9il\'r:4(^4f<M(W!pAV@CN(u1Jij%l9[^(6r[.NL6B`M+qbfOauR9?;+4-WV;aOD%E4\'f;U@Jk(9LGB7FA\\oA?S9RMc,2n&5;)E@BmnuV.AeLQE>#3?O;12gTUT\"k&ul:<Q#+:kC?7WVa,^YAM$K0=Yi?\\Q4aL[[c5%[\\nL^HjSdUF.t04G&Z>\\M8S?5Kl\"<_c2ZhC6%!!@hT`-Rn@9-Q/C+78XH.CbZ\"i8!I(2+!m;lHE%\'i`([86o^>Zbf$3Aq`iMCNYah=jtc`_P[g%ZZetI=N$na<3*:L`[%q^R\\Qg5ciOH_TV73/L4PGCr,H!F<(d?N=&piT;-\'[4N%(#kZ]6g((97_+@(GBf<Zla/k1!db_\'A\'`d74OS9L[IBCo`e8)@ZgHQ;6H;hhp?_>Q%G640!Yh0B+q3q#)gBUOGp=i[)\\Q\'QECY:Du04CRnDf>1QN^Yg*X\'CAi+oPd&Eo._<nqPUV`Z[eUQM4L-@H1^b3*9;9p]+LOWa$Q-^_\'04c`/OHT(C/Kjg:!-GiC<)=\"FET)br%KDGBJkC@A2KHPl5WhMT]E%ZNb$Ni5p`Z!?g#ZKN2!]mSWN?k.a\"hc$D51@WJf:?X.i!6WLd\"/6TNRo6B,gQ_TA`OBkatgUBMhO:F_/u9!W[E\"XQSl$=\'\'c6&6dA!(%CpX7A\'tcMQSp*e*$sQ65C.*`[-4XI;&\'\'f\\=,80S-Y\'W%9$:EXQ(_>c]ihu!1uIcb_Mjeq\"L]$6ZI\\;,ihE]am3A^I>d1>Z=8lLJS]o0qhTs\'a1/N75Q\'maoWeDJHJt>\\_0gm!nSK2\'PaiLDO0tk,G2\'0gN[Ud;7=Slr\\+\'C<+#q\';<jYU`E;;06:o7QaKm3!%D./LH#CFj7)8\\DJIf/h=C8arUZ)!+`fkO[\\.5]RN@Hf]hC;<0d+0)Je!$\'?Bni^!^m5iZ4XMJ&2>Yk$rI!9c>%<g(N6ZM7g[c8dej7\'dpH/))]/J*T03)O7!-/JVkN#(CbMC4hO`P0B!d[Zdt`FCp\"LGPMIntbG0r.kU!=kH,TOe%d7/:tpquK<dVjtb7[%>7Pt)eOn,sslJ-^]?(dYF*_Z1+6C)BuZ\"H>=>_\\5Ta\"=*-0?-t8\\a8cKC?j+HId7/2%!g,kT5_oId-q91S/<KrEU;.%l;59-9EX\\R.hbc=>>leOSTCMsJcf4&=Z,<3fiS_m#NgJWHRI6K#r6$1C(F+Klpcpb%>Bo\\8Q%/?&nNI((*e-MW/Zlo4^4A((-P;uAO-XZ`#Qi\"(GCDgLaF+fNnaC[7qbK>a]>U@/I&%uq\\jBO>[p[=naRlh3dBE.;*08FL&W_]Ok1Am$r*cnGc0)!EhZSRO9%<=KI5Ul3rT/`4\\\'I_gp!\'k1C$n-)jKrOO$REK2Ck_f&T,<)#fp@]S&<HJ;WZY:4FcCM?D>N-IGV4g9k<C-P(J+rtiV2\\nq&Nr%#F?HlMtl,US3Pg4,Ku30.HhS0>$tRk79FPVKlIum[r\'K/)(u8>q\\fk#+fg\'3)H[\"+qG,@PmlH#&:FQeW7>27p&D-db7+L#e.3Us-kERV^>%#%9g\'ieq.hZq[rCark>:jogq\"rM^*s;a?P=/=LShkoI)qekr^(0-*$,@*is8H7+?fnB7#Z+Xr433Im\\F;fa)C>?u$WoOn                    ";
    hz_font_load_woff2_from_memory_base85(HZ_BASE85_ENCODING_ADOBE, noto_sans_arabic_ui_woff2_base85);
    return 0;
#endif

    if (!glfwInit()) {
        fprintf(stderr, "Failed to initialize GLFW!\n");
        exit(-1);
    }

    glfwWindowHint(GLFW_CLIENT_API, GLFW_OPENGL_API);
    glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);
    glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
    glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 3);

    GLFWwindow *window = glfwCreateWindow(WIDTH, HEIGHT, "Hamza OpenGL 3 Demo", NULL, NULL);

    if (window == NULL) {
        fprintf(stderr, "Failed to create window!\n");
        exit(-1);
    }

    glfwMakeContextCurrent(window);

    if (!gladLoadGLLoader((GLADloadproc)glfwGetProcAddress)) {
        fprintf(stderr, "Failed to load OpenGL (gladLoadGLLoader)!\n");
        exit(-1);
    }

    printf("OpenGL version: %s\n", glGetString(GL_VERSION));
    printf("OpenGL vendor: %s\n", glGetString(GL_VENDOR));
    printf("GLSL version: %s\n", glGetString(GL_SHADING_LANGUAGE_VERSION));
    printf("OpenGL renderer: %s\n", glGetString(GL_RENDERER));

    // setup sdf glyph cache texture
    hz_glyph_cache_opts_t cache_opts = (hz_glyph_cache_opts_t){
        .width = 4096,
        .height = 4096,
        .max_sdf_distance = 8.0f, 
        .padd = 0.0f,
        .x_cells = 16,
        .y_cells = 16
    };

    hz_context_t *ctx = hz_context_create(&cache_opts);
    hz_renderer_gl3_t g;
    hz_gl3_device_init(&g, &cache_opts); 

    stbtt_fontinfo fontinfo, fontinfo2, notosans_kr_fontinfo;
    // if (!load_font_face(&fontinfo, argv[1])) {
    // if (!load_font_face(&fontinfo, "../../../data/fonts/Quran/OmarNaskh-Regular.ttf")) {
    if (!load_font_face(&fontinfo, "./TimesNewRoman.ttf")) {
        printf("Failed to load font file!\n");
        exit(-1);
    }

    if (!load_font_face(&fontinfo2, "./OmarNaskh-Regular.ttf")) {
        printf("Failed to load font file!\n");
        exit(-1);
    }

    if (!load_font_face(&notosans_kr_fontinfo, "./Orbit-Regular.ttf")) {
        printf("Failed to load font file!\n");
        exit(-1);
    }

    hz_font_t *font = hz_stbtt_font_create(&fontinfo);
    hz_font_t *font2 = hz_stbtt_font_create(&fontinfo2);
    hz_font_t *notosans_kr_font = hz_stbtt_font_create(&notosans_kr_fontinfo);
    hz_font_data_t *font_data = hz_font_data_create(font);
    hz_font_data_t *font_data2 = hz_font_data_create(font2);
    hz_font_data_t *notosans_kr_fd = hz_font_data_create(notosans_kr_font);

    uint16_t arialid = hz_context_stash_font(ctx, font_data);
    uint16_t omarid = hz_context_stash_font(ctx, font_data2);
    uint16_t notosans_kr_id = hz_context_stash_font(ctx, notosans_kr_fd);

    hz_buffer_t ar_sample, en_sample, ru_sample, sample_kr, sample_sy, urd_sample;
    hz_shaper_t *shaper  = hz_shaper_create();

    { // Arabic
        hz_shaper_set_direction(shaper, HZ_DIRECTION_RTL);
        hz_shaper_set_script(shaper, HZ_SCRIPT_ARABIC);
        hz_shaper_set_language(shaper, HZ_LANGUAGE_ARABIC);

        hz_feature_t features[] = {
            HZ_FEATURE_CCMP,
            HZ_FEATURE_ISOL,
            HZ_FEATURE_INIT,
            HZ_FEATURE_MEDI,
            HZ_FEATURE_FINA,
            HZ_FEATURE_RLIG,
            HZ_FEATURE_CALT,
            HZ_FEATURE_LIGA,
            HZ_FEATURE_DLIG,
            // HZ_FEATURE_SWSH,
            HZ_FEATURE_MSET,
            HZ_FEATURE_CURS,
            HZ_FEATURE_KERN,
            HZ_FEATURE_MARK,
            HZ_FEATURE_MKMK,
        };

        hz_shaper_set_features(shaper, ARRAYSIZE(features), features);

        hz_buffer_init(&ar_sample);
        hz_shape_sz1(shaper, font_data2, HZ_ENCODING_UTF8, "الله لا اله الا هو الحي القيوم لا تأخذه سنة ولا نوم له ما في السماوات و ما في الارض من ذا الذي يشفع عنده إلا بإذنه يعلم ما بين ايديهم وما خلفهم ولا يحيطون بشيء من علمه إلا بما شاء وسع كرسيه السماوات و الأرض ولا يؤوذه حفظهما وهو العلي العظيم", &ar_sample);

        hz_buffer_init(&urd_sample);
        hz_shape_sz1(shaper, font_data2, HZ_ENCODING_UTF8,"اللہ کے نام سے شروع جو بہت مہربان رحمت والا ہے اردو ترجمہ و تفسير سورۃ القرآن باب وار", &urd_sample);
    }

    { // English
        char buff[500];
        sprintf(buff,"gl:%s vendor:%s glsl:%s renderer:%s",
                                glGetString(GL_VERSION),
                                glGetString(GL_VENDOR),
                                glGetString(GL_SHADING_LANGUAGE_VERSION),
                                glGetString(GL_RENDERER)
                                );

        hz_buffer_init(&en_sample);
        hz_shaper_set_direction(shaper, HZ_DIRECTION_LTR);
        hz_shaper_set_script(shaper, HZ_SCRIPT_LATIN);
        hz_shaper_set_language(shaper, HZ_LANGUAGE_ENGLISH);
        hz_shape_sz1(shaper, font_data, HZ_ENCODING_UTF8, buff, &en_sample);
    }

    { // Cyrillic
        hz_buffer_init(&ru_sample);
        hz_shaper_set_direction(shaper, HZ_DIRECTION_LTR);
        hz_shaper_set_script(shaper, HZ_SCRIPT_CYRILLIC);
        hz_shaper_set_language(shaper, HZ_LANGUAGE_RUSSIAN);
        hz_shape_sz1(shaper, font_data, HZ_ENCODING_UTF8, "Са́мый холо́дный го́род на Земле́", &ru_sample);
    }

    { // Hangul
        hz_feature_t features[] = {
            HZ_FEATURE_CCMP,
            HZ_FEATURE_LJMO,
            HZ_FEATURE_VJMO,
            HZ_FEATURE_TJMO
        };

        hz_buffer_init(&sample_kr);
        hz_shaper_set_features(shaper, ARRAYSIZE(features), features);
        hz_shaper_set_direction(shaper, HZ_DIRECTION_LTR);
        hz_shaper_set_script(shaper, HZ_SCRIPT_HANGUL);
        hz_shaper_set_language(shaper, HZ_LANGUAGE_KOREAN);
        hz_shape_sz1(shaper, notosans_kr_fd, HZ_ENCODING_UTF8, "대박! 나도 스타 워즈를 좋아해요. 가장 좋아하는 에피소드는 뭐니", &sample_kr);
    }

    { // Syriac
        hz_feature_t features[] = {
            HZ_FEATURE_CCMP,
            HZ_FEATURE_ISOL,
            HZ_FEATURE_FINA,
            HZ_FEATURE_FIN2,
            HZ_FEATURE_FIN3,
            HZ_FEATURE_MEDI,
            HZ_FEATURE_MED2,
            HZ_FEATURE_INIT,
            HZ_FEATURE_RLIG,
            HZ_FEATURE_CALT,
            // optional
            HZ_FEATURE_LIGA,
            HZ_FEATURE_DLIG,
        };

        hz_buffer_init(&sample_sy);
        hz_shaper_set_features(shaper, ARRAYSIZE(features), features);
        hz_shaper_set_direction(shaper, HZ_DIRECTION_RTL);
        hz_shaper_set_script(shaper, HZ_SCRIPT_ARABIC);
        hz_shaper_set_language(shaper, HZ_LANGUAGE_SYRIAC);
        hz_shape_sz1(shaper, font_data2, HZ_ENCODING_UTF8, "ܝܠܦܢܰܢ ܠܫܳܢ̈ܐ ܢܘܟ ݂̈ܪܳܝܐ ܕܰܡܥܕܪܝܢ ܠܢ ܒܝܰܕܥܳܬ݂̈ܐ", &sample_sy);
    }

    glfwSwapInterval(0); // disable V-Sync
    glfwSetScrollCallback(window, (GLFWscrollfun)scroll_callback);
    glfwSetCursorPosCallback(window, (GLFWcursorposfun)cursor_callback);

    // create fs_quad_shader
    //GLuint fs_quad_shader = create_fs_quad_shader();

    // glDrawBuffer( GL_COLOR_ATTACHMENT0 );
    GLuint emptyVAO; glGenVertexArrays(1, &emptyVAO);

    int width, height;

    while (!glfwWindowShouldClose(window)) {
        glfwGetFramebufferSize(window, &width, &height);
        glViewport(0,0,width,height);
        half_height = height*0.5f;
        float half_width = width*0.5f;

        glfwPollEvents();
        glClearColor(1.0f, 1.0f, 1.0f, 1.0f);
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        hz_frame_begin(ctx);

        float zoom = get_linear_zoom();
        hz_camera_begin_ortho(ctx, camerax-half_width/zoom, camerax+half_width/zoom, cameray-half_height/zoom, cameray+half_height/zoom);

        hz_draw_buffer(ctx, &sample_kr, notosans_kr_id, (hz_vec3){200.0f,340.0f},
                            &(hz_buffer_style_t){
                                .shear = 0.0f,
                                .col = (hz_vec4){0.0f,0.0f,0.0f,1.0f},
                                .weight = 500.0f,
                                .scale = 1.0f
                            },
                            90.0f);

        hz_draw_buffer(ctx, &ar_sample, omarid, (hz_vec3){100.0f,540.0f},
                            &(hz_buffer_style_t){
                                .shear = 0.0f,
                                .col=(hz_vec4){0.0f, 0.0f, 0.0f, 1.0f},
                                .weight = 500.0f,
                                .scale = 1.0f
                            },
                            60.0f);

        hz_draw_buffer(ctx, &urd_sample, omarid, (hz_vec3){100.0f,200.0f},
                            &(hz_buffer_style_t){
                                .shear = 0.0f,
                                .col=(hz_vec4){0.0f, 0.0f, 0.0f, 1.0f},
                                .weight = 500.0f,
                                .scale = 1.0f
                            },
                            60.0f);

        hz_draw_buffer(ctx, &ru_sample, arialid, (hz_vec3){300.0f,100.0f},
                            &(hz_buffer_style_t){
                                .shear = 0.0f,
                                .col=(hz_vec4){0.0f, 0.0f, 0.0f, 1.0f},
                                .weight = 500.0f,
                                .scale = 1.0f
                            },
                            60.0f);


        hz_draw_buffer(ctx, &en_sample, arialid, (hz_vec3){200.0f,300.0f}, &(hz_buffer_style_t){
                                .shear = 0.0f,
                                .col=(hz_vec4){0.0f, 0.0f, 0.0f, 1.0f},
                                .weight = 500.0f,
                                .scale = 1.0f
                            },60.0f);
        // hz_draw_buffer(ctx, &ru_sample, arialid, (hz_vec4){1.0f,1.0f,0.0f,1.0f}, (hz_vec3){200.0f,400.0f},60.0f);
        // hz_draw_buffer(ctx, &sample_kr, notosans_kr_id, (hz_vec4){1.0f,0.0f,1.0f,1.0f}, (hz_vec3){500.0f,200.0f},80.0f);
        // hz_draw_buffer(ctx, &sample_sy, omarid, (hz_vec4){1.0f,0.0f,0.0f,1.0f}, (hz_vec3){500.0f,600.0f},80.0f);
        // hz_camera_end();

        hz_camera_begin_ortho(ctx, 0.0f, width, 0.0f, height);
        
        // hz_draw_buffer(ctx, &en_sample, arialid, (hz_vec3){0.0f,height},
        //                     &(hz_buffer_style_t){
        //                         .shear = 0.0f,
        //                         .col=(hz_vec4){1.0f,1.0f,1.0f,1.0f},
        //                         .weight = 500.0f,
        //                         .scale = 1.0f
        //                     },
        //                     24.0f);
        hz_frame_end(ctx);
        hz_gl3_render_frame(ctx, &g);

        // glBindVertexArray(emptyVAO);
        // glUseProgram(fs_quad_shader);
        // glActiveTexture(GL_TEXTURE0);
        // glBindTexture(GL_TEXTURE_2D, dev.sdf_texture);
        // glUniform1i(glGetUniformLocation(fs_quad_shader,"has_texture"),1);
        // glDrawArrays(GL_TRIANGLES,0,6);

        // glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
        // glUniform1i(glGetUniformLocation(fs_quad_shader,"has_texture"),0);
        // glDrawArrays(GL_TRIANGLES,0,6);
        // glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

        glfwSwapBuffers(window);
    }

    hz_context_release(ctx);

    glfwDestroyWindow(window);
    glfwTerminate();
    hz_deinit();
    return 0;
}
