import React, {
    useState,
    Dispatch,
    SetStateAction,
} from 'react'
import Switch1 from '@components/Switch'
import Box from '@components/common/Box'
import Text from '@components/common/Text'
import { ConferenceAppType } from '@lib/dataTypes/MeetingAssistType'
import { NextApiRequest, NextApiResponse } from 'next';
import supertokensNode from 'supertokens-node'
import { backendConfig } from '@config/backendConfig'
import Session from 'supertokens-node/recipe/session'

export async function getServerSideProps({ req, res }: { req: NextApiRequest, res: NextApiResponse }) {
    // Notice how the server uses `API` from `withSSRContext`, instead of the top-level `API`.
    // const SSR = withSSRContext({ req })
    // this runs on the backend, so we must call init on supertokens-node SDK
    supertokensNode.init(backendConfig())
    let session
    try {
        session = await Session.getSession(req, res, {
            overrideGlobalClaimValidators: async function () {
                return []
            },
        })
    } catch (err: any) {
        if (err.type === Session.Error.TRY_REFRESH_TOKEN) {
            return { props: { fromSupertokens: 'needs-refresh' } }
        } else if (err.type === Session.Error.UNAUTHORISED) {
            // this will force the frontend to try and refresh which will fail
            // clearing all cookies and redirecting the user to the login screen.
            return { props: { fromSupertokens: 'needs-refresh' } }
        }
        throw err
    }

    if (!session?.getUserId()) {
        return {
            redirect: {
                destination: '/User/Login/UserLogin',
                permanent: false,
            },
        }
    }

    return {
        props: {
        sub: session.getUserId(),
        }
    }
}

type Props = {
    isZoomAvailable: boolean,
    isGoogleMeetAvailable: boolean,
    zoomMeet: boolean,
    googleMeet: boolean,
    enableConference: boolean,
    setParentZoomMeet: Dispatch<SetStateAction<boolean>>,
    setParentGoogleMeet: Dispatch<SetStateAction<boolean>>,
    setParentEnableConference: Dispatch<SetStateAction<boolean>>,
    setParentConferenceApp: Dispatch<SetStateAction<ConferenceAppType | null | undefined>>,
}

function CMPWStep3(props: Props) {
    const [zoomMeet, setZoomMeet] = useState<boolean>(props?.zoomMeet)
    const [googleMeet, setGoogleMeet] = useState<boolean>(props?.googleMeet)
    const [enableConference, setEnableConference] = useState<boolean>(props?.enableConference)
    const [notAvailable, setNotAvailable] = useState<boolean>(false)

    const isZoomAvailable = props?.isZoomAvailable
    const isGoogleMeetAvailable = props?.isGoogleMeetAvailable
    const setParentZoomMeet = props?.setParentZoomMeet
    const setParentGoogleMeet = props?.setParentGoogleMeet
    const setParentConferenceApp = props?.setParentConferenceApp
    const setParentEnableConference = props?.setParentEnableConference

    const onEnableVirtual = (value: boolean) => {
        setEnableConference(value)
        setParentEnableConference(value)
        
        if (!isZoomAvailable && !isGoogleMeetAvailable) {
          setNotAvailable(true)
          setParentConferenceApp(null)
        } else {
          setNotAvailable(false)
        }
    
        if (value === false) {
          setParentConferenceApp(null)
        }
    }

    const onZoomMeet = (value: boolean) => {
        setZoomMeet(value)
        setParentZoomMeet(value)
        setGoogleMeet(!value)
        setParentGoogleMeet(!value)
    
        if (value) {
          setParentConferenceApp('zoom')
        } else {
          setParentConferenceApp('google')
        }
    }

    const onGoogleMeet = (value: boolean) => {
        setGoogleMeet(value)
        setParentGoogleMeet(value)
        setZoomMeet(!value)
        setParentZoomMeet(!value)
        if (value) {
          setParentConferenceApp('google')
        } else {
          setParentConferenceApp('zoom')
        }
        
    }

    return (
        <Box flex={1} justifyContent="center" alignItems="center" minHeight="65vh" width="100%">
          <Box flex={1} pt={{ phone: 'm', tablet: 's' }} flexDirection="row" justifyContent="space-between" alignItems="flex-start" width="70%">
            <Text variant="optionHeader" pt={{ phone: 'm', tablet: 's' }}>
              Enable a virtual meeting for this event?
            </Text>
            <Box flexDirection="row" justifyContent="flex-end" pt={{ phone: 'm', tablet: 's' }}>
              <Switch1
                checked={enableConference}
                onValueChange={onEnableVirtual}
                style={{marginBottom: 20}}
              />
            </Box>
          </Box>
          <Box flex={1} width="70%" pt={{ phone: 'm', tablet: 's' }}>
            {
              notAvailable
              ? (
                <Box width="100%">
                  <Text variant="subheader">
                    3rd party providers are not enabled. Please enable either zoom or google to use this feature.
                  </Text>
                </Box>
              ) : (
                <Box flex={1} width="100%">
                  {
                    enableConference
                    ? (
                      <Box width="100%">
                        <Box width="100%">
                          {
                            isZoomAvailable
                            ? (
                              <Box  width="100%">
                                <Box flex={1} pt={{ phone: 'm', tablet: 's' }} flexDirection="row" justifyContent="space-between" alignItems="center" width="100%">
                                  <Text variant="optionHeader" pt={{ phone: 'm', tablet: 's' }}>
                                    Enable a Zoom Meeting for this event?
                                  </Text>
                                  <Box flexDirection="row" justifyContent="flex-end" pt={{ phone: 'm', tablet: 's' }}>
                                    <Switch1
                                      checked={zoomMeet}
                                      onValueChange={onZoomMeet}
                                      style={{marginBottom: 20}}
                                    />
                                  </Box>
                                </Box>
                                <Box>
                                </Box>
                              </Box>
                            ): null
                          }
                        </Box>
                        <Box width="100%">
                          {
                            isGoogleMeetAvailable
                            ? (
                              <Box flex={1} pt={{ phone: 'm', tablet: 's' }} flexDirection="row" justifyContent="space-between" alignItems="center" width="100%">
                                <Text variant="optionHeader" pt={{ phone: 'm', tablet: 's' }}>
                                  Enable a Google Meet for this event?
                                </Text>
                                <Box flexDirection="row" justifyContent="flex-end" pt={{ phone: 'm', tablet: 's' }}>
                                  <Switch1
                                    checked={googleMeet}
                                    onValueChange={onGoogleMeet}
                                    style={{marginBottom: 20}}
                                  />
                                </Box>
                              </Box>
                            ) : null
                          }
                        </Box>
                      </Box>
                    ) : null
                  }
                </Box>
              )
            }
          </Box>
        </Box>
    )

}


export default CMPWStep3
