import React from 'react';
import { View, ViewToken } from 'react-native';
import { Link } from 'expo-router';
import Animated, {
  useSharedValue,
  useAnimatedStyle,
  withTiming,
  SharedValue,
} from 'react-native-reanimated';
import { FlatList } from 'react-native-gesture-handler';
import { Text } from '@/components/core/text';
import { useColorScheme } from 'nativewind';
import { COLORS } from '@/theme/colors';

type ItemType = {
  id: number;
  title: string;
  description: string;
  date: string;
  author: string;
  imageUrl: string;
};

export default function Feed() {
  const viewableItems = useSharedValue<ViewToken[]>([]);

  return (
    <View>
      <FlatList
        data={data}
        id=""
        contentContainerStyle={{ paddingTop: 10, paddingBottom: 35 }}
        onViewableItemsChanged={({ viewableItems: vItems }) => {
          viewableItems.value = vItems;
        }}
        renderItem={({ item }) => {
          return <ListItem item={item as unknown as ItemType} viewableItems={viewableItems} />;
        }}
      />
    </View>
  );
}

type ListItemProps = {
  viewableItems: SharedValue<ViewToken[]>;
  item: ItemType;
};

const ListItem: React.FC<ListItemProps> = React.memo(({ item, viewableItems }) => {
  const { colorScheme } = useColorScheme();
  const bgColor = colorScheme === 'dark' ? COLORS.dark.card : COLORS.light.card;
  const rStyle = useAnimatedStyle(() => {
    const isVisible = Boolean(
      viewableItems.value
        .filter((viewableItem) => viewableItem.isViewable)
        .find((viewableItem) => (viewableItem.item as ItemType).id === item.id)
    );

    return {
      opacity: withTiming(isVisible ? 1 : 0),
      transform: [
        {
          scale: withTiming(isVisible ? 1 : 0.6),
        },
      ],
    };
  }, [viewableItems]);

  return (
    <Animated.View
      style={[
        {
          height: 120,
          width: '96%',
          backgroundColor: bgColor,
          alignSelf: 'center',
          borderRadius: 15,
          marginTop: 20,
        },
        rStyle,
      ]}
    >
      <Link
        href={{
          pathname: '/feed/[id]',
          params: {
            id: item.id,
            title: item.title,
            description: item.description,
            date: item.date,
            author: item.author,
            imageUrl: item.imageUrl,
          },
        }}
      >
        <View className="flex-1 flex flex-col justify-between px-3 py-2">
          <View>
            <Text variant="title2">{item.title}</Text>
            <Text variant="callout">{item.description}</Text>
          </View>
          <View className="flex flex-row w-full justify-between">
            <Text variant="footnote" className="italic">
              {item.date}
            </Text>
            <Text variant="footnote" className="font-bold">
              {item.author}
            </Text>
          </View>
        </View>
      </Link>
    </Animated.View>
  );
});

/**Dummy Data generated by ChatGPT */
const data: ItemType[] = [
  {
    id: 1,
    title: 'Exploring the Mountains',
    description: 'A journey through the rocky mountains.',
    date: '2024-07-01',
    author: 'John Doe',
    imageUrl: 'https://picsum.photos/seed/696/3000/2000',
  },
  {
    id: 2,
    title: 'City Life Adventures',
    description: 'Discovering hidden gems in the city.',
    date: '2024-07-02',
    author: 'Jane Smith',
    imageUrl: 'https://picsum.photos/seed/696/3000/2000',
  },
  {
    id: 3,
    title: 'Beachside Relaxation',
    description: 'The best beaches to unwind.',
    date: '2024-07-03',
    author: 'Emily White',
    imageUrl: 'https://picsum.photos/seed/696/3000/2000',
  },
  {
    id: 4,
    title: 'Culinary Delights',
    description: 'Exploring exotic cuisines around the world.',
    date: '2024-07-04',
    author: 'Michael Brown',
    imageUrl: 'https://picsum.photos/seed/696/3000/2000',
  },
  {
    id: 5,
    title: 'Tech Innovations',
    description: 'The latest trends in technology.',
    date: '2024-07-05',
    author: 'Sophia Green',
    imageUrl: 'https://picsum.photos/seed/696/3000/2000',
  },
  {
    id: 6,
    title: 'Art and Culture',
    description: 'A dive into the world of art and culture.',
    date: '2024-07-06',
    author: 'David Johnson',
    imageUrl: 'https://picsum.photos/seed/696/3000/2000',
  },
  {
    id: 7,
    title: 'Wildlife Safari',
    description: 'An adventure into the wild.',
    date: '2024-07-07',
    author: 'Emma Davis',
    imageUrl: 'https://picsum.photos/seed/696/3000/2000',
  },
  {
    id: 8,
    title: 'Historical Sites',
    description: 'Exploring ancient historical sites.',
    date: '2024-07-08',
    author: 'James Wilson',
    imageUrl: 'https://picsum.photos/seed/696/3000/2000',
  },
  {
    id: 9,
    title: 'Fitness and Health',
    description: 'Tips for a healthier lifestyle.',
    date: '2024-07-09',
    author: 'Olivia Martinez',
    imageUrl: 'https://picsum.photos/seed/696/3000/2000',
  },
  {
    id: 10,
    title: 'Travel Tips',
    description: 'Essential tips for travelers.',
    date: '2024-07-10',
    author: 'Chris Lee',
    imageUrl: 'https://picsum.photos/seed/696/3000/2000',
  },
];
