# ポインタと参照

## ポインタ

ポインタというのは他の変数を参照する仕組みの1つです。
Windows のショートカットや Linux のシンボリックリンクと同じような機能です。

```cpp
int x = 5;
int* p = &x;
```

`int*` と書くと `int` 型の変数を指すポインタ型となります。
`&x` というのは `x` を指すポインタを得るための演算です。

## nullptr

どの変数も参照していないポインタのことをヌルポインタと呼びます。C++ではヌルポインタを表すキーワードとして `nullptr` が用意されています。

```cpp
int* p = nullptr;
```

!!! warning "NULL マクロ"
    C++11 より前はヌルポインタを表すのに `nullptr` というキーワードではなくC言語から受け継いだ `NULL` を使用するのが一般的でした。
    C言語の処理系では、 `void*` 型の変数は任意のポインタ型に暗黙変換される仕様があるため、 `NULL` を `(void*)0` などと定義している例が見られました。
    C++ではそのような暗黙変換は許されていないため、 `NULL` は単に `0` と定義される例が見られました。
    しかし `0` というリテラルは文脈によってヌルポインタと解釈されたり `int` 型と解釈されたりするため不都合が生じることもありました。

    C++11 ではこの問題を解決するために `nullptr` が導入されました。
    そのため C++11 以降では `NULL` ではなく `nullptr` を使うようにしましょう。

    詳細は [nullptr - cpprefjp C++日本語リファレンス][cpprefjp_nullptr] を参照してください。

[cpprefjp_nullptr]: https://cpprefjp.github.io/lang/cpp11/nullptr.html

## デリファレンス（逆参照）

ポインタが指す変数の値を参照するには次のように書きます。

```cpp
int x = 5;
int* p = &x;
int y = *p;
```

変数からポインタを得るときには `&` を付け、ポインタから値を得るには `*` を付けます。

## constポインタ

ポインタ型に `const` を付ける場合、 `const` の付ける位置によって意味合いが変わります。

`*` の前に `const` を付けると、ポインタが指し示す値を書き換えることができなくなります。

```cpp
int x = 123;
const int* p = &x;
*p = 456; // エラー
```

`*` の後に `const` を付けると、ポインタ自身を書き換えることができなくなります。

```cpp
int x = 123;
int* const p = &x;
p = nullptr; // エラー
```

併用することも可能です。

```cpp
const int* const p = &x;
```

## 参照

ポインタに似た仕組みとして参照があります。

```cpp
int x = 100;
int* p = &x; // pはポインタ型
int& r = x; // rは参照型

*p = 200;
r = 200;
```

`int&` と書くと `int` の参照型となります。

ポインタが指し示す値を扱うには `*` を付けてデリファレンスする必要がありますが、
参照では何も付与する必要がないため、見た目上では値型のように使うことが出来ます。
