---
title: rem vs em - Everything you need to know
description: We'll look at everything em and rem, their differences, when and how to use them, and practical examples of em and rem in action
slug: rem-vs-em
authors: fimber_elemuwa
tags: [css]
image: https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-12-21-em-vs-rem/social-2.png
hide_table_of_contents: false
---

**This article was last updated on November 28, 2024, to add sections for clear introduction to rem vs em.**

## Introduction

CSS is a crucial part of any website’s design, but understanding the nuances of how to use it can be tricky. One of the most important things to understand is the difference between using rem and em in CSS and why/when you should be using either.

Understanding the differences between these two units is crucial for web developers who want to create flexible and responsive web pages that are easy to maintain and modify. It also helps your website meet current web accessibility guidelines.

### What is em and rem in CSS?

In CSS, `em` and `rem` represent relative units that display the scalable and responsive results.

- **`em`:** It is relative to the font size of the parent. For example, if a parent font size is 20px, then `2em` will equal 40px.
- **`:rem`:** It's relative to root element font size, usually `<html>`. For example, if root font size is 16px, then `1rem` equals 16px.

Both units scale well and are suitable for accessibility and responsive design: use `rem` when you want to look for consistency across a project, and `em` to act on modules.

In this article, we'll look at everything em and rem, their differences, when and how to use them, and practical examples of em and rem in action. At the end of this article, you should have a solid understanding of both values. Without further ado, let’s jump right into it.

Steps we'll cover in this article:

- [em and rem units in CSS](#em-and-rem-units-in-css)
- [What is CSS em](#what-is-css-em)
- [What is CSS rem](#what-is-css-rem)
- [Differences between em and rem units](#differences-between-em-and-rem-units)
- [When to use em and rem units in CSS](#when-to-use-em-and-rem-units-in-css)

## Prerequisites

This article expects the reader to have a solid grasp of CSS. While you don’t have to be a CSS god(Hi Kevin Powell), it’d help if basic CSS terms do not confuse you.

## em and rem units in CSS

When looking for ways of specifying lengths in CSS, we’re quite spoiled for choices. All units for specifying lengths in CSS fall under two categories.

Absolute lengths: Absolute lengths, as the name suggests, are absolute; they’re fixed and don’t react relative to anything. This means no matter what happens, they’ll be the same size. Absolute lengths include cm, mm, in, px, pts, and pc.
Relative lengths: Relative lengths are units that specify a length relative to another unit, i.e., they respond based on other specified units or elements. They include %, vmax, vmin, vh, vw, ch, ex, and the units we’ll be talking about, em and rem.

If you’d like to know more about these units, you can do so here. Now let’s look at the two units we came here for.

## What is CSS em

Like I said above, em units in CSS are a relative unit of measurement used to size elements on a web page, mostly font size. Because it’s relative to its parent element, 1 em is equal to the font-size set in the parent element.

This means if you set the font-size in a parent div to 20px and set the font-size of the child div to 2 em, the font-size in the child div will equal 40px. Here’s an example.

First, let’s write the HTML

```tsx
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Document</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
   <div class="parent">
    I'm parent div set to 20px
    <div class="child">
        I'm the child div set to 2em, i.e 40px.
    </div>
   </div>
</body>
</html>

Next, the CSS.

.parent{
    font-size: 20px;
}

.child{
    font-size: 2em;
}

p {
    font-size: 1.5em;
 }
```

That will give us this.

<img src="https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-12-21-em-vs-rem/em-vs-rem-1.png"  alt="em-vs-rem" />

<br />

The em unit is useful because it allows you to adjust the size of elements on a page based on the font-size of a previously stated element, which helps create a consistent visual hierarchy. This can be useful for creating accessible websites that are easy to read for users with visual impairments.

It’s important to note that if you don’t specify the value of a parent element, the default value of the browser is taken as the parent element.

```tsx
p {
    font-size: 1.5em;
 }
```

In this example, the font-size property is set to 1.5em, meaning that the text size in the `<p>` element will be 1.5 times the size of the browser's default font-size if there’s no direct parent element.

Since most browsers scale their default font-size according to the screen size, this allows you to create flexible and responsive layouts that can adapt to different screen and font sizes.

Using the appropriate CSS properties, em units can also be used to set the size of other elements, such as margins, padding, and borders.

## What is CSS rem

Now that we know what em is, let’s look at rem. rem is another unit of measuring length in CSS, which stands for "root em". Since we know that an em is equal to the point size of the current font, we can deduce that the "root em" refers to the font size of the root element, which is usually the `<html>` element.

Confused? Let’s break it down more.

Like em, rem inherits its size from a parent element, but the parent element that rem looks at is not the div or section above it, but the very first element that surrounds it, which is the html element. Let’s make an example using the previous code. Same html code, with just an extra div.

```tsx
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Document</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
   <div class="parent">
    I'm parent div set to 20px
    <div class="child">
        I'm the child div set to 2em, i.e 40px.
    </div>
    <div class="child-2">
        I'm the child div set to 2em, i.e 60px.
    </div>
   </div>
</body>
</html>
```

Next, we’ll add the following CSS code to our CSS file.

```tsx
html{
    font-size: 30px;
 }

 .child-2{
    font-size: 2rem;
 }
```

The result would be this.

<img src="https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-12-21-em-vs-rem/em-vs-rem-2.png"  alt="em-vs-rem" />

<br />

As you can see, despite the child-2 div being inside another divs, it goes back all the way to the html element to inherit its font size.

Using the "rem" unit allows for a more scalable and flexible way to size elements on a page because if you change the font-size of the root element, all elements sized with the "rem" unit will be updated automatically to maintain their relative size.

## Advanced Usage of em and rem

I thought it would be useful to dive deeper into some advanced techniques for using em and rem units in our CSS. These units offer a lot of flexibility, and we can leverage them to create more dynamic and responsive designs.

Here are a few advanced strategies for using em and rem units:

### Compound Scaling with em

We can use em units to create compound scaling effects within nested elements. For instance, if we set a parent element to have a font size of 2em, and a child element to have a font size of 1.5em, the child’s font size will be 1.5 times the parent’s size, creating a scalable hierarchy.

```css
.parent {
  font-size: 2em;
}

.child {
  font-size: 1.5em; /* This will be 1.5 * 2em = 3em */
}
```

### Responsive Typography with rem

Using rem units for typography ensures that all text scales relative to the root font size. By adjusting the root font size with media queries, we can easily create responsive typography.

```css
html {
  font-size: 16px; /* Default root font size */
}

@media (max-width: 600px) {
  html {
    font-size: 14px; /* Smaller font size for small screens */
  }
}

body {
  font-size: 1rem; /* 1rem = 16px by default, 14px on small screens */
}
```

### Padding and Margins with rem

We can use rem units for padding and margins to maintain consistent spacing across different screen sizes. This approach ensures that spacing scales appropriately when the root font size changes.

```css
.container {
  padding: 2rem; /* Padding will scale with the root font size */
  margin: 1rem auto;
}
```

### Combining em and rem

In some cases, combining em and rem can offer the best of both worlds. For example, we can use rem units for global settings and em units for local, relative adjustments within components.

```css
.global {
  font-size: 1rem; /* Relative to root */
}

.component {
  font-size: 1.5em; /* Relative to its parent, .global in this case */
}
```

### Creating Modular Scales
We can use em or rem units to create modular scales, which are predefined ratios for scaling text and other elements. This method provides a consistent visual rhythm across the site.

```css
html {
  font-size: 16px; /* Base font size */
}

.h1 {
  font-size: 3rem; /* 48px */
}

.h2 {
  font-size: 2.25rem; /* 36px */
}

.h3 {
  font-size: 1.5rem; /* 24px */
}
```

By using these advanced techniques, we can create more adaptable and visually consistent designs.

## Differences between em and rem units

By now, you already know the difference between em and rem, but just for the sake of clarity, I’d like to restate the difference between both values.

In CSS, the rem unit is only relative to the document's root element, while the em unit is only relative to the immediate parent of the targeted element. This means that em sizes are inherited from parent elements, while rem sizes are inherited only from the root element.

## When to use em and rem units in CSS

It is a good idea to use the rem unit for global values such as font-sizes, margins, and padding, especially if you want to specify a font-size for the entire document and have it scale uniformly rather than being influenced by the font sizes of parent elements.

em is more suited for values that are specific to a particular element and its children. This allows you to create a consistent and flexible layout that adjusts well to different screen sizes and font sizes.
Potential problems with using em and rem units in CSS
em and rem are by far the best units to use today when specifying length, but like all things in life, they’re not perfect. These are a couple of problems you may run into when using em and rem:

- **Complex calculations:** Using em and rem units can lead to complex calculations, especially when nested elements are involved. This can make it difficult to accurately predict and control the size of elements on a page.
- **Inheritance issues:** Because em units are relative to the font size of their parent element, it can be difficult to understand and control how sizes are inherited across the page. This can lead to unexpected results and require additional debugging to resolve.
- **Performance issues:** In some very rare cases, using em and rem units can have a negative impact on performance, especially when combined with complex calculations or used excessively on a page.

Overall, while em and rem units can be helpful in certain situations, it's important to consider their potential drawbacks carefully and whether they are the best choice for your project.

## Accessibility Considerations

Let me share some thoughts on the ways in which we could make our web designs much more accessible using em and rem units in CSS. Things that really make a difference, for instance, are how the content would be perceived or accessed by a user with disabilities.

You can increase the accessibility of our web pages using em and rem units. Here's why:

### Scalable Font Sizes

Because I used either `em` or `rem` for the font sizes, then text on the webpage gets scalable to respond to user needs. This is beneficial to people with visual impairments who may desire a larger text size for comfort in reading. Browsers can scale these units more easily than fixed units.

### Consistent Spacing

By using rem for margins, padding, and other spacing properties, it's certain everything scales consistently. This could improve the predictability of the layout structure, which can help users with cognitive disabilities.

### Responsive Design

As already explained, both 'em' and 'rem' units are relative units, so designing a responsible layout becomes a lot easier. The layout then dynamically adapts according to screen size because the sizes of screens change, which is paramount as users are from diverse devices and can use assistive technologies.

### User Preferences

Some users set up their browser to have a different font size from the default to increase readability. By designing with `rem` units, user styles get respected, and our content stays accessible.

### Reduced Motion and Distraction

Em and rem units make it more in control to resize and reposition the elements, reducing unnecessary movement and distractions visually—a very challenging experience for users with attention-related disabilities. In short, we use em and rem units in our CSS to make the design more accessible and user-friendly. It is such a small but powerful way of using inclusive and adaptive design to cater to a wider scope of user needs.

## Implications on Performance

I just want to share with you some insights on what impacts performance when using em or rem units within CSS. Understanding them can serve to make better decisions for the design of our stylesheets to ensure the applications run smoothly.

### Reflow and Repaint

When we use em units, the browser needs to calculate sizes relative to their parent elements. That can lead to reflows and repaints when structures are deeply nested. This is one reason using rem units can make for less complexity; in this case, all calculations are only relative to the root element.

### Maintaining Consistency

The application will use rem units consistently. This is really helpful for responsive design. Changing the root font size will scale all other elements that use rem units in a proportional manner. It could lead to an easier rendering process within the browser.

### Memory Usage

Em units might mean more memory usage: Since each element with an em unit needs to remember the computed value based on its parent, rem units based on the root may keep memory usage small and predictable.

### CSS Calculations

The browser computes the size and position of elements used by CSS calculations. This makes em units slightly heavy computationally, as they have to reference parents in any given case. Rem units in some calculations have only one reference point; therefore, it can be much simpler, making it computationally faster.

### Layout Stability

Units in rem can maintain layout stability between multiple screen sizes and resolutions. Because the base font size inside the root element is set, and the unit for element sizes is rem, all other elements will be resized proportionally; as a result, layout shifting will be lesser, and user experience will increase.

### Readability and Maintenance

Using rem units will allow you to have more readability and maintainability in your stylesheet. Defining this base value globally makes the size of elements at a glance predictable and visible in your stylesheets. This could cut down on hours spent debugging layout concerns and actually make development a whole lot more productive.

For example, let's consider this quick example to draw out the points above:

```css
html {
  font-size: 16px; /* Base font size */
}

.container {
  padding: 2rem; /* Easier for the browser to calculate */
  margin: 1rem auto;
}

.child {
  font-size: 1.5em; /* Less work for the browser to calculate */
}
```

## Conclusion

If you made it here, congratulations! You now know all there is to know about em and rem and why we need them. Though they’re both similar, they’re distinctly different and should be treated accordingly.

Personally, I prefer using rem over em for components, but I usually use em for headers and text elements. No real reason, just my quirk. I hope you found this article helpful, and I hope it becomes your cheatsheet whenever you get confused about em and rem.
