---
title: Theming
---

Ant Design allows you to customize design tokens to satisfy UI diversity from business or brand requirements, including primary color, border radius, border color, etc.
Design Tokens are the smallest element that affects the theme. By modifying the Design Token, we can present various themes or components.

> For more information, refer to the [Ant Design documentation about theme customization &#8594](https://ant.design/docs/react/customize-theme)

## Predefined Themes

[`RefineThemes`](https://github.com/refinedev/refine/blob/main/packages/antd/src/definitions/themes/index.ts) has predefined themes that you can use by importing them from `@refinedev/antd` package.

```ts
const { Blue, Purple, Magenta, Red, Orange, Yellow } = RefineThemes;
```

```tsx
import { Refine } from "@refinedev/core";
import { ThemedLayoutV2, RefineThemes } from "@refinedev/antd";

import { ConfigProvider } from "antd";

const App: React.FC = () => {
  return (
    <ConfigProvider theme={RefineThemes.Blue}>
      <Refine
      /* ... */
      >
        <ThemedLayoutV2>{/* ... */}</ThemedLayoutV2>
      </Refine>
    </ConfigProvider>
  );
};
```

If you want to see how themes change the look of the application, check out this [example &#8594](/docs/examples/themes/refine-themes-antd/)

## Theme customization

You can use either the [`<ConfigProvider/>`](https://ant.design/components/config-provider/#components-config-provider-demo-theme) component or the `RefineThemes` provided by Refine to change the theme. This is not required if you decide to use the default theme.

### Overriding the themes

You can not only override or extend the default themes, but also create your own, just like this:

```tsx
import { Refine } from "@refinedev/core";
import { ThemedLayoutV2 } from "@refinedev/antd";

import { ConfigProvider } from "antd";

const API_URL = "https://api.fake-rest.refine.dev";

const App: React.FC = () => {
  return (
    // highlight-start
    <ConfigProvider
      theme={{
        components: {
          Button: {
            borderRadius: 0,
          },
          Typography: {
            colorTextHeading: "#1890ff",
          },
        },
        token: {
          colorPrimary: "#f0f",
        },
      }}
    >
      {/* highlight-end */}
      <Refine
      /* ... */
      >
        <ThemedLayoutV2>{/* ... */}</ThemedLayoutV2>
      </Refine>
      // highlight-next-line
    </ConfigProvider>
  );
};
```

### Use Preset Algorithms

Themes with different styles can be quickly generated by modifying the algorithm. Ant Design 5.0 provides three sets of [preset algorithms by default](https://ant.design/docs/react/customize-theme#theme-presets): the default algorithm `theme.defaultAlgorithm`, the dark algorithm `theme.darkAlgorithm` and the compact algorithm `theme.compactAlgorithm`.

You can switch between algorithms by modifying the algorithm property of theme in [`<ConfigProvider/>`](https://ant.design/components/config-provider/#components-config-provider-demo-theme).

> For more information, refer to the [Ant Design documentation on customizing themes&#8594](https://ant.design/docs/react/customize-theme#use-preset-algorithms)

#### Switching to dark theme

Let's start with adding a switch to the `Header` component:

```tsx
import { Space, Button } from "antd";

interface HeaderProps {
  theme: "light" | "dark";
  setTheme: (theme: "light" | "dark") => void;
}

const Header: FC<HeaderProps> = (props) => {
  return (
    <Space
      direction="vertical"
      align="end"
      style={{
        padding: "1rem",
      }}
    >
      <Button
        onClick={() => {
          props.setTheme(props.theme === "light" ? "dark" : "light");
        }}
        icon={props.theme === "light" ? <IconMoonStars /> : <IconSun />}
      />
    </Space>
  );
};
```

Then, we can use the `theme` property of the `ConfigProvider` component to switch between light and dark themes:

```tsx
import { Refine } from "@refinedev/core";
import { ThemedLayoutV2 } from "@refinedev/antd";
import { ConfigProvider, theme } from "antd";

import { Header } from "./Header";

const App: React.FC = () => {
  // highlight-next-line
  const [currentTheme, setCurrentTheme] = useState<"light" | "dark">("dark");

  return (
    <ConfigProvider
      // highlight-start
      theme={{
        algorithm:
          currentTheme === "light"
            ? theme.defaultAlgorithm
            : theme.darkAlgorithm,
      }}
      // highlight-end
    >
      <Refine
      /* ... */
      >
        <ThemedLayoutV2 Header={Header}>{/* ... */}</ThemedLayoutV2>
      </Refine>
    </ConfigProvider>
  );
};
```

### `useNotificationProvider` compatible with Theme

The `notificationProvider` export has been deprecated because it cannot consume the current `theme` context by default. To enable the `Notification` component to consume the current `theme` context, import the `App` component from `antd` and the `useNotificationProvider` export from `@refinedev/antd` and pass it as highlighted below:

```tsx
import { Refine } from "@refinedev/core";
import { ThemedLayoutV2, useNotificationProvider } from "@refinedev/antd";

import { ConfigProvider, App as AntdApp } from "antd";

const API_URL = "https://api.fake-rest.refine.dev";

const App: React.FC = () => {
    return (
        <ConfigProvider theme={RefineThemes.Blue}>
            {/* highlight-next-line */}
            <AntdApp>
                <Refine
                    //...
                    {/* highlight-next-line */}
                    notificationProvider={useNotificationProvider}
                >
                    <ThemedLayoutV2>{/* ... */}</ThemedLayoutV2>
                </Refine>
                {/* highlight-next-line */}
            </AntdApp>
        </ConfigProvider>
    );
};
```

If you want to customize the default layout elements provided with `@refinedev/antd` package, check out the [Custom Layout](/docs/advanced-tutorials/custom-layout) tutorial.

## Example

<CodeSandboxExample path="customization-theme-antd" />
