---
title: Notifications
---

One of the most important parts of an application is the notifications and the visual feedbacks. Refine has this built-in notification integration that works automatically when it's needed in cases such as when a request fails or when a form is submitted.

While this integration is not coupled with the UI integrations, it will be a wise choice to use the one that is provided by the UI libraries for a consistent design language. This is why Refine's UI integrations also provides a [`notificationProvider`](/docs/notification/notification-provider/) to be used with the notification integration of refine.

## Notification Providers

Refine let's you set a notification API by providing the `notificationProvider` property to the `<Refine />` component. `notificationProvider` is an object with close and open methods. Refine uses these methods to show and hide notifications. These methods can be called from anywhere in the application with `useNotification` hook.

An `notificationProvider` must include `open` and `close` methods with the following types:

```ts
interface NotificationProvider {
  open: (params: OpenNotificationParams) => void;
  close: (key: string) => void;
}

interface OpenNotificationParams {
  key?: string;
  message: string;
  type: "success" | "error" | "progress";
  description?: string;
  cancelMutation?: () => void;
  undoableTimeout?: number;
}
```

Once you provide the notification provider, Refine seamlessly integrate with [data hooks](/docs/guides-concepts/data-fetching/#data-hooks) to displays user-friendly notifications for various data-related actions, ensuring a clear and informative user experience. This includes:

- **Form Submission**: Whether a [form](/docs/data/hooks/use-form/) is successfully submitted or encounters errors, Refine will display the appropriate notification to keep the user informed.
- **Resource Management**: [Creation](/docs/data/hooks/use-create/), [deletion](/docs/data/hooks/use-delete/), [update](/docs/data/hooks/use-update/), [import](/docs/core/hooks/utilities/use-import/), and [export](/docs/core/hooks/utilities/use-export/) of resources are all accompanied by success or error notifications, providing immediate feedback to the user.
- **Data Fetching**: Refine also displays notifications for failed data fetching operations, including those using [useList](/docs/data/hooks/use-list/), [useInfiniteList](/docs/data/hooks/use-infinite-list/), [useMany](/docs/data/hooks/use-many/), [useOne](/docs/data/hooks/use-one/).
- Auth Actions: [Login](/docs/authentication/hooks/use-login/), [logout](/docs/authentication/hooks/use-logout/), [register](/docs/authentication/hooks/use-register/), [update password](/docs/authentication/hooks/use-update-password/), and [forgot password](/docs/authentication/hooks/use-forgot-password/) actions are all integrated with Refine's notification provider to display error notifications.

### Built-in Notification Providers

Using of the prebuilt notification providers are optional and can be customized, extended or even swapped with a custom implementation if needed.

As an example, we'll demonstrate how to open and close notifications using the `useNotification` hook. However, in most cases, you won't need to do this, as Refine typically manages notifications for you automatically.

<Tabs wrapContent={false}>

<TabItem default value="antd" label="Ant Design">

import NotificationAntd from "./notifications-antd";

<NotificationAntd />

</TabItem>

<TabItem value="material-ui" label="Material UI">

import NotificationMui from "./notifications-mui";

<NotificationMui />

</TabItem>

<TabItem value="mantine" label="Mantine">

import NotificationMantine from "./notifications-mantine";

<NotificationMantine />

</TabItem>

<TabItem value="chakra-ui" label="Chakra UI">

import NotificationChakraUI from "./notifications-chakra-ui";

<NotificationChakraUI />

</TabItem>
</Tabs>

### Undoable

Refine also supports undoable notification.

You can trigger an undoable notification by setting the `type` to `progress`. After timeout, the notification will be closed automatically. If the user clicks the undo button, the `cancelMutation` callback will be called.

```tsx
const { open } = useNotification();

open?.({
  type: "progress",
  message: "Progress",
  undoableTimeout: 5,
  cancelMutation: () => {
    // when undo button is clicked, run this callback
  },
});
```

Mutation hooks such as `useUpdate`, `useDelete` and `useForm` supports undoable notifications. It can be used for canceling the mutation.

```ts
import { useForm } from "@refinedev/core";

// automatically cancel the mutation when undo button is clicked
useForm({ mutationMode: "undoable" });
```

## Customizing Notifications

### With props

All data hooks have a `successNotification` and `errorNotification` prop that can be used to customize the notification that will be shown when the hook is called.

We will look `useUpdate` and `useForm` hooks as an example but all data hooks have the same props and they work the same way.

<Tabs>

<TabItem default value="useUpdate" label="useUpdate">

```tsx
import { useUpdate } from "@refinedev/core";

const { mutate } = useUpdate();

mutate({
  // it will be called when the mutation is successful
  // By setting it to `false`, you can disable the notification.
  successNotification: (data, values, resource) => {
    return {
      message: `${data.title} Successfully fetched.`,
      description: "Success with no errors",
      type: "success",
    };
  },
  // it will be called when the mutation is failed
  errorNotification: (data, values, resource) => {
    return {
      message: `Something went wrong when getting ${data.id}`,
      description: "Error",
      type: "error",
    };
  },
});
```

</TabItem>

<TabItem value="useForm" label="useForm">

```tsx
import { useForm } from "@refinedev/core";

useForm({
  //  it will be called when the form is submitted successfully
  // By setting it to `false`, you can disable the notification.
  successNotification: (data, values, resource) => {
    return {
      message: `Successfully created ${data.title}`,
      description: "good job!",
      type: "success",
    };
  },
  // it will be called when the form is submitted with errors
  // By setting it to `false`, you can disable the notification.
  errorNotification: (error, values, resource) => {
    return {
      message: `Failed to create ${values.title}`,
      description: error.message,
      type: "error",
    };
  },
});
```

</TabItem>

</Tabs>

### With i18n <GuideBadge id="i18n/i18n-provider/" />

Refine's notification integration is also integrated with the [`i18n Provider`](/docs/i18n/i18n-provider/). This means that you can use the `i18n` integration to customize the notifications.

Refine uses following keys for the notifications and popultes `{{resource}}` and `{{statusCode}}`. You can override these keys in your `i18n` provider to customize the notifications.

```json title="/locales/en/common.json"
{
  "notifications": {
    "success": "Successful",
    "error": "Error (status code: {{statusCode}})",
    "undoable": "You have {{seconds}} seconds to undo",
    "createSuccess": "Successfully created {{resource}}",
    "createError": "There was an error creating {{resource}} (status code: {{statusCode}})",
    "deleteSuccess": "Successfully deleted {{resource}}",
    "deleteError": "Error when deleting {{resource}} (status code: {{statusCode}})",
    "editSuccess": "Successfully edited {{resource}}",
    "editError": "Error when editing {{resource}} (status code: {{statusCode}})",
    "importProgress": "Importing: {{processed}}/{{total}}"
  }
}
```
