---
title: 4 Ways to Generate UUIDs in Node.js
description: We'll explore how UUIDs work in Node.js and how they can help keep data distinct and make your projects easier to scale.
slug: node-js-uuid
authors: necati
tags: [dev-tools]
image: https://refine.ams3.cdn.digitaloceanspaces.com/blog/2024-04-24-node-uuid/social.png
hide_table_of_contents: false
---

**This article was last updated on November 7, 2024 to include performance considerations and advanced use cases for UUIDs in distributed systems and microservices.**

## Introduction

In software development, it's important to make sure that each piece of data is unique to prevent overlaps and errors. Universally Unique Identifiers, or UUIDs, help solve this problem by giving each item a unique label. They’re becoming crucial as more systems and services link over various networks. They make sure every piece of data remains unique, preventing any overlaps and errors.

In this blog post, we'll look at how UUIDs work in Node.js and how they can help keep data distinct and make your projects easier to scale.

Steps we'll cover:

- [What is a UUID?](#what-is-a-uuid)
- [Why Use UUIDs in Your Node.js Projects?](#why-use-uuids-in-your-nodejs-projects)
- [How to Generate UUID'S?](#how-to-generate-uuids)
- [Understanding and Choosing the Right UUID Versions](#understanding-and-choosing-the-right-uuid-versions)
- [UUIDs in Distributed Systems and Microservices](#uuids-in-distributed-systems-and-microservices)

## What is a UUID?

A UUID is a 128-bit number used to uniquely label information in computer systems, as defined in RFC 4122. There are several versions, but the ones we often use are UUIDv1 and UUIDv4. UUIDv1 generates IDs based on the time they're created and includes this timestamp. It’s useful because you can trace when and where it was generated, which can help in some debugging scenarios.

On the other hand, UUIDv4 is totally random, providing a much higher level of security due to its unpredictability—ideal for sensitive applications where privacy is a concern.

Each type serves distinct purposes, catering to the specific needs of data identification and security in software development.

## Why Use UUIDs in Your Node.js Projects?

Using UUIDs in Node.js enhances data uniqueness and integrity, making it ideal for distributed systems and preventing ID collisions. It facilitates data merging from different sources due to the guaranteed uniqueness of each identifier. Additionally, UUIDs increase security by making it difficult to predict IDs, protecting against unauthorized data access. They also allow for more flexible database schema design, enabling agile development and easier database migrations by handling IDs at the application level.

Integrating UUIDs in our Node.js applications helps us ensure that every data entry is unique—this is vital for systems where multiple instances or databases need to merge or sync without conflicts. It also boosts our system's security, as the randomness of UUIDv4 makes it tough for anyone to guess the ID sequences, protecting against some forms of cyber attacks. Plus, using UUIDs lets us be more agile with our database schema, since IDs are handled at the application level, making it easier to evolve our database without messy migrations.

## How to Generate UUID'S?

To generate UUIDs in a Node.js environment, you can use different methods and packages depending on your specific needs and preferences. Below, I’ll guide you through three different approaches: using the built-in Node.js [`crypto`](https://nodejs.org/docs/latest-v14.x/api/crypto.html#crypto_crypto_randomuuid_options) module, the popular [`uuid`](https://github.com/uuidjs/uuid) npm package, and the [`nanoid`](https://github.com/ai/nanoid) npm package.

### 1. Using Node.js Crypto Module

Node.js includes a built-in module called `crypto` that can be used to generate UUIDs, specifically UUID v4, which are random:

```javascript
const crypto = require("crypto");

let uuid = crypto.randomUUID();
```

This function utilizes the `crypto` module's `randomUUID` method to generate random bytes, which are then formatted into a UUID v4 compliant formatr.

When you log, you can get similar like the following:

```
00a1dv45-dx19-2301-2471-223932594567
```

### 2. UUID npm Package

The `uuid` npm package is a very popular choice for generating UUIDs in Node.js applications. It supports multiple versions of UUIDs:

#### Installation

```bash
npm install uuid
```

#### Usage

```javascript
const { v4: uuidv4 } = require("uuid");

// Generate a UUID v4
const uuid = uuidv4();
console.log(`UUID: ${uuid}`);
// ⇨ '1b9d6bcd-bbfd-4b2d-9b5d-ab8dfbbd4bed'
```

This package provides simple, straightforward methods for generating UUIDs and supports UUID versions 1, 3, 4, and 5.

#### API table

|                                                    |                                              |                   |
| -------------------------------------------------- | -------------------------------------------- | ----------------- |
| [`uuid.NIL`](#uuidnil)                             | The nil UUID string (all zeros)              | New in `uuid@8.3` |
| [`uuid.parse()`](#uuidparsestr)                    | Convert UUID string to array of bytes        | New in `uuid@8.3` |
| [`uuid.stringify()`](#uuidstringifyarr-offset)     | Convert array of bytes to UUID string        | New in `uuid@8.3` |
| [`uuid.v1()`](#uuidv1options-buffer-offset)        | Create a version 1 (timestamp) UUID          |                   |
| [`uuid.v3()`](#uuidv3name-namespace-buffer-offset) | Create a version 3 (namespace w/ MD5) UUID   |                   |
| [`uuid.v4()`](#uuidv4options-buffer-offset)        | Create a version 4 (random) UUID             |                   |
| [`uuid.v5()`](#uuidv5name-namespace-buffer-offset) | Create a version 5 (namespace w/ SHA-1) UUID |                   |
| [`uuid.validate()`](#uuidvalidatestr)              | Test a string to see if it is a valid UUID   | New in `uuid@8.3` |
| [`uuid.version()`](#uuidversionstr)                | Detect RFC version of a UUID                 | New in `uuid@8.3` |

### 3. Nano ID Package

Nano ID is a tiny, secure URL-friendly unique string ID generator for JavaScript, which can be used as an alternative to UUIDs. It offers a similar level of uniqueness and randomness and is a great choice when shorter IDs are needed:

#### Installation

```bash
npm install nanoid
```

#### Usage

```javascript
const { nanoid } = require("nanoid");

// Generate a Nano ID
const id = nanoid();
console.log(`Nano ID: ${id}`);
```

Nano ID generates URL-friendly IDs by default, which are shorter and more memory-efficient than UUIDs.

Each of these methods provides unique identifiers suitable for various applications, from managing database keys to tracking unique user sessions. Choose the method that best fits your project’s requirements.

## Understanding and Choosing the Right UUID Versions

What I wanted to do was to provide some views on the different UUID versions and in what cases to use each; in particular, considering how to make data identifiers unique and secure in Node.js. The following is a quick rundown of UUID versions, how they differ, and where each might be most useful.

### UUIDv1 - Time-based

UUIDv1 is generated based on the time it was created, and it includes a timestamp, along with the MAC address of the machine that created it. Use it where you may want traceability-while you can tell when and where the ID was generated, for example:

- **Use Case**: When doing any form of logging or event tracking and want to find an entry's origin.
- **Pros**: Timestamp informs about ID creation time and helps in debugging.
- **Cons**: Not truly random, it can leak information about the machine and network, which can be a security concern.

### UUIDv4 - Randomly Generated

UUIDv4 is generated purely by chance; there's no timestamp or any hardware-based information in the ID. Therefore, it is even safer and more unpredictable; its creation makes it ideal for applications where high security and privacy are required.

- **Relevant Use Case**: User IDs or other sensitive data with which we want to minimize predictability.
- **Pros**: Highly random in nature and thus ideal for privacy and security. Not very predictable, unlike UUIDv1.
- **Disadvantages**: Not traceable because there is no timestamp per se; hence, not useful for tracking purposes.

### UUIDv3 and UUIDv5 - Namespace-Based

Both UUIDv3 and UUIDv5 are generated from a name and a namespace. They differ in the hashing algorithm since UUIDv3 uses MD5, while in UUIDv5, a more secure SHA-1 is used.

- **Use Case**: Creation of unique IDs based on already available data; for example, generating IDs from usernames or email addresses.
- **Advantages**: Deterministic generation - same input gives the same UUID, hence useful for reference consistency.
- **Limitations**: The randomness is limited, as there can be nothing outside the dataset on which the model is trained. Not appropriate for cases like codes, identity numbers that require high uniqueness in different systems.

While UUIDv4 often best suits most general-purpose cases with its randomness and unpredictability, UUIDv1 is good to go if traceability is required, or UUIDv5, if determinism needs to be maintained.

### 4. Bonus: short-uuid

[short-uuid](https://github.com/oculus42/short-uuid) simplifies the generation and translation of UUIDs into shorter or alternative formats, and vice versa.

The latest version, 4.0.0, brings some significant changes like improved error handling, modern ECMAScript support, and consistent-length value generation. This means you can now easily handle UUIDs in a more efficient and error-free manner.

To get started, you just need to install the library and use its simple API to generate or translate UUIDs. It's particularly handy when you need shorter identifiers for your application, like in URLs or database keys.

```
const short = require('short-uuid');

// Quick start with flickrBase58 format
short.generate(); // Example: 73WakrfVbNJBaAmhQtEeDv
```

- `short-uuid` takes RFC4122 v4-compliant UUIDs and translates them into shorter formats.
- Version 4.0.0 ensures consistent-length values unless otherwise specified, achieved by padding with the first character in the alphabet.
- Translators enable conversion back and forth between RFC compliant UUIDs and shortened formats.

## UUIDs in Distributed Systems and Microservices

I thought it might be useful to touch on the role of UUIDs in distributed systems and microservices, where they really shine due to their unique properties. The following code examples demonstrate several use cases of UUIDs in distributed systems and microservices.

1. **Generating UUIDs Independently within Multiple Services**

Every microservice can generate UUIDs autonomously without depending on an integrated ID generator. Here is how you could generate UUIDs for a Node.js environment by using the `uuid` library:

```javascript
// Install uuid library if you haven't
// npm install uuid

const { v4: uuidv4 } = require("uuid");

// Every microservice shall create a unique UUID
const orderID = uuidv4();
console.log(`Generated Order ID: ${orderID}`);
```

This will generate a UUIDv4; those are random—so they don't include timestamps or any other traceable information—and therefore suitable for most use cases.

2. **UUIDv5 to Generate Deterministic Identifiers Based on Namespaces**

This would be the reason for using UUIDv5 if you needed to have consistency across services, say using the same UUID based off of a certain identifier such as an email address. This version includes methods for generating UUIDs based on a "name" and a "namespace," guaranteeing that the same name in the same namespace always returns the same UUID:

```javascript
const { v5: uuidv5 } = require("uuid");

// Define a namespace -- usually a fixed UUID for a project
const NAMESPACE = "6ba7b810-9dad-11d1-80b4-00c04fd430c8"; // Example namespace UUID

// Generate a UUID based on a unique name (like an email) within the namespace
const userID = uuidv5("user@example.com", NAMESPACE);
console.log(`Generated User ID(UUIDv5): ${userID}`);
```

Using UUIDv5 is useful in services that need to recreate the same UUID for a certain resource over different services.

3. **Using UUIDs for Database Replication in Microservices**

In a microservice context, the database might be distributed or synchronized across regions or environments. UUIDs can provide unique IDs across different databases. Following is an example of UUID insertion in a MongoDB collection:

```javascript
const { v4: uuidv4 } = require("uuid");
const MongoClient = require("mongodb").MongoClient;

async function insertOrder() {
  const client = new MongoClient("mongodb://localhost:27017", {
    useUnifiedTopology: true,
  });
  await client.connect();
  const db = client.db("orders_db");
  const orders = db.collection("orders");

  // Generate unique UUID for the order
  let orderID = uuidv4();
  const newOrder = { orderID, item: "Laptop", quantity: 1 };

  await orders.insertOne(newOrder);
  console.log(`Order inserted with ID: ${orderID}`);

  await client.close();
}

insertOrder().catch(console.error);
```

This helps with replicating or merging databases since UUIDs will be conflict-free and ensure the IDs are unique across replicas.

4. **UUID Storing as Binary for Performance**

Some databases, like MySQL, support storing UUIDs in binary format such as `BINARY(16)` over `CHAR(36)` to save storage size and increase performance. Here's how this might look for MySQL:

```sql
CREATE TABLE users (
    id BINARY(16) PRIMARY KEY,
    name VARCHAR(50)
);

-- Insert a UUID as binary
INSERT INTO users (id, name) VALUES (UUID_TO_BIN(UUID()), 'John Doe');

-- Lookup the UUID
SELECT BIN_TO_UUID(id) as id, name FROM users;
```

The result for this method will be a compact binary of UUID that is faster and more efficient for large scales of databases. These examples illustrate various ways in which UUID can be applied to distributed systems and microservices when it comes to unique, reliable, and scalable ID management.

## Conclusion

UUIDs are essential for ensuring data integrity and uniqueness in distributed systems. They provide a reliable way to label data and prevent ID collisions, making them ideal for applications that require secure and scalable data management. By integrating UUIDs into your Node.js projects, you can enhance your system's security, flexibility, and performance, ensuring that your data remains distinct and secure across various platforms and networks.
