---
title: How to Use JavaScript Array some
description: JavaScript some method implementation on Array with examples
slug: javascript-some-method
authors: abdullah_numan
tags: [javascript]
image: https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-10-25-js-some/social.png
hide_table_of_contents: false
---

**_This article was last updated on November 4, 2024 to include performance considerations and comparisons with other iteration methods for a better understanding of the JavaScript `some` method._**

## Introduction

This post is about the Array `some()` method in JavaScript. This is the second part of the series titled [JavaScript Iteration Methods Series](https://refine.dev/blog/javascript-slice/).

Refinedev JavaScript Iteration Methods Series is a tutorial series that publishes posts covering the usage of various iteration methods in JavaScript.

The JavaScript `Array.prototype.some` method is an iteration method that tests whether any one element in a collection satisfies a given condition. It is commonly used to find items that fulfill involved conditions such as matching the value of deeply nested object properties. However, it can also be used to check if a given value is included in an array. The test is defined via a callback function and passed as argument to the JS Array `some()` call.

In this post, we explore with examples what the **JavaScript some** method is and how it works. We consider what entails its callback test function, and the cases where it is used. We demonstrate examples of using JavaScript Array `some()` with and without the `thisArg` object. We also examine the impact of using the callback with arrow syntax on the `thisArg` object while using the JavaScript `some()` method. Towards the end, we see how to modify the caller array from inside the JavaScript Array `some()` method.

We'll be discussing things in quite depth, so let's start with the basics.

Steps we'll cover:

- [What is JavaScript some method?](#what-is-javascript-some-method)
  - [JavaScript `Array.prototype.some`: Details of the Callback Function](#javascript-arrayprototypesome-details-of-the-callback-function)
  - [How `Array.prototype.some()` Works?](#how-arrayprototypesome-works)
  - [How to Use JavaScript Array `some` Method?](#how-to-use-javascript-array-some-method)
- [`some()` with Large Arrays - Tips for Efficiency](#some-with-large-arrays---tips-for-efficiency)
  - [**Early Stop Saves Time**](#early-stop-saves-time)
  - [**Keeping the Callback Simple**](#keeping-the-callback-simple)
  - [**When You Need to Check Every Item**](#when-you-need-to-check-every-item)
- [When to Use JavaScript `some()`?](#when-to-use-javascript-some)
  - [Using JavaScript `Array.prototype.some` to Test Arrays of Nested Objects](#using-javascript-arrayprototypesome-to-test-arrays-of-nested-objects)
  - [JavaScript `some()` With `thisArg` Argument](#javascript-some-with-thisarg-argument)
  - [JavaScript `some(callback, thisArg)` Doesn't Work With Arrow Functions](#javascript-somecallback-thisarg-doesnt-work-with-arrow-functions)
  - [JS `some(callback, thisArg)` Works With Non-Arrow Functions](#js-somecallback-thisarg-works-with-non-arrow-functions)
- [Modifying the Caller Array](#modifying-the-caller-array)
- [Comparing `some()` with `find()` and `every()`](#comparing-some-with-find-and-every)
  - [**`some()`** – Checks if Any Item Matches](#some--checks-if-any-item-matches)
  - [**`find()`** – Finds First Matching Item](#find--finds-first-matching-item)
  - [**`every()`** – Checks if All Items Match](#every--checks-if-all-items-match)

## What is JavaScript some method?

`Array.prototype.some()` is a JavaScript iteration method used to check if an array contains at least one item that satisfies a test. The method is called on an array of items and the test is performed with a callback function and any necessary `thisArg` object passed to the execution context of the callback function:

```ts
// Method signature

some(callbackFn);
some(callbackFn, thisArg);
```

The first argument, `callbackFn`, is mandatory and have to be declared with the test logic. The second argument, `thisArg`, is optional.

### JavaScript `Array.prototype.some`: Details of the Callback Function

The JS `some()` method's callback function, `callbackFn` here, takes three arguments. The first is the element being traversed to, `element`, which is mandatory. The second argument is the current index, `index` and the third is `array`, the array being iterated:

```ts
const callbackFn = function(element, index, array){...}
```

Both the second and third arguments are optional. So, `some()` would have the following possible call signatures with the callback:

```ts
// Method call signatures

some(function(element){...});
some(function(element, index){...});
some(function(element, index, array){...});
some(function(element, index, array){...}, thisArg);
```

### How `Array.prototype.some()` Works?

**JavaScript some** tests whether there is one element that satisfies the test logic set in the callback function, `callbackFn`.

JS `some()` attempts to execute the callback function once for each item in the array. If it finds one that evaluates to a truthy value for `callbackFn`, it returns with the Boolean `true`. Otherwise, it seeks to traverse to the end of the array and returns `false` if all are falsy:

### How to Use JavaScript Array `some` Method?

To use the JavaScript `Array.prototype.some` method, can declare the test inside a callback function, so that we can pass it to `some()` at invocation. For example, as in the case below for checking if a number in the array is even or not:

```ts
const numbers = [1, 2, 3, 4, 5];

const even = (element) => element % 2 === 0;
const isThereEvenNumber = numbers.some(even);

console.log(isThereEvenNumber); // true
```

In the above chunk of code, `even` is our callback function where we apply the logic for testing if the item is an even number. We then pass `even` to `some()` which is invoked on `numbers` array. Apparently, we have at least one even number in `numbers` so `even` returns `true`. As a result, `some()` also returns `true`.

## `some()` with Large Arrays - Tips for Efficiency

I’ve put together a few tips and examples for using JavaScript’s some() method in a way that’s efficient with large arrays.

### **Early Stop Saves Time**

The real value of `some()` is that it can short-circuit at the first match and thereby save a lot of processing time with large arrays.

```javascript
const largeArray = Array.from({ length: 100000 }, (_, i) => i); // large array of 100,000 items

const hasNumberGreaterThan50000 = largeArray.some((num) => num > 50000);

console.log(hasNumberGreaterThan50000); // true, stopped at 50,001 and saved time
```

Here, `some()` short-circuits and bails early since it finds a match – it's considerably faster.

### **Keeping the Callback Simple**

Since `some()` calls the callback for each item until it finds a match, a simple callback keeps things efficient.

```javascript
const users = Array.from({ length: 100000 }, (_, i) => ({
  id: i,
  active: i === 99999,
}));

// Find the first active user
const hasActiveUser = users.some((user) => user.active);

console.log(hasActiveUser); // true, stops as soon as it finds the active user
```

Above, `some()` will stop on the active user when found using just a simple callback for performance.

### **When You Need to Check Every Item**

If you have to go through every item in an array for things like checking complex conditions across all items, sometimes `filter` or `reduce` works better.

```javascript
// Use `filter` when you need to check all items
const complexCondition = users.filter(
  (user) => user.id % 2 === 0 && user.id > 5000,
);

console.log(complexCondition.length); // Outputs all users who match the condition
```

For cases where you want every match, `filter` makes more sense than `some()`, as it’s designed to "collect" all items meeting a condition.

## When to Use JavaScript `some()`?

`Array.prototype.some` is used in situations that involves fairly complex tests to be performed on an item, such as the division operation in the `even` function above.

### Using JavaScript `Array.prototype.some` to Test Arrays of Nested Objects

The JavaScript `Array` `some()` method can be a good alternative to JS Array `includes()`, but it is more powerful.

It can be used in an array of objects for testing objects with deeply nested properties and complex test logic. For example, the `some()` method can be used to test if an array of `users` contains a `user` with an `address` in the United States:

<details>

<summary>Show nested users array example</summary>

```ts
const users = [
  {
    username: "jos",
    firstName: "Jos",
    lastName: "Bid",
    role: "dumper",
    address: {
      line1: "1600 Pennsylvania Avenue NW",
      city: "Washington, DC",
      zipCode: "20500",
      country: "United States",
    },
  },
  {
    username: "joe",
    firstName: "Joseph",
    lastName: "Hidin",
    role: "bumper",
    address: {
      line1: "1, Sleeping Pill Avenue",
      city: "Tehran",
      zipCode: "20500",
      country: "Iran",
    },
  },
  {
    username: "yo",
    firstName: "Joseph",
    lastName: "Bitin",
    role: "cumber",
    address: {
      line1: "2, Flyin Avenue",
      city: "Yeruham",
      zipCode: "8051108",
      country: "Israel",
    },
  },
];
```

</details>

```ts
const users = [
  {
    // user details
  },
  // more users
];

const findPresidentJoe = (j) =>
  j?.address?.country === "United States" && j?.role === "dumper";

console.log(users.some(findPresidentJoe)); // true
```

### JavaScript `some()` With `thisArg` Argument

We can pass in the `thisArg` object to JavaScript `some()` as the second argument. Passing the `thisArg` to JavaScript Array `some()` adds it to the execution context of the callback function.

We can then access the passed `thisArg` object from within our callback and use it for our advantage. Let's try doing that by making some modifications to our `even` callback in the earlier `numbers` array example.

Instead of checking for an even number, let's say we want to generalize our callback function to check if the item is divisible by any given number. We would like our callback now to be something like below:

```ts
function divisible(element, divisor) {
  return element % divisor === 0;
}
```

However, we cannot pass `divisor` as the second argument to `divisible()`, as our callback accepts `index` and `array` as the second and third arguments, respectively. And it becomes overcrowded if we introduce a fourth with `divisor`.

We can get around this problem by passing `divisor` as a property of the `thisArg` object, the second argument to `every()`. And then access the object with `this` from inside the callback:

```ts
const numbers = [1, 2, 3, 4, 5];

function divisible(element) {
  return element % this?.divisor === 0;
}

const isThereEvenNumber = numbers.some(divisible, { divisor: 2 });

console.log(isThereEvenNumber); // true
```

Here, we set the `thisArg` object to `{ divisor: 2 }`, which basically leads to checking if the item is even or not.

With this now, we can try other divisor options, like checking if we have a number divisible by 3 or 7. Thanks to `thisArg`, this has become very easy now:

```ts
const isThereAnyDivisibleByThree = numbers.some(divisible, { divisor: 3 });
const isThereAnyDivisibleBySeven = numbers.some(divisible, { divisor: 7 });

console.log(isThereAnyDivisibleByThree); // true
console.log(isThereAnyDivisibleBySeven); // false
```

### JavaScript `some(callback, thisArg)` Doesn't Work With Arrow Functions

If we look back at the first example that involves the `even()` callback, we defined it as an arrow function. And it worked.

We defined its extension, the `divisible()` function with named declaration syntax. And it worked as well.

If we declare `divisible()` as an arrow function, we run into problems:

```ts
const divisible = (element) => element % this?.divisor === 0;

const isThereEvenNumber = numbers.some(divisible, { divisor: 2 });
const isThereAnyDivisibleByThree = numbers.some(divisible, { divisor: 3 });
const isThereAnyDivisibleBySeven = numbers.some(divisible, { divisor: 7 });

console.log(isThereEvenNumber); // false
console.log(isThereAnyDivisibleByThree); // false
console.log(isThereAnyDivisibleBySeven); // false
```

All returning `false`, although we expect two to be `true` and one to be `false`.

If we investigate the problem with a modified `divisible()` function that logs `this` to the console, we see that `this` is `undefined` in strict mode:

```ts
// strict mode

const numbers = [1, 2, 3, 4, 5];

const divisible = (element) => {
  console.log(this);
  return element % this?.divisor === 0;
};

const isThereEvenNumber = numbers.some(divisible, { divisor: 2 });

console.log(isThereEvenNumber);
// undefined
// undefined
// undefined
// undefined
// undefined
// false
```

Now, if we introduce a `this.divisor` property to the lexical environment of `divisible()`, we get its value logged to the console:

```ts
const numbers = [1, 2, 3, 4, 5];
this.divisor = "Hi";

const divisible = (element) => {
  console.log(this);
  return element % this.divisor === 0;
};

const isThereEvenNumber = numbers.some(divisible, { divisor: 2 });

console.log(isThereEvenNumber);
// { divisor: 'Hi' }
// { divisor: 'Hi' }
// { divisor: 'Hi' }
// { divisor: 'Hi' }
// { divisor: 'Hi' }
// false
```

Here, clearly, we have `{ divisor: 'Hi' }` coming from `divisible()`'s closure. It turns out, the problem is due to the binding of `divisible()`'s `this` to it's lexical environment because of the arrow syntax. It was `undefined` before we introduced `this.divisor = 'Hi';`. Now `this` is `{ divisor: 'Hi' }`. In other words, `{ divisor: 2 `} is not being relayed to `divisible`'s `this`.

So, `some()` with `thisArg` does not work as expected with `callbackFn` defined with arrow syntax.

### JS `some(callback, thisArg)` Works With Non-Arrow Functions

But as we have seen before, it works with callbacks defined with named function declarations:

```ts
function divisible(element) {
  return element % this?.divisor === 0;
}

const isThereEvenNumber = numbers.some(divisible, { divisor: 2 });

console.log(isThereEvenNumber); // true
```

It also works with anonymous function expressions:

```ts
const divisible = function (element) {
  return element % this?.divisor === 0;
};

const isThereEvenNumber = numbers.some(divisible, { divisor: 2 });

console.log(isThereEvenNumber); // true
```

## Modifying the Caller Array

**JavaScript some method** sets the range of the items to be processed before the first invocation of the callback function.

`some()` itself does not modify the caller array, but the caller is available to the callback function as its third argument, `array`. And if an item is changed after traversal, the change is disregarded by the callback function. That is, the callback function only respects the existing value of an item at the time it is visited.

We can witness this in a scenario where the caller array is mutated from inside `some()`.

JavaScript `some()` itself does not modify the caller array, but the caller is available to the callback function as its third argument, `array`. This means we can deliberately mutate the caller when we need to from inside our callback, `divisible()`:

```ts
function divisible(element, index, array) {
  array[0] = 7;
  array[4] = 7;
  console.log(array);
  return element % this?.divisor === 0;
}
```

In this scenario, if an unvisited item is changed ahead of time, the callback function - here `divisible()` - finds the new value when it visits the item and so the new value is processed. In contrast, it disregards changes to items that are already traversed:

```ts
const divisible = function (element, index, array) {
  array[0] = 7;
  array[4] = 7;

  console.log(array);

  return element % this?.divisor === 0;
};

const isDivisibleBySeven = numbers.some(divisible, { divisor: 7 });

console.log(isDivisibleBySeven);
console.log(numbers);
/*
  [ 7, 2, 3, 4, 7 ]
  [ 7, 2, 3, 4, 7 ]
  [ 7, 2, 3, 4, 7 ]
  [ 7, 2, 3, 4, 7 ]
  [ 7, 2, 3, 4, 7 ]
  true
  [ 7, 2, 3, 4, 7 ]
*/
```

In the console log statements above, the `numbers` array is being logged five times due to the `console.log(array);` statement we placed inside `divisible()`.

As we can see, `numbers` is being mutated twice in the first call to `divisible()`. The first mutation happens when at `numbers[0]`, i.e. after being visited, which changes the value of itself to `7`. So, even though it was divisible by the divisor `7`, `some()` didn't immediately return `true`. Instead, it returned `true` in the next instance when it visited the unvisited and mutated value of `7` at `numbers[4]`.

This shows that the callback function processes the value of an item as it finds it at traversal and disregards the changes made to it when and after it is at that index.

## Comparing `some()` with `find()` and `every()`

I wanted to show a quick comparison of `some()`, `find()`, and `every()`, since each one is particularly strong for different scenarios with arrays.

### **`some()`** – Checks if Any Item Matches

The `some()` function tests whether _any_ item in the array fulfills a condition. It will short-circuit, that is, stop, the very first moment it finds a match as it makes its way through an array. This is a tremendous method whenever you want a fast check of something because all it needs to return is a "yes" or "no" answer.

```javascript
const numbers = [1, 2, 3, 4, 5];
const hasEven = numbers.some((num) => num % 2 === 0);

console.log(hasEven); // true, stops at the first even number
```

When you need to check if some item is satisfying the condition, then you should use `some()`.

### **`find()`** – Finds First Matching Item

`find()` goes one step further and actually _returns_ the first item that matches the condition. It's helpful when you would rather have the specific value, not just a boolean.

```javascript
const people = [{ name: "Alice" }, { name: "Bob" }, { name: "Charlie" }];
const foundPerson = people.find((person) => person.name.startsWith("B"));

console.log(foundPerson); // { name: 'Bob' }, stops after first match
```

Use `find()` when you want to retrieve the matching item itself, not simply know if one exists.

### **`every()`** – Checks if All Items Match

On the other hand, `every()` checks whether _each_ of the items in an array passes a test. This method tests all items and returns `true` if all items pass the test; otherwise, it stops on the first mismatch.

```javascript
const ages = [18, 22, 30];
const allAdults = ages.every((age) => age >= 18);

console.log(allAdults); // true, all ages are 18 or over
```

Use `every()` when you want to confirm every single item in the array passes a certain test.

In a nutshell:

- **`some()`** to check if at least one item matches.
- **`find()`** to return the first match.
- **`every()`** to confirm all items in the list match.

## Summary

In this post, we played around with the **JavaScript some** method which helps us test whether an array has at least one item that passes the test implemented using a callback function. We saw that the callback function could take only three arguments. Additional arguments can be bound to its execution context by setting its `this` value with a `thisArg` passed to `some()`.

We also found out that if we use arrow syntax to declare the callback function, its lexical context binding messes with the binding of `thisArg` to its `this` object. So, we should be using non-arrow functions to declare a callback function that uses `this` in its implementation.
