---
title: How to Use JavaScript Array Splice
description: This post is about various uses of the splice() method in JavaScript.
slug: javascript-splice-method
authors: abdullah_numan
tags: [javascript]
image: https://refine.ams3.cdn.digitaloceanspaces.com/blog/2023-10-20-js-splice/social-2.png
hide_table_of_contents: false
---

**This article was last updated on September 05, 2024, to add sections on Splice vs. Slice, Edge Cases with Splice, Performance Considerations, Splice in Multidimensional Arrays, and Using Splice with Destructuring.**

## Introduction

`Array.prototype.splice()` is a JavaScript `Array` method that is used to remove, replace or insert items starting at a chosen index. It is also an effective tool for adding items to the end of an array.

`Array.prototype.splice()` is a sparsely used but powerful JavaScript method for manipulating arrays **in place**. This means, we are mutating the actual array, not a copy of it. JavaScript `splice()` is useful for wiping all items after `n` number of initial items. It is handy for removing a number of intermediary items starting at an indicated index. It is especially needed for inserting items when we want to replace items or overload an array besides replacement. JavaScript `splice()` offers an alternative to `Array.prototype.push()` for growing the tail of the array as well.

Unlike several other `Array` methods, JavaScript `slice()` does not have a `String` equivalent.

In this post, we play around to illustrate various use cases of JavaScript `splice()` with trivial examples. We start with the method signature for `Array.prototype.splice()`, its possible parameters (`startIndex`, `deleteCount`, `item`s) and what they represent. We then go ahead and see examples of using combinations of passed arguments to manipulate an array. In particular, we observe how to wipe out items from a certain index onward. We also go through examples of replacing existing items with new ones. We explore how to insert items in the case of overloaded replacement and injection. We also figure out `splice()` way of adding items to the tail of the array.

Towards the end, we touch base on the nuances of the values passed as `startIndex` and learn how JavaScript `splice()` behaves when `startIndex` argument is negative or greater than the array length.

Steps we'll cover in this post:

- [JavaScript Splice](#javascript-splice)
  - [JS `Array.prototype.splice()` Method Signature](#js-arrayprototypesplice-method-signature)
  - [JS `splice()` Result / Return Value](#js-splice-result--return-value)
  - [JS `Array.prototype.splice()` without Arguments](#js-arrayprototypesplice-without-arguments)
  - [Wiping All After `n` Items - JavaScript `splice()` with Only `startIndex`](#wiping-all-after-n-items---javascript-splice-with-only-startindex)
  - [Deleting Intermediary Items with JS `Array.prototype.splice()`](#deleting-intermediary-items-with-js-arrayprototypesplice)
  - [Replacing Items with JS `Array.prototype.splice()`](#replacing-items-with-js-arrayprototypesplice)
  - [Items Injection with `Array.prototype.splice()`](#items-injection-with-arrayprototypesplice)
  - [Tailgrowing with `Array.prototype.splice()`](#tailgrowing-with-arrayprototypesplice)
  - [Inject Items Before Last `n` Items - JavaScript `slice()` with Negative `startIndex`](#inject-items-before-last-n-items---javascript-slice-with-negative-startindex)
- [Nuances of JS `Array.prototype.splice()`](#nuances-of-js-arrayprototypesplice)
- [Bonus: Splice vs. Slice in JavaScript](#bonus-splice-vs-slice-in-javascript)
- [Splice on Multidimensional Arrays](#splice-on-multidimensional-arrays)
- [Using Splice with Destructuring in JavaScript](#using-splice-with-destructuring-in-javascript)

## JavaScript Splice

When called by an array, JavaScript `splice()` takes three possible arguments: `startIndex`, `deleteCount` and `item`s list. Of these three types of arguments, `startIndex` indicates a zero-based index at which mutation kicks off. `deleteCount` denotes the number of items to be deleted and `item`s is a possibly infinite list of items to be inserted.

### JS `Array.prototype.splice()` Method Signature

Possible call signatures for JavaScript `splice()` are:

```js
splice()
splice(startIndex)
splice(startIndex, deleteCount, item1)
splice(startIndex, deleteCount, item1, item2, /*...*/, itemN)
```

None of the arguments for `Array.prototype.splice()` are required. However, when we want to mutate the caller array, we have to pass the `startIndex` as a minimum. Passing `deleteCount` depends on whether we want to delete any item/s or not. And passing the `item`s list depends on whether/what we want to insert or add to the array. It is also possible to pass all arguments at the same time. In the coming sections, we consider most of the cases.

### JS `splice()` Result / Return Value

As we'll start seeing next, `splice()` mutates the original array and returns an array containing **removed items**.

### JS `Array.prototype.splice()` without Arguments

Although, JavaScript `splice()` attempts to mutate the caller array, the returned value is **_an array of removed items_**. We can get a hint of the return value when an array calls it without passing any argument:

```js
const mnemonic = [
  "Please",
  "Send",
  "Cats",
  "Monkeys",
  "And",
  "Zebras",
  "In",
  "Large",
  "Cages",
  "Make",
  "Sure",
  "Padlocked",
];

// Return value
console.log(mnemonic.splice()); // []

console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys", "And", "Zebras", "In", "Large", "Cages", "Make", "Sure", "Padlocked"]
```

As we can see, `splice()` without argument returns an empty array `[]`. And as we'll see it become clearer in the next section, it is an array of removed items. In the case above, since no item is being removed, it is empty.

Notice that, although `splice()` is geared to remove items, it did not mutate the caller above, since we have not passed `startIndex`.

### Wiping All After `n` Items - JavaScript `splice()` with Only `startIndex`

When we pass `startIndex`, we should expect the caller array to mutate.

With **only** `startIndex` passed to JavaScript `splice()`, all items including the one **at and after** `startIndex` are removed:

```js
const mnemonic = [
  "Please",
  "Send",
  "Cats",
  "Monkeys",
  "And",
  "Zebras",
  "In",
  "Large",
  "Cages",
  "Make",
  "Sure",
  "Padlocked",
];

// Return value
console.log(mnemonic.splice(4)); // ["And", "Zebras", "In", "Large", "Cages", "Make", "Sure", "Padlocked"]

// Caller array
console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys"]
```

We can clearly see from this example that the returned array now has items including the one at `startIndex` (`4`) till the end **removed**. In other words, they were wiped out and the items upto `startIndex` are remaining in the original array.

It is important to notice that removal begins at `startIndex` **included**. This gives rise to a noticeable pattern where the zero based `startIndex` actually corresponds to first **n** items **remaining** in the caller array, i.e. `n = startIndex`:

```js
const mnemonic = [
  "Please",
  "Send",
  "Cats",
  "Monkeys",
  "And",
  "Zebras",
  "In",
  "Large",
  "Cages",
  "Make",
  "Sure",
  "Padlocked",
];

// Wiping all after first 4 items
console.log(mnemonic.splice(4)); // ["And", "Zebras", "In", "Large", "Cages", "Make", "Sure", "Padlocked"]
console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys"]

// Wiping all after first 2 items
console.log(mnemonic.splice(2)); // ["Cats", "Monkeys"]
console.log(mnemonic); // ["Please", "Send"]

// Wiping all items
console.log(mnemonic.splice(0)); // ["Please", "Send"]
console.log(mnemonic); // []
```

In the above examples, for each of the values passed as `startIndex`, we want that number of items to remain at the head of the original array. With `mnemonic.splice(0)`, we are emptying the original array.

### Deleting Intermediary Items with JS `Array.prototype.splice()`

We can remove intermediary items from an array by passing `deleteCount` as the second argument. This removes the indicated number of items beginning at `startIndex`:

```js
const mnemonic = [
  "Please",
  "Send",
  "Cats",
  "Monkeys",
  "And",
  "Zebras",
  "In",
  "Large",
  "Cages",
  "Make",
  "Sure",
  "Padlocked",
];
console.log(mnemonic.splice(4, 2)); // ["And", "Zebras"]
console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys", "In", "Large", "Cages", "Make", "Sure", "Padlocked"]
```

In the above call, we are passing `2` as `deleteCount`. Deleting starts at `startIndex` and ends at the index where `deleteCount` is achieved. Notice, removal slides the items after `deleteCount` to the **left**.

### Replacing Items with JS `Array.prototype.splice()`

We can replace removed items with new ones by passing in items as a list after the second argument. All items passed are added **_to `startIndex` onwards_**:

```js
const mnemonic = [
  "Please",
  "Send",
  "Cats",
  "Monkeys",
  "And",
  "Zebras",
  "In",
  "Large",
  "Cages",
  "Make",
  "Sure",
  "Padlocked",
];
console.log(mnemonic.splice(4, 0, "Ants", "Zebus")); // []
console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys", "Ants", "Zebus", "And", "Zebras", "In", "Large", "Cages", "Make", "Sure", "Padlocked"]
console.log(mnemonic.splice(4, 2)); // ["Ants", "Zebus"]
console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys", "And", "Zebras", "In", "Large", "Cages", "Make", "Sure", "Padlocked"]

// Items removed first and then added
console.log(mnemonic.splice(4, 2, "Alpacas", "Zorillas"));
console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys", "Alpacas", "Zorillas", "In", "Large", "Cages", "Make", "Sure", "Padlocked"]
```

Observe above that deletion gains precedence over insertion. It becomes more apparent when `deleteCount` is positive and items are replaced or added with arguments after . In the above series of mutations,`"Ants", "Zebus"` first inserted at `4`, with `0` items removed first. Then they were removed with `4` as `startIndex`, `2` as `deleteCount` and no items replacing them. In the last mutation, further removal of `2` items at index `4` happens, that are replaced subsequently with the new items.

We can actually go further and overload the array by adding more items than removed:

```js
const mnemonic = [
  "Please",
  "Send",
  "Cats",
  "Monkeys",
  "And",
  "Zebras",
  "In",
  "Large",
  "Cages",
  "Make",
  "Sure",
  "Padlocked",
];

// Add more items than removed
console.log(mnemonic.splice(4, 2, "Ants", "Zebus", "Donkeys", "Bears")); // ["And", "Zebras"]
console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys", "Ants", "Zebus", "Donkeys", "Bears", "In", "Large", "Cages", "Make", "Sure", "Padlocked"]
```

Notice that because we are replacing or overloading items, the items that come after get displaced towards the **right**.

### Items Injection with `Array.prototype.splice()`

You might already noticed that, if we want to only inject items at a certain point in the array, we can pass `0` as `deleteCount`:

```js
const mnemonic = [
  "Please",
  "Send",
  "Cats",
  "Monkeys",
  "And",
  "Zebras",
  "In",
  "Large",
  "Cages",
  "Make",
  "Sure",
  "Padlocked",
];
console.log(mnemonic.splice(4, 0, "Ants", "Zebus", "Donkeys", "Bears")); // []
console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys", "Ants", "Zebus", "Donkeys", "Bears", "And", "Zebras", "In", "Large", "Cages", "Make", "Sure", "Padlocked"]
```

New items are added starting at `startIndex` directly because with `deleteCount` set to `0`, so no items are deleted.

### Tailgrowing with `Array.prototype.splice()`

We can add to the tail by using the `length` of the caller array as `startIndex`:

```js
const mnemonic = [
  "Please",
  "Send",
  "Cats",
  "Monkeys",
  "And",
  "Zebras",
  "In",
  "Large",
  "Cages",
  "Make",
  "Sure",
  "Padlocked",
];
console.log(
  mnemonic.splice(mnemonic.length, 0, "Ants", "Zebus", "Donkeys", "Bears"),
); // []
console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys", "In", "Large", "Cages", "Make", "Sure", "Padlocked", "Ants", "Zebus", "Donkeys", "Bears"]
```

Caller `length` finds the tail for an incoming item as `length` is greater than the last index in the zero-index based array. Using `length` like this with `splice()` is a good occasional alternative to `Array.prototype.push()`.

It important to caution that using `-1` as `startIndex` **doesn't** add to the tail. That's because for `startIndex < 0`, mutation starts at `startIndex + mnemonic.length`. So, the last item always gets displaced as last and we never add to the tail:

```js
const mnemonic = [
  "Please",
  "Send",
  "Cats",
  "Monkeys",
  "And",
  "Zebras",
  "In",
  "Large",
  "Cages",
  "Make",
  "Sure",
  "Padlocked",
];
console.log(mnemonic.splice(-1, 0, "Ants", "Zebus", "Donkeys", "Bears")); // []
console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys", "And", "Zebras", "In", "Large", "Cages", "Make", "Sure" "Ants", "Zebus", "Donkeys", "Bears", "Padlocked"]
```

A variation of this next.

### Inject Items Before Last `n` Items - JavaScript `slice()` with Negative `startIndex`

We can inject new items before last `n` items by passing in `-n` as `startIndex`. This is another example when `startIndex < 0`:

```js
const mnemonic = [
  "Please",
  "Send",
  "Cats",
  "Monkeys",
  "And",
  "Zebras",
  "In",
  "Large",
  "Cages",
  "Make",
  "Sure",
  "Padlocked",
];

// Inject before last 2 items
console.log(mnemonic.splice(-2, 0, "Ants", "Zebus", "Donkeys", "Bears")); // []
console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys", "And", "Zebras", "In", "Large", "Cages", "Make", "Ants", "Zebus", "Donkeys", "Bears", "Sure", "Padlocked"]

// Inject before last 4 items
console.log(mnemonic.splice(-4, 0, "Apacas", "Zorillas")); // []
console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys", "And", "Zebras", "In", "Large", "Cages", "Make", "Ants", "Zebus", "Apacas", "Zorillas", "Donkeys", "Bears", "Sure", "Padlocked"]
```

Again, a negative `startIndex` brings `-n + mnemonic.length` to play. So, mutation starts at `n` items (**not** indexes) from the end of the array -- keeping last `n` items intact in the tail.

## Nuances of JS `Array.prototype.splice()`

In this section, let's consider other quirks of using JavaScript `slice()`.

### Higher Negatives of `startIndex`

There is another nuance of using negative `startIndex` with `splice()`. For example, when `startIndex < -mnemonic.length`, it is effectively `0`:

```js
const mnemonic = [
  "Please",
  "Send",
  "Cats",
  "Monkeys",
  "And",
  "Zebras",
  "In",
  "Large",
  "Cages",
  "Make",
  "Sure",
  "Padlocked",
];
console.log(mnemonic.splice(-100, 10, "Ants", "Zebus", "Donkeys", "Bears")); // ["Please", "Send", "Cats", "Monkeys", "And", "Zebras", "In", "Large", "Cages", "Make"]
console.log(mnemonic); // ["Ants", "Zebus", "Donkeys", "Bears", "Sure", "Padlocked"]
```

Here, with `startIndex` value of `-100`, removal starts at index `0` and first `10` items are removed. The passed items are added after removal at index `0`.

<br />

### Higher `startIndex` Value

When `startIndex` is greater than the length of the array, no items is deleted even if `deleteCount` is passed. Plus all passed items are tucked to the tail:

```js
const mnemonic = [
  "Please",
  "Send",
  "Cats",
  "Monkeys",
  "And",
  "Zebras",
  "In",
  "Large",
  "Cages",
  "Make",
  "Sure",
  "Padlocked",
];
console.log(mnemonic.splice(100, 10, "Ants", "Zebus", "Donkeys", "Bears")); // []
console.log(mnemonic); // ["Please", "Send", "Cats", "Monkeys", "And", "Zebras", "In", "Large", "Cages", "Make", "Sure", "Padlocked", "Ants", "Zebus", "Donkeys", "Bears"]
```

As we can see, `100` is far greater than `mnemonic.length`, and we asked for deletion of `10` items somewhere non-existent. So, none is deleted inside the array. And passed items are added to the tail.

Using an arbitrary higher index is, however, not safe for tailgrowing, since it does not always make sure `startIndex` is greater or equal to caller `length`.

## Bonus: Splice vs. Slice in JavaScript

I will briefly explain the difference between two methods in JavaScript: `splice()` and `slice()`. Generally, they are mistaken for being similar methods. The reason is that both these methods bear the same name.

### `splice()`

- **Mutates the original array**.
- Can be used to **remove**, **replace**, or **insert** elements at a specific index.
- It **mutates** the array in place and returns an array of removed elements.

Sample:

```js
let arr = [1, 2, 3, 4];
let removed = arr.splice(2, 2); // Starting from index 2, this will remove 2 elements
console.log(arr); // [1, 2, 5] - The original array is mutated
console.log(removed); // [3, 4] - The removed elements
```

### `slice()`

- **Does not modify the original array**.
- It returns the **shallow copy** of a portion of an array from the start index to the end index, one level deep.
- The original array is not modified.

Example:

```js
let arr = [1, 2, 3, 4];
let sliced = arr.slice(2, 4); // Copies elements from index 2 to 4 (excluding 4)
console.log(arr); // [1, 2, 3, 4, 5] - The original array is not changed
console.log(sliced); // [3, 4] - The sliced part
```

### Key Differences:

- `splice()` is a method that modifies the original array, whereas `slice()` doesn't.

Use the method `splice()` when you want to **alter** an array by removing, inserting, or replacing elements and use the method `slice()` when you want to **retrieve** a part of the array without having to change the array.

## Splice on Multidimensional Arrays

Here, I would like to share some insights about how the `splice()` method works in multidimensional arrays with JavaScript. If you have an array that contains other arrays, then it allows for changes through the `splice()` of the outer and inner arrays, respectively.

### Modifying the Outer Array

You can use `splice()` to add or remove entire inner arrays from the outer array:

```javascript
let matrix = [
  [1, 2, 3],
  [4, 5, 6],
  [7, 8, 9],
];
```

To pop out the second row (i.e., the array `[4, 5, 6]`):

```javascript
let removedRow = matrix.splice(1, 1); // Remove 1 element at index 1
console.log(matrix);
// Output: [[1, 2, 3], [7, 8, 9]]

console.log(removedRow);
// Output: [[4, 5, 6]]
```

### Changing Inner Arrays

You can also `splice()` the elements inside:

```javascript
// Delete the second element of the first row
let removedElement = matrix[0].splice(1, 1); // Removes 1 element at index 1 of the first inner array
console.log(matrix);
// Output: [[1, 3], [7, 8, 9]]

console.log(removedElement);
// Output: [2]
```

### Inserting Elements

Similarly, you can push elements into inner arrays:

```javascript
// Populate the second row
matrix[1].splice(2, 0, 10, 11);
// Pushes 10 and 11 into the second inner array at index 2

console.log(matrix);
// Output: [[1, 3], [7, 8, 10, 11, 9]]
```

### Application Example

Let's assume we have a multidimensional array representation of a schedule, where each inner array represents the tasks to be completed that day:

```javascript
let weeklySchedule = [
  ["Meeting", "Coding", "Review"], // Monday
  ["Planning", "Design"], // Tuesday
  ["Deployment", "Testing"], // Wednesday
];
```

If we have to scratch off "Design" on the schedule for Tuesday:

```javascript
weeklySchedule[1].splice(1, 1); // Removes 1 element at index 1 of Tuesday's array

console.log(weeklySchedule);
// Output: [["Meeting", "Coding", "Review"], ["Planning"], ["Deployment", "Testing"]]
```

### Key Points

- **Accessing Inner Arrays**: Access inner arrays using array indices before splicing.
- **Replacing Inner Array Elements**: The `splice()` function can be used to add, remove, or replace elements from the inner arrays.
- **Changing the Outer Array**: The `splice()` method is capable of adding or removing complete arrays that are enclosed within the outer array.

## Using Splice with Destructuring in JavaScript

Here's a cool trick in JavaScript: how to use `splice()` with destructuring. It can really simplify lots of things when you want to manipulate an array, yet also makes the code much more readable as you're working with specific elements within that array.

### Basic Splice + Destructuring

We can use `splice()` in an array to remove elements. Using destructuring, we are able to directly assign the removed items to variables.  
Here is an example to destructure and remove:

```javascript
let arr = [1, 2, 3, 4, 5];

// Splice the first two away, and destructure the result
let [first, second] = arr.splice(0, 2);

console.log(first); // Output: 1
console.log(second); // Output: 2
console.log(arr); // Output: [3, 4, 5] - The original array is now modified
```

Here, `splice()` removes the first two elements, destructuring assigns them to `first` and `second`, and the original array is mutated.

### Example: Element Swapping

You can also combine `splice()` and destructuring for more practical applications, such as swapping array elements:

```javascript
let numbers = [10, 20, 30, 40, 50];

// Swap 2nd and 4th elements
[numbers[1], numbers[3]] = [numbers[3], numbers[1]];

console.log(numbers); // Output: [10, 40, 30, 20, 50]
```

### Extracting Multiple Elements with Destructuring

You can even extract several parts of the array using `splice()` and apply these immediately with destructuring:

```javascript
let fruits = ["apple", "banana", "cherry", "date", "elderberry"];

// Remove 3 elements starting from index 1, destructively
const [firstFruit, secondFruit, thirdFruit] = fruits.splice(1, 3);

console.log(firstFruit); // Output: 'banana'
console.log(secondFruit); // Output: 'cherry'
console.log(thirdFruit); // Output: 'date'
console.log(fruits); // Output: ['apple', 'elderberry']
```

### Destructuring + Insertion

You can combine `splice()` with destructuring to delete and insert elements in a single shot:

```javascript
let arr = [1, 2, 3, 4, 5];

// Destructure two elements from the index 2 of the array
let [removed1, removed2] = arr.splice(2, 2, "a", "b");

console.log(removed1); // Output: 3
console.log(removed2); // Output: 4
console.log(arr); // Output: [1, 2, 'a', 'b', 5] - Array is modified with new elements
```

In this example, we delete elements `3` and `4` and then insert `'a'` and `'b'` into their places, while destructuring the deleted items into variables `removed1` and `removed2`.

### Why Use This?

- **Cleaner Code**: Use destructuring with `splice()` to avoid writing many lines of code while removing elements from it.
- **Direct Access**: The removed elements are available immediately for working, and the array is ready to be manipulated in a single step.

## Summary

In this post, explored various cases and patterns of using JavaScript `splice()`. We learned that `Array.prototype.splice()` is an array method and it doesn't have a `String` equivalent. We also understood that it mutates the original array and returns an array of removed items.

We found out how to wipe out all items after first `n` items by passing `n` as the `startIndex` to `splice()`. We illustrated how to remove intermediary items by passing `deleteCount` as the second argument. We also saw how to replace removed items by passing in new items as third argument and onward. We learned how to inject items with `0` passed as `deleteCount`.

We figured out an alternative to `Array.prototype.push()` by using `Array.prototype.splice()` with caller `length` passed as `startIndex`. We also discovered the pattern of adding items prior to last `n` items using `-n` as the `startIndex`. In the end, we explored the quirks of using extreme values of `startIndex`, particularly when it is more negative than `-length` of the caller array and when it is higher than `length`.
