---
title: Bun vs. Node.js
description: We'll go over the new Bun runtime that has created a buzz in the tech space lately.
slug: bun-js-vs-node
authors: victor_uma
tags: [javascript, dev-tools]
image: https://refine.ams3.cdn.digitaloceanspaces.com/blog/2023-06-04-bun-vs-node/social-3.png
hide_table_of_contents: false
---

**This article was last updated on August 12, 2024 to add sections for Cross-Platform Support and Comparison with Other Runtimes (e.g., Deno).**

## Introduction

Do you want to try out a new runtime environment? [Bun](https://bun.sh/) is the new Javascript runtime that claims to be better than [Node.js](https://nodejs.org/en). This article will show how we can test this with benchmark scores.

## Prerequisites

While there is no prerequisite to follow in this tutorial, you should at least know the fundamentals of Javascript and building basic web applications.

You need the followings:

- [Node 14](https://nodejs.org/en) or higher
- [Bun runtime](https://bun.sh/) installed
- npm installed
- A code editor

## Overview

In this tutorial, we will go over the new Bun runtime that has created a buzz in the tech space lately. We will talk about what a runtime does and why some developers are switching over to using bun. We will also carry out some benchmark test to see if bun really has the fastest runtime as the Bun team says. Let's dive rignt in.

Steps we'll cover:

- [What is a runtime](#what-is-a-runtime)
- [What is Bun](#what-is-bun)
- [Why is Bun Fast](#why-is-bun-fast)
- [Installing the Bun runtime](#installing-the-bun-runtime)
- [How does Bun compare with Node](#how-does-bun-compare-with-node)
- [Benchmarking Bun](#benchmarking-bun)
- [Comparison of Bun with Other JavaScript Runtimes (e.g., Deno and Node.js)](#comparison-of-bun-with-other-javascript-runtimes-eg-deno-and-nodejs)

## What is a runtime

Imagine you have a big box of LEGO blocks and you want to build a cool spaceship. You have all the instructions on how to put the pieces together, but you need something to actually assemble the spaceship and make it work. That's where a runtime comes in!
A runtime is like a special helper that takes care of building and running your LEGO spaceship. It's a program that makes sure all the pieces fit together correctly and that the spaceship does what it's supposed to do.

When you give the instructions to the runtime, it reads them step by step and starts putting the LEGO pieces in the right places. It follows the instructions precisely, making sure each piece is connected properly and that everything is in the right order.
Once the spaceship is built, the runtime also takes care of making it work. It powers up the spaceship, activates its engines, and controls all its cool features. It's like a small computer inside the spaceship that runs all the commands and makes sure everything runs smoothly.

In the world of programming, a runtime is similar. It's a special program that helps run other programs. It reads the instructions of a program, executes them step by step, and makes sure everything works as intended.
So, just like the helper in assembling the LEGO spaceship, a runtime is the special program that helps build and run other programs correctly. It's like a smart assistant that makes sure everything works smoothly, just like you'd expect from your awesome LEGO spaceship!.

### The Javascript runtime

The JavaScript runtime is like a translator between JavaScript and the computer. When you tell JavaScript to do something, like add two numbers together, the runtime listens and understands what you want. It then takes that instruction and talks to the computer in a language it understands.

The runtime also takes care of other important things. It makes sure JavaScript follows the rules and doesn't make any mistakes. It keeps an eye on JavaScript as it runs, like a teacher watching over a student to make sure they're doing their work correctly.

When the runtime talks to the computer and gets the result, it passes it back to JavaScript. It's like the runtime whispers the answer to JavaScript, and then JavaScript can do something with that answer. Maybe it shows the answer on the screen or uses it to make a cool animation.

## What is Bun

Bun is a JavaScript runtime built from scratch using the [Zig](https://ziglang.org/) programming language, with a focus on fast startup, efficient code execution, and a cohesive developer experience. It provides tools and features to optimize and streamline the development of JavaScript applications and is designed to be compatible with existing JavaScript ecosystems.

When you tell Bun what you want it to do, it listens carefully. It takes your instructions, which are written in a special language called JavaScript, and starts executing them step by step. It's like telling the a car where you want to go and how you want to get there.
But Bun doesn't just understand JavaScript—it's also really good at making JavaScript code run really fast! It's like having a car engine that can make your car zoom down the road faster than any other car.

Bun is built using a special programming language called Zig. Zig is like a magical tool that allows the people who created Bun to build it in a way that makes it very efficient and fast. It's like the secret recipe that makes Bun so powerful.
With Bun, you can do all sorts of things. You can build websites and applications that work really quickly and smoothly. You can also use the Bun CLI (Command Line Interface) to run your JavaScript and TypeScript files, bundle your code together, and even manage your project's dependencies.

## Why is Bun Fast

The Bun runtime exhibits impressive speed due to several key factors:

**Lightweight Design**: Bun is designed to be lightweight, resulting in a smaller codebase and reduced resource requirements. This optimized design allows Bun to deliver better performance in terms of both speed and memory usage compared to other runtimes.

**Low-Level Implementation**: The Bun runtime is built using Zig, a relatively new low-level programming language. Zig's characteristics enable developers to write code with fine-grained control over memory management and execution, contributing to the runtime's efficiency.

**Performance Optimization**: Instead of relying on the V8 engine, Bun utilizes the JavaScriptCore from WebKit, which is widely recognized for its superior performance. By leveraging this core engine, Bun benefits from its optimized execution of JavaScript code, resulting in improved runtime speed.

**Integrated Functionality**: Bun offers native tools and features that streamline the development process. It includes a built-in bundler, replacing the need for external tools like Webpack, as well as a native transpiler that supports writing TypeScript code directly. Additionally, Bun incorporates a test runner similar to Jest and automatically loads environment variables without requiring additional installations of packages like dotenv.

## Installing the Bun runtime

To install Bun, you can follow these steps:
Open your computer's terminal or command prompt. In the terminal, enter the following command:

```tsx
curl -fsSL https://bun.sh/install | bash
```

For macOS users, run this after:

```tsx
exec / bin / zsh;
```

This command will initiate the installation process for Bun by downloading the installation script from the official Bun website. Press Enter and allow the installation script to run. It will handle the necessary steps to install Bun and its dependencies on your system. Wait for the installation process to complete. The script will take care of all the required tasks to ensure Bun is properly installed on your computer.

Once the installation is finished, you will have successfully installed Bun. You can now start using the Bun runtime to run your JavaScript and TypeScript applications and take advantage of its bundled tools and optimized performance.

## How does Bun compare with Node

In this section, we'll look at how Bun compares to Node and do some benchmarking between this two runtime.

1. **Performance**: Bun emphasizes faster startup times and runtime performance by utilizing the JavaScriptCore engine from WebKit, renowned for its speed. In contrast, Node.js relies on the V8 engine, which is also highly optimized but may have performance distinctions compared to JavaScriptCore.
2. **Size and Dependencies**: Bun strives to be a lightweight runtime with a smaller codebase and minimal dependencies. It incorporates built-in tools like a bundler and transpiler, reducing reliance on external dependencies. In contrast, Node.js is a more comprehensive runtime with a larger codebase and extensive support for external modules and libraries.
3. **Compatibility**: Although Bun aims to serve as a drop-in replacement for Node.js, there may be variances in API compatibility. While Bun natively implements many Node.js and Web APIs, some specific Node.js modules or APIs might not be fully supported.
4. **Tooling**: Bun provides an integrated toolkit for JavaScript development, including a bundler, transpiler, and package manager. Node.js, on the other hand, boasts a rich ecosystem of third-party tools and libraries for diverse development tasks, such as popular bundlers like Webpack and package managers like npm or Yarn.
5. **Community and Ecosystem**: Node.js benefits from a mature and extensive community, offering substantial support, well-documented resources, and a vast ecosystem of modules and libraries. In contrast, Bun, being relatively newer, may have a smaller community and a more focused ecosystem.

## Benchmarking Bun

This benchmarking test is running on my Mac M1, 8gb ram computer. For the benchmarking tool, we will be using [k6](https://k6.io/) an open source tool by Grafana labs. You can find the installation guide for this tool [here](https://k6.io/docs/get-started/installation/).

Here is our computer software version:

- Node v16.14.2
- Bun v0.4.0

For our code, I have gotten a simple HTTP server code from the [Bun](https://bun.sh/) and [Node.js](https://nodejs.org/en/docs/guides/getting-started-guide) official sites. Here is the Hello World code in Bun and Node

**Node**

```tsx
const http = require("http");
const hostname = "127.0.0.1";
const port = 3000;

const server = http.createServer((req, res) => {
  res.statusCode = 200;
  res.setHeader("Content-Type", "text/plain");
  res.end("Hello World");
});

server.listen(port, hostname, () => {
  console.log(`Server running at http://${hostname}:${port}/`);
});
```

Run the command in your terminal:

```tsx
node server.js
```

Your node server will be running on port: http://localhost:3000/

**Bun**

```tsx
export default {
  port: 3001,
  fetch(_) {
    return new Response("Hello World");
  },
};
```

Your Bun server will be running on port: http://localhost:3001/

Run the command in your terminal:

```tsx
bun run bunserver.js
```

Create a `script.js` file and paste this test script:

```tsx
import http from "k6/http";
import { sleep } from "k6";

export default function () {
  http.get("http://localhost:3001/"); // this will change depending on the server you're testing for
  sleep(1);
}
```

In your terminal run:

```tsx
k6 run script.js
```

Here is the result for our Node server:

<div className="centered-image">
   <img style={{alignSelf:"center"}}  src="https://refine.ams3.cdn.digitaloceanspaces.com/blog/2023-06-04-bun-vs-node/node.png"  alt="bun vs node js" />
</div>

<br/>

Here is the result for our Bun server:

<div className="centered-image">
   <img style={{alignSelf:"center"}}  src="https://refine.ams3.cdn.digitaloceanspaces.com/blog/2023-06-04-bun-vs-node/bun.png
"  alt="bun vs node js" />
</div>

<br/>

We can now see and compare Bun speed to Node. If you want to go further you can introduce different latencies, more users and duration with the script:

```tsx
k6 run --vus 10 --duration 30s script.js
```

## Cross-Platform Support with Bun and Node.js

I wanted to discuss the cross-platform support of Bun and Node.js, which I think is a really critical point that we should consider when choosing a runtime for our projects. Here, I've pointed out performance across various operating systems with some command examples:

### Node.js Cross-Platform Support

Over the last ten years, Node.js has become one of the prominent things to all OSs—macOS, Linux, or Windows. At such time, this makes Node.js a very reliable choice for any developer working with diverse environments. Here are some key points:

- **macOS and Linux**: Node.js works very well on Unix-based systems like macOS and Linux. It fits so well into things such as shell scripting, package management with npm, or Yarn, and many other tools more at home most typically in these environments.

```bash
# macOS/Linux
If you are using macOS or Linux, Node.js can be installed using
curl -fsSL https://deb.nodesource.com/setup_14.x | sudo
sudo apt-get install -y nodejs
```

- **Windows**: Node.js also interfaces very well with Windows; it has native installers and is properly integrated, even though this also implies being able to properly work with Windows Subsystem for Linux. The Node.js team has worked out most of the historical differences in path handling and command execution.

```powershell
## Installing Node.js on Windows with the Node.js Installer
winget install OpenJS.NodeJS
```

- **ARM and Other Architectures**: Node.js supports a bunch of processor architectures, including ARM. The major importance here is due to the fact that the devices like Raspberry Pis and Apple M1/M2 are seeing increased adoption.

```bash
# Install Node.js on Raspberry Pi, which has an ARM architecture
curl -fsSL https://deb.nodesource.com/setup_14.x | sudo
sudo apt-get install -y nodejs
```

### Bun Cross-Platform Support

Bun is a pretty new runtime for which many impressive gains have been made in terms of performance and features; cross-platform support is still young. Here's an overview of its current state:

- **macOS**: Bun works great with macOS, especially on Apple's M1/M2 chips. Its performance is at the best level with this platform, and it provides a good service for developers working on macOS.

```bash
# Install Bun on macOS
curl -fsSL https://bun.sh/install | bash
```

- **Linux**: Bun also works on Linux but, since it's newer, there might be some rough edges, or things could be not as good as with Node.js in terms of community tooling.

```bash
# Install Bun on Linux
curl -fsSL https://bun.sh/install | bash
```

- **Windows**: Windows support for Bun is still very new. It should be possible to run Bun on Windows, but you might require WSL or possibly come across other limitations when running on a native Windows environment.

```bash
# Install Bun on Windows using WSL
curl -fsSL https://bun.sh/install | bash
```

- **ARM and Other Architectures**: Bun is being gradually developed to work with a growing number of architectures; ARM is one example. However, the performance and stability on these architectures have not reached that of Node.js quite yet.

```bash
# Install Bun on ARM architecture
curl -fsSL https://bun.sh/install | bash
```

## Comparison of Bun with Other JavaScript Runtimes (e.g., Deno and Node.js)

It will be good to benchmark Bun vs. other JavaScript runtimes like Deno and Node.js. That might put us in perspective about how Bun compares to other JavaScript run-times out there so that we can make a little more educated decision about our use case for the project. Here is a rundown of comparison:

### Overview of Runtimes

- **Node.js**: This is the most established JavaScript runtime, best known for its vast ecosystem, mature tooling, and solid community support. It is based on the V8 JavaScript engine and has been the primary runtime utilized for server-side JavaScript over the last decade.

- **Deno**: Deno is a new runtime created by Ryan Dahl, the original creator of Node.js. It tries to fix some problems in Node.js, such as security issues and dependency management. Built using V8 and Rust, with its primary focus being on having modern features—like having out-of-the-box support for TypeScript, secured by default, and an inbuilt package manager—this is Deno.

- **Bun**: Bun is a newcomer to the JavaScript runtime arena, focusing on performance, speed, and developer experience. It's built with the Zig programming language and uses WebKit's JavaScriptCore engine. It comes with built-in tools, such as a bundler, transpiler, and test runner, to make everything easy for developers.

### Performance

- **Node.js**: Known for good performance in general, especially with I/O-bound applications. It relies on V8, which is optimized around the best possible speed and efficiency, meaning that in certain scenarios it may not be as fast as some newer runtimes.

- **Deno**: Deno is comparable in performance to Node.js, and it has some advantages in certain areas, such as cold start times and memory usage. Because it uses Rust for the core runtime, it will be that much more memory-safe and, sometimes, even faster.

- **Bun**: Bun is particularly speed-optimized with very fast start-up and execution time compared to Node.js and Deno. It achieves this with the help of JavaScriptCore engine and lightweight design, making it an excellent choice for performance-critical applications.

### Security

- **Node.js**: Security features are not turned on by default for Node.js; hence, developers need to be proactive for securing their application—careful management of dependencies and configuration of runtime to avoid basic security pitfalls.

- **Deno**: Security is one of the things that are greatly advertised by Deno. Deno runs by default in a secured, sandboxed environment with explicit permission required for file, network, and environment access, thus reducing the risks of vulnerabilities from third-party packages.

- **Bun**: Bun is yet to mature for its security model. It might have the best of a few security characteristics, but it still may not match Deno's inbuilt protection. A developer using Bun has to be extra cautious in practicing security, more so because it's a younger runtime than Deno, and the community is quite smaller.

### Ecosystem and Tooling

- **Node.js**: It has a very large ecosystem since it contains millions of packages available via npm. Also, it is very flexible in use; this is because mature tooling is provided to do anything with it, from build processes to deployment.

- **Deno**: Growing quickly in an ecosystem, Deno is on its expanding front yet stands way much smaller than Node.js. The module system in Deno is decentralized — it fetches packages directly from URLs as compared to the centralized repository in npm. This allows more fragmentation but then, even better flexibility.

- **Bun**: Bun aims to be a middle ground with a built-in bundler, transpiler, and test runner that reduces the amount of external dependencies but also makes developing easy. However, it is emerging and doesn't have as many third-party packages as Node.js does.

### TypeScript Support

- **Node.js**: Although Node.js does not come natively with TypeScript, it plays along very well with TypeScript due to the commonly used transpilers Babel or using the TypeScript compiler (tsc). And now due to the strong community support, the TypeScript is being welcomed on strongly, and most of the packages have TypeScript definitions.

- **Deno**: Deno was designed from the ground up with TypeScript being a first-class language; there isn't any need for the user to need special configuration. This makes it good for developers looking to use TypeScript without any extra setup.

- **Bun**: Bun also supports native TypeScript. One can do author in TypeScript without any need for additional configuration or tooling. This helps greatly in easing the pain in the adoption of TypeScript within projects that make use of Bun.

### Community and Adoption

- **Node.js**: The largest and most active community, extensive documentation, tutorials, and support—literally, it is applied in any work, from small projects to enterprise-level applications.

- **Deno**: Around Deno, a community is developing very quickly, mostly for developers who are using modern JavaScript features and focusing on security. Although not as big as the Node.js ecosystem, it has a more alive and friendly community and is more focused on modern best practices.

- **Bun**: It is quite a new runtime, so naturally the community around it is small, albeit rapidly developing. It generates a lot of interest due to claims of performance, but it might take yet a while for adoption on the level of Node.js and even Deno.

Final words;

- **Node.js**: Good for developers who need a stable, fully-supported runtime with rich features in tools and libraries.

- **Deno**: A good option for a security-centric project looking for JavaScript features at the forefront with first-class TypeScript integration. This is also a good alternative for developers who don't want to get lost in the way package management is done in Node.js.

- **Bun**: Ideal for developers with the need for maximum performance, fast startup time, and all-in-one development based on built-in tools; however, its ecosystem is still in maturity, so at this stage, it can better match with critical performance or smaller projects.

### Conclusion

In conclusion, Bun and Node.js are two JavaScript runtimes that offer different approaches and features for developers. Bun focuses on delivering fast startup times, optimized performance, and a lightweight design with integrated tools like a bundler and transpiler. It utilizes the JavaScriptCore engine from WebKit to achieve its performance goals. On the other hand, Node.js has a larger ecosystem, extensive community support, and compatibility with a wide range of programming languages. It relies on the V8 engine and offers a rich set of third-party tools and libraries. Choosing between Bun and Node.js depends on factors such as performance requirements, specific project needs, and the availability of suitable tooling and community support. Ultimately, developers can leverage the strengths of each runtime to build robust and efficient JavaScript applications.
