---
title: How to Install and Use NVM?
description: Guide to using NVM with Mac and Windows.
slug: install-nvm-mac-and-windows
authors: chidume_nnamdi
tags: [javascript, dev-tools]
image: https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-11-09-nvm/social-2.png
hide_table_of_contents: false
---

**This article was last updated on July 05, 2024, to add sections for Advanced NVM Usage and Best Practices for Using NVM.**

## Introduction

Nodejs became a game changer in the software development world after its initial release in May 2009. Since then, software development has not been the same. The majority of big companies turned towards Nodejs and many JavaScript frameworks were birthed. Finally, JavaScript ceased from being a browser-only language to an all-platform language. We can create mobile apps, and desktop apps now with JavaScript.

To run Nodejs all you need to do is to install the executable and off you go!
Now, Nodejs got better and better with each new version released but sometimes these versions can be a problem, and we might see ourselves running incompatible Nodejs in our projects. So we might just have to install the right Nodejs version, but do we have to install and re-install Nodejs versions all the time?

The answer is NO because a brilliant does all the versioning work for us. The tool is called nvm, Node Version Manager.

Steps we'll cover:

- [What's `nvm`?](#whats-nvm)
- [Installation of nvm](#installation-of-nvm)
- [Installing a Node version](#installing-a-node-version)
- [Switching environments](#switching-environments)
- [Advanced NVM Usage](#advanced-nvm-usage)
- [Useful nvm commands](#useful-nvm-commands)
- [NVM Best Practices](#nvm-best-practices)

## What's `nvm`?

nvm like we have learned is a Node Version Manager. It allows us to install and switch between different versions of Nodejs.

Instead of manually downloading and uninstalling the version of Nodejs nvm does it behind the hood for us. Just that it does not uninstall any version. It just manages them and switches to anyone at will.

Let's say you are trying to run a Node project and the project works correctly (for e.g) Node version 14 but your Node version is 10. You will likely get this warning:

```
This project requires Node version 14
```

I encountered that on a project I was part of. I am running Node version 16.17.0 but the tests in the project were written in Node version 14.18.1. I constantly got failures concerning V8 buffer overflows and most of the tests were failing because of that. It wasn't until after a long time I got to realize that the tests were initially written in a Node 14.18.1 environment. So I had to use nvm and switch from 16 to 14 and all the tests passed.

One of the quick and most powerful commands of the nvm is the `use` command. This command does the switching of the Node version.

```
nvm use 14.18.0
```

This command will switch your current Node environment version to switch from its current version to version 14.18.0.

Let's say you are on Node 16.0.1

```
node --v
16.0.1
```

Now, to switch to Node version 14.18.0, we run the previous command:

```
nvm use 14.18.0
```

Let's check if the Node version is correctly switched:

```
node -v
14.18.0
```

Yes, it is now on Node version 14.18.0.

## Installation of nvm

The easiest way to install the nvm in Linux or MacOS is by using the curl or wget tool. you run any of the commands below:

```
curl -o- https://raw.githubusercontent.com/nvm-sh/nvm/v0.39.2/install.sh | bash

wget -qO- https://raw.githubusercontent.com/nvm-sh/nvm/v0.39.2/install.sh | bash
```

These commands download the `install.sh` script from `https://raw.githubusercontent.com/nvm-sh/nvm/v0.39.2/` and run the script.

This script contains instructions on where to fetch the nvm repository and install it on the executing machine. This nvm repository will be cloned to `.nvm` directory in your machine, and then add the below snippet to either of the profile configuration files: `~/.bash_profile`, `~/.zshrc`, `~/.profile`, or `~/.bashrc`.

```
export NVM_DIR="$([ -z "${XDG_CONFIG_HOME-}" ] && printf %s "${HOME}/.nvm" || printf %s "${XDG_CONFIG_HOME}/nvm")"
[ -s "$NVM_DIR/nvm.sh" ] && \. "$NVM_DIR/nvm.sh" # This loads nvm
```

Installation on a Windows machine is different from the above. In Windows, you will use the nvm-windows tool. This NVM for Windows is the Microsoft/npm/Google recommended Node.js version manager for Windows.
N.B It is not the same thing as nvm.

You can download the latest release of the NVM Windows here. To view all releases of the NVM Windows, you can see it here. For more instructions on how to install the NVM Windows, upgrade, and more, visit their Github repo page.

Now, when done with the installation of the nvm we can verify if the installation is successful and that the nvm is available globally on our machine we will run the below command:

```
nvm -v
```

This will print the version of the nvm currently on your machine.

```
Node Version Manager (v0.35.3)

Note: <version> refers to any version-like string nvm understands. This includes:
  - full or partial version numbers, starting with an optional "v" (0.10, v0.1.2, v1)
  - default (built-in) aliases: node, stable, unstable, iojs, system
  - custom aliases you define with `nvm alias foo`

 Any options that produce colorized output should respect the `--no-colors` option.
```

See mine above, as you can see I have nvm version 0.35.3 on my machine.

## Installing a Node version

We can install a particular Nodejs version using `nvm`. This is done by running the command:

```
nvm install 10.1.0
```

This will install the Nodejs v10.1.0 on your machine. Now, if there is Nodejs v14 in your machine, it will not be removed. Instead nvm will install the Nodejs v10.1.0 to sit in my machine alongside the v14. Now, the current version of Node that the `node` will use is the version that was mainly installed directly, not from nvm.

You can install as many as Nodejs versions you want:

```
nvm install 11.2.1
```

```
nvm install 16.2.3
```

To install the latest Node version, we will run the below command:

```
nvm install node
```

This will install the latest version of Nodejs on your machine.

If we run the `install` command with only the major version, then nvm will install the latest version of Nodejs in that major version.

For example, this:

```
nvm install 10
```

will install the latest version of Nodejs v10. Let's say the latest version of Nodejs in major version 10 is 10.19.0. It will be installed.

## Switching environments

We can switch Node environments by using the nvm `use` command.

```
nvm use 14
```

This switches to Nodejs version 14. We must have the Nodejs 14 already installed on our machine before we can switch to it.

```
nvm use 12.0.1
```

This will switch to Nodejs v12.0.1.

There is something that we need to understand. If we only state the major version in the `use` command, nvm will switch to using the major version of the Nodejs regardless of the minor and patch versions.

For example, this:

```
nvm use 14
```

This is just 14, the major version. There is no minor or patch version. Now, this will switch to using the latest Node version in the machine with a major version of 14.
If there are Node versions: `14.17.1`, `14.18.1`, and `14.18.0`. nvm will pick the latest `14.18.1` and switch to it.

## Advanced NVM Usage

In this new section, I put a part on advanced NVM usage into our article on NVM installation and usage to automatize the switching of versions for Node. I integrated NVM with other well-known development tools.

#### Automating Node Version Switch

Automating process-based switching of Node versions with respect to the project directory will save time and achieve consistency in managing the problem. This can be done with `.nvmrc` files. An `.nvmrc` file specifies a version of Node for a project, and NVM will automatically switch to that version when changing to the project directory.

Below are some simple steps to create and use `.nvmrc` files for automatic Node version switching:

1. Place in the project's root folder a file named `.nvmrc` that contains the version of Node you want to install.
2. Run the `nvm use` command to switch to the Node version specified in the `.nvmrc` file.

```bash
# Create a .nvmrc file in your project root
echo "14.18.1" > .nvmrc

# Use the Node version specified in .nvmrc
nvm use
```

- Create a `.nvmrc` file in your project root: `echo "14.18.1" > .nvmrc`
- Use the Node version that's defined in `.nvmrc`: `nvm use`

#### Combining NVM with Development Tools

With NVM built into popular development tools such as Visual Studio Code and Docker, this guarantees the right Node version is used and assures a common one in the development environment. This embeds NVM inside frequent development tools to give good assurance of correct Node versioning, with no issues or time waste due to conflicts in different environments.

Here is how you can configure all these tools so they can use the right Node version managed by NVM:

1. **Visual Studio Code**: Update your VS Code settings to point to the Node version controlled by NVM.
2. **Docker**: Configure Docker to use NVM and the correct Node version in your Docker containers.

- For example, a Visual Studio Code settings.json:
  ```json
  {
    "terminal.integrated.shellArgs.linux": ["-l"],
    "terminal.integrated.env.linux": {
      "NVM_DIR": "$HOME/.nvm",
      "PATH": "$NVM_DIR/versions/node/$(nvm version)/bin:$PATH"
    }
  }
  ```

It will enable you to automate the switching of Node versions and include NVM as part of the development tools that you are working with. This integration of NVM with the tooling will become a smooth and efficient workflow.

## Useful nvm commands

Let's see some other useful nvm commands:

### `list`

This command lists/displays all the Nodejs versions we have on our machine.

```
nvm list
```

Let's see the output:

```
v14.17.1
v14.18.1
v16.9.0
-> v16.17 system
default -> node (-> v16.17.0)
node -> stable (-> v16.17.0) (default)
stable -> 16.17 (-> v16.17.0) (default)
iojs -> N/A (default)
unstable -> N/A (default)
lts/* -> lts/gallium (-> v16.17.0)
lts/argon -> v4.9.1 (-> N/A)
lts/boron -> v6.17.1 (-> N/A)
lts/carbon -> v8.17.0 (-> N/A)
lts/dubnium -> v10.24.1 (-> N/A)
lts/erbium -> v12.22.12 (-> N/A)
lts/fermium -> v14.20.0 (-> N/A)
lts/gallium -> v16.17.0
```

See that it displays all the Nodejs versions in my machine and LTS(long-term support) versions. See that the arrowed item in the list is our current Nodejs version.

### `uninstall`

This is the reverse action of `install`. This command will uninstall a Node version from the machine:

```
nvm uninstall 13.10.1
```

This will remove the installed Nodejs version 13.10.1 from your machine.

### `run`

This command lets you run a script with a Nodejs version without changing the current version.

```
nvm run 6.10.3 app.js
```

This runs app.js using node 6.10.3. You don't have to switch the current version to 6.10.3 using the `use` command before you can run the script. So this `run` command does the switching before running the script and switches back to the current version after executing the script.

## NVM Best Practices

I have organized the best practices section into our article on installing and using NVM. The following section provides some of the most important tips that help in the efficient and effective utilization of NVM.

### Keeping NVM and Node Versions Up-To-Date

Security, performance, and compatibility with the latest features and fixes in bug calls require that NVM and Node versions be updated regularly. Keep your development environment up to date so you can take advantage of the latest improvements in the Node.js ecosystem.

**Best Practices**:

- Install updates for NVM on a regular basis.
- Always use the latest stable versions of Node for new projects.
- Regularly check and update Node versions for existing projects in order to meet compatibility and security requirements.

```bash
# Update NVM to the latest version
nvm install-latest-npm

# Install the latest Node version
nvm install node

# Update existing Node versions and reinstall global packages
nvm install 14.18.1 --reinstall-packages-from=14.17.0
```

- Update NVM: `nvm install-latest-npm`
- Install the current version of node: `nvm install node`
- Update existing Node versions: `nvm install 14.18.1 --reinstall-packages-from=14.17.0`

### Leveraging NVM for Various Environments

NVM helps to manage multiple Node environments on a single machine. It is beneficial when you jump between development, test, and production, which most likely requires different Node versions.

**Best Practices**:

- Place `.nvmrc` files within your project directories outlining what version of Node your project should be using.
- Setup NVM scripts for auto-switching between Node versions based on the particular task you're running.
- Test your app on various Node versions to ensure it's compatible and performs well in every environment.

```bash
# Create an .nvmrc file in your project root to specify Node version
echo "14.18.1" > .nvmrc

# Switch to the Node version specified in .nvmrc
nvm use

```

- Create an `.nvmrc` file in your project root: `echo "14.18.1" > .nvmrc`
- Switch to the specified Node version in `.nvmrc`: `nvm use`

### Managing Global Packages

It becomes very crucial when effectively managing global packages with NVM, whereby much consideration has to be taken so that there is no conflict and consistency maintained across the different Node versions.

**Best Practices**:

- Install global packages for each Node version so that the environments are isolated from one another.
- Using the `reinstall-packages` NVM feature, global packages are moved with a Node upgrade.
- Regularly audit and clean up your global packages to keep your dev environment nice and trimmed.

```bash
# Reinstall global packages when upgrading Node versions
nvm install 14.18.1 --reinstall-packages-from=14.17.0

# List global packages for the current Node version
npm list -g --depth=0

# Remove an unused global package
npm uninstall -g package-name
```

- Reinstall global packages when upgrading Node versions: `nvm install 14.18.1 --reinstall-packages-from=14.17.0`
- List all globally installed packages: `npm list -g --depth=0`

These best practices allow for a smooth and efficient flow with NVM, allowing you an environment that is current, consistent, and best optimized for different project requirements.

## Conclusion

We learned a lot from this article. We started by introducing Nodejs and then followed by nvm. Next, we learned about nvm and what it does to Node versions in our machine.

Next, we learned how nvm switches between Node versions in our machine, and the command we can use to do that. We learned also how to install and uninstall Node versions using the nvm tool.

Finally, we saw lists of useful nvm commands that we can to ease our use of the nvm tool.
