---
title: Next.js 13 - What are the new features?
description: Next.js 13 is out! Let's see the new features in this release.
slug: new-features-in-next-js-13
authors: michael
tags: [nextjs]
image: https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-11-12-next-13/social.png
hide_table_of_contents: false
---

## Introduction

Next.js is a popular React.js framework for building complex and performant web applications. Since its launch in 2016, Next.js has been constantly improved and updated for better performance and a great developer experience. Next.js 13 was announced at the just concluded [Next.js Conference](https://nextjs.org/conf) by Next.js and Vercel CEO, Guillermo Rauch. The announcement also re-affirms Next.js commitment and mission to revolutionize web development.

This new release comes with upgrades and new additions that let you create dynamic applications without limits and supports the latest features in React.js 18. Some even say Next.js 13 is the real React.js 18 (well, that's true to some extent).

This article will explore the latest features and also take you through installing and upgrading your codebases to Next.js 13. Let's dive in!

Steps we'll cover:

- [Introduction](#introduction)
- [Installing Next.js 13](#installing-nextjs-13)
- [Latest Features in Next.js 13](#latest-features-in-nextjs-13)
- [Upgrading to Next.js 13](#upgrading-to-nextjs-13)

### Installing Next.js 13

To create a Next.js 13 app, you need to have [Node.js 14](https://nodejs.org/) or later installed on your computer. Next, run any of the commands below to install Next.js 13:

```bash
npx create-next-app@latest --experimental-app
# or
yarn create next-app --experimental-app
# or
pnpm create next-app --experimental-app
```

With the exception of a new app folder, the generated files and folders are almost the same as before in older versions.

Next, run `npm run dev` to start the development server and you're good to go!

## Latest Features in Next.js 13

Now, let's look at some of the latest and most interesting features in the new release.

### A brand new `app/` directory

One of the notable features in Next.js 13 is a brand new `app` directory (beta). The `app` directory file-system based router is an upgrade to Next.js page routing system. The app folder comes with a few [special files](https://beta.nextjs.org/docs/routing/fundamentals#special-files) that can be used for different states and UI of your application. For example, you can use an optional `error.tsx` file to isolate errors to specific parts of your application.

The `app` directory supports nested layouts, streaming, React's Server Components - the `app` directory will treat components inside of it as Server Components by default - nested routing and more. It works well alongside the `pages` folder so it can be adopted incrementally.

### Turbopack (Alpha)

Next.js 13 is built on top of Turbopack, a native, rust-based compiler, and successor to Webpack. Turbopack takes only 1.8 seconds during startup compared to Webpack, which takes 16.5 seconds. It's also 700x faster during updates than Webpack, 10x faster updates than Vite, and 4x faster cold starts than Webpack.

It's interesting to note that Turbopack development is being led by the creator of Webpack, Tobias Koppers. Turbopack alpha is currently supported only in development. You can run the following command to try it out `next dev --turbo`.

### A new way to fetch data

Another cool feature in Next.js 13 is a new data fetching method built on top of the native [fetch() Web API](https://developer.mozilla.org/en-US/docs/Web/API/Fetch_API). It allows you to use `async/await` to asynchronously fetch data in Server Components created with JavaScript `await` syntax, with performance benefits such as automatic caching and data re-validation.

It also comes with a `use` hook for unwrapping values from a Promise. The `use` hook can be used inside normal components, hooks and on the client. Think of it like a React-only version of `await`. `use` is being developed by the React team and uses Suspense under the hood.

Below is an example taken from [here](https://github.com/acdlite/rfcs/blob/first-class-promises/text/0000-first-class-support-for-promises.md) showing how to fetch data with `async/await` in Server Components.

```ts title="app/page.tsx"
async function Note({ id, isEditing }) {
  const note = await db.posts.get(id);
  return (
    <div>
      <h1>{note.title}</h1>
      <section>{note.body}</section>
      {isEditing ? <NoteEditor note={note} /> : null}
    </div>
  );
}
```

### Brand new `next/font` for font optimization

Next.js 13 introduces a brand new font module, `next/font`, that automatically optimizes your local and Google fonts for performance, prevents layout shifts using the CSS size-adjust property and many other performance benefits.

CSS and font files are downloaded at build time with the rest of your static assets, and no requests are sent to Google by the browser.

Example usage:

```ts
import { Inter } from '@next/font/google';

const inter = Inter();

<html className={inter.className}>
```

### Newly improved `next/link` and `next/image` components

The `next/link` and `next/image` components are some of the core features in Next.js that got an upgrade in Next.js 13. `next/image` saw new performance upgrades such as less client-side JavaScript, faster image load times and a required alt attribute for accessibility.

The `next/link` component used for client-side transition now has a built-in `<a>` element by default. This means you won't have to manually add `<a>` like before.

Before:

```ts
import Link from 'next/link'
...
<Link href="/about">
  <a>About</a>
</Link>

```

Next.js 13:

```ts
import Link from 'next/link'
<Link href="/about">
  About // <a> will be added automatically in Next.js 13
</Link>
```

## `@vercel/og` for Open Graph Image Generation

Next.js 13 ships with a new library, `@vercel/og` that optimizes the generation of social media cards via the Open Graph Protocol. The library exports an `ImageResponse` function that generates a static image which you can then use in your `<meta>` tags.

For example, given an api route `pages/api/og.tsx`, you can generate a social card image with HTML and CSS like below:

```ts
import { ImageResponse } from "@vercel/og";

export const config = {
  runtime: "experimental-edge",
};

export default function () {
  return new ImageResponse(
    (
      <div
        style={{
          fontSize: 128,
          background: "white",
          width: "100%",
          height: "100%",
          display: "flex",
          textAlign: "center",
          alignItems: "center",
          justifyContent: "center",
        }}
      >
        Hello world!
      </div>
    ),
    {
      width: 1200,
      height: 600,
    },
  );
}
```

Now, if you navigate to `http://localhost:3000/api/og`, you should see an image generated with the text `Hello World!`.

You can read more about OG image generation [here](https://vercel.com/docs/concepts/functions/edge-functions/og-image-generation).

Some other new upgrades and additions include [Streaming](https://nextjs.org/blog/next-13#streaming) and [Middleware API](https://nextjs.org/blog/next-13#middleware-api-updates) updates.

<br/>
<div>
<a href="https://discord.gg/refine">
  <img  src="https://refine.ams3.cdn.digitaloceanspaces.com/website/static/img/discord_big_blue.png" alt="discord banner" />
</a>
</div>

## Upgrading to Next.js 13

In one of the previous sections, we talked about creating a new Next.js 13 app. In this section, we'll look at how you can upgrade your existing codebases to Next.js 13. The easiest way to upgrade your app to Next.js 13 is to run any of the following commands using your preferred package manager:

```bash
npm i next@latest react@latest react-dom@latest eslint-config-next@latest
# or
yarn upgrade next react react-dom eslint-config-next --latest
# or
pnpm up next react react-dom eslint-config-next --latest
```

If you want to only upgrade specific features, you can run a codemod transformation for those particular features. For example, the following command will upgrade links in your pages folder to Next.js 13 when run from the root of your app:

```bash
npx @next/codemod new-link ./pages
```

You can check the documentations [here](https://nextjs.org/docs/advanced-features/codemods) for the available Next.js codemods.

## Conclusion

Next.js is one of the popular tools for building the Web and it's constantly evolving. This article summarized the latest features in Next.js 13 and you learned how to upgrade your codebases to use these new features.

The documentation for Next.js 13 is currently in Beta and can be found [here](https://beta.nextjs.org/docs/getting-started), and there is now also a dark mode available! :)

With Next.js the possibilities are endless and I can't wait to see the cool stuff you build next. Happy coding! you build next. Happy coding!s and I can't wait to see the cool stuff you build next. Happy coding!
