---
title: Material UI Icons in React
description: A complete guide on using Material UI Icons in a React app
slug: mui-icons-in-react
authors: doro_onome
tags: [material-ui, react]
image: https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-08-24-mui-icons/mui-icons-social.png
hide_table_of_contents: false
---

import Tabs from '@theme/Tabs';
import TabItem from '@theme/TabItem';

## Introduction

React is the most commonly used frontend framework for creating a responsive web interface. It is accompanied by libraries that provide elements with a means of working together to maximize the aesthetic value of any web design layout effectively. Some of these libraries include Material UI, Ant Design, React-Bootstrap, Semantic UI, Fluent UI, Blueprint UI, e.t.c.

<!--truncate-->

These libraries, whether component or utility-based, perform specific roles in aiding React developers to create web structures that are responsive and pretty. For this article, we will focus on Material UI, investigate its icons and highlight their functions in a React application.

Steps we’ll cover:

- [What is Material UI?](#what-is-material-ui)
- [Getting started with Material UI icons in React](#getting-started-with-material-ui-icons-in-react)
- [How to apply Material UI icons to your project](#how-to-apply-material-ui-icons-to-your-project)
- [SvgIcon API](#svgicon-api)
- [The Icon component(Font icons)](#the-icon-componentfont-icons)
- [Building a Task Tracker with React and Material UI](#building-a-task-tracker-with-react-and-material-ui)
- [Testing Material UI Icons](#testing-material-ui-icons)

## Prerequisites

To properly understand the scope of this article, you must have:

- Thorough knowledge of HTML, CSS, and JavaScript
- A working installation of Node.Js

## What is Material UI?

Material UI is an open-source React component library based on Google’s [Material Design](https://material.io/design). It provides a complete set of UI tools to help developers build and maintain React applications and effectively import components into several parts of their projects. Over the years, more developers have incorporated Material UI into their project UI because it makes the web design process significantly easier and faster.

Material UI offers components that can serve specific purposes on a web page. Some of which include Form components, Data display components, Feedback components, Navigation components, Layout components, e.t.c. Material UI icons are a significant example of the Data display components.

## Getting started with Material UI icons in React

Material UI icons are a pre-made set of icons that can be extracted from the Material UI component system and embedded into any React application. They are commonly used in application toolbars to represent frequent operations and actions. These icons make it easier for developers to create shortcuts on the app and substitute lengthy text descriptions with easy-to-understand representations.

Material UI offers three types of icon support:

- Uniform [Material Icons](https://mui.com/material-ui/icons/#material-svg-icons) that are exported as React Components
- The [`SvgIcon`](https://mui.com/material-ui/icons/#svgicon) component - A React wrapper for custom SVG icons.
- The [`Icon`](https://mui.com/material-ui/icons/#icon-font-icons) component - A React wrapper for custom font icons.

## How to apply Material UI icons to your project

### Step 1 - Install the Material UI Library

Install the Material UI library into your project as part of your package.json dependencies with the following command:

<Tabs
defaultValue="npm"
values={[
{label: 'use npm', value: 'npm'},
{label: 'use yarn', value: 'yarn'}
]}>
<TabItem value="npm">

```bash
npm install @mui/material @emotion/react @emotion/styled
```

  </TabItem>
    <TabItem value="yarn">

```bash
yarn add @mui/material @emotion/react @emotion/styled
```

  </TabItem>
</Tabs>

### Step 2 - Install the Material UI Icons package

Install the Material UI icons themselves into your project with the following command:

<Tabs
defaultValue="npm"
values={[
{label: 'use npm', value: 'npm'},
{label: 'use yarn', value: 'yarn'}
]}>
<TabItem value="npm">

```bash
npm install @mui/icons-material
```

  </TabItem>
    <TabItem value="yarn">

```bash
yarn add @mui/icons-material
```

  </TabItem>
</Tabs>

You should now see the updated dependencies on your package.json like this:

<img src="https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-08-24-mui-icons/package.png" alt="package" />

<br/>

### Step 3 - Import the Icons

The next step would be to import the Material UI icons into your desired project directory by using one of the following techniques:

You can import each icon you intend to use for that particular React component individually like this:

```tsx
import ArrowRight from "@mui/icons-material/ArrowRight";
import Camera from "@mui/icons-material/Camera";
```

You can import them together in one declaration like this:

```tsx
import { ArrowRight, Camera } from "@mui/icons-material";
```

Each Material UI icon comes with a theme:

- Filled(The default theme)
- Outlined
- Rounded
- Two-tone
- Sharp

To use a theme other than the default, append the theme name to the icon name when importing it. For instance, you can import a Camera icon with an outlined theme like this:

```
 @material-ui/icons/CameraOutlined
```

### Step 4 - Displaying the icon on the DOM

After successfully installing the Material UI library and icons package into your application, you can call the individual icons into your JSX as a component and export them into `App.js`, ensuring that they are displayed on the virtual DOM.

For instance, if you want to display Camera icons bearing all the primary themes. You can do that with the code below:

```tsx
import React from "react";
import Grid from "@mui/material/Grid";
import Typography from "@mui/material/Typography";
import CameraIcon from "@mui/icons-material/Camera";
import CameraOutlinedIcon from "@mui/icons-material/CameraOutlined";
import CameraRoundedIcon from "@mui/icons-material/CameraRounded";
import CameraTwoToneIcon from "@mui/icons-material/CameraTwoTone";
import CameraSharpIcon from "@mui/icons-material/CameraSharp";
import ThreeSixtyIcon from "@mui/icons-material/ThreeSixty";

const Home = () => {
  return (
    <div>
      <Grid container sx={{ color: "text.primary" }}>
        <Grid item xs={4}>
          <Typography>Filled</Typography>
        </Grid>
        <Grid item xs={8}>
          <CameraIcon />
        </Grid>
        <Grid item xs={4}>
          <Typography>Outlined</Typography>
        </Grid>
        <Grid item xs={8}>
          <CameraOutlinedIcon />
        </Grid>
        <Grid item xs={4}>
          <Typography>Rounded</Typography>
        </Grid>
        <Grid item xs={8}>
          <CameraRoundedIcon />
        </Grid>
        <Grid item xs={4}>
          <Typography>Two Tone</Typography>
        </Grid>
        <Grid item xs={8}>
          <CameraTwoToneIcon />
        </Grid>
        <Grid item xs={4}>
          <Typography>Sharp</Typography>
        </Grid>
        <Grid item xs={8}>
          <CameraSharpIcon />
        </Grid>
        <Grid item xs={4}>
          <Typography>Edge-cases</Typography>
        </Grid>
        <Grid item xs={8}>
          <ThreeSixtyIcon />
        </Grid>
      </Grid>
    </div>
  );
};

export default Home;
```

The code above demonstrates how to include the Camera icon and its primary themes in a standard React project. The icons are then displayed as virtual DOM components.

Here’s the result:

<img src="https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-08-24-mui-icons/icon-dom.png" alt="iconDom" />

<br/>

## SvgIcon API

For cases where you may need a custom SVG icon that is not readily available in the Material icon specifications, you can use the `SvgIcon` wrapper. This component is an extension of the native `<svg>` element:

- It has built-in accessibility.
- SVG elements have a default `viewport` size of 24px by 24px that can customized with the `viewport` attribute.
- The component inherits the current color by default. You can, at your discretion, use the `color` prop to apply one of the theme colors.

Below is a simple representation of how to call a customized Material UI `SvgIcon` component:

```tsx
import React from "react";
import SvgIcon from "@mui/material/SvgIcon";
import CameraIcon from "./Components/CameraIcon";

const HomeIcon = (props) => {
  return (
    <div>
      <SvgIcon {...props}>
        <path d="M9.4 10.5l4.77-8.26C13.47 2.09 12.75 2 12 2c-2.4 0-4.6.85-6.32 2.25l3.66 6.35.06-.1zM21.54 9c-.92-2.92-3.15-5.26-6-6.34L11.88 9h9.66zm.26 1h-7.49l.29.5 4.76 8.25C21 16.97 22 14.61 22 12c0-.69-.07-1.35-.2-2zM8.54 12l-3.9-6.75C3.01 7.03 2 9.39 2 12c0 .69.07 1.35.2 2h7.49l-1.15-2zm-6.08 3c.92 2.92 3.15 5.26 6 6.34L12.12 15H2.46zm11.27 0l-3.9 6.76c.7.15 1.42.24 2.17.24 2.4 0 4.6-.85 6.32-2.25l-3.66-6.35-.93 1.6z" />
      </SvgIcon>
    </div>
  );
};

// And the code for the color setting would look the following way:
<div className={Class.root}>
  <CameraIcon />
  <CameraIcon color="primary" />
  <CameraIcon color="secondary" />
  <CameraIcon color="action" />
  <CameraIcon color="disabled" />
</div>;

export default HomeIcon;
```

Here’s the result:

<img src="https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-08-24-mui-icons/svg-icon.png" alt="svgIcon" />

<br/>

## The Icon component(Font icons)

The `Icon` component displays any icon font that supports ligatures. To use an icon, just enclose its name in the `Icon` component's font ligature.

Here’s a simple illustration of how to import font icons into your React app with the `Icon` component:

```tsx
import * as React from "react";
import Box from "@mui/material/Box";
import { green } from "@mui/material/colors";
import Icon from "@mui/material/Icon";

export default function Icons() {
  return (
    <Box
      sx={{
        "& > :not(style)": {
          m: 2,
        },
      }}
    >
      <Icon>camera</Icon>
      <Icon color="primary">camera</Icon>
      <Icon sx={{ color: green[500] }}>camera</Icon>
      <Icon fontSize="small">camera</Icon>
      <Icon sx={{ fontSize: 30 }}>camera</Icon>
    </Box>
  );
}
```

Here’s the result

<img src="https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-08-24-mui-icons/icon-icon.png" alt="iconIcon" />

<br/>
<br/>

Notice how we could customize the `font size of some of the icons.

You can also use FontAwesome icons like this:

```tsx
<FontAwesomeIcon icon="fa-solid fa-aperture" />
```

## Building a Task Tracker with React and Material UI

Material UI icons are a very dynamic element of web design. You can use them to represent commands, actions and directories in a React application. We can showcase their uses and functions in a simple application like a task tracker.

The task tracker will have three components:

- The `Form` component
- The `Task` component
- The `TaskList` component

### The Form Component

This component includes an input element where users can type their tasks, as well as an add button represented by Material UI's `AddCircleIcon.` It also has a `Select` element where users can customize the task tracker by displaying tasks that have been completed or those that have yet to be completed.

Here’s the code for the `Form` component

```tsx
import React, { useState } from "react";
import AddCircleIcon from "@mui/icons-material/AddCircle";

const Form = () => {
  const [inputText, setInputText] = useState("");
  const [todos, setTodos] = useState([]);
  const [status, setStatus] = useState("All");
  const [filteredTodos, setFilteredTodos] = useState([]);

  const inputTextHandler = (e) => {
    setInputText(e.target.value);
  };

  const submitTodoHandler = (e) => {
    e.preventDefault();

    setTodos([
      ...todos,
      {
        text: inputText,
        completed: false,
        id: Math.floor(Math.random() * 10000),
      },
    ]);

    setInputText("");
  };

  const statusHandler = (e) => {
    setStatus(e.target.value);
  };

  return (
    <section>
      <form className="form-div">
        <div>
          <input
            type="text"
            placeholder="Enter New Task Here"
            value={inputText}
            className="input"
            onChange={inputTextHandler}
          />
          <AddCircleIcon onClick={submitTodoHandler} className="add-icon" />
          <select onChange={statusHandler}>
            <option>All</option>
            <option>Completed</option>
            <option>Uncompleted</option>
          </select>
        </div>
      </form>
    </section>
  );
};

export default Form;
```

Here’s the `Form` component:

<img src="https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-08-24-mui-icons/task-form.png" alt="task form" />

<br/>

<br/>
<div>
<a href="https://discord.gg/refine">
  <img  src="https://refine.ams3.cdn.digitaloceanspaces.com/website/static/img/discord_big_blue.png" alt="discord banner" />
</a>
</div>

### The Task Component

This component displays the user interface and functionalities within the individual tasks that the user specifies. You can delete tasks from the Material UI by clicking the `DeleteIcon.` By clicking the `CheckCircleIcon` from Material UI, you can mark a "todo" as "completed."

Here’s the code for the `Task` component:

```tsx
import React, { useState } from "react";
import CheckCircleIcon from "@mui/icons-material/CheckCircle";
import DeleteIcon from "@mui/icons-material/Delete";

const Task = () => {
  const [todos, setTodos] = useState([]);
  const [filteredTodos, setFilteredTodos] = useState([]);

  const deleteHandler = () => {
    setTodos(todos.filter((el) => el.id !== todo.id));
  };

  const completeHandler = () => {
    setTodos(
      todos.map((item) => {
        if (item.id === todo.id) {
          return {
            ...item,
            completed: !item.completed,
          };
        }
        return item;
      }),
    );
  };

  return (
    <section className="task-section">
      <div className="class-list">
        <p className={`${todo.completed ? "pargh2" : "pargh"}`}>{text}</p>
      </div>
      <div>
        <button onClick={completeHandler} className="btn-2">
          <CheckCircleIcon className="icon2" />
        </button>
        <button onClick={deleteHandler} className="btn-1">
          <DeleteIcon className="icon1" />
        </button>
      </div>
    </section>
  );
};

export default Task;
```

Here’s the `Task` component

<img src="https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-08-24-mui-icons/task-component.png" alt="task component" />

<br/>

### The TaskList Component

This component maps the various tasks created by the user and displays them as an unordered list in the DOM.

Here’s the code:

```tsx
import React from "react";
import Task from "./Task";

const Tasklist = () => {
  const [todos, setTodos] = useState([]);
  const [filteredTodos, setFilteredTodos] = useState([]);

  return (
    <div>
      <ul>
        {filteredTodos.map((todo) => (
          <Task
            text={todo.text}
            key={todo.id}
            todos={todos}
            setTodos={setTodos}
            todo={todo}
          />
        ))}
      </ul>
    </div>
  );
};

export default Tasklist;
```

Here’s the final result:

<img src="https://refine.ams3.cdn.digitaloceanspaces.com/blog/2022-08-24-mui-icons/tasklist.png" alt="task list" />

<br/>

## Testing Material UI Icons

Material UI provides a wide range of regression tests. Material UI components are tested internally. `@testing-library/react` is a library with a first-class API for this approach. Material UI icons exported from `@mui/icons-material` have a `data-testid` attribute for testing purposes.

For instance:

```tsx
import CameraIcon from "@mui/icons-material/Camera";
```

Once mounted, it has the following property:

```tsx
<svg data-testid="CameraIcon"></svg>
```

## Conclusion

This article reviewed React Material UI and explored its icons, installation, and application processes. We also highlighted a possible use case in a task-tracking application.
You can access the source code on [Github Repo](https://github.com/refinedev/task-manager-with-react-blog).

You can also see the deployed application [here](https://task-manager-with-react.vercel.app/).
