export const getOrgEntity = (
  name: string
): {
  name: string;
  latex: string;
  requireLatexMath: boolean;
  html: string;
  ascii: string;
  latin1: string;
  utf8: string;
} | null => {
  const e = orgEntities.find((x) => x[0] === name);
  if (!e) return null;

  return {
    name,
    latex: e[1],
    requireLatexMath: e[2],
    html: e[3],
    ascii: e[4],
    latin1: e[5],
    utf8: e[6],
  };
};

// (json-encode org-entities)
const orgEntities: [string, string, boolean, string, string, string, string][] =
  [
    // * Letters
    // ** Latin
    ['Agrave', '\\`{A}', false, '&Agrave;', 'A', 'À', 'À'],
    ['agrave', '\\`{a}', false, '&agrave;', 'a', 'à', 'à'],
    ['Aacute', "\\'{A}", false, '&Aacute;', 'A', 'Á', 'Á'],
    ['aacute', "\\'{a}", false, '&aacute;', 'a', 'á', 'á'],
    ['Acirc', '\\^{A}', false, '&Acirc;', 'A', 'Â', 'Â'],
    ['acirc', '\\^{a}', false, '&acirc;', 'a', 'â', 'â'],
    ['Amacr', '\\bar{A}', false, '&Amacr;', 'A', 'Ã', 'Ã'],
    ['amacr', '\\bar{a}', false, '&amacr;', 'a', 'ã', 'ã'],
    ['Atilde', '\\~{A}', false, '&Atilde;', 'A', 'Ã', 'Ã'],
    ['atilde', '\\~{a}', false, '&atilde;', 'a', 'ã', 'ã'],
    ['Auml', '\\"{A}', false, '&Auml;', 'Ae', 'Ä', 'Ä'],
    ['auml', '\\"{a}', false, '&auml;', 'ae', 'ä', 'ä'],
    ['Aring', '\\AA{}', false, '&Aring;', 'A', 'Å', 'Å'],
    ['AA', '\\AA{}', false, '&Aring;', 'A', 'Å', 'Å'],
    ['aring', '\\aa{}', false, '&aring;', 'a', 'å', 'å'],
    ['AElig', '\\AE{}', false, '&AElig;', 'AE', 'Æ', 'Æ'],
    ['aelig', '\\ae{}', false, '&aelig;', 'ae', 'æ', 'æ'],
    ['Ccedil', '\\c{C}', false, '&Ccedil;', 'C', 'Ç', 'Ç'],
    ['ccedil', '\\c{c}', false, '&ccedil;', 'c', 'ç', 'ç'],
    ['Egrave', '\\`{E}', false, '&Egrave;', 'E', 'È', 'È'],
    ['egrave', '\\`{e}', false, '&egrave;', 'e', 'è', 'è'],
    ['Eacute', "\\'{E}", false, '&Eacute;', 'E', 'É', 'É'],
    ['eacute', "\\'{e}", false, '&eacute;', 'e', 'é', 'é'],
    ['Ecirc', '\\^{E}', false, '&Ecirc;', 'E', 'Ê', 'Ê'],
    ['ecirc', '\\^{e}', false, '&ecirc;', 'e', 'ê', 'ê'],
    ['Euml', '\\"{E}', false, '&Euml;', 'E', 'Ë', 'Ë'],
    ['euml', '\\"{e}', false, '&euml;', 'e', 'ë', 'ë'],
    ['Igrave', '\\`{I}', false, '&Igrave;', 'I', 'Ì', 'Ì'],
    ['igrave', '\\`{i}', false, '&igrave;', 'i', 'ì', 'ì'],
    ['Iacute', "\\'{I}", false, '&Iacute;', 'I', 'Í', 'Í'],
    ['iacute', "\\'{i}", false, '&iacute;', 'i', 'í', 'í'],
    ['Icirc', '\\^{I}', false, '&Icirc;', 'I', 'Î', 'Î'],
    ['icirc', '\\^{i}', false, '&icirc;', 'i', 'î', 'î'],
    ['Iuml', '\\"{I}', false, '&Iuml;', 'I', 'Ï', 'Ï'],
    ['iuml', '\\"{i}', false, '&iuml;', 'i', 'ï', 'ï'],
    ['Ntilde', '\\~{N}', false, '&Ntilde;', 'N', 'Ñ', 'Ñ'],
    ['ntilde', '\\~{n}', false, '&ntilde;', 'n', 'ñ', 'ñ'],
    ['Ograve', '\\`{O}', false, '&Ograve;', 'O', 'Ò', 'Ò'],
    ['ograve', '\\`{o}', false, '&ograve;', 'o', 'ò', 'ò'],
    ['Oacute', "\\'{O}", false, '&Oacute;', 'O', 'Ó', 'Ó'],
    ['oacute', "\\'{o}", false, '&oacute;', 'o', 'ó', 'ó'],
    ['Ocirc', '\\^{O}', false, '&Ocirc;', 'O', 'Ô', 'Ô'],
    ['ocirc', '\\^{o}', false, '&ocirc;', 'o', 'ô', 'ô'],
    ['Otilde', '\\~{O}', false, '&Otilde;', 'O', 'Õ', 'Õ'],
    ['otilde', '\\~{o}', false, '&otilde;', 'o', 'õ', 'õ'],
    ['Ouml', '\\"{O}', false, '&Ouml;', 'Oe', 'Ö', 'Ö'],
    ['ouml', '\\"{o}', false, '&ouml;', 'oe', 'ö', 'ö'],
    ['Oslash', '\\O', false, '&Oslash;', 'O', 'Ø', 'Ø'],
    ['oslash', '\\o{}', false, '&oslash;', 'o', 'ø', 'ø'],
    ['OElig', '\\OE{}', false, '&OElig;', 'OE', 'OE', 'Œ'],
    ['oelig', '\\oe{}', false, '&oelig;', 'oe', 'oe', 'œ'],
    ['Scaron', '\\v{S}', false, '&Scaron;', 'S', 'S', 'Š'],
    ['scaron', '\\v{s}', false, '&scaron;', 's', 's', 'š'],
    ['szlig', '\\ss{}', false, '&szlig;', 'ss', 'ß', 'ß'],
    ['Ugrave', '\\`{U}', false, '&Ugrave;', 'U', 'Ù', 'Ù'],
    ['ugrave', '\\`{u}', false, '&ugrave;', 'u', 'ù', 'ù'],
    ['Uacute', "\\'{U}", false, '&Uacute;', 'U', 'Ú', 'Ú'],
    ['uacute', "\\'{u}", false, '&uacute;', 'u', 'ú', 'ú'],
    ['Ucirc', '\\^{U}', false, '&Ucirc;', 'U', 'Û', 'Û'],
    ['ucirc', '\\^{u}', false, '&ucirc;', 'u', 'û', 'û'],
    ['Uuml', '\\"{U}', false, '&Uuml;', 'Ue', 'Ü', 'Ü'],
    ['uuml', '\\"{u}', false, '&uuml;', 'ue', 'ü', 'ü'],
    ['Yacute', "\\'{Y}", false, '&Yacute;', 'Y', 'Ý', 'Ý'],
    ['yacute', "\\'{y}", false, '&yacute;', 'y', 'ý', 'ý'],
    ['Yuml', '\\"{Y}', false, '&Yuml;', 'Y', 'Y', 'Ÿ'],
    ['yuml', '\\"{y}', false, '&yuml;', 'y', 'ÿ', 'ÿ'],
    // ** Latin (special face)
    ['fnof', '\\textit{f}', false, '&fnof;', 'f', 'f', 'ƒ'],
    ['real', '\\Re', true, '&real;', 'R', 'R', 'ℜ'],
    ['image', '\\Im', true, '&image;', 'I', 'I', 'ℑ'],
    ['weierp', '\\wp', true, '&weierp;', 'P', 'P', '℘'],
    ['ell', '\\ell', true, '&ell;', 'ell', 'ell', 'ℓ'],
    ['imath', '\\imath', true, '&imath;', '[dotless i]', 'dotless i', 'ı'],
    ['jmath', '\\jmath', true, '&jmath;', '[dotless j]', 'dotless j', 'ȷ'],
    // ** Greek
    ['Alpha', 'A', false, '&Alpha;', 'Alpha', 'Alpha', 'Α'],
    ['alpha', '\\alpha', true, '&alpha;', 'alpha', 'alpha', 'α'],
    ['Beta', 'B', false, '&Beta;', 'Beta', 'Beta', 'Β'],
    ['beta', '\\beta', true, '&beta;', 'beta', 'beta', 'β'],
    ['Gamma', '\\Gamma', true, '&Gamma;', 'Gamma', 'Gamma', 'Γ'],
    ['gamma', '\\gamma', true, '&gamma;', 'gamma', 'gamma', 'γ'],
    ['Delta', '\\Delta', true, '&Delta;', 'Delta', 'Delta', 'Δ'],
    ['delta', '\\delta', true, '&delta;', 'delta', 'delta', 'δ'],
    ['Epsilon', 'E', false, '&Epsilon;', 'Epsilon', 'Epsilon', 'Ε'],
    ['epsilon', '\\epsilon', true, '&epsilon;', 'epsilon', 'epsilon', 'ε'],
    [
      'varepsilon',
      '\\varepsilon',
      true,
      '&epsilon;',
      'varepsilon',
      'varepsilon',
      'ε',
    ],
    ['Zeta', 'Z', false, '&Zeta;', 'Zeta', 'Zeta', 'Ζ'],
    ['zeta', '\\zeta', true, '&zeta;', 'zeta', 'zeta', 'ζ'],
    ['Eta', 'H', false, '&Eta;', 'Eta', 'Eta', 'Η'],
    ['eta', '\\eta', true, '&eta;', 'eta', 'eta', 'η'],
    ['Theta', '\\Theta', true, '&Theta;', 'Theta', 'Theta', 'Θ'],
    ['theta', '\\theta', true, '&theta;', 'theta', 'theta', 'θ'],
    ['thetasym', '\\vartheta', true, '&thetasym;', 'theta', 'theta', 'ϑ'],
    ['vartheta', '\\vartheta', true, '&thetasym;', 'theta', 'theta', 'ϑ'],
    ['Iota', 'I', false, '&Iota;', 'Iota', 'Iota', 'Ι'],
    ['iota', '\\iota', true, '&iota;', 'iota', 'iota', 'ι'],
    ['Kappa', 'K', false, '&Kappa;', 'Kappa', 'Kappa', 'Κ'],
    ['kappa', '\\kappa', true, '&kappa;', 'kappa', 'kappa', 'κ'],
    ['Lambda', '\\Lambda', true, '&Lambda;', 'Lambda', 'Lambda', 'Λ'],
    ['lambda', '\\lambda', true, '&lambda;', 'lambda', 'lambda', 'λ'],
    ['Mu', 'M', false, '&Mu;', 'Mu', 'Mu', 'Μ'],
    ['mu', '\\mu', true, '&mu;', 'mu', 'mu', 'μ'],
    ['nu', '\\nu', true, '&nu;', 'nu', 'nu', 'ν'],
    ['Nu', 'N', false, '&Nu;', 'Nu', 'Nu', 'Ν'],
    ['Xi', '\\Xi', true, '&Xi;', 'Xi', 'Xi', 'Ξ'],
    ['xi', '\\xi', true, '&xi;', 'xi', 'xi', 'ξ'],
    ['Omicron', 'O', false, '&Omicron;', 'Omicron', 'Omicron', 'Ο'],
    ['omicron', '\\textit{o}', false, '&omicron;', 'omicron', 'omicron', 'ο'],
    ['Pi', '\\Pi', true, '&Pi;', 'Pi', 'Pi', 'Π'],
    ['pi', '\\pi', true, '&pi;', 'pi', 'pi', 'π'],
    ['Rho', 'P', false, '&Rho;', 'Rho', 'Rho', 'Ρ'],
    ['rho', '\\rho', true, '&rho;', 'rho', 'rho', 'ρ'],
    ['Sigma', '\\Sigma', true, '&Sigma;', 'Sigma', 'Sigma', 'Σ'],
    ['sigma', '\\sigma', true, '&sigma;', 'sigma', 'sigma', 'σ'],
    ['sigmaf', '\\varsigma', true, '&sigmaf;', 'sigmaf', 'sigmaf', 'ς'],
    ['varsigma', '\\varsigma', true, '&sigmaf;', 'varsigma', 'varsigma', 'ς'],
    ['Tau', 'T', false, '&Tau;', 'Tau', 'Tau', 'Τ'],
    ['Upsilon', '\\Upsilon', true, '&Upsilon;', 'Upsilon', 'Upsilon', 'Υ'],
    ['upsih', '\\Upsilon', true, '&upsih;', 'upsilon', 'upsilon', 'ϒ'],
    ['upsilon', '\\upsilon', true, '&upsilon;', 'upsilon', 'upsilon', 'υ'],
    ['Phi', '\\Phi', true, '&Phi;', 'Phi', 'Phi', 'Φ'],
    ['phi', '\\phi', true, '&phi;', 'phi', 'phi', 'ɸ'],
    ['varphi', '\\varphi', true, '&varphi;', 'varphi', 'varphi', 'φ'],
    ['Chi', 'X', false, '&Chi;', 'Chi', 'Chi', 'Χ'],
    ['chi', '\\chi', true, '&chi;', 'chi', 'chi', 'χ'],
    ['acutex', '\\acute x', true, '&acute;x', "'x", "'x", '𝑥́'],
    ['Psi', '\\Psi', true, '&Psi;', 'Psi', 'Psi', 'Ψ'],
    ['psi', '\\psi', true, '&psi;', 'psi', 'psi', 'ψ'],
    ['tau', '\\tau', true, '&tau;', 'tau', 'tau', 'τ'],
    ['Omega', '\\Omega', true, '&Omega;', 'Omega', 'Omega', 'Ω'],
    ['omega', '\\omega', true, '&omega;', 'omega', 'omega', 'ω'],
    ['piv', '\\varpi', true, '&piv;', 'omega-pi', 'omega-pi', 'ϖ'],
    ['varpi', '\\varpi', true, '&piv;', 'omega-pi', 'omega-pi', 'ϖ'],
    [
      'partial',
      '\\partial',
      true,
      '&part;',
      '[partial differential]',
      '[partial differential]',
      '∂',
    ],
    // ** Hebrew
    ['alefsym', '\\aleph', true, '&alefsym;', 'aleph', 'aleph', 'ℵ'],
    ['aleph', '\\aleph', true, '&aleph;', 'aleph', 'aleph', 'ℵ'],
    ['gimel', '\\gimel', true, '&gimel;', 'gimel', 'gimel', 'ℷ'],
    ['beth', '\\beth', true, '&beth;', 'beth', 'beth', 'ב'],
    ['dalet', '\\daleth', true, '&daleth;', 'dalet', 'dalet', 'ד'],
    // ** Icelandic
    ['ETH', '\\DH{}', false, '&ETH;', 'D', 'Ð', 'Ð'],
    ['eth', '\\dh{}', false, '&eth;', 'dh', 'ð', 'ð'],
    ['THORN', '\\TH{}', false, '&THORN;', 'TH', 'Þ', 'Þ'],
    ['thorn', '\\th{}', false, '&thorn;', 'th', 'þ', 'þ'],
    // * Punctuation
    // ** Dots and Marks
    ['dots', '\\dots{}', false, '&hellip;', '...', '...', '…'],
    ['cdots', '\\cdots{}', true, '&ctdot;', '...', '...', '⋯'],
    ['hellip', '\\dots{}', false, '&hellip;', '...', '...', '…'],
    ['middot', '\\textperiodcentered{}', false, '&middot;', '.', '·', '·'],
    ['iexcl', '!`', false, '&iexcl;', '!', '¡', '¡'],
    ['iquest', '?`', false, '&iquest;', '?', '¿', '¿'],
    // ** Dash-like
    ['shy', '\\-', false, '&shy;', '', '', ''],
    ['ndash', '--', false, '&ndash;', '-', '-', '–'],
    ['mdash', '---', false, '&mdash;', '--', '--', '—'],
    // ** Quotations
    ['quot', '\\textquotedbl{}', false, '&quot;', '"', '"', '"'],
    ['acute', '\\textasciiacute{}', false, '&acute;', "'", '´', '´'],
    ['ldquo', '\\textquotedblleft{}', false, '&ldquo;', '"', '"', '“'],
    ['rdquo', '\\textquotedblright{}', false, '&rdquo;', '"', '"', '”'],
    ['bdquo', '\\quotedblbase{}', false, '&bdquo;', '"', '"', '„'],
    ['lsquo', '\\textquoteleft{}', false, '&lsquo;', '`', '`', '‘'],
    ['rsquo', '\\textquoteright{}', false, '&rsquo;', "'", "'", '’'],
    ['sbquo', '\\quotesinglbase{}', false, '&sbquo;', ',', ',', '‚'],
    ['laquo', '\\guillemotleft{}', false, '&laquo;', '<<', '«', '«'],
    ['raquo', '\\guillemotright{}', false, '&raquo;', '>>', '»', '»'],
    ['lsaquo', '\\guilsinglleft{}', false, '&lsaquo;', '<', '<', '‹'],
    ['rsaquo', '\\guilsinglright{}', false, '&rsaquo;', '>', '>', '›'],
    // * Other
    // ** Misc. (often used)
    ['circ', '\\^{}', false, '&circ;', '^', '^', '∘'],
    ['vert', '\\vert{}', true, '&vert;', '|', '|', '|'],
    ['vbar', '|', false, '|', '|', '|', '|'],
    ['brvbar', '\\textbrokenbar{}', false, '&brvbar;', '|', '¦', '¦'],
    ['S', '\\S', false, '&sect;', 'paragraph', '§', '§'],
    ['sect', '\\S', false, '&sect;', 'paragraph', '§', '§'],
    ['amp', '\\&', false, '&amp;', '&', '&', '&'],
    ['lt', '\\textless{}', false, '&lt;', '<', '<', '<'],
    ['gt', '\\textgreater{}', false, '&gt;', '>', '>', '>'],
    ['tilde', '\\textasciitilde{}', false, '~', '~', '~', '~'],
    ['slash', '/', false, '/', '/', '/', '/'],
    ['plus', '+', false, '+', '+', '+', '+'],
    ['under', '\\_', false, '_', '_', '_', '_'],
    ['equal', '=', false, '=', '=', '=', '='],
    ['asciicirc', '\\textasciicircum{}', false, '^', '^', '^', '^'],
    [
      'dagger',
      '\\textdagger{}',
      false,
      '&dagger;',
      '[dagger]',
      '[dagger]',
      '†',
    ],
    ['dag', '\\dag{}', false, '&dagger;', '[dagger]', '[dagger]', '†'],
    [
      'Dagger',
      '\\textdaggerdbl{}',
      false,
      '&Dagger;',
      '[doubledagger]',
      '[doubledagger]',
      '‡',
    ],
    [
      'ddag',
      '\\ddag{}',
      false,
      '&Dagger;',
      '[doubledagger]',
      '[doubledagger]',
      '‡',
    ],
    // ** Whitespace
    ['nbsp', '~', false, '&nbsp;', ' ', ' ', ' '],
    ['ensp', '\\hspace*{.5em}', false, '&ensp;', ' ', ' ', ' '],
    ['emsp', '\\hspace*{1em}', false, '&emsp;', ' ', ' ', ' '],
    ['thinsp', '\\hspace*{.2em}', false, '&thinsp;', ' ', ' ', ' '],
    // ** Currency
    ['curren', '\\textcurrency{}', false, '&curren;', 'curr.', '¤', '¤'],
    ['cent', '\\textcent{}', false, '&cent;', 'cent', '¢', '¢'],
    ['pound', '\\pounds{}', false, '&pound;', 'pound', '£', '£'],
    ['yen', '\\textyen{}', false, '&yen;', 'yen', '¥', '¥'],
    ['euro', '\\texteuro{}', false, '&euro;', 'EUR', 'EUR', '€'],
    ['EUR', '\\texteuro{}', false, '&euro;', 'EUR', 'EUR', '€'],
    ['dollar', '\\$', false, '$', '$', '$', '$'],
    ['USD', '\\$', false, '$', '$', '$', '$'],
    // ** Property Marks
    ['copy', '\\textcopyright{}', false, '&copy;', '(c)', '©', '©'],
    ['reg', '\\textregistered{}', false, '&reg;', '(r)', '®', '®'],
    ['trade', '\\texttrademark{}', false, '&trade;', 'TM', 'TM', '™'],
    // ** Science et al.
    ['minus', '\\minus', true, '&minus;', '-', '-', '−'],
    ['pm', '\\textpm{}', false, '&plusmn;', '+-', '±', '±'],
    ['plusmn', '\\textpm{}', false, '&plusmn;', '+-', '±', '±'],
    ['times', '\\texttimes{}', false, '&times;', '*', '×', '×'],
    ['frasl', '/', false, '&frasl;', '/', '/', '⁄'],
    ['colon', '\\colon', true, ':', ':', ':', ':'],
    ['div', '\\textdiv{}', false, '&divide;', '/', '÷', '÷'],
    ['frac12', '\\textonehalf{}', false, '&frac12;', '1/2', '½', '½'],
    ['frac14', '\\textonequarter{}', false, '&frac14;', '1/4', '¼', '¼'],
    ['frac34', '\\textthreequarters{}', false, '&frac34;', '3/4', '¾', '¾'],
    [
      'permil',
      '\\textperthousand{}',
      false,
      '&permil;',
      'per thousand',
      'per thousand',
      '‰',
    ],
    ['sup1', '\\textonesuperior{}', false, '&sup1;', '^1', '¹', '¹'],
    ['sup2', '\\texttwosuperior{}', false, '&sup2;', '^2', '²', '²'],
    ['sup3', '\\textthreesuperior{}', false, '&sup3;', '^3', '³', '³'],
    [
      'radic',
      '\\sqrt{\\,}',
      true,
      '&radic;',
      '[square root]',
      '[square root]',
      '√',
    ],
    ['sum', '\\sum', true, '&sum;', '[sum]', '[sum]', '∑'],
    ['prod', '\\prod', true, '&prod;', '[product]', '[n-ary product]', '∏'],
    ['micro', '\\textmu{}', false, '&micro;', 'micro', 'µ', 'µ'],
    ['macr', '\\textasciimacron{}', false, '&macr;', '[macron]', '¯', '¯'],
    ['deg', '\\textdegree{}', false, '&deg;', 'degree', '°', '°'],
    ['prime', '\\prime', true, '&prime;', "'", "'", '′'],
    ['Prime', '\\prime{}\\prime', true, '&Prime;', "''", "''", '″'],
    ['infin', '\\infty', true, '&infin;', '[infinity]', '[infinity]', '∞'],
    ['infty', '\\infty', true, '&infin;', '[infinity]', '[infinity]', '∞'],
    [
      'prop',
      '\\propto',
      true,
      '&prop;',
      '[proportional to]',
      '[proportional to]',
      '∝',
    ],
    [
      'propto',
      '\\propto',
      true,
      '&prop;',
      '[proportional to]',
      '[proportional to]',
      '∝',
    ],
    ['not', '\\textlnot{}', false, '&not;', '[angled dash]', '¬', '¬'],
    ['neg', '\\neg{}', true, '&not;', '[angled dash]', '¬', '¬'],
    ['land', '\\land', true, '&and;', '[logical and]', '[logical and]', '∧'],
    ['wedge', '\\wedge', true, '&and;', '[logical and]', '[logical and]', '∧'],
    ['lor', '\\lor', true, '&or;', '[logical or]', '[logical or]', '∨'],
    ['vee', '\\vee', true, '&or;', '[logical or]', '[logical or]', '∨'],
    ['cap', '\\cap', true, '&cap;', '[intersection]', '[intersection]', '∩'],
    ['cup', '\\cup', true, '&cup;', '[union]', '[union]', '∪'],
    [
      'smile',
      '\\smile',
      true,
      '&smile;',
      '[cup product]',
      '[cup product]',
      '⌣',
    ],
    [
      'frown',
      '\\frown',
      true,
      '&frown;',
      '[Cap product]',
      '[cap product]',
      '⌢',
    ],
    ['int', '\\int', true, '&int;', '[integral]', '[integral]', '∫'],
    [
      'therefore',
      '\\therefore',
      true,
      '&there4;',
      '[therefore]',
      '[therefore]',
      '∴',
    ],
    [
      'there4',
      '\\therefore',
      true,
      '&there4;',
      '[therefore]',
      '[therefore]',
      '∴',
    ],
    ['because', '\\because', true, '&because;', '[because]', '[because]', '∵'],
    ['sim', '\\sim', true, '&sim;', '~', '~', '∼'],
    [
      'cong',
      '\\cong',
      true,
      '&cong;',
      '[approx. equal to]',
      '[approx. equal to]',
      '≅',
    ],
    [
      'simeq',
      '\\simeq',
      true,
      '&cong;',
      '[approx. equal to]',
      '[approx. equal to]',
      '≅',
    ],
    [
      'asymp',
      '\\asymp',
      true,
      '&asymp;',
      '[almost equal to]',
      '[almost equal to]',
      '≈',
    ],
    [
      'approx',
      '\\approx',
      true,
      '&asymp;',
      '[almost equal to]',
      '[almost equal to]',
      '≈',
    ],
    ['ne', '\\ne', true, '&ne;', '[not equal to]', '[not equal to]', '≠'],
    ['neq', '\\neq', true, '&ne;', '[not equal to]', '[not equal to]', '≠'],
    [
      'equiv',
      '\\equiv',
      true,
      '&equiv;',
      '[identical to]',
      '[identical to]',
      '≡',
    ],
    [
      'triangleq',
      '\\triangleq',
      true,
      '&triangleq;',
      '[defined to]',
      '[defined to]',
      '≜',
    ],
    ['le', '\\le', true, '&le;', '<=', '<=', '≤'],
    ['leq', '\\le', true, '&le;', '<=', '<=', '≤'],
    ['ge', '\\ge', true, '&ge;', '>=', '>=', '≥'],
    ['geq', '\\ge', true, '&ge;', '>=', '>=', '≥'],
    [
      'lessgtr',
      '\\lessgtr',
      true,
      '&lessgtr;',
      '[less than or greater than]',
      '[less than or greater than]',
      '≶',
    ],
    [
      'lesseqgtr',
      '\\lesseqgtr',
      true,
      '&lesseqgtr;',
      '[less than or equal or greater than or equal]',
      '[less than or equal or greater than or equal]',
      '⋚',
    ],
    ['ll', '\\ll', true, '&Lt;', '<<', '<<', '≪'],
    ['Ll', '\\lll', true, '&Ll;', '<<<', '<<<', '⋘'],
    ['lll', '\\lll', true, '&Ll;', '<<<', '<<<', '⋘'],
    ['gg', '\\gg', true, '&Gt;', '>>', '>>', '≫'],
    ['Gg', '\\ggg', true, '&Gg;', '>>>', '>>>', '⋙'],
    ['ggg', '\\ggg', true, '&Gg;', '>>>', '>>>', '⋙'],
    ['prec', '\\prec', true, '&pr;', '[precedes]', '[precedes]', '≺'],
    [
      'preceq',
      '\\preceq',
      true,
      '&prcue;',
      '[precedes or equal]',
      '[precedes or equal]',
      '≼',
    ],
    [
      'preccurlyeq',
      '\\preccurlyeq',
      true,
      '&prcue;',
      '[precedes or equal]',
      '[precedes or equal]',
      '≼',
    ],
    ['succ', '\\succ', true, '&sc;', '[succeeds]', '[succeeds]', '≻'],
    [
      'succeq',
      '\\succeq',
      true,
      '&sccue;',
      '[succeeds or equal]',
      '[succeeds or equal]',
      '≽',
    ],
    [
      'succcurlyeq',
      '\\succcurlyeq',
      true,
      '&sccue;',
      '[succeeds or equal]',
      '[succeeds or equal]',
      '≽',
    ],
    ['sub', '\\subset', true, '&sub;', '[subset of]', '[subset of]', '⊂'],
    ['subset', '\\subset', true, '&sub;', '[subset of]', '[subset of]', '⊂'],
    ['sup', '\\supset', true, '&sup;', '[superset of]', '[superset of]', '⊃'],
    [
      'supset',
      '\\supset',
      true,
      '&sup;',
      '[superset of]',
      '[superset of]',
      '⊃',
    ],
    [
      'nsub',
      '\\not\\subset',
      true,
      '&nsub;',
      '[not a subset of]',
      '[not a subset of',
      '⊄',
    ],
    [
      'sube',
      '\\subseteq',
      true,
      '&sube;',
      '[subset of or equal to]',
      '[subset of or equal to]',
      '⊆',
    ],
    [
      'nsup',
      '\\not\\supset',
      true,
      '&nsup;',
      '[not a superset of]',
      '[not a superset of]',
      '⊅',
    ],
    [
      'supe',
      '\\supseteq',
      true,
      '&supe;',
      '[superset of or equal to]',
      '[superset of or equal to]',
      '⊇',
    ],
    ['setminus', '\\setminus', true, '&setminus;', '" ', '"', '⧵'],
    ['forall', '\\forall', true, '&forall;', '[for all]', '[for all]', '∀'],
    [
      'exist',
      '\\exists',
      true,
      '&exist;',
      '[there exists]',
      '[there exists]',
      '∃',
    ],
    [
      'exists',
      '\\exists',
      true,
      '&exist;',
      '[there exists]',
      '[there exists]',
      '∃',
    ],
    [
      'nexist',
      '\\nexists',
      true,
      '&exist;',
      '[there does not exists]',
      '[there does not  exists]',
      '∄',
    ],
    [
      'nexists',
      '\\nexists',
      true,
      '&exist;',
      '[there does not exists]',
      '[there does not  exists]',
      '∄',
    ],
    ['empty', '\\emptyset', true, '&empty;', '[empty set]', '[empty set]', '∅'],
    [
      'emptyset',
      '\\emptyset',
      true,
      '&empty;',
      '[empty set]',
      '[empty set]',
      '∅',
    ],
    ['isin', '\\in', true, '&isin;', '[element of]', '[element of]', '∈'],
    ['in', '\\in', true, '&isin;', '[element of]', '[element of]', '∈'],
    [
      'notin',
      '\\notin',
      true,
      '&notin;',
      '[not an element of]',
      '[not an element of]',
      '∉',
    ],
    [
      'ni',
      '\\ni',
      true,
      '&ni;',
      '[contains as member]',
      '[contains as member]',
      '∋',
    ],
    ['nabla', '\\nabla', true, '&nabla;', '[nabla]', '[nabla]', '∇'],
    ['ang', '\\angle', true, '&ang;', '[angle]', '[angle]', '∠'],
    ['angle', '\\angle', true, '&ang;', '[angle]', '[angle]', '∠'],
    ['perp', '\\perp', true, '&perp;', '[up tack]', '[up tack]', '⊥'],
    ['parallel', '\\parallel', true, '&parallel;', '||', '||', '∥'],
    ['sdot', '\\cdot', true, '&sdot;', '[dot]', '[dot]', '⋅'],
    ['cdot', '\\cdot', true, '&sdot;', '[dot]', '[dot]', '⋅'],
    [
      'lceil',
      '\\lceil',
      true,
      '&lceil;',
      '[left ceiling]',
      '[left ceiling]',
      '⌈',
    ],
    [
      'rceil',
      '\\rceil',
      true,
      '&rceil;',
      '[right ceiling]',
      '[right ceiling]',
      '⌉',
    ],
    [
      'lfloor',
      '\\lfloor',
      true,
      '&lfloor;',
      '[left floor]',
      '[left floor]',
      '⌊',
    ],
    [
      'rfloor',
      '\\rfloor',
      true,
      '&rfloor;',
      '[right floor]',
      '[right floor]',
      '⌋',
    ],
    ['lang', '\\langle', true, '&lang;', '<', '<', '⟨'],
    ['rang', '\\rangle', true, '&rang;', '>', '>', '⟩'],
    ['langle', '\\langle', true, '&lang;', '<', '<', '⟨'],
    ['rangle', '\\rangle', true, '&rang;', '>', '>', '⟩'],
    ['hbar', '\\hbar', true, '&hbar;', 'hbar', 'hbar', 'ℏ'],
    ['mho', '\\mho', true, '&mho;', 'mho', 'mho', '℧'],
    // ** Arrows
    ['larr', '\\leftarrow', true, '&larr;', '<-', '<-', '←'],
    ['leftarrow', '\\leftarrow', true, '&larr;', '<-', '<-', '←'],
    ['gets', '\\gets', true, '&larr;', '<-', '<-', '←'],
    ['lArr', '\\Leftarrow', true, '&lArr;', '<=', '<=', '⇐'],
    ['Leftarrow', '\\Leftarrow', true, '&lArr;', '<=', '<=', '⇐'],
    ['uarr', '\\uparrow', true, '&uarr;', '[uparrow]', '[uparrow]', '↑'],
    ['uparrow', '\\uparrow', true, '&uarr;', '[uparrow]', '[uparrow]', '↑'],
    ['uArr', '\\Uparrow', true, '&uArr;', '[dbluparrow]', '[dbluparrow]', '⇑'],
    [
      'Uparrow',
      '\\Uparrow',
      true,
      '&uArr;',
      '[dbluparrow]',
      '[dbluparrow]',
      '⇑',
    ],
    ['rarr', '\\rightarrow', true, '&rarr;', '->', '->', '→'],
    ['to', '\\to', true, '&rarr;', '->', '->', '→'],
    ['rightarrow', '\\rightarrow', true, '&rarr;', '->', '->', '→'],
    ['rArr', '\\Rightarrow', true, '&rArr;', '=>', '=>', '⇒'],
    ['Rightarrow', '\\Rightarrow', true, '&rArr;', '=>', '=>', '⇒'],
    ['darr', '\\downarrow', true, '&darr;', '[downarrow]', '[downarrow]', '↓'],
    [
      'downarrow',
      '\\downarrow',
      true,
      '&darr;',
      '[downarrow]',
      '[downarrow]',
      '↓',
    ],
    [
      'dArr',
      '\\Downarrow',
      true,
      '&dArr;',
      '[dbldownarrow]',
      '[dbldownarrow]',
      '⇓',
    ],
    [
      'Downarrow',
      '\\Downarrow',
      true,
      '&dArr;',
      '[dbldownarrow]',
      '[dbldownarrow]',
      '⇓',
    ],
    ['harr', '\\leftrightarrow', true, '&harr;', '<->', '<->', '↔'],
    ['leftrightarrow', '\\leftrightarrow', true, '&harr;', '<->', '<->', '↔'],
    ['hArr', '\\Leftrightarrow', true, '&hArr;', '<=>', '<=>', '⇔'],
    ['Leftrightarrow', '\\Leftrightarrow', true, '&hArr;', '<=>', '<=>', '⇔'],
    ['crarr', '\\hookleftarrow', true, '&crarr;', "<-'", "<-'", '↵'],
    ['hookleftarrow', '\\hookleftarrow', true, '&crarr;', "<-'", "<-'", '↵'],
    // ** Function names
    ['arccos', '\\arccos', true, 'arccos', 'arccos', 'arccos', 'arccos'],
    ['arcsin', '\\arcsin', true, 'arcsin', 'arcsin', 'arcsin', 'arcsin'],
    ['arctan', '\\arctan', true, 'arctan', 'arctan', 'arctan', 'arctan'],
    ['arg', '\\arg', true, 'arg', 'arg', 'arg', 'arg'],
    ['cos', '\\cos', true, 'cos', 'cos', 'cos', 'cos'],
    ['cosh', '\\cosh', true, 'cosh', 'cosh', 'cosh', 'cosh'],
    ['cot', '\\cot', true, 'cot', 'cot', 'cot', 'cot'],
    ['coth', '\\coth', true, 'coth', 'coth', 'coth', 'coth'],
    ['csc', '\\csc', true, 'csc', 'csc', 'csc', 'csc'],
    ['deg', '\\deg', true, '&deg;', 'deg', 'deg', 'deg'],
    ['det', '\\det', true, 'det', 'det', 'det', 'det'],
    ['dim', '\\dim', true, 'dim', 'dim', 'dim', 'dim'],
    ['exp', '\\exp', true, 'exp', 'exp', 'exp', 'exp'],
    ['gcd', '\\gcd', true, 'gcd', 'gcd', 'gcd', 'gcd'],
    ['hom', '\\hom', true, 'hom', 'hom', 'hom', 'hom'],
    ['inf', '\\inf', true, 'inf', 'inf', 'inf', 'inf'],
    ['ker', '\\ker', true, 'ker', 'ker', 'ker', 'ker'],
    ['lg', '\\lg', true, 'lg', 'lg', 'lg', 'lg'],
    ['lim', '\\lim', true, 'lim', 'lim', 'lim', 'lim'],
    ['liminf', '\\liminf', true, 'liminf', 'liminf', 'liminf', 'liminf'],
    ['limsup', '\\limsup', true, 'limsup', 'limsup', 'limsup', 'limsup'],
    ['ln', '\\ln', true, 'ln', 'ln', 'ln', 'ln'],
    ['log', '\\log', true, 'log', 'log', 'log', 'log'],
    ['max', '\\max', true, 'max', 'max', 'max', 'max'],
    ['min', '\\min', true, 'min', 'min', 'min', 'min'],
    ['Pr', '\\Pr', true, 'Pr', 'Pr', 'Pr', 'Pr'],
    ['sec', '\\sec', true, 'sec', 'sec', 'sec', 'sec'],
    ['sin', '\\sin', true, 'sin', 'sin', 'sin', 'sin'],
    ['sinh', '\\sinh', true, 'sinh', 'sinh', 'sinh', 'sinh'],
    ['sup', '\\sup', true, '&sup;', 'sup', 'sup', 'sup'],
    ['tan', '\\tan', true, 'tan', 'tan', 'tan', 'tan'],
    ['tanh', '\\tanh', true, 'tanh', 'tanh', 'tanh', 'tanh'],
    // ** Signs & Symbols
    ['bull', '\\textbullet{}', false, '&bull;', '*', '*', '•'],
    ['bullet', '\\textbullet{}', false, '&bull;', '*', '*', '•'],
    ['star', '\\star', true, '*', '*', '*', '⋆'],
    ['lowast', '\\ast', true, '&lowast;', '*', '*', '∗'],
    ['ast', '\\ast', true, '&lowast;', '*', '*', '*'],
    ['odot', '\\odot', true, 'o', '[circled dot]', '[circled dot]', 'ʘ'],
    [
      'oplus',
      '\\oplus',
      true,
      '&oplus;',
      '[circled plus]',
      '[circled plus]',
      '⊕',
    ],
    [
      'otimes',
      '\\otimes',
      true,
      '&otimes;',
      '[circled times]',
      '[circled times]',
      '⊗',
    ],
    [
      'check',
      '\\checkmark',
      true,
      '&checkmark;',
      '[checkmark]',
      '[checkmark]',
      '✓',
    ],
    [
      'checkmark',
      '\\checkmark',
      true,
      '&check;',
      '[checkmark]',
      '[checkmark]',
      '✓',
    ],
    // ** Miscellaneous (seldom used)
    ['para', '\\P{}', false, '&para;', '[pilcrow]', '¶', '¶'],
    ['ordf', '\\textordfeminine{}', false, '&ordf;', '_a_', 'ª', 'ª'],
    ['ordm', '\\textordmasculine{}', false, '&ordm;', '_o_', 'º', 'º'],
    ['cedil', '\\c{}', false, '&cedil;', '[cedilla]', '¸', '¸'],
    ['oline', '\\overline{~}', true, '&oline;', '[overline]', '¯', '‾'],
    ['uml', '\\textasciidieresis{}', false, '&uml;', '[diaeresis]', '¨', '¨'],
    ['zwnj', '\\/{}', false, '&zwnj;', '', '', '‌'],
    ['zwj', '', false, '&zwj;', '', '', '‍'],
    ['lrm', '', false, '&lrm;', '', '', '‎'],
    ['rlm', '', false, '&rlm;', '', '', '‏'],
    // ** Smilies
    ['smiley', '\\ddot\\smile', true, '&#9786;', ':-)', ':-)', '☺'],
    ['blacksmile', '\\ddot\\smile', true, '&#9787;', ':-)', ':-)', '☻'],
    ['sad', '\\ddot\\frown', true, '&#9785;', ':-(', ':-(', '☹'],
    ['frowny', '\\ddot\\frown', true, '&#9785;', ':-(', ':-(', '☹'],
    // ** Suits
    ['clubs', '\\clubsuit', true, '&clubs;', '[clubs]', '[clubs]', '♣'],
    ['clubsuit', '\\clubsuit', true, '&clubs;', '[clubs]', '[clubs]', '♣'],
    ['spades', '\\spadesuit', true, '&spades;', '[spades]', '[spades]', '♠'],
    [
      'spadesuit',
      '\\spadesuit',
      true,
      '&spades;',
      '[spades]',
      '[spades]',
      '♠',
    ],
    ['hearts', '\\heartsuit', true, '&hearts;', '[hearts]', '[hearts]', '♥'],
    [
      'heartsuit',
      '\\heartsuit',
      true,
      '&heartsuit;',
      '[hearts]',
      '[hearts]',
      '♥',
    ],
    [
      'diams',
      '\\diamondsuit',
      true,
      '&diams;',
      '[diamonds]',
      '[diamonds]',
      '◆',
    ],
    [
      'diamondsuit',
      '\\diamondsuit',
      true,
      '&diams;',
      '[diamonds]',
      '[diamonds]',
      '◆',
    ],
    [
      'diamond',
      '\\diamondsuit',
      true,
      '&diamond;',
      '[diamond]',
      '[diamond]',
      '◆',
    ],
    [
      'Diamond',
      '\\diamondsuit',
      true,
      '&diamond;',
      '[diamond]',
      '[diamond]',
      '◆',
    ],
    ['loz', '\\lozenge', true, '&loz;', '[lozenge]', '[lozenge]', '⧫'],
    ['_ ', '\\hspace*{0.5em}', false, '&ensp;', ' ', ' ', ' '],
    ['_  ', '\\hspace*{1.0em}', false, '&ensp;&ensp;', '  ', '  ', '  '],
    [
      '_   ',
      '\\hspace*{1.5em}',
      false,
      '&ensp;&ensp;&ensp;',
      '   ',
      '   ',
      '   ',
    ],
    [
      '_    ',
      '\\hspace*{2.0em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;',
      '    ',
      '    ',
      '    ',
    ],
    [
      '_     ',
      '\\hspace*{2.5em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;',
      '     ',
      '     ',
      '     ',
    ],
    [
      '_      ',
      '\\hspace*{3.0em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '      ',
      '      ',
      '      ',
    ],
    [
      '_       ',
      '\\hspace*{3.5em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '       ',
      '       ',
      '       ',
    ],
    [
      '_        ',
      '\\hspace*{4.0em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '        ',
      '        ',
      '        ',
    ],
    [
      '_         ',
      '\\hspace*{4.5em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '         ',
      '         ',
      '         ',
    ],
    [
      '_          ',
      '\\hspace*{5.0em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '          ',
      '          ',
      '          ',
    ],
    [
      '_           ',
      '\\hspace*{5.5em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '           ',
      '           ',
      '           ',
    ],
    [
      '_            ',
      '\\hspace*{6.0em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '            ',
      '            ',
      '            ',
    ],
    [
      '_             ',
      '\\hspace*{6.5em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '             ',
      '             ',
      '             ',
    ],
    [
      '_              ',
      '\\hspace*{7.0em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '              ',
      '              ',
      '              ',
    ],
    [
      '_               ',
      '\\hspace*{7.5em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '               ',
      '               ',
      '               ',
    ],
    [
      '_                ',
      '\\hspace*{8.0em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '                ',
      '                ',
      '                ',
    ],
    [
      '_                 ',
      '\\hspace*{8.5em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '                 ',
      '                 ',
      '                 ',
    ],
    [
      '_                  ',
      '\\hspace*{9.0em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '                  ',
      '                  ',
      '                  ',
    ],
    [
      '_                   ',
      '\\hspace*{9.5em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '                   ',
      '                   ',
      '                   ',
    ],
    [
      '_                    ',
      '\\hspace*{10.0em}',
      false,
      '&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;&ensp;',
      '                    ',
      '                    ',
      '                    ',
    ],
  ];
