use qlty_config::config::issue_transformer::IssueTransformer;
use qlty_types::analysis::v1::{Issue, Location};
use std::path::Path;
use tracing::debug;

use crate::executor::staging_area::StagingArea;

#[derive(Debug, Clone)]
pub struct SourceExtractor {
    pub staging_area: StagingArea,
}

impl IssueTransformer for SourceExtractor {
    fn transform(&self, mut issue: Issue) -> Option<Issue> {
        // TODO: this is overly restrictive at the moment since plugin formatters
        // also generate `fmt` issues (ie: prettier). We should find a way to ensure
        // that we only skip issues that are generated by qlty itself
        if issue.rule_key == "fmt" {
            debug!("Skipping `fmt` issue: {:?}", issue);
            return Some(issue);
        }

        if let Some(location) = &issue.location {
            let (snippet, context) = self.extract_snippet_and_context(location);
            issue.snippet = snippet.unwrap_or_else(|| "".to_owned());
            issue.snippet_with_context = context.unwrap_or_else(|| "".to_owned());
        }

        Some(issue)
    }

    fn clone_box(&self) -> Box<dyn IssueTransformer> {
        Box::new(self.clone())
    }
}

impl SourceExtractor {
    fn extract_snippet_and_context(&self, location: &Location) -> (Option<String>, Option<String>) {
        let relative_path = Path::new(&location.path);
        match self.staging_area.read_lines(relative_path) {
            Ok(lines) => {
                let snippet = self.snippet_from_lines(location, &lines);
                let context = self.snippet_with_context(location, &lines, 10);
                (snippet, context)
            }
            Err(error) => {
                debug!("Failed to read file for snippet extraction: {}", error);
                (None, None)
            }
        }
    }

    fn snippet_from_lines(&self, location: &Location, lines: &[String]) -> Option<String> {
        self.extract_lines(location, lines, 0, 0)
    }

    fn snippet_with_context(
        &self,
        location: &Location,
        lines: &[String],
        context_lines: usize,
    ) -> Option<String> {
        self.extract_lines(location, lines, context_lines, context_lines)
    }

    fn extract_lines(
        &self,
        location: &Location,
        lines: &[String],
        before: usize,
        after: usize,
    ) -> Option<String> {
        location.range.as_ref().and_then(|range| {
            let start_line = range.start_line as usize;
            let end_line = range.end_line as usize;

            // add lines.len() != 0 to handle empty files
            if !lines.is_empty() {
                if start_line != 0 && end_line != 0 {
                    let start_index = start_line.saturating_sub(before).max(1) - 1;
                    let end_index = (end_line - 1 + after).min(lines.len() - 1);

                    return Some(lines.get(start_index..=end_index)?.join("\n"));
                }
                if start_line != 0 && end_line == 0 {
                    let single_line_index = start_line - 1;
                    let start_index = single_line_index.saturating_sub(before).max(0);
                    let end_index = (single_line_index + after).min(lines.len() - 1);

                    return Some(lines.get(start_index..=end_index)?.join("\n"));
                }
            }

            debug!("start_line and/or end_line not provided or zero in issue location");
            None
        })
    }
}

#[cfg(test)]
mod test {
    use super::*;
    use crate::executor::staging_area::Mode;
    use anyhow::{Context, Result};
    use qlty_types::analysis::v1::Range;
    use std::fs::File;
    use std::io::Write;
    use std::path::PathBuf;
    use tempfile::tempdir;

    fn setup() -> Result<(StagingArea, PathBuf)> {
        let temp_source = tempdir()?;
        let temp_dest = tempdir()?;

        let staging_area = StagingArea::new(
            temp_source.path().to_path_buf(),
            temp_dest.path().to_path_buf(),
            Mode::ReadWrite,
        );

        Ok((staging_area, temp_source.path().to_path_buf()))
    }

    fn create_temp_file(
        staging_area: &StagingArea,
        source_dir: &Path,
        file_path: &str,
        contents: &[&str],
    ) -> Result<()> {
        let full_file_path = source_dir.join(file_path);

        if let Some(parent) = full_file_path.parent() {
            std::fs::create_dir_all(parent)
                .with_context(|| format!("Failed to create parent directory at {:?}", parent))?;
        }

        let mut file = File::create(&full_file_path)
            .with_context(|| format!("Failed to create file at {:?}", full_file_path))?;

        writeln!(file, "{}", contents.join("\n"))
            .with_context(|| format!("Failed to write to file at {:?}", full_file_path))?;

        staging_area
            .stage(&full_file_path.strip_prefix(source_dir)?)
            .with_context(|| {
                format!(
                    "Failed to stage file from {:?} to staging area",
                    full_file_path
                )
            })?;

        Ok(())
    }

    #[test]
    fn test_basic_extraction() -> Result<()> {
        let (staging_area, source_dir) = setup()?;
        create_temp_file(
            &staging_area,
            &source_dir,
            "test_file.txt",
            &["Line 1", "Line 2", "Line 3"],
        )?;

        let location = Location {
            path: "test_file.txt".to_string(),
            range: Some(Range {
                start_line: 2,
                end_line: 2,
                ..Default::default()
            }),
        };
        let lines = staging_area
            .read_lines(Path::new("test_file.txt"))
            .with_context(|| "Failed to read lines from staged test file")?;
        let se = SourceExtractor {
            staging_area: staging_area.clone(),
        };
        let result = se.extract_lines(&location, &lines, 0, 0);
        assert_eq!(result, Some("Line 2".to_string()));
        Ok(())
    }

    #[test]
    fn test_extraction_with_context() -> Result<()> {
        let (staging_area, source_dir) = setup()?;
        create_temp_file(
            &staging_area,
            &source_dir,
            "context_file.txt",
            &["Line 1", "Line 2", "Line 3", "Line 4", "Line 5"],
        )?;

        let location = Location {
            path: "context_file.txt".to_string(),
            range: Some(Range {
                start_line: 3,
                end_line: 3,
                ..Default::default()
            }),
        };
        let lines = staging_area
            .read_lines(Path::new("context_file.txt"))
            .with_context(|| "Failed to read lines from staged context file")?;
        let se = SourceExtractor {
            staging_area: staging_area.clone(),
        };
        let result = se.extract_lines(&location, &lines, 1, 1);
        assert_eq!(result, Some("Line 2\nLine 3\nLine 4".to_string()));
        Ok(())
    }

    #[test]
    fn test_out_of_bounds_extraction() -> Result<()> {
        let (staging_area, source_dir) = setup()?;
        create_temp_file(
            &staging_area,
            &source_dir,
            "bounds_file.txt",
            &["Line 1", "Line 2"],
        )?;

        let location = Location {
            path: "bounds_file.txt".to_string(),
            range: Some(Range {
                start_line: 5,
                end_line: 6,
                ..Default::default()
            }),
        };
        let lines = staging_area
            .read_lines(Path::new("bounds_file.txt"))
            .with_context(|| "Failed to read lines from staged bounds file")?;
        let se = SourceExtractor {
            staging_area: staging_area.clone(),
        };
        let result = se.extract_lines(&location, &lines, 0, 0);
        assert_eq!(result, None);
        Ok(())
    }

    #[test]
    fn test_empty_file_extraction() -> Result<()> {
        let (staging_area, source_dir) = setup()?;
        create_temp_file(&staging_area, &source_dir, "empty_file.txt", &[])?;

        let location = Location {
            path: "empty_file.txt".to_string(),
            range: Some(Range {
                start_line: 1,
                end_line: 1,
                ..Default::default()
            }),
        };
        let lines = staging_area
            .read_lines(Path::new("empty_file.txt"))
            .with_context(|| "Failed to read lines from staged empty file")?;
        let se = SourceExtractor {
            staging_area: staging_area.clone(),
        };
        let result = se.extract_lines(&location, &lines, 0, 0);
        assert_eq!(
            result,
            Some("".to_string()),
            "Expected Some(\"\") for an empty file but got {:?}",
            result
        );
        Ok(())
    }

    #[test]
    fn test_single_line_file_extraction() -> Result<()> {
        let (staging_area, source_dir) = setup()?;
        create_temp_file(
            &staging_area,
            &source_dir,
            "single_line_file.txt",
            &["line 1"],
        )?;

        let location = Location {
            path: "single_line_file.txt".to_string(),
            range: Some(Range {
                start_line: 1,
                end_line: 1,
                ..Default::default()
            }),
        };
        let lines = staging_area
            .read_lines(Path::new("single_line_file.txt"))
            .with_context(|| "Failed to read lines from staged single-line file")?;
        let se = SourceExtractor {
            staging_area: staging_area.clone(),
        };
        let result = se.extract_lines(&location, &lines, 0, 0);
        assert_eq!(
            result,
            Some("line 1".to_string()),
            "Expected 'line 1' for a file with a single line but got {:?}",
            result
        );
        Ok(())
    }

    #[test]
    fn test_extraction_at_file_bounds() -> Result<()> {
        let (staging_area, source_dir) = setup()?;
        create_temp_file(
            &staging_area,
            &source_dir,
            "boundary_file.txt",
            &["Line 1", "Line 2", "Line 3"],
        )?;

        let location = Location {
            path: "boundary_file.txt".to_string(),
            range: Some(Range {
                start_line: 1,
                end_line: 3,
                ..Default::default()
            }),
        };
        let lines = staging_area
            .read_lines(Path::new("boundary_file.txt"))
            .with_context(|| "Failed to read lines from staged boundary file")?;
        let se = SourceExtractor {
            staging_area: staging_area.clone(),
        };
        let result = se.extract_lines(&location, &lines, 0, 0);
        assert_eq!(
            result,
            Some("Line 1\nLine 2\nLine 3".to_string()),
            "Expected exact content of the file but got {:?}",
            result
        );
        Ok(())
    }
}
