// Copyright 2020, Chef.  All rights reserved.
// https://github.com/q191201771/lal
//
// Use of this source code is governed by a MIT-style license
// that can be found in the License file.
//
// Author: Chef (191201771@qq.com)

package rtprtcp_test

import (
	"testing"

	"github.com/q191201771/lal/pkg/rtprtcp"

	"github.com/q191201771/naza/pkg/assert"
	"github.com/q191201771/naza/pkg/nazalog"
)

var testRtpData = []byte{
	0x90, 0x00, 0x1d, 0x55, 0x4a, 0x22, 0xc9, 0xf7, 0x08, 0x0c, 0x09, 0xb8, 0xbe, 0xde, 0x00, 0x01,
	0x31, 0x00, 0x02, 0x00, 0x03, 0x33, 0x65, 0xc7, 0xcc, 0x20, 0x3d, 0x19, 0x12, 0x97, 0xf4, 0xc6,
	0x16, 0xa6, 0x2c, 0xe2, 0xb8, 0x05, 0xd8, 0xce, 0x42, 0x00, 0x33, 0x19, 0x3d, 0xd5, 0xa1, 0x25,
	0x63, 0x2d, 0x59, 0x03, 0x91, 0xf7, 0x72, 0x18, 0x52, 0xe0, 0x4d, 0x13, 0x2f, 0x7a, 0xc8, 0xf7,
	0x3d, 0x89, 0x0c, 0x82, 0x7d, 0xc3, 0x1c, 0xd7, 0x9f, 0xfd, 0x47, 0x72, 0x0e, 0xdc, 0xc0, 0x90,
	0xe9, 0x02, 0x85, 0x80, 0xbb, 0x52, 0x07, 0x05, 0x03, 0x75, 0x81, 0x9e, 0x33, 0x81, 0x06, 0x1a,
	0x4d, 0x29, 0xa3, 0x39, 0xe5, 0x72, 0x7b, 0x2a, 0x3b, 0x16, 0x88, 0x25, 0x16, 0x97, 0x1c, 0x6e,
	0x27, 0x71, 0xee, 0x4c, 0xdb, 0x11, 0xbc, 0xe3, 0x83, 0xb4, 0xd4, 0x1e, 0x7b, 0xa6, 0x3b, 0x88,
	0xe4, 0xa2, 0x6d, 0x4e, 0x37, 0xb6, 0xa9, 0xb4, 0xac, 0xa9, 0x9f, 0xf4, 0x97, 0xe7, 0x1b, 0x2c,
	0x2b, 0x50, 0x86, 0x51, 0x01, 0xe9, 0xa0, 0x84, 0x90, 0x4f, 0x46, 0x07, 0x9c, 0x6c, 0xdb, 0x4b,
	0xdf, 0xec, 0x2f, 0x37, 0xc3, 0x43, 0xfb, 0x8f, 0xd7, 0xb4, 0x7c, 0x17, 0xa8, 0x09, 0xb7, 0x69,
	0x1e, 0x6f, 0x95, 0x01, 0xcb, 0xe7, 0xb7, 0x3f, 0xd4, 0x06, 0x44, 0x37, 0xb5, 0x7c,
}

func TestCompareSeq(t *testing.T) {
	assert.Equal(t, 0, rtprtcp.CompareSeq(0, 0))
	assert.Equal(t, 0, rtprtcp.CompareSeq(1024, 1024))
	assert.Equal(t, 0, rtprtcp.CompareSeq(65535, 65535))

	assert.Equal(t, 1, rtprtcp.CompareSeq(1, 0))
	assert.Equal(t, 1, rtprtcp.CompareSeq(16383, 0))

	assert.Equal(t, -1, rtprtcp.CompareSeq(65534, 0))
	assert.Equal(t, -1, rtprtcp.CompareSeq(65535, 0))
	assert.Equal(t, -1, rtprtcp.CompareSeq(65534, 1))
	assert.Equal(t, -1, rtprtcp.CompareSeq(65535, 1))

	assert.Equal(t, -1, rtprtcp.CompareSeq(0, 1))
	assert.Equal(t, -1, rtprtcp.CompareSeq(0, 16383))

	assert.Equal(t, 1, rtprtcp.CompareSeq(0, 65534))
	assert.Equal(t, 1, rtprtcp.CompareSeq(0, 65535))
	assert.Equal(t, 1, rtprtcp.CompareSeq(1, 65534))
	assert.Equal(t, 1, rtprtcp.CompareSeq(1, 65535))

	assert.Equal(t, 1, rtprtcp.CompareSeq(16384, 0))
	assert.Equal(t, -1, rtprtcp.CompareSeq(0, 16384))
	assert.Equal(t, -1, rtprtcp.CompareSeq(10400, 33489))
}

func TestSubSeq(t *testing.T) {
	assert.Equal(t, 0, rtprtcp.SubSeq(0, 0))
	assert.Equal(t, 0, rtprtcp.SubSeq(1024, 1024))
	assert.Equal(t, 0, rtprtcp.SubSeq(65535, 65535))

	assert.Equal(t, 1, rtprtcp.SubSeq(1, 0))
	assert.Equal(t, 16383, rtprtcp.SubSeq(16383, 0))

	assert.Equal(t, -49152, rtprtcp.SubSeq(16384, 0))
	assert.Equal(t, -2, rtprtcp.SubSeq(65534, 0))
	assert.Equal(t, -1, rtprtcp.SubSeq(65535, 0))
	assert.Equal(t, -3, rtprtcp.SubSeq(65534, 1))
	assert.Equal(t, -2, rtprtcp.SubSeq(65535, 1))

	assert.Equal(t, -1, rtprtcp.SubSeq(0, 1))
	assert.Equal(t, -16383, rtprtcp.SubSeq(0, 16383))

	assert.Equal(t, 49152, rtprtcp.SubSeq(0, 16384))
	assert.Equal(t, 2, rtprtcp.SubSeq(0, 65534))
	assert.Equal(t, 1, rtprtcp.SubSeq(0, 65535))
	assert.Equal(t, 3, rtprtcp.SubSeq(1, 65534))
	assert.Equal(t, 2, rtprtcp.SubSeq(1, 65535))
}

func TestParseRtpHeader(t *testing.T) {
	h, err := rtprtcp.ParseRtpHeader(testRtpData)
	nazalog.Assert(nil, err)

	assert.Equal(t, uint8(0), h.Mark)
	assert.Equal(t, uint16(0xbede), h.ExtensionProfile)
	assert.Equal(t, uint8(1), h.Extension)
	assert.Equal(t, int(4), len(h.Extensions))
	assert.Equal(t, []byte{0x31, 0x00, 0x02, 0x00}, h.Extensions)
}
