ChatEngine does not contain any pre-made templates or GUI code as part of the main package (there are plenty in the [chat-engine-examples](http://www.github.com/pubnub/chat-engine-examples) repository however).

ChatEngine leaves the design and interaction flow up to the developer. ChatEngine provides events, methods, and data, but does not make any assumptions of template engines.

ChatEngine works on both the front and back ends, with popular frameworks like [jQuery](https://jquery.com/), [React](https://reactjs.org/), [Angular](https://angularjs.org/), [React Native](https://facebook.github.io/react-native/), [NodeJS](https://nodejs.org/en/), and good old [Javascript](https://www.javascript.com/).

For example, to publish a message to a {@link Chat| ```Chat``` } you may bind a DOM ```onSubmit()``` {@link Event| ```Event``` } to the {@link Chat#emit| ```Chat.emit()``` } method.

So, let's create a new {@link Chat| ```Chat``` }:

```js
// create a new chat
let chat = new ChatEngine.Chat('example-chat');
```

Create a place for message output and give the {@link User| ```User``` } a way to emit the {@link Event| ```Event``` }:

```HTML
<div data-id="output"></div>
<button type="submit" onclick="sendChat()">Send</button>
```

The ```sendChat()``` method emits ```Chat.emit```.

```js
// send a message to all users in the chat
function sendChat() {

    chat.emit('message', {
        text: 'hello world!'
    });

};
```

Then, when the message is received by the client:

```js
// get the DOM element in Javascript
let output = document.getElementById('output');

// when the event is received
chat.on('message', (payload) => {

    output.innerHTML = output.innerHTML + payload.data.text;

});
```

## Data Binding (Angular and React)

ChatEngine has a local data system that keeps data about {@link Chat| Chats } and {@link User| Users } in sync. So when working with a data-bound system we get some functionality for free!

For example in Angular:

```HTML
<ul data-id="online-list">
    <li ng-repeat="(uuid, user) in chat.users">
        <a href="#" ng-click="chat.invite(user)">{{user.state.username}}</a>
    </li>
</ul>
```

Note that binding ChatEngine data to Angular requires forcefully telling Angular that something has changed. The following is a small Angular plugin that, when loaded, will re-render your Angular app anytime anything changes in ChatEngine.

```js
// plugin
angular.module('my-app', [])
    .service('ngChatEngine', ['$timeout', function($timeout) {

        this.bind = function(ChatEngine) {

            // updates angular when anything changes
            ChatEngine.onAny(function(event, payload) {
                $timeout(function() {});
            });

        }

    }]);
```

```js
// in your app
angular.module('chatApp', ['my-app'])
    .run(['$rootScope', 'ngChatEngine', function($rootScope, ngChatEngine) {

        // ChatEngine Configure
        $rootScope.ChatEngine = new ChatEngineCore({});

        // bind open chat framework angular plugin
        ngChatEngine.bind($rootScope.ChatEngine);

    }])
```
