/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.facebook.presto.verifier.source;

import com.facebook.presto.verifier.framework.SnapshotQuery;
import com.facebook.presto.verifier.framework.SourceQuery;
import org.jdbi.v3.sqlobject.config.RegisterColumnMapper;
import org.jdbi.v3.sqlobject.config.RegisterConstructorMapper;
import org.jdbi.v3.sqlobject.customizer.Bind;
import org.jdbi.v3.sqlobject.customizer.Define;
import org.jdbi.v3.sqlobject.statement.SqlQuery;
import org.jdbi.v3.sqlobject.statement.SqlUpdate;

import java.util.List;

@RegisterColumnMapper(StringToStringMapColumnMapper.class)
@RegisterColumnMapper(StringListColumnMapper.class)
public interface VerifierDao
{
    @SqlUpdate("CREATE TABLE <table_name> (\n" +
            "  id int(11) unsigned NOT NULL PRIMARY KEY AUTO_INCREMENT,\n" +
            "  suite varchar(256) NOT NULL,\n" +
            "  name varchar(256) DEFAULT NULL,\n" +
            "  control_catalog varchar(256) NOT NULL,\n" +
            "  control_schema varchar(256) NOT NULL,\n" +
            "  control_query text NOT NULL,\n" +
            "  control_query_id mediumtext DEFAULT NULL,\n" +
            "  control_username varchar(256) DEFAULT NULL,\n" +
            "  control_password varchar(256) DEFAULT NULL,\n" +
            "  control_session_properties text DEFAULT NULL,\n" +
            "  control_client_tags text DEFAULT NULL,\n" +
            "  control_partitions text DEFAULT NULL,\n" +
            "  test_catalog varchar(256) NOT NULL,\n" +
            "  test_schema varchar(256) NOT NULL,\n" +
            "  test_query text NOT NULL,\n" +
            "  test_query_id mediumtext DEFAULT NULL,\n" +
            "  test_username varchar(256) DEFAULT NULL,\n" +
            "  test_password varchar(256) DEFAULT NULL,\n" +
            "  test_session_properties text DEFAULT NULL,\n" +
            "  test_client_tags text DEFAULT NULL,\n" +
            "  test_partitions text DEFAULT NULL)")
    void createVerifierQueriesTable(@Define("table_name") String tableName);

    @SqlQuery("SELECT\n" +
            "  suite,\n" +
            "  name,\n" +
            "  control_query,\n" +
            "  name control_query_id,\n" +
            "  control_catalog,\n" +
            "  control_schema,\n" +
            "  control_username,\n" +
            "  control_password,\n" +
            "  control_session_properties,\n" +
            "  control_client_tags,\n" +
            "  control_partitions,\n" +
            "  test_query,\n" +
            "  test_query_id,\n" +
            "  test_catalog,\n" +
            "  test_schema,\n" +
            "  test_username,\n" +
            "  test_password,\n" +
            "  test_session_properties,\n" +
            "  test_client_tags,\n" +
            "  test_partitions\n" +
            "FROM\n" +
            "  <table_name>\n" +
            "WHERE\n" +
            "  suite  = :suite\n" +
            "ORDER BY\n" +
            "  id\n" +
            "LIMIT\n" +
            "  :limit")
    @RegisterConstructorMapper(SourceQuery.class)
    List<SourceQuery> getSourceQueries(
            @Define("table_name") String tableName,
            @Bind("suite") String suite,
            @Bind("limit") int limit);

    @SqlUpdate("CREATE TABLE <table_name> (\n" +
            "  id int(11) unsigned NOT NULL PRIMARY KEY AUTO_INCREMENT,\n" +
            "  suite varchar(256) NOT NULL,\n" +
            "  name varchar(256) NOT NULL DEFAULT '.',  -- default to '.' to make UNIQUE constraint happy if name is not provided.\n" +
            "  is_explain BOOLEAN NOT NULL DEFAULT false,\n" +
            "  snapshot json NOT NULL,\n" +
            "  updated_at datetime NOT NULL DEFAULT now(),\n" +
            "  UNIQUE(suite, name, is_explain)\n" +
            ")")
    void createVerifierSnapshotsTable(@Define("table_name") String tableName);

    @SqlQuery("SELECT\n" +
            "  suite,\n" +
            "  name,\n" +
            "  is_explain,\n" +
            "  snapshot\n" +
            "FROM <table_name>\n" +
            "WHERE\n" +
            "  suite  = :suite\n" +
            "ORDER BY\n" +
            "  id\n" +
            "LIMIT\n" +
            "  :limit")
    @RegisterConstructorMapper(SnapshotQuery.class)
    List<SnapshotQuery> getSnapshotQueries(
            @Define("table_name") String tableName,
            @Bind("suite") String suite,
            @Bind("limit") int limit);

    @SqlUpdate("INSERT INTO <table_name>\n" +
            "  (suite,\n" +
            "  name,\n" +
            "  is_explain,\n" +
            "  snapshot,\n" +
            "  updated_at)\n" +
            "VALUES(:suite,\n" +
            "  :name,\n" +
            "  :is_explain,\n" +
            "  :snapshot,\n" +
            "  now())\n" +
            "ON DUPLICATE KEY UPDATE\n" +
            "  snapshot = :snapshot,\n" +
            "  updated_at = now()")
    void saveSnapShot(
            @Define("table_name") String tableName,
            @Bind("suite") String suite,
            @Bind("name") String name,
            @Bind("is_explain") boolean explain,
            @Bind("snapshot") String snapshot);
}
