/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.facebook.presto.spark.execution.shuffle;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;

import static com.google.common.base.MoreObjects.toStringHelper;
import static java.util.Objects.requireNonNull;

/**
 * This class is a 1:1 strict API mapping to LocalShuffleInfo in
 * presto-native-execution/presto_cpp/main/operators/LocalPersistentShuffle.h.
 * Please refrain changes to this API class. If any changes have to be made to
 * this class, one should make sure to make corresponding changes in the above
 * C++ struct and its corresponding serde functionalities.
 */
public class PrestoSparkLocalShuffleWriteInfo
        implements PrestoSparkShuffleWriteInfo
{
    private final int numPartitions;
    // Unique identifier for each query generated by Presto.
    private final String queryId;
    // Unique identifier for each shuffle stage generated by Spark
    private final int shuffleId;
    private final String rootPath;

    @JsonCreator
    public PrestoSparkLocalShuffleWriteInfo(
            @JsonProperty("numPartitions") int numPartitions,
            @JsonProperty("queryId") String queryId,
            @JsonProperty("shuffleId") int shuffleId,
            @JsonProperty("rootPath") String rootPath)
    {
        this.numPartitions = numPartitions;
        this.queryId = requireNonNull(queryId, "queryId is null");
        this.shuffleId = shuffleId;
        this.rootPath = requireNonNull(rootPath, "rootPath is null");
    }

    @JsonProperty
    public int getNumPartitions()
    {
        return numPartitions;
    }

    @JsonProperty
    public String getQueryId()
    {
        return queryId;
    }

    @JsonProperty
    public String getRootPath()
    {
        return rootPath;
    }

    @JsonProperty
    public int getShuffleId()
    {
        return shuffleId;
    }

    @Override
    public String toString()
    {
        return toStringHelper(this)
                .add("numPartitions", numPartitions)
                .add("queryId", queryId)
                .add("shuffleId", shuffleId)
                .add("rootPath", rootPath)
                .toString();
    }
}
