/**********************************************************************
 * Copyright (C) 2022 Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 ***********************************************************************/

import fs from 'node:fs';
import path from 'node:path';

import { app } from 'electron';

import type { ConfigurationRegistry } from '../plugin/configuration-registry.js';

/**
 * On Windows, launching program automatically on startup is done via %APPDATA%\Roaming\Microsoft\Windows\Start Menu\Programs\Startup folder
 * This class manages the creation and deletion of the startup file
 * It uses a vbs script as using a bat script displays a blank terminal window
 */
export class WindowsStartup {
  private podmanDesktopBinaryPath;

  private startupFile;
  private windowsStartupFoler;
  private exeDirectory;
  private configurationRegistry: ConfigurationRegistry;

  constructor(configurationRegistry: ConfigurationRegistry) {
    // configuration settings
    this.configurationRegistry = configurationRegistry;

    // grab current path of the binary
    this.podmanDesktopBinaryPath = app.getPath('exe');

    // Folder of the binary path
    this.exeDirectory = path.dirname(this.podmanDesktopBinaryPath);

    // Path to the startup folder ?
    this.windowsStartupFoler = path.resolve(app.getPath('appData'), 'Microsoft/Windows/Start Menu/Programs/Startup');

    // Path to the startup folder ?
    this.startupFile = path.resolve(this.windowsStartupFoler, 'podman-desktop.vbs');
  }

  // enable only if we're not using a temporary path / portable mode
  shouldEnable(): boolean {
    if (!process.env['PORTABLE_EXECUTABLE_FILE'] && this.podmanDesktopBinaryPath.startsWith(app.getPath('temp'))) {
      console.warn('Skipping start on boot option as the app is running from a temporary path');
      return false;
    }
    return true;
  }

  async enable(): Promise<void> {
    // Check the preferences for login.minimize has been enabled
    // as this may change each time it's enabled (changed from true to false, etc.)
    // it's also to make sure that settings weren't changed while async function was running
    // so we check the configuration within the function
    const preferencesConfig = this.configurationRegistry.getConfiguration('preferences');
    const minimize = preferencesConfig.get<boolean>('login.minimize');

    // We pass in "--minimize" so electron can read the flag on first startup.
    const minimizeSettings = minimize ? ' --minimized' : '';

    // check if we are using the portable mode.
    // in that case we need to register the binary path to the portable file
    // and not where it is being expanded
    if (process.env['PORTABLE_EXECUTABLE_FILE'] && process.env['PORTABLE_EXECUTABLE_DIR']) {
      this.podmanDesktopBinaryPath = process.env['PORTABLE_EXECUTABLE_FILE'];
      this.exeDirectory = process.env['PORTABLE_EXECUTABLE_DIR'];
    } else if (this.podmanDesktopBinaryPath.startsWith(app.getPath('temp'))) {
      console.warn('Skipping startup installation as the app is running from a temporary path');
      return;
    }

    // do we have an updated version of the binary being installed in AppData/Local
    // if so, we need to update the startup file to point to the new binary
    // this is the case when we update the app
    const programsData = path.resolve(app.getPath('appData'), '..', 'local/Programs/podman-desktop');
    const podmanDesktopInPrograms = path.resolve(programsData, 'Podman Desktop.exe');
    if (fs.existsSync(podmanDesktopInPrograms)) {
      this.podmanDesktopBinaryPath = podmanDesktopInPrograms;
      this.exeDirectory = programsData;
    }

    if (!this.windowsStartupFoler) {
      console.warn(
        `Windows startup folder not found ${this.windowsStartupFoler}, cancelling the startup file creation`,
      );
      return;
    }

    // write the script file
    // also make sure we try to execute the binary only if it exists to avoid any start error/annoying popup
    // in case user removed the app
    // Use """" instead of chr(34) escape codes in the script to avoid any issue with the path / adding --minimized
    const vbsContent =
      `' This file is automatically generated from Podman Desktop preferences.
set FSO = CreateObject("Scripting.FileSystemObject")
Set WshShell = CreateObject("WScript.Shell")
if FSO.FileExists("${this.podmanDesktopBinaryPath}") then
  WshShell.CurrentDirectory = "${this.exeDirectory}"
  WshShell.Run """" & "${this.podmanDesktopBinaryPath}" & """` +
      minimizeSettings +
      `", 0, False
end if
Set WshShell = Nothing
`;

    await fs.promises.writeFile(this.startupFile, vbsContent, 'utf-8');
    console.info(
      `Installing Podman Desktop startup file at ${this.startupFile} using ${this.podmanDesktopBinaryPath} location.`,
    );
  }

  async disable(): Promise<void> {
    // remove the file at this.podmanDesktopBinaryPath only if it exists
    if (fs.existsSync(this.startupFile)) {
      await fs.promises.unlink(this.startupFile);
    }
  }
}
