#!/usr/bin/env python3

from sys import argv
from collections import OrderedDict
import re
import msgpack  # way faster than pickle


class Message:
    def __init__(self, name, section, index):
        self.name = name
        self.section = section
        self.index = index

        self.bytes = []  # XXX: bytearray would be better


def try_convert_int(s):
    try:
        return int(s, base=0)
    except:
        return s


def parse_command(source):
    if source[0] != "[":
        return None, [], {}, source
    source = source[1:]  # "["

    inside_brackets = ""
    while source[0] != "]":
        if source[0] == "\n":
            return None, [], {}, source

        inside_brackets += source[0]
        source = source[1:]
    source = source[1:]  # "]"

    command, *raw_args = inside_brackets.split(" ")

    args = []
    named_args = {}

    for arg in raw_args:
        if "=" in arg:
            key, value = arg.split("=", 1)

            if "," in value:
                named_args[key.lower()] = [try_convert_int(v) for v in value.lower().split(",")]
            else:
                named_args[key.lower()] = try_convert_int(value.lower())
        else:
            if "," in arg:
                args.append([try_convert_int(value) for value in arg.lower().split(",")])
            else:
                args.append(try_convert_int(arg.lower()))

    # args = []
    # for arg in raw_args:
    #     args.append(try_convert_int(arg.lower()))

    return command.lower(), args, named_args, source


def color_to_code(color, style):
    COLORS = {
        "diary": {
            "normal": 0x00,
            "red": 0x07,
        },
        "inspect": {
            "dark": 0x17,
        },
        "button": {
            "blue": 0x10,
            "green": 0x11,
            "red": 0x12,
            "yellow": 0x13,
            "gray": 0x14,
            "grey": 0x14,
        },
        "popup": {
            "red": 0x28,
            "pink": 0x29,
            "purple": 0x2A,
            "blue": 0x2B,
            "teal": 0x2C,
            "green": 0x2D,
            "yellow": 0x2E,
            "normal": 0x2F,
        },
        "sign": {
            "normal": 0x18,
            "red": 0x19,
            "blue": 0x1A,
            "green": 0x1B,
        },
    }

    if type(color) is int:
        return color

    return COLORS.get(
        style,
        {
            # [style:left], [style:right]
            "normal": 0x0A,
            "red": 0x20,
            "pink": 0x21,
            "purple": 0x22,
            "blue": 0x23,
            "cyan": 0x24,
            "green": 0x25,
            "yellow": 0x26,
        },
    ).get(color)


CHARSET_STANDARD = {
    # "𝅘𝅥𝅮": 0x00,
    "!": 0x01,
    '"': 0x02,
    "#": 0x03,
    "$": 0x04,
    "%": 0x05,
    "&": 0x06,
    "'": 0x07,
    "(": 0x08,
    ")": 0x09,
    "*": 0x0A,
    "+": 0x0B,
    ",": 0x0C,
    "-": 0x0D,
    ".": 0x0E,
    "/": 0x0F,
    "0": 0x10,
    "1": 0x11,
    "2": 0x12,
    "3": 0x13,
    "4": 0x14,
    "5": 0x15,
    "6": 0x16,
    "7": 0x17,
    "8": 0x18,
    "9": 0x19,
    ":": 0x1A,
    ";": 0x1B,
    "<": 0x1C,
    "=": 0x1D,
    ">": 0x1E,
    "?": 0x1F,
    "@": 0x20,
    "A": 0x21,
    "B": 0x22,
    "C": 0x23,
    "D": 0x24,
    "E": 0x25,
    "F": 0x26,
    "G": 0x27,
    "H": 0x28,
    "I": 0x29,
    "J": 0x2A,
    "K": 0x2B,
    "L": 0x2C,
    "M": 0x2D,
    "N": 0x2E,
    "O": 0x2F,
    "P": 0x30,
    "Q": 0x31,
    "R": 0x32,
    "S": 0x33,
    "T": 0x34,
    "U": 0x35,
    "V": 0x36,
    "W": 0x37,
    "X": 0x38,
    "Y": 0x39,
    "Z": 0x3A,
    "[": 0x3B,
    "¥": 0x3C,
    "]": 0x3D,
    "^": 0x3E,
    "_": 0x3F,
    "`": 0x40,
    "a": 0x41,
    "b": 0x42,
    "c": 0x43,
    "d": 0x44,
    "e": 0x45,
    "f": 0x46,
    "g": 0x47,
    "h": 0x48,
    "i": 0x49,
    "j": 0x4A,
    "k": 0x4B,
    "l": 0x4C,
    "m": 0x4D,
    "n": 0x4E,
    "o": 0x4F,
    "p": 0x50,
    "q": 0x51,
    "r": 0x52,
    "s": 0x53,
    "t": 0x54,
    "u": 0x55,
    "v": 0x56,
    "w": 0x57,
    "x": 0x58,
    "y": 0x59,
    "z": 0x5A,
    "{": 0x5B,
    "|": 0x5C,
    "}": 0x5D,
    "~": 0x5E,
    "°": 0x5F,
    "À": 0x60,
    "Á": 0x61,
    "Â": 0x62,
    "Ä": 0x63,
    "Ç": 0x64,
    "È": 0x65,
    "É": 0x66,
    "Ê": 0x67,
    "Ë": 0x68,
    "Ì": 0x69,
    "Í": 0x6A,
    "Î": 0x6B,
    "Ï": 0x6C,
    "Ñ": 0x6D,
    "Ò": 0x6E,
    "Ó": 0x6F,
    "Ô": 0x70,
    "Ö": 0x71,
    "Ù": 0x72,
    "Ú": 0x73,
    "Û": 0x74,
    "Ü": 0x75,
    "ß": 0x76,
    "à": 0x77,
    "á": 0x78,
    "â": 0x79,
    "ä": 0x7A,
    "ç": 0x7B,
    "è": 0x7C,
    "é": 0x7D,
    "ê": 0x7E,
    "ë": 0x7F,
    "ì": 0x80,
    "í": 0x81,
    "î": 0x82,
    "ï": 0x83,
    "ñ": 0x84,
    "ò": 0x85,
    "ó": 0x86,
    "ô": 0x87,
    "ö": 0x88,
    "ù": 0x89,
    "ú": 0x8A,
    "û": 0x8B,
    "ü": 0x8C,
    "¡": 0x8D,
    "¿": 0x8E,
    "ª": 0x8F,
    # "♥": 0x90,
    # "★": 0x91,
    # "↑": 0x92,
    # "↓": 0x93,
    # "←": 0x94,
    # "→": 0x95,
    # "●": 0x96,
    # "✖": 0x97,
    "“": 0xA2,
    "”": 0xA3,
    "‘": 0xA4,
    "’": 0xA5,
    " ": 0xF7,
    "　": 0xF7,
    # "Ⓐ": [0xFF, 0x24, 0xFF, 0x05, 0x10, 0x98, 0xFF, 0x25],
    # "Ⓑ": [0xFF, 0x24, 0xFF, 0x05, 0x11, 0x99, 0xFF, 0x25],
    # "Ⓢ": [0xFF, 0x24, 0xFF, 0x05, 0x12, 0xA1, 0xFF, 0x25],
    # "▲": [0xFF, 0x24, 0xFF, 0x05, 0x13, 0x9D, 0xFF, 0x25],
    # "▼": [0xFF, 0x24, 0xFF, 0x05, 0x13, 0x9E, 0xFF, 0x25],
    # "◀": [0xFF, 0x24, 0xFF, 0x05, 0x13, 0x9F, 0xFF, 0x25],
    # "▶": [0xFF, 0x24, 0xFF, 0x05, 0x13, 0xA0, 0xFF, 0x25],
    # "Ⓛ": [0xFF, 0x24, 0xFF, 0x05, 0x14, 0x9A, 0xFF, 0x25],
    # "Ⓡ": [0xFF, 0x24, 0xFF, 0x05, 0x14, 0x9B, 0xFF, 0x25],
    # "Ⓩ": [0xFF, 0x24, 0xFF, 0x05, 0x14, 0x9C, 0xFF, 0x25],
}

CHARSET_CREDITS = {
    "A": 0x00,
    "B": 0x01,
    "C": 0x02,
    "D": 0x03,
    "E": 0x04,
    "F": 0x05,
    "G": 0x06,
    "H": 0x07,
    "I": 0x08,
    "J": 0x09,
    "K": 0x0A,
    "L": 0x0B,
    "M": 0x0C,
    "N": 0x0D,
    "O": 0x0E,
    "P": 0x0F,
    "Q": 0x10,
    "R": 0x11,
    "S": 0x12,
    "T": 0x13,
    "U": 0x14,
    "V": 0x15,
    "W": 0x16,
    "X": 0x17,
    "Y": 0x18,
    "Z": 0x19,
    "'": 0x1A,
    ".": 0x1B,
    ",": 0x1C,
    "0": 0x1D,
    "1": 0x1E,
    "2": 0x1F,
    "3": 0x20,
    "4": 0x21,
    "5": 0x22,
    "6": 0x23,
    "7": 0x24,
    "8": 0x25,
    "9": 0x26,
    "©": 0x27,
    "&": 0x28,
    " ": 0xF7,
    "　": 0xF8,
}

CHARSET_KANA = {
    "あ": 0x00,
    "い": 0x01,
    "う": 0x02,
    "え": 0x03,
    "お": 0x04,
    "か": 0x05,
    "き": 0x06,
    "く": 0x07,
    "け": 0x08,
    "こ": 0x09,
    "さ": 0x0A,
    "し": 0x0B,
    "す": 0x0C,
    "せ": 0x0D,
    "そ": 0x0E,
    "た": 0x0F,
    "ち": 0x10,
    "つ": 0x11,
    "て": 0x12,
    "と": 0x13,
    "な": 0x14,
    "に": 0x15,
    "ぬ": 0x16,
    "ね": 0x17,
    "の": 0x18,
    "は": 0x19,
    "ひ": 0x1A,
    "ふ": 0x1B,
    "へ": 0x1C,
    "ほ": 0x1D,
    "ま": 0x1E,
    "み": 0x1F,
    "む": 0x20,
    "め": 0x21,
    "も": 0x22,
    "や": 0x23,
    "ゆ": 0x24,
    "よ": 0x25,
    "ら": 0x26,
    "り": 0x27,
    "る": 0x28,
    "れ": 0x29,
    "ろ": 0x2A,
    "わ": 0x2B,
    "を": 0x2C,
    "ん": 0x2D,
    "ゔ": 0x2E,
    "が": 0x2F,
    "ぎ": 0x30,
    "ぐ": 0x31,
    "げ": 0x32,
    "ご": 0x33,
    "ざ": 0x34,
    "じ": 0x35,
    "ず": 0x36,
    "ぜ": 0x37,
    "ぞ": 0x38,
    "だ": 0x39,
    "ぢ": 0x3A,
    "づ": 0x3B,
    "で": 0x3C,
    "ど": 0x3D,
    "ば": 0x3E,
    "び": 0x3F,
    "ぶ": 0x40,
    "べ": 0x41,
    "ぼ": 0x42,
    "ぱ": 0x43,
    "ぴ": 0x44,
    "ぷ": 0x45,
    "ぺ": 0x46,
    "ぽ": 0x47,
    "ぁ": 0x48,
    "ぃ": 0x49,
    "ぅ": 0x4A,
    "ぇ": 0x4B,
    "ぉ": 0x4C,
    "っ": 0x4D,
    "ゃ": 0x4E,
    "ゅ": 0x4F,
    "ょ": 0x50,
    "ア": 0x51,
    "イ": 0x52,
    "ウ": 0x53,
    "エ": 0x54,
    "オ": 0x55,
    "カ": 0x56,
    "キ": 0x57,
    "ク": 0x58,
    "ケ": 0x59,
    "コ": 0x5A,
    "サ": 0x5B,
    "シ": 0x5C,
    "ス": 0x5D,
    "セ": 0x5E,
    "ソ": 0x5F,
    "タ": 0x60,
    "チ": 0x61,
    "ツ": 0x62,
    "テ": 0x63,
    "ト": 0x64,
    "ナ": 0x65,
    "ニ": 0x66,
    "ヌ": 0x67,
    "ネ": 0x68,
    "ノ": 0x69,
    "ハ": 0x6A,
    "ヒ": 0x6B,
    "フ": 0x6C,
    "ヘ": 0x6D,
    "ホ": 0x6E,
    "マ": 0x6F,
    "ミ": 0x70,
    "ム": 0x71,
    "メ": 0x72,
    "モ": 0x73,
    "ヤ": 0x74,
    "ユ": 0x75,
    "ヨ": 0x76,
    "ラ": 0x77,
    "リ": 0x78,
    "ル": 0x79,
    "レ": 0x7A,
    "ロ": 0x7B,
    "ワ": 0x7C,
    "ヲ": 0x7D,
    "ン": 0x7E,
    "ヴ": 0x7F,
    "ガ": 0x80,
    "ギ": 0x81,
    "グ": 0x82,
    "ゲ": 0x83,
    "ゴ": 0x84,
    "ザ": 0x85,
    "ジ": 0x86,
    "ズ": 0x87,
    "ゼ": 0x88,
    "ゾ": 0x89,
    "ダ": 0x8A,
    "ヂ": 0x8B,
    "ヅ": 0x8C,
    "デ": 0x8D,
    "ド": 0x8E,
    "バ": 0x8F,
    "ビ": 0x90,
    "ブ": 0x91,
    "ベ": 0x92,
    "ボ": 0x93,
    "パ": 0x94,
    "ピ": 0x95,
    "プ": 0x96,
    "ペ": 0x97,
    "ポ": 0x98,
    "ァ": 0x99,
    "ィ": 0x9A,
    "ゥ": 0x9B,
    "ェ": 0x9C,
    "ォ": 0x9D,
    "ッ": 0x9E,
    "ャ": 0x9F,
    "ュ": 0xA0,
    "ョ": 0xA1,
    "ー": 0xA2,
    "～": 0xA3,
    "~": 0xA3,
    # "―": 0xA4,
    # "―": 0xA5,
    # "―": 0xA6,
    "０": 0xA7,
    "0": 0xA7,
    "１": 0xA8,
    "1": 0xA8,
    "２": 0xA9,
    "2": 0xA9,
    "３": 0xAA,
    "3": 0xAA,
    "４": 0xAB,
    "4": 0xAB,
    "５": 0xAC,
    "5": 0xAC,
    "６": 0xAD,
    "6": 0xAD,
    "７": 0xAE,
    "7": 0xAE,
    "８": 0xAF,
    "8": 0xAF,
    "９": 0xB0,
    "9": 0xB0,
    "[up]": 0xB1,
    "[down]": 0xB2,
    "[left]": 0xB3,
    "[right]": 0xB4,
    "！": 0xB5,
    "!": 0xB5,
    "？": 0xB6,
    "?": 0xB6,
    "＋": 0xB7,
    "+": 0xB7,
    "－": 0xB8,
    "／": 0xB9,
    "/": 0xB9,
    "．": 0xBA,
    ".": 0xBA,
    "＆": 0xBB,
    "&": 0xBB,
    "＃": 0xBC,
    "#": 0xBC,
    "[heart]": 0xBD,
    "[star]": 0xBE,
    "（": 0xBF,
    "(": 0xBF,
    "）": 0xC0,
    ")": 0xC0,
    "『": 0xC1,
    "』": 0xC2,
    "・": 0xC3,
    "[hiragana smalln]": 0xC4,
    "[katakana smalln]": 0xC5,
    "星": 0xC6,
    # 0xC7 is unused
    " ": 0xF7,
    "　": 0xF8,
}

CHARSET_LATIN = {
    "A": 0x00,
    "Ａ": 0x00,
    "B": 0x01,
    "Ｂ": 0x01,
    "C": 0x02,
    "Ｃ": 0x02,
    "D": 0x03,
    "Ｄ": 0x03,
    "E": 0x04,
    "Ｅ": 0x04,
    "F": 0x05,
    "Ｆ": 0x05,
    "G": 0x06,
    "Ｇ": 0x06,
    "H": 0x07,
    "Ｈ": 0x07,
    "I": 0x08,
    "Ｉ": 0x08,
    "J": 0x09,
    "Ｊ": 0x09,
    "K": 0x0A,
    "Ｋ": 0x0A,
    "L": 0x0B,
    "Ｌ": 0x0B,
    "M": 0x0C,
    "Ｍ": 0x0C,
    "N": 0x0D,
    "Ｎ": 0x0D,
    "O": 0x0E,
    "Ｏ": 0x0E,
    "P": 0x0F,
    "Ｐ": 0x0F,
    "Q": 0x10,
    "Ｑ": 0x10,
    "R": 0x11,
    "Ｒ": 0x11,
    "S": 0x12,
    "Ｓ": 0x12,
    "T": 0x13,
    "Ｔ": 0x13,
    "U": 0x14,
    "Ｕ": 0x14,
    "V": 0x15,
    "Ｖ": 0x15,
    "W": 0x16,
    "Ｗ": 0x16,
    "X": 0x17,
    "Ｘ": 0x17,
    "Y": 0x18,
    "Ｙ": 0x18,
    "Z": 0x19,
    "Ｚ": 0x19,
    "z": 0x1A,
    "ｚ": 0x1A,
    " ": 0xF7,
    "　": 0xF8,
}

CHARSET_KANJI = {
    "上": 0x00,
    "下": 0x01,
    "左": 0x02,
    "右": 0x03,
    "中": 0x04,
    "東": 0x05,
    "西": 0x06,
    "南": 0x07,
    "北": 0x08,
    "一": 0x09,
    "二": 0x0A,
    "三": 0x0B,
    "名": 0x0C,
    "国": 0x0D,
    "城": 0x0E,
    "姫": 0x0F,
    "大": 0x10,
    "王": 0x11,
    "花": 0x12,
    "世": 0x13,
    "界": 0x14,
    "草": 0x15,
    "気": 0x16,
    "間": 0x17,
    "門": 0x18,
    "家": 0x19,
    "地": 0x1A,
    "岩": 0x1B,
    "駅": 0x1C,
    "山": 0x1D,
    "海": 0x1E,
    "火": 0x1F,
    "水": 0x20,
    "氷": 0x21,
    "日": 0x22,
    "根": 0x23,
    "雲": 0x24,
    "口": 0x25,
    "原": 0x26,
    "前": 0x27,
    "店": 0x28,
    "天": 0x29,
    "森": 0x2A,
    "木": 0x2B,
    "力": 0x2C,
    "空": 0x2D,
    "人": 0x2E,
    "島": 0x2F,
    "出": 0x30,
    "入": 0x31,
    "本": 0x32,
    "石": 0x33,
    "村": 0x34,
    "休": 0x35,
    "先": 0x36,
    "見": 0x37,
    "近": 0x38,
    "方": 0x39,
    "法": 0x3A,
    "手": 0x3B,
    "紙": 0x3C,
    "引": 0x3D,
    "場": 0x3E,
    "所": 0x3F,
    "使": 0x40,
    "回": 0x41,
    "道": 0x42,
    "物": 0x43,
    "弟": 0x44,
    "子": 0x45,
    "汽": 0x46,
    "車": 0x47,
    "何": 0x48,
    "黒": 0x49,
    "分": 0x4A,
    "時": 0x4B,
    "屋": 0x4C,
    "音": 0x4D,
    "目": 0x4E,
    "行": 0x4F,
    "絵": 0x50,
    "月": 0x51,
    "野": 0x52,
    "外": 0x53,
    "図": 0x54,
    "部": 0x55,
    "小": 0x56,
    "風": 0x57,
    "魔": 0x58,
    "元": 0x59,
    "太": 0x5A,
    "陽": 0x5B,
    "実": 0x5C,
    "赤": 0x5D,
    "雪": 0x5E,
    "谷": 0x5F,
    "通": 0x60,
    "[circle]": 0x61,
    "[cross]": 0x62,
    "長": 0x63,
    "話": 0x64,
    "色": 0x65,
    "光": 0x66,
    "合": 0x67,
    "青": 0x68,
    "黄": 0x69,
    "[note]": 0x6A,
    "当": 0x6B,
    "数": 0x6C,
    "兄": 0x6D,
    "用": 0x6E,
    "心": 0x6F,
    "今": 0x70,
    "正": 0x71,
    "直": 0x72,
    "全": 0x73,
    "体": 0x74,
    "夜": 0x75,
    "面": 0x76,
    "虫": 0x77,
    "ｘ": 0x78,
    "x": 0x78,
    " ": 0xF7,
    "　": 0xF8,
}

CHARSET_BUTTONS = {
    "[~a]": 0x00,
    "[~b]": 0x01,
    "[~start]": 0x02,
    "[~c-up]": 0x03,
    "[~c-down]": 0x04,
    "[~c-left]": 0x05,
    "[~c-right]": 0x06,
    "[~z]": 0x07,
    "[~l]": 0x08,
    "[~r]": 0x09,
}

CHARSET_IQUE = {
    " ": 0xF7,
    "栗": [0x5F, 0x01],
    "萨": [0x5F, 0x02],
    "带": [0x5F, 0x03],
    "块": [0x5F, 0x04],
    "珍": [0x5F, 0x05],
    "美": [0x5F, 0x06],
    "啧": [0x5F, 0x07],
    "荣": [0x5F, 0x08],
    "吗": [0x5F, 0x09],
    "前": [0x5F, 0x0A],
    "破": [0x5F, 0x0B],
    "孙": [0x5F, 0x0C],
    "因": [0x5F, 0x0D],
    "激": [0x5F, 0x0E],
    "曾": [0x5F, 0x0F],
    "震": [0x5F, 0x10],
    "芒": [0x5F, 0x11],
    "峰": [0x5F, 0x12],
    "略": [0x5F, 0x13],
    "居": [0x5F, 0x14],
    "课": [0x5F, 0x15],
    "父": [0x5F, 0x16],
    "扑": [0x5F, 0x17],
    "数": [0x5F, 0x18],
    "腻": [0x5F, 0x19],
    "吸": [0x5F, 0x1A],
    "化": [0x5F, 0x1B],
    "死": [0x5F, 0x1C],
    "逸": [0x5F, 0x1D],
    "婪": [0x5F, 0x1E],
    "售": [0x5F, 0x1F],
    "采": [0x5F, 0x20],
    "质": [0x5F, 0x21],
    "积": [0x5F, 0x22],
    "窄": [0x5F, 0x23],
    "徘": [0x5F, 0x24],
    "睦": [0x5F, 0x25],
    "悉": [0x5F, 0x26],
    "骂": [0x5F, 0x27],
    "践": [0x5F, 0x28],
    "益": [0x5F, 0x29],
    "魄": [0x5F, 0x2A],
    "侧": [0x5F, 0x2B],
    "辖": [0x5F, 0x2C],
    "酱": [0x5F, 0x2D],
    "咐": [0x5F, 0x2E],
    "剔": [0x5F, 0x2F],
    "妇": [0x5F, 0x30],
    "掀": [0x5F, 0x31],
    "子": [0x60, 0x01],
    "秃": [0x60, 0x02],
    "长": [0x60, 0x03],
    "埃": [0x60, 0x04],
    "贵": [0x60, 0x05],
    "光": [0x60, 0x06],
    "嗯": [0x60, 0x07],
    "幸": [0x60, 0x08],
    "倒": [0x60, 0x09],
    "探": [0x60, 0x0A],
    "坏": [0x60, 0x0B],
    "进": [0x60, 0x0C],
    "此": [0x60, 0x0D],
    "动": [0x60, 0x0E],
    "害": [0x60, 0x0F],
    "危": [0x60, 0x10],
    "强": [0x60, 0x11],
    "雷": [0x60, 0x12],
    "玻": [0x60, 0x13],
    "企": [0x60, 0x14],
    "程": [0x60, 0x15],
    "贝": [0x60, 0x16],
    "鼻": [0x60, 0x17],
    "声": [0x60, 0x18],
    "莓": [0x60, 0x19],
    "盾": [0x60, 0x1A],
    "汽": [0x60, 0x1B],
    "衰": [0x60, 0x1C],
    "魁": [0x60, 0x1D],
    "率": [0x60, 0x1E],
    "广": [0x60, 0x1F],
    "捉": [0x60, 0x20],
    "埋": [0x60, 0x21],
    "鸦": [0x60, 0x22],
    "仆": [0x60, 0x23],
    "徊": [0x60, 0x24],
    "脾": [0x60, 0x25],
    "含": [0x60, 0x26],
    "蔼": [0x60, 0x27],
    "艳": [0x60, 0x28],
    "阵": [0x60, 0x29],
    "宜": [0x60, 0x2A],
    "罕": [0x60, 0x2B],
    "肚": [0x60, 0x2C],
    "娱": [0x60, 0x2D],
    "妄": [0x60, 0x2E],
    "栅": [0x60, 0x2F],
    "郑": [0x60, 0x30],
    "兼": [0x60, 0x31],
    "小": [0x61, 0x01],
    "鹫": [0x61, 0x02],
    "矛": [0x61, 0x03],
    "莫": [0x61, 0x04],
    "的": [0x61, 0x05],
    "就": [0x61, 0x06],
    "好": [0x61, 0x07],
    "某": [0x61, 0x08],
    "提": [0x61, 0x09],
    "十": [0x61, 0x0A],
    "果": [0x61, 0x0B],
    "疑": [0x61, 0x0C],
    "忙": [0x61, 0x0D],
    "及": [0x61, 0x0E],
    "怕": [0x61, 0x0F],
    "使": [0x61, 0x10],
    "烈": [0x61, 0x11],
    "沙": [0x61, 0x12],
    "尼": [0x61, 0x13],
    "鹅": [0x61, 0x14],
    "晚": [0x61, 0x15],
    "胆": [0x61, 0x16],
    "坚": [0x61, 0x17],
    "警": [0x61, 0x18],
    "搅": [0x61, 0x19],
    "避": [0x61, 0x1A],
    "脸": [0x61, 0x1B],
    "淘": [0x61, 0x1C],
    "梧": [0x61, 0x1D],
    "幅": [0x61, 0x1E],
    "温": [0x61, 0x1F],
    "占": [0x61, 0x20],
    "喝": [0x61, 0x21],
    "叉": [0x61, 0x22],
    "搞": [0x61, 0x23],
    "务": [0x61, 0x24],
    "挤": [0x61, 0x25],
    "欲": [0x61, 0x26],
    "骚": [0x61, 0x27],
    "驴": [0x61, 0x28],
    "刷": [0x61, 0x29],
    "赫": [0x61, 0x2A],
    "刮": [0x61, 0x2B],
    "勃": [0x61, 0x2C],
    "卧": [0x61, 0x2D],
    "堕": [0x61, 0x2E],
    "睹": [0x61, 0x2F],
    "综": [0x61, 0x30],
    "瞧": [0x61, 0x31],
    "刺": [0x62, 0x01],
    "哥": [0x62, 0x02],
    "毒": [0x62, 0x03],
    "斯": [0x62, 0x04],
    "星": [0x62, 0x05],
    "是": [0x62, 0x06],
    "任": [0x62, 0x07],
    "受": [0x62, 0x08],
    "醒": [0x62, 0x09],
    "分": [0x62, 0x0A],
    "村": [0x62, 0x0B],
    "问": [0x62, 0x0C],
    "闻": [0x62, 0x0D],
    "迎": [0x62, 0x0E],
    "肯": [0x62, 0x0F],
    "伦": [0x62, 0x10],
    "拥": [0x62, 0x11],
    "漠": [0x62, 0x12],
    "拓": [0x62, 0x13],
    "格": [0x62, 0x14],
    "愚": [0x62, 0x15],
    "接": [0x62, 0x16],
    "叶": [0x62, 0x17],
    "型": [0x62, 0x18],
    "拌": [0x62, 0x19],
    "钧": [0x62, 0x1A],
    "类": [0x62, 0x1B],
    "捣": [0x62, 0x1C],
    "捡": [0x62, 0x1D],
    "伏": [0x62, 0x1E],
    "暖": [0x62, 0x1F],
    "座": [0x62, 0x20],
    "肤": [0x62, 0x21],
    "轨": [0x62, 0x22],
    "右": [0x62, 0x23],
    "悠": [0x62, 0x24],
    "社": [0x62, 0x25],
    "圣": [0x62, 0x26],
    "拔": [0x62, 0x27],
    "犟": [0x62, 0x28],
    "梳": [0x62, 0x29],
    "诱": [0x62, 0x2A],
    "池": [0x62, 0x2B],
    "卓": [0x62, 0x2C],
    "框": [0x62, 0x2D],
    "君": [0x62, 0x2E],
    "敛": [0x62, 0x2F],
    "嫌": [0x62, 0x30],
    "斤": [0x62, 0x31],
    "头": [0x63, 0x01],
    "哈": [0x63, 0x02],
    "壳": [0x63, 0x03],
    "枯": [0x63, 0x04],
    "杖": [0x63, 0x05],
    "对": [0x63, 0x06],
    "何": [0x63, 0x07],
    "尊": [0x63, 0x08],
    "跟": [0x63, 0x09],
    "耶": [0x63, 0x0A],
    "英": [0x63, 0x0B],
    "赛": [0x63, 0x0C],
    "各": [0x63, 0x0D],
    "渴": [0x63, 0x0E],
    "莉": [0x63, 0x0F],
    "绑": [0x63, 0x10],
    "繁": [0x63, 0x11],
    "置": [0x63, 0x12],
    "莱": [0x63, 0x13],
    "细": [0x63, 0x14],
    "蠢": [0x63, 0x15],
    "剩": [0x63, 0x16],
    "深": [0x63, 0x17],
    "饰": [0x63, 0x18],
    "拿": [0x63, 0x19],
    "良": [0x63, 0x1A],
    "扔": [0x63, 0x1B],
    "跺": [0x63, 0x1C],
    "嘴": [0x63, 0x1D],
    "坑": [0x63, 0x1E],
    "遭": [0x63, 0x1F],
    "桥": [0x63, 0x20],
    "祖": [0x63, 0x21],
    "承": [0x63, 0x22],
    "膝": [0x63, 0x23],
    "赋": [0x63, 0x24],
    "怂": [0x63, 0x25],
    "凋": [0x63, 0x26],
    "详": [0x63, 0x27],
    "沟": [0x63, 0x28],
    "撤": [0x63, 0x29],
    "攥": [0x63, 0x2A],
    "绳": [0x63, 0x2B],
    "枕": [0x63, 0x2C],
    "插": [0x63, 0x2D],
    "絞": [0x63, 0x2E],
    "萌": [0x63, 0x2F],
    "亡": [0x63, 0x30],
    "撑": [0x63, 0x31],
    "飞": [0x64, 0x01],
    "金": [0x64, 0x02],
    "炎": [0x64, 0x03],
    "骨": [0x64, 0x04],
    "从": [0x64, 0x05],
    "真": [0x64, 0x06],
    "言": [0x64, 0x07],
    "敬": [0x64, 0x08],
    "起": [0x64, 0x09],
    "交": [0x64, 0x0A],
    "雄": [0x64, 0x0B],
    "只": [0x64, 0x0C],
    "邪": [0x64, 0x0D],
    "秘": [0x64, 0x0E],
    "波": [0x64, 0x0F],
    "架": [0x64, 0x10],
    "衷": [0x64, 0x11],
    "烹": [0x64, 0x12],
    "咒": [0x64, 0x13],
    "侵": [0x64, 0x14],
    "称": [0x64, 0x15],
    "余": [0x64, 0x16],
    "稀": [0x64, 0x17],
    "古": [0x64, 0x18],
    "洗": [0x64, 0x19],
    "Ａ": [0x64, 0x1A],
    "式": [0x64, 0x1B],
    "脚": [0x64, 0x1C],
    "吐": [0x64, 0x1D],
    "建": [0x64, 0x1E],
    "袭": [0x64, 0x1F],
    "修": [0x64, 0x20],
    "胃": [0x64, 0x21],
    "钮": [0x64, 0x22],
    "富": [0x64, 0x23],
    "倔": [0x64, 0x24],
    "恿": [0x64, 0x25],
    "零": [0x64, 0x26],
    "支": [0x64, 0x27],
    "岔": [0x64, 0x28],
    "悚": [0x64, 0x29],
    "赊": [0x64, 0x2A],
    "枝": [0x64, 0x2B],
    "绅": [0x64, 0x2C],
    "讶": [0x64, 0x2D],
    "晬": [0x64, 0x2E],
    "绍": [0x64, 0x2F],
    "讯": [0x64, 0x30],
    "订": [0x64, 0x31],
    "行": [0x65, 0x01],
    "面": [0x65, 0x02],
    "芽": [0x65, 0x03],
    "魔": [0x65, 0x04],
    "底": [0x65, 0x05],
    "打": [0x65, 0x06],
    "语": [0x65, 0x07],
    "家": [0x65, 0x08],
    "些": [0x65, 0x09],
    "完": [0x65, 0x0A],
    "顺": [0x65, 0x0B],
    "玩": [0x65, 0x0C],
    "恶": [0x65, 0x0D],
    "凡": [0x65, 0x0E],
    "园": [0x65, 0x0F],
    "迹": [0x65, 0x10],
    "题": [0x65, 0x11],
    "饪": [0x65, 0x12],
    "糕": [0x65, 0x13],
    "守": [0x65, 0x14],
    "奖": [0x65, 0x15],
    "抗": [0x65, 0x16],
    "烤": [0x65, 0x17],
    "娃": [0x65, 0x18],
    "餐": [0x65, 0x19],
    "Ｄ": [0x65, 0x1A],
    "射": [0x65, 0x1B],
    "碍": [0x65, 0x1C],
    "棍": [0x65, 0x1D],
    "议": [0x65, 0x1E],
    "闹": [0x65, 0x1F],
    "晴": [0x65, 0x20],
    "狗": [0x65, 0x21],
    "伸": [0x65, 0x22],
    "堂": [0x65, 0x23],
    "执": [0x65, 0x24],
    "途": [0x65, 0x25],
    "争": [0x65, 0x26],
    "伫": [0x65, 0x27],
    "脉": [0x65, 0x28],
    "轰": [0x65, 0x29],
    "憾": [0x65, 0x2A],
    "茂": [0x65, 0x2B],
    "咆": [0x65, 0x2C],
    "钉": [0x65, 0x2D],
    "痕": [0x65, 0x2E],
    "屑": [0x65, 0x2F],
    "咽": [0x65, 0x30],
    "驯": [0x65, 0x31],
    "红": [0x66, 0x01],
    "龟": [0x66, 0x02],
    "钳": [0x66, 0x03],
    "法": [0x66, 0x04],
    "里": [0x66, 0x05],
    "败": [0x66, 0x06],
    "无": [0x66, 0x07],
    "梦": [0x66, 0x08],
    "伙": [0x66, 0x09],
    "全": [0x66, 0x0A],
    "消": [0x66, 0x0B],
    "笑": [0x66, 0x0C],
    "忧": [0x66, 0x0D],
    "智": [0x66, 0x0E],
    "越": [0x66, 0x0F],
    "恩": [0x66, 0x10],
    "宇": [0x66, 0x11],
    "凯": [0x66, 0x12],
    "幽": [0x66, 0x13],
    "据": [0x66, 0x14],
    "励": [0x66, 0x15],
    "持": [0x66, 0x16],
    "源": [0x66, 0x17],
    "罐": [0x66, 0x18],
    "清": [0x66, 0x19],
    "Ｂ": [0x66, 0x1A],
    "断": [0x66, 0x1B],
    "备": [0x66, 0x1C],
    "仿": [0x66, 0x1D],
    "赴": [0x66, 0x1E],
    "供": [0x66, 0x1F],
    "朗": [0x66, 0x20],
    "般": [0x66, 0x21],
    "五": [0x66, 0x22],
    "璃": [0x66, 0x23],
    "髦": [0x66, 0x24],
    "疏": [0x66, 0x25],
    "逗": [0x66, 0x26],
    "猖": [0x66, 0x27],
    "掘": [0x66, 0x28],
    "释": [0x66, 0x29],
    "巅": [0x66, 0x2A],
    "坪": [0x66, 0x2B],
    "哮": [0x66, 0x2C],
    "贺": [0x66, 0x2D],
    "锢": [0x66, 0x2E],
    "咂": [0x66, 0x2F],
    "唐": [0x66, 0x30],
    "堪": [0x66, 0x31],
    "色": [0x67, 0x01],
    "汪": [0x67, 0x02],
    "朱": [0x67, 0x03],
    "卫": [0x67, 0x04],
    "感": [0x67, 0x05],
    "狠": [0x67, 0x06],
    "表": [0x67, 0x07],
    "朋": [0x67, 0x08],
    "伴": [0x67, 0x09],
    "解": [0x67, 0x0A],
    "息": [0x67, 0x0B],
    "尽": [0x67, 0x0C],
    "虑": [0x67, 0x0D],
    "慧": [0x67, 0x0E],
    "愉": [0x67, 0x0F],
    "客": [0x67, 0x10],
    "碎": [0x67, 0x11],
    "茜": [0x67, 0x12],
    "床": [0x67, 0x13],
    "或": [0x67, 0x14],
    "厉": [0x67, 0x15],
    "焦": [0x67, 0x16],
    "汁": [0x67, 0x17],
    "月": [0x67, 0x18],
    "洁": [0x67, 0x19],
    "Ｅ": [0x67, 0x1A],
    "突": [0x67, 0x1B],
    "武": [0x67, 0x1C],
    "替": [0x67, 0x1D],
    "磋": [0x67, 0x1E],
    "妈": [0x67, 0x1F],
    "據": [0x67, 0x20],
    "窝": [0x67, 0x21],
    "宽": [0x67, 0x22],
    "闭": [0x67, 0x23],
    "卦": [0x67, 0x24],
    "凑": [0x67, 0x25],
    "侦": [0x67, 0x26],
    "獗": [0x67, 0x27],
    "政": [0x67, 0x28],
    "愁": [0x67, 0x29],
    "枚": [0x67, 0x2A],
    "褐": [0x67, 0x2B],
    "恰": [0x67, 0x2C],
    "廉": [0x67, 0x2D],
    "槽": [0x67, 0x2E],
    "妒": [0x67, 0x2F],
    "诡": [0x67, 0x30],
    "摄": [0x67, 0x31],
    "蓝": [0x68, 0x01],
    "食": [0x68, 0x02],
    "盖": [0x68, 0x03],
    "锤": [0x68, 0x04],
    "会": [0x68, 0x05],
    "地": [0x68, 0x06],
    "达": [0x68, 0x07],
    "友": [0x68, 0x08],
    "怎": [0x68, 0x09],
    "方": [0x68, 0x0A],
    "闲": [0x68, 0x0B],
    "情": [0x68, 0x0C],
    "兴": [0x68, 0x0D],
    "痛": [0x68, 0x0E],
    "既": [0x68, 0x0F],
    "砰": [0x68, 0x10],
    "颜": [0x68, 0x11],
    "制": [0x68, 0x12],
    "单": [0x68, 0x13],
    "睬": [0x68, 0x14],
    "论": [0x68, 0x15],
    "急": [0x68, 0x16],
    "熟": [0x68, 0x17],
    "角": [0x68, 0x18],
    "剂": [0x68, 0x19],
    "Ｃ": [0x68, 0x1A],
    "尚": [0x68, 0x1B],
    "涉": [0x68, 0x1C],
    "扮": [0x68, 0x1D],
    "犹": [0x68, 0x1E],
    "配": [0x68, 0x1F],
    "研": [0x68, 0x20],
    "窜": [0x68, 0x21],
    "阔": [0x68, 0x22],
    "旺": [0x68, 0x23],
    "谣": [0x68, 0x24],
    "陌": [0x68, 0x25],
    "焉": [0x68, 0x26],
    "奴": [0x68, 0x27],
    "慌": [0x68, 0x28],
    "焚": [0x68, 0x29],
    "秒": [0x68, 0x2A],
    "毯": [0x68, 0x2B],
    "溃": [0x68, 0x2C],
    "妥": [0x68, 0x2D],
    "仇": [0x68, 0x2E],
    "忌": [0x68, 0x2F],
    "戒": [0x68, 0x30],
    "拾": [0x68, 0x31],
    "王": [0x69, 0x01],
    "人": [0x69, 0x02],
    "蜜": [0x69, 0x03],
    "巫": [0x69, 0x04],
    "用": [0x69, 0x05],
    "教": [0x69, 0x06],
    "所": [0x69, 0x07],
    "没": [0x69, 0x08],
    "嗨": [0x69, 0x09],
    "风": [0x69, 0x0A],
    "荡": [0x69, 0x0B],
    "享": [0x69, 0x0C],
    "甜": [0x69, 0x0D],
    "苦": [0x69, 0x0E],
    "简": [0x69, 0x0F],
    "姐": [0x69, 0x10],
    "砖": [0x69, 0x11],
    "柠": [0x69, 0x12],
    "撞": [0x69, 0x13],
    "私": [0x69, 0x14],
    "两": [0x69, 0x15],
    "俘": [0x69, 0x16],
    "瓜": [0x69, 0x17],
    "男": [0x69, 0x18],
    "烘": [0x69, 0x19],
    "Ｆ": [0x69, 0x1A],
    "咖": [0x69, 0x1B],
    "驾": [0x69, 0x1C],
    "众": [0x69, 0x1D],
    "豫": [0x69, 0x1E],
    "饲": [0x69, 0x1F],
    "沿": [0x69, 0x20],
    "且": [0x69, 0x21],
    "舌": [0x69, 0x22],
    "摔": [0x69, 0x23],
    "讲": [0x69, 0x24],
    "织": [0x69, 0x25],
    "卤": [0x69, 0x26],
    "役": [0x69, 0x27],
    "悲": [0x69, 0x28],
    "稽": [0x69, 0x29],
    "辱": [0x69, 0x2A],
    "淡": [0x69, 0x2B],
    "咕": [0x69, 0x2C],
    "衬": [0x69, 0x2D],
    "弦": [0x69, 0x2E],
    "辑": [0x69, 0x2F],
    "勺": [0x69, 0x30],
    "树": [0x6A, 0x01],
    "花": [0x6A, 0x02],
    "蜂": [0x6A, 0x03],
    "婆": [0x6A, 0x04],
    "它": [0x6A, 0x05],
    "训": [0x6A, 0x06],
    "切": [0x6A, 0x07],
    "什": [0x6A, 0x08],
    "邮": [0x6A, 0x09],
    "把": [0x6A, 0x0A],
    "差": [0x6A, 0x0B],
    "咳": [0x6A, 0x0C],
    "伤": [0x6A, 0x0D],
    "誓": [0x6A, 0x0E],
    "徒": [0x6A, 0x0F],
    "穿": [0x6A, 0x10],
    "技": [0x6A, 0x11],
    "檬": [0x6A, 0x12],
    "椰": [0x6A, 0x13],
    "致": [0x6A, 0x14],
    "优": [0x6A, 0x15],
    "虏": [0x6A, 0x16],
    "药": [0x6A, 0x17],
    "肖": [0x6A, 0x18],
    "牛": [0x6A, 0x19],
    "注": [0x6A, 0x1A],
    "喱": [0x6A, 0x1B],
    "驶": [0x6A, 0x1C],
    "喂": [0x6A, 0x1D],
    "术": [0x6A, 0x1E],
    "农": [0x6A, 0x1F],
    "踏": [0x6A, 0x20],
    "桩": [0x6A, 0x21],
    "绵": [0x6A, 0x22],
    "氛": [0x6A, 0x23],
    "疲": [0x6A, 0x24],
    "统": [0x6A, 0x25],
    "莽": [0x6A, 0x26],
    "辜": [0x6A, 0x27],
    "躁": [0x6A, 0x28],
    "兔": [0x6A, 0x29],
    "箭": [0x6A, 0x2A],
    "央": [0x6A, 0x2B],
    "哝": [0x6A, 0x2C],
    "衫": [0x6A, 0x2D],
    "缉": [0x6A, 0x2E],
    "箩": [0x6A, 0x2F],
    "飕": [0x6A, 0x30],
    "喏": [0x6B, 0x01],
    "超": [0x6B, 0x02],
    "催": [0x6B, 0x03],
    "钱": [0x6B, 0x04],
    "给": [0x6B, 0x05],
    "他": [0x6B, 0x06],
    "国": [0x6B, 0x07],
    "介": [0x6B, 0x08],
    "递": [0x6B, 0x09],
    "吹": [0x6B, 0x0A],
    "劲": [0x6B, 0x0B],
    "干": [0x6B, 0x0C],
    "放": [0x6B, 0x0D],
    "末": [0x6B, 0x0E],
    "劳": [0x6B, 0x0F],
    "衣": [0x6B, 0x10],
    "裤": [0x6B, 0x11],
    "糖": [0x6B, 0x12],
    "苹": [0x6B, 0x13],
    "适": [0x6B, 0x14],
    "秀": [0x6B, 0x15],
    "惊": [0x6B, 0x16],
    "区": [0x6B, 0x17],
    "钥": [0x6B, 0x18],
    "奶": [0x6B, 0x19],
    "踱": [0x6B, 0x1A],
    "烦": [0x6B, 0x1B],
    "漂": [0x6B, 0x1C],
    "锻": [0x6B, 0x1D],
    "惕": [0x6B, 0x1E],
    "市": [0x6B, 0x1F],
    "拱": [0x6B, 0x20],
    "弯": [0x6B, 0x21],
    "滩": [0x6B, 0x22],
    "纯": [0x6B, 0x23],
    "惫": [0x6B, 0x24],
    "辨": [0x6B, 0x25],
    "播": [0x6B, 0x26],
    "肆": [0x6B, 0x27],
    "损": [0x6B, 0x28],
    "臂": [0x6B, 0x29],
    "哭": [0x6B, 0x2A],
    "六": [0x6B, 0x2B],
    "诀": [0x6B, 0x2C],
    "饶": [0x6B, 0x2D],
    "授": [0x6B, 0x2E],
    "勉": [0x6B, 0x2F],
    "劈": [0x6B, 0x30],
    "库": [0x6C, 0x01],
    "级": [0x6C, 0x02],
    "眠": [0x6C, 0x03],
    "乌": [0x6C, 0x04],
    "这": [0x6C, 0x05],
    "多": [0x6C, 0x06],
    "整": [0x6C, 0x07],
    "意": [0x6C, 0x08],
    "员": [0x6C, 0x09],
    "唷": [0x6C, 0x0A],
    "丈": [0x6C, 0x0B],
    "伟": [0x6C, 0x0C],
    "然": [0x6C, 0x0D],
    "四": [0x6C, 0x0E],
    "冷": [0x6C, 0x0F],
    "服": [0x6C, 0x10],
    "左": [0x6C, 0x11],
    "除": [0x6C, 0x12],
    "酸": [0x6C, 0x13],
    "足": [0x6C, 0x14],
    "未": [0x6C, 0x15],
    "病": [0x6C, 0x16],
    "硬": [0x6C, 0x17],
    "匙": [0x6C, 0x18],
    "赢": [0x6C, 0x19],
    "寒": [0x6C, 0x1A],
    "软": [0x6C, 0x1B],
    "浮": [0x6C, 0x1C],
    "炼": [0x6C, 0x1D],
    "泼": [0x6C, 0x1E],
    "民": [0x6C, 0x1F],
    "墙": [0x6C, 0x20],
    "缘": [0x6C, 0x21],
    "炽": [0x6C, 0x22],
    "粹": [0x6C, 0x23],
    "聊": [0x6C, 0x24],
    "血": [0x6C, 0x25],
    "短": [0x6C, 0x26],
    "虐": [0x6C, 0x27],
    "喊": [0x6C, 0x28],
    "牌": [0x6C, 0x29],
    "睁": [0x6C, 0x2A],
    "返": [0x6C, 0x2B],
    "碌": [0x6C, 0x2C],
    "疼": [0x6C, 0x2D],
    "芮": [0x6C, 0x2E],
    "黏": [0x6C, 0x2F],
    "猫": [0x6C, 0x30],
    "啪": [0x6D, 0x01],
    "加": [0x6D, 0x02],
    "神": [0x6D, 0x03],
    "电": [0x6D, 0x04],
    "个": [0x6D, 0x05],
    "么": [0x6D, 0x06],
    "蘑": [0x6D, 0x07],
    "必": [0x6D, 0x08],
    "工": [0x6D, 0x09],
    "终": [0x6D, 0x0A],
    "夫": [0x6D, 0x0B],
    "斗": [0x6D, 0x0C],
    "恙": [0x6D, 0x0D],
    "奔": [0x6D, 0x0E],
    "丢": [0x6D, 0x0F],
    "传": [0x6D, 0x10],
    "边": [0x6D, 0x11],
    "迷": [0x6D, 0x12],
    "橙": [0x6D, 0x13],
    "集": [0x6D, 0x14],
    "职": [0x6D, 0x15],
    "慢": [0x6D, 0x16],
    "铃": [0x6D, 0x17],
    "储": [0x6D, 0x18],
    "厦": [0x6D, 0x19],
    "赔": [0x6D, 0x1A],
    "敏": [0x6D, 0x1B],
    "族": [0x6D, 0x1C],
    "胖": [0x6D, 0x1D],
    "袍": [0x6D, 0x1E],
    "宅": [0x6D, 0x1F],
    "澡": [0x6D, 0x20],
    "低": [0x6D, 0x21],
    "蔽": [0x6D, 0x22],
    "沉": [0x6D, 0x23],
    "背": [0x6D, 0x24],
    "沸": [0x6D, 0x25],
    "尤": [0x6D, 0x26],
    "祸": [0x6D, 0x27],
    "掩": [0x6D, 0x28],
    "鲁": [0x6D, 0x29],
    "酬": [0x6D, 0x2A],
    "拨": [0x6D, 0x2B],
    "篝": [0x6D, 0x2C],
    "祥": [0x6D, 0x2D],
    "抚": [0x6D, 0x2E],
    "填": [0x6D, 0x2F],
    "俏": [0x6D, 0x30],
    "嗒": [0x6E, 0x01],
    "邦": [0x6E, 0x02],
    "奇": [0x6E, 0x03],
    "宝": [0x6E, 0x04],
    "世": [0x6E, 0x05],
    "另": [0x6E, 0x06],
    "菇": [0x6E, 0x07],
    "须": [0x6E, 0x08],
    "作": [0x6E, 0x09],
    "盛": [0x6E, 0x0A],
    "妻": [0x6E, 0x0B],
    "庆": [0x6E, 0x0C],
    "三": [0x6E, 0x0D],
    "相": [0x6E, 0x0E],
    "累": [0x6E, 0x0F],
    "遍": [0x6E, 0x10],
    "扇": [0x6E, 0x11],
    "森": [0x6E, 0x12],
    "调": [0x6E, 0x13],
    "唉": [0x6E, 0x14],
    "隐": [0x6E, 0x15],
    "弱": [0x6E, 0x16],
    "薯": [0x6E, 0x17],
    "设": [0x6E, 0x18],
    "港": [0x6E, 0x19],
    "偿": [0x6E, 0x1A],
    "睛": [0x6E, 0x1B],
    "坦": [0x6E, 0x1C],
    "唠": [0x6E, 0x1D],
    "羡": [0x6E, 0x1E],
    "唯": [0x6E, 0x1F],
    "嘛": [0x6E, 0x20],
    "值": [0x6E, 0x21],
    "春": [0x6E, 0x22],
    "疙": [0x6E, 0x23],
    "夸": [0x6E, 0x24],
    "腾": [0x6E, 0x25],
    "蒜": [0x6E, 0x26],
    "毫": [0x6E, 0x27],
    "溢": [0x6E, 0x28],
    "兜": [0x6E, 0x29],
    "贫": [0x6E, 0x2A],
    "符": [0x6E, 0x2B],
    "熙": [0x6E, 0x2C],
    "毙": [0x6E, 0x2D],
    "嘱": [0x6E, 0x2E],
    "喀": [0x6E, 0x2F],
    "曙": [0x6E, 0x30],
    "绒": [0x6F, 0x01],
    "图": [0x6F, 0x02],
    "史": [0x6F, 0x03],
    "师": [0x6F, 0x04],
    "界": [0x6F, 0x05],
    "难": [0x6F, 0x06],
    "充": [0x6F, 0x07],
    "走": [0x6F, 0x08],
    "呢": [0x6F, 0x09],
    "亲": [0x6F, 0x0A],
    "告": [0x6F, 0x0B],
    "祝": [0x6F, 0x0C],
    "少": [0x6F, 0x0D],
    "耐": [0x6F, 0x0E],
    "喔": [0x6F, 0x0F],
    "彩": [0x6F, 0x10],
    "窗": [0x6F, 0x11],
    "林": [0x6F, 0x12],
    "酒": [0x6F, 0x13],
    "耗": [0x6F, 0x14],
    "才": [0x6F, 0x15],
    "健": [0x6F, 0x16],
    "效": [0x6F, 0x17],
    "妙": [0x6F, 0x18],
    "玛": [0x6F, 0x19],
    "驭": [0x6F, 0x1A],
    "耍": [0x6F, 0x1B],
    "仅": [0x6F, 0x1C],
    "叨": [0x6F, 0x1D],
    "慕": [0x6F, 0x1E],
    "厅": [0x6F, 0x1F],
    "荒": [0x6F, 0x20],
    "吊": [0x6F, 0x21],
    "藤": [0x6F, 0x22],
    "瘩": [0x6F, 0x23],
    "宠": [0x6F, 0x24],
    "穆": [0x6F, 0x25],
    "撼": [0x6F, 0x26],
    "诚": [0x6F, 0x27],
    "验": [0x6F, 0x28],
    "奈": [0x6F, 0x29],
    "穷": [0x6F, 0x2A],
    "猪": [0x6F, 0x2B],
    "攘": [0x6F, 0x2C],
    "铰": [0x6F, 0x2D],
    "裔": [0x6F, 0x2E],
    "滞": [0x6F, 0x2F],
    "囱": [0x6F, 0x30],
    "毛": [0x70, 0x01],
    "巴": [0x70, 0x02],
    "派": [0x70, 0x03],
    "陈": [0x70, 0x04],
    "重": [0x70, 0x05],
    "忘": [0x70, 0x06],
    "满": [0x70, 0x07],
    "非": [0x70, 0x08],
    "普": [0x70, 0x09],
    "笔": [0x70, 0x0A],
    "诉": [0x70, 0x0B],
    "缺": [0x70, 0x0C],
    "旅": [0x70, 0x0D],
    "极": [0x70, 0x0E],
    "旱": [0x70, 0x0F],
    "啸": [0x70, 0x10],
    "首": [0x70, 0x11],
    "击": [0x70, 0x12],
    "顽": [0x70, 0x13],
    "虽": [0x70, 0x14],
    "显": [0x70, 0x15],
    "康": [0x70, 0x16],
    "眩": [0x70, 0x17],
    "具": [0x70, 0x18],
    "归": [0x70, 0x19],
    "双": [0x70, 0x1A],
    "仙": [0x70, 0x1B],
    "模": [0x70, 0x1C],
    "尝": [0x70, 0x1D],
    "肃": [0x70, 0x1E],
    "码": [0x70, 0x1F],
    "芜": [0x70, 0x20],
    "橱": [0x70, 0x21],
    "泉": [0x70, 0x22],
    "煞": [0x70, 0x23],
    "慷": [0x70, 0x24],
    "奏": [0x70, 0x25],
    "攀": [0x70, 0x26],
    "陆": [0x70, 0x27],
    "惦": [0x70, 0x28],
    "赘": [0x70, 0x29],
    "慰": [0x70, 0x2A],
    "屿": [0x70, 0x2B],
    "瘫": [0x70, 0x2C],
    "链": [0x70, 0x2D],
    "芬": [0x70, 0x2E],
    "肠": [0x70, 0x2F],
    "逝": [0x70, 0x30],
    "怪": [0x71, 0x01],
    "布": [0x71, 0x02],
    "克": [0x71, 0x03],
    "李": [0x71, 0x04],
    "新": [0x71, 0x05],
    "量": [0x71, 0x06],
    "欠": [0x71, 0x07],
    "常": [0x71, 0x08],
    "通": [0x71, 0x09],
    "签": [0x71, 0x0A],
    "出": [0x71, 0x0B],
    "席": [0x71, 0x0C],
    "早": [0x71, 0x0D],
    "限": [0x71, 0x0E],
    "落": [0x71, 0x0F],
    "谷": [0x71, 0x10],
    "易": [0x71, 0x11],
    "乱": [0x71, 0x12],
    "皮": [0x71, 0x13],
    "影": [0x71, 0x14],
    "存": [0x71, 0x15],
    "福": [0x71, 0x16],
    "晕": [0x71, 0x17],
    "煎": [0x71, 0x18],
    "眼": [0x71, 0x19],
    "赏": [0x71, 0x1A],
    "惯": [0x71, 0x1B],
    "威": [0x71, 0x1C],
    "拍": [0x71, 0x1D],
    "袖": [0x71, 0x1E],
    "凝": [0x71, 0x1F],
    "陡": [0x71, 0x20],
    "董": [0x71, 0x21],
    "堵": [0x71, 0x22],
    "碾": [0x71, 0x23],
    "慨": [0x71, 0x24],
    "乞": [0x71, 0x25],
    "宿": [0x71, 0x26],
    "魅": [0x71, 0x27],
    "劝": [0x71, 0x28],
    "母": [0x71, 0x29],
    "盼": [0x71, 0x2A],
    "洽": [0x71, 0x2B],
    "痪": [0x71, 0x2C],
    "哔": [0x71, 0x2D],
    "忽": [0x71, 0x2E],
    "畜": [0x71, 0x2F],
    "逞": [0x71, 0x30],
    "炸": [0x72, 0x01],
    "路": [0x72, 0x02],
    "云": [0x72, 0x03],
    "暗": [0x72, 0x04],
    "来": [0x72, 0x05],
    "啊": [0x72, 0x06],
    "很": [0x72, 0x07],
    "直": [0x72, 0x08],
    "封": [0x72, 0x09],
    "名": [0x72, 0x0A],
    "抓": [0x72, 0x0B],
    "培": [0x72, 0x0C],
    "纪": [0x72, 0x0D],
    "脑": [0x72, 0x0E],
    "向": [0x72, 0x0F],
    "婚": [0x72, 0x10],
    "吉": [0x72, 0x11],
    "洲": [0x72, 0x12],
    "学": [0x72, 0x13],
    "联": [0x72, 0x14],
    "份": [0x72, 0x15],
    "颗": [0x72, 0x16],
    "汤": [0x72, 0x17],
    "盘": [0x72, 0x18],
    "镜": [0x72, 0x19],
    "透": [0x72, 0x1A],
    "呼": [0x72, 0x1B],
    "佳": [0x72, 0x1C],
    "挣": [0x72, 0x1D],
    "雅": [0x72, 0x1E],
    "海": [0x72, 0x1F],
    "峭": [0x72, 0x20],
    "昂": [0x72, 0x21],
    "肥": [0x72, 0x22],
    "炬": [0x72, 0x23],
    "借": [0x72, 0x24],
    "丐": [0x72, 0x25],
    "仰": [0x72, 0x26],
    "拆": [0x72, 0x27],
    "惭": [0x72, 0x28],
    "忆": [0x72, 0x29],
    "顿": [0x72, 0x2A],
    "荐": [0x72, 0x2B],
    "伍": [0x72, 0x2C],
    "谬": [0x72, 0x2D],
    "趁": [0x72, 0x2E],
    "搁": [0x72, 0x2F],
    "措": [0x72, 0x30],
    "弹": [0x73, 0x01],
    "之": [0x73, 0x02],
    "雾": [0x73, 0x03],
    "诺": [0x73, 0x04],
    "和": [0x73, 0x05],
    "一": [0x73, 0x06],
    "汀": [0x73, 0x07],
    "样": [0x73, 0x08],
    "信": [0x73, 0x09],
    "太": [0x73, 0x0A],
    "关": [0x73, 0x0B],
    "理": [0x73, 0x0C],
    "念": [0x73, 0x0D],
    "筋": [0x73, 0x0E],
    "炫": [0x73, 0x0F],
    "嘘": [0x73, 0x10],
    "聪": [0x73, 0x11],
    "掉": [0x73, 0x12],
    "者": [0x73, 0x13],
    "押": [0x73, 0x14],
    "乓": [0x73, 0x15],
    "七": [0x73, 0x16],
    "菜": [0x73, 0x17],
    "读": [0x73, 0x18],
    "异": [0x73, 0x19],
    "瓶": [0x73, 0x1A],
    "援": [0x73, 0x1B],
    "衡": [0x73, 0x1C],
    "脱": [0x73, 0x1D],
    "较": [0x73, 0x1E],
    "输": [0x73, 0x1F],
    "卑": [0x73, 0x20],
    "窥": [0x73, 0x21],
    "巢": [0x73, 0x22],
    "惑": [0x73, 0x23],
    "贴": [0x73, 0x24],
    "贩": [0x73, 0x25],
    "仗": [0x73, 0x26],
    "姑": [0x73, 0x27],
    "惧": [0x73, 0x28],
    "策": [0x73, 0x29],
    "嬉": [0x73, 0x2A],
    "冠": [0x73, 0x2B],
    "泪": [0x73, 0x2C],
    "屈": [0x73, 0x2D],
    "呐": [0x73, 0x2E],
    "嚷": [0x73, 0x2F],
    "拖": [0x73, 0x30],
    "兵": [0x74, 0x01],
    "心": [0x74, 0x02],
    "老": [0x74, 0x03],
    "灰": [0x74, 0x04],
    "平": [0x74, 0x05],
    "定": [0x74, 0x06],
    "说": [0x74, 0x07],
    "？": [0x74, 0x08],
    "她": [0x74, 0x09],
    "棒": [0x74, 0x0A],
    "键": [0x74, 0x0B],
    "由": [0x74, 0x0C],
    "品": [0x74, 0x0D],
    "谁": [0x74, 0x0E],
    "遇": [0x74, 0x0F],
    "吓": [0x74, 0x10],
    "栋": [0x74, 0x11],
    "改": [0x74, 0x12],
    "包": [0x74, 0x13],
    "求": [0x74, 0x14],
    "乒": [0x74, 0x15],
    "予": [0x74, 0x16],
    "茶": [0x74, 0x17],
    "懂": [0x74, 0x18],
    "聚": [0x74, 0x19],
    "钻": [0x74, 0x1A],
    "狡": [0x74, 0x1B],
    "矮": [0x74, 0x1C],
    "悬": [0x74, 0x1D],
    "胁": [0x74, 0x1E],
    "踪": [0x74, 0x1F],
    "鄙": [0x74, 0x20],
    "蜡": [0x74, 0x21],
    "纹": [0x74, 0x22],
    "索": [0x74, 0x23],
    "彼": [0x74, 0x24],
    "哒": [0x74, 0x25],
    "姆": [0x74, 0x26],
    "娘": [0x74, 0x27],
    "怖": [0x74, 0x28],
    "艇": [0x74, 0x29],
    "晃": [0x74, 0x2A],
    "构": [0x74, 0x2B],
    "俯": [0x74, 0x2C],
    "睿": [0x74, 0x2D],
    "粒": [0x74, 0x2E],
    "辉": [0x74, 0x2F],
    "逮": [0x74, 0x30],
    "杀": [0x75, 0x01],
    "嘿": [0x75, 0x02],
    "呜": [0x75, 0x03],
    "马": [0x75, 0x04],
    "。": [0x75, 0x05],
    "…": [0x75, 0x06],
    "灵": [0x75, 0x07],
    "如": [0x75, 0x08],
    "写": [0x75, 0x09],
    "绝": [0x75, 0x0A],
    "速": [0x75, 0x0B],
    "甚": [0x75, 0x0C],
    "彻": [0x75, 0x0D],
    "便": [0x75, 0x0E],
    "旋": [0x75, 0x0F],
    "唬": [0x75, 0x10],
    "币": [0x75, 0x11],
    "态": [0x75, 0x12],
    "裹": [0x75, 0x13],
    "严": [0x75, 0x14],
    "寻": [0x75, 0x15],
    "船": [0x75, 0x16],
    "升": [0x75, 0x17],
    "喷": [0x75, 0x18],
    "资": [0x75, 0x19],
    "画": [0x75, 0x1A],
    "猾": [0x75, 0x1B],
    "粗": [0x75, 0x1C],
    "挂": [0x75, 0x1D],
    "拳": [0x75, 0x1E],
    "楼": [0x75, 0x1F],
    "症": [0x75, 0x20],
    "烛": [0x75, 0x21],
    "丝": [0x75, 0x22],
    "俗": [0x75, 0x23],
    "笆": [0x75, 0x24],
    "咚": [0x75, 0x25],
    "秩": [0x75, 0x26],
    "嗅": [0x75, 0x27],
    "挺": [0x75, 0x28],
    "弛": [0x75, 0x29],
    "熔": [0x75, 0x2A],
    "撕": [0x75, 0x2B],
    "涡": [0x75, 0x2C],
    "竭": [0x75, 0x2D],
    "蔚": [0x75, 0x2E],
    "妨": [0x75, 0x2F],
    "慈": [0x75, 0x30],
    "手": [0x76, 0x01],
    "虎": [0x76, 0x02],
    "噜": [0x76, 0x03],
    "力": [0x76, 0x04],
    "现": [0x76, 0x05],
    "不": [0x76, 0x06],
    "证": [0x76, 0x07],
    "夺": [0x76, 0x08],
    "看": [0x76, 0x09],
    "错": [0x76, 0x0A],
    "听": [0x76, 0x0B],
    "至": [0x76, 0x0C],
    "丽": [0x76, 0x0D],
    "久": [0x76, 0x0E],
    "转": [0x76, 0x0F],
    "岸": [0x76, 0x10],
    "预": [0x76, 0x11],
    "第": [0x76, 0x12],
    "艺": [0x76, 0x13],
    "哎": [0x76, 0x14],
    "围": [0x76, 0x15],
    "艘": [0x76, 0x16],
    "控": [0x76, 0x17],
    "豆": [0x76, 0x18],
    "料": [0x76, 0x19],
    "沃": [0x76, 0x1A],
    "携": [0x76, 0x1B],
    "牺": [0x76, 0x1C],
    "幻": [0x76, 0x1D],
    "残": [0x76, 0x1E],
    "究": [0x76, 0x1F],
    "畏": [0x76, 0x20],
    "壶": [0x76, 0x21],
    "察": [0x76, 0x22],
    "囚": [0x76, 0x23],
    "剧": [0x76, 0x24],
    "揭": [0x76, 0x25],
    "操": [0x76, 0x26],
    "恼": [0x76, 0x27],
    "列": [0x76, 0x28],
    "绷": [0x76, 0x29],
    "您": [0x76, 0x2A],
    "页": [0x76, 0x2B],
    "腹": [0x76, 0x2C],
    "乖": [0x76, 0x2D],
    "耳": [0x76, 0x2E],
    "嗜": [0x76, 0x2F],
    "眨": [0x76, 0x30],
    "大": [0x77, 0x01],
    "舞": [0x77, 0x02],
    "妖": [0x77, 0x03],
    "机": [0x77, 0x04],
    "在": [0x77, 0x05],
    "做": [0x77, 0x06],
    "明": [0x77, 0x07],
    "似": [0x77, 0x08],
    "送": [0x77, 0x09],
    "过": [0x77, 0x0A],
    "外": [0x77, 0x0B],
    "假": [0x77, 0x0C],
    "奥": [0x77, 0x0D],
    "连": [0x77, 0x0E],
    "屋": [0x77, 0x0F],
    "物": [0x77, 0x10],
    "示": [0x77, 0x11],
    "试": [0x77, 0x12],
    "垂": [0x77, 0x13],
    "匆": [0x77, 0x14],
    "内": [0x77, 0x15],
    "燃": [0x77, 0x16],
    "排": [0x77, 0x17],
    "颤": [0x77, 0x18],
    "壁": [0x77, 0x19],
    "谱": [0x77, 0x1A],
    "款": [0x77, 0x1B],
    "牲": [0x77, 0x1C],
    "烫": [0x77, 0x1D],
    "案": [0x77, 0x1E],
    "廊": [0x77, 0x1F],
    "痒": [0x77, 0x20],
    "褪": [0x77, 0x21],
    "窟": [0x77, 0x22],
    "杂": [0x77, 0x23],
    "势": [0x77, 0x24],
    "免": [0x77, 0x25],
    "立": [0x77, 0x26],
    "罪": [0x77, 0x27],
    "蕾": [0x77, 0x28],
    "逼": [0x77, 0x29],
    "幕": [0x77, 0x2A],
    "估": [0x77, 0x2B],
    "罩": [0x77, 0x2C],
    "霍": [0x77, 0x2D],
    "泣": [0x77, 0x2E],
    "淌": [0x77, 0x2F],
    "谓": [0x77, 0x30],
    "炮": [0x78, 0x01],
    "步": [0x78, 0x02],
    "冰": [0x78, 0x03],
    "（": [0x78, 0x04],
    "又": [0x78, 0x05],
    "努": [0x78, 0x06],
    "自": [0x78, 0x07],
    "书": [0x78, 0x08],
    "吧": [0x78, 0x09],
    "次": [0x78, 0x0A],
    "下": [0x78, 0x0B],
    "令": [0x78, 0x0C],
    "够": [0x78, 0x0D],
    "门": [0x78, 0x0E],
    "顶": [0x78, 0x0F],
    "拜": [0x78, 0x10],
    "运": [0x78, 0x11],
    "暴": [0x78, 0x12],
    "钓": [0x78, 0x13],
    "捎": [0x78, 0x14],
    "弃": [0x78, 0x15],
    "烧": [0x78, 0x16],
    "丧": [0x78, 0x17],
    "抖": [0x78, 0x18],
    "垒": [0x78, 0x19],
    "仓": [0x78, 0x1A],
    "素": [0x78, 0x1B],
    "逐": [0x78, 0x1C],
    "抵": [0x78, 0x1D],
    "肿": [0x78, 0x1E],
    "厨": [0x78, 0x1F],
    "划": [0x78, 0x20],
    "椅": [0x78, 0x21],
    "勘": [0x78, 0x22],
    "臊": [0x78, 0x23],
    "导": [0x78, 0x24],
    "响": [0x78, 0x25],
    "臣": [0x78, 0x26],
    "饱": [0x78, 0x27],
    "寐": [0x78, 0x28],
    "杳": [0x78, 0x29],
    "馨": [0x78, 0x2A],
    "稿": [0x78, 0x2B],
    "飓": [0x78, 0x2C],
    "跨": [0x78, 0x2D],
    "狈": [0x78, 0x2E],
    "涌": [0x78, 0x2F],
    "蜗": [0x78, 0x30],
    "酷": [0x79, 0x01],
    "间": [0x79, 0x02],
    "冻": [0x79, 0x03],
    "开": [0x79, 0x04],
    "可": [0x79, 0x05],
    "那": [0x79, 0x06],
    "己": [0x79, 0x07],
    "疯": [0x79, 0x08],
    "计": [0x79, 0x09],
    "停": [0x79, 0x0A],
    "猜": [0x79, 0x0B],
    "忍": [0x79, 0x0C],
    "更": [0x79, 0x0D],
    "隆": [0x79, 0x0E],
    "房": [0x79, 0x0F],
    "访": [0x79, 0x10],
    "露": [0x79, 0x11],
    "同": [0x79, 0x12],
    "街": [0x79, 0x13],
    "范": [0x79, 0x14],
    "视": [0x79, 0x15],
    "：": [0x79, 0x16],
    "防": [0x79, 0x17],
    "污": [0x79, 0x18],
    "文": [0x79, 0x19],
    "抒": [0x79, 0x1A],
    "阴": [0x79, 0x1B],
    "渐": [0x79, 0x1C],
    "遗": [0x79, 0x1D],
    "奋": [0x79, 0x1E],
    "肴": [0x79, 0x1F],
    "倦": [0x79, 0x20],
    "盹": [0x79, 0x21],
    "渊": [0x79, 0x22],
    "南": [0x79, 0x23],
    "饵": [0x79, 0x24],
    "衔": [0x79, 0x25],
    "惩": [0x79, 0x26],
    "址": [0x79, 0x27],
    "栽": [0x79, 0x28],
    "嘟": [0x79, 0x29],
    "雀": [0x79, 0x2A],
    "延": [0x79, 0x2B],
    "陛": [0x79, 0x2C],
    "欺": [0x79, 0x2D],
    "吻": [0x79, 0x2E],
    "评": [0x79, 0x2F],
    "瞄": [0x79, 0x30],
    "霸": [0x7A, 0x01],
    "谍": [0x7A, 0x02],
    "白": [0x7A, 0x03],
    "始": [0x7A, 0x04],
    "以": [0x7A, 0x05],
    "困": [0x7A, 0x06],
    "位": [0x7A, 0x07],
    "狂": [0x7A, 0x08],
    "上": [0x7A, 0x09],
    "止": [0x7A, 0x0A],
    "并": [0x7A, 0x0B],
    "苛": [0x7A, 0x0C],
    "追": [0x7A, 0x0D],
    "卖": [0x7A, 0x0E],
    "啦": [0x7A, 0x0F],
    "山": [0x7A, 0x10],
    "塞": [0x7A, 0x11],
    "种": [0x7A, 0x12],
    "喇": [0x7A, 0x13],
    "选": [0x7A, 0x14],
    "糟": [0x7A, 0x15],
    "压": [0x7A, 0x16],
    "御": [0x7A, 0x17],
    "染": [0x7A, 0x18],
    "涸": [0x7A, 0x19],
    "律": [0x7A, 0x1A],
    "蝙": [0x7A, 0x1B],
    "劣": [0x7A, 0x1C],
    "骸": [0x7A, 0x1D],
    "周": [0x7A, 0x1E],
    "抽": [0x7A, 0x1F],
    "炭": [0x7A, 0x20],
    "昼": [0x7A, 0x21],
    "泳": [0x7A, 0x22],
    "北": [0x7A, 0x23],
    "沛": [0x7A, 0x24],
    "垄": [0x7A, 0x25],
    "罚": [0x7A, 0x26],
    "纵": [0x7A, 0x27],
    "盆": [0x7A, 0x28],
    "囔": [0x7A, 0x29],
    "羽": [0x7A, 0x2A],
    "挫": [0x7A, 0x2B],
    "杞": [0x7A, 0x2C],
    "凌": [0x7A, 0x2D],
    "项": [0x7A, 0x2E],
    "茁": [0x7A, 0x2F],
    "捷": [0x7A, 0x30],
    "黑": [0x7B, 0x01],
    "火": [0x7B, 0x02],
    "霹": [0x7B, 0x03],
    "）": [0x7B, 0x04],
    "实": [0x7B, 0x05],
    "永": [0x7B, 0x06],
    "年": [0x7B, 0x07],
    "冒": [0x7B, 0x08],
    "中": [0x7B, 0x09],
    "答": [0x7B, 0x0A],
    "与": [0x7B, 0x0B],
    "刻": [0x7B, 0x0C],
    "随": [0x7B, 0x0D],
    "勋": [0x7B, 0x0E],
    "咪": [0x7B, 0x0F],
    "岛": [0x7B, 0x10],
    "入": [0x7B, 0x11],
    "丛": [0x7B, 0x12],
    "叭": [0x7B, 0x13],
    "择": [0x7B, 0x14],
    "退": [0x7B, 0x15],
    "雪": [0x7B, 0x16],
    "捶": [0x7B, 0x17],
    "土": [0x7B, 0x18],
    "毁": [0x7B, 0x19],
    "誉": [0x7B, 0x1A],
    "蝠": [0x7B, 0x1B],
    "甩": [0x7B, 0x1C],
    "爬": [0x7B, 0x1D],
    "推": [0x7B, 0x1E],
    "屉": [0x7B, 0x1F],
    "陷": [0x7B, 0x20],
    "塌": [0x7B, 0x21],
    "层": [0x7B, 0x22],
    "痴": [0x7B, 0x23],
    "枪": [0x7B, 0x24],
    "靴": [0x7B, 0x25],
    "氓": [0x7B, 0x26],
    "劫": [0x7B, 0x27],
    "幢": [0x7B, 0x28],
    "拘": [0x7B, 0x29],
    "嘭": [0x7B, 0x2A],
    "额": [0x7B, 0x2B],
    "漏": [0x7B, 0x2C],
    "咯": [0x7B, 0x2D],
    "概": [0x7B, 0x2E],
    "厄": [0x7B, 0x2F],
    "径": [0x7B, 0x30],
    "黄": [0x7C, 0x01],
    "焰": [0x7C, 0x02],
    "雳": [0x7C, 0x03],
    "结": [0x7C, 0x04],
    "每": [0x7C, 0x05],
    "远": [0x7C, 0x06],
    "噢": [0x7C, 0x07],
    "险": [0x7C, 0x08],
    "让": [0x7C, 0x09],
    "应": [0x7C, 0x0A],
    "着": [0x7C, 0x0B],
    "乐": [0x7C, 0x0C],
    "折": [0x7C, 0x0D],
    "章": [0x7C, 0x0E],
    "ー": [0x7C, 0x0F],
    "附": [0x7C, 0x10],
    "救": [0x7C, 0x11],
    "爆": [0x7C, 0x12],
    "钟": [0x7C, 0x13],
    "攻": [0x7C, 0x14],
    "临": [0x7C, 0x15],
    "尘": [0x7C, 0x16],
    "板": [0x7C, 0x17],
    "壤": [0x7C, 0x18],
    "灭": [0x7C, 0x19],
    "银": [0x7C, 0x1A],
    "栖": [0x7C, 0x1B],
    "戚": [0x7C, 0x1C],
    "烬": [0x7C, 0x1D],
    "垫": [0x7C, 0x1E],
    "螺": [0x7C, 0x1F],
    "展": [0x7C, 0x20],
    "倾": [0x7C, 0x21],
    "遮": [0x7C, 0x22],
    "醉": [0x7C, 0x23],
    "庞": [0x7C, 0x24],
    "偏": [0x7C, 0x25],
    "囊": [0x7C, 0x26],
    "弓": [0x7C, 0x27],
    "匿": [0x7C, 0x28],
    "航": [0x7C, 0x29],
    "扯": [0x7C, 0x2A],
    "扭": [0x7C, 0x2B],
    "炉": [0x7C, 0x2C],
    "鞋": [0x7C, 0x2D],
    "峡": [0x7C, 0x2E],
    "厚": [0x7C, 0x2F],
    "括": [0x7C, 0x30],
    "绿": [0x7D, 0x01],
    "气": [0x7D, 0x02],
    "女": [0x7D, 0x03],
    "束": [0x7D, 0x04],
    "愿": [0x7D, 0x05],
    "记": [0x7D, 0x06],
    "继": [0x7D, 0x07],
    "经": [0x7D, 0x08],
    "妹": [0x7D, 0x09],
    "查": [0x7D, 0x0A],
    "抱": [0x7D, 0x0B],
    "游": [0x7D, 0x0C],
    "磨": [0x7D, 0x0D],
    "赚": [0x7D, 0x0E],
    "否": [0x7D, 0x0F],
    "近": [0x7D, 0x10],
    "魂": [0x7D, 0x11],
    "裂": [0x7D, 0x12],
    "形": [0x7D, 0x13],
    "敌": [0x7D, 0x14],
    "柜": [0x7D, 0x15],
    "旧": [0x7D, 0x16],
    "轻": [0x7D, 0x17],
    "凉": [0x7D, 0x18],
    "帽": [0x7D, 0x19],
    "磁": [0x7D, 0x1A],
    "鸟": [0x7D, 0x1B],
    "留": [0x7D, 0x1C],
    "锐": [0x7D, 0x1D],
    "检": [0x7D, 0x1E],
    "梯": [0x7D, 0x1F],
    "雨": [0x7D, 0x20],
    "斜": [0x7D, 0x21],
    "怜": [0x7D, 0x22],
    "协": [0x7D, 0x23],
    "印": [0x7D, 0x24],
    "僻": [0x7D, 0x25],
    "译": [0x7D, 0x26],
    "恭": [0x7D, 0x27],
    "托": [0x7D, 0x28],
    "恒": [0x7D, 0x29],
    "苗": [0x7D, 0x2A],
    "窃": [0x7D, 0x2B],
    "悄": [0x7D, 0x2C],
    "尾": [0x7D, 0x2D],
    "拐": [0x7D, 0x2E],
    "隙": [0x7D, 0x2F],
    "崎": [0x7D, 0x30],
    "兄": [0x7E, 0x01],
    "医": [0x7E, 0x02],
    "孩": [0x7E, 0x03],
    "哇": [0x7E, 0x04],
    "望": [0x7E, 0x05],
    "希": [0x7E, 0x06],
    "续": [0x7E, 0x07],
    "历": [0x7E, 0x08],
    "天": [0x7E, 0x09],
    "尔": [0x7E, 0x0A],
    "歉": [0x7E, 0x0B],
    "找": [0x7E, 0x0C],
    "却": [0x7E, 0x0D],
    "别": [0x7E, 0x0E],
    "唱": [0x7E, 0x0F],
    "故": [0x7E, 0x10],
    "指": [0x7E, 0x11],
    "植": [0x7E, 0x12],
    "粉": [0x7E, 0x13],
    "旦": [0x7E, 0x14],
    "台": [0x7E, 0x15],
    "投": [0x7E, 0x16],
    "缓": [0x7E, 0x17],
    "爽": [0x7E, 0x18],
    "臭": [0x7E, 0x19],
    "际": [0x7E, 0x1A],
    "锋": [0x7E, 0x1B],
    "肮": [0x7E, 0x1C],
    "丑": [0x7E, 0x1D],
    "标": [0x7E, 0x1E],
    "绕": [0x7E, 0x1F],
    "荫": [0x7E, 0x20],
    "坡": [0x7E, 0x21],
    "篱": [0x7E, 0x22],
    "辛": [0x7E, 0x23],
    "夏": [0x7E, 0x24],
    "帅": [0x7E, 0x25],
    "询": [0x7E, 0x26],
    "阶": [0x7E, 0x27],
    "娇": [0x7E, 0x28],
    "司": [0x7E, 0x29],
    "约": [0x7E, 0x2A],
    "财": [0x7E, 0x2B],
    "挠": [0x7E, 0x2C],
    "嗖": [0x7E, 0x2D],
    "铭": [0x7E, 0x2E],
    "趾": [0x7E, 0x2F],
    "岖": [0x7E, 0x30],
    "弟": [0x7F, 0x01],
    "疗": [0x7F, 0x02],
    "卡": [0x7F, 0x03],
    "咔": [0x7F, 0x04],
    "都": [0x7F, 0x05],
    "件": [0x7F, 0x06],
    "想": [0x7F, 0x07],
    "城": [0x7F, 0x08],
    "呀": [0x7F, 0x09],
    "博": [0x7F, 0x0A],
    "密": [0x7F, 0x0B],
    "今": [0x7F, 0x0C],
    "松": [0x7F, 0x0D],
    "迫": [0x7F, 0x0E],
    "歌": [0x7F, 0x0F],
    "浪": [0x7F, 0x10],
    "跑": [0x7F, 0x11],
    "几": [0x7F, 0x12],
    "懒": [0x7F, 0x13],
    "踩": [0x7F, 0x14],
    "固": [0x7F, 0x15],
    "掷": [0x7F, 0x16],
    "节": [0x7F, 0x17],
    "阳": [0x7F, 0x18],
    "篮": [0x7F, 0x19],
    "敲": [0x7F, 0x1A],
    "爪": [0x7F, 0x1B],
    "脏": [0x7F, 0x1C],
    "陋": [0x7F, 0x1D],
    "缝": [0x7F, 0x1E],
    "圈": [0x7F, 0x1F],
    "垠": [0x7F, 0x20],
    "崩": [0x7F, 0x21],
    "善": [0x7F, 0x22],
    "侍": [0x7F, 0x23],
    "籁": [0x7F, 0x24],
    "丰": [0x7F, 0x25],
    "育": [0x7F, 0x26],
    "亏": [0x7F, 0x27],
    "谚": [0x7F, 0x28],
    "渡": [0x7F, 0x29],
    "叛": [0x7F, 0x2A],
    "拯": [0x7F, 0x2B],
    "闯": [0x7F, 0x2C],
    "诅": [0x7F, 0x2D],
    "矫": [0x7F, 0x2E],
    "扬": [0x7F, 0x2F],
    "脊": [0x7F, 0x30],
    "蛋": [0x80, 0x01],
    "灯": [0x80, 0x02],
    "米": [0x80, 0x03],
    "战": [0x80, 0x04],
    "要": [0x80, 0x05],
    "事": [0x80, 0x06],
    "候": [0x80, 0x07],
    "堡": [0x80, 0x08],
    "邀": [0x80, 0x09],
    "士": [0x80, 0x0A],
    "爱": [0x80, 0x0B],
    "练": [0x80, 0x0C],
    "口": [0x80, 0x0D],
    "典": [0x80, 0x0E],
    "怯": [0x80, 0x0F],
    "漫": [0x80, 0x10],
    "腿": [0x80, 0x11],
    "描": [0x80, 0x12],
    "散": [0x80, 0x13],
    "准": [0x80, 0x14],
    "孪": [0x80, 0x15],
    "造": [0x80, 0x16],
    "省": [0x80, 0x17],
    "灿": [0x80, 0x18],
    "静": [0x80, 0x19],
    "万": [0x80, 0x1A],
    "孵": [0x80, 0x1B],
    "巨": [0x80, 0x1C],
    "拼": [0x80, 0x1D],
    "箱": [0x80, 0x1E],
    "组": [0x80, 0x1F],
    "扎": [0x80, 0x20],
    "尺": [0x80, 0x21],
    "橘": [0x80, 0x22],
    "挑": [0x80, 0x23],
    "萦": [0x80, 0x24],
    "肌": [0x80, 0x25],
    "悍": [0x80, 0x26],
    "号": [0x80, 0x27],
    "润": [0x80, 0x28],
    "拂": [0x80, 0x29],
    "谅": [0x80, 0x2A],
    "锅": [0x80, 0x2B],
    "呸": [0x80, 0x2C],
    "爵": [0x80, 0x2D],
    "寂": [0x80, 0x2E],
    "泽": [0x80, 0x2F],
    "矗": [0x80, 0x30],
    "仔": [0x81, 0x01],
    "笼": [0x81, 0x02],
    "欧": [0x81, 0x03],
    "车": [0x81, 0x04],
    "当": [0x81, 0x05],
    "得": [0x81, 0x06],
    "有": [0x81, 0x07],
    "恢": [0x81, 0x08],
    "请": [0x81, 0x09],
    "待": [0x81, 0x0A],
    "爷": [0x81, 0x0B],
    "嚯": [0x81, 0x0C],
    "毕": [0x81, 0x0D],
    "段": [0x81, 0x0E],
    "场": [0x81, 0x0F],
    "壮": [0x81, 0x10],
    "监": [0x81, 0x11],
    "述": [0x81, 0x12],
    "专": [0x81, 0x13],
    "按": [0x81, 0x14],
    "巷": [0x81, 0x15],
    "圆": [0x81, 0x16],
    "频": [0x81, 0x17],
    "烂": [0x81, 0x18],
    "恐": [0x81, 0x19],
    "麻": [0x81, 0x1A],
    "召": [0x81, 0x1B],
    "朵": [0x81, 0x1C],
    "猛": [0x81, 0x1D],
    "盒": [0x81, 0x1E],
    "筑": [0x81, 0x1F],
    "儿": [0x81, 0x20],
    "寸": [0x81, 0x21],
    "竞": [0x81, 0x22],
    "贸": [0x81, 0x23],
    "招": [0x81, 0x24],
    "肉": [0x81, 0x25],
    "哧": [0x81, 0x26],
    "虔": [0x81, 0x27],
    "沐": [0x81, 0x28],
    "涯": [0x81, 0x29],
    "擦": [0x81, 0x2A],
    "冬": [0x81, 0x2B],
    "扼": [0x81, 0x2C],
    "灾": [0x81, 0x2D],
    "寞": [0x81, 0x2E],
    "垮": [0x81, 0x2F],
    "纠": [0x81, 0x30],
    "鼹": [0x82, 0x01],
    "鬼": [0x82, 0x02],
    "苏": [0x82, 0x03],
    "泡": [0x82, 0x04],
    "遥": [0x82, 0x05],
    "到": [0x82, 0x06],
    "等": [0x82, 0x07],
    "复": [0x82, 0x08],
    "举": [0x82, 0x09],
    "废": [0x82, 0x0A],
    "欣": [0x82, 0x0B],
    "商": [0x82, 0x0C],
    "竟": [0x82, 0x0D],
    "趣": [0x82, 0x0E],
    "担": [0x82, 0x0F],
    "观": [0x82, 0x10],
    "狱": [0x82, 0x11],
    "戴": [0x82, 0x12],
    "横": [0x82, 0x13],
    "味": [0x82, 0x14],
    "惜": [0x82, 0x15],
    "暂": [0x82, 0x16],
    "偶": [0x82, 0x17],
    "算": [0x82, 0x18],
    "胶": [0x82, 0x19],
    "痹": [0x82, 0x1A],
    "唤": [0x82, 0x1B],
    "棵": [0x82, 0x1C],
    "怀": [0x82, 0x1D],
    "摆": [0x82, 0x1E],
    "滴": [0x82, 0x1F],
    "嗬": [0x82, 0x20],
    "蜘": [0x82, 0x21],
    "革": [0x82, 0x22],
    "捏": [0x82, 0x23],
    "裁": [0x82, 0x24],
    "梭": [0x82, 0x25],
    "敞": [0x82, 0x26],
    "哼": [0x82, 0x27],
    "互": [0x82, 0x28],
    "钩": [0x82, 0x29],
    "铺": [0x82, 0x2A],
    "季": [0x82, 0x2B],
    "嘎": [0x82, 0x2C],
    "权": [0x82, 0x2D],
    "茬": [0x82, 0x2E],
    "俊": [0x82, 0x2F],
    "僧": [0x82, 0x30],
    "鼠": [0x83, 0x01],
    "将": [0x83, 0x02],
    "莎": [0x83, 0x03],
    "队": [0x83, 0x04],
    "夜": [0x83, 0x05],
    "也": [0x83, 0x06],
    "去": [0x83, 0x07],
    "原": [0x83, 0x08],
    "办": [0x83, 0x09],
    "话": [0x83, 0x0A],
    "喜": [0x83, 0x0B],
    "店": [0x83, 0x0C],
    "变": [0x83, 0x0D],
    "哟": [0x83, 0x0E],
    "皱": [0x83, 0x0F],
    "❝": [0x83, 0x10],
    "身": [0x83, 0x11],
    "胸": [0x83, 0x12],
    "乏": [0x83, 0x13],
    "瞬": [0x83, 0x14],
    "遵": [0x83, 0x15],
    "降": [0x83, 0x16],
    "半": [0x83, 0x17],
    "器": [0x83, 0x18],
    "皇": [0x83, 0x19],
    "翅": [0x83, 0x1A],
    "忠": [0x83, 0x1B],
    "仍": [0x83, 0x1C],
    "绚": [0x83, 0x1D],
    "脆": [0x83, 0x1E],
    "湿": [0x83, 0x1F],
    "丘": [0x83, 0x20],
    "蛛": [0x83, 0x21],
    "紫": [0x83, 0x22],
    "艰": [0x83, 0x23],
    "判": [0x83, 0x24],
    "属": [0x83, 0x25],
    "懦": [0x83, 0x26],
    "撒": [0x83, 0x27],
    "稳": [0x83, 0x28],
    "辈": [0x83, 0x29],
    "栏": [0x83, 0x2A],
    "册": [0x83, 0x2B],
    "吱": [0x83, 0x2C],
    "祈": [0x83, 0x2D],
    "咙": [0x83, 0x2E],
    "孔": [0x83, 0x2F],
    "漆": [0x83, 0x30],
    "岩": [0x84, 0x01],
    "军": [0x84, 0x02],
    "阿": [0x84, 0x03],
    "洞": [0x84, 0x04],
    "空": [0x84, 0x05],
    "许": [0x84, 0x06],
    "该": [0x84, 0x07],
    "貌": [0x84, 0x08],
    "被": [0x84, 0x09],
    "胡": [0x84, 0x0A],
    "若": [0x84, 0x0B],
    "营": [0x84, 0x0C],
    "孤": [0x84, 0x0D],
    "特": [0x84, 0x0E],
    "眉": [0x84, 0x0F],
    "纸": [0x84, 0x10],
    "站": [0x84, 0x11],
    "针": [0x84, 0x12],
    "二": [0x84, 0x13],
    "减": [0x84, 0x14],
    "价": [0x84, 0x15],
    "燥": [0x84, 0x16],
    "抢": [0x84, 0x17],
    "翡": [0x84, 0x18],
    "腌": [0x84, 0x19],
    "膀": [0x84, 0x1A],
    "根": [0x84, 0x1B],
    "乘": [0x84, 0x1C],
    "盯": [0x84, 0x1D],
    "铙": [0x84, 0x1E],
    "河": [0x84, 0x1F],
    "晒": [0x84, 0x20],
    "网": [0x84, 0x21],
    "序": [0x84, 0x22],
    "掠": [0x84, 0x23],
    "赖": [0x84, 0x24],
    "骗": [0x84, 0x25],
    "呵": [0x84, 0x26],
    "兆": [0x84, 0x27],
    "柱": [0x84, 0x28],
    "讽": [0x84, 0x29],
    "迄": [0x84, 0x2A],
    "驱": [0x84, 0x2B],
    "校": [0x84, 0x2C],
    "祷": [0x84, 0x2D],
    "亚": [0x84, 0x2E],
    "溉": [0x84, 0x2F],
    "启": [0x84, 0x30],
    "石": [0x85, 0x01],
    "团": [0x85, 0x02],
    "姨": [0x85, 0x03],
    "哦": [0x85, 0x04],
    "时": [0x85, 0x05],
    "规": [0x85, 0x06],
    "再": [0x85, 0x07],
    "安": [0x85, 0x08],
    "参": [0x85, 0x09],
    "八": [0x85, 0x0A],
    "哪": [0x85, 0x0B],
    "业": [0x85, 0x0C],
    "独": [0x85, 0x0D],
    "鲜": [0x85, 0x0E],
    "识": [0x85, 0x0F],
    "片": [0x85, 0x10],
    "目": [0x85, 0x11],
    "戏": [0x85, 0x12],
    "百": [0x85, 0x13],
    "巧": [0x85, 0x14],
    "著": [0x85, 0x15],
    "滋": [0x85, 0x16],
    "昏": [0x85, 0x17],
    "翠": [0x85, 0x18],
    "豪": [0x85, 0x19],
    "狼": [0x85, 0x1A],
    "怨": [0x85, 0x1B],
    "尖": [0x85, 0x1C],
    "副": [0x85, 0x1D],
    "钹": [0x85, 0x1E],
    "环": [0x85, 0x1F],
    "郁": [0x85, 0x20],
    "巡": [0x85, 0x21],
    "淹": [0x85, 0x22],
    "奉": [0x85, 0x23],
    "货": [0x85, 0x24],
    "骑": [0x85, 0x25],
    "愤": [0x85, 0x26],
    "嚓": [0x85, 0x27],
    "轮": [0x85, 0x28],
    "嗓": [0x85, 0x29],
    "腰": [0x85, 0x2A],
    "档": [0x85, 0x2B],
    "碗": [0x85, 0x2C],
    "氧": [0x85, 0x2D],
    "卟": [0x85, 0x2E],
    "谋": [0x85, 0x2F],
    "捕": [0x85, 0x30],
    "球": [0x86, 0x01],
    "叠": [0x86, 0x02],
    "比": [0x86, 0x03],
    "，": [0x86, 0x04],
    "知": [0x86, 0x05],
    "矩": [0x86, 0x06],
    "见": [0x86, 0x07],
    "最": [0x86, 0x08],
    "于": [0x86, 0x09],
    "犯": [0x86, 0x0A],
    "呆": [0x86, 0x0B],
    "总": [0x86, 0x0C],
    "爸": [0x86, 0x0D],
    "殊": [0x86, 0x0E],
    "潮": [0x86, 0x0F],
    "❞": [0x86, 0x10],
    "录": [0x86, 0x11],
    "室": [0x86, 0x12],
    "合": [0x86, 0x13],
    "依": [0x86, 0x14],
    "收": [0x86, 0x15],
    "补": [0x86, 0x16],
    "冲": [0x86, 0x17],
    "雕": [0x86, 0x18],
    "华": [0x86, 0x19],
    "籍": [0x86, 0x1A],
    "恨": [0x86, 0x1B],
    "卷": [0x86, 0x1C],
    "景": [0x86, 0x1D],
    "肩": [0x86, 0x1E],
    "境": [0x86, 0x1F],
    "葱": [0x86, 0x20],
    "逻": [0x86, 0x21],
    "啰": [0x86, 0x22],
    "句": [0x86, 0x23],
    "宾": [0x86, 0x24],
    "童": [0x86, 0x25],
    "怒": [0x86, 0x26],
    "卿": [0x86, 0x27],
    "嘶": [0x86, 0x28],
    "蒙": [0x86, 0x29],
    "扛": [0x86, 0x2A],
    "官": [0x86, 0x2B],
    "骤": [0x86, 0x2C],
    "嘲": [0x86, 0x2D],
    "炙": [0x86, 0x2E],
    "纱": [0x86, 0x2F],
    "峻": [0x86, 0x30],
    "盗": [0x87, 0x01],
    "罗": [0x87, 0x02],
    "帕": [0x87, 0x03],
    "！": [0x87, 0x04],
    "道": [0x87, 0x05],
    "点": [0x87, 0x06],
    "亮": [0x87, 0x07],
    "后": [0x87, 0x08],
    "先": [0x87, 0x09],
    "傻": [0x87, 0x0A],
    "住": [0x87, 0x0B],
    "买": [0x87, 0x0C],
    "讨": [0x87, 0x0D],
    "期": [0x87, 0x0E],
    "、": [0x87, 0x0F],
    "认": [0x87, 0x10],
    "音": [0x87, 0x11],
    "管": [0x87, 0x12],
    "玫": [0x87, 0x13],
    "靠": [0x87, 0x14],
    "换": [0x87, 0x15],
    "治": [0x87, 0x16],
    "距": [0x87, 0x17],
    "桶": [0x87, 0x18],
    "浓": [0x87, 0x19],
    "阻": [0x87, 0x1A],
    "咄": [0x87, 0x1B],
    "饿": [0x87, 0x1C],
    "佩": [0x87, 0x1D],
    "瓣": [0x87, 0x1E],
    "呕": [0x87, 0x1F],
    "澈": [0x87, 0x20],
    "饭": [0x87, 0x21],
    "砍": [0x87, 0x22],
    "购": [0x87, 0x23],
    "浴": [0x87, 0x24],
    "悔": [0x87, 0x25],
    "杰": [0x87, 0x26],
    "惨": [0x87, 0x27],
    "鸣": [0x87, 0x28],
    "词": [0x87, 0x29],
    "崖": [0x87, 0x2A],
    "遣": [0x87, 0x2B],
    "届": [0x87, 0x2C],
    "晰": [0x87, 0x2D],
    "泄": [0x87, 0x2E],
    "浸": [0x87, 0x2F],
    "媲": [0x87, 0x30],
    "贼": [0x88, 0x01],
    "汉": [0x88, 0x02],
    "瑞": [0x88, 0x03],
    "谢": [0x88, 0x04],
    "闪": [0x88, 0x05],
    "但": [0x88, 0x06],
    "照": [0x88, 0x07],
    "胜": [0x88, 0x08],
    "测": [0x88, 0x09],
    "拒": [0x88, 0x0A],
    "紧": [0x88, 0x0B],
    "东": [0x88, 0x0C],
    "厌": [0x88, 0x0D],
    "局": [0x88, 0x0E],
    "即": [0x88, 0x0F],
    "锁": [0x88, 0x10],
    "崇": [0x88, 0x11],
    "俱": [0x88, 0x12],
    "瑰": [0x88, 0x13],
    "摇": [0x88, 0x14],
    "产": [0x88, 0x15],
    "愈": [0x88, 0x16],
    "离": [0x88, 0x17],
    "巾": [0x88, 0x18],
    "饼": [0x88, 0x19],
    "虚": [0x88, 0x1A],
    "牙": [0x88, 0x1B],
    "蛰": [0x88, 0x1C],
    "献": [0x88, 0x1D],
    "僵": [0x88, 0x1E],
    "舒": [0x88, 0x1F],
    "岁": [0x88, 0x20],
    "帘": [0x88, 0x21],
    "惹": [0x88, 0x22],
    "寄": [0x88, 0x23],
    "报": [0x88, 0x24],
    "茸": [0x88, 0x25],
    "宁": [0x88, 0x26],
    "昨": [0x88, 0x27],
    "烟": [0x88, 0x28],
    "灌": [0x88, 0x29],
    "葡": [0x88, 0x2A],
    "踢": [0x88, 0x2B],
    "翰": [0x88, 0x2C],
    "科": [0x88, 0x2D],
    "饥": [0x88, 0x2E],
    "洪": [0x88, 0x2F],
    "拦": [0x88, 0x30],
    "木": [0x89, 0x01],
    "高": [0x89, 0x02],
    "塔": [0x89, 0x03],
    "你": [0x89, 0x04],
    "耀": [0x89, 0x05],
    "发": [0x89, 0x06],
    "还": [0x89, 0x07],
    "利": [0x89, 0x08],
    "宫": [0x89, 0x09],
    "失": [0x89, 0x0A],
    "张": [0x89, 0x0B],
    "西": [0x89, 0x0C],
    "保": [0x89, 0x0D],
    "顾": [0x89, 0x0E],
    "跳": [0x89, 0x0F],
    "牢": [0x89, 0x10],
    "擅": [0x89, 0x11],
    "部": [0x89, 0x12],
    "莲": [0x89, 0x13],
    "杆": [0x89, 0x14],
    "反": [0x89, 0x15],
    "萎": [0x89, 0x16],
    "则": [0x89, 0x17],
    "伞": [0x89, 0x18],
    "荷": [0x89, 0x19],
    "思": [0x89, 0x1A],
    "齿": [0x89, 0x1B],
    "迈": [0x89, 0x1C],
    "端": [0x89, 0x1D],
    "咀": [0x89, 0x1E],
    "振": [0x89, 0x1F],
    "暑": [0x89, 0x20],
    "桌": [0x89, 0x21],
    "覆": [0x89, 0x22],
    "癫": [0x89, 0x23],
    "吝": [0x89, 0x24],
    "浑": [0x89, 0x25],
    "陪": [0x89, 0x26],
    "棕": [0x89, 0x27],
    "兽": [0x89, 0x28],
    "铅": [0x89, 0x29],
    "萄": [0x89, 0x2A],
    "涨": [0x89, 0x2B],
    "九": [0x89, 0x2C],
    "呻": [0x89, 0x2D],
    "咦": [0x89, 0x2E],
    "恕": [0x89, 0x2F],
    "浇": [0x89, 0x30],
    "乃": [0x8A, 0x01],
    "跷": [0x8A, 0x02],
    "妞": [0x8A, 0x03],
    "成": [0x8A, 0x04],
    "正": [0x8A, 0x05],
    "生": [0x8A, 0x06],
    "桃": [0x8A, 0x07],
    "象": [0x8A, 0x08],
    "殿": [0x8A, 0x09],
    "礼": [0x8A, 0x0A],
    "刚": [0x8A, 0x0B],
    "宴": [0x8A, 0x0C],
    "护": [0x8A, 0x0D],
    "糊": [0x8A, 0x0E],
    "饮": [0x8A, 0x0F],
    "噩": [0x8A, 0x10],
    "躲": [0x8A, 0x11],
    "系": [0x8A, 0x12],
    "菟": [0x8A, 0x13],
    "移": [0x8A, 0x14],
    "增": [0x8A, 0x15],
    "缩": [0x8A, 0x16],
    "伐": [0x8A, 0x17],
    "诗": [0x8A, 0x18],
    "拉": [0x8A, 0x19],
    "付": [0x8A, 0x1A],
    "咬": [0x8A, 0x1B],
    "创": [0x8A, 0x1C],
    "凶": [0x8A, 0x1D],
    "嚼": [0x8A, 0x1E],
    "熬": [0x8A, 0x1F],
    "德": [0x8A, 0x20],
    "井": [0x8A, 0x21],
    "湖": [0x8A, 0x22],
    "宣": [0x8A, 0x23],
    "啬": [0x8A, 0x24],
    "呱": [0x8A, 0x25],
    "闷": [0x8A, 0x26],
    "榈": [0x8A, 0x27],
    "叔": [0x8A, 0x28],
    "旷": [0x8A, 0x29],
    "妆": [0x8A, 0x2A],
    "烁": [0x8A, 0x2B],
    "嘉": [0x8A, 0x2C],
    "吟": [0x8A, 0x2D],
    "腥": [0x8A, 0x2E],
    "逆": [0x8A, 0x2F],
    "缭": [0x8A, 0x30],
    "伊": [0x8B, 0x01],
    "噬": [0x8B, 0x02],
    "咻": [0x8B, 0x03],
    "功": [0x8B, 0x04],
    "为": [0x8B, 0x05],
    "能": [0x8B, 0x06],
    "公": [0x8B, 0x07],
    "已": [0x8B, 0x08],
    "霜": [0x8B, 0x09],
    "千": [0x8B, 0x0A],
    "镇": [0x8B, 0x0B],
    "欢": [0x8B, 0x0C],
    "叫": [0x8B, 0x0D],
    "涂": [0x8B, 0x0E],
    "命": [0x8B, 0x0F],
    "决": [0x8B, 0x10],
    "藏": [0x8B, 0x11],
    "诞": [0x8B, 0x12],
    "弗": [0x8B, 0x13],
    "况": [0x8B, 0x14],
    "基": [0x8B, 0x15],
    "浆": [0x8B, 0x16],
    "碰": [0x8B, 0x17],
    "盐": [0x8B, 0x18],
    "仁": [0x8B, 0x19],
    "班": [0x8B, 0x1A],
    "朝": [0x8B, 0x1B],
    "铁": [0x8B, 0x1C],
    "呃": [0x8B, 0x1D],
    "缠": [0x8B, 0x1E],
    "垃": [0x8B, 0x1F],
    "阱": [0x8B, 0x20],
    "摸": [0x8B, 0x21],
    "溜": [0x8B, 0x22],
    "恋": [0x8B, 0x23],
    "例": [0x8B, 0x24],
    "馆": [0x8B, 0x25],
    "庭": [0x8B, 0x26],
    "袋": [0x8B, 0x27],
    "皙": [0x8B, 0x28],
    "怡": [0x8B, 0x29],
    "匣": [0x8B, 0x2A],
    "施": [0x8B, 0x2B],
    "岗": [0x8B, 0x2C],
    "初": [0x8B, 0x2D],
    "祟": [0x8B, 0x2E],
    "贯": [0x8B, 0x2F],
    "徽": [0x8B, 0x30],
    "钢": [0x8C, 0x01],
    "草": [0x8C, 0x02],
    "像": [0x8C, 0x03],
    "了": [0x8C, 0x04],
    "勇": [0x8C, 0x05],
    "性": [0x8C, 0x06],
    "主": [0x8C, 0x07],
    "往": [0x8C, 0x08],
    "度": [0x8C, 0x09],
    "载": [0x8C, 0x0A],
    "稍": [0x8C, 0x0B],
    "处": [0x8C, 0x0C],
    "滚": [0x8C, 0x0D],
    "午": [0x8C, 0x0E],
    "条": [0x8C, 0x0F],
    "坐": [0x8C, 0x10],
    "墟": [0x8C, 0x11],
    "共": [0x8C, 0x12],
    "蹈": [0x8C, 0x13],
    "习": [0x8C, 0x14],
    "本": [0x8C, 0x15],
    "枫": [0x8C, 0x16],
    "触": [0x8C, 0x17],
    "鸡": [0x8C, 0x18],
    "煮": [0x8C, 0x19],
    "翻": [0x8C, 0x1A],
    "洋": [0x8C, 0x1B],
    "吞": [0x8C, 0x1C],
    "线": [0x8C, 0x1D],
    "懈": [0x8C, 0x1E],
    "圾": [0x8C, 0x1F],
    "谜": [0x8C, 0x20],
    "净": [0x8C, 0x21],
    "融": [0x8C, 0x22],
    "耻": [0x8C, 0x23],
    "义": [0x8C, 0x24],
    "柔": [0x8C, 0x25],
    "济": [0x8C, 0x26],
    "阁": [0x8C, 0x27],
    "斑": [0x8C, 0x28],
    "哀": [0x8C, 0x29],
    "障": [0x8C, 0x2A],
    "苍": [0x8C, 0x2B],
    "谨": [0x8C, 0x2C],
    "趟": [0x8C, 0x2D],
    "汗": [0x8C, 0x2E],
    "薄": [0x8C, 0x2F],
    "删": [0x8C, 0x30],
    "盔": [0x8D, 0x01],
    "伪": [0x8D, 0x02],
    "嗦": [0x8D, 0x03],
    "回": [0x8D, 0x04],
    "敢": [0x8D, 0x05],
    "微": [0x8D, 0x06],
    "需": [0x8D, 0x07],
    "日": [0x8D, 0x08],
    "负": [0x8D, 0x09],
    "逢": [0x8D, 0x0A],
    "休": [0x8D, 0x0B],
    "谈": [0x8D, 0x0C],
    "扰": [0x8D, 0x0D],
    "代": [0x8D, 0x0E],
    "鲸": [0x8D, 0x0F],
    "吃": [0x8D, 0x10],
    "棺": [0x8D, 0x11],
    "跃": [0x8D, 0x12],
    "曲": [0x8D, 0x13],
    "掌": [0x8D, 0x14],
    "混": [0x8D, 0x15],
    "养": [0x8D, 0x16],
    "倍": [0x8D, 0x17],
    "乳": [0x8D, 0x18],
    "蓄": [0x8D, 0x19],
    "容": [0x8D, 0x1A],
    "叮": [0x8D, 0x1B],
    "飘": [0x8D, 0x1C],
    "费": [0x8D, 0x1D],
    "域": [0x8D, 0x1E],
    "脖": [0x8D, 0x1F],
    "喻": [0x8D, 0x20],
    "维": [0x8D, 0x21],
    "堆": [0x8D, 0x22],
    "羞": [0x8D, 0x23],
    "嘈": [0x8D, 0x24],
    "茎": [0x8D, 0x25],
    "逛": [0x8D, 0x26],
    "迟": [0x8D, 0x27],
    "允": [0x8D, 0x28],
    "默": [0x8D, 0x29],
    "秋": [0x8D, 0x2A],
    "傲": [0x8D, 0x2B],
    "慎": [0x8D, 0x2C],
    "噗": [0x8D, 0x2D],
    "辆": [0x8D, 0x2E],
    "揍": [0x8D, 0x2F],
    "・": [0x8D, 0x30],
    "甲": [0x8E, 0x01],
    "装": [0x8E, 0x02],
    "水": [0x8E, 0x03],
    "我": [0x8E, 0x04],
    "而": [0x8E, 0x05],
    "乎": [0x8E, 0x06],
    "帮": [0x8E, 0x07],
    "活": [0x8E, 0x08],
    "责": [0x8E, 0x09],
    "赶": [0x8E, 0x0A],
    "觉": [0x8E, 0x0B],
    "确": [0x8E, 0x0C],
    "耽": [0x8E, 0x0D],
    "野": [0x8E, 0x0E],
    "鱼": [0x8E, 0x0F],
    "刨": [0x8E, 0x10],
    "材": [0x8E, 0x11],
    "扫": [0x8E, 0x12],
    "演": [0x8E, 0x13],
    "握": [0x8E, 0x14],
    "弄": [0x8E, 0x15],
    "芳": [0x8E, 0x16],
    "逃": [0x8E, 0x17],
    "酪": [0x8E, 0x18],
    "群": [0x8E, 0x19],
    "迅": [0x8E, 0x1A],
    "蛮": [0x8E, 0x1B],
    "考": [0x8E, 0x1C],
    "笨": [0x8E, 0x1D],
    "征": [0x8E, 0x1E],
    "塘": [0x8E, 0x1F],
    "户": [0x8E, 0x20],
    "挖": [0x8E, 0x21],
    "滑": [0x8E, 0x22],
    "引": [0x8E, 0x23],
    "志": [0x8E, 0x24],
    "猎": [0x8E, 0x25],
    "挡": [0x8E, 0x26],
    "帐": [0x8E, 0x27],
    "隔": [0x8E, 0x28],
    "穴": [0x8E, 0x29],
    "喉": [0x8E, 0x2A],
    "销": [0x8E, 0x2B],
    "鼓": [0x8E, 0x2C],
    "嗤": [0x8E, 0x2D],
    "搭": [0x8E, 0x2E],
    "尸": [0x8E, 0x2F],
    "登": [0x8E, 0x30],
    "虫": [0x8F, 0x01],
    "热": [0x8F, 0x02],
    "晶": [0x8F, 0x03],
    "们": [0x8F, 0x04],
    "赞": [0x8F, 0x05],
    "其": [0x8F, 0x06],
    "助": [0x8F, 0x07],
    "赌": [0x8F, 0x08],
    "领": [0x8F, 0x09],
    "快": [0x8F, 0x0A],
    "精": [0x8F, 0x0B],
    "状": [0x8F, 0x0C],
    "误": [0x8F, 0x0D],
    "睡": [0x8F, 0x0E],
    "取": [0x8F, 0x0F],
    "偷": [0x8F, 0x10],
    "流": [0x8F, 0x11],
    "潜": [0x8F, 0x12],
    "邻": [0x8F, 0x13],
    "愧": [0x8F, 0x14],
    "勒": [0x8F, 0x15],
    "香": [0x8F, 0x16],
    "获": [0x8F, 0x17],
    "油": [0x8F, 0x18],
    "体": [0x8F, 0x19],
    "绪": [0x8F, 0x1A],
    "吵": [0x8F, 0x1B],
    "楚": [0x8F, 0x1C],
    "贪": [0x8F, 0x1D],
    "乡": [0x8F, 0x1E],
    "旁": [0x8F, 0x1F],
    "晓": [0x8F, 0x20],
    "躺": [0x8F, 0x21],
    "狭": [0x8F, 0x22],
    "审": [0x8F, 0x23],
    "禁": [0x8F, 0x24],
    "谎": [0x8F, 0x25],
    "挨": [0x8F, 0x26],
    "泥": [0x8F, 0x27],
    "沮": [0x8F, 0x28],
    "嘻": [0x8F, 0x29],
    "擔": [0x8F, 0x2A],
    "靓": [0x8F, 0x2B],
    "丫": [0x8F, 0x2C],
    "吩": [0x8F, 0x2D],
    "杠": [0x8F, 0x2E],
    "杯": [0x8F, 0x2F],
    "屏": [0x8F, 0x30],
}


def check_if_correct_charset(char, cur_charset, filename, lineno):
    warn_msg = f"{filename}:{lineno}: Warning: character '{char}' is present but is completely in a wrong charset currently set. Add {{}} before the character to silence this warning."

    if char == " " or char == "　":
        return -1, cur_charset
    elif char in CHARSET_KANA and cur_charset is not CHARSET_KANA:
        print(warn_msg.format("[Charset Kana]"))
        return 0, CHARSET_KANA
    elif char in CHARSET_LATIN and cur_charset is not CHARSET_LATIN:
        print(warn_msg.format("[Charset Latin]"))
        return 1, CHARSET_LATIN
    elif char in CHARSET_KANJI and cur_charset is not CHARSET_KANJI:
        print(warn_msg.format("[Charset Kanji]"))
        return 2, CHARSET_KANJI
    elif char in CHARSET_BUTTONS and cur_charset is not CHARSET_BUTTONS:
        print(warn_msg.format("[Charset Buttons]"))
        return 3, CHARSET_BUTTONS

    return -1, cur_charset


def strip_c_comments(text):
    def replacer(match):
        s = match.group(0)
        if s.startswith("/"):
            return " "
        else:
            return s

    pattern = re.compile(
        r'//.*?$|/\*.*?\*/|\'(?:\\.|[^\\\'])*\'|"(?:\\.|[^\\"])*"',
        re.DOTALL | re.MULTILINE,
    )
    return re.sub(pattern, replacer, text)


if __name__ == "__main__":
    if len(argv) < 3:
        print("usage: parse_compile.py [version] [in.msg] [out.msgpack] [--c]")
        exit(1)

    version = argv[1]
    filename = argv[2]
    outfile = argv[3]
    is_output_format_c = "--c" in argv

    messages = []

    message = None
    with open(filename, "r") as f:
        source = strip_c_comments(f.read())
        lineno = 1

        directive = ""
        indent_level = 0

        if version == "jp":
            charset = CHARSET_KANA
        elif version == "ique":
            charset = CHARSET_IQUE
        else:
            charset = CHARSET_STANDARD
        font_stack = [0]
        sound_stack = [0]
        color_stack = [0x0A]
        fx_stack = []
        style = None
        explicit_end = False
        choiceindex = -1

        while len(source) > 0:
            if source[0] == "\r" or source[0] == "\t":
                source = source[1:]
                continue

            if source[0] == "\n":
                lineno += 1
                source = source[1:]

                for i in range(indent_level):
                    if source[0] == "\t":
                        source = source[1:]
                    else:
                        break

                continue

            if message is None:
                directive = ""
                while source[0] != " ":
                    if source[0] == "\n":
                        lineno += 1
                    elif source[0] == "\r":
                        pass
                    else:
                        directive += source[0]
                    source = source[1:]

                directive = directive.split(":")

                if directive[0] != "#message":
                    print(f"{filename}:{lineno}: expected #message directive")
                    exit(1)
                if is_output_format_c:
                    if len(directive) != 2:
                        print(f"{filename}:{lineno}: expected #message:NAME directive")
                        exit(1)

                    message = Message(directive[1], None, None)
                else:
                    if len(directive) != 3:
                        print(f"{filename}:{lineno}: expected #message:SECTION:INDEX directive")
                        exit(1)

                    section = int(directive[1], 16)

                    if directive[2].startswith("(") and directive[2].endswith(")"):
                        name = directive[2][1:-1]
                        index = None
                    else:
                        name = None
                        index = int(directive[2], 16)

                    directive = ""

                    message = Message(name, section, index)
                messages.append(message)

                if version == "jp":
                    charset = CHARSET_KANA
                elif version == "ique":
                    charset = CHARSET_IQUE
                else:
                    charset = CHARSET_STANDARD

                while source[0] != "{":
                    source = source[1:]

                    if source[0] == "\n":
                        lineno += 1
                    elif source[0] == "\r":
                        pass
                    elif source[0] == "{":
                        break
                    elif source[0] != " " and source[0] != "\t":
                        print(f"{filename}:{lineno}: expected opening brace ('{{')")
                        exit(1)

                source = source[1:]  # {

                # count indent level
                indent_level = 0
                """
                while source[0] == " " or source[0] == "\t" or source[0] == "\n" or source[0] == "\r":
                    if source[0] == " " or source[0] == "\t":
                        indent_level += 1
                    source = source[1:]
                """
            else:
                command, args, named_args, source = parse_command(source)

                if command:
                    if command == "end":
                        message.bytes += [0xFD]
                        explicit_end = True
                    elif command == "raw":
                        message.bytes += [*args]
                    elif command == "br":
                        message.bytes += [0xF0]
                    elif command == "wait":
                        message.bytes += [0xF1]
                    elif command == "pause":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                            exit(1)

                        message.bytes += [0xF2, args[0]]
                    elif command == "next":
                        message.bytes += [0xFB]
                    elif command == "yield":
                        message.bytes += [0xFF, 0x04]
                    elif command == "savecolor":
                        message.bytes += [0xFF, 0x24]
                    elif command == "restorecolor":
                        message.bytes += [0xFF, 0x25]
                    elif command == "color":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: color command requires 1 parameter")
                            exit(1)

                        color = color_to_code(args[0], style)

                        if color is None:
                            print(f"{filename}:{lineno}: unknown color")
                            exit(1)

                        message.bytes += [0xFF, 0x05, color]
                        # color_stack.append(color)
                    # elif command == "/color":
                    #    color_stack.pop()
                    #    message.bytes += [0xFF, 0x05, color_stack[0]]
                    elif command == "style":
                        message.bytes += [0xFC]

                        style = args[0]
                        args = args[1:]
                        if type(style) is int:
                            message.bytes += [style, *args]
                        else:
                            if style == "right":
                                message.bytes += [0x01]
                            elif style == "left":
                                message.bytes += [0x02]
                            elif style == "center":
                                message.bytes += [0x03]
                            elif style == "tattle":
                                message.bytes += [0x04]
                            elif style == "choice":
                                pos = named_args.get("pos")

                                if not isinstance(pos, list) or len(pos) != 2:
                                    print(f"{filename}:{lineno}: 'choice' style requires pos=_,_")
                                    exit(1)

                                size = named_args.get("size")

                                if not isinstance(size, list) or len(size) != 2:
                                    print(f"{filename}:{lineno}: 'choice' style requires size=_,_")
                                    exit(1)

                                message.bytes += [
                                    0x05,
                                    pos[0],
                                    pos[1],
                                    size[0],
                                    size[1],
                                ]
                            elif style == "inspect":
                                message.bytes += [0x06]
                            elif style == "sign":
                                message.bytes += [0x07]
                            elif style == "lamppost":
                                height = named_args.get("height")

                                if not isinstance(height, int):
                                    print(f"{filename}:{lineno}: 'lamppost' style requires height=_")
                                    exit(1)

                                message.bytes += [0x08, height]
                            elif style == "postcard":
                                index = named_args.get("index")

                                if not isinstance(index, int):
                                    print(f"{filename}:{lineno}: 'postcard' style requires index=_")
                                    exit(1)

                                message.bytes += [0x09, index]
                            elif style == "popup":
                                message.bytes += [0x0A]
                            elif style == "popup2":
                                message.bytes += [0x0B]
                            elif style == "upgrade":
                                pos = named_args.get("pos")

                                if not isinstance(pos, list) or len(pos) != 2:
                                    print(f"{filename}:{lineno}: 'upgrade' style requires pos=_,_")
                                    exit(1)

                                size = named_args.get("size")

                                if not isinstance(size, list) or len(size) != 2:
                                    print(f"{filename}:{lineno}: 'upgrade' style requires size=_,_")
                                    exit(1)

                                message.bytes += [
                                    0x0C,
                                    pos[0],
                                    pos[1],
                                    size[0],
                                    size[1],
                                ]
                            elif style == "narrate":
                                message.bytes += [0x0D]
                            elif style == "epilogue":
                                message.bytes += [0x0E]
                    elif command == "font":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: font command requires 1 parameter")
                            exit(1)

                        font = args[0]

                        if font == "standard":
                            font = 0
                        elif font == "menu":
                            font = 1
                        elif font == "menu2":
                            font = 2
                        elif font == "title":
                            font = 3
                        elif font == "subtitle":
                            font = 4

                        if type(font) is not int:
                            print(f"{filename}:{lineno}: unknown font '{font}'")
                            exit(1)

                        message.bytes += [0xFF, 0x00, font]
                        # font_stack.append(font)

                        if font == 3 or font == 4:
                            charset = CHARSET_CREDITS
                        else:
                            if version == "jp":
                                charset = CHARSET_KANA
                            elif version == "ique":
                                charset = CHARSET_IQUE
                            else:
                                charset = CHARSET_STANDARD
                    # elif command == "/font":
                    #     font_stack.pop()
                    #     message.bytes += [0xFF, 0x00, font_stack[0]]

                    #     if font == 3 or font == 4:
                    #         charset = CHARSET_CREDITS
                    #     else:
                    #         charset = CHARSET
                    elif command == "charset":
                        if version != "jp":
                            print(f"{filename}:{lineno}: charset command is only supported in the JP version")
                            exit(1)

                        if len(args) != 1:
                            print(f"{filename}:{lineno}: charset command requires 1 parameter")
                            exit(1)

                        arg_charset = args[0]

                        if arg_charset == "kana":
                            arg_charset = 0
                        elif arg_charset == "latin":
                            arg_charset = 1
                        elif arg_charset == "kanji":
                            arg_charset = 2
                        elif arg_charset == "buttons":
                            arg_charset = 3

                        if type(arg_charset) is not int:
                            print(f"{filename}:{lineno}: unknown charset '{arg_charset}'")
                            exit(1)

                        message.bytes += [0xF3 + arg_charset]

                        if arg_charset == 0:
                            charset = CHARSET_KANA
                        elif arg_charset == 1:
                            charset = CHARSET_LATIN
                        elif arg_charset == 2:
                            charset = CHARSET_KANJI
                        elif arg_charset == 3:
                            charset = CHARSET_BUTTONS

                    elif command == "variant0":
                        message.bytes += [0xF3]

                    elif command == "inputoff":
                        message.bytes += [0xFF, 0x07]
                    elif command == "inputon":
                        message.bytes += [0xFF, 0x08]
                    elif command == "delayoff":
                        message.bytes += [0xFF, 0x09]
                    elif command == "delayon":
                        message.bytes += [0xFF, 0x0A]
                    elif command == "charwidth":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                            exit(1)

                        message.bytes += [0xFF, 0x0B, args[0]]
                    elif command == "scroll":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                            exit(1)

                        message.bytes += [0xFF, 0x0C, args[0]]
                    elif command == "size":
                        args = args[0]

                        if len(args) == 1:
                            args.append(args[0])

                        if len(args) != 2:
                            print(f"{filename}:{lineno}: {command} command requires 2 parameters")
                            exit(1)

                        message.bytes += [0xFF, 0x0D, args[0], args[1]]
                    elif command == "sizereset":
                        message.bytes += [0xFF, 0x0E]
                    elif command == "speed":
                        delay = named_args.get("delay")

                        if not isinstance(delay, int):
                            print(f"{filename}:{lineno}: {command} command requires delay=_")
                            exit(1)

                        chars = named_args.get("chars")

                        if not isinstance(delay, int):
                            print(f"{filename}:{lineno}: {command} command requires chars=_")
                            exit(1)

                        message.bytes += [0xFF, 0x0F, delay, chars]
                    # elif command == "pos":
                    #     if "y" not in named_args:
                    #         print(f"{filename}:{lineno}: pos command requires parameter: y (x is optional)")
                    #         exit(1)

                    #     if "x" in named_args:
                    #         message.bytes += [0xFF, 0x10, named_args["x"], named_args["y"]]
                    #     else:
                    #         message.bytes += [0xFF, 0x11, named_args["y"]]
                    elif command == "setposx":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                            exit(1)

                        message.bytes += [0xFF, 0x10, args[0] >> 8, args[0] & 0xFF]
                    elif command == "setposy":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                            exit(1)

                        message.bytes += [0xFF, 0x11, *args]
                    elif command == "right":
                        if len(args) == 0:
                            if version == "jp":
                                charset_byte, charset = check_if_correct_charset("[right]", charset, filename, lineno)
                                if charset_byte != -1:
                                    message.bytes += [0xF3 + charset_byte]
                                message.bytes += [0xB4]
                            else:
                                message.bytes += [0x95]
                        else:
                            if len(args) != 1:
                                print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                                exit(1)

                            message.bytes += [0xFF, 0x12, args[0]]
                    elif command == "down":
                        if len(args) == 0:
                            if version == "jp":
                                charset_byte, charset = check_if_correct_charset("[down]", charset, filename, lineno)
                                if charset_byte != -1:
                                    message.bytes += [0xF3 + charset_byte]
                                message.bytes += [0xB2]
                            else:
                                message.bytes += [0x93]
                        else:
                            if len(args) != 1:
                                print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                                exit(1)

                            message.bytes += [0xFF, 0x13, args[0]]
                    elif command == "up":
                        if len(args) == 0:
                            if version == "jp":
                                charset_byte, charset = check_if_correct_charset("[up]", charset, filename, lineno)
                                if charset_byte != -1:
                                    message.bytes += [0xF3 + charset_byte]
                                message.bytes += [0xB1]
                            else:
                                message.bytes += [0x92]
                        else:
                            if len(args) != 1:
                                print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                                exit(1)

                            message.bytes += [0xFF, 0x14, args[0]]
                    elif command == "inlineimage":
                        index = named_args.get("index")

                        if not isinstance(index, int):
                            print(f"{filename}:{lineno}: {command} command requires index=_")
                            exit(1)

                        message.bytes += [0xFF, 0x15, index]
                    elif command == "animsprite":
                        spriteid = named_args.get("spriteid")
                        raster = named_args.get("raster")

                        # TODO: named sprite id and raster

                        if not isinstance(spriteid, int):
                            print(f"{filename}:{lineno}: {command} command requires spriteid=_")
                            exit(1)
                        if not isinstance(raster, int):
                            print(f"{filename}:{lineno}: {command} command requires raster=_")
                            exit(1)

                        message.bytes += [
                            0xFF,
                            0x16,
                            spriteid >> 8,
                            spriteid & 0xFF,
                            raster,
                        ]
                    elif command == "itemicon":
                        itemid = named_args.get("itemid")

                        # TODO: itemname

                        if not isinstance(itemid, int):
                            print(f"{filename}:{lineno}: {command} command requires itemid=_")
                            exit(1)

                        message.bytes += [0xFF, 0x17, itemid >> 8, itemid & 0xFF]
                    elif command == "image":
                        index = named_args.get("index")
                        pos = named_args.get("pos")  # xx,y
                        hasborder = named_args.get("hasborder")
                        alpha = named_args.get("alpha")
                        fadeamount = named_args.get("fadeamount")

                        if not isinstance(index, int):
                            print(f"{filename}:{lineno}: {command} command requires index=_")
                            exit(1)
                        if not isinstance(pos, list) or len(pos) != 2:
                            print(f"{filename}:{lineno}: {command} command requires pos=_,_")
                            exit(1)
                        if not isinstance(hasborder, int):
                            print(f"{filename}:{lineno}: {command} command requires hasborder=_")
                            exit(1)
                        if not isinstance(alpha, int):
                            print(f"{filename}:{lineno}: {command} command requires alpha=_")
                            exit(1)
                        if not isinstance(fadeamount, int):
                            print(f"{filename}:{lineno}: {command} command requires fadeamount=_")
                            exit(1)

                        message.bytes += [
                            0xFF,
                            0x18,
                            index,
                            pos[0] >> 8,
                            pos[0] & 0xFF,
                            pos[1],
                            hasborder,
                            alpha,
                            fadeamount,
                        ]
                    elif command == "hideimage":
                        fadeamount = named_args.get("fadeamount", 0)

                        if not isinstance(fadeamount, int):
                            print(f"{filename}:{lineno}: {command} command requires fadeamount=_")
                            exit(1)

                        message.bytes += [0xFF, 0x19, fadeamount]
                    elif command == "animdelay":
                        index = named_args.get("index")
                        delay = named_args.get("delay")

                        if not isinstance(index, int):
                            print(f"{filename}:{lineno}: {command} command requires index=_")
                            exit(1)
                        if not isinstance(delay, int):
                            print(f"{filename}:{lineno}: {command} command requires delay=_")
                            exit(1)

                        message.bytes += [0xFF, 0x1A, 0, index, delay]
                    elif command == "animloop":
                        if len(args) != 2:
                            print(f"{filename}:{lineno}: {command} command requires 2 parameters")
                            exit(1)
                        message.bytes += [0xFF, 0x1B, args[0], args[1]]
                    elif command == "animdone":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                            exit(1)
                        message.bytes += [0xFF, 0x1C, args[0]]
                    elif command == "setcursorpos":
                        index = named_args.get("index")
                        pos = named_args.get("pos")

                        if not isinstance(index, int):
                            print(f"{filename}:{lineno}: {command} command requires index=_")
                            exit(1)
                        if not isinstance(pos, list) or len(pos) != 2:
                            print(f"{filename}:{lineno}: {command} command requires pos=_,_")
                            exit(1)

                        message.bytes += [0xFF, 0x1D, index, pos, pos]
                    elif command == "cursor":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: cursor command requires 1 parameter")
                            exit(1)

                        message.bytes += [0xFF, 0x1E, *args]
                    elif command == "option" and choiceindex == -1:
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: option command requires 1 parameter")
                            exit(1)

                        message.bytes += [0xFF, 0x21, *args]
                    elif command == "endchoice" and choiceindex == -1:
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                            exit(1)

                        message.bytes += [0xFF, 0x1F, args[0]]
                    elif command == "setcancel":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                            exit(1)

                        message.bytes += [0xFF, 0x20, args[0]]
                    # elif command == "startfx":
                    #     message.bytes += [0xFF, 0x26, resolve_effect(args[0]), *args[1:]]
                    # elif command == "endfx":
                    #     message.bytes += [0xFF, 0x27, resolve_effect(args[0]), *args[1:]]
                    elif command == "/fx":
                        message.bytes += [0xFF, 0x27, fx_stack.pop()]
                    elif command == "shake":
                        fx_stack.append(0x00)
                        message.bytes += [0xFF, 0x26, 0x00]
                    elif command == "/shake":
                        fx_stack.pop()
                        message.bytes += [0xFF, 0x27, 0x00]
                    elif command == "wave":
                        fx_stack.append(0x01)
                        message.bytes += [0xFF, 0x26, 0x01]
                    elif command == "/wave":
                        fx_stack.pop()
                        message.bytes += [0xFF, 0x27, 0x01]
                    elif command == "noiseoutline":
                        fx_stack.append(0x02)
                        message.bytes += [0xFF, 0x26, 0x02]
                    elif command == "/noiseoutline":
                        fx_stack.pop()
                        message.bytes += [0xFF, 0x27, 0x02]
                    elif command == "static":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                            exit(1)

                        fx_stack.append(0x03)
                        message.bytes += [0xFF, 0x26, 0x03, args[0]]
                    elif command == "/static":
                        fx_stack.pop()
                        message.bytes += [0xFF, 0x27, 0x03]
                    elif command == "blur":
                        _dir = named_args.get("dir")

                        if _dir == "x":
                            _dir = 0
                        elif _dir == "y":
                            _dir = 1
                        elif _dir == "xy":
                            _dir = 2

                        if not isinstance(_dir, int):
                            print(f"{filename}:{lineno}: {command} command requires dir=_")
                            exit(1)

                        fx_stack.append(0x05)
                        message.bytes += [0xFF, 0x26, 0x05, _dir]
                    elif command == "/blur":
                        fx_stack.pop()
                        message.bytes += [0xFF, 0x27, 0x05]
                    elif command == "rainbow":
                        fx_stack.append(0x06)
                        message.bytes += [0xFF, 0x26, 0x06]
                    elif command == "/rainbow":
                        fx_stack.pop()
                        message.bytes += [0xFF, 0x27, 0x06]
                    elif command == "ditherfade":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                            exit(1)

                        fx_stack.append(0x07)
                        message.bytes += [0xFF, 0x26, 0x07, args[0]]
                    elif command == "/ditherfade":
                        fx_stack.pop()
                        message.bytes += [0xFF, 0x27, 0x07]
                    elif command == "printrising":
                        fx_stack.append(0x0A)
                        message.bytes += [0xFF, 0x26, 0x0A]
                    elif command == "/printrising":
                        fx_stack.pop()
                        message.bytes += [0xFF, 0x27, 0x0A]
                    elif command == "printgrowing":
                        fx_stack.append(0x0B)
                        message.bytes += [0xFF, 0x26, 0x0B]
                    elif command == "/printgrowing":
                        fx_stack.pop()
                        message.bytes += [0xFF, 0x27, 0x0B]
                    elif command == "sizejitter":
                        fx_stack.append(0x0C)
                        message.bytes += [0xFF, 0x26, 0x0C]
                    elif command == "/sizejitter":
                        fx_stack.pop()
                        message.bytes += [0xFF, 0x27, 0x0C]
                    elif command == "sizewave":
                        fx_stack.append(0x0D)
                        message.bytes += [0xFF, 0x26, 0x0D]
                    elif command == "/sizewave":
                        fx_stack.pop()
                        message.bytes += [0xFF, 0x27, 0x0D]
                    elif command == "dropshadow":
                        fx_stack.append(0x0E)
                        message.bytes += [0xFF, 0x26, 0x0E]
                    elif command == "/dropshadow":
                        fx_stack.pop()
                        message.bytes += [0xFF, 0x27, 0x0E]
                    elif command == "var":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: var command requires 1 parameter")
                            exit(1)

                        message.bytes += [0xFF, 0x28, *args]
                    elif command == "centerx":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                            exit(1)

                        message.bytes += [0xFF, 0x29, *args]
                    elif command == "rewindoff":
                        message.bytes += [0xFF, 0x2A, 0]
                    elif command == "rewindon":
                        message.bytes += [0xFF, 0x2A, 1]
                    elif command == "customvoice":
                        soundids = named_args.get("soundids")

                        if not isinstance(soundids, list) or len(pos) != 2:
                            print(f"{filename}:{lineno}: {command} command requires soundids=_,_")
                            exit(1)

                        message.bytes += [
                            0xFF,
                            0x2C,
                            soundids[0] >> 24,
                            (soundids[0] >> 16) & 0xFF,
                            (soundids[0] >> 8) & 0xFF,
                            soundids[0] & 0xFF,
                            soundids[1] >> 24,
                            (soundids[1] >> 16) & 0xFF,
                            (soundids[1] >> 8) & 0xFF,
                            soundids[1] & 0xFF,
                        ]
                    elif command == "volume":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                            exit(1)

                        message.bytes += [0xFF, 0x2E, *args]
                    elif command == "voice":
                        if len(args) != 1:
                            print(f"{filename}:{lineno}: {command} command requires 1 parameter")
                            exit(1)

                        sound = args[0]

                        if sound == "normal":
                            sound = 0
                        elif sound == "bowser":
                            sound = 1
                        elif sound == "star" or sound == "spirit":
                            sound = 2

                        if type(sound) is not int:
                            print(f"{filename}:{lineno}: unknown voice '{sound}'")
                            exit(1)

                        message.bytes += [0xFF, 0x2F, sound]
                        # sound_stack.append(sound)
                    # elif command == "/sound":
                    #     sound_stack.pop()
                    #     message.bytes += [0xFF, 0x2F, sound_stack[0]]
                    elif command == "a":
                        color_code = color_to_code("blue", "button")
                        assert color_code is not None
                        if version == "jp":
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0xF6,
                                0x00,
                                0xFF,
                                0x25,
                            ]
                        else:
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0x98,
                                0xFF,
                                0x25,
                            ]
                    elif command == "b":
                        color_code = color_to_code(
                            named_args.get("color", "green"),
                            named_args.get("ctx", "button"),
                        )
                        assert color_code is not None
                        if version == "jp":
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0xF6,
                                0x01,
                                0xFF,
                                0x25,
                            ]
                        else:
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0x99,
                                0xFF,
                                0x25,
                            ]
                    elif command == "l":
                        color_code = color_to_code(
                            named_args.get("color", "gray"),
                            named_args.get("ctx", "button"),
                        )
                        assert color_code is not None
                        if version == "jp":
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0xF6,
                                0x08,
                                0xFF,
                                0x25,
                            ]
                        else:
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0x9A,
                                0xFF,
                                0x25,
                            ]
                    elif command == "r":
                        color_code = color_to_code(
                            named_args.get("color", "gray"),
                            named_args.get("ctx", "button"),
                        )
                        assert color_code is not None
                        if version == "jp":
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0xF6,
                                0x09,
                                0xFF,
                                0x25,
                            ]
                        else:
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0x9B,
                                0xFF,
                                0x25,
                            ]
                    elif command == "z":
                        color_code = color_to_code("grey", "button")
                        assert color_code is not None
                        if version == "jp":
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0xF6,
                                0x07,
                                0xFF,
                                0x25,
                            ]
                        else:
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0x9C,
                                0xFF,
                                0x25,
                            ]
                    elif command == "c-up":
                        color_code = color_to_code(
                            named_args.get("color", "yellow"),
                            named_args.get("ctx", "button"),
                        )
                        assert color_code is not None
                        if version == "jp":
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0xF6,
                                0x03,
                                0xFF,
                                0x25,
                            ]
                        else:
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0x9D,
                                0xFF,
                                0x25,
                            ]
                    elif command == "c-down":
                        color_code = color_to_code(
                            named_args.get("color", "yellow"),
                            named_args.get("ctx", "button"),
                        )
                        assert color_code is not None
                        if version == "jp":
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0xF6,
                                0x04,
                                0xFF,
                                0x25,
                            ]
                        else:
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0x9E,
                                0xFF,
                                0x25,
                            ]
                    elif command == "c-left":
                        color_code = color_to_code(
                            named_args.get("color", "yellow"),
                            named_args.get("ctx", "button"),
                        )
                        assert color_code is not None
                        if version == "jp":
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0xF6,
                                0x05,
                                0xFF,
                                0x25,
                            ]
                        else:
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0x9F,
                                0xFF,
                                0x25,
                            ]
                    elif command == "c-right":
                        color_code = color_to_code(
                            named_args.get("color", "yellow"),
                            named_args.get("ctx", "button"),
                        )
                        assert color_code is not None
                        if version == "jp":
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0xF6,
                                0x06,
                                0xFF,
                                0x25,
                            ]
                        else:
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0xA0,
                                0xFF,
                                0x25,
                            ]
                    elif command == "start":
                        color_code = color_to_code(
                            named_args.get("color", "red"),
                            named_args.get("ctx", "button"),
                        )  #
                        assert color_code is not None
                        if version == "jp":
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0xF6,
                                0x02,
                                0xFF,
                                0x25,
                            ]
                        else:
                            message.bytes += [
                                0xFF,
                                0x24,
                                0xFF,
                                0x05,
                                color_code,
                                0xA1,
                                0xFF,
                                0x25,
                            ]
                    elif command == "~a":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[~a]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0x00]
                        else:
                            message.bytes += [0x98]
                    elif command == "~b":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[~b]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0x01]
                        else:
                            message.bytes += [0x99]
                    elif command == "~l":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[~l]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0x08]
                        else:
                            message.bytes += [0x9A]
                    elif command == "~r":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[~r]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0x09]
                        else:
                            message.bytes += [0x9B]
                    elif command == "~z":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[~z]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0x07]
                        else:
                            message.bytes += [0x9C]
                    elif command == "~c-up":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[~c-up]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0x03]
                        else:
                            message.bytes += [0x9D]
                    elif command == "~c-down":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[~c-down]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0x04]
                        else:
                            message.bytes += [0x9E]
                    elif command == "~c-left":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[~c-left]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0x05]
                        else:
                            message.bytes += [0x9F]
                    elif command == "~c-right":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[~c-right]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0x06]
                        else:
                            message.bytes += [0xA0]
                    elif command == "~start":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[~start]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0x02]
                        else:
                            message.bytes += [0xA1]
                    elif command == "note":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[note]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0x6A]
                        else:
                            message.bytes += [0x00]
                    elif command == "heart":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[heart]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0xBD]
                        else:
                            message.bytes += [0x90]
                    elif command == "star":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[star]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0xBE]
                        else:
                            message.bytes += [0x91]
                    elif command == "left":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[left]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0xB3]
                        else:
                            message.bytes += [0x94]
                    elif command == "circle":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[circle]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0x61]
                        else:
                            message.bytes += [0x96]
                    elif command == "cross":
                        if version == "jp":
                            charset_byte, charset = check_if_correct_charset("[cross]", charset, filename, lineno)
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0x62]
                        else:
                            message.bytes += [0x97]
                    elif command == "katakana":
                        if version != "jp":
                            print(f"{filename}:{lineno}: Command katakana is only supported in the JP version")
                            exit(1)

                        kana_char = args[0]

                        if kana_char == "smalln":
                            charset_byte, charset = check_if_correct_charset(
                                "[katakana smalln]", charset, filename, lineno
                            )
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0xC5]
                        else:
                            print(f"{filename}:{lineno}: Invalid or unimplemented katakana character name {kana_char}")
                            exit(1)
                    elif command == "hiragana":
                        if version != "jp":
                            print(f"{filename}:{lineno}: Command hiragana is only supported in the JP version")
                            exit(1)

                        kana_char = args[0]

                        if kana_char == "smalln":
                            charset_byte, charset = check_if_correct_charset(
                                "[hiragana smalln]", charset, filename, lineno
                            )
                            if charset_byte != -1:
                                message.bytes += [0xF3 + charset_byte]
                            message.bytes += [0xC4]
                        else:
                            print(f"{filename}:{lineno}: Invalid or unimplemented hiragana character name {kana_char}")
                            exit(1)
                    elif command == "fullspace":
                        message.bytes += [0xF8]
                    elif command == "halfspace":
                        message.bytes += [0xF9]
                    elif command == "savepos":
                        message.bytes += [0xFF, 0x22]
                    elif command == "restorepos":
                        message.bytes += [0xFF, 0x23]
                    elif command == "enablecdownnext":
                        message.bytes += [0xFF, 0x2B]
                    elif command == "beginchoice":
                        choiceindex = 0
                        message.bytes += [0xFF, 0x09]  # delayoff
                    elif command == "option" and choiceindex >= 0:
                        message.bytes += [0xFF, 0x1E, choiceindex]  # cursor n
                        message.bytes += [0xFF, 0x21, choiceindex]  # option n
                        choiceindex += 1
                    elif command == "endchoice" and choiceindex >= 0:
                        cancel = named_args.get("cancel")

                        message.bytes += [0xFF, 0x21, 255]  # option 255
                        message.bytes += [0xFF, 0x0A]  # delayon

                        if isinstance(cancel, int):
                            message.bytes += [0xFF, 0x20, cancel]  # setcancel n

                        message.bytes += [0xFF, 0x1F, choiceindex]  # endchoice n

                        choiceindex = -1
                    elif command == "animation" and choiceindex >= 0:
                        # TODO
                        print(f"{filename}:{lineno}: '{command}' tag is not yet implemented")
                        exit(1)
                    else:
                        print(f"{filename}:{lineno}: unknown command '{command}'")
                        exit(1)
                else:
                    if source[0] == "}":
                        if not explicit_end:
                            print(f"{filename}:{lineno}: warning: string lacks an [end] command")
                            # message.bytes += [0xFD]
                        explicit_end = False

                        # sanity check
                        for b in message.bytes:
                            if not isinstance(b, int):
                                print(b)

                        # padding
                        while len(message.bytes) % 4 != 0:
                            message.bytes += [0x00]

                        message = None
                        source = source[1:]  # }
                        indent_level = 0
                        choiceindex = -1
                        continue

                    if source[0] == "\\":
                        source = source[1:]

                    if version == "jp" and charset is not CHARSET_CREDITS:
                        charset_byte, charset = check_if_correct_charset(source[0], charset, filename, lineno)
                        if charset_byte != -1:
                            message.bytes += [0xF3 + charset_byte]
                        elif (
                            source[0] not in CHARSET_KANA
                            and source[0] not in CHARSET_LATIN
                            and source[0] not in CHARSET_KANJI
                            and source[0] not in CHARSET_BUTTONS
                        ):
                            print(f"{filename}:{lineno}: unsupported character '{source[0]}' for current font")
                            exit(1)

                        data = charset[source[0]]

                        if type(data) is int:
                            message.bytes.append(data)
                        else:
                            message.bytes += data

                        source = source[1:]
                    else:
                        if source[0] in charset:
                            data = charset[source[0]]

                            if type(data) is int:
                                message.bytes.append(data)
                            else:
                                message.bytes += data

                            source = source[1:]
                        else:
                            print(f"{filename}:{lineno}: unsupported character '{source[0]}' for current font")
                            exit(1)

        if message != None:
            print(f"{filename}: missing [end]")
            exit(1)

    if is_output_format_c:
        with open(outfile, "w") as f:
            f.write(f"#include <ultra64.h>\n")

            for message in messages:
                f.write(f"static s8 {message.name}[] = {{\n")
                for b in message.bytes:
                    f.write(f"0x{b:02X},")
                f.write(f"\n}};\n")

    else:
        with open(outfile, "wb") as f:
            msgpack.pack(
                [
                    {
                        "section": message.section,
                        "index": message.index,
                        "name": message.name,
                        "bytes": bytes(message.bytes),
                    }
                    for message in messages
                ],
                f,
            )
