import { createJSCodeshiftTransformationRule } from '@wakaru/shared/rule'
import type { ASTTransformation } from '@wakaru/shared/rule'
import type { MemberExpression, StringLiteral } from 'jscodeshift'

/**
 * Restore template literal syntax from string concatenation.
 *
 * @example
 * // TypeScript / Babel / SWC / esbuild
 * "the ".concat(first, " take the ").concat(second, " and ").concat(third);
 * ->
 * `the ${first} take the ${second} and ${third}`
 */
export const transformAST: ASTTransformation = (context) => {
    const { root, j } = context

    root
        .find(j.CallExpression, {
            callee: {
                type: 'MemberExpression',
                object: {
                    type: 'StringLiteral',
                },
                property: {
                    type: 'Identifier',
                    name: 'concat',
                },
            },
        })
        .forEach((path) => {
            const object = (path.node.callee as MemberExpression).object as StringLiteral

            // goes up the tree to find the parent CallExpression and check if it's a concat
            // this is to find the start of the concat chain
            // and collect all arguments
            let parent = path
            const args = [object]
            while (parent) {
                // @ts-expect-error skip check for object and property
                args.push(...parent.node.arguments)
                if (j.match(parent?.parent?.parent, {
                    type: 'CallExpression',
                    callee: {
                        type: 'MemberExpression',
                        object: {
                            type: 'CallExpression',
                            // @ts-expect-error skip check for object and property
                            callee: {
                                type: 'MemberExpression',
                            },
                        },
                        property: {
                            type: 'Identifier',
                            name: 'concat',
                        },
                    },
                })) {
                    parent = parent.parent.parent
                    continue
                }

                break
            }

            if (!j.CallExpression.check(parent.node)) return

            const templateLiteral = args.reduce((acc, arg) => {
                if (j.StringLiteral.check(arg)) {
                    const escaped = arg.value
                        .replace(/(?<!\\)([\n])/g, '\\n')
                        .replace(/(?<!\\)([\t])/g, '\\t')
                        .replace(/(?<!\\)([\r])/g, '\\r')
                        .replace(/(?<!\\)([\b])/g, '\\b')
                        .replace(/(?<!\\)([\f])/g, '\\f')
                        .replace(/(?<!\\)([\v])/g, '\\v')
                        .replace(/(?<!\\)([\0])/g, '\\0')
                        .replace(/(?<!\\)`/g, '\\`')
                        .replace(/(?<!\\)\$/g, '\\$')
                    return acc + escaped
                }

                return `${acc}\${${j(arg).toSource()}}`
            }, '')

            j(parent).replaceWith(j.templateLiteral([j.templateElement({ raw: templateLiteral, cooked: templateLiteral }, true)], []))
        })
}

export default createJSCodeshiftTransformationRule({
    name: 'un-template-literal',
    transform: transformAST,
})
