//
// SPDX-FileCopyrightText: 2016-2024 by pi-lar GmbH
// SPDX-License-Identifier: OSL-3.0
//
#ifndef NP_MAPREDUCE_H_
#define NP_MAPREDUCE_H_

#include <stdbool.h>

#include "np_list.h"
#include "np_tree.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Implementation of a map reduce data structures
 * map functions receive each item of a collection and must use the single
 * linked list 'map_result'. reduce function receive each item from this single
 * linked list and must fill the associative array 'reduce_results' after
 * applying their reduce functionality. This could be the counting of elements
 * of just the more complicated calculations. The associative array structures
 * 'map_args' and 'reduce_args' can be used to pass additional arguments to the
 * map / reduce functions, the main purpose for the map function is to pass the
 * input value(s) in the 'io'
 *
 * TODO:
 * - make map and reduce function executable/callable as a np_job_t
 * - use a spinlock to protect the map_result / reduce_result from concurrent
 * access
 * - use a condition to trigger the execution of the reduce function
 */

struct np_map_reduce_s;

typedef int8_t (*np_cmp_func)(struct np_map_reduce_s *mr_struct,
                              const void             *element);

typedef bool (*np_map_func)(struct np_map_reduce_s *mr_struct,
                            const void             *element);
typedef bool (*np_reduce_func)(struct np_map_reduce_s *mr_struct,
                               const void             *element);

typedef struct np_mr_args_s {
  np_tree_t *kv_pairs;
  void      *io;
  // empty yet;
} np_mr_args_t;

typedef struct np_map_reduce_s {
  np_cmp_func cmp; // a function to compare two elements to select teh best "map
                   // starting point"

  np_map_func  map;               // the function to call that execute a "map"
  np_mr_args_t map_args;          // passing arguments to mp
  np_sll_t(void_ptr, map_result); // a list of potential "map" candidates

  np_reduce_func
      reduce; // the reduce function to work on the list generated by "map"
  np_mr_args_t reduce_args;   // argumenst for the "reduce" part
  np_tree_t   *reduce_result; // a associative array of results (potentially
                              // shared between different "reduce" calls)

} np_map_reduce_t;

#ifdef __cplusplus
}
#endif

#endif // NP_MAPREDUCE_H_
