#!/usr/bin/env python
# SPDX-FileCopyrightText: 2016-2024 by pi-lar GmbH
# SPDX-License-Identifier: OSL-3.0
import sys
import re
from pprint import pprint
import string
import random

def id_generator(size=6, chars=string.ascii_uppercase + string.digits):
    return ''.join(random.choice(chars) for _ in range(size))

# regex to detect functions. may be executed with a MULTILINE flag
# I recommend https://regex101.com for testing purposes
interface_detection = "(?P<interface>#if\s+?(?P<interface_modifier>PROTECTED|PUBLIC)(?P<interface_priority>_HIGH|_MID|_LOW)?_INTERFACE(?P<interface_content>(.|(\n(?!\r)))*?)#endif\s*?//\s*?(?P=interface_modifier)(?P=interface_priority)?_INTERFACE)"
fn_detection = "(?P<fn_decl>(?P<fn_comment>(\/\/.+)|\/\*((?!(\*\/))[\s\S])*\*\/)?\s*(\n(?!\r))+\s*(?P<fn_sig>(?P<fn_modifier>(PUBLIC|PROTECTED|PRIVATE)?\s+((static|inline|extern)\s+)*)?\s*(?P<fn_returns>\S+\s*\*?){1}\s+(?P<fn_name>\S+){1}\s*\((?P<fn_params>[a-zA-Z0-9_*,\s\n\r]*)\)))[\s\r(?!\n)]*\{"

full_detection = interface_detection + "|" + fn_detection

class Fn_Desc:
    def __init__(self, match):
        self.match      = match
        self.fn_name    = ""
        self.fn_params  = ""
        self.fn_modifier= ["PRIVATE"]
        self.fn_comment = ""
        self.fn_returns = "void"
        self.fn_sig = ""
        self.fn_decl = ""

    def getComment(self):
        # TODO: generate one if none is present
        return self.fn_comment

class Scanner:
    def __init__(self,guard,prefix="",suffix=""):
        self.guard = guard
        self.warnings = []
        self.defaults = {'iterface':{'modifier':'PROTECTED','priority':'_LOW'}}
        self.interfaces = {"PUBLIC":{'_HIGH':[],'_MID':[],'_LOW':[]},"PROTECTED":{'_HIGH':[],'_MID':[],'_LOW':[]}}
        self.functions = []
        self.prefix =   "%s\r\n%s\r\n\r\n%s" % (
                        prefix,
                        "// Generated by makeheaders",
                        '\r\n'.join([
                            "#ifndef PUBLIC_INTERFACE\r\n  #define PUBLIC_INTERFACE 0\r\n#endif",
                            "#ifndef PUBLIC_LOW_INTERFACE\r\n  #define PUBLIC_LOW_INTERFACE 0\r\n#endif",
                            "#ifndef PUBLIC_MID_INTERFACE\r\n  #define PUBLIC_MID_INTERFACE 0\r\n#endif",
                            "#ifndef PUBLIC_HIGH_INTERFACE\r\n  #define PUBLIC_HIGH_INTERFACE 0\r\n#endif",
                            "#ifndef PROTECTED_INTERFACE\r\n  #define PROTECTED_INTERFACE 0\r\n#endif",
                            "#ifndef PROTECTED_LOW_INTERFACE\r\n  #define PROTECTED_LOW_INTERFACE 0\r\n#endif",
                            "#ifndef PROTECTED_MID_INTERFACE\r\n  #define PROTECTED_MID_INTERFACE 0\r\n#endif",
                            "#ifndef PROTECTED_HIGH_INTERFACE\r\n  #define PROTECTED_HIGH_INTERFACE 0\r\n#endif",
                            "#ifndef PUBLIC\r\n  #define PUBLIC\r\n#endif",
                            "#ifndef PROTECTED\r\n  #define PROTECTED\r\n#endif",
                            "#ifndef PRIVATE\r\n  #define PRIVATE\r\n#endif",
                        ])
                        )
        self.suffix =   suffix

    def scan(self, name, content):
        ret = None
        if name.endswith('.c'):
            ret = self.scanCFile(content)
        elif name.endswith('.h'):
            ret = self.scanHFile(content)
        else:
            self.warnings += "Could not scan \"%s\""%(name)

        return ret

    def scanHFile(self, content):
        isInPublic = False
        lastpos = 0
        for match in re.finditer(interface_detection, content, re.MULTILINE):
            if lastpos < match.start(0):
                pre_part = content[lastpos:match.start(0)].strip()
                if pre_part:
                    self.interfaces[self.defaults['iterface']['modifier']][self.defaults['iterface']['priority']].append(pre_part)

            interface_modifier = match.group("interface_modifier").strip().upper()
            interface_priority = match.group("interface_priority")
            if not interface_priority:
                interface_priority = self.defaults['iterface']['priority']
            interface_content = "\r\n%s\r\n" % (match.group("interface_content").strip())

            if interface_modifier not in self.interfaces.keys():
                self.interfaces[interface_modifier] = {'_HIGH':[],'_MID':[],'_LOW':[]}
            self.interfaces[interface_modifier][interface_priority].append(interface_content.strip())
            lastpos = match.end(0)

        post_part = content[lastpos:]
        if post_part:
            self.interfaces[self.defaults['iterface']['modifier']][self.defaults['iterface']['priority']].append(post_part.strip())

    def scanCFile(self, content):
        for match in  re.finditer(
                                full_detection, content, re.MULTILINE
                            ):
            fn_name = match.group("fn_name")
            interface = match.group("interface")

            # handle interface found
            if(interface):
                interface_modifier = match.group("interface_modifier").strip().upper()
                interface_priority = match.group("interface_priority")
                if not interface_priority:
                    interface_priority = self.defaults['iterface']['priority']
                interface_content = match.group("interface_content").strip()

                if interface_modifier not in self.interfaces.keys():
                    self.interfaces[interface_modifier] = {'_HIGH':[],'_MID':[],'_LOW':[]}

                if interface_content:
                    interface_content = "%s\r\n" % (interface_content)
                    self.interfaces[interface_modifier][interface_priority].append(interface_content)

            # handle function signature
            if(fn_name):
                tmp = Fn_Desc(match);
                self.functions.append(tmp)
                tmp.fn_name = fn_name

                for group in [  "fn_params","fn_comment","fn_returns",
                                "fn_decl","fn_sig"]:
                    tmp_prop = match.group(group)
                    if tmp_prop:
                        setattr(tmp, group, tmp_prop.strip())

                tmp_prop = match.group("fn_modifier")
                if tmp_prop:
                    tmp.fn_modifier  = list(filter(bool,tmp_prop.strip().split(" ")))

    def export(self, level, includePrefix=True, includeSuffix=True):
        ret = ""

        if self.guard and self.guard != "" :
            ret += "\r\n#ifndef %s\r\n#define %s\r\n" % (self.guard,self.guard)

        if includePrefix:
            ret += self.prefix

        if level in self.interfaces.keys():
            ret += "\r\n".join(self.interfaces[level]['_HIGH'])
            ret += "\r\n".join(self.interfaces[level]['_MID'])
            ret += "\r\n".join(self.interfaces[level]['_LOW'])

        # filter functions for desired modifier
        filterd_fns = list(filter(
                        lambda f: (level in f.fn_modifier), self.functions
                        ))
        seen = set()
        filterd_fns_unique = []
        for fn in filterd_fns:
            ident = fn.fn_name
            if ident not in seen:
                filterd_fns_unique.append(fn)
                seen.add(ident)
            else:
                self.warnings.append(
                    "Dupplicate function definition \"%s\""%(ident)
                )

        # get the corresponging declaration
        reduced_fns = ["%s%s;"%(x.getComment() , x.fn_sig) for x in filterd_fns_unique]

        # add to resultset
        ret += "\r\n\r\n".join(reduced_fns)

        if includeSuffix:
            ret += self.suffix

        if self.guard and self.guard != "":
            ret += "\r\n#endif // %s\r\n" % (self.guard)

        return ret

if __name__ == "__main__":
    import argparse
    parser = argparse.ArgumentParser(description='Create h files from .c files')
    parser.add_argument('-t','--target',
                                    type=argparse.FileType('w+'),
                                    default=sys.stdout,
                                    help='target to write to')

    parser.add_argument('-l','--level',
                                    default="private",
                                    choices=["public","protected","private"],
                                    help='level to scan. values: public|protected|private')

    parser.add_argument('source',   type=argparse.FileType('r'),
                                    nargs='+',
                                    help='file to parse.')

    parser.add_argument('-p',       type=argparse.FileType('r'),
                                    default=None,
                                    help='file to take prefix string from')
    parser.add_argument('--prefix',
                                    nargs='*',
                                    default=None,
                                    help='prefix string')

    parser.add_argument('-s',       type=argparse.FileType('r'),
                                    default=None,
                                    help='file to take suffix string from')
    parser.add_argument('--suffix',
                                    nargs='*',
                                    default=None,
                                    help='suffix string')
    parser.add_argument('-g','--guard',
                                    default="_"+id_generator()+"_H",
                                    help='Guard string')
    args = parser.parse_args()
    args.level = args.level.upper()
    prefix =""
    if args.p:
        prefix += args.p.read()
    if args.prefix:
        prefix += "\r\n".join(args.prefix)


    suffix =""
    if args.suffix:
        suffix += "\r\n".join(args.suffix)
    if args.s:
        suffix += args.s.read()

    scanner = Scanner(args.guard, prefix, suffix);
    indent = "   "
    scanner.prefix += "\r\n/*\r\n%sScanned %3i files:\r\n" % (indent,len(args.source))
    i = 0
    for file_c in args.source:
        inc = True
        i += 1
        scanner.prefix += "%s%2i. %s\r\n" % (indent,i, file_c.name)
        scanner.scan(file_c.name, file_c.read())

    scanner.prefix += "*/\r\n"
    new_content = scanner.export(args.level)

    try:
        old_content = args.target.read()
    except:
        old_content = ""

    if old_content != new_content:
        args.target.write(new_content)
    else:
        print "do not overwrite \"%s\"" % (args.target.name)
