BSL430.NET.Console App
------------------
BSL430.NET.Console is simple .NET console app wrapping BSL430.NET main public methods. It was created because lack of support on **Linux** platform, WPF GUI app is Windows only, therefore console app was also needed, for making BSL430.NET available also for Linux developers and *open-source world*. For more details please read docs here at wiki [BSL430.NET Library](https://github.com/parezj/BSL430.NET/wiki/BSL430.NET.Library) or [FirmwareTools](https://github.com/parezj/BSL430.NET/wiki/BSL430.NET.FirmwareTools.Library), especially [**Device Set Up**](https://github.com/parezj/BSL430.NET/wiki/BSL430.NET.Library#Device-Set-Up), because each time you init communication with MCU, you need to correctly set device up (*MCU*, *InvokeMechanism*, *BaudRate*, *Password*). With few commands (and params) all main public methods of the main and aux package are exposed. **Upload**, **Download**, **Erase** and **FwTools** commands, described in table below.

```
Note: This app uses unmanaged libraries for communication, so bitness matters.
```
- [Compatibility](#Compatibility)
- [Commands](#Commands)
- [Params](#Params)
- [Examples](#Examples)
  - [BSL430.NET](#BSL430NET)
    - [Scan Devices](#Scan-Devices)
    - [Upload to MCU](#Upload-to-MCU)
    - [Download from MCU](#Download-from-MCU)
    - [Erase MCU](#Erase-MCU)
  - [BSL430.NET.FirmwareTools](#BSL430NETFirmwareTools)
    - [Convert](#Convert)
    - [Combine](#Combine)
    - [Validate](#Validate)
    - [Get BSL Password](#Get-BSL-Password)
    - [Compare](#Compare)

Compatibility
-------------
BSL430.NET.Console is built using **.NET Core 3**. Each architecture (x86, amd64) has its own build for **Framework Dependent** deployment (standard build, correct unmanaged libraries need to be put in correct place) and for **Self Contained** deployment (.NET Core is packed with app - /p:*PublishSingleFile*=true and /p:*PublishTrimmed*=true flags are set for trimming not needed assemblies and packing into single executable). These optimizations makes both apps having similar size, which is interesting.
- Framework
  - .NET Core 3.0
    - Framework Dependent
    - Self Contained
- OS
  - Windows 7, 8, 8.1, 10
    - x86
    - amd64
  - Linux
    - amd64

Commands
--------

| Cmd | Long Name      | Main Param                    | Required Params  | Optional Params | Description |
|:---:|----------------|-------------------------------|------------------|-----------------|-------------|
| -s  | --scan         | [all/ftdi/libftdi/usb/serial] |                  | t, w, x         | Scan for available devices.
| -u  | --upload       | device name eg.: FTDI1        | f, m, i          | b, p, x         | Upload firmware to specified device eg. FTDI1 from TI-TXT, Intel-HEX, SREC or ELF file. Firmware file, MCU family and Invoke Mechanism are required.
| -d  | --download     | device name eg.: USB2         | f, m, i, p, a, z | b, o, y, x      | Download firmware from specified device eg. USB2. and write it to TI-TXT, Intel-HEX or SREC file (ELF is not supported yet). Firmware file, MCU family, Invoke Mechanism, Password, Start Addres and Data Size are required.
| -e  | --erase        | device name eg.: COM3         | m, i             | x               | Erase specified device eg. COM3. Firmware file, MCU family and Invoke Mechanism are required.
|     |                |                               |                  |                 |
| -c  | --convert      | firmware file path            | f, o             | l, y            | Convert specified firmware file and write it to -f path. Set -o output fw format: TI-TXT, Intel-HEX or SREC (TI-TXT is default. ELF is not supported yet). Destination firmware path and format are required.
| -n  | --combine      | first firmware file path      | j, f, o          | l, y            | Combine specified firmware file with another from -j path and write it to -f path. Set -o output fw format: TI-TXT, Intel-HEX or SREC (TI-TXT is default. ELF is not supported yet). Second and Destination firmware paths and format are required.
| -k  | --compare      | first firmware file path      | j                |                 | Compare two firmware files (format auto-detected; TI-TXT, Intel-HEX, SREC and ELF supported) and print result (Equal - True/False, Match - percentage [0.0 ; 100.0] %, BytesDiff - count of different byte nodes). Second firmware path is required.
| -v  | --validate     | firmware file path            |                  |                 | Read and parse firmware file (format auto-detected; TI-TXT, Intel-HEX, SREC and ELF supported) and print information (first addr, last addr, size, crc16..) and status if firmware is valid or not.
| -r  | --get-password | firmware file path            | m                |                 | Read and parse firmware file (format auto-detected; TI-TXT, Intel-HEX, SREC and ELF supported) and print BSL password (last 16/20/32 bytes of interrupt vector table 0xFFE0 - 0xFFFF). MCU family is required.
|     |                |                               |                  |                 |
| -g  | --force-ascii  |                               |                  |                 | Optional. Default output text encoding is UTF-8, however if you need ASCII only, use this option.

Params
------

| Param | Long Name            | Type                   | Description                                       |
|:-----:|--------------------- |----------------------- |---------------------------------------------------|
| -f    | --file               | string (path)          | Full or relative path to firmware file. (TI-TXT, Intel-HEX, SREC or ELF).
| -j    | --second-file        | string (path)          | Full or relative path to another firmware file, usually for Combine or Compare. (TI-TXT, Intel-HEX, SREC or ELF). 
| -p    | --password           | hex string (16/20/32)  | BSL password is 16/20/32 bytes long. Enter 32 chars long hex string. For upload and erase this is optional.
| -a    | --addrstart          | int                    | (Download only) start download from 16bit address specified. eg. '0F00'
| -z    | --datasize           | int                    | (Download only) download specified amount of bytes in hex. eg. 'FFFF'
| -b    | --baudrate           | enum (BaudRate)        | Sets baud rate. Default is 9600 bps (9600).
| -i    | --invoke             | enum (InvokeMechanism) | Sets invoke mechanism mode. Default is SHARED_JTAG (0).
| -m    | --mcu                | enum (MCU)             | Sets MCU family. Default is MSP430_F5xx (4).
| -l    | --fill-ff            | bool                   | Optional. Force output fw monolithic structure as missing addr nodes are filled with 0xFF.
| -x    | --xml                | string (path)          | Optional. XML Log output file. Can be only filename or absolute path.
| -t    | --usb-ignore-ti-vid  | bool                   | Optional. When scanning USB HID, ignore TI VID(0x2047) and get all USB HID devices present in system.
| -w    | --ftdi-ignore-unknow | bool                   | Optional. When scanning FTDI, ignore UNKNOWN type devices, that are most likely already connected.
| -o    | --output-format      | enum (FwFormat)        | Optional. Download or convert firmware format output mode: TI-TXT, Intel-HEX or SREC (TI-TXT is default. ELF is not supported yet as output).
| -y    | --fw-line-length     | int                    | Optional. Defines amount of data bytes present at final firmware text file at single row.  When = 0, default values are set (TI-TXT = 16, Intel-HEX = 32, SREC = 32).


Examples
--------
### BSL430.NET
Run this app without any cmd or param to show welcome screen with all info, commands, params, examples and enums.
```powershell
BSL430.NET.Console.exe
```
![Scan](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_home.png)

#### Scan Devices
Scan for all available devices.
```powershell
BSL430.NET.Console.exe -s all
```
Scan for all available devices and save XML output file.
```powershell
BSL430.NET.Console.exe --scan all -x scan.xml
```
![Scan](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_scan2.png)

Scan for single device
```powershell
BSL430.NET.Console.exe -s ftdi
BSL430.NET.Console.exe -s serial
BSL430.NET.Console.exe -s usb
BSL430.NET.Console.exe -s libftdi
```
![Scan](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_scan.png)

#### Upload to MCU
Upload fw.hex to MSP430_F5xx via FTDI1 with default baud rate (9600 bd/s), standard invoke mechanism (shared JTAG pins).
```powershell
BSL430.NET.Console.exe -u FTDI1 -f fw.hex -m MSP430_F5xx -i SHARED_JTAG
```
![Upload](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_upload.png)

Same action, same settings, but now faster (115200 bd/s) and save XML log.
```powershell
BSL430.NET.Console.exe --upload FTDI1 -f fw.hex -m MSP430_F5xx -i SHARED_JTAG -b 115200 -x log.xml
```
![Upload](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_upload2.png)

#### Download from MCU
Download fw.txt (TI-TXT format) from MSP430_F5xx via COM14 with default baud rate (9600 bd/s), standard invoke mechanism (shared JTAG pins), from start address 0x8000, 0x7FFE byte count, with BSL password 32x 0xFF.
```powershell
BSL430.NET.Console.exe -d COM14 -f fw.txt -o TI_TXT -m MSP430_F5xx -i SHARED_JTAG -a 8000 -z 7FFE -p FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
```
![Download](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_download.png)

When MCU is not connected or disconnected, this error is shown:
![Download](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_download_fail.png)

When incorrect password is provided, this error is shown:
![Download](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_download_fail_pw.png)

Same action, same settings, but now faster (115200 bd/s), set output firmware line length to 64 bytes and save XML log.
```powershell
BSL430.NET.Console.exe --download COM14 -f fw.txt -o TI_TXT -m MSP430_F5xx -i SHARED_JTAG -a 8000 -z 7FFE -b 115200 -x log.xml -y 64 -p FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
```
![Download](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_download2.png)

#### Erase MCU
Erase MSP430_F5xx via FTDI1 with standard invoke mechanism (shared JTAG pins).
```powershell
BSL430.NET.Console.exe -e FTDI1 -m MSP430_F5xx -i SHARED_JTAG
```
Same action, same settings, but now save also XML log.
```powershell
BSL430.NET.Console.exe --erase FTDI1 -m MSP430_F5xx -i SHARED_JTAG -x log.xml
```
![Erase](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_erase.png)

### BSL430.NET.FirmwareTools

#### Convert
Convert firmware fw_ti.txt (auto-detected) to Intel-HEX format and save it as fw_intel.hex.
```powershell
BSL430.NET.Console.exe -c fw_ti.txt -f fw_intel.hex -o INTEL_HEX
```
Same action, but now also fill data gaps with 0xFF and set custom output line length to 64 bytes.
```powershell
BSL430.NET.Console.exe --convert fw_ti.txt -f fw_intel.hex -o INTEL_HEX -l true -y 64
```
![Convert](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_convert.png)

#### Combine
Combine TI-TXT firmware fw_ti.txt (auto-detected) with INTEL-HEX firmware fw_intel.hex (auto-detected) and save it as SREC firmware fw_srec.s19.
```powershell
BSL430.NET.Console.exe -n fw_ti.txt -j fw_intel.hex -f fw_srec.s19 -o SREC
```
Same action, but now also fill data gaps with 0xFF and set custom output line length to 24 bytes.
```powershell
BSL430.NET.Console.exe --combine fw_ti.txt -j fw_intel.hex -f fw_srec.s19 -o SREC -l true -y 24
```
![Combine](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_combine.png)

#### Compare
Compare ELF firmware fw_elf.out with TI-TXT firmware fw_ti.txt.
```powershell
BSL430.NET.Console.exe -k fw_elf.out -j fw_ti.txt

BSL430.NET.Console.exe --compare fw_elf.out -j fw_ti.txt
```
![Compare](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_compare.png)
![Compare](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_compare2.png)

#### Validate
Validate ELF firmware fw_elf.out and get basic info.
```powershell
BSL430.NET.Console.exe -v fw_elf.out

BSL430.NET.Console.exe --validate fw_ti.txt
```
![Validate](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_validate.png)

#### Get BSL Password
Get BSL password (16/20/32 bytes long) to use it for correct download/upload of MSP430_F5xx MCU. Read docs for more info.
```powershell
BSL430.NET.Console.exe -r fw_intel.hex -m MSP430_F5xx

BSL430.NET.Console.exe --get-password fw_elf.out -m MSP430_F5xx
```
![Password](https://raw.githubusercontent.com/parezj/BSL430.NET/master/img/screenshots/console_get_password.png)