////////////////////////////////////////////////////////////////// ////
//// 																////
//// AES Decryption Core for FPGA									////
//// 																////
//// This file is part of the AES Decryption Core for FPGA project 	////
//// http://www.opencores.org/cores/xxx/ 							////
//// 																////
//// Description 													////
//// Implementation of  AES Decryption Core for FPGA according to 	////
//// core specification document.		 							////
//// 																////
//// To Do: 														////
//// - 																////
//// 																////
//// Author(s): 													////
//// - scheng, schengopencores@opencores.org 						////
//// 																////
//////////////////////////////////////////////////////////////////////
//// 																////
//// Copyright (C) 2009 Authors and OPENCORES.ORG 					////
//// 																////
//// This source file may be used and distributed without 			////
//// restriction provided that this copyright statement is not 		////
//// removed from the file and that any derivative work contains 	////
//// the original copyright notice and the associated disclaimer. 	////
//// 																////
//// This source file is free software; you can redistribute it 	////
//// and/or modify it under the terms of the GNU Lesser General 	////
//// Public License as published by the Free Software Foundation; 	////
//// either version 2.1 of the License, or (at your option) any 	////
//// later version. 												////
//// 																////
//// This source is distributed in the hope that it will be 		////
//// useful, but WITHOUT ANY WARRANTY; without even the implied 	////
//// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 		////
//// PURPOSE. See the GNU Lesser General Public License for more 	////
//// details. 														////
//// 																////
//// You should have received a copy of the GNU Lesser General 		////
//// Public License along with this source; if not, download it 	////
//// from http://www.opencores.org/lgpl.shtml 						////
//// 																//// ///
///////////////////////////////////////////////////////////////////
////																////
//// Key Schedule buffer											////
////																////
//// The key schedule buffer is required for decryption because		////
//// round keys are consumed in reversed order than they are		////
//// generated by the Key Expander. The KschBuffer sits between the	////
//// Key Expander and the decryptor. Round keys coming out from the	////
//// Key Expander is first stored in the KschBuffer, and later read	////
//// out in reversed order by the decryptor.						////
////																////
////////////////////////////////////////////////////////////////////////

module KschBuffer(
	// Key schedule buffer is required for decryption because round keys are consumed in
	// reversed order than they are generated by the Key Expander. The KschBuffer sits
	// between the Key Expander and the decryptor. Round keys coming out from the Key
	// Expander is first stored in the KschBuffer, and later read out in reversed order
	// by the decryptor.
	input	[0:127]	rkey_in,	// Round key from Key Expander
	input	rkey_vld_in,		// High when rkey_in has a valid round key. This occurs when
								// the Key Expander is updating the key schedule.
	
	output	[0:127]	rkey_out,	// Round key to decryptor.
	input	next_rkey,			// Assert high by decryptor to request for next round key.
	output	rkey_vld_out,		// High indicates to decryptor that a valid round key is
								// present at rkey_out.
	
	input	[0:1]	klen_sel,	// Key length select. 00->128-bit, 01->192-bit, 10->256-bit, 11->invalid
	input	clk, rst
	);
	
	(* RAM_STYLE="distributed" *) reg [127:0] lutram [15:0];
	
	reg	[3:0]	rd_addr_cnt;
	reg	[3:0]	wr_addr_cnt;
	reg			first_rkey_vld_in;
	reg	[3:0]	nr;
	
	// Do not change to always_ff. This is the coding template for inferring RAM by
	// Vivado synthesizer.
	always @(posedge clk)
		if (rkey_vld_in) lutram[wr_addr_cnt] <= rkey_in;

	assign rkey_out = lutram[rd_addr_cnt];

	// No. of rounds for various key lengths. nr is used in the read and write address
	// counter of the key schedule RAM buffer.
	always_comb
		unique case (klen_sel)
			2'b00 : nr <= 10;	// 128-bit
			2'b01 : nr <= 12;	// 192-bit
			2'b10 : nr <= 14;	// 256-bit
		endcase
		
	// Address counter used by the decryptor (reads from RAM). Counts down
	// from nr to 0.
	always_ff @(posedge clk)
	begin
		if (rst) rd_addr_cnt <= nr;
		else
		begin
			if (next_rkey)
			begin
				if (rd_addr_cnt == 0) rd_addr_cnt <= nr;
				else rd_addr_cnt--;
			end
		end
	end
	
	// Address counter for Key Expander (writes to RAM). Counts up from 0 to nr.
	always_ff @(posedge clk)
	begin
		if (rst) wr_addr_cnt <= 0;
		else
		begin
			if (rkey_vld_in)
			begin
				if (wr_addr_cnt == nr) wr_addr_cnt <= 0;
				else wr_addr_cnt++;
			end
		end
	end
	
	// Key Expander always has the priority to update key schedule in RAM. rkey_vld_out
	// is held low during key schedule update, stopping the decryptor to start decrypting
	// a new block.
	//
	// Also rkey_vld_out is held low upon reset until the first valid rkey is present.
	always_ff @(posedge clk)
	begin
		if (rst) first_rkey_vld_in <= 0;
		else if (rkey_vld_in) first_rkey_vld_in <= 1;
	end
	
	assign rkey_vld_out = ~rkey_vld_in & first_rkey_vld_in;
endmodule
