/**
  * @file NCriticalPathReportParser.h
  * @author Oleksandr Pyvovarov (APivovarov@quicklogic.com or
  aleksandr.pivovarov.84@gmail.com or
  * https://github.com/w0lek)
  * @date 2024-03-12
  * @copyright Copyright 2021 The Foedag team

  * GPL License

  * Copyright (c) 2021 The Open-Source FPGA Foundation

  * This program is free software: you can redistribute it and/or modify
  * it under the terms of the GNU General Public License as published by
  * the Free Software Foundation, either version 3 of the License, or
  * (at your option) any later version.

  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  * GNU General Public License for more details.

  * You should have received a copy of the GNU General Public License
  * along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#pragma once

#include <map>
#include <memory>
#include <string>
#include <vector>

namespace FOEDAG {

enum Role { PATH, SEGMENT, OTHER };

/**
 * Helper structure for Path Information, containing parsed data for a specific
 * path.
 */
struct PathInfo {
  int index = -1;
  std::string start;
  std::string end;
  std::string slack;
  std::string id() const { return start + ":" + end; }
  bool isValid() const { return (index != -1); }
};

struct Line {
  std::string line;
  Role role;
  bool isMultiColumn = true;
};

struct Element {
  std::vector<Line> lines;

  int currentRole() {
    if (!lines.empty()) {
      return lines[0].role;
    }
    return OTHER;
  }
};
using ElementPtr = std::shared_ptr<Element>;

struct Group {
  Group() { getNextCurrentElement(); }
  std::vector<ElementPtr> elements;
  bool isPath() { return pathInfo.isValid(); }

  void getNextCurrentElement() {
    currentElement = std::make_shared<Element>();
    elements.push_back(currentElement);
  }

  ElementPtr currentElement;
  PathInfo pathInfo;
};
using GroupPtr = std::shared_ptr<Group>;

/**
 * @brief Parser for the Critical Path Report generated by VPR.
 *
 * This parser is designed to process the Critical Path Report output generated
 * by VPR (Versatile Place and Route) tool.
 */
class NCriticalPathReportParser {
 public:
  static std::vector<GroupPtr> parseReport(
      const std::vector<std::string>& lines);
  static void parseMetaData(const std::vector<std::string>& lines,
                            std::map<int, std::pair<int, int>>& metadata);
};

}  // namespace FOEDAG
