/***************************************************************************
 tracker.cpp - Tracker
  ---------------------
 begin                : 20.02.2020
 copyright            : (C) 2020 by David Signer
 email                : david (at) opengis.ch
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "rubberbandmodel.h"
#include "tracker.h"

#include <QTimer>
#include <qgsdistancearea.h>
#include <qgsproject.h>
#include <qgssensormanager.h>

#define MAXIMUM_DISTANCE_FAILURES 20

Tracker::Tracker( QgsVectorLayer *layer )
  : mLayer( layer )
{
}

RubberbandModel *Tracker::model() const
{
  return mRubberbandModel;
}

void Tracker::setModel( RubberbandModel *model )
{
  if ( mRubberbandModel == model )
    return;

  mRubberbandModel = model;
}

QgsFeature Tracker::feature() const
{
  return mFeature;
}

void Tracker::setFeature( const QgsFeature &feature )
{
  if ( mFeature == feature )
    return;

  mFeature = feature;
}

void Tracker::trackPosition()
{
  if ( !model() || std::isnan( model()->currentCoordinate().x() ) || std::isnan( model()->currentCoordinate().y() ) )
  {
    return;
  }

  if ( !qgsDoubleNear( mMaximumDistance, 0.0 ) && mCurrentDistance > mMaximumDistance )
  {
    // Simple logic to avoid getting stuck in an infinite erroneous distance having somehow actually moved beyond the safeguard threshold
    if ( ++mMaximumDistanceFailuresCount < MAXIMUM_DISTANCE_FAILURES )
    {
      return;
    }
  }

  mSkipPositionReceived = true;
  model()->addVertex();

  mMaximumDistanceFailuresCount = 0;
  mCurrentDistance = 0.0;
  mTimeIntervalFulfilled = qgsDoubleNear( mTimeInterval, 0.0 );
  mMinimumDistanceFulfilled = qgsDoubleNear( mMinimumDistance, 0.0 );
  mSensorCaptureFulfilled = !mSensorCapture;
}

void Tracker::positionReceived()
{
  if ( mSkipPositionReceived )
  {
    // When calling model()->addVertex(), the signal we listen to for new position received is triggered, skip that one
    mSkipPositionReceived = false;
    return;
  }

  if ( !qgsDoubleNear( mMinimumDistance, 0.0 ) || !qgsDoubleNear( mMaximumDistance, 0.0 ) )
  {
    QVector<QgsPointXY> points = mRubberbandModel->flatPointSequence( QgsProject::instance()->crs() );

    auto pointIt = points.constEnd() - 1;

    QVector<QgsPointXY> flatPoints;

    flatPoints << *pointIt;
    pointIt--;
    flatPoints << *pointIt;

    QgsDistanceArea distanceArea;
    distanceArea.setEllipsoid( QgsProject::instance()->ellipsoid() );
    distanceArea.setSourceCrs( QgsProject::instance()->crs(), QgsProject::instance()->transformContext() );
    mCurrentDistance = distanceArea.measureLine( flatPoints );
  }

  if ( !qgsDoubleNear( mMinimumDistance, 0.0 ) )
  {
    if ( mCurrentDistance > mMinimumDistance )
    {
      mMinimumDistanceFulfilled = true;
    }
  }
  else
  {
    mMinimumDistanceFulfilled = true;
  }

  if ( ( !mConjunction && mMinimumDistanceFulfilled ) || ( mMinimumDistanceFulfilled && mTimeIntervalFulfilled && mSensorCaptureFulfilled ) )
  {
    trackPosition();
  }
}

void Tracker::timeReceived()
{
  mTimeIntervalFulfilled = true;

  if ( !mConjunction || ( mMinimumDistanceFulfilled && mSensorCaptureFulfilled ) )
  {
    trackPosition();
  }
}

void Tracker::sensorDataReceived()
{
  mSensorCaptureFulfilled = true;

  if ( !mConjunction || ( mMinimumDistanceFulfilled && mTimeIntervalFulfilled ) )
  {
    trackPosition();
  }
}

void Tracker::start()
{
  mIsActive = true;
  emit isActiveChanged();

  if ( mTimeInterval > 0 )
  {
    connect( &mTimer, &QTimer::timeout, this, &Tracker::timeReceived );
    mTimer.start( mTimeInterval * 1000 );
  }
  else
  {
    mTimeIntervalFulfilled = true;
  }
  if ( mMinimumDistance > 0 || ( qgsDoubleNear( mTimeInterval, 0.0 ) && !mSensorCapture ) )
  {
    connect( mRubberbandModel, &RubberbandModel::currentCoordinateChanged, this, &Tracker::positionReceived );
  }
  else
  {
    mMinimumDistanceFulfilled = true;
  }
  if ( mSensorCapture )
  {
    connect( QgsProject::instance()->sensorManager(), &QgsSensorManager::sensorDataCaptured, this, &Tracker::sensorDataReceived );
  }
  else
  {
    mSensorCaptureFulfilled = true;
  }

  //set the start time
  setStartPositionTimestamp( QDateTime::currentDateTime() );

  if ( mMeasureType == Tracker::SecondsSinceStart )
  {
    model()->setMeasureValue( 0 );
  }

  mSkipPositionReceived = false;
  mMaximumDistanceFailuresCount = 0;
  mCurrentDistance = mMaximumDistance;

  //track first position
  trackPosition();
}

void Tracker::stop()
{
  //track last position
  trackPosition();

  mIsActive = false;
  emit isActiveChanged();

  if ( mTimeInterval > 0 )
  {
    mTimer.stop();
    disconnect( &mTimer, &QTimer::timeout, this, &Tracker::trackPosition );
  }
  if ( mMinimumDistance > 0 || ( qgsDoubleNear( mTimeInterval, 0.0 ) && !mSensorCapture ) )
  {
    disconnect( mRubberbandModel, &RubberbandModel::currentCoordinateChanged, this, &Tracker::positionReceived );
  }
  if ( mSensorCapture )
  {
    disconnect( QgsProject::instance()->sensorManager(), &QgsSensorManager::sensorDataCaptured, this, &Tracker::sensorDataReceived );
  }
}
