/***************************************************************************
  activelayerfeatureslocatorfilter.cpp

 ---------------------
 begin                : 30.08.2023
 copyright            : (C) 2023 by Mathieu Pellerin
 email                : mathieu@opengis.ch
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "activelayerfeatureslocatorfilter.h"
#include "featurelistextentcontroller.h"
#include "locatormodelsuperbridge.h"
#include "qgsquickmapsettings.h"

#include <QAction>
#include <qgsexpressioncontextutils.h>
#include <qgsfeedback.h>
#include <qgsmaplayermodel.h>
#include <qgsproject.h>
#include <qgsvectorlayer.h>

#include <math.h>


ActiveLayerFeaturesLocatorFilter::ActiveLayerFeaturesLocatorFilter( LocatorModelSuperBridge *locatorBridge, QObject *parent )
  : QgsLocatorFilter( parent )
  , mLocatorBridge( locatorBridge )
{
  setUseWithoutPrefix( false );
}

ActiveLayerFeaturesLocatorFilter *ActiveLayerFeaturesLocatorFilter::clone() const
{
  return new ActiveLayerFeaturesLocatorFilter( mLocatorBridge );
}

QString ActiveLayerFeaturesLocatorFilter::fieldRestriction( QString &searchString, bool *isRestricting )
{
  QString _fieldRestriction;
  searchString = searchString.trimmed();
  if ( isRestricting )
  {
    *isRestricting = searchString.startsWith( '@' );
  }
  if ( searchString.startsWith( '@' ) )
  {
    _fieldRestriction = searchString.left( std::min( searchString.indexOf( ' ' ), searchString.length() ) ).remove( 0, 1 );
    searchString = searchString.mid( _fieldRestriction.length() + 2 );
  }
  return _fieldRestriction;
}

QStringList ActiveLayerFeaturesLocatorFilter::prepare( const QString &string, const QgsLocatorContext &locatorContext )
{
  // Normally skip very short search strings, unless when specifically searching using this filter or try to match fields
  if ( string.length() < 3 && !locatorContext.usingPrefix && !string.startsWith( '@' ) )
    return QStringList();

  QgsVectorLayer *layer = qobject_cast<QgsVectorLayer *>( mLocatorBridge->activeLayer() );
  if ( !layer )
    return QStringList();

  mLayerIsSpatial = layer->isSpatial();
  mDispExpression = QgsExpression( layer->displayExpression() );
  mContext.appendScopes( QgsExpressionContextUtils::globalProjectLayerScopes( layer ) );
  mDispExpression.prepare( &mContext );

  // determine if search is restricted to a specific field
  QString searchString = string;
  bool isRestricting = false;
  QString _fieldRestriction = fieldRestriction( searchString, &isRestricting );
  bool allowNumeric = false;
  double numericalValue = searchString.toDouble( &allowNumeric );

  // search in display expression if no field restriction
  if ( !isRestricting )
  {
    QgsFeatureRequest req;
    req.setSubsetOfAttributes( qgis::setToList( mDispExpression.referencedAttributeIndexes( layer->fields() ) ) );
    if ( !mDispExpression.needsGeometry() )
    {
#if _QGIS_VERSION_INT >= 33500
      req.setFlags( Qgis::FeatureRequestFlag::NoGeometry );
#else
      req.setFlags( QgsFeatureRequest::NoGeometry );
#endif
    }
    QString enhancedSearch = searchString;
    enhancedSearch.replace( ' ', '%' );
    req.setFilterExpression( QStringLiteral( "%1 ILIKE '%%2%'" )
                               .arg( layer->displayExpression(), enhancedSearch ) );
    req.setLimit( mMaxTotalResults );
    mDisplayTitleIterator = layer->getFeatures( req );
  }
  else
  {
    mDisplayTitleIterator = QgsFeatureIterator();
  }

  // build up request expression
  QStringList expressionParts;
  QStringList completionList;
  const QgsFields fields = layer->fields();
  QgsAttributeList subsetOfAttributes = qgis::setToList( mDispExpression.referencedAttributeIndexes( layer->fields() ) );
  for ( const QgsField &field : fields )
  {
#if _QGIS_VERSION_INT >= 33300
    if ( field.configurationFlags().testFlag( Qgis::FieldConfigurationFlag::NotSearchable ) )
#else
    if ( field.configurationFlags().testFlag( QgsField::ConfigurationFlag::NotSearchable ) )
#endif
    {
      continue;
    }

    if ( isRestricting )
    {
      if ( !field.name().startsWith( _fieldRestriction ) )
      {
        continue;
      }

      int index = layer->fields().indexFromName( field.name() );
      if ( !subsetOfAttributes.contains( index ) )
      {
        subsetOfAttributes << index;
      }

      // if we are trying to find a field (and not searching anything yet)
      // keep the list of matching fields to display them as results
      if ( searchString.isEmpty() && _fieldRestriction != field.name() )
      {
        mFieldsCompletion << field.name();
      }
    }
    else if ( searchString.isEmpty() )
    {
      mFieldsCompletion << field.name();
    }

    // the completion list (returned by the current method) is used by the locator line edit directly
    completionList.append( QStringLiteral( "@%1 " ).arg( field.name() ) );

    if ( field.type() == QMetaType::QString )
    {
      expressionParts << QStringLiteral( "%1 ILIKE '%%2%'" ).arg( QgsExpression::quotedColumnRef( field.name() ), searchString );
    }
    else if ( allowNumeric && field.isNumeric() )
    {
      expressionParts << QStringLiteral( "%1 = %2" ).arg( QgsExpression::quotedColumnRef( field.name() ), QString::number( numericalValue, 'g', 17 ) );
    }
  }

  QString expression = QStringLiteral( "(%1)" ).arg( expressionParts.join( QLatin1String( " ) OR ( " ) ) );

  QgsFeatureRequest req;
  if ( !mDispExpression.needsGeometry() )
  {
#if _QGIS_VERSION_INT >= 33500
    req.setFlags( Qgis::FeatureRequestFlag::NoGeometry );
#else
    req.setFlags( QgsFeatureRequest::NoGeometry );
#endif
  }
  req.setFilterExpression( expression );
  if ( isRestricting )
  {
    req.setSubsetOfAttributes( subsetOfAttributes );
  }

  req.setLimit( mMaxTotalResults );
  mFieldIterator = layer->getFeatures( req );

  mLayerId = layer->id();
  mLayerName = layer->name();
  mAttributeAliases.clear();
  for ( int idx = 0; idx < layer->fields().size(); ++idx )
  {
    mAttributeAliases.append( layer->attributeDisplayName( idx ) );
  }

  return completionList;
}

void ActiveLayerFeaturesLocatorFilter::fetchResults( const QString &string, const QgsLocatorContext &, QgsFeedback *feedback )
{
  QgsFeatureIds featuresFound;
  QgsFeature f;
  QString searchString = string;
  fieldRestriction( searchString );

  if ( searchString.trimmed().isEmpty() )
  {
    // propose available fields for restriction
    for ( const QString &field : std::as_const( mFieldsCompletion ) )
    {
      QgsLocatorResult result;
      result.displayString = QStringLiteral( "@%1" ).arg( field );
      result.group = mLayerName;
      result.description = tr( "Limit the search to the field '%1'" ).arg( field );
#if _QGIS_VERSION_INT >= 33300
      result.setUserData( QVariantMap( { { QStringLiteral( "type" ), QVariant::fromValue( ResultType::FieldRestriction ) },
                                         { QStringLiteral( "search_text" ), QStringLiteral( "%1 @%2 " ).arg( prefix(), field ) } } ) );
#else
      result.userData = QVariantMap( { { QStringLiteral( "type" ), QVariant::fromValue( ResultType::FieldRestriction ) },
                                       { QStringLiteral( "search_text" ), QStringLiteral( "%1 @%2 " ).arg( prefix(), field ) } } );
#endif
      result.score = 1;
      emit resultFetched( result );
    }
    return;
  }

  // search in display title
  if ( mDisplayTitleIterator.isValid() )
  {
    while ( mDisplayTitleIterator.nextFeature( f ) )
    {
      if ( feedback->isCanceled() )
        return;

      mContext.setFeature( f );

      QgsLocatorResult result;
      result.displayString = mDispExpression.evaluate( &mContext ).toString();
      result.group = mLayerName;

#if _QGIS_VERSION_INT >= 33300
      result.setUserData( QVariantList() << f.id() << mLayerId );
#else
      result.userData = QVariantList() << f.id() << mLayerId;
#endif
      result.score = static_cast<double>( searchString.length() ) / result.displayString.size();
      result.actions << QgsLocatorResult::ResultAction( OpenForm, tr( "Open form" ), QStringLiteral( "qrc:/themes/qfield/nodpi/ic_baseline-list_white_24dp.svg" ) );
      if ( mLayerIsSpatial )
      {
        result.actions << QgsLocatorResult::ResultAction( Navigation, tr( "Set feature as destination" ), QStringLiteral( "qrc:/themes/qfield/nodpi/ic_navigation_flag_purple_24dp.svg" ) );
      }

      emit resultFetched( result );

      featuresFound << f.id();
      if ( featuresFound.count() >= mMaxTotalResults )
        break;
    }
  }

  // search in fields
  while ( mFieldIterator.nextFeature( f ) )
  {
    if ( feedback->isCanceled() )
      return;

    // do not display twice the same feature
    if ( featuresFound.contains( f.id() ) )
      continue;

    QgsLocatorResult result;

    mContext.setFeature( f );

    // find matching field content
    int idx = 0;
    const QgsAttributes attributes = f.attributes();
    for ( const QVariant &var : attributes )
    {
      QString attrString = var.toString();
      if ( attrString.contains( searchString, Qt::CaseInsensitive ) )
      {
        if ( idx < mAttributeAliases.count() )
        {
          result.displayString = QStringLiteral( "%1 (%2)" ).arg( attrString, mAttributeAliases[idx] );
        }
        else
        {
          result.displayString = attrString;
        }
        break;
      }
      idx++;
    }
    if ( result.displayString.isEmpty() )
      continue; //not sure how this result slipped through...
    result.group = mLayerName;
    result.description = mDispExpression.evaluate( &mContext ).toString();
#if _QGIS_VERSION_INT >= 33300
    result.setUserData( QVariantList() << f.id() << mLayerId );
#else
    result.userData = QVariantList() << f.id() << mLayerId;
#endif
    result.score = static_cast<double>( searchString.length() ) / result.displayString.size();
    result.actions << QgsLocatorResult::ResultAction( OpenForm, tr( "Open form" ), QStringLiteral( "qrc:/themes/qfield/nodpi/ic_baseline-list_white_24dp.svg" ) );
    if ( mLayerIsSpatial )
    {
      result.actions << QgsLocatorResult::ResultAction( Navigation, tr( "Set feature as destination" ), QStringLiteral( "qrc:/themes/qfield/nodpi/ic_navigation_flag_purple_24dp.svg" ) );
    }

    emit resultFetched( result );

    featuresFound << f.id();
    if ( featuresFound.count() >= mMaxTotalResults )
      break;
  }
}

void ActiveLayerFeaturesLocatorFilter::triggerResult( const QgsLocatorResult &result )
{
  triggerResultFromAction( result, Normal );
}

void ActiveLayerFeaturesLocatorFilter::triggerResultFromAction( const QgsLocatorResult &result, const int actionId )
{
#if _QGIS_VERSION_INT >= 33601
  QVariantMap data = result.userData().toMap();
#else
  QVariantMap data = result.getUserData().toMap();
#endif
  switch ( data.value( QStringLiteral( "type" ) ).value<ResultType>() )
  {
    case ResultType::FieldRestriction:
    {
      QTimer::singleShot( 100, [=] { emit mLocatorBridge->requestSearchTextChange( data.value( "search_text" ).toString() ); } );
      break;
    }

    case ResultType::Feature:
    {
#if _QGIS_VERSION_INT >= 33601
      QVariantList dataList = result.userData().toList();
#else
      QVariantList dataList = result.getUserData().toList();
#endif
      QgsFeatureId fid = dataList.at( 0 ).toLongLong();
      QString layerId = dataList.at( 1 ).toString();
      QgsVectorLayer *layer = qobject_cast<QgsVectorLayer *>( QgsProject::instance()->mapLayer( layerId ) );
      if ( !layer )
        return;

      QgsFeature feature;
      QgsFeatureRequest featureRequest = QgsFeatureRequest().setFilterFid( fid );

      switch ( actionId )
      {
        case OpenForm:
        {
          QMap<QgsVectorLayer *, QgsFeatureRequest> requests;
          requests.insert( layer, featureRequest );
          mLocatorBridge->featureListController()->model()->setFeatures( requests );
          mLocatorBridge->featureListController()->selection()->setFocusedItem( 0 );
          mLocatorBridge->featureListController()->requestFeatureFormState();
          break;
        }

        case Navigation:
        {
          if ( !mLocatorBridge->navigation() )
            return;

          QgsFeatureIterator it = layer->getFeatures( featureRequest );
          it.nextFeature( feature );
          if ( feature.hasGeometry() )
          {
            mLocatorBridge->navigation()->setDestinationFeature( feature, layer );
          }
          else
          {
            mLocatorBridge->emitMessage( tr( "Feature has no geometry" ) );
          }
          break;
        }

        case Normal:
        {
          QgsFeatureIterator it = layer->getFeatures( featureRequest.setNoAttributes() );
          it.nextFeature( feature );
          const QgsGeometry geom = feature.geometry();
          if ( geom.isNull() || geom.constGet()->isEmpty() )
          {
            mLocatorBridge->emitMessage( tr( "Feature has no geometry" ) );
            return;
          }
          QgsRectangle r = mLocatorBridge->mapSettings()->mapSettings().layerExtentToOutputExtent( layer, geom.boundingBox() );

          // zoom in if point cannot be distinguished from others
          // code taken from QgsMapCanvas::zoomToSelected
          if ( !mLocatorBridge->keepScale() )
          {
            if ( layer->geometryType() == Qgis::GeometryType::Point && r.isEmpty() )
            {
              int scaleFactor = 5;
              const QgsPointXY center = mLocatorBridge->mapSettings()->mapSettings().mapToLayerCoordinates( layer, r.center() );
              const QgsRectangle extentRect = mLocatorBridge->mapSettings()->mapSettings().mapToLayerCoordinates( layer, mLocatorBridge->mapSettings()->visibleExtent() ).scaled( 1.0 / scaleFactor, &center );
              const QgsFeatureRequest pointRequest = QgsFeatureRequest().setFilterRect( extentRect ).setLimit( 1000 ).setNoAttributes();
              QgsFeatureIterator fit = layer->getFeatures( pointRequest );
              QgsFeature pointFeature;
              QgsPointXY closestPoint;
              double closestSquaredDistance = pow( extentRect.width() + extentRect.height(), 2.0 );
              bool pointFound = false;
              while ( fit.nextFeature( pointFeature ) )
              {
                QgsPointXY point = pointFeature.geometry().asPoint();
                double sqrDist = point.sqrDist( center );
                if ( sqrDist > closestSquaredDistance || sqrDist < 4 * std::numeric_limits<double>::epsilon() )
                  continue;
                pointFound = true;
                closestPoint = point;
                closestSquaredDistance = sqrDist;
              }
              if ( pointFound )
              {
                // combine selected point with closest point and scale this rect
                r.combineExtentWith( mLocatorBridge->mapSettings()->mapSettings().layerToMapCoordinates( layer, closestPoint ) );
                const QgsPointXY rCenter = r.center();
                r.scale( scaleFactor, &rCenter );
              }
            }
            else if ( !r.isEmpty() )
            {
              r.scale( 1.25 );
            }
          }

          if ( r.isEmpty() || mLocatorBridge->keepScale() )
          {
            mLocatorBridge->mapSettings()->setCenter( QgsPoint( r.center() ), true );
          }
          else
          {
            mLocatorBridge->mapSettings()->setExtent( r, true );
          }

          mLocatorBridge->locatorHighlightGeometry()->setProperty( "qgsGeometry", geom );
          mLocatorBridge->locatorHighlightGeometry()->setProperty( "crs", layer->crs() );
          break;
        }
      }
      break;
    }
  }
}
