/*
 * Copyright (C) 2024 OpenAni and contributors.
 *
 * 此源代码的使用受 GNU AFFERO GENERAL PUBLIC LICENSE version 3 许可证的约束, 可以在以下链接找到该许可证.
 * Use of this source code is governed by the GNU AGPLv3 license, which can be found at the following link.
 *
 * https://github.com/open-ani/ani/blob/main/LICENSE
 */

package me.him188.ani.app.data.repository

import androidx.compose.ui.unit.Dp
import androidx.datastore.core.DataStore
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.KSerializer
import kotlinx.serialization.Serializable
import kotlinx.serialization.builtins.serializer
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder

sealed class WindowStateRepository : Repository() {
    abstract val flow: Flow<SavedWindowState?>
    abstract suspend fun update(states: SavedWindowState)
}

@Serializable
data class SavedWindowState(
    val x: @Serializable(DpSerializer::class) Dp, // dp
    val y: @Serializable(DpSerializer::class) Dp, // dp
    val width: @Serializable(DpSerializer::class) Dp, // dp
    val height: @Serializable(DpSerializer::class) Dp, // dp
)

private object DpSerializer : KSerializer<Dp> {
    override val descriptor = Float.serializer().descriptor

    override fun serialize(encoder: Encoder, value: Dp) {
        encoder.encodeFloat(value.value)
    }

    override fun deserialize(decoder: Decoder): Dp = Dp(decoder.decodeFloat())
}

class WindowStateRepositoryImpl(
    private val store: DataStore<SavedWindowState?>,
) : WindowStateRepository() {
    override val flow: Flow<SavedWindowState?> = store.data

    override suspend fun update(states: SavedWindowState) {
        store.updateData {
            states
        }
    }

}
