pub use self::key::*;

use crate::errno::EINVAL;
use crate::process::VThread;
use crate::syscalls::{SysErr, SysIn, SysOut, Syscalls};
use crate::ucred::Ucred;
use crate::{info, warn};
use std::fmt::{Display, Formatter};
use std::num::NonZeroI32;
use std::ops::Index;
use std::ptr::read;
use std::sync::Arc;
use thiserror::Error;

mod key;

/// An implementation of PS4 registry manager.
pub struct RegMgr {}

impl RegMgr {
    pub fn new(sys: &mut Syscalls) -> Arc<Self> {
        let mgr = Arc::new(Self {});

        sys.register(532, &mgr, Self::sys_regmgr_call);
        sys.register(605, &mgr, Self::sys_workaround8849);

        mgr
    }

    fn sys_regmgr_call(self: &Arc<Self>, td: &Arc<VThread>, i: &SysIn) -> Result<SysOut, SysErr> {
        // Get arguments.
        let op: u32 = i.args[0].try_into().unwrap();
        let buf: *mut i32 = i.args[2].into();
        let req: *const u8 = i.args[3].into();
        let reqlen: usize = i.args[4].into();

        // TODO: Check the result of priv_check(td, 682).
        if buf.is_null() {
            todo!("regmgr_call with buf = null");
        }

        if req.is_null() {
            todo!("regmgr_call with req = null");
        }

        if reqlen > 2048 {
            todo!("regmgr_call with reqlen > 2048");
        }

        // Execute the operation.
        let r = match op {
            0x18 => {
                let v1 = unsafe { read::<u64>(req as _) };
                let v2 = unsafe { read::<u32>(req.add(8) as _) };
                let value = unsafe { read::<i32>(req.add(12) as _) };

                info!(
                    "Attempting to set registry with v1: {}, v2: {}, value: {}.",
                    v1, v2, value
                );

                self.decode_key(v1, v2, td.cred(), 2).and_then(|k| {
                    info!("Setting registry key {} to value {}.", k, value);
                    self.set_int(k, value)
                })
            }
            0x19 => {
                let v1 = unsafe { read::<u64>(req as _) };
                let v2 = unsafe { read::<u32>(req.add(8) as _) };

                self.decode_key(v1, v2, td.cred(), 1)
                    .and_then(|k| todo!("regmgr_call({op}) with matched key = {k}"))
            }
            0x27 | 0x40.. => Err(RegError::V800d0219),
            v => todo!("regmgr_call({v})"),
        };

        // Write the result.
        unsafe {
            *buf = match r {
                Ok(v) => v,
                Err(e) => {
                    warn!(e, "regmgr_call({op}) failed");
                    e.code()
                }
            }
        };

        Ok(SysOut::ZERO)
    }

    fn decode_key(&self, v1: u64, v2: u32, cred: &Ucred, v3: u32) -> Result<RegKey, RegError> {
        // Check checksum.
        let a = (v1 & 0xff) as i32;
        let b = ((v1 >> 8) & 0xff) as i32;
        let c = ((v1 >> 16) & 0xff) as i32;
        let d = ((v1 >> 24) & 0xff) as i32;
        let e = ((v1 >> 32) & 0xff) as i32;
        let f = ((v1 >> 40) & 0xff) as i32;
        let g = (a + b + c + d + e * f) as u16;
        let h = (v1 >> 48) as u16;

        if g != h {
            return Err(RegError::ChecksumMismatched(h, g));
        }

        // Decrypt the request.
        let i = (f ^ 0x6b) as usize;

        if i > 12 {
            todo!("regmgr_call with multiplier ^ 0x6b > 12");
        }

        let x = e ^ SBOX1[ReverseIndex(i)] as i32;
        let sbox = if x == 0x19 {
            &SBOX2
        } else {
            if x != 0x72 {
                todo!("regmgr_call with x != 0x19 && x != 0x72: {:#x}", x)
            } else {
                &SBOX1
            }
        };

        // Construct the key.
        let mut key: u32 = (a ^ (sbox[i + 3] as i32)) as u32;

        key |= ((c ^ (sbox[i + 2] as i32)) as u32) << 8;
        key |= ((d ^ (sbox[i + 1] as i32)) as u32) << 16;
        key |= ((b ^ (sbox[i] as i32)) as u32) << 24;

        // Lookup the entry.
        let key = RegKey::new(key);
        let entry = self.lookup(key).ok_or(RegError::NotFound(key))?;
        let web = if cred.is_libkernel_web() || cred.is_webprocess_webapp_or_webmas() {
            1
        } else {
            0
        };

        if v3 == 1 {
            if cred.is_nongame() && entry.unk3 & 0x10 == 0 {
                return Err(RegError::V800d0216);
            }
            if entry.unk3 & 1 == 0 {
                return Err(RegError::V800d0214);
            }
            if (!(entry.unk3 >> 12) & web) != 0 {
                return Err(RegError::V800d021f);
            }
        }

        if cred.is_nongame() {
            todo!("decode regmgr_call request with non-game cred");
        } else if (entry.unk3 & 2) == 0 {
            Err(RegError::V800d0214)
        } else if (!(entry.unk3 >> 13)) & web != 0 {
            Err(RegError::V800d021f)
        } else {
            match entry.unk2 {
                8 => todo!("decode regmgr_call request with unk2 = 8"),
                16 => todo!("decode regmgr_call request with unk2 = 16"),
                _ => {
                    if v2 != 0 {
                        Err(RegError::V800d0206)
                    } else {
                        Ok(key)
                    }
                }
            }
        }
    }

    /// See `sceRegMgrGetInt` on the PS4 for a reference.
    fn get_int(&self, key: RegKey, out: &mut i32) -> Result<i32, RegError> {
        let mut buf = [0u8; 4];

        if let Err(e) = self.check_param(key, 0, buf.len()) {
            todo!("sceRegMgrGetInt with regMgrComCheckParam({key}, 0, 4) = Err({e})");
        }

        match self.get_value(key, &mut buf) {
            Ok(v) => {
                *out = i32::from_le_bytes(buf);
                Ok(v)
            }
            Err(e) => todo!("sceRegMgrGetInt({key}) with regMgrComSetReg() = {e}"),
        }
    }

    /// See `sceRegMgrSetInt` on the PS4 for a reference.
    fn set_int(&self, key: RegKey, value: i32) -> Result<i32, RegError> {
        let value = value.to_le_bytes();

        if let Err(e) = self.check_param(key, 0, value.len()) {
            todo!("sceRegMgrSetInt with regMgrComCheckParam({key}, 0, 4) = Err({e})");
        }

        match self.set_value(key, &value) {
            Ok(v) => Ok(v),
            Err(e) => todo!("sceRegMgrSetInt({key}) with regMgrComSetReg() = {e}"),
        }
    }

    /// See `regMgrComSetReg` on the PS4 for a reference.
    fn set_value(&self, key: RegKey, value: &[u8]) -> Result<i32, RegError> {
        'l1: for e in &UNK_ENTRIES1 {
            for i in 0..e.unk2 {
                if (i << (e.unk1 & 0xff)) + e.key.value() == key.value() {
                    if (e.unk3 & 1) == 0 {
                        break 'l1;
                    } else {
                        return Ok(0);
                    }
                }
            }
        }

        // Lookup a target entry.
        let entry = self.lookup(key).ok_or(RegError::NotFound(key))?;

        if value.len() > entry.len {
            return Err(RegError::V800d0208);
        }

        if entry.len < 5 {
            let mut buf = [0u8; 4];
            self.get_value(key, &mut buf[..entry.len])?;

            if entry.unk4 == 1 {
                todo!("regMgrComSetReg({key}) with unk4 = 1");
            } else {
                for i in 0..value.len() {
                    if buf[i] != value[i] {
                        todo!("regMgrComSetReg({key}) with new value");
                    }
                }
            }
        } else {
            todo!("regMgrComSetReg({key}) with len >= 5");
        }

        Ok(0)
    }

    /// See `regMgrComGetReg` on the PS4 for a reference.
    fn get_value(&self, key: RegKey, buf: &mut [u8]) -> Result<i32, RegError> {
        // Special cases for specific keys.
        match key {
            RegKey::REGISTRY_RECOVER
            | RegKey::SYSTEM_UPDATE_MODE
            | RegKey::SYSTEM_POWER_SHUTDOWN_STATUS
            | RegKey::SYSTEM_SPECIFIC_IDU_MODE
            | RegKey::SYSTEM_SPECIFIC_SHOW_MODE
            | RegKey::SYSTEM_SPECIFIC_ARCADE_MODE
            | RegKey::DEVENV_TOOL_BOOT_PARAM => {
                todo!("regMgrComGetReg({key})");
            }
            _ => {}
        }

        // Lookup the entry.
        let entry = match self.lookup(key) {
            Some(v) => v,
            None => return Ok(0),
        };

        if entry.len > buf.len() {
            Err(RegError::V800d0208)
        } else if entry.unk5 == 2 {
            match entry.unk6 {
                16 | 2 => todo!("regMgrComGetReg({key}) with unk6 = 16 | 2"),
                1 => {
                    if entry.unk4 == 1 {
                        todo!("regMgrComGetReg({key}) with unk6 = 1 and unk4 = 1");
                    } else {
                        let data = entry.value.to_le_bytes();

                        buf[..entry.len].copy_from_slice(&data[..entry.len]);

                        return Ok(0);
                    }
                }
                v => todo!("regMgrComGetReg({key}) with unk6 = {v}"),
            }
        } else {
            todo!("regMgrComGetReg({key}) with unk5 != 2");
        }
    }

    /// See `regMgrComCheckParam` on the PS4 for a reference.
    fn check_param(&self, key: RegKey, v1: u16, len: usize) -> Result<&'static RegEntry, RegError> {
        let entry = match self.lookup(key) {
            Some(v) => v,
            None => return Ok(&ENTRIES[0]),
        };

        if entry.unk4 != v1 {
            return Err(RegError::V800d0207);
        }

        if v1 != 2 {
            if v1 == 1 {
                todo!("regMgrComCheckParam with v1 = 1");
            }

            if v1 != 0 {
                return Err(RegError::V800d0207);
            }
        }

        if entry.len == len {
            Ok(entry)
        } else {
            Err(RegError::V800d0208)
        }
    }

    fn lookup(&self, key: RegKey) -> Option<&'static RegEntry> {
        for e in &ENTRIES {
            if e.key == key {
                return Some(e);
            } else if e.unk1 >= 2 {
                for v in 1..e.unk1 {
                    if ((v << (e.unk2 & 0x1f)) + e.key.value()) == key.value() {
                        return Some(e);
                    }
                }
            }
        }

        None
    }

    fn sys_workaround8849(self: &Arc<Self>, _: &Arc<VThread>, i: &SysIn) -> Result<SysOut, SysErr> {
        let key = {
            let arg: usize = i.args[0].into();
            let key: u32 = arg.try_into().unwrap();

            RegKey::new(key)
        };

        match key {
            RegKey::NP_DEBUG
            | RegKey::BROWSER_DEBUG_NOTIFICATION
            | RegKey::DEVENV_TOOL_TRC_NOTIFY
            | RegKey::DEVENV_TOOL_USE_DEFAULT_LIB
            | RegKey::DEVENV_TOOL_SYS_PRX_PRELOAD => {
                let mut out = 0;
                let ret = self.get_int(key, &mut out).unwrap();

                match NonZeroI32::new(ret) {
                    None => Ok(ret.into()),
                    Some(v) => Err(SysErr::Raw(v)),
                }
            }
            _ => Err(SysErr::Raw(EINVAL)),
        }
    }
}

/// Contains information for a registry entry.
pub struct RegEntry {
    pub key: RegKey,
    pub unk4: u16,
    pub len: usize,
    pub unk8: u16,
    pub unk6: u8,
    pub unk5: u8,
    pub value: u32,
    pub unk9: u16,
    pub unk1: u32,
    pub unk2: u8,
    pub unk3: u32,
}

/// Represents an error when the operation on a registry is failed.
#[derive(Debug, Error)]
pub enum RegError {
    NotFound(RegKey),
    ChecksumMismatched(u16, u16),
    V800d0206,
    V800d0207,
    V800d0208,
    V800d0214,
    V800d0216,
    V800d0219,
    V800d021f,
}

impl RegError {
    pub fn code(&self) -> i32 {
        match self {
            Self::NotFound(_) => 0x800d0203u32 as i32,
            Self::ChecksumMismatched(_, _) => 0x800d0204u32 as i32,
            Self::V800d0206 => 0x800d0206u32 as i32,
            Self::V800d0207 => 0x800d0207u32 as i32,
            Self::V800d0208 => 0x800d0208u32 as i32,
            Self::V800d0214 => 0x800d0214u32 as i32,
            Self::V800d0216 => 0x800d0216u32 as i32,
            Self::V800d0219 => 0x800d0219u32 as i32,
            Self::V800d021f => 0x800d021fu32 as i32,
        }
    }
}

impl Display for RegError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
            Self::NotFound(k) => write!(f, "entry {k} not found"),
            Self::ChecksumMismatched(e, g) => {
                write!(f, "checksum mismatched (expected {e:#x}, got {g:#x})")
            }
            v => write!(f, "{:#x}", v.code()),
        }
    }
}

struct RegUnk6 {
    key: RegKey,
    unk1: u32,
    unk2: u32,
    unk3: u32,
    unk4: u32,
}

const SBOX1: [u8; 16] = [
    0x68, 0xe8, 0x98, 0x03, 0x9a, 0x70, 0x23, 0x5a, 0x63, 0xee, 0xf5, 0x7b, 0xff, 0xa4, 0x4c, 0x8c,
];

const SBOX2: [u8; 16] = [
    0x14, 0xee, 0xde, 0xe1, 0x80, 0xac, 0xf3, 0x78, 0x47, 0x43, 0xdb, 0x40, 0x93, 0xdd, 0xb1, 0x34,
];

pub struct ReverseIndex(pub usize);

impl<T> Index<ReverseIndex> for [T] {
    type Output = T;

    fn index(&self, idx: ReverseIndex) -> &T {
        let len = self.len();
        &self[len - 1 - idx.0]
    }
}

const UNK_ENTRIES1: [RegUnk6; 3] = [
    RegUnk6 {
        key: RegKey::NET_WIFI_FREQ_BAND,
        unk1: 0,
        unk2: 1,
        unk3: 1,
        unk4: 0,
    },
    RegUnk6 {
        key: RegKey::AUDIOOUT_CONNECTOR_TYPE,
        unk1: 0,
        unk2: 1,
        unk3: 2,
        unk4: 0x40,
    },
    RegUnk6 {
        key: RegKey::AUDIOOUT_CODEC,
        unk1: 0,
        unk2: 1,
        unk3: 2,
        unk4: 0x40,
    },
];

const ENTRIES: [RegEntry; 560] = [
    RegEntry {
        key: RegKey::new(0x1010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x0,
        unk6: 1,
        unk5: 2,
        value: 0x900001,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x1020000),
        unk4: 0x0,
        len: 4,
        unk8: 0x0,
        unk6: 1,
        unk5: 2,
        value: 0x400060,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x1030000),
        unk4: 0x0,
        len: 4,
        unk8: 0x0,
        unk6: 1,
        unk5: 2,
        value: 0x2000d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x1040000),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x1050000),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 360,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x1060000),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0x900000,
        unk9: 380,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x1070000),
        unk4: 0x0,
        len: 4,
        unk8: 0x210,
        unk6: 1,
        unk5: 2,
        value: 0x7e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x1080000),
        unk4: 0x0,
        len: 4,
        unk8: 0x220,
        unk6: 1,
        unk5: 2,
        value: 0x700000,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x1400000),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x1800100),
        unk4: 0x0,
        len: 4,
        unk8: 0x200,
        unk6: 16,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0xa,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x1800200),
        unk4: 0x0,
        len: 4,
        unk8: 0x210,
        unk6: 16,
        unk5: 2,
        value: 0x28,
        unk9: 0,
        unk1: 0xa,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 100,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x2020000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 104,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x1355,
    },
    RegEntry {
        key: RegKey::new(0x2030000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2040000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2050000),
        unk4: 0x1,
        len: 65,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x50,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x2060000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x2070000),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2080000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x20a0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x11,
    },
    RegEntry {
        key: RegKey::new(0x20b0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x6020,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 64,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x1355,
    },
    RegEntry {
        key: RegKey::new(0x20c0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 372,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x20e0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x11,
    },
    RegEntry {
        key: RegKey::new(0x20f0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 1,
        unk5: 2,
        value: 0x4,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2100000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x2110000),
        unk4: 0x0,
        len: 4,
        unk8: 0x220,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 396,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2120000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x2800200),
        unk4: 0x0,
        len: 4,
        unk8: 0x210,
        unk6: 1,
        unk5: 2,
        value: 0x2501000,
        unk9: 128,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2800300),
        unk4: 0x0,
        len: 4,
        unk8: 0x30,
        unk6: 1,
        unk5: 2,
        value: 0x184,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2800400),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2800500),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x1,
    },
    RegEntry {
        key: RegKey::new(0x2800600),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 1,
        unk5: 2,
        value: 0x1501000,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2800700),
        unk4: 0x2,
        len: 8,
        unk8: 0x2020,
        unk6: 2,
        unk5: 2,
        value: 0x91,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2800800),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2800900),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x7550001,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2800a00),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x7550000,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2800b00),
        unk4: 0x1,
        len: 30,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x99,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2800c00),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2800d00),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2804000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 368,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2820100),
        unk4: 0x0,
        len: 4,
        unk8: 0x8020,
        unk6: 1,
        unk5: 2,
        value: 0x2,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x2820200),
        unk4: 0x0,
        len: 4,
        unk8: 0x420,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x1,
    },
    RegEntry {
        key: RegKey::new(0x2820300),
        unk4: 0x0,
        len: 4,
        unk8: 0x8020,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2820400),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0xe10,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2820500),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x2820600),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x3c,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x2820700),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2820800),
        unk4: 0x0,
        len: 4,
        unk8: 0x8020,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2820900),
        unk4: 0x2,
        len: 8,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0xb7,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2820b00),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2820c00),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x1011,
    },
    RegEntry {
        key: RegKey::new(0x2820e00),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2780100),
        unk4: 0x0,
        len: 4,
        unk8: 0x210,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2780200),
        unk4: 0x0,
        len: 4,
        unk8: 0x210,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2780300),
        unk4: 0x0,
        len: 4,
        unk8: 0x210,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2780400),
        unk4: 0x0,
        len: 4,
        unk8: 0x210,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x27c0100),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0xffffffff,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x11,
    },
    RegEntry {
        key: RegKey::new(0x2860100),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0xffffffff,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2860300),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0xffffffff,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2860500),
        unk4: 0x1,
        len: 40,
        unk8: 0x10,
        unk6: 2,
        unk5: 2,
        value: 0xbf,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2880100),
        unk4: 0x0,
        len: 4,
        unk8: 0x8020,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2880200),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x2900100),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2a00100),
        unk4: 0x1,
        len: 30,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0xe7,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2b00100),
        unk4: 0x0,
        len: 4,
        unk8: 0x210,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2b00200),
        unk4: 0x0,
        len: 4,
        unk8: 0x210,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2b80100),
        unk4: 0x2,
        len: 8,
        unk8: 0x10,
        unk6: 2,
        unk5: 2,
        value: 0x105,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc0100),
        unk4: 0x2,
        len: 8,
        unk8: 0x210,
        unk6: 2,
        unk5: 2,
        value: 0x10d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc0200),
        unk4: 0x2,
        len: 8,
        unk8: 0x210,
        unk6: 2,
        unk5: 2,
        value: 0x115,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc0300),
        unk4: 0x2,
        len: 8,
        unk8: 0x210,
        unk6: 2,
        unk5: 2,
        value: 0x11d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc0400),
        unk4: 0x2,
        len: 8,
        unk8: 0x210,
        unk6: 2,
        unk5: 2,
        value: 0x125,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc0500),
        unk4: 0x2,
        len: 8,
        unk8: 0x210,
        unk6: 2,
        unk5: 2,
        value: 0x12d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc0600),
        unk4: 0x2,
        len: 8,
        unk8: 0x210,
        unk6: 2,
        unk5: 2,
        value: 0x135,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc0700),
        unk4: 0x2,
        len: 8,
        unk8: 0x210,
        unk6: 2,
        unk5: 2,
        value: 0x13d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc0800),
        unk4: 0x2,
        len: 8,
        unk8: 0x210,
        unk6: 2,
        unk5: 2,
        value: 0x145,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc0900),
        unk4: 0x2,
        len: 8,
        unk8: 0x210,
        unk6: 2,
        unk5: 2,
        value: 0x14d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc0a00),
        unk4: 0x2,
        len: 8,
        unk8: 0x210,
        unk6: 2,
        unk5: 2,
        value: 0x155,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc0b00),
        unk4: 0x2,
        len: 8,
        unk8: 0x210,
        unk6: 2,
        unk5: 2,
        value: 0x15d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc4001),
        unk4: 0x1,
        len: 16,
        unk8: 0x210,
        unk6: 16,
        unk5: 2,
        value: 0x165,
        unk9: 0,
        unk1: 0x14,
        unk2: 0x8,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc4002),
        unk4: 0x2,
        len: 8,
        unk8: 0x210,
        unk6: 16,
        unk5: 2,
        value: 0x2a5,
        unk9: 0,
        unk1: 0x14,
        unk2: 0x8,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2bc4003),
        unk4: 0x2,
        len: 8,
        unk8: 0x210,
        unk6: 16,
        unk5: 2,
        value: 0x345,
        unk9: 0,
        unk1: 0x14,
        unk2: 0x8,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2be0100),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2be0200),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2c30100),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x2c30200),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x3800100),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x9,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x3800200),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x310,
    },
    RegEntry {
        key: RegKey::new(0x3800300),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0xff,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x310,
    },
    RegEntry {
        key: RegKey::new(0x3800400),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x5553,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x310,
    },
    RegEntry {
        key: RegKey::new(0x3800500),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x310,
    },
    RegEntry {
        key: RegKey::new(0x3800600),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x3800700),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x3800800),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x3e5,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x310,
    },
    RegEntry {
        key: RegKey::new(0x3800900),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x3800a00),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x111,
    },
    RegEntry {
        key: RegKey::new(0x5010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0xf,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x5020000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x2,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x5030000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x5040000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x5050000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x5060000),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x55,
    },
    RegEntry {
        key: RegKey::new(0x5070000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0xffffc7c0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x55,
    },
    RegEntry {
        key: RegKey::new(0x5080000),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0xffffb9b0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x55,
    },
    RegEntry {
        key: RegKey::new(0x5090000),
        unk4: 0x1,
        len: 30,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x3f5,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x50a0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x5140000),
        unk4: 0x2,
        len: 8,
        unk8: 0x220,
        unk6: 2,
        unk5: 2,
        value: 0x413,
        unk9: 108,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x5150000),
        unk4: 0x2,
        len: 8,
        unk8: 0x220,
        unk6: 2,
        unk5: 2,
        value: 0x41b,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x5160000),
        unk4: 0x2,
        len: 8,
        unk8: 0x220,
        unk6: 2,
        unk5: 2,
        value: 0x423,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x5170000),
        unk4: 0x2,
        len: 8,
        unk8: 0x220,
        unk6: 2,
        unk5: 2,
        value: 0x42b,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x5180000),
        unk4: 0x2,
        len: 8,
        unk8: 0x220,
        unk6: 2,
        unk5: 2,
        value: 0x433,
        unk9: 424,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1180c69e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7020000),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 1,
        unk5: 2,
        value: 0x1180c69e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7030000),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7040000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7050000),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 1,
        unk5: 2,
        value: 0x26d3ffde,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7060000),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 1,
        unk5: 2,
        value: 0x2fffffff,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7070000),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7800100),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x43b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7800200),
        unk4: 0x1,
        len: 17,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x47b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x7800300),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x58b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7800500),
        unk4: 0x2,
        len: 8,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x68b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7800600),
        unk4: 0x1,
        len: 65,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x70b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7800700),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xb1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x7800800),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0xb5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7800900),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0xb9b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7800a00),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xbdb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7800b00),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xc1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7800c00),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xc5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7800d00),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xc9b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7800e00),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0xcdb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7800f00),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0xd1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7801000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xd5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x7801100),
        unk4: 0x2,
        len: 8,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xd9b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7805c01),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xe1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x311,
    },
    RegEntry {
        key: RegKey::new(0x7805c02),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xe5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x310,
    },
    RegEntry {
        key: RegKey::new(0x7805c03),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xe9b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x310,
    },
    RegEntry {
        key: RegKey::new(0x7805c04),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xedb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x310,
    },
    RegEntry {
        key: RegKey::new(0x7805c05),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x7805c06),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x7805c07),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf9b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x311,
    },
    RegEntry {
        key: RegKey::new(0x7806401),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xfdb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7806402),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x101b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7806403),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x105b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7806c01),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x109b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x7806c02),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x10db,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7806c03),
        unk4: 0x2,
        len: 16,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x111b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7807801),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x121b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7807c01),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x125b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7807c02),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x129b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7807c03),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x12db,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808001),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x131b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808002),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x135b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808003),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x139b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808004),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x13db,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808005),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x141b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808006),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x145b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808007),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x149b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808008),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x14db,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808009),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x151b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780800a),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x155b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780800b),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x159b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780800c),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x15db,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780800d),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x161b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780800e),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x165b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780800f),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x169b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808010),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x16db,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808011),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x171b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808012),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x175b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808013),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x179b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808014),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x17db,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808015),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x181b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808016),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x185b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808017),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x189b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808018),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x18db,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808019),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x191b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780801a),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x195b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780801b),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x199b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780801c),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x19db,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780801d),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x1a1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780801e),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x1a5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780801f),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x1a9b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808080),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1adb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x7808081),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1b1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808082),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1b5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x7808083),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1b9b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808084),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1bdb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7808085),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1c1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x7809001),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1c5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x7809002),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x1c9b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x7809003),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1cdb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x7809004),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 16,
        unk5: 2,
        value: 0x1d1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x7809005),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1d5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x7809006),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1d9b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x780a001),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1ddb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780a002),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1e1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780a003),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1e5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780a004),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1e9b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780a005),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1edb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x780a006),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1f1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780a007),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1f5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x780a008),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1f9b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x780a009),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1fdb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x780a00a),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x201b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x780a00b),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x205b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x780a00c),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x209b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780a00d),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x20db,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780a00e),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x211b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780a00f),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x215b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x780b003),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x219b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x33,
    },
    RegEntry {
        key: RegKey::new(0x780b004),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x21db,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b007),
        unk4: 0x1,
        len: 17,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x221b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b008),
        unk4: 0x1,
        len: 37,
        unk8: 0x4010,
        unk6: 16,
        unk5: 2,
        value: 0x232b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b009),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x257b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b00a),
        unk4: 0x0,
        len: 4,
        unk8: 0x4010,
        unk6: 16,
        unk5: 2,
        value: 0x25bb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b00b),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x25fb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b00c),
        unk4: 0x1,
        len: 17,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x263b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b00d),
        unk4: 0x1,
        len: 65,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x274b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b00e),
        unk4: 0x1,
        len: 3,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x2b5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b00f),
        unk4: 0x1,
        len: 6,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x2b8b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b010),
        unk4: 0x1,
        len: 36,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x2beb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b011),
        unk4: 0x1,
        len: 11,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x2e2b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b012),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x2edb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b013),
        unk4: 0x1,
        len: 65,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0x2f1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780b801),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x332b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b802),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x336b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b803),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x33ab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b804),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x33eb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b805),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x342b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b806),
        unk4: 0x2,
        len: 8,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x346b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b807),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x34eb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b808),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x352b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b809),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x356b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b80a),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x35ab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b80b),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x35eb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b80c),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x362b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b80d),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x366b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b80e),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x36ab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b80f),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x36eb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b810),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x372b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b811),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x376b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b812),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x37ab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b813),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x37eb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b814),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x382b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b815),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x386b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b816),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x38ab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b817),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x38eb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b818),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x392b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b819),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x396b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b81a),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x39ab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780b81b),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x39eb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x780bc01),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3a2b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780bc02),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3a6b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780bc03),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3aab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780bd01),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3aeb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780bd02),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3b2b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780bd03),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3b6b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c001),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3bab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c002),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3beb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c003),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3c2b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c004),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3c6b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c005),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3cab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c006),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3ceb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c007),
        unk4: 0x1,
        len: 32,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3d2b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c008),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3f2b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c009),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3f6b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c00a),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3fab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x780c00b),
        unk4: 0x1,
        len: 41,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x3feb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c00c),
        unk4: 0x1,
        len: 41,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x427b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c00d),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x450b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c00e),
        unk4: 0x1,
        len: 128,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x454b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c010),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x4d4b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c011),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x4d8b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c012),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x4dcb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c013),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x4e0b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c014),
        unk4: 0x1,
        len: 32,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x4e4b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c015),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x504b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c016),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x508b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c017),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x50cb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c018),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x510b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c019),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x514b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c01a),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x518b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c01b),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x51cb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c01c),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x520b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c01d),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x524b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c01e),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x528b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c01f),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x52cb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c020),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x530b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c021),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x534b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c022),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x538b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c023),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x53cb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c024),
        unk4: 0x2,
        len: 8,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x540b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c025),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x548b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c026),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x54cb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c027),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x550b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c028),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x554b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c029),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x558b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c02a),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x55cb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c02b),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x560b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c02c),
        unk4: 0x1,
        len: 37,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x564b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c041),
        unk4: 0x1,
        len: 841,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x589b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c042),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x8d2b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c043),
        unk4: 0x1,
        len: 601,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x8d6b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c501),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xb2fb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c502),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xb33b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c504),
        unk4: 0x1,
        len: 81,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xb37b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c505),
        unk4: 0x1,
        len: 81,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xb88b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c506),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xbd9b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c601),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xbddb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c602),
        unk4: 0x1,
        len: 512,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xbe1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c603),
        unk4: 0x2,
        len: 240,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xde1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780c701),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xed1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780d001),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xed5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780d002),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xed9b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780d101),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xeddb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780d102),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xee1b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780dc01),
        unk4: 0x1,
        len: 37,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xee5b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780dc02),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf0ab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780dc03),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf0eb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780dc04),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf12b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x780dc05),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf16b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780dc06),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf1ab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780dc07),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf1eb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780e001),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf22b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780e002),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf26b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780e003),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf2ab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780e004),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf2eb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780e005),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf32b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780e101),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf36b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780e401),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf3ab,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780e402),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xf3eb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780e403),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 16,
        unk5: 2,
        value: 0xf42b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780f801),
        unk4: 0x2,
        len: 64,
        unk8: 0x1000,
        unk6: 16,
        unk5: 2,
        value: 0xf46b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780f802),
        unk4: 0x2,
        len: 8,
        unk8: 0x1000,
        unk6: 16,
        unk5: 2,
        value: 0xf86b,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x780f803),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 16,
        unk5: 2,
        value: 0xf8eb,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x9010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x9020000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x9030000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x9040000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x9050000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x9060000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x9070000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x9400100),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x9400200),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x3,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x9400300),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0xc,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x9400400),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0xa030000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 72,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x333,
    },
    RegEntry {
        key: RegKey::new(0xa040000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 76,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x333,
    },
    RegEntry {
        key: RegKey::new(0xa060000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 116,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x333,
    },
    RegEntry {
        key: RegKey::new(0xa070000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x2e,
        unk9: 120,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x333,
    },
    RegEntry {
        key: RegKey::new(0xa080000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 136,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x333,
    },
    RegEntry {
        key: RegKey::new(0xa0a0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 140,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0xa0d0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x220,
        unk6: 1,
        unk5: 2,
        value: 0x729,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0xa0f0000),
        unk4: 0x2,
        len: 128,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0xf92b,
        unk9: 216,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x30,
    },
    RegEntry {
        key: RegKey::new(0xa100000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 348,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0xa110000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 352,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x311,
    },
    RegEntry {
        key: RegKey::new(0xa120000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 388,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0xa130000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 392,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0xa140000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 400,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0xa150000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 404,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0xa160000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 408,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0xa170000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x301,
    },
    RegEntry {
        key: RegKey::new(0xa180000),
        unk4: 0x2,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0xffffffff,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x311,
    },
    RegEntry {
        key: RegKey::new(0xa190000),
        unk4: 0x2,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0xffffffff,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x311,
    },
    RegEntry {
        key: RegKey::new(0xa1a0000),
        unk4: 0x2,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0xffffffff,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x311,
    },
    RegEntry {
        key: RegKey::new(0xa1b0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0xa1c0000),
        unk4: 0x2,
        len: 1024,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0xf9ab,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0xb030000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 344,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0xb040000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0xb050000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0xb060000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 152,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0xb070000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x18,
        unk9: 156,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0xb080000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 160,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0xb090000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 356,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0xb400101),
        unk4: 0x2,
        len: 32,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0xfdab,
        unk9: 0,
        unk1: 0x14,
        unk2: 0x8,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0xc010000),
        unk4: 0x2,
        len: 8,
        unk8: 0x4020,
        unk6: 2,
        unk5: 2,
        value: 0x1002b,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x30,
    },
    RegEntry {
        key: RegKey::new(0xc020000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0xc400101),
        unk4: 0x2,
        len: 32,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x10033,
        unk9: 0,
        unk1: 0x14,
        unk2: 0x8,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x12010100),
        unk4: 0x2,
        len: 1000,
        unk8: 0x220,
        unk6: 16,
        unk5: 2,
        value: 0x102b3,
        unk9: 0,
        unk1: 0x20,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x14140100),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x17fb3,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14140200),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x17fc3,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14140300),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x17fd3,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14140400),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14140500),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x17fe3,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14140600),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x17ff3,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14140700),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14140800),
        unk4: 0x1,
        len: 256,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18003,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14140c00),
        unk4: 0x1,
        len: 128,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18103,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14140d00),
        unk4: 0x1,
        len: 128,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18183,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14140e00),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18203,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14140f00),
        unk4: 0x2,
        len: 8,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18213,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14190100),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14190600),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14190700),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14190800),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14190900),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14190a00),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x2,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x141e0100),
        unk4: 0x1,
        len: 33,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1821b,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e0200),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x2,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e0300),
        unk4: 0x1,
        len: 27,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1823c,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e0400),
        unk4: 0x1,
        len: 65,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18257,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e0500),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e4001),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e6001),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e6002),
        unk4: 0x1,
        len: 256,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18298,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e6003),
        unk4: 0x1,
        len: 128,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18398,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e6004),
        unk4: 0x1,
        len: 128,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18418,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e6005),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18498,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e6006),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x184a8,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e6007),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x184b8,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e6008),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e6009),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x184c8,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e600a),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x184d8,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e600b),
        unk4: 0x1,
        len: 16,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x184e8,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x141e600c),
        unk4: 0x2,
        len: 8,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x184f8,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x141e8001),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e8002),
        unk4: 0x1,
        len: 256,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18500,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x141e8003),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1f90,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x303,
    },
    RegEntry {
        key: RegKey::new(0x14230100),
        unk4: 0x1,
        len: 33,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18600,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14230200),
        unk4: 0x1,
        len: 33,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18621,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14230300),
        unk4: 0x1,
        len: 33,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18642,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14230400),
        unk4: 0x1,
        len: 33,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18663,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14230500),
        unk4: 0x1,
        len: 65,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18684,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14230600),
        unk4: 0x1,
        len: 65,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x186c5,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14230700),
        unk4: 0x1,
        len: 27,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18706,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14230800),
        unk4: 0x1,
        len: 27,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18721,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14234001),
        unk4: 0x1,
        len: 33,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1873c,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14234002),
        unk4: 0x1,
        len: 27,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1875d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14234003),
        unk4: 0x1,
        len: 33,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18778,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14234004),
        unk4: 0x1,
        len: 27,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18799,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14234005),
        unk4: 0x1,
        len: 33,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x187b4,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14234006),
        unk4: 0x1,
        len: 65,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x187d5,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14234007),
        unk4: 0x1,
        len: 33,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18816,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14234008),
        unk4: 0x1,
        len: 65,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18837,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x3,
    },
    RegEntry {
        key: RegKey::new(0x14280100),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x313,
    },
    RegEntry {
        key: RegKey::new(0x14280200),
        unk4: 0x1,
        len: 256,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18878,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x313,
    },
    RegEntry {
        key: RegKey::new(0x14280300),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1f90,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x313,
    },
    RegEntry {
        key: RegKey::new(0x142d0100),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x30,
    },
    RegEntry {
        key: RegKey::new(0x142d0200),
        unk4: 0x1,
        len: 33,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18978,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x30,
    },
    RegEntry {
        key: RegKey::new(0x142d0300),
        unk4: 0x1,
        len: 65,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x18999,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x30,
    },
    RegEntry {
        key: RegKey::new(0x142d0400),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x6,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x30,
    },
    RegEntry {
        key: RegKey::new(0x142e0100),
        unk4: 0x1,
        len: 65,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x189da,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x100,
    },
    RegEntry {
        key: RegKey::new(0x14320101),
        unk4: 0x1,
        len: 33,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x18a1b,
        unk9: 0,
        unk1: 0xa,
        unk2: 0x8,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x14320102),
        unk4: 0x1,
        len: 27,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x18b65,
        unk9: 0,
        unk1: 0xa,
        unk2: 0x8,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x14320103),
        unk4: 0x1,
        len: 65,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x18c73,
        unk9: 0,
        unk1: 0xa,
        unk2: 0x8,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x14320104),
        unk4: 0x2,
        len: 8,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x18efd,
        unk9: 0,
        unk1: 0xa,
        unk2: 0x8,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x14700000),
        unk4: 0x1,
        len: 16,
        unk8: 0x230,
        unk6: 2,
        unk5: 2,
        value: 0x18f4d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x14710000),
        unk4: 0x1,
        len: 16,
        unk8: 0x230,
        unk6: 2,
        unk5: 2,
        value: 0x18f5d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x14740000),
        unk4: 0x1,
        len: 16,
        unk8: 0x230,
        unk6: 2,
        unk5: 2,
        value: 0x18f6d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x14750000),
        unk4: 0x1,
        len: 16,
        unk8: 0x230,
        unk6: 2,
        unk5: 2,
        value: 0x18f7d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x14760000),
        unk4: 0x1,
        len: 16,
        unk8: 0x230,
        unk6: 2,
        unk5: 2,
        value: 0x18f8d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x14770000),
        unk4: 0x0,
        len: 4,
        unk8: 0x230,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x14780000),
        unk4: 0x1,
        len: 256,
        unk8: 0x230,
        unk6: 2,
        unk5: 2,
        value: 0x18f9d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x19010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x19600000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x19800000),
        unk4: 0x1,
        len: 17,
        unk8: 0x8020,
        unk6: 2,
        unk5: 2,
        value: 0x1909d,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x1311,
    },
    RegEntry {
        key: RegKey::new(0x19810000),
        unk4: 0x0,
        len: 4,
        unk8: 0x30,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x1311,
    },
    RegEntry {
        key: RegKey::new(0x1e010000),
        unk4: 0x2,
        len: 1024,
        unk8: 0x10,
        unk6: 2,
        unk5: 2,
        value: 0x190ae,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x31,
    },
    RegEntry {
        key: RegKey::new(0x1e020000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x20010000),
        unk4: 0x2,
        len: 1024,
        unk8: 0x4020,
        unk6: 2,
        unk5: 2,
        value: 0x194ae,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x33,
    },
    RegEntry {
        key: RegKey::new(0x20020100),
        unk4: 0x2,
        len: 1024,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x198ae,
        unk9: 0,
        unk1: 0xa,
        unk2: 0x10,
        unk3: 0x33,
    },
    RegEntry {
        key: RegKey::new(0x20034001),
        unk4: 0x2,
        len: 2000,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1c0ae,
        unk9: 0,
        unk1: 0x2,
        unk2: 0x8,
        unk3: 0x33,
    },
    RegEntry {
        key: RegKey::new(0x20400000),
        unk4: 0x2,
        len: 1024,
        unk8: 0x4020,
        unk6: 2,
        unk5: 2,
        value: 0x1d04e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x33,
    },
    RegEntry {
        key: RegKey::new(0x20410000),
        unk4: 0x0,
        len: 4,
        unk8: 0x1000,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x33,
    },
    RegEntry {
        key: RegKey::new(0x23010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x656e67,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x23020000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x656e67,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x23030000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x656e67,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x23040000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x23050000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x23060000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x23070000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x23080000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x23090000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x230a0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x656e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x230b0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x656e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x230c0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x656e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x230d0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x230e0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x230f0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x2020,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x28010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x28020000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x28030000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x29010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x29020000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x29030000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2a010000),
        unk4: 0x2,
        len: 32,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1d44e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x2d010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x320c0000),
        unk4: 0x1,
        len: 43,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1d46e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x37040000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x37050000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x33,
    },
    RegEntry {
        key: RegKey::new(0x37060000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x33,
    },
    RegEntry {
        key: RegKey::new(0x37090000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x12c,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x370a0000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x3c020000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x33,
    },
    RegEntry {
        key: RegKey::new(0x3c030000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x33,
    },
    RegEntry {
        key: RegKey::new(0x3c040000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x33,
    },
    RegEntry {
        key: RegKey::new(0x41810000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x41820000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x41010100),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1d499,
        unk9: 0,
        unk1: 0x20,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x41010200),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1d519,
        unk9: 0,
        unk1: 0x20,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x41010300),
        unk4: 0x2,
        len: 16,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1d599,
        unk9: 0,
        unk1: 0x20,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x41010400),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1d799,
        unk9: 0,
        unk1: 0x20,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x42800100),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1d819,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x42800200),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1d859,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x42800300),
        unk4: 0x2,
        len: 16,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1d899,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x42800400),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1d999,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x43800100),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1d9d9,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x43800200),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1da19,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x43800300),
        unk4: 0x2,
        len: 16,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1da59,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x43800400),
        unk4: 0x0,
        len: 4,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1db59,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0xc8800100),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1db99,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0xc8800200),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1dbd9,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0xc8800300),
        unk4: 0x2,
        len: 16,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1dc19,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0xc8800400),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1dd19,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x45010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x45020000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x2,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x45030000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x45050000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x45060000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x45400000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x55010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x56010000),
        unk4: 0x1,
        len: 33,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1dd59,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x64800100),
        unk4: 0x1,
        len: 17,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1dd7a,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x64800200),
        unk4: 0x2,
        len: 6,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1de8a,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x64800300),
        unk4: 0x2,
        len: 8,
        unk8: 0x4020,
        unk6: 16,
        unk5: 2,
        value: 0x1deea,
        unk9: 0,
        unk1: 0x10,
        unk2: 0x10,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x46010100),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0xf0,
    },
    RegEntry {
        key: RegKey::new(0x46010200),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0xf0,
    },
    RegEntry {
        key: RegKey::new(0x46010300),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x2027,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0xf0,
    },
    RegEntry {
        key: RegKey::new(0x49010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x310,
    },
    RegEntry {
        key: RegKey::new(0x49020100),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x49020200),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x49020300),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x20202,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x49020400),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x64,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x49020500),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x64,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x49020600),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x2,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x49020700),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x49020800),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x49020900),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x49020a00),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x64,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x49020b00),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x20202,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x49020c00),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x355,
    },
    RegEntry {
        key: RegKey::new(0x4b010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x33,
    },
    RegEntry {
        key: RegKey::new(0x4f010000),
        unk4: 0x0,
        len: 4,
        unk8: 0x10,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x4f400101),
        unk4: 0x2,
        len: 6,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1df6a,
        unk9: 0,
        unk1: 0xa,
        unk2: 0x8,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x4f400102),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1dfa6,
        unk9: 0,
        unk1: 0xa,
        unk2: 0x8,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x4f400103),
        unk4: 0x2,
        len: 64,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1dfce,
        unk9: 0,
        unk1: 0xa,
        unk2: 0x8,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x4f400104),
        unk4: 0x2,
        len: 8,
        unk8: 0x20,
        unk6: 16,
        unk5: 2,
        value: 0x1e24e,
        unk9: 0,
        unk1: 0xa,
        unk2: 0x8,
        unk3: 0x10,
    },
    RegEntry {
        key: RegKey::new(0x50400100),
        unk4: 0x2,
        len: 48,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1e29e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x70030000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x70040000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x70050000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0x0,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
    RegEntry {
        key: RegKey::new(0x73010100),
        unk4: 0x1,
        len: 32,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1e2ce,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x73010200),
        unk4: 0x1,
        len: 32,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1e2ee,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x73010300),
        unk4: 0x1,
        len: 32,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1e30e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x73010400),
        unk4: 0x1,
        len: 32,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1e32e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x73010500),
        unk4: 0x1,
        len: 32,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1e34e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x73010600),
        unk4: 0x1,
        len: 32,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1e36e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x73010700),
        unk4: 0x1,
        len: 256,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1e38e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x73010800),
        unk4: 0x1,
        len: 32,
        unk8: 0x20,
        unk6: 2,
        unk5: 2,
        value: 0x1e48e,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x0,
    },
    RegEntry {
        key: RegKey::new(0x76800000),
        unk4: 0x0,
        len: 4,
        unk8: 0x20,
        unk6: 1,
        unk5: 2,
        value: 0xffffffff,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x30,
    },
    RegEntry {
        key: RegKey::new(0x7802c9c8),
        unk4: 0x0,
        len: 4,
        unk8: 0x30,
        unk6: 1,
        unk5: 2,
        value: 0x1,
        unk9: 0,
        unk1: 0x0,
        unk2: 0x0,
        unk3: 0x300,
    },
];
